;; gorilla-repl.fileformat = 1

;; **
;;; # Predicting power consumptions with HTM
;; **

;; **
;;; _by Marcus Lewis_
;;; 
;;; ## Abstract
;;; 
;;; Comportex, an implementation of Hierarchical Temporal Memory, is used to stream Numenta's "hotgym" data set and predict subsequent values. The data is meticulously encoded into the language of HTM, and the HTM region is tuned and observed.
;;; 
;;; You can run my final solution in the browser [here](#TODO).
;;; 
;;; ## Jump around
;;; 
;;; **This page**
;;; 
;;; - [The data](#the_data)
;;; - [Encode the data](#encoding_the_data)
;;;   - [Timestamps](#encoding_timestamps)
;;;   - [Consumptions](#encoding_consumptions)
;;;     - [Moving window](#moving_window)
;;;       - [Settle down](#settle_down)
;;;     - [Probabilistic](#probabilistic)
;;;       - [The random casualties](#the_random_casualties)
;;;     - [Whining](#whining)
;;;     - [A third way](#a_third_way)
;;;   - [Put them all together](#encoding_combined)
;;; - [Choose a rubric](#choose_a_rubric) (**It's fine if you start here**)
;;; - [Design an HTM](#design_an_htm)
;;;   - [A third way](#a_third_way_again)
;;; - [Read the crystal ball](#read_the_crystal_ball)
;;; - [Did you just beat NuPIC?](#beat_nupic) (**Clickbait**)
;;; 
;;; **[Appendix](?path=hotgym_appendix_2015_09_19.clj)**
;;; - Errata from my talk
;;; - Encoder math and trials
;;;   - Coordinate encoder
;;;     - Doing the math
;;;     - That took a while
;;;     - Doing the trials
;;;   - Sampled linear encoder
;;; - Running on the JVM
;;; 
;;; ## This is code
;;; 
;;; This page is just a [ComportexViz](https://github.com/nupic-community/comportexviz) worksheet. Feel free to [tweak](link-to-worksheet) it. You might enjoy generating your own images -- every image in this page is created by the Clojure code above it.
;;; 
;;; Feel free to ignore the code.
;;; 
;;; Let's get this out of the way:
;; **

;; @@
(ns comportexviz.notebook.hotgym
  (:require [clj-time.core :as t]
            [clj-time.format :as f]
            [clj-time.predicates :as pr]
            [clojure.data.csv :as csv]
            [clojure.java.io :as io]
            [clojure.set :as set]
            [comportexviz.server.launchpad :refer [start-runner stop-all-runners]]
            [comportexviz.server.notebook :refer [viz]]
            [gorilla-plot.core :as plot]
            [org.nfrac.comportex.core :as core]
            [org.nfrac.comportex.cells :as cells]
            [org.nfrac.comportex.encoders :as e]
            [org.nfrac.comportex.protocols :as p]
            [org.nfrac.comportex.repl]
            [org.nfrac.comportex.topology :as topology])
  (:use [clojure.pprint]
        [clojure.stacktrace]))

(org.nfrac.comportex.repl/truncate-large-data-structures)

(def be-really-slow? true)
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;comportexviz.notebook.hotgym/be-really-slow?</span>","value":"#'comportexviz.notebook.hotgym/be-really-slow?"}
;; <=

;; **
;;; ## <a name="the_data" /> The data
;; **

;; **
;;; Here's a glance at the dataset:
;; **

;; @@
(def the-data
  (with-open [in-file (io/reader "data/rec-center-hourly.csv")]
    (doall (->> (csv/read-csv in-file)
                (drop 3)
                (map (fn [[timestamp-str consumption-str]]
                       [timestamp-str
                        (Double/parseDouble consumption-str)]))))))

(pprint (take 10 the-data))
;; @@
;; ->
;;; ([&quot;7/2/10 0:00&quot; 21.2]
;;;  [&quot;7/2/10 1:00&quot; 16.4]
;;;  [&quot;7/2/10 2:00&quot; 4.7]
;;;  [&quot;7/2/10 3:00&quot; 4.7]
;;;  [&quot;7/2/10 4:00&quot; 4.6]
;;;  [&quot;7/2/10 5:00&quot; 23.5]
;;;  [&quot;7/2/10 6:00&quot; 47.5]
;;;  [&quot;7/2/10 7:00&quot; 45.4]
;;;  [&quot;7/2/10 8:00&quot; 46.1]
;;;  [&quot;7/2/10 9:00&quot; 41.5])
;;; 
;; <-
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; **
;;; Each row is a `[timestamp power-consumption]` pair. The power consumption is measured in kilowatts.
;;; 
;;; Here's more data, zoomed out:
;; **

;; @@
(plot/list-plot (map (fn [i]
                       (let [[timestamp consumption] (nth the-data i)]
                         [i consumption]))
                     (range 0 200))
                :joined true
                :color "red")
;; @@

;; **
;;; It covers every hour between the dates:
;; **

;; @@
[(first the-data)
 (last the-data)]
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;7/2/10 0:00&quot;</span>","value":"\"7/2/10 0:00\""},{"type":"html","content":"<span class='clj-double'>21.2</span>","value":"21.2"}],"value":"[\"7/2/10 0:00\" 21.2]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;12/31/10 22:00&quot;</span>","value":"\"12/31/10 22:00\""},{"type":"html","content":"<span class='clj-double'>5.0</span>","value":"5.0"}],"value":"[\"12/31/10 22:00\" 5.0]"}],"value":"[[\"7/2/10 0:00\" 21.2] [\"12/31/10 22:00\" 5.0]]"}
;; <=

;; **
;;; The minimum and maximum power consumptions are:
;; **

;; @@
[(apply min-key second the-data)
 (apply max-key second the-data)]
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;7/3/10 4:00&quot;</span>","value":"\"7/3/10 4:00\""},{"type":"html","content":"<span class='clj-double'>4.4</span>","value":"4.4"}],"value":"[\"7/3/10 4:00\" 4.4]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;11/13/10 8:00&quot;</span>","value":"\"11/13/10 8:00\""},{"type":"html","content":"<span class='clj-double'>90.9</span>","value":"90.9"}],"value":"[\"11/13/10 8:00\" 90.9]"}],"value":"[[\"7/3/10 4:00\" 4.4] [\"11/13/10 8:00\" 90.9]]"}
;; <=

;; **
;;; Let's shape the data into the sequence of inputs that will go into our model. Just fetch everything we might use.
;; **

;; @@
(def inputs
  (->> the-data
       (map (fn [[timestamp-string consumption]]
              (let [timestamp (f/parse (f/formatter "MM/dd/yy HH:mm") timestamp-string)]
                {:timestamp timestamp-string
                 :day-of-week (t/day-of-week timestamp)
                 :hour-of-day (t/hour timestamp)
                 :is-weekend? (pr/weekend? timestamp)
                 :consumption consumption})))))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;comportexviz.notebook.hotgym/inputs</span>","value":"#'comportexviz.notebook.hotgym/inputs"}
;; <=

;; @@
(pprint (->> inputs
             (take-nth 24)
             (take 4)))
;; @@
;; ->
;;; ({:timestamp &quot;7/2/10 0:00&quot;,
;;;   :day-of-week 5,
;;;   :hour-of-day 0,
;;;   :is-weekend? false,
;;;   :consumption 21.2}
;;;  {:timestamp &quot;7/3/10 0:00&quot;,
;;;   :day-of-week 6,
;;;   :hour-of-day 0,
;;;   :is-weekend? true,
;;;   :consumption 22.5}
;;;  {:timestamp &quot;7/4/10 0:00&quot;,
;;;   :day-of-week 7,
;;;   :hour-of-day 0,
;;;   :is-weekend? true,
;;;   :consumption 22.2}
;;;  {:timestamp &quot;7/5/10 0:00&quot;,
;;;   :day-of-week 1,
;;;   :hour-of-day 0,
;;;   :is-weekend? false,
;;;   :consumption 22.4})
;;; 
;; <-
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; **
;;; ## <a name="encode_the_data" /> Encode the data
;; **

;; **
;;; We need to feed this information into a region of cortex. This region takes in bits that are either active or inactive. So we need an encoding scheme, converting each value into a set of bits. These bits should illustrate commonalities between values.
;; **

;; **
;;; ### Timestamps
;;; 
;;; Let's prepare a bag of tools, and later we'll decide which ones to use.
;;; 
;;; For this data, it seems reasonable to compress the timestamp down to:
;;; 
;;; 1. Hour of day
;;; 2. Day of week
;;; 3. Whether it's the weekend
;;; 
;;; There's some redundancy between 2 and 3, but we might only use a subset of these.
;;; 
;;; First, the easy one.
;; **

;; @@
(let [example-weekend-encoder (e/category-encoder [10] [true false])]
  (for [is-weekend? [true false]]
    [is-weekend? (viz example-weekend-encoder is-weekend?
                      {:display-mode :one-d})]))
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-unkown'>true</span>","value":"true"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAAeCAYAAAALvL+DAAAAQ0lEQVQYV2P8z8DwnwEKfvr5HWTEEPjh53cApoJBQmIDpoohrOXdu3dw7////7+REUPg7du3DTDvMzExHcBUMXS1AAAARJhyLe2f7AAAAABJRU5ErkJggg==' />"}],"value":"[true #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1720d352 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1720d352\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-unkown'>false</span>","value":"false"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAAeCAYAAAALvL+DAAAAQ0lEQVQYV2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoYwlr+MzDAvf/Tz+8gI4bADz+/AzDvM0hIbMBUMXS1AACbeJhyEY8ZUAAAAABJRU5ErkJggg==' />"}],"value":"[false #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5a929f97 \"comportexviz.server.notebook$eval28204$fn$reify__28208@5a929f97\"]]"}],"value":"([true #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1720d352 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1720d352\"]] [false #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5a929f97 \"comportexviz.server.notebook$eval28204$fn$reify__28208@5a929f97\"]])"}
;; <=

;; **
;;; The only question is how many bits to use. We'll get to that.
;;; 
;;; "Day of week" and "hour of day" are alike: a range of numbers where order is meaningful.
;; **

;; @@
(let [example-hour-encoder (e/linear-encoder [25] 2 [0 24])]
  (for [hour (range 24)]
    [hour (viz example-hour-encoder hour
               {:display-mode :one-d})]))
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>0</span>","value":"0"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABLCAYAAABX/ybHAAAAUUlEQVQ4T2P8z8DwnwEKfvr5HWTEEPjh53cApoJBQmIDpgrCWt69ewe35f///42MGAJv375tgNnCxMR0AFPFqBbUIBwNsdEEA8kxo/mFnOICAKbDxAArqpMjAAAAAElFTkSuQmCC' />"}],"value":"[0 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5e2881db \"comportexviz.server.notebook$eval28204$fn$reify__28208@5e2881db\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>1</span>","value":"1"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABLCAYAAABX/ybHAAAAUUlEQVQ4T2P8z8DwnwEKfvr5HWTEEPjh53cApoJBQmIDpgrCWt69ewe35f///42MGAJv375tgNnCxMR0AFPFqBbUIBwNsdEEA8kxo/mFnOICAKbDxAArqpMjAAAAAElFTkSuQmCC' />"}],"value":"[1 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x34d514da \"comportexviz.server.notebook$eval28204$fn$reify__28208@34d514da\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>2</span>","value":"2"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABLCAYAAABX/ybHAAAAUElEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOMP5nYIBr+enndxBT4Ief3wGYFgYJiQ3kaCHDYaNaSI3K0RAbDTFIRiUy7w+jBAMAVwrEAN0nAMkAAAAASUVORK5CYII=' />"}],"value":"[2 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x51f70c74 \"comportexviz.server.notebook$eval28204$fn$reify__28208@51f70c74\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>3</span>","value":"3"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABLCAYAAABX/ybHAAAAVElEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKogrOU/AwPclp9+fgcZMQR++PkdgNnCICGxAVMFYS308cuoLXRIMKOBPBrIkOKAZoUSAAdRxAA9RlcfAAAAAElFTkSuQmCC' />"}],"value":"[3 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4c5af0d2 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4c5af0d2\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>4</span>","value":"4"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABLCAYAAABX/ybHAAAAUElEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKqgiZb/DAxwh/308zvIiCHww8/vAMxhDBISGzBVENYyaL0/6rDBmSxH42W4xAsAt4nEAFOcmqgAAAAASUVORK5CYII=' />"}],"value":"[4 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x56e696db \"comportexviz.server.notebook$eval28204$fn$reify__28208@56e696db\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>5</span>","value":"5"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABLCAYAAABX/ybHAAAAUElEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoYLFr+MzDA/fLTz+8gI4bADz+/AzC/MEhIbMBUQVjLcAqxUb8Mm8Q/GpUkRiUAZ9DEAFc20uAAAAAASUVORK5CYII=' />"}],"value":"[5 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4a092b19 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4a092b19\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>6</span>","value":"6"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABLCAYAAABX/ybHAAAATklEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoYwlr+MzDAvf/Tz+8gI4bADz+/AzDvM0hIbMBUQVjLCA/kUe+P5Cw2WGMfABgXxABRJXhEAAAAAElFTkSuQmCC' />"}],"value":"[6 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x23740a80 \"comportexviz.server.notebook$eval28204$fn$reify__28208@23740a80\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>7</span>","value":"7"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABLCAYAAABX/ybHAAAAUElEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoYWVr+MzDAQ+ynn99BRgyBH35+B2AhxiAhsQFTBWEto/EymiwhaWg0V9KmUAIAyE/EAI0sKwEAAAAASUVORK5CYII=' />"}],"value":"[7 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x38b3b25c \"comportexviz.server.notebook$eval28204$fn$reify__28208@38b3b25c\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>8</span>","value":"8"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABLCAYAAABX/ybHAAAAVElEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahBgh9p+BAa7ip5/fQUYMgR9+fgdggcwgIbEBUwVhLaNROZqSIWloNCMTUY4BAHiWxAAfU6qNAAAAAElFTkSuQmCC' />"}],"value":"[8 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5a81abdf \"comportexviz.server.notebook$eval28204$fn$reify__28208@5a81abdf\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>9</span>","value":"9"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABLCAYAAABX/ybHAAAAUklEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahNQIsf8MDHBDf/r5HWTEEPjh53cAFi8MEhIbMFUQ1jIa+6OJH5KGRl7eBwAo3cQAcrwqHgAAAABJRU5ErkJggg==' />"}],"value":"[9 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x59531275 \"comportexviz.server.notebook$eval28204$fn$reify__28208@59531275\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>10</span>","value":"10"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABLCAYAAABX/ybHAAAAUklEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahAMUYv8ZGODu+Onnd5ARQ+CHn98BWFQySEhswFRBWMtoghnNL5A0NMSLCwDZFcQAgmHluAAAAABJRU5ErkJggg==' />"}],"value":"[10 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5af023d5 \"comportexviz.server.notebook$eval28204$fn$reify__28208@5af023d5\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>11</span>","value":"11"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABLCAYAAABX/ybHAAAAUklEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahEMnxP4zMMCd/tPP7yAjhsAPP78DsNhnkJDYgKmCsJbRNDaaxSBpaDCVMACJXMQAytdTAQAAAABJRU5ErkJggg==' />"}],"value":"[11 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x7f745b85 \"comportexviz.server.notebook$eval28204$fn$reify__28208@7f745b85\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>12</span>","value":"12"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABLCAYAAABX/ybHAAAAUklEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahMM6xP4zMMB9+9PP7yAjhsAPP78DsATDICGxAVMFYS2jyXI0V0LSEM0KJQA5o8QAttnw1AAAAABJRU5ErkJggg==' />"}],"value":"[12 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x28fcc8e7 \"comportexviz.server.notebook$eval28204$fn$reify__28208@28fcc8e7\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>13</span>","value":"13"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABLCAYAAABX/ybHAAAAVElEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhhppg/jMwwAPop5/fQUYMgR9+fgdgaYxBQmIDpgrCWkZT8mhGhqQh4soxAOnbxABrklzXAAAAAElFTkSuQmCC' />"}],"value":"[13 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x697da737 \"comportexviz.server.notebook$eval28204$fn$reify__28208@697da737\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>14</span>","value":"14"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABLCAYAAABX/ybHAAAAUElEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhRnGC+c/AAA/Tn35+BxkxBH74+R2AJUsGCYkNmCoIaxlN/CM47wMAmiLEAA0bZx8AAAAASUVORK5CYII=' />"}],"value":"[14 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3248df46 \"comportexviz.server.notebook$eval28204$fn$reify__28208@3248df46\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>15</span>","value":"15"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABLCAYAAABX/ybHAAAAUElEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNhAJ5j8DAzwafvr5HWTEEPjh53cAlpIZJCQ2YKogrGU0vwyX4gIASmnEAMN7hQgAAAAASUVORK5CYII=' />"}],"value":"[15 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6fbd02f9 \"comportexviz.server.notebook$eval28204$fn$reify__28208@6fbd02f9\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>16</span>","value":"16"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABLCAYAAABX/ybHAAAAUElEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNkQSzH8GBnjM/fTzO8iIIfDDz+8ALPEzSEhswFRBWMtoFhuUJQwA+qHEAEFt/xoAAAAASUVORK5CYII=' />"}],"value":"[16 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x21484419 \"comportexviz.server.notebook$eval28204$fn$reify__28208@21484419\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>17</span>","value":"17"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABLCAYAAABX/ybHAAAAUElEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNnwTzH8GBnhk//TzO8iIIfDDz+8ALL8wSEhswFRBWMtorqR9oQQAqujEAKHnQPsAAAAASUVORK5CYII=' />"}],"value":"[17 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x25081b18 \"comportexviz.server.notebook$eval28204$fn$reify__28208@25081b18\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>18</span>","value":"18"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABLCAYAAABX/ybHAAAAUUlEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDkGnF/+MzDA08dPP7+DmAI//PwOwLIYg4TEBnK0jGZkEssxAFsvxACK29K1AAAAAElFTkSuQmCC' />"}],"value":"[18 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x51ce4210 \"comportexviz.server.notebook$eval28204$fn$reify__28208@51ce4210\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>19</span>","value":"19"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABLCAYAAABX/ybHAAAAUklEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDmG3Pzyn4EBnqR++vkdZMQQ+OHndwCWKxkkJDZgqiCsZWTnfQALdsQAE9b1pQAAAABJRU5ErkJggg==' />"}],"value":"[19 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x56e3281a \"comportexviz.server.notebook$eval28204$fn$reify__28208@56e3281a\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>20</span>","value":"20"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABLCAYAAABX/ybHAAAAUklEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDmGjvnlPwMDPBX+9PM7yIgh8MPP7wAsIzNISGzAVEFYyzAqLgC7rsQAx0qvmwAAAABJRU5ErkJggg==' />"}],"value":"[20 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x35e8d653 \"comportexviz.server.notebook$eval28204$fn$reify__28208@35e8d653\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>21</span>","value":"21"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABLCAYAAABX/ybHAAAAU0lEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDlmcOeX/wwM8IT708/vICOGwA8/vwOwvM8gIbEBUwVhLYO1hAEAa/XEAHryDqcAAAAASUVORK5CYII=' />"}],"value":"[21 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x505dcb2d \"comportexviz.server.notebook$eval28204$fn$reify__28208@505dcb2d\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>22</span>","value":"22"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABLCAYAAABX/ybHAAAAU0lEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDlm2OWX/wwM8LT+08/vICOGwA8/vwOw4oJBQmIDpgrCWuhSKAEAHDzEALXra6YAAAAASUVORK5CYII=' />"}],"value":"[22 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x73a0cdf4 \"comportexviz.server.notebook$eval28204$fn$reify__28208@73a0cdf4\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>23</span>","value":"23"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABLCAYAAABX/ybHAAAAUUlEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDlmNL8cYPzPwADPHj/9/A5iCvzw8zsAK2EYJCQ2kKOF9HIMAMx0xABjVA6uAAAAAElFTkSuQmCC' />"}],"value":"[23 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3ccc8862 \"comportexviz.server.notebook$eval28204$fn$reify__28208@3ccc8862\"]]"}],"value":"([0 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5e2881db \"comportexviz.server.notebook$eval28204$fn$reify__28208@5e2881db\"]] [1 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x34d514da \"comportexviz.server.notebook$eval28204$fn$reify__28208@34d514da\"]] [2 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x51f70c74 \"comportexviz.server.notebook$eval28204$fn$reify__28208@51f70c74\"]] [3 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4c5af0d2 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4c5af0d2\"]] [4 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x56e696db \"comportexviz.server.notebook$eval28204$fn$reify__28208@56e696db\"]] [5 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4a092b19 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4a092b19\"]] [6 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x23740a80 \"comportexviz.server.notebook$eval28204$fn$reify__28208@23740a80\"]] [7 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x38b3b25c \"comportexviz.server.notebook$eval28204$fn$reify__28208@38b3b25c\"]] [8 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5a81abdf \"comportexviz.server.notebook$eval28204$fn$reify__28208@5a81abdf\"]] [9 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x59531275 \"comportexviz.server.notebook$eval28204$fn$reify__28208@59531275\"]] [10 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5af023d5 \"comportexviz.server.notebook$eval28204$fn$reify__28208@5af023d5\"]] [11 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x7f745b85 \"comportexviz.server.notebook$eval28204$fn$reify__28208@7f745b85\"]] [12 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x28fcc8e7 \"comportexviz.server.notebook$eval28204$fn$reify__28208@28fcc8e7\"]] [13 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x697da737 \"comportexviz.server.notebook$eval28204$fn$reify__28208@697da737\"]] [14 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3248df46 \"comportexviz.server.notebook$eval28204$fn$reify__28208@3248df46\"]] [15 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6fbd02f9 \"comportexviz.server.notebook$eval28204$fn$reify__28208@6fbd02f9\"]] [16 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x21484419 \"comportexviz.server.notebook$eval28204$fn$reify__28208@21484419\"]] [17 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x25081b18 \"comportexviz.server.notebook$eval28204$fn$reify__28208@25081b18\"]] [18 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x51ce4210 \"comportexviz.server.notebook$eval28204$fn$reify__28208@51ce4210\"]] [19 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x56e3281a \"comportexviz.server.notebook$eval28204$fn$reify__28208@56e3281a\"]] [20 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x35e8d653 \"comportexviz.server.notebook$eval28204$fn$reify__28208@35e8d653\"]] [21 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x505dcb2d \"comportexviz.server.notebook$eval28204$fn$reify__28208@505dcb2d\"]] [22 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x73a0cdf4 \"comportexviz.server.notebook$eval28204$fn$reify__28208@73a0cdf4\"]] [23 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3ccc8862 \"comportexviz.server.notebook$eval28204$fn$reify__28208@3ccc8862\"]])"}
;; <=

;; @@
(let [example-day-encoder (e/linear-encoder [8] 2 [1 8])]
  (for [day (range 1 8)]
    [day (viz example-day-encoder day
               {:display-mode :one-d})]))
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>1</span>","value":"1"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAAYCAYAAADd5VyeAAAAQklEQVQYV2P8z8DwnwEKfvr5HWTEEPjh53cApoJBQmIDpgrCWt69ewe35f///42MGAJv375tgNnCxMR0AFPFiNICANNah3M5pFwhAAAAAElFTkSuQmCC' />"}],"value":"[1 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x37007dd0 \"comportexviz.server.notebook$eval28204$fn$reify__28208@37007dd0\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>2</span>","value":"2"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAAYCAYAAADd5VyeAAAAQklEQVQYV2P8z8DwnwEKfvr5HWTEEPjh53cApoJBQmIDpgrCWt69ewe35f///42MGAJv375tgNnCxMR0AFPFiNICANNah3M5pFwhAAAAAElFTkSuQmCC' />"}],"value":"[2 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x61341bc4 \"comportexviz.server.notebook$eval28204$fn$reify__28208@61341bc4\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>3</span>","value":"3"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAAYCAYAAADd5VyeAAAAQElEQVQYV2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOMP5nYIBr+enndxBT4Ief3wGYFgYJiQ3kaCHDYcNICwCDoYdzWVxOFgAAAABJRU5ErkJggg==' />"}],"value":"[3 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5b80d600 \"comportexviz.server.notebook$eval28204$fn$reify__28208@5b80d600\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>4</span>","value":"4"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAAYCAYAAADd5VyeAAAAR0lEQVQYV2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKogrOU/AwPclp9+fgcZMQR++PkdgNnCICGxAVMFYS308QtdbAEAM+iHcxLN8tkAAAAASUVORK5CYII=' />"}],"value":"[4 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xd344da6 \"comportexviz.server.notebook$eval28204$fn$reify__28208@d344da6\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>5</span>","value":"5"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAAYCAYAAADd5VyeAAAARUlEQVQYV2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKqgiZb/DAxwh/308zvIiCHww8/vAMxhDBISGzBVENYyWL0PAOQgh3OxNkfmAAAAAElFTkSuQmCC' />"}],"value":"[5 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x39ee4f2d \"comportexviz.server.notebook$eval28204$fn$reify__28208@39ee4f2d\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>6</span>","value":"6"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAAYCAYAAADd5VyeAAAARUlEQVQYV2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoYLFr+MzDA/fLTz+8gI4bADz+/AzC/MEhIbMBUQVgLXUIMAJRnh3P3xDeDAAAAAElFTkSuQmCC' />"}],"value":"[6 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x78c589ef \"comportexviz.server.notebook$eval28204$fn$reify__28208@78c589ef\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>7</span>","value":"7"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAAYCAYAAADd5VyeAAAARUlEQVQYV2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoYwlr+MzDAvf/Tz+8gI4bADz+/AzDvM0hIbMBUQVgL6YEMAESuh3OaDOrAAAAAAElFTkSuQmCC' />"}],"value":"[7 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x25a86021 \"comportexviz.server.notebook$eval28204$fn$reify__28208@25a86021\"]]"}],"value":"([1 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x37007dd0 \"comportexviz.server.notebook$eval28204$fn$reify__28208@37007dd0\"]] [2 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x61341bc4 \"comportexviz.server.notebook$eval28204$fn$reify__28208@61341bc4\"]] [3 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5b80d600 \"comportexviz.server.notebook$eval28204$fn$reify__28208@5b80d600\"]] [4 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xd344da6 \"comportexviz.server.notebook$eval28204$fn$reify__28208@d344da6\"]] [5 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x39ee4f2d \"comportexviz.server.notebook$eval28204$fn$reify__28208@39ee4f2d\"]] [6 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x78c589ef \"comportexviz.server.notebook$eval28204$fn$reify__28208@78c589ef\"]] [7 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x25a86021 \"comportexviz.server.notebook$eval28204$fn$reify__28208@25a86021\"]])"}
;; <=

;; **
;;; Now, by tuning the numbers of these encoders we adjust:
;;; 
;;; - The radius for number similarity.
;;;   - i.e. how far should the window reach?
;;; - The bit difference between a number and its surrounding numbers.
;;;   - i.e. how far should we move the window of bits on each increment?
;;; - The importance of this piece of data. The more active bits devoted to one piece of information, the more impact it will have on HTM columns.
;; **

;; @@
(defn show-example-linear-encoder
  ([start end radius step-distance]
   (show-example-linear-encoder start end radius step-distance (range start end)))
  ([start end radius step-distance samples]
   (let [n-extra (long (* 2 radius step-distance))
         n-active (+ step-distance n-extra)
         n-bits (-> (- end start)
                    (* step-distance)
                    (+ n-extra)
                    Math/ceil
                    long)
         enc (e/linear-encoder [n-bits] n-active [start (dec end)])]
     (println (format "Showing (e/linear-encoder [%s] %s [%s %s])"
                      n-bits n-active start end))
     (for [i samples]
       [i (viz enc i)]))))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;comportexviz.notebook.hotgym/show-example-linear-encoder</span>","value":"#'comportexviz.notebook.hotgym/show-example-linear-encoder"}
;; <=

;; **
;;; Play with the similarity radius:
;; **

;; @@
(show-example-linear-encoder 0 24 2 2)
;; @@
;; ->
;;; Showing (e/linear-encoder [56] 10 [0 24])
;;; 
;; <-
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>0</span>","value":"0"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAVCAYAAAAnzezqAAAAgElEQVRIS2P8z8DwnwEKfvr5HWTftMkemf9twQI4/////42MjIz1MHmq8AfcAT/8/A7AfMQgIbGB4cWLAGT+1/Z2ARifiYnpwL9//xyoyWcc8BAYcAeMRsFoFKCHAM0LHvSCbMAdgJ4LaF3woBdkGAXRaBSMRgHd08BoLhjxBREAidyvcfdtdOEAAAAASUVORK5CYII=' />"}],"value":"[0 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x43663885 \"comportexviz.server.notebook$eval28204$fn$reify__28208@43663885\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>1</span>","value":"1"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAVCAYAAAAnzezqAAAAqUlEQVRIS2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGRkZ4fIg9RTzB9wBb9++bYD5kImJ6QBnfDyczyAhseFre7sAsvy/f/8cqMlnHPAQGHAHDHgU/GdggOcCUKpn37QJnurpkgsG3AE//PwOwFI1KNUzvHgRgMyneS4Y8BAYcAeMRsFoFKCHAM2rX/TqfMAdgJ4LaF3woFfnjAMeAgPugIGOAgCi2LVx1ysVdQAAAABJRU5ErkJggg==' />"}],"value":"[1 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6d629498 \"comportexviz.server.notebook$eval28204$fn$reify__28208@6d629498\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>2</span>","value":"2"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAVCAYAAAAnzezqAAAAmElEQVRIS2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGRkZ4fIg9RTzB9wBb9++bYD5kImJ6QBnfDyczyAhseFre7sAsvy/f/8cqMlnHPAQGHAHjEbBaBQMeAgMuAMGPBf8Z2CA1wWgsp990yZ42U+XumDAHfDDz+8ArGwHlf0ML14EIPNpXhcMeAgMuANGo2DERwEAlsa7cRdwC/AAAAAASUVORK5CYII=' />"}],"value":"[2 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x641dca66 \"comportexviz.server.notebook$eval28204$fn$reify__28208@641dca66\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>3</span>","value":"3"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAVCAYAAAAnzezqAAAAqklEQVRIS2N89+7dfwYo+P//f6OQsHA9jP/Tz+8g+6ZN9sj8bwsWwPkg9YyMjHD1ZPEH3AFv375tgPmQiYnpAGd8PJzPICGxgeHFiwCYPIj/tb1dAFn9v3//HCjhMw54CAy4AwZ9FFA91aPnGkJRQHMHEIoCaqd69FxDMBfQPARGo2DAQ2DAHTCaC0aj4D8DA7xFRJcWEHpdMOAO+OHndwC5xUPrFhB6XQAAWmvBcbZ0k9wAAAAASUVORK5CYII=' />"}],"value":"[3 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1ae9fd6f \"comportexviz.server.notebook$eval28204$fn$reify__28208@1ae9fd6f\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>4</span>","value":"4"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAVCAYAAAAnzezqAAAAoklEQVRIS2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd/DbggX2+NSj6yfIH3AHvH37tgHmIyYmpgP//v1zQOZzxsfD5RkkJDZ8bW8XwKceXT8hPiN6CKAHOfumTfAgp0sU0N0B6FGAHuQML14EwIJ8NApokgZGo2DAs+GgjwKql/1odQ3BkpDmDiAUBdQu+9HrhoEPAUK5YDQKaB4CIz4KAIeYxHH4XdyMAAAAAElFTkSuQmCC' />"}],"value":"[4 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x68b94ca7 \"comportexviz.server.notebook$eval28204$fn$reify__28208@68b94ca7\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>5</span>","value":"5"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAVCAYAAAAnzezqAAAApklEQVRIS2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd/DbggX2+NSj6yfIH3AHvH37tgHmIyYmpgP//v1zQOZzxsfD5RkkJDZ8bW8XwKceXT8hPuOAh8CAO2A0CkajAD0E0Ase9k2b4AUPXQoiujsAPRegFzwML14EwAoeuhREdA+BAU8Do1EwGgWEQoDqLSC0FhdGe4DmTTB0BxDKBdRuAaG3kAC6H8RxpoeSpAAAAABJRU5ErkJggg==' />"}],"value":"[5 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1186c37c \"comportexviz.server.notebook$eval28204$fn$reify__28208@1186c37c\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>6</span>","value":"6"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAVCAYAAAAnzezqAAAApElEQVRIS2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDyI/23BAjgfm3508zD4A+6At2/fNsB8xMTEdODfv38OyHzO+Hi4PIOExAaGFy8CYPIg/tf2dgF8+tHNQ+czkhoCFAc5WhQPvANIjQJKg3w0CjCy4WgUDHg2HI2C0ShADwGaV7+E6gK6OwA9F9C6+iVYF9A9BAY8DQx0FAAAZxDEcQL4q7cAAAAASUVORK5CYII=' />"}],"value":"[6 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1bc275d0 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1bc275d0\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>7</span>","value":"7"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAVCAYAAAAnzezqAAAAnElEQVRIS2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDyI/23BAjgfm3508zD4A+6At2/fNsB8xMTEdODfv38OyHzO+Hi4PIOExAaGFy8CYPIg/tf2dgF8+tHNQ+czDngIDLgDRqNgNApIDQGKCx60go7kbEh1B5CaCygteCguiKgeAgOeBkajYDQKBjwEBtwBA50LAI5cxHHgz4fSAAAAAElFTkSuQmCC' />"}],"value":"[7 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4e95f142 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4e95f142\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>8</span>","value":"8"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAVCAYAAAAnzezqAAAAo0lEQVRIS2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef38FvCxbYk6Ifw/wBd8Dbt28bYD5gYmI68O/fPwd8fM74eLh6BgmJDV/b2wVI0Y9uPuOAh8CAO2DQRwF6qmfftAme6umSC2juAEJRgJ7qGV68CIClerrkApqHAKFcQHMHjEbBaBSQGgIUV7/o1f2AO4DUXEBp9UtydUz1FtBoFKC1uADA48RxsYT9pgAAAABJRU5ErkJggg==' />"}],"value":"[8 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6c6b5043 \"comportexviz.server.notebook$eval28204$fn$reify__28208@6c6b5043\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>9</span>","value":"9"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAVCAYAAAAnzezqAAAAk0lEQVRIS2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef38FvCxbYk6Ifw/wBd8Dbt28bYD5gYmI68O/fPwd8fM74eLh6BgmJDV/b2wVI0Y9uPuOAh8CAO2A0CkajYMBDYMAdMOhzAXrZz75pE7zsp0tdQHMHEIoC9LKf4cWLAFjZT5e6gOYhQCgX0NwBo1Ew4qMAAPNqxHGbj+FOAAAAAElFTkSuQmCC' />"}],"value":"[9 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4bd1e391 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4bd1e391\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>10</span>","value":"10"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAVCAYAAAAnzezqAAAAnUlEQVRIS2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef30H2TZvsYepB/G8LFsD5xJjHOOAOePv2bQPMB0xMTAf+/fvngI/PGR8PV88gIbGB4cWLAJh6EP9re7sAKeYNfAiMRgGlIUByqkfPZQPuAEpzAampHj2XUZwLRqNgNAooTgOjuWC0ICIUAlRvAZFaF9DcAYRyAbVbQOh1AQAaxcRx7qlAXAAAAABJRU5ErkJggg==' />"}],"value":"[10 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x43b573ce \"comportexviz.server.notebook$eval28204$fn$reify__28208@43b573ce\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>11</span>","value":"11"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAVCAYAAAAnzezqAAAAoUlEQVRIS2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fge/LVhgT4p+xgF3wNu3bxtgLmZiYjrw798/B1L4nPHxcP0MEhIbvra3C5Cin+QQQA9y9k2b4EFOlyigugNIjQL0IGd48SIAFuSjUUBWGhiNAlILotFcQHHZj17XUBoFJFc+6A6gNBeQWvaj1zUU1wUUh8BoFAx4CAy4AwY6FwAAx6fEcanhGSEAAAAASUVORK5CYII=' />"}],"value":"[11 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x224b851b \"comportexviz.server.notebook$eval28204$fn$reify__28208@224b851b\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>12</span>","value":"12"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAVCAYAAAAnzezqAAAAnklEQVRIS2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fge/LVhgT4p+xgF3wNu3bxtgLmZiYjrw798/B1L4nPHxcP0MEhIbvra3C5Cif+BDYDQKBjwEBtwBQy4XoBc87Js2wQseuhREVHcAqVGAXvAwvHgRACt46FIQUT0ESM0FVHfAaBSMRgGlIUByCwi9xTXgDqA0F5DaAkJvcQEA+i7Ecajif2gAAAAASUVORK5CYII=' />"}],"value":"[12 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x14943059 \"comportexviz.server.notebook$eval28204$fn$reify__28208@14943059\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>13</span>","value":"13"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAVCAYAAAAnzezqAAAAjElEQVRIS2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fge/LVhgT4p+xgF3wNu3bxtgLmZiYjrw798/B1L4nPHxcP0MEhIbvra3C5Cif+BDYDQKBjwEBtwBo7lgNApIDQH0sp990yZ42U+XuoDqDiA1F6CX/QwvXgTAyn661AVUD4EBTwOjUTDiowAAk3nEcYmx0yQAAAAASUVORK5CYII=' />"}],"value":"[13 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x509d8431 \"comportexviz.server.notebook$eval28204$fn$reify__28208@509d8431\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>14</span>","value":"14"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAVCAYAAAAnzezqAAAAkElEQVRIS2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fgfZN22yh+kH8b8tWADnYzV/wB3w9u3bBpiLmZiYDvz798+BFD5nfDxcP4OExAaGFy8CYPpB/K/t7QL4zGMc8BAYcAeMRsFoFFA7BAgWPOgF3YA7gNq5gFDBg17QUb0gGo2C0SggOQ2M5oIRXxABAM5rxHFsDXcqAAAAAElFTkSuQmCC' />"}],"value":"[14 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x385ea6ef \"comportexviz.server.notebook$eval28204$fn$reify__28208@385ea6ef\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>15</span>","value":"15"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAVCAYAAAAnzezqAAAAoklEQVRIS2N89+7dfwYo+P//fyMjI2M9JXwhYWG4/p9+fge/LVhgj9e8AXfA27dvG2AuZGJiOvDv3z8HSvic8fFw8xgkJDZ8bW8XwGce44CHwIA7YMhHAXqqZ9+0CZ7q6ZILKHYApVGAnuoZXrwIgKV6uuQCikOA0lxAsQNGo2A0CqgdAgSrX/TqfsAdQO1cQKj6Ra/uKa6OSW4BjUYBWosLAAEBxHFxLXJ4AAAAAElFTkSuQmCC' />"}],"value":"[15 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x46c38fa2 \"comportexviz.server.notebook$eval28204$fn$reify__28208@46c38fa2\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>16</span>","value":"16"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAVCAYAAAAnzezqAAAAj0lEQVRIS2N89+7dfwYo+P//fyMjI2M9JXwhYWG4/p9+fge/LVhgj9e8AXfA27dvG2AuZGJiOvDv3z8HSvic8fFw8xgkJDZ8bW8XwGce44CHwIA7YDQKRqNgwENgwB0w5HMBetnPvmkTvOynS11AsQMojQL0sp/hxYsAWNlPl7qA4hCgNBdQ7IDRKBjxUQAAM4jEcYQijk8AAAAASUVORK5CYII=' />"}],"value":"[16 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x73e069db \"comportexviz.server.notebook$eval28204$fn$reify__28208@73e069db\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>17</span>","value":"17"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAVCAYAAAAnzezqAAAAnElEQVRIS2N89+7dfwYo+P//fyMjI2M9JXwhYWG4/p9+fgfZN22yh5kH4n9bsADOB9s34A54+/ZtA8yFTExMB/79++dACZ8zPh5uHoOExAaGFy8CYOaB+F/b2wWQzR/4EBiNAlqHAEaqR89lA+4AWucC9FSPnstongtGo2A0CgimgdFcMFoQURoCJLeAqF0XUOwASnMBqS0g9LoAAFrUxHGi4WsfAAAAAElFTkSuQmCC' />"}],"value":"[17 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4fe0d79a \"comportexviz.server.notebook$eval28204$fn$reify__28208@4fe0d79a\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>18</span>","value":"18"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAVCAYAAAAnzezqAAAAnklEQVRIS2N89+7dfwYo+P//fyMjI2M9NflCwsJw8376+R38tmCBPYr5A+6At2/fNsBcxMTEdODfv38O1ORzxsfDzWeQkNjwtb1dANl8RmqHAHqQs2/aBA9yukQByQ6gdhSgBznDixcBsCAfjQKsaWA0CgY8Gw77KMAo+9HrGlpHAUEH0DoK0Mt+9LqG5nUBwRAYjYIBD4EBd8BA5wIAB8XEcYXszpoAAAAASUVORK5CYII=' />"}],"value":"[18 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x2c47957 \"comportexviz.server.notebook$eval28204$fn$reify__28208@2c47957\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>19</span>","value":"19"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAVCAYAAAAnzezqAAAAmklEQVRIS2N89+7dfwYo+P//fyMjI2M9NflCwsJw8376+R38tmCBPYr5A+6At2/fNsBcxMTEdODfv38O1ORzxsfDzWeQkNjwtb1dANl8xgEPgQF3wGgUjEYBtUMAveBh37QJXvDQpSAi2QHUzgXoBQ/DixcBsIKHLgURySEw4GlgNApGo4DWIYDRAkJvcQ24A2idC9BbQOgtLgA6TMRxbBygIAAAAABJRU5ErkJggg==' />"}],"value":"[19 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x58204832 \"comportexviz.server.notebook$eval28204$fn$reify__28208@58204832\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>20</span>","value":"20"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAVCAYAAAAnzezqAAAAmklEQVRIS2N89+7dfwYo+P//fyMjI2M9NflCwsJw8376+R1k37TJHmY+iM844A54+/ZtA8xFTExMB/79++dATT5nfDzcfAYJiQ0ML14EwMwH8ekeAt8WLIBHATjK6R0FGA6gdxR8bW8XQI7igQ+B0SgY8BAYcAeM5oLRKKB2CBCqfmleF5DsAGrnAkLVL83rApJDYMDTwEBHAQALF8GysqJhjAAAAABJRU5ErkJggg==' />"}],"value":"[20 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x28fe4c5c \"comportexviz.server.notebook$eval28204$fn$reify__28208@28fe4c5c\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>21</span>","value":"21"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAVCAYAAAAnzezqAAAAiklEQVRIS2N89+7dfwYo+P//fyMjI2M9NflCwsJw8376+R1k37TJHmY+iM844A54+/ZtA8xFTExMB/79++dATT5nfDzcfAYJiQ0ML14EwMwH8Qc+BEajYMBDYMAdMOJywbcFC+AFEbjgo3cUYDiA3lHwtb1dALmgG/gQGI2CAQ+BAXfAaC4Y8VEAAO8CvDRZCnGnAAAAAElFTkSuQmCC' />"}],"value":"[21 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x616b02de \"comportexviz.server.notebook$eval28204$fn$reify__28208@616b02de\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>22</span>","value":"22"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAVCAYAAAAnzezqAAAAkklEQVRIS2N89+7dfwYo+P//fyMjI2M9LflCwsJw83/6+R1kHHAHvH37tgHmYyYmpgP//v1zoCWfMz4ebh+DhMSGgQ+B0SigdQigp3r2TZvsYWmMLrmAoANonQvQUz3DixcBsBCgSy4gGAIDngZGo2A0CugdAt8WLIAXRODqf8AdQO9c8LW9XQC5uh/4EBjxUQAAeiq2tqHA1MsAAAAASUVORK5CYII=' />"}],"value":"[22 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x40c4c165 \"comportexviz.server.notebook$eval28204$fn$reify__28208@40c4c165\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>23</span>","value":"23"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAVCAYAAAAnzezqAAAAgElEQVRIS2N89+7dfwYo+P//fyMjI2M9LflCwsJw83/6+R1kHHAHvH37tgHmYyYmpgP//v1zoCWfMz4ebh+DhMSGgQ+B0SgY8BAYcAeM5oLRKKB1CKCX/eybNtnDSlq61AUEHUDrXIBe9jO8eBEACwG61AUEQ2DA08BoFIz4KAAAoTaxOO7Iqz8AAAAASUVORK5CYII=' />"}],"value":"[23 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1a7f6097 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1a7f6097\"]]"}],"value":"([0 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x43663885 \"comportexviz.server.notebook$eval28204$fn$reify__28208@43663885\"]] [1 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6d629498 \"comportexviz.server.notebook$eval28204$fn$reify__28208@6d629498\"]] [2 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x641dca66 \"comportexviz.server.notebook$eval28204$fn$reify__28208@641dca66\"]] [3 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1ae9fd6f \"comportexviz.server.notebook$eval28204$fn$reify__28208@1ae9fd6f\"]] [4 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x68b94ca7 \"comportexviz.server.notebook$eval28204$fn$reify__28208@68b94ca7\"]] [5 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1186c37c \"comportexviz.server.notebook$eval28204$fn$reify__28208@1186c37c\"]] [6 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1bc275d0 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1bc275d0\"]] [7 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4e95f142 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4e95f142\"]] [8 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6c6b5043 \"comportexviz.server.notebook$eval28204$fn$reify__28208@6c6b5043\"]] [9 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4bd1e391 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4bd1e391\"]] [10 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x43b573ce \"comportexviz.server.notebook$eval28204$fn$reify__28208@43b573ce\"]] [11 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x224b851b \"comportexviz.server.notebook$eval28204$fn$reify__28208@224b851b\"]] [12 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x14943059 \"comportexviz.server.notebook$eval28204$fn$reify__28208@14943059\"]] [13 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x509d8431 \"comportexviz.server.notebook$eval28204$fn$reify__28208@509d8431\"]] [14 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x385ea6ef \"comportexviz.server.notebook$eval28204$fn$reify__28208@385ea6ef\"]] [15 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x46c38fa2 \"comportexviz.server.notebook$eval28204$fn$reify__28208@46c38fa2\"]] [16 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x73e069db \"comportexviz.server.notebook$eval28204$fn$reify__28208@73e069db\"]] [17 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4fe0d79a \"comportexviz.server.notebook$eval28204$fn$reify__28208@4fe0d79a\"]] [18 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x2c47957 \"comportexviz.server.notebook$eval28204$fn$reify__28208@2c47957\"]] [19 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x58204832 \"comportexviz.server.notebook$eval28204$fn$reify__28208@58204832\"]] [20 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x28fe4c5c \"comportexviz.server.notebook$eval28204$fn$reify__28208@28fe4c5c\"]] [21 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x616b02de \"comportexviz.server.notebook$eval28204$fn$reify__28208@616b02de\"]] [22 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x40c4c165 \"comportexviz.server.notebook$eval28204$fn$reify__28208@40c4c165\"]] [23 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1a7f6097 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1a7f6097\"]])"}
;; <=

;; @@
(show-example-linear-encoder 0 7 2 1)
;; @@
;; ->
;;; Showing (e/linear-encoder [11] 5 [0 7])
;;; 
;; <-
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>0</span>","value":"0"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABAAAAAJCAYAAAA7KqwyAAAAcklEQVQoU2P8z8DwnwEKfvr5HWTftMkemf9twQI4/////42MjIz1MHkwn2IDfvj5HYCZyCAhsYHhxYsAZP7X9nYBGJ+JienAv3//HJD5lLtg8HmBUKgLCws3wMOIgYEBIwxINgA9FgiFuqCgICLWgE4BAA9ijEAK6RcqAAAAAElFTkSuQmCC' />"}],"value":"[0 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x74e471e7 \"comportexviz.server.notebook$eval28204$fn$reify__28208@74e471e7\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>1</span>","value":"1"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABAAAAAJCAYAAAA7KqwyAAAAi0lEQVQoU2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGRkZ4fIg9YwUG/D27dsGmA1MTEwHOOPj4XwGCYkNX9vbBZDl//3754DMZ/zPwAD3AsjJ7Js2wZ1MlBcoNuCHn98BmJNATmZ48SIAmU91LwgLCyPCiIGBgeQwwDCAVC8ICgoivAz0KwB32o9AyyYgmwAAAABJRU5ErkJggg==' />"}],"value":"[1 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x632e7e24 \"comportexviz.server.notebook$eval28204$fn$reify__28208@632e7e24\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>2</span>","value":"2"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABAAAAAJCAYAAAA7KqwyAAAAj0lEQVQoU2N89+7dfwYo+P//f6OQsHA9jP/Tz+8g+6ZN9sj8bwsWwPkg9YwUG/D27dsGmA1MTEwHOOPj4XwGCYkNDC9eBMDkQfyv7e0CyOoJugDDyYyMcC8S5QWCBhDyArqT//3754Dihf8MDPBYICbUhYWFEWHEwMDASLEBP/z8DiCHMqFQFxQURKgHagQAXhaSQEd8T7AAAAAASUVORK5CYII=' />"}],"value":"[2 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5b2fb61c \"comportexviz.server.notebook$eval28204$fn$reify__28208@5b2fb61c\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>3</span>","value":"3"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABAAAAAJCAYAAAA7KqwyAAAAc0lEQVQoU2N89+7dfwYo+P//f6OQsHA9jP/Tz+8g+6ZN9sj8bwsWwPkg9YwUG/D27dsGmA1MTEwHOOPj4XwGCYkNDC9eBMDkQfyv7e0CyOopd8Hg9wJ6qAsLCyPCiIGBgWAYEDSAUCygh7qgoOABeKwAGQD7JZVAG2lZjwAAAABJRU5ErkJggg==' />"}],"value":"[3 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x57057e89 \"comportexviz.server.notebook$eval28204$fn$reify__28208@57057e89\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>4</span>","value":"4"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABAAAAAJCAYAAAA7KqwyAAAAgklEQVQoU2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd/DbggX2KOopNuDt27cNMBOZmJgO/Pv3zwGZzxkfD5dnkJDY8LW9XQBZnhHdBehOZt+0Ce5korxAsgHoXkB3MsOLFwEwJ1PFCxybNsHDCGQwyWGAYQCpXuCYNWsC3EtABgDLNJVAOcdeLwAAAABJRU5ErkJggg==' />"}],"value":"[4 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xb956c7 \"comportexviz.server.notebook$eval28204$fn$reify__28208@b956c7\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>5</span>","value":"5"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABAAAAAJCAYAAAA7KqwyAAAAhUlEQVQoU2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDyIz0ixAW/fvm2AmcjExHTg379/Dsh8zvh4uDyDhMQGhhcvAmDyID7JLvi2YAHcC2Avk+oFDANI9cLX9nYBZC9iuIBQqHNs2gQPI5BBlBuA7gVCoc4xa9YEeCwAGQB9DJKBGUx0NAAAAABJRU5ErkJggg==' />"}],"value":"[5 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1076c7aa \"comportexviz.server.notebook$eval28204$fn$reify__28208@1076c7aa\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>6</span>","value":"6"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABAAAAAJCAYAAAA7KqwyAAAAbklEQVQoU2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDyIz0ixAW/fvm2AmcjExHTg379/Dsh8zvh4uDyDhMQGhhcvAmDyID7lLhh6XuDYtAkeRqCwIDkMMAwgNRY4Zs2aAI8FIAMARn+PwvwBDykAAAAASUVORK5CYII=' />"}],"value":"[6 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x36cf1cee \"comportexviz.server.notebook$eval28204$fn$reify__28208@36cf1cee\"]]"}],"value":"([0 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x74e471e7 \"comportexviz.server.notebook$eval28204$fn$reify__28208@74e471e7\"]] [1 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x632e7e24 \"comportexviz.server.notebook$eval28204$fn$reify__28208@632e7e24\"]] [2 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5b2fb61c \"comportexviz.server.notebook$eval28204$fn$reify__28208@5b2fb61c\"]] [3 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x57057e89 \"comportexviz.server.notebook$eval28204$fn$reify__28208@57057e89\"]] [4 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xb956c7 \"comportexviz.server.notebook$eval28204$fn$reify__28208@b956c7\"]] [5 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1076c7aa \"comportexviz.server.notebook$eval28204$fn$reify__28208@1076c7aa\"]] [6 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x36cf1cee \"comportexviz.server.notebook$eval28204$fn$reify__28208@36cf1cee\"]])"}
;; <=

;; @@
(show-example-linear-encoder 0 24 1 1)
;; @@
;; ->
;;; Showing (e/linear-encoder [26] 3 [0 24])
;;; 
;; <-
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>0</span>","value":"0"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAhElEQVQ4T2P8z8DwnwEKfvr5Hfy2YIE9jP////9GRkbGeor4NLfgh5/fAZgLGSQkNnxtbxeA8ZmYmA78+/fPgRI+I7V9ICws3AB3MAMDA+0toHYQCQoKIoKcLj6geRzQPIjevXsHz2jUyFgYqYjmFrx9+xaebqmRsTBSEc19QHMLaB1EANIZVroPDSE9AAAAAElFTkSuQmCC' />"}],"value":"[0 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x347e870f \"comportexviz.server.notebook$eval28204$fn$reify__28208@347e870f\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>1</span>","value":"1"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAf0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Vfk0t+Dt27cNMBczMTEd+PfvnwM1+Yz/GRjgQfTTz+/gtwUL7CkJImFhYbiDQebQ3oIffn4HYC5mkJDY8LW9XYCSIBIUFESYRxcf0DwORoMIPedjJFOaBxG1iwoMH9DcAmqXRegZDQB261a69bxyCwAAAABJRU5ErkJggg==' />"}],"value":"[1 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x96fc014 \"comportexviz.server.notebook$eval28204$fn$reify__28208@96fc014\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>2</span>","value":"2"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAeElEQVQ4T2N89+7dfwYo+P//fyMjI2M9Vfk0t+Dt27cNMBczMTEd+PfvnwM1+YzU9oGwsDDcwSCH0t4CageRoKDgAVgQg33wn4EBnop++vkd/LZggT0lqQgjiGhuwQ8/P4SXJCQ2fG1vF6AkFY0GEcGMOBpEJAcRABu9VrptHtpFAAAAAElFTkSuQmCC' />"}],"value":"[2 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x7c3ab8b4 \"comportexviz.server.notebook$eval28204$fn$reify__28208@7c3ab8b4\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>3</span>","value":"3"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAnUlEQVQ4T2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGRkZ4fIg9QT5NLfg7du3DTAXMjExHeCMj4fzGSQkNnxtbxdAlv/3758DKXxGdB8Q9DKBIBIWFkY4kIGBgfYWoAcRqUGArl5QUPAALAhBNO19QPM4oHkQ/WdggGc0amQsjFREcwt++PkhYp0KGQsjFdHcBzS3gNZBBAAYNVm6wO8a5gAAAABJRU5ErkJggg==' />"}],"value":"[3 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x75f792d9 \"comportexviz.server.notebook$eval28204$fn$reify__28208@75f792d9\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>4</span>","value":"4"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAoUlEQVQ4T2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGRkZ4fIg9QT5NLfg7du3DTAXMjExHeCMj4fzGSQkNnxtbxdAlv/3758DKXxGavtAWFgY4UAGBgbaW0DtIBIUFDwAC0IQjeEDgqmCQCoiGERUtwA9iEhNJejqR4PoAKEgxAii/wwM8LKIGmUPRiqiuQU//PwQGYMKZQ96EAEAq3FculxT3PQAAAAASUVORK5CYII=' />"}],"value":"[4 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x35f13597 \"comportexviz.server.notebook$eval28204$fn$reify__28208@35f13597\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>5</span>","value":"5"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAkklEQVQ4T2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGRkZ4fIg9QT5NLfg7du3DTAXMjExHeCMj4fzGSQkNnxtbxdAlv/3758DKXxGavtAWFgY4UAGBgbaW0DtIBIUFDwAC0IQTXsf0DwO6B5EBDMOgYxGMBVR3QL0ICI1I6GrJ5iKqO4D9FREdQtoHUQA1YBfumDcJc4AAAAASUVORK5CYII=' />"}],"value":"[5 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x42e92c5d \"comportexviz.server.notebook$eval28204$fn$reify__28208@42e92c5d\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>6</span>","value":"6"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAg0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Vfk0t+Dt27cNMBczMTEd+PfvnwM1+YzoPhASFoYH0U8/v4PfFiywJyXIhIWF4Q4G6aO9BehBxBkfj3CBhMSGr+3tAqQEmaCg4AGYevr4gOZxMBpE6DkfI5nSPYgoLYsIZjSqW0Dtsgg9owEAPrxfundxNV4AAAAASUVORK5CYII=' />"}],"value":"[6 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6b3fd3ac \"comportexviz.server.notebook$eval28204$fn$reify__28208@6b3fd3ac\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>7</span>","value":"7"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAe0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Vfk0t+Dt27cNMBczMTEd+PfvnwM1+YzU9oGwsDDcwSCH0t4CageRoKDgAVgQY/WBkLAwPBX99PM7+G3BAntSUhXBIKK6BehBxBkfj4gkCYkNX9vbBUhJVaNBRDAjjgYRyUEEAKfpX7ptBRT5AAAAAElFTkSuQmCC' />"}],"value":"[7 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x2217f548 \"comportexviz.server.notebook$eval28204$fn$reify__28208@2217f548\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>8</span>","value":"8"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAnElEQVQ4T2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd/DbggX2+NSj62ekuQVv375tgLmIiYnpwL9//xyQ+Zzx8XB5BgmJDV/b2wXwqUfXT9AHGF5GC0J0eWFhYYSDGBgYaG8BoSBC9zIhvqCg4AFYEIJo2vuAUCqiOA7oHkSUZiyCqYjqFqAHEaUZi2AqoroP0FMR1S2gdRABAHHLX7opIBmpAAAAAElFTkSuQmCC' />"}],"value":"[8 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1e763110 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1e763110\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>9</span>","value":"9"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAm0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd/DbggX2+NSj62ekuQVv375tgLmIiYnpwL9//xyQ+Zzx8XB5BgmJDV/b2wXwqUfXT3UfCAsLIxzEwMBAewuoHUSCgoIHYEEIogn6ACNVoKUydHmSg4hiCwgFEXqqIMQfDSKMjIgeZASDiNKyh2AqoroF6KmI0rIHPYgAO61fuii1L5wAAAAASUVORK5CYII=' />"}],"value":"[9 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x15766d47 \"comportexviz.server.notebook$eval28204$fn$reify__28208@15766d47\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>10</span>","value":"10"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAl0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd/DbggX2+NSj62ekuQVv375tgLmIiYnpwL9//xyQ+Zzx8XB5BgmJDV/b2wXwqUfXT3UfCAsLIxzEwMBAewuoHUSCgoIHYEEIomnvA2qnIow4GPAgwsg4aBkRXZ7kVESxBYSCCD3jEOKTnIoo9gGhVESxBbQOIgAFj1+6Lh6Q2gAAAABJRU5ErkJggg==' />"}],"value":"[10 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xc17ef6 \"comportexviz.server.notebook$eval28204$fn$reify__28208@c17ef6\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>11</span>","value":"11"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAhUlEQVQ4T2N89+7dfwYo+P//fyMjI2M9Vfk0t+Dt27cNMBczMTEd+PfvnwM1+YyEfCAkLAwPsp9+fge/LVhgjy8IhYWF4Q4GqaO9BYSCiDM+HuEiCYkNX9vbBfAFoaCg4AGYPH18QPM4GA0i9JyPkUwHPIhILZtIzmgUW0AoiEgtm9AzGgBuvF+6vYnnRgAAAABJRU5ErkJggg==' />"}],"value":"[11 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1d617613 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1d617613\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>12</span>","value":"12"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAe0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Vfk0t+Dt27cNMBczMTEd+PfvnwM1+YzU9oGwsDDcwSCH0t4CageRoKDgAVgQE+UDIWFheKr66ed38NuCBfb4UhnJQUSxBYSCiDM+HhFpEhIbvra3C+BLZaNBNBpEmGURpakIANfpX7qPQh6qAAAAAElFTkSuQmCC' />"}],"value":"[12 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x32b03a1e \"comportexviz.server.notebook$eval28204$fn$reify__28208@32b03a1e\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>13</span>","value":"13"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAe0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Vfk0t+Dt27cNMBczMTEd+PfvnwM1+YzU9oGwsDDcwSCH0t4CageRoKDgAVgQE+UDIWFheKr66ed38NuCBfb4UhnJQUSxBYSCiDM+HhFpEhIbvra3C+BLZaNBNBpEmGURpakIANfpX7qPQh6qAAAAAElFTkSuQmCC' />"}],"value":"[13 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x140a755a \"comportexviz.server.notebook$eval28204$fn$reify__28208@140a755a\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>14</span>","value":"14"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAj0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef38FvCxbY49VPcwvevn3bAHMBExPTgX///jng43PGx8PVM0hIbPja3i6ATz0jtX0gLCyMcAADAwPtLaB2EAkKCh6ABRmIJtkHhFIZxUFEsgWkBhGhVDYaRAQzIslBRGrZQ3IqotgCQqmI1LIHPYgAa61futjTxx0AAAAASUVORK5CYII=' />"}],"value":"[14 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1f379fe3 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1f379fe3\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>15</span>","value":"15"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAi0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef38FvCxbY49VPcwvevn3bAHMBExPTgX///jng43PGx8PVM0hIbPja3i6ATz0jtX0gLCyMcAADAwPtLaB2EAkKCh6ABRmIpr0PaB4Hgy6ICGVEilMRyRaQGkSEMiLFqYhkH5Caiki2gNZBBAA1j1+6IrL/8gAAAABJRU5ErkJggg==' />"}],"value":"[15 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x9f0b3f8 \"comportexviz.server.notebook$eval28204$fn$reify__28208@9f0b3f8\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>16</span>","value":"16"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAgklEQVQ4T2N89+7dfwYo+P//fyMjI2M9Vfk0t+Dt27cNMBczMTEd+PfvnwM1+Yyk+kBIWBgehD/9/A5+W7DAHjlIhYWF4Q4GidPeAlKDiDM+HuFCCYkNX9vbBZCDVFBQ8ACMTx8f0DwORoOIYDIddEFEqKyiOKORbAGpQUSorELPaACevF+6jPVARwAAAABJRU5ErkJggg==' />"}],"value":"[16 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3a230812 \"comportexviz.server.notebook$eval28204$fn$reify__28208@3a230812\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>17</span>","value":"17"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAfElEQVQ4T2N89+7dfwYo+P//fyMjI2M9Vfk0t+Dt27cNMBczMTEd+PfvnwM1+YzU9oGwsDDcwSCH0t4CageRoKDgAVgQk+UDIWFheCr76ed38NuCBfbIqY7iICLZAlKDiDM+HhGJEhIbvra3CyCnutEgOjAaRATLIlKDCAAH+F+6hH9iegAAAABJRU5ErkJggg==' />"}],"value":"[17 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x7d440aa9 \"comportexviz.server.notebook$eval28204$fn$reify__28208@7d440aa9\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>18</span>","value":"18"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAj0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fge/LVhgj6Kf5ha8ffu2AWYjExPTgX///jmQwueMj4frZ5CQ2PC1vV0AWT8jpT5AD1JhYWGEhQwMDLS3gNIgQg9SQUHBA7AgAtG09wHN42DQBRGhjEVxKiLZAlKDiFDGojgVkewDUlMRyRbQOogA0ctfuiv4O7MAAAAASUVORK5CYII=' />"}],"value":"[18 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5eaf5a65 \"comportexviz.server.notebook$eval28204$fn$reify__28208@5eaf5a65\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>19</span>","value":"19"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAl0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fge/LVhgj6Kf5ha8ffu2AWYjExPTgX///jmQwueMj4frZ5CQ2PC1vV0AWT8jtX3AsWkT3IEgi2hvAbWDiGPWrAmwIKKKD9BTnbCwMCJOqBFEBC2gNIjQU52goOCB0SDCmxEpDiJCZQ/FqYhkC0hNRYTKHvQgAgB1nl+6QiNMIAAAAABJRU5ErkJggg==' />"}],"value":"[19 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x52cc5d3b \"comportexviz.server.notebook$eval28204$fn$reify__28208@52cc5d3b\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>20</span>","value":"20"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAkklEQVQ4T2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fge/LVhgj6Kf5ha8ffu2AWYjExPTgX///jmQwueMj4frZ5CQ2PC1vV0AWT8jtX3AsWkT3IEgi2hvAbWDiGPWrAmwIKKPD2geB4M+iNAzprCwMCLZUiMVEbSA0iBCz5iCgoIHqJqKCPqA0lRE0AJaBxEAGXFfuo6EysAAAAAASUVORK5CYII=' />"}],"value":"[20 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x708dbce8 \"comportexviz.server.notebook$eval28204$fn$reify__28208@708dbce8\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>21</span>","value":"21"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAgUlEQVQ4T2N89+7dfwYo+P//fyMjI2M9Vfk0t+Dt27cNMBczMTEd+PfvnwM1+YyU+kBIWBgepD/9/A5ybNoEdyDIobS3gNIg4oyPhwcxg4TEBo5ZsybAgpg+PqB5HIwGEcFkOuiDCL3sEhYWRiRbamQ0ghZQGkToZZegoOAB5IwGAFyPX7qCcwXsAAAAAElFTkSuQmCC' />"}],"value":"[21 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x2d41f112 \"comportexviz.server.notebook$eval28204$fn$reify__28208@2d41f112\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>22</span>","value":"22"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAd0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Vfk0t+Dt27cNMBczMTEd+PfvnwM1+YzU9oGwsDDcwSCH0t4CageRoKDgAVgQU8UHQsLC8FT308/vIMemTfA4pI8FlAYRZ3w8IlIlJDZwzJo1YTSIUDLiaBARLJsIBREAxbxfumG5eAsAAAAASUVORK5CYII=' />"}],"value":"[22 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x999faad \"comportexviz.server.notebook$eval28204$fn$reify__28208@999faad\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>23</span>","value":"23"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAiklEQVQ4T2N89+7dfwYo+P//fyMjI2M9JXwhYWG4/p9+fgcZaW7B27dvG2AuZmJiOvDv3z8HSvic8fFw8xgkJDZQ3QfCwsIICxgYGGhvAbWDSFBQ8AAsiEE07X1A7VSEEQeDPojQMxbHpk3wZE6VOCBoAaVBhJ6xOGbNmkDVVETQB5SmIoIW0DqIAAFzXPuT/BJrAAAAAElFTkSuQmCC' />"}],"value":"[23 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x390c4b20 \"comportexviz.server.notebook$eval28204$fn$reify__28208@390c4b20\"]]"}],"value":"([0 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x347e870f \"comportexviz.server.notebook$eval28204$fn$reify__28208@347e870f\"]] [1 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x96fc014 \"comportexviz.server.notebook$eval28204$fn$reify__28208@96fc014\"]] [2 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x7c3ab8b4 \"comportexviz.server.notebook$eval28204$fn$reify__28208@7c3ab8b4\"]] [3 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x75f792d9 \"comportexviz.server.notebook$eval28204$fn$reify__28208@75f792d9\"]] [4 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x35f13597 \"comportexviz.server.notebook$eval28204$fn$reify__28208@35f13597\"]] [5 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x42e92c5d \"comportexviz.server.notebook$eval28204$fn$reify__28208@42e92c5d\"]] [6 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6b3fd3ac \"comportexviz.server.notebook$eval28204$fn$reify__28208@6b3fd3ac\"]] [7 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x2217f548 \"comportexviz.server.notebook$eval28204$fn$reify__28208@2217f548\"]] [8 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1e763110 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1e763110\"]] [9 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x15766d47 \"comportexviz.server.notebook$eval28204$fn$reify__28208@15766d47\"]] [10 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xc17ef6 \"comportexviz.server.notebook$eval28204$fn$reify__28208@c17ef6\"]] [11 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1d617613 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1d617613\"]] [12 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x32b03a1e \"comportexviz.server.notebook$eval28204$fn$reify__28208@32b03a1e\"]] [13 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x140a755a \"comportexviz.server.notebook$eval28204$fn$reify__28208@140a755a\"]] [14 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1f379fe3 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1f379fe3\"]] [15 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x9f0b3f8 \"comportexviz.server.notebook$eval28204$fn$reify__28208@9f0b3f8\"]] [16 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3a230812 \"comportexviz.server.notebook$eval28204$fn$reify__28208@3a230812\"]] [17 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x7d440aa9 \"comportexviz.server.notebook$eval28204$fn$reify__28208@7d440aa9\"]] [18 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5eaf5a65 \"comportexviz.server.notebook$eval28204$fn$reify__28208@5eaf5a65\"]] [19 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x52cc5d3b \"comportexviz.server.notebook$eval28204$fn$reify__28208@52cc5d3b\"]] [20 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x708dbce8 \"comportexviz.server.notebook$eval28204$fn$reify__28208@708dbce8\"]] [21 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x2d41f112 \"comportexviz.server.notebook$eval28204$fn$reify__28208@2d41f112\"]] [22 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x999faad \"comportexviz.server.notebook$eval28204$fn$reify__28208@999faad\"]] [23 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x390c4b20 \"comportexviz.server.notebook$eval28204$fn$reify__28208@390c4b20\"]])"}
;; <=

;; @@
(show-example-linear-encoder 0 24 4 1)
;; @@
;; ->
;;; Showing (e/linear-encoder [32] 9 [0 24])
;;; 
;; <-
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>0</span>","value":"0"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAiklEQVQ4T2P8z8DwnwEKfvr5HWTftMkemf9twQI4/////42MjIz1MHmi+DS34Ief3wGYixgkJDYwvHgRgMz/2t4uAOMzMTEd+PfvnwMpfEaa+4DmFgy5IBIWFm6AxyEDAwPV4wDDAmoHkaCgICJVYvMBpRmLYBBR3QL0IKI0Y40GEcGyaDSISA4iAElCllhcQnLLAAAAAElFTkSuQmCC' />"}],"value":"[0 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6145e469 \"comportexviz.server.notebook$eval28204$fn$reify__28208@6145e469\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>1</span>","value":"1"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAsElEQVQ4T2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGRkZ4fIg9QT5NLfg7du3DTAXMjExHeCMj4fzGSQkNnxtbxdAlv/3758DKXzG/wwM8CACBQn7pk3wIKFKENHcgh9+fgdgXgYFCcOLFwHI/EEXRMLCwog4ZGBgoHocYFhA7SASFBREBDldfEDtVDTwQURp2UMwFVHdAvRURGnGIpiKqO4D9FREdQtoHUQAMKqZWLQbiIcAAAAASUVORK5CYII=' />"}],"value":"[1 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x64677116 \"comportexviz.server.notebook$eval28204$fn$reify__28208@64677116\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>2</span>","value":"2"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAArElEQVQ4T2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGRkZ4fIg9QT5NLfg7du3DTAXMjExHeCMj4fzGSQkNnxtbxdAlv/3758DKXxGmvuA5hbQPIj+MzDAUxEo1bBv2gRPNeSkImFhYUQcMjAwMNLcgh9+fgdgqQKUahhevAhA5pOaigQFBRHm0cUHo0GEXhaNgFREafFMMIiobgF6RiM1Y6EX3+gZDQCu1pxYSlwzBQAAAABJRU5ErkJggg==' />"}],"value":"[2 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x7128bf16 \"comportexviz.server.notebook$eval28204$fn$reify__28208@7128bf16\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>3</span>","value":"3"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAp0lEQVQ4T2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGRkZ4fIg9QT5NLfg7du3DTAXMjExHeCMj4fzGSQkNnxtbxdAlv/3758DKXxGmvuA5hYMuSASFhZGxCEDAwPV4wDDAmoHkaCg4AFYKgPRjP8ZGOAZDZSx2DdtgmcscjIahg9obsEPPz+ElyQkNjC8eBEA9yIZGW00iA4QKotGg4jkIAIAw9WfWFfNsDIAAAAASUVORK5CYII=' />"}],"value":"[3 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4f710891 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4f710891\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>4</span>","value":"4"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAtElEQVQ4T2N89+7dfwYo+P//f6OQsHA9jP/Tz+8g+6ZN9sj8bwsWwPkg9YyMjHD1WPk0t+Dt27cNMBcyMTEd4IyPh/MZJCQ2MLx4EQCTB/G/trcLIKv/9++fAz4+IyEfkBwk6EFGcwsIBRGpQYIeZFQPImFhYUQcMjAw0N4CageRoKDgAXiqo4sPqJ2KMOKA5kH0n4EBXhZRo+zB8AHNLfjh54eIdSqUPRipiOY+oLkFtA4iAK0+olhTSnmlAAAAAElFTkSuQmCC' />"}],"value":"[4 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1d9dbba \"comportexviz.server.notebook$eval28204$fn$reify__28208@1d9dbba\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>5</span>","value":"5"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAqElEQVQ4T2N89+7dfwYo+P//f6OQsHA9jP/Tz+8g+6ZN9sj8bwsWwPkg9YyMjHD1WPk0t+Dt27cNMBcyMTEd4IyPh/MZJCQ2MLx4EQCTB/G/trcLIKv/9++fAz4+I819QHMLBjyISE01wsLCiDhkYGAgGAcUW0AoiEhNNYKCggfgqY4uPiCUikaDaBikov8MDPDimhrFM0ZGo7kFP/z8EBmDCsUzekYDAC16pVhBaNtnAAAAAElFTkSuQmCC' />"}],"value":"[5 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5ee9f556 \"comportexviz.server.notebook$eval28204$fn$reify__28208@5ee9f556\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>6</span>","value":"6"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAmklEQVQ4T2N89+7dfwYo+P//f6OQsHA9jP/Tz+8g+6ZN9sj8bwsWwPkg9YyMjHD1WPk0t+Dt27cNMBcyMTEd4IyPh/MZJCQ2MLx4EQCTB/G/trcLIKv/9++fAz4+I819QHMLhlwQCQsLI+KQgYGB6nGAYQG1g0hQUPAAPNUR4wNSMxbJQUSxBYSCiNSMNRpEBwiVPaNBRHIQAQBEeqhYjtiAHAAAAABJRU5ErkJggg==' />"}],"value":"[6 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6a8d3192 \"comportexviz.server.notebook$eval28204$fn$reify__28208@6a8d3192\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>7</span>","value":"7"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAr0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd/DbggX2+NSj62ekuQVv375tgLmIiYnpwL9//xyQ+Zzx8XB5BgmJDV/b2wXwqUfXj+ED9CBh37QJHiRUCSKqW4AeROhBwvDiRQAsSAZFEAkLCyPijIGBgepxgGEBtYNIUFDwADxI6eID9IxGaSoafEFEatlDciqi2AJCqYjUsofkVESxDwilIootoHUQAQC9p6hYQJAYHgAAAABJRU5ErkJggg==' />"}],"value":"[7 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x2dff89d2 \"comportexviz.server.notebook$eval28204$fn$reify__28208@2dff89d2\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>8</span>","value":"8"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAoUlEQVQ4T2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd/DbggX2+NSj62ekuQVv375tgLmIiYnpwL9//xyQ+Zzx8XB5BgmJDV/b2wXwqUfXT3sfDL8gQk817Js2wVMNMalIWFgYEWcMDAwYcUB1C9BTEXqqYXjxIgCWaohJRYKCggfg6uniA/RUNBpEIzAVkVo8kxxEFFtAKKORWjyjZzQANuOoWAq1xpAAAAAASUVORK5CYII=' />"}],"value":"[8 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x623483dc \"comportexviz.server.notebook$eval28204$fn$reify__28208@623483dc\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>9</span>","value":"9"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAnklEQVQ4T2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd/DbggX2+NSj62ekuQVv375tgLmIiYnpwL9//xyQ+Zzx8XB5BgmJDV/b2wXwqUfXT3sfjAYReqoRFhZGxBkDAwPV4wDDAmqnIkFBwQOwVAaiMXyAnrHYN22CZyxiMhrBIKK6BehBhJ6xGF68CIB7mYiMNhpEBMui0SAiOYgAsBCoWGQCxl8AAAAASUVORK5CYII=' />"}],"value":"[9 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x73e69989 \"comportexviz.server.notebook$eval28204$fn$reify__28208@73e69989\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>10</span>","value":"10"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAsUlEQVQ4T2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDyI/23BAjgfm35Gmlvw9u3bBpiLmJiYDvz7988Bmc8ZHw+XZ5CQ2MDw4kUATB7E/9reLoBPP8k+IBgkaEFMewtIDSJCQYIexFT3gbCwMCLOGBgYaG8BtYNIUFDwADyV0cUHpGY0QskUIw7oHkSUlj0EUxHVLUAPIkrLHoKpiOo+QE9FVLeA1kEEAGbjqFjGX3pGAAAAAElFTkSuQmCC' />"}],"value":"[10 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x70e437c6 \"comportexviz.server.notebook$eval28204$fn$reify__28208@70e437c6\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>11</span>","value":"11"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAn0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDyI/23BAjgfm35Gmlvw9u3bBpiLmJiYDvz7988Bmc8ZHw+XZ5CQ2MDw4kUATB7E/9reLoBPP+19MPKCiFCqERYWRsQZAwMDyXFAsgWkpiJCqUZQUPAAPJXRxQekpqLRICJY2A3+IKK0eCaY0ahuAXpGo7R4Rs9oAB22qFg4ua3UAAAAAElFTkSuQmCC' />"}],"value":"[11 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x21849111 \"comportexviz.server.notebook$eval28204$fn$reify__28208@21849111\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>12</span>","value":"12"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAmElEQVQ4T2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDyI/23BAjgfm35Gmlvw9u3bBpiLmJiYDvz7988Bmc8ZHw+XZ5CQ2MDw4kUATB7E/9reLoBPP+19MBpE6KlIWFgYEWcMDAxUjwMMC6idigQFBQ/AUxk5PiCUsSgOIpItIDWICGWs0SAiWPaMBhHJQQQA1HqoWPrp8tQAAAAASUVORK5CYII=' />"}],"value":"[12 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x67e8cc9e \"comportexviz.server.notebook$eval28204$fn$reify__28208@67e8cc9e\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>13</span>","value":"13"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAmElEQVQ4T2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDyI/23BAjgfm35Gmlvw9u3bBpiLmJiYDvz7988Bmc8ZHw+XZ5CQ2MDw4kUATB7E/9reLoBPP+19MBpE6KlIWFgYEWcMDAxUjwMMC6idigQFBQ/AUxk5PiCUsSgOIpItIDWICGWs0SAiWPaMBhHJQQQA1HqoWPrp8tQAAAAASUVORK5CYII=' />"}],"value":"[13 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x20852b3b \"comportexviz.server.notebook$eval28204$fn$reify__28208@20852b3b\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>14</span>","value":"14"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAoElEQVQ4T2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef38FvCxbY49VPcwvevn3bAHMBExPTgX///jng43PGx8PVM0hIbPja3i6ATz0jzX1AcwsGPIjQUw37pk3wVIMtFQkLCyPiiIGBgWAcUGwBoSBCTzUML14EwFINtlQkKCh4AC5PFx8QSkWjQTQCUxGh4pnijEayBaRmNELFM3pGAwDG46hYz43bEQAAAABJRU5ErkJggg==' />"}],"value":"[14 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x718e62ea \"comportexviz.server.notebook$eval28204$fn$reify__28208@718e62ea\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>15</span>","value":"15"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAnElEQVQ4T2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef38FvCxbY49VPcwvevn3bAHMBExPTgX///jng43PGx8PVM0hIbPja3i6ATz0jzX1AcwuGXBAJCwsj4oiBgYHqcYBhAbWDSFBQ8AAsVYFogj5Az1jsmzbBMxa2jEZyEFFsAaEgQs9YDC9eBMCDAEtGGw2iDaNBhFFcU5qKAEAfqFgzFRHUAAAAAElFTkSuQmCC' />"}],"value":"[15 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x740769d3 \"comportexviz.server.notebook$eval28204$fn$reify__28208@740769d3\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>16</span>","value":"16"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAsElEQVQ4T2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef30H2TZvsYepB/G8LFsD5YPNobsHbt28bYC5gYmI68O/fPwd8fM74eLh6BgmJDQwvXgTA1IP4X9vbBZD1U+wDjCBBD2JKg4igBZQGEXqQoAcx1YNIWFgYEUcMDAy0t4DaQSQoKHgAnqro4gNqpyKMOBjwICK17CE5FVFsAaEgIrXsITkVUewDQqmIYgtoHUQA9uOoWGrEyo8AAAAASUVORK5CYII=' />"}],"value":"[16 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x552ad7f1 \"comportexviz.server.notebook$eval28204$fn$reify__28208@552ad7f1\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>17</span>","value":"17"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAoUlEQVQ4T2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef30H2TZvsYepB/G8LFsD5YPNobsHbt28bYC5gYmI68O/fPwd8fM74eLh6BgmJDQwvXgTA1IP4X9vbBZD1094Ho0GEnmqEhYURccTAwEBxHBC0gNJUhJ5qBAUFD8BTFV18QGkqGg0ijMIPvbAb/EFEavFMckaj2AJCGY3U4hk9owEArbaoWPxGAYYAAAAASUVORK5CYII=' />"}],"value":"[17 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5c7fe29c \"comportexviz.server.notebook$eval28204$fn$reify__28208@5c7fe29c\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>18</span>","value":"18"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAo0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef30H2TZvsYepB/G8LFsD5YPNobsHbt28bYC5gYmI68O/fPwd8fM74eLh6BgmJDQwvXgTA1IP4X9vbBZD1094Ho0GEnoo4Nm2CxyEoLqgeBxgWUDsVccyaNQGeqqjhA/SMJSwsjEjGdLGA0iBCz1iCgoIHRoMIpSwaDSKM4hu9uCYURAA+eqhYPQvgIgAAAABJRU5ErkJggg==' />"}],"value":"[18 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5e6442cb \"comportexviz.server.notebook$eval28204$fn$reify__28208@5e6442cb\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>19</span>","value":"19"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAsUlEQVQ4T2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fge/LVhgj6Kf5ha8ffu2AWYjExPTgX///jmQwueMj4frZ5CQ2PC1vV0AWT8jqT5ADxL2TZvgQUKVICLZAlKDCD1IGF68CIAFCV2CiGPTJngcgiymehxgWEDtIOKYNWsCPMjo4gNqJ9OhF0ToZY+wsDAiZ1MjDghaQGkqQi97BAUFD1A1FRH0AaWpiKAFtA4iAJGYqFhBP+evAAAAAElFTkSuQmCC' />"}],"value":"[19 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1162e125 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1162e125\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>20</span>","value":"20"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAArElEQVQ4T2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fge/LVhgj6Kf5ha8ffu2AWYjExPTgX///jmQwueMj4frZ5CQ2PC1vV0AWT8jzX1AcwsGXRChpxr2TZvgqQaUijg2bYLHISguSI4Dki0gNYjQUw3DixcBsFQDSkUcs2ZNgPPp4gNSU9FoEBEs7IZfEKEXz8LCwojCjxrJlKAFlGY09OJZUFDwAHJGAwDktqhYEhtYvAAAAABJRU5ErkJggg==' />"}],"value":"[20 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x127439a7 \"comportexviz.server.notebook$eval28204$fn$reify__28208@127439a7\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>21</span>","value":"21"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAm0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fge/LVhgj6Kf5ha8ffu2AWYjExPTgX///jmQwueMj4frZ5CQ2PC1vV0AWT8jzX1AcwuGXBBxbNoEj0NQXFA9DjAsoHYQccyaNQGWisjyAXrGYt+0CZ6xQBmN4iAi2QJSgwg9YzG8eBEADxIJiQ2jQXRgNIgIFtekBhEAN+OoWHLfulEAAAAASUVORK5CYII=' />"}],"value":"[21 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x31bc5277 \"comportexviz.server.notebook$eval28204$fn$reify__28208@31bc5277\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>22</span>","value":"22"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAqUlEQVQ4T2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fgfZN22yh+kH8RlpbsHbt28bYDYyMTEd+PfvnwMpfM74eLh+BgmJDQwvXgTA9IP4VPfBtwUL4EEEDnJqBxGGBdQOoq/t7QLIQUx1H3Bs2gSPQ5BFtLeA2kHEMWvWBHgqoosPqJ2KMOJg0AURobKH4lREsgWkBhGhsofiVESyD0hNRSRbQOsgAgAbI6WZd7zl8QAAAABJRU5ErkJggg==' />"}],"value":"[22 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4d322c91 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4d322c91\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>23</span>","value":"23"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAkklEQVQ4T2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fgfZN22yh+kH8RlpbsHbt28bYDYyMTEd+PfvnwMpfM74eLh+BgmJDQwvXgTA9IP4tPfBaBChpyKOTZvgcQiKC6rHAYYF1E5FHLNmTYCnIrr4gNqpaDSIMApDgsl00KUiQsUzxRmNZAtIDSJCxTN6RgMAHa2i2l+M9VsAAAAASUVORK5CYII=' />"}],"value":"[23 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4bdf87db \"comportexviz.server.notebook$eval28204$fn$reify__28208@4bdf87db\"]]"}],"value":"([0 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6145e469 \"comportexviz.server.notebook$eval28204$fn$reify__28208@6145e469\"]] [1 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x64677116 \"comportexviz.server.notebook$eval28204$fn$reify__28208@64677116\"]] [2 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x7128bf16 \"comportexviz.server.notebook$eval28204$fn$reify__28208@7128bf16\"]] [3 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4f710891 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4f710891\"]] [4 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1d9dbba \"comportexviz.server.notebook$eval28204$fn$reify__28208@1d9dbba\"]] [5 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5ee9f556 \"comportexviz.server.notebook$eval28204$fn$reify__28208@5ee9f556\"]] [6 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6a8d3192 \"comportexviz.server.notebook$eval28204$fn$reify__28208@6a8d3192\"]] [7 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x2dff89d2 \"comportexviz.server.notebook$eval28204$fn$reify__28208@2dff89d2\"]] [8 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x623483dc \"comportexviz.server.notebook$eval28204$fn$reify__28208@623483dc\"]] [9 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x73e69989 \"comportexviz.server.notebook$eval28204$fn$reify__28208@73e69989\"]] [10 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x70e437c6 \"comportexviz.server.notebook$eval28204$fn$reify__28208@70e437c6\"]] [11 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x21849111 \"comportexviz.server.notebook$eval28204$fn$reify__28208@21849111\"]] [12 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x67e8cc9e \"comportexviz.server.notebook$eval28204$fn$reify__28208@67e8cc9e\"]] [13 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x20852b3b \"comportexviz.server.notebook$eval28204$fn$reify__28208@20852b3b\"]] [14 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x718e62ea \"comportexviz.server.notebook$eval28204$fn$reify__28208@718e62ea\"]] [15 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x740769d3 \"comportexviz.server.notebook$eval28204$fn$reify__28208@740769d3\"]] [16 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x552ad7f1 \"comportexviz.server.notebook$eval28204$fn$reify__28208@552ad7f1\"]] [17 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5c7fe29c \"comportexviz.server.notebook$eval28204$fn$reify__28208@5c7fe29c\"]] [18 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5e6442cb \"comportexviz.server.notebook$eval28204$fn$reify__28208@5e6442cb\"]] [19 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1162e125 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1162e125\"]] [20 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x127439a7 \"comportexviz.server.notebook$eval28204$fn$reify__28208@127439a7\"]] [21 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x31bc5277 \"comportexviz.server.notebook$eval28204$fn$reify__28208@31bc5277\"]] [22 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4d322c91 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4d322c91\"]] [23 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4bdf87db \"comportexviz.server.notebook$eval28204$fn$reify__28208@4bdf87db\"]])"}
;; <=

;; @@
(show-example-linear-encoder 0 24 10 1)
;; @@
;; ->
;;; Showing (e/linear-encoder [44] 21 [0 24])
;;; 
;; <-
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>0</span>","value":"0"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAiElEQVRIS2P8z8DwnwEKfvr5HWTftMkeH//bggVw+f///zcyMjLWw9QTxae7hT/8/A7AXMggIbGB4cWLAHz8r+3tAjB5JiamA//+/XMghc9Idx/S3cJhF6TCwsIN8DTBwMBA8zjEsJDWQSooKIjIBQPiQ1qn0tEgpXrhPRqko0F6kNQaf+SVpQATfyBqXM9v6gAAAABJRU5ErkJggg==' />"}],"value":"[0 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x11c150af \"comportexviz.server.notebook$eval28204$fn$reify__28208@11c150af\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>1</span>","value":"1"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAArUlEQVRIS2N89+7dfwYo+P//f6OQsHA9jP/Tz+8g+6ZN9vj43xYsgMuD9DMyMsL1Y+XT3cK3b982wHzAxMR0gDM+Hs5nkJDYwPDiRQBMHhv/a3u7ALL+f//+OeDjM/5nYIAHKU2CED2I6W7hDz+/A/iCDD1ISQ1C9CCmeZAKCwsj0gQDAwP9LaR1kAoKCiKibEB8SOtUihGHo0GKXniTWliPBinV67/RIB1+QQoACyMjao98JPUAAAAASUVORK5CYII=' />"}],"value":"[1 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x75687bf2 \"comportexviz.server.notebook$eval28204$fn$reify__28208@75687bf2\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>2</span>","value":"2"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAp0lEQVRIS2N89+7dfwYo+P//f6OQsHA9jP/Tz+8g+6ZN9vj43xYsgMuD9DMyMsL1Y+XT3cK3b982wHzAxMR0gDM+Hs5nkJDYwPDiRQBMHhv/a3u7ALL+f//+OeDjM9Ldh3S3kO5B+p+BAZ5KaZEqhYWFEWmCgYGBke4W/vDzO4AvFaKnUlJTpaCgIML8AfHhaJCil6Wklp2jqZTq1dFokI4G6QZKy1IACYsmampY3rsAAAAASUVORK5CYII=' />"}],"value":"[2 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1a7a9b8d \"comportexviz.server.notebook$eval28204$fn$reify__28208@1a7a9b8d\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>3</span>","value":"3"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAq0lEQVRIS2N89+7dfwYo+P//f6OQsHA9jP/Tz+8g+6ZN9vj43xYsgMuD9DMyMsL1Y+XT3cK3b982wHzAxMR0gDM+Hs5nkJDYwPDiRQBMHhv/a3u7ALL+f//+OeDjM9Ldh3S3cNgFqbCwMCJNMDAw0DwOMSykdZAKCgoegKdykA//MzDAMz4tMjqGD+lu4Q8/P4SXaZDRR4OU6mXnaJCOBulBUpsUGCUNvTM+AA63KWp1iWcvAAAAAElFTkSuQmCC' />"}],"value":"[3 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x773f13d0 \"comportexviz.server.notebook$eval28204$fn$reify__28208@773f13d0\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>4</span>","value":"4"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAtElEQVRIS2N89+7dfwYo+P//f6OQsHA9jP/Tz+8g+6ZN9vj43xYsgMuD9DMyMsL1Y+XT3cK3b982wHzAxMR0gDM+Hs5nkJDYwPDiRQBMHhv/a3u7ALL+f//+OeDjM9Ldh3S3cNgFqbCwMCJNMDAw0DwOMSykdZAKCgoegKfyAfEhrVPpwAfpfwYGeFlKi7ITw4d0t/CHnx8iFdGg7MRIpXT3Id0tHA1S9CYGqU0KjGxB7yAFABqnLGr8Tc6wAAAAAElFTkSuQmCC' />"}],"value":"[4 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6c6061b0 \"comportexviz.server.notebook$eval28204$fn$reify__28208@6c6061b0\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>5</span>","value":"5"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAArklEQVRIS2N89+7dfwYo+P//f6OQsHA9jP/Tz+8g+6ZN9vj43xYsgMuD9DMyMsL1Y+XT3cK3b982wHzAxMR0gDM+Hs5nkJDYwPDiRQBMHhv/a3u7ALL+f//+OeDjM9Ldh3S3cNgFqbCwMCJNMDAw0DwOMSykdZAKCgoegKfyAfEhrVPpaJBSvfAe+CD9z8AAr55oUR1h+JDuFv7w80NkTBpURxgZn+4+pLuF9A5SAC1bL2qFkg44AAAAAElFTkSuQmCC' />"}],"value":"[5 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5ae12025 \"comportexviz.server.notebook$eval28204$fn$reify__28208@5ae12025\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>6</span>","value":"6"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAqUlEQVRIS2N89+7dfwYo+P//f6OQsHA9jP/Tz+8g+6ZN9vj43xYsgMuD9DMyMsL1Y+XT3cK3b982wHzAxMR0gDM+Hs5nkJDYwPDiRQBMHhv/a3u7ALL+f//+OeDjM9Ldh3S3cNgFqbCwMCJNMDAw0DwOMSykdZAKCgoegKfyAfEhrVPpaJBSvfAeDdJhGKT/GRjgNT4taniMREN3C3/4+SHKOhrU8OhlKQBG0zJq4PJFHQAAAABJRU5ErkJggg==' />"}],"value":"[6 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x14096e07 \"comportexviz.server.notebook$eval28204$fn$reify__28208@14096e07\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>7</span>","value":"7"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAmElEQVRIS2N89+7dfwYo+P//f6OQsHA9jP/Tz+8g+6ZN9vj43xYsgMuD9DMyMsL1Y+XT3cK3b982wHzAxMR0gDM+Hs5nkJDYwPDiRQBMHhv/a3u7ALL+f//+OeDjM9Ldh3S3cNgFqbCwMCJNMDAw0DwOMSykdZAKCgoegKfyAfEhrVPpaJBSvfAeDdLRIMVoARCq8UdeWQoAZw81avuk9k0AAAAASUVORK5CYII=' />"}],"value":"[7 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1ba74ec5 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1ba74ec5\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>8</span>","value":"8"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAqElEQVRIS2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDw2/rcFC+Dy2MxjpLuFb9++bYC5mImJ6cC/f/8ckPmc8fFweQYJiQ0ML14EwOSx8b+2twvgMw/Dh1QPQrQoor+F6EFK7SBEjyKa+1BYWBiRBhgYGOhvIa2DVFBQ8AA8VQ+ID9EzPrWzBUYcjgYppYX1aJA2jqZS9EREqIYfeWUpAJcPNWqMFXs9AAAAAElFTkSuQmCC' />"}],"value":"[8 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x7425d1e0 \"comportexviz.server.notebook$eval28204$fn$reify__28208@7425d1e0\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>9</span>","value":"9"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAn0lEQVRIS2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDw2/rcFC+Dy2MxjpLuFb9++bYC5mImJ6cC/f/8ckPmc8fFweQYJiQ0ML14EwOSx8b+2twvgM4/+Phx5QUrtVCksLIxIAwwMDBhxSHML0VMptVOloKDgAXiqHhAfoqfS0SBFL0sJlZ2jqbSR5olmNJVSWsOPvFQKAMcPNWp8iRNJAAAAAElFTkSuQmCC' />"}],"value":"[9 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x27163749 \"comportexviz.server.notebook$eval28204$fn$reify__28208@27163749\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>10</span>","value":"10"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAApUlEQVRIS2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDw2/rcFC+Dy2MxjpLuFb9++bYC5mImJ6cC/f/8ckPmc8fFweQYJiQ0ML14EwOSx8b+2twvgM4/+PhwNUkpTqbCwMCINMDAw0DwOMSykdSoVFBQ8AE/V2HxIaRCiFxwEg5TmFqIHKbUz+miQHhgNUvT6kFD9RzDjjwbpkAtSAPcPNWoXX/WGAAAAAElFTkSuQmCC' />"}],"value":"[10 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x78b00cfa \"comportexviz.server.notebook$eval28204$fn$reify__28208@78b00cfa\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>11</span>","value":"11"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAArklEQVRIS2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDw2/rcFC+Dy2MxjpLuFb9++bYC5mImJ6cC/f/8ckPmc8fFweQYJiQ0ML14EwOSx8b+2twvgM4/+PhwNUkpTqbCwMCINMDAw0DwOMSykdSoVFBQ8AE/VA+JDWqfSwReklKZK9LKYYCqluYXoqZTaZSfBVEpzH6KnUppbOBqklDYpCGZ8WqdSACceNWrkmTopAAAAAElFTkSuQmCC' />"}],"value":"[11 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x59c0b44f \"comportexviz.server.notebook$eval28204$fn$reify__28208@59c0b44f\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>12</span>","value":"12"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAq0lEQVRIS2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDw2/rcFC+Dy2MxjpLuFb9++bYC5mImJ6cC/f/8ckPmc8fFweQYJiQ0ML14EwOSx8b+2twvgM4/+PhwNUkpTqbCwMCINMDAw0DwOMSykdSoVFBQ8AE/VA+JDWqfS0SCleuE9+IKU0oyOXr0RzPg0txA941O7OiKY8WnuQ/SMT3ML6R2kAFceNWq5FBDlAAAAAElFTkSuQmCC' />"}],"value":"[12 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x330d446 \"comportexviz.server.notebook$eval28204$fn$reify__28208@330d446\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>13</span>","value":"13"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAq0lEQVRIS2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDw2/rcFC+Dy2MxjpLuFb9++bYC5mImJ6cC/f/8ckPmc8fFweQYJiQ0ML14EwOSx8b+2twvgM4/+PhwNUkpTqbCwMCINMDAw0DwOMSykdSoVFBQ8AE/VA+JDWqfS0SCleuE9+IKU0oyOXr0RzPg0txA941O7OiKY8WnuQ/SMT3ML6R2kAFceNWq5FBDlAAAAAElFTkSuQmCC' />"}],"value":"[13 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x158a1716 \"comportexviz.server.notebook$eval28204$fn$reify__28208@158a1716\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>14</span>","value":"14"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAlUlEQVRIS2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDw2/rcFC+Dy2MxjpLuFb9++bYC5mImJ6cC/f/8ckPmc8fFweQYJiQ0ML14EwOSx8b+2twvgM4/+PhwNUkpTqbCwMCINMDAw0DwOMSykdSoVFBQ8AE/VA+JDWqfS0SCleuE9GqSjQUpyjT/yylIAtx41aiimJp8AAAAASUVORK5CYII=' />"}],"value":"[14 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6da50e81 \"comportexviz.server.notebook$eval28204$fn$reify__28208@6da50e81\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>15</span>","value":"15"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAqUlEQVRIS2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef30H2TZvsYeqx8b8tWACXB5tPdwvfvn3bAHMhExPTgX///jng43PGx8PVM0hIbGB48SIAph4b/2t7uwCyeQR9SHEQokcRoSCluoWEgpTSIESPIpoHqbCwMCLOGRgY6G8hrYNUUFDwADwVD4gPaZ1KMeJwNEgpzfijQYpRH44GKXqNj17Dj7yyFADnHjVqykjYLAAAAABJRU5ErkJggg==' />"}],"value":"[15 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x311badb6 \"comportexviz.server.notebook$eval28204$fn$reify__28208@311badb6\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>16</span>","value":"16"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAApElEQVRIS2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef30H2TZvsYeqx8b8tWACXB5tPdwvfvn3bAHMhExPTgX///jng43PGx8PVM0hIbGB48SIAph4b/2t7uwCyefT34WiQUpoqhYWFEXHOwMBAMA6pbiGhVEppqhQUFDwAT8UD4kNCqXQ0SNHLUvSyczSVYlRPVE80o6l0NEjRa3z0Gh69LAUAFy01amnKQf8AAAAASUVORK5CYII=' />"}],"value":"[16 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x348ccd37 \"comportexviz.server.notebook$eval28204$fn$reify__28208@348ccd37\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>17</span>","value":"17"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAsElEQVRIS2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef30H2TZvsYeqx8b8tWACXB5tPdwvfvn3bAHMhExPTgX///jng43PGx8PVM0hIbGB48SIAph4b/2t7uwCyefT34WiQoqdaUlMpx6ZN8DQBikuaxyGGhbROpRyzZk2Ap2JifEhqEKJndGFhYUQ2GhALCQUppRldUFDwwGiQopSlo0GKXh8SzBajqZTqJQ29gxQA8Q81an1+lzkAAAAASUVORK5CYII=' />"}],"value":"[17 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x717b4e17 \"comportexviz.server.notebook$eval28204$fn$reify__28208@717b4e17\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>18</span>","value":"18"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAtElEQVRIS2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef30H2TZvsYeqx8b8tWACXB5tPdwvfvn3bAHMhExPTgX///jng43PGx8PVM0hIbGB48SIAph4b/2t7uwCyefT34WiQoqdaUlMpx6ZN8DQBikuaxyGGhbROpRyzZk2Ap+IB8SGtU+ngD1JSUyV62SksLIwomYiJQ6pbSCiVUlp2CgoKHiAplVLdh4RSKdUtHA3SIR+kAMsANWo4gK/fAAAAAElFTkSuQmCC' />"}],"value":"[18 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x34ef97bf \"comportexviz.server.notebook$eval28204$fn$reify__28208@34ef97bf\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>19</span>","value":"19"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAt0lEQVRIS2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef30H2TZvsYeqx8b8tWACXB5tPdwvfvn3bAHMhExPTgX///jng43PGx8PVM0hIbGB48SIAph4b/2t7uwCyefT34WiQoqdaUlMpx6ZN8DQBikuaxyGGhbROpRyzZk2Ap+IB8SGtU+lokFK98B78QUpqRkevjoSFhRGFPTHZguoWEsr4lFZHgoKCB0jK+FT3IaGMT3UL6R2kAKTxNWrwEPKmAAAAAElFTkSuQmCC' />"}],"value":"[19 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x58f5fa89 \"comportexviz.server.notebook$eval28204$fn$reify__28208@58f5fa89\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>20</span>","value":"20"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAr0lEQVRIS2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef30H2TZvsYeqx8b8tWACXB5tPdwvfvn3bAHMhExPTgX///jng43PGx8PVM0hIbGB48SIAph4b/2t7uwCyefT34WiQoqdaUlMpx6ZN8DQBikuaxyGGhbROpRyzZk2Ap+IB8SGtU+lokFK98B4N0hEYpKSWneg1vLCwMKL+JKakobqFhMpSSmt4QUHBA8hlKQB+4jVqCtpx+AAAAABJRU5ErkJggg==' />"}],"value":"[20 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xac261d5 \"comportexviz.server.notebook$eval28204$fn$reify__28208@ac261d5\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>21</span>","value":"21"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAmklEQVRIS2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef30H2TZvsYeqx8b8tWACXB5tPdwvfvn3bAHMhExPTgX///jng43PGx8PVM0hIbGB48SIAph4b/2t7uwCyefT34WiQoqdaUlMpx6ZN8DQBikuaxyGGhbROpRyzZk2Ap+IB8SGtU+lokFK98B4N0tEgbRwtS9FbAOhlKQBY0zVqfYPB3AAAAABJRU5ErkJggg==' />"}],"value":"[21 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x21628d6c \"comportexviz.server.notebook$eval28204$fn$reify__28208@21628d6c\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>22</span>","value":"22"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAApklEQVRIS2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fgfZN22yh+nHxmeku4Vv375tgLmIiYnpwL9//xxI4XPGx8P1M0hIbGB48SIAph8bn2QfkhqE3xYsgAcxOMpIDVKKLSQ1SEkNwq/t7QLIUURzH3Js2gRPEyCL6W8hrYOUY9asCfBUOyA+pHUqxYjD0SBFL8xJzfijQUqwPhwNUvQaf+SVpQAhWTKrnD6GLwAAAABJRU5ErkJggg==' />"}],"value":"[22 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6978b490 \"comportexviz.server.notebook$eval28204$fn$reify__28208@6978b490\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>23</span>","value":"23"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAlUlEQVRIS2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fgfZN22yh+nHxmeku4Vv375tgLmIiYnpwL9//xxI4XPGx8P1M0hIbGB48SIAph8bn/4+HA1S9FRLaqrk2LQJniZAcUtyHFJsIamplNRUyTFr1gR4qh0QH5KaSkeDFL0sHU2lBKsnihPNaCodLUsJ1fDoZSkAjPkv7JyXQ18AAAAASUVORK5CYII=' />"}],"value":"[23 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1eae2a84 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1eae2a84\"]]"}],"value":"([0 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x11c150af \"comportexviz.server.notebook$eval28204$fn$reify__28208@11c150af\"]] [1 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x75687bf2 \"comportexviz.server.notebook$eval28204$fn$reify__28208@75687bf2\"]] [2 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1a7a9b8d \"comportexviz.server.notebook$eval28204$fn$reify__28208@1a7a9b8d\"]] [3 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x773f13d0 \"comportexviz.server.notebook$eval28204$fn$reify__28208@773f13d0\"]] [4 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6c6061b0 \"comportexviz.server.notebook$eval28204$fn$reify__28208@6c6061b0\"]] [5 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5ae12025 \"comportexviz.server.notebook$eval28204$fn$reify__28208@5ae12025\"]] [6 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x14096e07 \"comportexviz.server.notebook$eval28204$fn$reify__28208@14096e07\"]] [7 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1ba74ec5 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1ba74ec5\"]] [8 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x7425d1e0 \"comportexviz.server.notebook$eval28204$fn$reify__28208@7425d1e0\"]] [9 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x27163749 \"comportexviz.server.notebook$eval28204$fn$reify__28208@27163749\"]] [10 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x78b00cfa \"comportexviz.server.notebook$eval28204$fn$reify__28208@78b00cfa\"]] [11 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x59c0b44f \"comportexviz.server.notebook$eval28204$fn$reify__28208@59c0b44f\"]] [12 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x330d446 \"comportexviz.server.notebook$eval28204$fn$reify__28208@330d446\"]] [13 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x158a1716 \"comportexviz.server.notebook$eval28204$fn$reify__28208@158a1716\"]] [14 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6da50e81 \"comportexviz.server.notebook$eval28204$fn$reify__28208@6da50e81\"]] [15 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x311badb6 \"comportexviz.server.notebook$eval28204$fn$reify__28208@311badb6\"]] [16 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x348ccd37 \"comportexviz.server.notebook$eval28204$fn$reify__28208@348ccd37\"]] [17 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x717b4e17 \"comportexviz.server.notebook$eval28204$fn$reify__28208@717b4e17\"]] [18 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x34ef97bf \"comportexviz.server.notebook$eval28204$fn$reify__28208@34ef97bf\"]] [19 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x58f5fa89 \"comportexviz.server.notebook$eval28204$fn$reify__28208@58f5fa89\"]] [20 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xac261d5 \"comportexviz.server.notebook$eval28204$fn$reify__28208@ac261d5\"]] [21 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x21628d6c \"comportexviz.server.notebook$eval28204$fn$reify__28208@21628d6c\"]] [22 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6978b490 \"comportexviz.server.notebook$eval28204$fn$reify__28208@6978b490\"]] [23 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1eae2a84 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1eae2a84\"]])"}
;; <=

;; **
;;; Play with the step distance:
;; **

;; @@
(show-example-linear-encoder 0 24 0.5 1)
;; @@
;; ->
;;; Showing (e/linear-encoder [25] 2 [0 24])
;;; 
;; <-
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>0</span>","value":"0"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABQAAAAPCAYAAADkmO9VAAAAcUlEQVQ4T2P8z8DwnwEKfvr5Hfy2YIE9jP////9GRkbGepL4VDfwh5/fAZgLGCQkNnxtbxeA8ZmYmA78+/fPgRQ+I9VdSHUDqe7ld+/ewWOZrFhFTwVUN/Dt27cNpMQioVhnpLoLqW7gqJdJzrvosQ4AjWxFJuMBcMUAAAAASUVORK5CYII=' />"}],"value":"[0 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x236fc525 \"comportexviz.server.notebook$eval28204$fn$reify__28208@236fc525\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>1</span>","value":"1"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABQAAAAPCAYAAADkmO9VAAAAaElEQVQ4T2N89+7dfwYo+P//fyMjI2M9RXyqG/j27dsGmIuYmJgO/Pv3z4ESPuN/Bga4l3/6+R38tmCBPUVeprqBP/z8DsBcxCAhseFre7vAqJdJinVGqqdDqhtI9YRNdRdS3UBqexkAJeRFJkvyZRMAAAAASUVORK5CYII=' />"}],"value":"[1 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4499ec9 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4499ec9\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>2</span>","value":"2"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABQAAAAPCAYAAADkmO9VAAAAbUlEQVQ4T2N89+7dfwYo+P//fyMjI2M9RXyqG/j27dsGmIuYmJgO/Pv3z4ESPiPVXUh1A6nu5f8MDPBY/unnd/DbggX2FMUy1Q384ed3AOYiBgmJDV/b2wUoimWqu5DqBlLdy1RPh1Q3kNoJGwC+TUUmIW2aBAAAAABJRU5ErkJggg==' />"}],"value":"[2 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6320edba \"comportexviz.server.notebook$eval28204$fn$reify__28208@6320edba\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>3</span>","value":"3"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABQAAAAPCAYAAADkmO9VAAAAZ0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9RXyqG/j27dsGmIuYmJgO/Pv3z4ESPiPVXUh1A0e9THks/2dggCfsn35+B78tWGBPUcKmuoE//PwOwFzEICGx4Wt7uwBFCZvqLqS6gdT2MgBWxUUmLMIiyQAAAABJRU5ErkJggg==' />"}],"value":"[3 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x60736722 \"comportexviz.server.notebook$eval28204$fn$reify__28208@60736722\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>4</span>","value":"4"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABQAAAAPCAYAAADkmO9VAAAAi0lEQVQ4T2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGRkZ4fIg9Rh8qhv49u3bBpgLmJiYDnDGx8P5DBISG762twsgy//7988BH58R3YUEvUSqlyk2EN3LhLxESH7Uy1gSKqFYRZenerKheiz/Z2CA52Wy8i66l6lu4A8/vwOwvElO3kVP6AA28kgmuR8+BgAAAABJRU5ErkJggg==' />"}],"value":"[4 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3de8c0e4 \"comportexviz.server.notebook$eval28204$fn$reify__28208@3de8c0e4\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>5</span>","value":"5"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABQAAAAPCAYAAADkmO9VAAAAeklEQVQ4T2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGRkZ4fIg9Rh8qhv49u3bBpgLmJiYDnDGx8P5DBISG762twsgy//7988BH5+R6i6kuoE09zLBWCQ1lik2EN3LhGKRkDxGLFPsQvRYptjAUS8fIBSLhOQBPfJLJtpQy90AAAAASUVORK5CYII=' />"}],"value":"[5 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x759fbe46 \"comportexviz.server.notebook$eval28204$fn$reify__28208@759fbe46\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>6</span>","value":"6"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABQAAAAPCAYAAADkmO9VAAAAYElEQVQ4T2N89+7dfwYo+P//fyMjI2M9RXyqG/j27dsGmIuYmJgO/Pv3z4ESPiO6C4WEheFe/unnd/DbggX2JAUB1Q1E9zJnfDw8CBgkJDZ8bW8XICUIRr08IhP28M/LACQQSyaXZ9rZAAAAAElFTkSuQmCC' />"}],"value":"[6 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1136c072 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1136c072\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>7</span>","value":"7"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABQAAAAPCAYAAADkmO9VAAAAa0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9RXyqG/j27dsGmIuYmJgO/Pv3z4ESPiPVXUh1A2nuZSFhYXgs//TzO/htwQJ7kmId3csUG4juZc74eHisM0hIbPja3i5ASqxjxDLFLhx6Xh7+eRkACi5LJuNRcWUAAAAASUVORK5CYII=' />"}],"value":"[7 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6a87ef01 \"comportexviz.server.notebook$eval28204$fn$reify__28208@6a87ef01\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>8</span>","value":"8"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABQAAAAPCAYAAADkmO9VAAAAa0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9RXyqG/j27dsGmIuYmJgO/Pv3z4ESPiPVXUh1A0e9TP1YFhIWhifsn35+B78tWGBPUkJHj2WKDUSPZc74eHhCZ5CQ2PC1vV2AlISOkbApduGg9zIA8D1LJu/UtyIAAAAASUVORK5CYII=' />"}],"value":"[8 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1193ed55 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1193ed55\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>9</span>","value":"9"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABQAAAAPCAYAAADkmO9VAAAAgUlEQVQ4T2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd/DbggX2+NQzUt3At2/fNsBsZGJiOvDv3z8HZD5nfDxcnkFCYsPX9nYBfOoJuhA9CAjyCXmZoAFoYc5IyMvoQUCIP+plzIRNcqQMvVgmOe+ip0N0L1NsIHrCJjXvoid0ACcfSyZIxk/KAAAAAElFTkSuQmCC' />"}],"value":"[9 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x19761caf \"comportexviz.server.notebook$eval28204$fn$reify__28208@19761caf\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>10</span>","value":"10"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABQAAAAPCAYAAADkmO9VAAAAd0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd/DbggX2+NQzUt3At2/fNsBsZGJiOvDv3z8HZD5nfDxcnkFCYsPX9nYBfOqp78Kh72X0WCfIJ+RlggagJTNGQrGMHuuE+ARjmWQXjnoZo3AgOQypHcsAXfJLJlwn2OEAAAAASUVORK5CYII=' />"}],"value":"[10 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3e647aa3 \"comportexviz.server.notebook$eval28204$fn$reify__28208@3e647aa3\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>11</span>","value":"11"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABQAAAAPCAYAAADkmO9VAAAAa0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9RXyqG/j27dsGmIuYmJgO/Pv3z4ESPiMhFwoJC8OD4Kef38FvCxbY4w0SqhtIyMuc8fHwIGGQkNjwtb1dAF+QjHp5NGGTk7cJJWySCwtCCZvUvA0ARBBLJuXevfgAAAAASUVORK5CYII=' />"}],"value":"[11 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6bce6589 \"comportexviz.server.notebook$eval28204$fn$reify__28208@6bce6589\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>12</span>","value":"12"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABQAAAAPCAYAAADkmO9VAAAAdklEQVQ4T2N89+7dfwYo+P//fyMjI2M9RXyqG/j27dsGmIuYmJgO/Pv3z4ESPiPVXUh1A+nuZSFhYXis//TzO/htwQJ7vKmAkJdJNpCQlznj4+GpgEFCYsPX9nYBfKmAYCyT7MKh72WS8zYhL5NsIKFYJjVvAwAqLksmzV9qpQAAAABJRU5ErkJggg==' />"}],"value":"[12 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6539d4e0 \"comportexviz.server.notebook$eval28204$fn$reify__28208@6539d4e0\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>13</span>","value":"13"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABQAAAAPCAYAAADkmO9VAAAAdklEQVQ4T2N89+7dfwYo+P//fyMjI2M9RXyqG/j27dsGmIuYmJgO/Pv3z4ESPiPVXUh1A+nuZSFhYXis//TzO/htwQJ7vKmAkJdJNpCQlznj4+GpgEFCYsPX9nYBfKmAYCyT7MKh72WS8zYhL5NsIKFYJjVvAwAqLksmzV9qpQAAAABJRU5ErkJggg==' />"}],"value":"[13 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4c032f8c \"comportexviz.server.notebook$eval28204$fn$reify__28208@4c032f8c\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>14</span>","value":"14"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABQAAAAPCAYAAADkmO9VAAAAe0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef38FvCxbYo6inuoFv375tgNnAxMR04N+/fw74+Jzx8XD1DBISG762twsgq2ck1YWEgoT6BpLqZUJBQn0XUj0MR71MMGFTHMsE8y563icUyyQbSCiWCeVd9CAAAEcfSybt2Y2jAAAAAElFTkSuQmCC' />"}],"value":"[14 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x51afe5cd \"comportexviz.server.notebook$eval28204$fn$reify__28208@51afe5cd\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>15</span>","value":"15"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABQAAAAPCAYAAADkmO9VAAAAcUlEQVQ4T2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef38FvCxbYo6inuoFv375tgNnAxMR04N+/fw74+Jzx8XD1DBISG762twsgq2ekugupbuCAe5lQKiA5DAkaSKqXCaUC6ruQ1Fge9TJmYULtWAYAffJLJpJ0A48AAAAASUVORK5CYII=' />"}],"value":"[15 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4955305c \"comportexviz.server.notebook$eval28204$fn$reify__28208@4955305c\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>16</span>","value":"16"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABQAAAAPCAYAAADkmO9VAAAAa0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9RXyqG/j27dsGmIuYmJgO/Pv3z4ESPiOpLhQSFoYHyU8/v4PfFiywRwkiqhtIqpc54+PhQcQgIbHha3u7AHIQjXp5NGGTk7dJTdgECw9SEzahvA4AZBBLJpVRr2gAAAAASUVORK5CYII=' />"}],"value":"[16 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x62606342 \"comportexviz.server.notebook$eval28204$fn$reify__28208@62606342\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>17</span>","value":"17"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABQAAAAPCAYAAADkmO9VAAAAdklEQVQ4T2N89+7dfwYo+P//fyMjI2M9RXyqG/j27dsGmIuYmJgO/Pv3z4ESPiPVXUh1Awfcy0LCwvBU8NPP7+C3BQvsUVIFqV4maCCpXuaMj4enCgYJiQ1f29sFkFMFybFM0IXDz8sE8zqpXiZoIKmxTCivAwBKLksmnxf9xQAAAABJRU5ErkJggg==' />"}],"value":"[17 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x37ca5fcd \"comportexviz.server.notebook$eval28204$fn$reify__28208@37ca5fcd\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>18</span>","value":"18"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABQAAAAPCAYAAADkmO9VAAAAa0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9RXyqG/j27dsGmIuYmJgO/Pv3z4ESPiPVXUh1A0e9TP9YFhIWhif8n35+B78tWGCPkhFIjWWCBpIay5zx8fCMwCAhseFre7sAckYgOWETdOGg9zIAMExLJlI7oikAAAAASUVORK5CYII=' />"}],"value":"[18 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x19e54574 \"comportexviz.server.notebook$eval28204$fn$reify__28208@19e54574\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>19</span>","value":"19"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABQAAAAPCAYAAADkmO9VAAAAdElEQVQ4T2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fgcZqW7g27dvG2AuYmJiOvDv3z8HUvic8fFw/QwSEhsodiFGEFHqZQwDKfUyehCNepn0hD0CYhk9735bsMAeJe+TmrAJGkhqwkbPu1/b2wWQ8z4AH9NIZzrfoBsAAAAASUVORK5CYII=' />"}],"value":"[19 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x795241c4 \"comportexviz.server.notebook$eval28204$fn$reify__28208@795241c4\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>20</span>","value":"20"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABQAAAAPCAYAAADkmO9VAAAAYElEQVQ4T2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fgcZqW7g27dvG2AuYmJiOvDv3z8HUvic8fFw/QwSEhuo78Lh72WMVEGplzEMpDSW0VMFxbE86uVG6ochtWMZAKrFRahEJ0zUAAAAAElFTkSuQmCC' />"}],"value":"[20 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x184947df \"comportexviz.server.notebook$eval28204$fn$reify__28208@184947df\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>21</span>","value":"21"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABQAAAAPCAYAAADkmO9VAAAAZUlEQVQ4T2N89+7dfwYo+P//fyMjI2M9RXyqG/j27dsGmIuYmJgO/Pv3z4ESPiOlLhQSFoYH0U8/v4PUN5BSL3PGx8ODjEFCYgP1XUj1MBx0XsbICJR6GcNASr2MnhEojuWh52UAx8hFqFeUfsQAAAAASUVORK5CYII=' />"}],"value":"[21 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x7d98472c \"comportexviz.server.notebook$eval28204$fn$reify__28208@7d98472c\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>22</span>","value":"22"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABQAAAAPCAYAAADkmO9VAAAAaElEQVQ4T2N89+7dfwYo+P//fyMjI2M9RXyqG/j27dsGmIuYmJgO/Pv3z4ESPiPVXUh1Awedl4WEheGp4qef30GKwxDDQEq9zBkfD08lDBISG6jvQkpjefB7GSPvU+plDAMpjWX0vA8A5MtFqI25+r0AAAAASUVORK5CYII=' />"}],"value":"[22 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4e65730e \"comportexviz.server.notebook$eval28204$fn$reify__28208@4e65730e\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>23</span>","value":"23"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABQAAAAPCAYAAADkmO9VAAAAXUlEQVQ4T2N89+7dfwYo+P//fyMjI2M9RXyqG/j27dsGmIuYmJgO/Pv3z4ESPiPVXUh1A0e9PPCxLCQsDM8IP/38DlKcbDAMpDSWOePj4RmDQUJiA/VdSGnCprmXAQHdRaiPGnqdAAAAAElFTkSuQmCC' />"}],"value":"[23 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x25d25767 \"comportexviz.server.notebook$eval28204$fn$reify__28208@25d25767\"]]"}],"value":"([0 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x236fc525 \"comportexviz.server.notebook$eval28204$fn$reify__28208@236fc525\"]] [1 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4499ec9 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4499ec9\"]] [2 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6320edba \"comportexviz.server.notebook$eval28204$fn$reify__28208@6320edba\"]] [3 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x60736722 \"comportexviz.server.notebook$eval28204$fn$reify__28208@60736722\"]] [4 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3de8c0e4 \"comportexviz.server.notebook$eval28204$fn$reify__28208@3de8c0e4\"]] [5 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x759fbe46 \"comportexviz.server.notebook$eval28204$fn$reify__28208@759fbe46\"]] [6 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1136c072 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1136c072\"]] [7 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6a87ef01 \"comportexviz.server.notebook$eval28204$fn$reify__28208@6a87ef01\"]] [8 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1193ed55 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1193ed55\"]] [9 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x19761caf \"comportexviz.server.notebook$eval28204$fn$reify__28208@19761caf\"]] [10 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3e647aa3 \"comportexviz.server.notebook$eval28204$fn$reify__28208@3e647aa3\"]] [11 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6bce6589 \"comportexviz.server.notebook$eval28204$fn$reify__28208@6bce6589\"]] [12 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6539d4e0 \"comportexviz.server.notebook$eval28204$fn$reify__28208@6539d4e0\"]] [13 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4c032f8c \"comportexviz.server.notebook$eval28204$fn$reify__28208@4c032f8c\"]] [14 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x51afe5cd \"comportexviz.server.notebook$eval28204$fn$reify__28208@51afe5cd\"]] [15 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4955305c \"comportexviz.server.notebook$eval28204$fn$reify__28208@4955305c\"]] [16 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x62606342 \"comportexviz.server.notebook$eval28204$fn$reify__28208@62606342\"]] [17 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x37ca5fcd \"comportexviz.server.notebook$eval28204$fn$reify__28208@37ca5fcd\"]] [18 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x19e54574 \"comportexviz.server.notebook$eval28204$fn$reify__28208@19e54574\"]] [19 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x795241c4 \"comportexviz.server.notebook$eval28204$fn$reify__28208@795241c4\"]] [20 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x184947df \"comportexviz.server.notebook$eval28204$fn$reify__28208@184947df\"]] [21 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x7d98472c \"comportexviz.server.notebook$eval28204$fn$reify__28208@7d98472c\"]] [22 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4e65730e \"comportexviz.server.notebook$eval28204$fn$reify__28208@4e65730e\"]] [23 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x25d25767 \"comportexviz.server.notebook$eval28204$fn$reify__28208@25d25767\"]])"}
;; <=

;; @@
(show-example-linear-encoder 0 24 0.5 4)
;; @@
;; ->
;;; Showing (e/linear-encoder [100] 8 [0 24])
;;; 
;; <-
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>0</span>","value":"0"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAAnElEQVRYR+2XsRWAIAxECRNYso1OwEpqxRqM4Ag6EHT6tCE2YmFjkYIUR3ddXo7PccTGsHnO5f12xNhXzcwzEY1NtfoBT+/XuiHj3LKH0FVtrV1LKUNLTeo3qH5AWCx9BWCx8BUAxeIkAsWgWEihNMtBMSgWbwAUg+KfUobvlvCKUM757cUqevCXevUDppSmlr33L6thsTiJtFt8Axw00mlRJEd1AAAAAElFTkSuQmCC' />"}],"value":"[0 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x51bd1747 \"comportexviz.server.notebook$eval28204$fn$reify__28208@51bd1747\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>1</span>","value":"1"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAAwElEQVRYR2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGRkZ4fIg9TTnD3oHvn37tgEWQkxMTAc44+PhfAYJiQ1f29sFkOX//fvnQE8+46APwUHvwCEXxTTPlaSWAuhRPOgciB7F9M6lhOzDyMWDLgRHo5jCgp3xPwMDvC4eFHUvei4f9A784ed3AFa3Doa6Fz1Xj0YxxcXWaBSTWveO5mIKC+bRXExxrkVPg6O5eDQXUzlXEmpBj+biEZeLAVDO2GmCaQxeAAAAAElFTkSuQmCC' />"}],"value":"[1 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x73ccfa9c \"comportexviz.server.notebook$eval28204$fn$reify__28208@73ccfa9c\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>2</span>","value":"2"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAAwUlEQVRYR2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGRkZ4fIg9TTnD3oHvn37tgEWQkxMTAc44+PhfAYJiQ1f29sFkOX//fvnQE8+46APwUHvwNEopjTXj0Yxhbl+NBdTXNOM5uLRXDxaFxNovo0W1NQuqCkutyjNtej60aN40DkQvaCmd4uZkH0YdfGgC8HRKKY0F/9nYID3iwdFPxg9Fw96B/7w8zsA6+cOhn4weq5mHPQhOOgdONijGAA75+RpCucPHwAAAABJRU5ErkJggg==' />"}],"value":"[2 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x16b9738c \"comportexviz.server.notebook$eval28204$fn$reify__28208@16b9738c\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>3</span>","value":"3"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAAmUlEQVRYR2N89+7dfwYo+P//fyMjI2P9oOIPege+ffu2ARZiTExMB/79++cwmPiMgz4EB70Dh1wUCwkLw3PxTz+/g98WLLAf0FyNHsWDzoHoUcwZHw/P1QwSEhu+trcLDGSuxsjFgy4ER6OYwoJ/NIopbnyM5mJKm2+juXg0Fw90Y2E0F4/m4tHmFoF++GhBPVpQjxbU+IdaAOsw6mlJJKNuAAAAAElFTkSuQmCC' />"}],"value":"[3 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3238412b \"comportexviz.server.notebook$eval28204$fn$reify__28208@3238412b\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>4</span>","value":"4"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAAy0lEQVRYR2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd/DbggX2+NSj66eYP+gd+Pbt2wZYiDAxMR349++fAzKfMz4eLs8gIbHha3u7AD716Pop5TMO+hAc9A4cjWK0UoHkXD0axWilAqm5ejQXk5zm0NPskM/FFIcArXPxgDuQUBSTmuuorZ5gLh7wECRUUA+4A4dcFA94Cxo916NH8aBzIHoUD3QLGr0UwMjFgy4ER6OY2s2t0SgmtfEwmotJDbHRgprCXIteUAMAEl7qaRArg+kAAAAASUVORK5CYII=' />"}],"value":"[4 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xb066f04 \"comportexviz.server.notebook$eval28204$fn$reify__28208@b066f04\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>5</span>","value":"5"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAAtElEQVRYR2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd/DbggX2+NSj66eYP+gd+Pbt2wZYiDAxMR349++fAzKfMz4eLs8gIbHha3u7AD716Pop5TMO+hAc9A4cjWK0UoHkXD0axWilAqm5ejQXk5zm0NPsaC4ezcWjdTFq8w4jU40W1KMF9UA3+UcL6tGCerAX1BS3RmgdxQPuQEK5mNQWMLXVE2xRD3gIEqqLB9yBgz2KATl96mlAd8StAAAAAElFTkSuQmCC' />"}],"value":"[5 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1f8a6ce2 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1f8a6ce2\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>6</span>","value":"6"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAAyElEQVRYR2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef38FvCxbYk6KfkPkY8oPegW/fvm2AhQATE9OBf//+OeDjc8bHw9UzSEhs+NreLkCKfkLmo8szDvoQHPQOHHZRTHIuJFAqEDSP1CgmaCClDkLXT2oUk5oLKVVPci6mewiORjGBgp9QEiAYxXSve9EzCaEoHnAHEsrF9K57Sa6LBzwER6N4NBfTu4lPal08mosJtX5Gc/FoLh7Nxfj74aPNLYq7CIO9uQUA/sfqaRUNkBgAAAAASUVORK5CYII=' />"}],"value":"[6 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x42f2498c \"comportexviz.server.notebook$eval28204$fn$reify__28208@42f2498c\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>7</span>","value":"7"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAAyklEQVRYR2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef38FvCxbYk6KfkPkY8oPegW/fvm2AhQATE9OBf//+OeDjc8bHw9UzSEhs+NreLkCKfkLmo8szDvoQHPQOHI1iAqUCwVw9GsUESgVCuXo0FxNMY4TS6GguJhRChORHc/FoLkZLI3RvDw67XExxuUYo16LLk5qL6e5AUqOYUN1JbXmS62K6h+BoFNO6oKZ7uUdqLh5wBxLKxfTuB5PcLx7wECSUiwfcgYM9igEl9eppmzts1QAAAABJRU5ErkJggg==' />"}],"value":"[7 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x323379e0 \"comportexviz.server.notebook$eval28204$fn$reify__28208@323379e0\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>8</span>","value":"8"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAAnklEQVRYR2N89+7dfwYo+P//fyMjI2P9oOIPege+ffu2ARZiTExMB/79++cwmPiMgz4EB70Dh10UCwkLw3P5Tz+/g98WLLCnaa4nNYrp7kBSo5gzPh6e6xkkJDZ8bW8XoGWuJzkX0z0ER6OYwoJ/NIopbnyM5mJKm2+juXg0F6OlIbrXJKO5eDQXo+XC0eYWepIYLahHC+rRghr/UAsA6z/qae5kIbEAAAAASUVORK5CYII=' />"}],"value":"[8 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xa754eed \"comportexviz.server.notebook$eval28204$fn$reify__28208@a754eed\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>9</span>","value":"9"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAAw0lEQVRYR2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fge/LVhgT4p+gvYNege+ffu2AeZjJiamA//+/XMghc8ZHw/XzyAhseFre7sAKfoJ2cc46ENw0DtwNIpJLBUwcvVoFJNYKqDn6tFcTLCmIJRGh30upjiECIUgpbmY5g6kNIoJ1aWUylOci2kegqNRTO+CmuYtaPRcTWoU092BpOZiWregKa6L6R6Co1E8movRsj3d0+BoLibUnCIkP5qLKczFABJt6mnyoT8cAAAAAElFTkSuQmCC' />"}],"value":"[9 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4654de3d \"comportexviz.server.notebook$eval28204$fn$reify__28208@4654de3d\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>10</span>","value":"10"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAAtElEQVRYR2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fge/LVhgT4p+gvYNege+ffu2AeZjJiamA//+/XMghc8ZHw/XzyAhseFre7sAKfoJ2cc46ENw0DtwNIpJLBUwcvVoFJNYKqDn6tFcTLCmIJRGR3MxoRAiJD+ai0dzMYE0QvP24GguJpRLCcmP5uLRXDyai/F30ihublHcWqF1Lqa5AyktqAn1aymVH41iipPAYI9iADmM6mlvsAbrAAAAAElFTkSuQmCC' />"}],"value":"[10 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x16ba5152 \"comportexviz.server.notebook$eval28204$fn$reify__28208@16ba5152\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>11</span>","value":"11"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAAtUlEQVRYR2N89+7dfwYo+P//fyMjI2M9JXwhYWG4/p9+fge/LVhgT4l5jIPegW/fvm2A+ZCJienAv3//HCjhc8bHw81jkJDY8LW9XYAS8wZ/CI5G8WCLYkpLAQz91I5iqjuQ2rmY0lIAXT/VczHVQ3A0iiks+CmOYqrXvehtAUqjmOYOpDQXU7vupXoupnkIjkbxaC4m0AWgeRoczcUUdsIGf00yGsWjUUygoB2ti0frYgrrYgD+1upp9ZX7agAAAABJRU5ErkJggg==' />"}],"value":"[11 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3fefc61c \"comportexviz.server.notebook$eval28204$fn$reify__28208@3fefc61c\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>12</span>","value":"12"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAAtElEQVRYR2N89+7dfwYo+P//fyMjI2M9JXwhYWG4/p9+fge/LVhgT4l5jIPegW/fvm2A+ZCJienAv3//HCjhc8bHw81jkJDY8LW9XYAS8wZ/CI5G8WgUk1jsUL2YGc3FowU1iQX3aEE96BoLo7l4NBcPt1xMaSbD0E/t5hbVHUjtXExpixxdP9Vb1FQPwdEoJrEUoHoUU72Bit5ApjSKae5ASnMxtZtXo1FM8VgMehoc7FEMACYE6mmrly65AAAAAElFTkSuQmCC' />"}],"value":"[12 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1f4de52a \"comportexviz.server.notebook$eval28204$fn$reify__28208@1f4de52a\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>13</span>","value":"13"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAAoklEQVRYR2N89+7dfwYo+P//fyMjI2P9oOIPege+ffu2ARZiTExMB/79++cwmPiM1A5BIWFheBL56ed38NuCBfYUJZlB70BqRzFnfDw8yTBISGz42t4uQEmSGY1iitPgaBSP5mISC+7RXExpY4HqNcloLh7NxaO5mEALfrS5NdrcIrGXN1pQD7qqbjQXD7ZcTGkSwdBP7SimugOp3dyi9tAJANJt6mkd3lL7AAAAAElFTkSuQmCC' />"}],"value":"[13 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x7f8865c8 \"comportexviz.server.notebook$eval28204$fn$reify__28208@7f8865c8\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>14</span>","value":"14"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAAt0lEQVRYR2N89+7dfwYo+P//fyMjI2M9NflCwsJw8376+R38tmCBPUnmD3oHvn37tgHmIyYmpgP//v1zoCafMz4ebj6DhMSGr+3tAqSYzzjoQ3DQO3A0ikdzMYW5ejQXU1qzUFyTjPhcTGkM0LwmodiBtI5iSuv2wR+CtK6LR6OY1BCguNxDb9FTO4qp7kBq52JK6170XE/1XEz1EByNYhI7SaNRTPU0OJqLSR3qGC2oKRzqQM/FABJ86mnZzniaAAAAAElFTkSuQmCC' />"}],"value":"[14 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3bfc8baa \"comportexviz.server.notebook$eval28204$fn$reify__28208@3bfc8baa\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>15</span>","value":"15"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAAnklEQVRYR2N89+7dfwYo+P//fyMjI2M9NflCwsJw8376+R38tmCBPUnmD3oHvn37tgHmIyYmpgP//v1zoCafMz4ebj6DhMSGr+3tAqSYzzjoQ3DQO3A0ikdzMYW5ejQXU1qzUFyTjObi0Vw8mosJNO9G6+LR5haFXYDRgnrAq7rRXDzYczGlSYTmvTqKHUjr5hal/ezBH4K0zsXDPooBOZvqaSRV4OQAAAAASUVORK5CYII=' />"}],"value":"[15 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x7a6265ef \"comportexviz.server.notebook$eval28204$fn$reify__28208@7a6265ef\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>16</span>","value":"16"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAAuUlEQVRYR2N89+7dfwYo+P//fyMjI2M9LflCwsJw83/6+R38tmCBPV77Br0D37592wDzARMT04F///450JLPGR8Pt49BQmLD1/Z2AXz2MQ76EBz0DhyNYkpLBXpHMcnFGL2jmNRSgu65mOQQHI1iCgt+mkcxyXUveq6ndRRT7EBa52JS6170XD4axaNRTHLBipYLKQ7B0VxMYgt6NBejp1mK0+BoQU1qP3i0LkZrXo3WxQNekwz2XAwA/uXqaXA6iS4AAAAASUVORK5CYII=' />"}],"value":"[16 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xaf6ab3c \"comportexviz.server.notebook$eval28204$fn$reify__28208@af6ab3c\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>17</span>","value":"17"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAAuElEQVRYR2N89+7dfwYo+P//fyMjI2M9LflCwsJw83/6+R38tmCBPV77Br0D37592wDzARMT04F///450JLPGR8Pt49BQmLD1/Z2AXz2MQ76EBz0DhyNYrRSYTQXU5rLR3MxpTUNyWlwNBeP5mIq182juXg0F6MlKbo3t0iOAXo3t0h2IL0LalLr8tEoJjlK0XuVIz6KSW69oIcgrXMxxQ6kdRSTWvei53Ka52KKQ3A0ikkc6kCPYgAmE+pp071zRwAAAABJRU5ErkJggg==' />"}],"value":"[17 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x36ec298f \"comportexviz.server.notebook$eval28204$fn$reify__28208@36ec298f\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>18</span>","value":"18"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAAoUlEQVRYR2N89+7dfwYo+P//fyMjI2P9oOIPege+ffu2ARZiTExMB/79++cwmPiMgz4EB70DR6MYrVQQEhaGlxI//fwOfluwwB5vqUHvKCbZgfSOYs74eHipwSAhseFre7sAvlKD7rmY5BAcjWK0gn80iiltTJCcBkdz8WhBTeXm12guHs3FaElqtC5GTxKjBTXdM8loc2u0uUXlsZphl4sB613qaWT/vWAAAAAASUVORK5CYII=' />"}],"value":"[18 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x69124523 \"comportexviz.server.notebook$eval28204$fn$reify__28208@69124523\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>19</span>","value":"19"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAAuElEQVRYR2N89+7dfwYo+P//fyMjI2M9PflCwsJw+376+R38tmCBPYr9g96Bb9++bYC5mImJ6cC/f/8c6MnnjI+H288gIbHha3u7ALL9jIM+BAe9A0ejmECpMJqLKc31o7mY0pqHYBoczcWU1u0DXVATTCIDHcWESoEBr4sJhuBoFFPYfKN7FBMs99BzPb2jmGQH0jsXE6p70XP1aBSjFyOjUUywYKW0BT2aiwkUzKO5mO5pcLAX1AASi+ppnC6yVAAAAABJRU5ErkJggg==' />"}],"value":"[19 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xb608f3 \"comportexviz.server.notebook$eval28204$fn$reify__28208@b608f3\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>20</span>","value":"20"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAAoElEQVRYR2N89+7dfwYo+P//fyMjI2M9PflCwsJw+376+R38tmCBPYr9g96Bb9++bYC5mImJ6cC/f/8c6MnnjI+H288gIbHha3u7ALL9jIM+BAe9A0ejmECpMJqLKc31o7mY0pqHYBoczcWjuZjGdfNoLh7NxYO+NTPoHThaUI8W1CO9oCZYjg50LibowIHOxYRa5APeLyYYgqNRTGEpAAA5quppG5N8zAAAAABJRU5ErkJggg==' />"}],"value":"[20 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3fa6ec92 \"comportexviz.server.notebook$eval28204$fn$reify__28208@3fa6ec92\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>21</span>","value":"21"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAAu0lEQVRYR2N89+7dfwYo+P//fyMjI2P9QPKFhIXh9v/08zvIOOgd+Pbt2wZYiDExMR349++fw0DyOePj4e5hkJDYMPhDcDSKSUwygz6KMUqRwRbFGA4cbLkYvRQZdLl4NIqpXdAPeBSj173fFiywR2kLDHQuJujAgc7F6AXz1/Z2AeS2wGgUE2pfjkYxoRAiJE8wBEdzMYHm1mguJpTGCMkTTIOjBTWBfjbBEBzNxaO5mMZjNQTT4GDPxQCKzuTrR25TVgAAAABJRU5ErkJggg==' />"}],"value":"[21 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x68d10ab8 \"comportexviz.server.notebook$eval28204$fn$reify__28208@68d10ab8\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>22</span>","value":"22"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAAtElEQVRYR2N89+7dfwYo+P//fyMjI2P9QPKFhIXh9v/08zvIOOgd+Pbt2wZYiDExMR349++fw0DyOePj4e5hkJDYMPhDcDSKSUwyo1FMaTGFUcyM5mISC/7RgprSgn40F4/mYkpDgFL9o7l42OdijCQy2JpbGA4cbHUxehIZdC3q0SimNBcPuihGL5i/LVhgj9IvH+hcTNCBA52L0ZtXX9vbBZD75QOeiwmG4GgUE+gXE4piAPmQ2e95odniAAAAAElFTkSuQmCC' />"}],"value":"[22 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x139161dc \"comportexviz.server.notebook$eval28204$fn$reify__28208@139161dc\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>23</span>","value":"23"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAAkUlEQVRYR2N89+7dfwYo+P//fyMjI2P9oOIPege+ffu2ARZiTExMB/79++cwmPiMgz4EB70DR6OYxFJBSFgYXor89PM7OOjSIIYDB1sUc8bHw0sVBgmJDYM/BAdbLh6NYkrr8tFcTGldPpqLqZ4GRwvq0bqYzi3s0Vw8mospDQFK9Y/WxaN1MaVpiFL9Q65FDQBlbtRxnRwlDAAAAABJRU5ErkJggg==' />"}],"value":"[23 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x23d002cd \"comportexviz.server.notebook$eval28204$fn$reify__28208@23d002cd\"]]"}],"value":"([0 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x51bd1747 \"comportexviz.server.notebook$eval28204$fn$reify__28208@51bd1747\"]] [1 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x73ccfa9c \"comportexviz.server.notebook$eval28204$fn$reify__28208@73ccfa9c\"]] [2 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x16b9738c \"comportexviz.server.notebook$eval28204$fn$reify__28208@16b9738c\"]] [3 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3238412b \"comportexviz.server.notebook$eval28204$fn$reify__28208@3238412b\"]] [4 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xb066f04 \"comportexviz.server.notebook$eval28204$fn$reify__28208@b066f04\"]] [5 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1f8a6ce2 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1f8a6ce2\"]] [6 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x42f2498c \"comportexviz.server.notebook$eval28204$fn$reify__28208@42f2498c\"]] [7 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x323379e0 \"comportexviz.server.notebook$eval28204$fn$reify__28208@323379e0\"]] [8 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xa754eed \"comportexviz.server.notebook$eval28204$fn$reify__28208@a754eed\"]] [9 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4654de3d \"comportexviz.server.notebook$eval28204$fn$reify__28208@4654de3d\"]] [10 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x16ba5152 \"comportexviz.server.notebook$eval28204$fn$reify__28208@16ba5152\"]] [11 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3fefc61c \"comportexviz.server.notebook$eval28204$fn$reify__28208@3fefc61c\"]] [12 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1f4de52a \"comportexviz.server.notebook$eval28204$fn$reify__28208@1f4de52a\"]] [13 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x7f8865c8 \"comportexviz.server.notebook$eval28204$fn$reify__28208@7f8865c8\"]] [14 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3bfc8baa \"comportexviz.server.notebook$eval28204$fn$reify__28208@3bfc8baa\"]] [15 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x7a6265ef \"comportexviz.server.notebook$eval28204$fn$reify__28208@7a6265ef\"]] [16 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xaf6ab3c \"comportexviz.server.notebook$eval28204$fn$reify__28208@af6ab3c\"]] [17 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x36ec298f \"comportexviz.server.notebook$eval28204$fn$reify__28208@36ec298f\"]] [18 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x69124523 \"comportexviz.server.notebook$eval28204$fn$reify__28208@69124523\"]] [19 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xb608f3 \"comportexviz.server.notebook$eval28204$fn$reify__28208@b608f3\"]] [20 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3fa6ec92 \"comportexviz.server.notebook$eval28204$fn$reify__28208@3fa6ec92\"]] [21 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x68d10ab8 \"comportexviz.server.notebook$eval28204$fn$reify__28208@68d10ab8\"]] [22 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x139161dc \"comportexviz.server.notebook$eval28204$fn$reify__28208@139161dc\"]] [23 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x23d002cd \"comportexviz.server.notebook$eval28204$fn$reify__28208@23d002cd\"]])"}
;; <=

;; @@
(show-example-linear-encoder 0 24 0.5 10)
;; @@
;; ->
;;; Showing (e/linear-encoder [250] 20 [0 24])
;;; 
;; <-
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>0</span>","value":"0"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAAAwCAYAAAChS3wfAAAA9klEQVRoQ+2ZvRGCQBSE35GYypDQjVZwY0dqRBuU4FABdkMDOppwJvL36OD2I9she9x3u/sIySzZ//nG+Dx03Wmt320765TSPYRwnd5noeUH8Imxn76o1fXDhuGy1q+mOU66KIp+HMdzTjrInwD5AYAALiBugyAAAiBAEJLK/r7L+CCUffnxA/AukHv2911m1wXkTgAIuDoMArlvgHABt9LDBVwXwAXU7gAQAIFtHZa7A0AABEBgsxGiC2SeA6qqus3/Ps1Mbh+wG4CaC5RluSDPCQABCyAgFoRwAW+DIAACWl2AIEQQogtsyxAugAvgAlIbIV+Hf8M27OQkehOwAAAAAElFTkSuQmCC' />"}],"value":"[0 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x32e22224 \"comportexviz.server.notebook$eval28204$fn$reify__28208@32e22224\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>1</span>","value":"1"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAAAwCAYAAAChS3wfAAABC0lEQVRoQ+2aMRKCQAxFszSOnbgNt9ET7HgDj6JWXIMjOJwAb0MDnY42rMMoC4MFtZtHl8k2hHz+T/6atm29fB/v/WVr7WmIX87dHkWxm+aNMSHfn//7WH0BmqY5D184SZJqfTyGWLLses/zzTTfdd0+ptio7wD1BQACsIByGgQCQAAIIISYBTQPQ7AALAALwAKwACzwqQAbIZaiCrfCCCGEEEIIIYQQQggpFkJeJLjDvRu8KsvgBsfoDltrR+9TRIz6Ajydq4afYO8GS10fpnFs7nCapuP70gFAQAwQgAWU0yAQAAJAACGk6Wbo4iwQ+9XYnwLMWSA27T+/2bo4C6jrgPk4rK4A2iHwBtAuCvNZXgfXAAAAAElFTkSuQmCC' />"}],"value":"[1 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x2ea4da36 \"comportexviz.server.notebook$eval28204$fn$reify__28208@2ea4da36\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>2</span>","value":"2"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAAAwCAYAAAChS3wfAAABI0lEQVRoQ+1aMRKCMBBMaBw7MQ2/0Rcw/oCnqBXf4AkOL8Df0ECnMzbERiIcM6FlbpfuhqS53GZ372L7vvfm93nv79ba6zQ+OhfiT54/31V1iq2X+zcfwyeg67rbeKJJkjTDMJyn8b4own+TZY9XWR5i6+X+rccWvgLgE0AIkAXAaZAQIAQIAQqhmPbf1XXQ/hBeQJof9QmQLCC1v2nby1ghEF4ArgKkF4BLACEghBBcBRACoiNEFhBNUPU6gBAgBOZuUP0cQMw9Fl1huASsCSFtcwA5p2AFrLGAulmgvAMIgRUvoK0CnHP/WacxBu4OWCQADQJpmjahw8UKIASMJQTIAlhmiCxAFqAQmitBsgBZgCwweyCh/WUovQDtsKBBdBb4AvYYHPPqzBZYAAAAAElFTkSuQmCC' />"}],"value":"[2 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3c3f8412 \"comportexviz.server.notebook$eval28204$fn$reify__28208@3c3f8412\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>3</span>","value":"3"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAAAwCAYAAAChS3wfAAABKUlEQVRoQ+2aTQ6CMBCFp2yMO7EbbqMnIN6Ao6grrsERDCfA27CBnSZuKCGxFUYj6Y4yj92kaRdD33zzU9W2raH3Z4y5KqXOY3uvtbNfaXrflOXBrg/2syic/Ws/P29xtngHNE1zsX80iqKq67rj2N5mmVunJLlRXZ/s+mA/8nz3bz8/b2m2En8DxDsAEvCkQPBRn1HOOwaszgG+Egg96nMK4Qb4UgASCDzxgQRAgWmxp0ABJEJ+/QBQABQIvAMECoACk44XagHUAp5NUZTDoc8BQAFQABSYjAJBAVBAVjWotf7MOolIXCb45QBpHaE4jis33cYNgARIXFMUEgAFGAZBAWFzASRCPAZAAkwCa38ZOisBcQ7gElj7y9DZREjcDeAdIXEOkC6BHs+PHPO4lYtcAAAAAElFTkSuQmCC' />"}],"value":"[3 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x65f83939 \"comportexviz.server.notebook$eval28204$fn$reify__28208@65f83939\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>4</span>","value":"4"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAAAwCAYAAAChS3wfAAABIklEQVRoQ+2asQ2DMBBFzzRRuhA3bJNMgLIBoySpWIMRIiYg29BAl0hpME1wJCLZOiEhcffpLGMX5/t6d/42fd87+n7Oubsx5hoaH6318588f76r6sRZH9t/9Xn1Aei67jadYJIkzTAM59B4XxT+f8qyx6ssD5z1sf3XnjfqM0B9ACABUEA5BiEBSAASQCHEqeXRC0hrhkABUAAUAAVAAc1XYqAAKAAKgAIcH2BX194HUOELzJsfcQGIUWDuA1DbXqaMUeELiM+AmC8gPgCQQKQQkpYB1tqft0lEUW9QfAC0SSBN08ZTDBkACZCBBJgU2PqboMUUEBcArgS2/iZoMQXEZQC3FxAXAEgAFAhfikqzw/8wCAlAApBA0BeQ9jh6XgiNqQYc89BQU70AAAAASUVORK5CYII=' />"}],"value":"[4 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x46a8d4a0 \"comportexviz.server.notebook$eval28204$fn$reify__28208@46a8d4a0\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>5</span>","value":"5"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAAAwCAYAAAChS3wfAAABKElEQVRoQ+2aMQ7CMAxFnS6IjZKlt4ETVNygRwGmXqNHQD1BuU2XdgOJpWGhQZjJShfXv1ukRJHcfD/7J24cx0CfL4Rwdc6dJeO993H+qyzvz6Y5SNZL91t8vvkADMNwmf9YlmXdNE1HyXhbVXE9FcXtUdc7yXrpfkvPd+ZPgPkAQAJCCvCsv2nbmPVNUGB1AZBKgGd96vvTnPVNUGB1J0BKgdUFABIABWTNECTAan9QAIWQ8koQFEikgDoHiDteqYWQ+gCkSkCbA8QdJbEjpN4EhQSY6w0JgAJpvQAooOweABQABUCBn6s/c4WQ9/57l0lECIC1QijP8y7eY+AEQALkIAFjvQAowDEICUACttrhvxwACUACkIDopaj2l6HoBdAOMz/AOgXePN0c80ogR7QAAAAASUVORK5CYII=' />"}],"value":"[5 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4ed5c329 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4ed5c329\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>6</span>","value":"6"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAAAwCAYAAAChS3wfAAABGUlEQVRoQ+2aQQ6CMBBFWzbGndgNt9ETEG/gUdQV1+AIhhPgbdjAThM31BgFE1xNyqbMY0OatGkync+b8sd2XefN9/HeX6y1J8l469w4/5nnt0dZ7iTrpfvNPl99ANq2PQ8nliRJ3ff9XjJeH4/jepNl13tRbCTrpfvNPd+qzwD1AUACUEA5BpEAEkACFEIhtTt3gdgvQ1AACkABKAAFNP8SgwJQAApAASgABT4RwBfAGlPoDVIIKSuEnHM/L9MYo84b/AuANgmkaVoPdc/7TQZoc4eRQKgEpj7AqqrGnqAYeoSCKbC4AEgpMO0JMk1zGL+qEfQIBVNgcRkgpcDiAoAEhHeBxWUAEhB2ikKBST8AhVDslaB2CrwAW/Qc88joLwcAAAAASUVORK5CYII=' />"}],"value":"[6 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xe8ac1f5 \"comportexviz.server.notebook$eval28204$fn$reify__28208@e8ac1f5\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>7</span>","value":"7"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAAAwCAYAAAChS3wfAAABH0lEQVRoQ+2aMRKCMBBFExrHTkzDbfQEjDfgKGrFNTiCwwnwNjTQ6YwNsREcY0GxFOzk0VEkk1n28/7uxvZ9783n8d5frbVnyfveuWn9K8/vz6o6SPaTnmd2ffQB6LruMn6hJEmaYRiOkvdtUUz7mSy7PcpyJ9lPep659Tb6DIg+AEgACkSOQSSABJAARmhJr04toK0YggJQAApAAQkFwr/+pq6nDlAUHSH1AZBSIOwAmbY9jRkVRUdIfQZIO0LqA4AEhEZIfQYgAeFcAAoE1R9GSJsThAILU2Dt02Dn3Hd2aYwRzwa1tcD+ArC0BNY+DU7TtJlqFTIACRiLBKCArCkKBZTdCYICGKHACUIBKAAFRKMxbTdDoQAUgAI/5fAb77wc89vHRV4AAAAASUVORK5CYII=' />"}],"value":"[7 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x2a9f00ec \"comportexviz.server.notebook$eval28204$fn$reify__28208@2a9f00ec\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>8</span>","value":"8"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAAAwCAYAAAChS3wfAAABGElEQVRoQ+2ZQQ6CMBBFWzbGndgNt9ETEG/AUdQV1+AIhhPgbdjAThM31JBIjXVZa5rMYzeLDk2Zz/vT0eM4WvV6rLVnrfUxJN4a49Y/yvK6atvdkm+O703j4l+8L3S/WvwBDMNwWr5QlmXdNE37kHhdVS6fKoqL6vvDkm+Ob3W9Cckfuj9/PRWABJAAFJCNQSiAERL+D0ACkSWQnPf3e53YRij5A4gtgdS8/997geQrAAlE7gWQQGo3QFDAu/KDAhihuL1A8hhEAkgACUSdC2CEMEKJzQJ9JwgFhFHAGPOeXSqlos8GUzNCXwcgTQJ5nnduWk0FIAGlkQAUkNULQAEogBH6dIJQAApAAVE3QvQCtMMeBqVT4AmDkxzzDub1dAAAAABJRU5ErkJggg==' />"}],"value":"[8 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x2a92dccb \"comportexviz.server.notebook$eval28204$fn$reify__28208@2a92dccb\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>9</span>","value":"9"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAAAwCAYAAAChS3wfAAABGElEQVRoQ+2asQ3CMBBF7TSIjuAm28AEERswClBljYyAMkHYJk3SgUQTU0CMFMpcokN+6dycrIu/353/2a7rvPl83vuLtfYkud46F+I98/z2KMudZPzJ+40+AW3bnoc/kiRJ3ff9XnK9Ph5DfJNl13tRbCTjT92vjf4ERJ8AJAAFIscgEkACSIBCaM7anF5AezMEBaAAFIACUCDmJzEoAAWgABSAAlDgnQF8AayxCL1BCiHhQmj8ArSqquAGa3CHnXNfr9IYI+4N/l0CpCUwdoNN0xyGOkODO5ymaR32wwlAAsYiASgg2w5DgdFMEBSgEFJeCUKBmSmgbTJ08V5AfQLmloC2ydDFewH1J0B6UlT7RMjPHRC7BF6iqhzzbGVDUAAAAABJRU5ErkJggg==' />"}],"value":"[9 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4abe9e3b \"comportexviz.server.notebook$eval28204$fn$reify__28208@4abe9e3b\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>10</span>","value":"10"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAAAwCAYAAAChS3wfAAABJklEQVRoQ+2aMQ6CQBBFZ2mMnbgNt9ETEG/AUdSKa3AEwwnwNjTQaWLD2sgax4Jqlknm09GwYZjP+7N/3TiOgT5XCOHqnDtL3u+9j89/leX92TQHyfUW38d8AYZhuMxfIMuybpqmo+T9tqrielQUt0dd7yTXW3ofZ74DzBcAEgAFjGMQEoAEIAEYIUkvzr3/pm2j9zcxC6gvgDQFuPenvj/NHWdiFlDfAdKzgPoCQALCRkh9B0ACwjtCoADbBIUR0uYEQYHEFFg9B+C5R2oKqCtAagmsnQPwnEA8F1AXhUECLPuEBIxRwHv/zSaJyNw/4K8A1iSQ53kXd6TQAZAAOUgAFJCNxrQ5QVAAFIAR+nWCoAAoAAqIHpDQdjIUswDGYYZB6xR4AzaBHPNISyHTAAAAAElFTkSuQmCC' />"}],"value":"[10 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x12636bf0 \"comportexviz.server.notebook$eval28204$fn$reify__28208@12636bf0\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>11</span>","value":"11"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAAAwCAYAAAChS3wfAAABHklEQVRoQ+2aMQ6DMAxFHZaqW2kWbtOeAPUGHKXtxDU4QsUJ6G1YYGulLgQhNamUzrEj+bN5wWD8/PMTzDzPjr6Xc+5ujLmmjI/Whvt/6vq56/uTz7fF764LMcfzGPUFmKbp5r9AURTDsiznlPG+aUI+qqoHjePF59viV9seUuaP3w8dAASAgKwKsE/9WOWkERAvgLQKcE/97FRAvAOAgLAKAAHutT9UIDJ7UAFhOwwVkJ4BQAAIKDdDQAAIAAHRTVF4Ael1AOww7LDyIQgEgIAuBKy1v7NJIhI/G+TeD/grgDYvUJblEE6j0QFAgAwQUGaHoQKxDAIBIKB8JQgEEiOQ25+h7CqQfQFSI5Dbn6HsXiD7Dki9I5R9AbQjsAIP+Bzz8xe2lgAAAABJRU5ErkJggg==' />"}],"value":"[11 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x40298130 \"comportexviz.server.notebook$eval28204$fn$reify__28208@40298130\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>12</span>","value":"12"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAAAwCAYAAAChS3wfAAABFklEQVRoQ+2asQ3CMBBFz2kQHcFNtoEJIjbIKECVNTICygRhmzRJBxJNnIYYyfT/LN135+qkr/t+Pn+7eZ6DfFcI4e6cuyL3R+9jvU9dP99dd0LWd+YFmKbptileFMWwLMsZud83TawvVfV4te0BWZ8dQAvQAqSAbQySArwIGT8DaAFagBbgLAAdP5Nxm+Ow9nsAKUAKkAKkAClg+VWYFABTIL347Po+5gAmcoHsBEBbIM0BZBwv2yFsIhfIrgPQuUB2AtACxijgvf9lkSICzwa1LfAngDULlGU5ROqwA2gBcbSAMgXQf4Kyo4C6ANoWQP8Jyo4C6h2gPQuoC0ALkALYXEA7C8xuFiAFwJ+jeRFK3wOsU2AF6WAc80v5ItEAAAAASUVORK5CYII=' />"}],"value":"[12 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x7ce1797a \"comportexviz.server.notebook$eval28204$fn$reify__28208@7ce1797a\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>13</span>","value":"13"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAAAwCAYAAAChS3wfAAABHklEQVRoQ+2aMQ7CMAxFnSyIjZClt4ETVNygRwGmXqNHQD1BuU2XdgOJpWGhQZg5cST/bplqWfl+9nfMPM+BPl8I4WqMOUue997H/7/q+v7sukPSeNQnYJqmy5pha+2wLMtR8rxtmhgPVdXt0ba7lPEY9TdAfQIgAWEK8Kq/6ftY9VVQQDwB0hLgVZ/G8bRWfRUUEL8B0hQQTwAkAArIDkOQAOv9QQE0Qpk7QVCgMAokd4C441VaI5Q9AaVJILUDxB0vcUcouwkKCTDXGxIABcqaBUCBxHsAUAAUAAV+Vn/qGiHv/Xf3SERIgLZGyDk3xL0DbgAkQAYSUDYLgAIcg5AAJKBrHP6rAZAAJAAJiL4Uzf0yFLMAxmHmB2inwBt9Nxzz+v/KrQAAAABJRU5ErkJggg==' />"}],"value":"[13 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x42906f9b \"comportexviz.server.notebook$eval28204$fn$reify__28208@42906f9b\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>14</span>","value":"14"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAAAwCAYAAAChS3wfAAABEUlEQVRoQ+2aMQ6CQBBFd2mMnbgNt9ETEG/AUdSKa3AEwwnwNjTQaWLDGqOsCVYWy5DMoyFUbCbzeX/4Y/u+9+Zzee/P1tqj5PPWufD+R55f71W1i3oe9QXouu40VjhJkmYYhr3k87oownlMll1uZbmJeR6rvgPUFwAJQAHlGEQCSAAJYISieu0/ZwtmgbmHISgABaAAFIACmn+JQQEoAAWgABSAAu8KkAsQjSnMBjFCyoyQc+6bPRpj1GWDPwXQJoE0TZvR97zudIC2dBgJLE0C0xxgVddhJyjGjtDiKCBeAGkKTHeCTNsewlc6wo7Q4igg3gHSFBAvABIQngXEOwAJCG+KQoHJPgBGaG4nqJ0CT5xOHPNX3PkqAAAAAElFTkSuQmCC' />"}],"value":"[14 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4b25a0d2 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4b25a0d2\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>15</span>","value":"15"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAAAwCAYAAAChS3wfAAABGElEQVRoQ+2asQ3CMBBF7TSIjuAm28AEERtkFKDKGhkBZYKwTZqkA4kmpiFGmNa2TvJL58463c/79896nmerPp+19qq1Pks6741x93nV9f3ZdYeg98u+ANM0XdaKFkUxLMtylHTeNo27n6qq26NtdyHvp7PvgOwLgASgQOYYRAJIAAlghIJ668CzBLNA7GEICkABKAAFJFHA/+tv+t4lQFkkQskLII0CfgKkxvG0dmgWiVDyDpCWCCUvABIQZoSSdwASELYXgALe9IcRiu0EoYBwCoTeBhtjvrtGpZS43WDsCOyvANIlEHobXJbl4GYLOgAJKI0EoICsUBQKRH4TBAUwQp4ThAJQAAqIWo3FfhkKBaAAFPgZh98wJRzzassXFAAAAABJRU5ErkJggg==' />"}],"value":"[15 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x47098887 \"comportexviz.server.notebook$eval28204$fn$reify__28208@47098887\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>16</span>","value":"16"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAAAwCAYAAAChS3wfAAABF0lEQVRoQ+2ZsQ3CMBBF7TSIjuAm28AEERswClBljYyAMkHYJk3SgUQTo0jECNNhF7HuubvudPLz//+sh2Gw6n2stRet9WlJ9dYY18+zLG+rptnN/U31o65d/Vf/4gfQ9/15nmiWZe04jvsl1evj0fWniuKquu4w9zfV96rahPSrxd8A8QMAAVRAuAyCAAiAAEZoSd7fzyJ+Fgj2/n7WSc0IRR9AaioQ6v39rJNcFoh+A0AgsTgMAqEbIFTAW/mhAollAVQg9hsAAiCQVhyO/gaAAAiAQFIbIbJAoA8wxnz+GpVS4vYBPwOQpgJ5nrfud5kbAAJKg4AwI4QK+DIIAiAgKwtghDBCZIHvMIQKoAKogKiNkB+HX8PtHPMqIu83AAAAAElFTkSuQmCC' />"}],"value":"[16 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x30d403b5 \"comportexviz.server.notebook$eval28204$fn$reify__28208@30d403b5\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>17</span>","value":"17"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAAAwCAYAAAChS3wfAAABF0lEQVRoQ+2aMQ6CMBSGWxbjJnbhNnoC4g04ijpxDY5gOAHehgU2TVyog1KTutFq2vRj60Dy8tKf73/8T47jqMX70VqfpZTHkM9bpUx9j7K83ptm51Rv8g0YhuE0dzDLsm6apn3I53VVmXpFUVxudb1xqVcmfwOSbwASgAKJYxAJIAEkgBFy8tJ/nh2YBXwPQ1AACkABKAAFUv4lBgWgABSAAlAACrw6QC5ANJZgNogRCtwI2X+AVm1r0uAl6bBS6pMtCiGCzwZ/3oDQJWCnwaLvD7NvWZIO53nemfe5AUhASCQABcIeh6GAtRMEBTBCnp0gFIiMAq6bodHPAt4bEJsEXDdDo58FvN+A0DdFfW+EfH0DUpfAE+MEHPORcIgUAAAAAElFTkSuQmCC' />"}],"value":"[17 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x71794e83 \"comportexviz.server.notebook$eval28204$fn$reify__28208@71794e83\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>18</span>","value":"18"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAAAwCAYAAAChS3wfAAABI0lEQVRoQ+2asQ3CMBBFz2kQHcFNtoEJIjbIKECVNTICygRhmzROBxJNnIYYcVSWLaHT/XTuTqf7ef/8baZp8vT+vPdXY8xZ0nlvbaj3Vdf3Z9cdoupX3wDn3GXtWFEUwzzPR0nnbdOE+qmqbo+23cXUb9RPgPoGQAKggHIMQgKQACQAIxTlnf+8K3Dvv+n74P1V7ALZGyCNAtz70zie1glWsQtknwBpu0D2BkACwoxQ9gmABITdCIEC7BIURijVCYICwikQnQPwXUY6BZIbIF0CsTkAzz3E5QLJURgkwLJPSEAZBay1nyyRiNT9A34aoE0CZVkO4QYJEwAJkIEEQAFZ0ViqEwQFQAEYoW8nCAqAAqCAqAcSqS9DsQtgHWYY1E6BBXbbHPOrxynLAAAAAElFTkSuQmCC' />"}],"value":"[18 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x173c434e \"comportexviz.server.notebook$eval28204$fn$reify__28208@173c434e\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>19</span>","value":"19"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAAAwCAYAAAChS3wfAAABK0lEQVRoQ+2aQQ6CMBBFWzbGndgNt9ETEG/AUdQV1+AIhhPgbdjAThM3lJBITUZj08BmOp/d7KaTef2/M+i+7616f9baq9b6zCneG+PyfeX5fVPXhzn/KX5WlYt/nk98Abquu8wVS5KkGYbhyCneFoXLX2XZTbXtac5/ih9luft3Hi2+A8QXAAhEpgLeW5+qXGwIBBcgNgR8tz5VuehUILgDgEBkRggI+Lw/VIA89qACMEJxzQOCZRAIAAEgENVECEYIRsgzBaZOECoAFYAKQAU4rcLo6o6uxoKtMCZCmAjJugSNMZ9dolJK3FT4qwDSjFCapo3bHqMDgIDSQEDYWwAqQGUQCAAB4U4QCDBDYOmfoexVYPUCcENg6Z+h7N8Cq3cAt4nQ6gWQjsAIUFIc89rvQWYAAAAASUVORK5CYII=' />"}],"value":"[19 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1c7aac9e \"comportexviz.server.notebook$eval28204$fn$reify__28208@1c7aac9e\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>20</span>","value":"20"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAAAwCAYAAAChS3wfAAABGklEQVRoQ+2asQ2DMBBFzzRRuhA3bJNMgLIBoySpWIMRIiYg29BAl0hpME1wJEcCnVyd79O5477u693524zj6Oj7Oefuxpir5PPRWv//n7J8vpvmtFqPegGGYbgtCmVZ1k3TdJZ83leVr4eK4vGq68NaPUZ9B6gXABYABZRjEBaABWABDEKSZ/9wd8EuwF2GQAFQABQABUABzVdioAAoAAqAApIpEM7+u7b1OYCKXCBaAOkUCHMA6vvL0tEqcoHoDpCeC0QLAAsIH4S4HWCt/WWHRCQ+G4wWQJsF8jzvPCXQAbAAGVggMQpsvQlKngJsAVKzwNaboOQpwO6A1HYBtgCwACgg+1KUG4f/YRAWgAVgAdG5APdxdDgIzSnJHPOcUbpmAAAAAElFTkSuQmCC' />"}],"value":"[20 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xca3dd0f \"comportexviz.server.notebook$eval28204$fn$reify__28208@ca3dd0f\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>21</span>","value":"21"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAAAwCAYAAAChS3wfAAABG0lEQVRoQ+2asRHCMAxFZTccHcZNtoEJcmyQUYAqa2QELhOEbdIkHdzRxGmIIUrhcyt9dy6l89eTvmzGcQz0PSGEuzHmKul+9D7G8ynL57tpTqv41CdgGIbbkhFrbTdN01nSfV9VMT4qiserrg//8Rn1L0B9AiABYRTgVX/XtrHqq6BAdgKkSYBXfer7y1L1VVAg+wVIo0B2AiABUEDWMAQJsN4fFEAjlOgEQQHhFNg4QNzxkt4IJRMgXQLcAeKOlzhHKGmCQgLM9YYEQAHZswAowPYAoAAoAAqsVn/qGiHv/W9XSERIgLZGyDnXxT0BXgAkQAYSUDYLgAIcg5AAJKBrHN7UAEgAEoAERP0UTf0MxSyAcZj5AdopMAO9kRzz9h50SQAAAABJRU5ErkJggg==' />"}],"value":"[21 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3795974e \"comportexviz.server.notebook$eval28204$fn$reify__28208@3795974e\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>22</span>","value":"22"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAAAwCAYAAAChS3wfAAABC0lEQVRoQ+2aMRKCMBBFExqxc6ThNnoCxhtwFLWitabiCExOoLehgU5nbMBhlIzGgqHdfelS7p/8vJ/s2q7rBvNZwzCcrbVHSfttkvh6nll2e1TV7qc+9QK0bXuaFImi6Nr3/V7Sfp3nvj6TpvW9KDbf9Vn1J0C9AFgACijHIBbAAliAICQp+4dvGd4Cc48hKAAFoAAUgAKav8SgABSAAlAACkCBtwL0BWiNKewNEoSUBaHYOd/7HO89db3BPwG0WSAuy8uUezgBWMAYiwWEUSDsA6yc8zNB44yQeAosFkCaBcKZINM0B3/rp2ktngKLT4C0GaHFAmABKCDrUxQLBJOhUIAgNJMEtVPgBbeaHPMw5+esAAAAAElFTkSuQmCC' />"}],"value":"[22 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x42c694c1 \"comportexviz.server.notebook$eval28204$fn$reify__28208@42c694c1\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>23</span>","value":"23"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAAAwCAYAAAChS3wfAAAA+UlEQVRoQ+2ZsRHCMAxF7TSEljTZBibIsUFGATagzhScJ4Bt0jg9TcxxF/sOUbiXXjqXVvT9/pf8sizJbV9K6ea9v2g+H7qu3O89DC9vvgAxxmv+403TPNd1PWk+78ex3Nf1/YMOQAJIAArYxiAUwAgZfwOQABJAAmQBzfFXxnvisJwHQAHlFJAtvwvhmCVvYiJULYB2CcgJkJvnc+4AExOhagdonwhVC4AEoIDuLIAExB4ACmCEhBOEAsYo0IZQdp9fR6h+Nygp8FcAaxJop+lesgAdgAScRwJQwFYWgAJQACP06wShABSAAqq3w3IvQBYgDgsMWqfAB/1EAX0MwA66AAAAAElFTkSuQmCC' />"}],"value":"[23 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xb91ce5b \"comportexviz.server.notebook$eval28204$fn$reify__28208@b91ce5b\"]]"}],"value":"([0 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x32e22224 \"comportexviz.server.notebook$eval28204$fn$reify__28208@32e22224\"]] [1 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x2ea4da36 \"comportexviz.server.notebook$eval28204$fn$reify__28208@2ea4da36\"]] [2 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3c3f8412 \"comportexviz.server.notebook$eval28204$fn$reify__28208@3c3f8412\"]] [3 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x65f83939 \"comportexviz.server.notebook$eval28204$fn$reify__28208@65f83939\"]] [4 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x46a8d4a0 \"comportexviz.server.notebook$eval28204$fn$reify__28208@46a8d4a0\"]] [5 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4ed5c329 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4ed5c329\"]] [6 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xe8ac1f5 \"comportexviz.server.notebook$eval28204$fn$reify__28208@e8ac1f5\"]] [7 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x2a9f00ec \"comportexviz.server.notebook$eval28204$fn$reify__28208@2a9f00ec\"]] [8 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x2a92dccb \"comportexviz.server.notebook$eval28204$fn$reify__28208@2a92dccb\"]] [9 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4abe9e3b \"comportexviz.server.notebook$eval28204$fn$reify__28208@4abe9e3b\"]] [10 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x12636bf0 \"comportexviz.server.notebook$eval28204$fn$reify__28208@12636bf0\"]] [11 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x40298130 \"comportexviz.server.notebook$eval28204$fn$reify__28208@40298130\"]] [12 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x7ce1797a \"comportexviz.server.notebook$eval28204$fn$reify__28208@7ce1797a\"]] [13 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x42906f9b \"comportexviz.server.notebook$eval28204$fn$reify__28208@42906f9b\"]] [14 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4b25a0d2 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4b25a0d2\"]] [15 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x47098887 \"comportexviz.server.notebook$eval28204$fn$reify__28208@47098887\"]] [16 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x30d403b5 \"comportexviz.server.notebook$eval28204$fn$reify__28208@30d403b5\"]] [17 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x71794e83 \"comportexviz.server.notebook$eval28204$fn$reify__28208@71794e83\"]] [18 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x173c434e \"comportexviz.server.notebook$eval28204$fn$reify__28208@173c434e\"]] [19 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1c7aac9e \"comportexviz.server.notebook$eval28204$fn$reify__28208@1c7aac9e\"]] [20 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xca3dd0f \"comportexviz.server.notebook$eval28204$fn$reify__28208@ca3dd0f\"]] [21 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3795974e \"comportexviz.server.notebook$eval28204$fn$reify__28208@3795974e\"]] [22 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x42c694c1 \"comportexviz.server.notebook$eval28204$fn$reify__28208@42c694c1\"]] [23 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xb91ce5b \"comportexviz.server.notebook$eval28204$fn$reify__28208@b91ce5b\"]])"}
;; <=

;; **
;;; Play with both:
;; **

;; @@
(show-example-linear-encoder 0 24 3 10)
;; @@
;; ->
;;; Showing (e/linear-encoder [300] 70 [0 24])
;;; 
;; <-
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>0</span>","value":"0"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEgAAAAzCAYAAAA0CE5FAAABMUlEQVRoQ+2aMQ6CQBBFd2lsJTTcRk+w8UZqxTU4guEEeBsuoNGGtVAwLIkTMhJl8+imgGJ2Hv/P7FhvjDev5+7ceVVVG018Lcv+fe/90Vq77763yJgECQd4c67uTtjk+ck0zU4TX4pi3b2fJEndtu12ybGlgoQKIkEgpkJcjVh0qhWqrhax6BOkVbHYVCtUXRCTjCyIzaxiICb0ZtH/pEEMxDCKs45TQAzEQAzEXhn4yciWXuzzxJNejF5MeasCYiCmu5fDKGIUMYoYRYziMwP/eFWNUcQoYhRVKiVtn4BYgFiWZYd+/ccYQ4KkBNGLDXuxNE3fC2VU0HiHEsQEWR8lCMRAbNK4A8RATLd2jIpNrSAmisNeDRUTNv9BDMR04w0QA7EvVxAqhopNmjiGKvYALdgWrwz4YaYAAAAASUVORK5CYII=' />"}],"value":"[0 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x29901e4a \"comportexviz.server.notebook$eval28204$fn$reify__28208@29901e4a\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>1</span>","value":"1"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEgAAAAzCAYAAAA0CE5FAAABVElEQVRoQ+2bvQ3CMBBG7TSIjuAm28AEERswClCxRkZATADb0CQdCJoYIcgfBYdxkMB66U5BLo57+b67XHRRFFY9LmvtamzMooovabofbLcTl/iUZfXvb+dprevz/jImQcIfmOf5sqqQKIp2w/m8jlWSbNThMKvuvxMf1+tR+7yyLKf/HGsqSKggEgRiXoiDmKSyIAZiINY2ur0bUxADMRADsVYz3fszRpJ5erHXvSJGUaogVAwVQ8VQMVTsnoHbRPPXJpCoGCrm+VYFo4hR9Hsvh1HEKGIUMYoYxR82ilapervjk22O5+2P4LY7SJCgYuc03blsb0jbHsFtd1BB3QoyxjTrP0opTYKEBIFYtxeL47h55PRRQaGpVu+IBZ8gX8RCUy0QEyaIIOaaIBBDxZwmiiAGYn6b/aiYawX59mIYReFjFoyi8L1Y8BUEYl8ed4SO2BVjXDSvfooHewAAAABJRU5ErkJggg==' />"}],"value":"[1 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4cd94599 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4cd94599\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>2</span>","value":"2"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEgAAAAzCAYAAAA0CE5FAAABf0lEQVRoQ+2bMW7CQBBFd92gdDHbcJvkBCg34CiBimtwhIgTkNvQ2F0ipbFpYsPOIg+jdSRwHt1okS2N583/s177uq5b9/tr23bjvX+/jOch9PHPcvk52+9fuvVb4u/drv//tevL+91dTILigkgeUFVV664iiqI4NE3zehk/rVb9ulssPtzx+Nat3xJ/bbfPQ9eX97u32FNBSgWRIBCLWoYVYRATqp00aRADsXERyzWG0khOziiSIIGcNIq5xlAayckZRSpIVJBUMRIEYiZVS4xibgU9vGpJ4zg2YpNL0Ngq9uiqJWc1ELPOYvSgP1YxEFM27SfXpFEx43ZH7iwGYiD2z18UStnHKMbvATGKmjFkFlNUC8SGkQIxEDMipCUMFUPFogNh2gGtEML5uI9zjv0ggViSIBCLESvL8tAfEKOC0jOZIKaoFoiJM5jSGIKYtYLYUYyNJYiBWN6ogYpZexBGEaPILDb0tZI2nMp1VAwVQ8VMPQXEFGSsH6/IWewE5F1Jr2/S/0YAAAAASUVORK5CYII=' />"}],"value":"[2 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x68a7189d \"comportexviz.server.notebook$eval28204$fn$reify__28208@68a7189d\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>3</span>","value":"3"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEgAAAAzCAYAAAA0CE5FAAABfklEQVRoQ+2aQW7CMBBF7Wyq7gjecJv2BIgbcJTCKtfgCBUnoLdhk+xAYpMgFRLwJLKxEhCOXnYjW440me//Z/J1URSVuj5VVa211j/38dSYJj7N538f2+1Xvf5IfNxsmv1d58v3vV1MguyCaH2gPM9XdUUkSbIry/L7Pv5cLpt1NZv9qv1+Ua8/Eh+ybOI6X77v3WJNBXkqiAQBMevKCIUwEBOs3bqkgRgQA2IuodtbeAIxIAbEgJijme59x/honl7M7j2lkEQo+ioIFoPFYDFYDBa7ZKBrohk6nhh6PywGi3lYypcghCJC0frPF9yaIBQRighFhCJCEaH4nwHcHR32mOjtLwhFhGKQUDTG3Ow+SimaVdGLtRIExGyIpWm6awxiVFDbkwnEPOMNICY8mHIC6YVYX9uvtAnHpoO8ECNBguYli/W1/UqbcGw2YCAWeknLiSIQA2LOCePTIRYba8kh/tNZbHQJGprFYmOtlwvF0VXQ0Cw2ugQBMfe44wwkhkmv4MJWTwAAAABJRU5ErkJggg==' />"}],"value":"[3 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x7e5382ad \"comportexviz.server.notebook$eval28204$fn$reify__28208@7e5382ad\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>4</span>","value":"4"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEgAAAAzCAYAAAA0CE5FAAABbklEQVRoQ+2aQQ6CMBBFWzbGndgNt9ETGG/gUdSV1+AIxhPobdzAThM34EYw1KRDU4gBHrumTUom8/pnfkbneV6qz1eW5VFrvXetl8bU+6/N5ja7XFbV+TbrZ5rW59vcJ/1P7/sEyJ0QOsuyQ5UBURRdi6JYu9bz3a4+r5LkrO73bXW+zfpxOi187pP+p+99TQYJGUSAQMz5ZEiIgpig2gRIChAq5lZtMkjKIFQMFUPFfHpJ794NxEAMxEDMw6/yfmMkmadQDCwUQx1E23EcnaM4+QBJiIU6iLbjODpHcfIZJBWKkw8QiP1ZxQanWnZd1Ddigw9Q34gNTbVsE190FEMf6cFnEIgF2h2hhSKICdMdIDb2AKFiFIrOgTDbYDPGfMd7lFLIvFU5/wQIxJqIxXF8rQfCyKDfmUwQE0x6EBNmMEHMN4PoxZq9GIiBmNCdg1jHAaJQpFCkF+ty/AUVQ8U6fqQpFCkUvWYU7V7sDVfjSa+9TaF3AAAAAElFTkSuQmCC' />"}],"value":"[4 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5718a163 \"comportexviz.server.notebook$eval28204$fn$reify__28208@5718a163\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>5</span>","value":"5"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEgAAAAzCAYAAAA0CE5FAAABYUlEQVRoQ+2bQQ6CMBBFWzbGndiNt9ETGG/gUdSV1+AIxhPobdzAThM34EYwYtJPFQ2Rx65pCWTSN3/+UGyWZYW5X0VRbKy1K9947Fw1f53Pj4P9flquf2d8SZLq/ibPV+/X+jwB8m8Im6bputwBURQd8jyf+cbD5bJabyaTnTmdFuX6d8bn7XYU8nz1fm3P21/voM4jVU8xBKhjiHUdqTqiICZUmwCpAP1axUBMFI6o2L8FCMT8hTFJWiVpCkUKRa+3VN4NxEBMIKQChIqhYt4OqexAomKoGCoW8tFBIkVHMVDVUDFUDBX7ag4CMRADMRC7R6DJp3HVvgidp92h3DxeDC+GF2szSTvnHsd7jDHkoFoOegkQheJzoRjH8aE6EMYOej2TCWJC1kFMnMEEsdAdRKH4XCiCGIgF9pxVHQRiIBbkxVAxVKzlHIQX+9CL9e3/sGAv1vsAKcT69n9YXcVuEaZJrzn3P3YAAAAASUVORK5CYII=' />"}],"value":"[5 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x516bd394 \"comportexviz.server.notebook$eval28204$fn$reify__28208@516bd394\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>6</span>","value":"6"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEgAAAAzCAYAAAA0CE5FAAABcElEQVRoQ+2bQW7CMBBF7WxQdw3ecBt6AtQbcJTCimtwBMQJ6G3YJLtW6ibpAhKEIzF8jFAgj52FrUTD/35jz+DLsqzd8VPX9dJ7/6WMxyG08/9ms+/Rdjtt1l8z/l2v2/m3PF99X3k+ATIEURTFovnFsyzbVVX1oYzf5vN2vZtMNm6//2zWXzP+Wa3eleep75c636MgQ0EECItJW0ZsSSxmURuLYTEspiS2JIrWnqJ+T6J4OTGGYpaioBgUg2JQLOG+SsZ6vCdBMSgm3ZB2FAfFoBgUg2JQ7BCBW4oGqZfy1nrOYpzFtEJoB/MkindOFFNLzXFp+uVKz4MLkGqx1FJzXJp+udLz4BSknsUGFyAs1jOK9Z1aIYRTO49zTs6kUy32dAF6tMX6Tq08z3dtAxgK6vZkYjHjcNoJEBY7pxgWUxX06EQRihltwFDM6LR/OgVhsfMbRyhm/JMAikGxtEt6LIbF7qwgKAbFpO6OmGL/y1pJr7sYElgAAAAASUVORK5CYII=' />"}],"value":"[6 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1d85ad4 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1d85ad4\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>7</span>","value":"7"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEgAAAAzCAYAAAA0CE5FAAABbUlEQVRoQ+2bMY7CMBBF7TSIboMbbsOeAO0NOMouFdfgCIgTwG1okm5X2iahIUEYCWP9oBjr0VkiIRnmzf+ZcWxd1625fNq2XVtrv5X1zLn++P/l8jjZ7xfd+Z5Z/223/feHuB71fiwBCiREVVU/3T9cFMWhaZpPZT1drfrzmfl8Z06nr+58z6x/N5sP5ffV6/ePJ4NCJQXEQEwqGTJiqmr5KpedimUfIFXFVNXyVS47Fcs+g1QVyz5AIPbYGI+uYsmplm8cx0Ys+QCNjVhqqjX4s5hapJPPIBB78bOYahRBLNBRBLF3DxAqhlGUhhA46VDLFcRADMSUQWdwsAhiIAZiIHaJwBCj8djRND4o5INod9DueO/Rc2rtDufcdfuOMYYa5NWguwBhFG+NYlmWh37DFxl0vycTxAKyDmKBPZggFptBGMVbowhiIKa9KYCKxdYgjCJGMaqjCGIgpr2Mg1GMzSCMIkYxqqPoI3YG/rdJr5x9eqwAAAAASUVORK5CYII=' />"}],"value":"[7 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x637cf6a8 \"comportexviz.server.notebook$eval28204$fn$reify__28208@637cf6a8\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>8</span>","value":"8"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEgAAAAzCAYAAAA0CE5FAAABZklEQVRoQ+2aQW7CMBBF7Wwqdk294Tb0BKg34ChtV1yDI1Q5QbgNm2TXSt0kbEiqulIH9I1s0GNnhZhk8PP/nhnf9/3oTp9xHN+996/K+CmE+f7v9Xr/0DSrab5zxl+73fz9FM+jvo8nQMaC6LrubfqHq6pqh2F4VsaLzWaezy2XH+5weJnmO2f8ud0+Kr+vPn98PyvI2lJADMSkLQPEQEyzLR4V+1+1QQzEQExSKctYghiIgRiIKekX0h3WHqJexyhiFKUMKTJvIQhiIAZiV7UBIAZiIAZipwikqNtZ6Q3qYpbvia9TF6MuJqVDOGpYyIEYiIFYSh8UQvhp33HOsQdFe9CfAHEW+30Wq+u6nRu+UqwgtScx7mHM3aOYHLG7D5CKmNqTGPcw5u5RBDHDGILYpQECscJVLLdqxaXp4hArPkC5EcutWnEGsTgVK34FqekO1SgWHyAQQ8UuqouhYrdmFEtXsSM+4EmvtBKoOQAAAABJRU5ErkJggg==' />"}],"value":"[8 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x71fd401d \"comportexviz.server.notebook$eval28204$fn$reify__28208@71fd401d\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>9</span>","value":"9"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEgAAAAzCAYAAAA0CE5FAAABaElEQVRoQ+2bMQ6CMBSGWxbjJnbhNnoC4w08ijp5DY9gOIHehgU2TVzARTCWxNq0Smk+tyamgef7+P/+PmRVVY14fpqm2Usptz7Xc6W6/e6r1WWS54t2/2/Wt+Ox+/4vrs94vxTI0BBlWe7aXzRJknNd10uf6+lm0+0vsuwkimLd7v/N+no4zHxej+39STrI0EEUCMScHhkgZlJtEAMxEPNpbHvGEcRADMRA7IeHaeNhlbPY57Ond6PoGm/ocUh0cUd0BfKNmGu8occh0cUd0XWQb6MYXYFAbGQqNrhq6fFHaIgFV6DQEBtatfRQPzijGFwHgdif4w5Xowhihr+eQWxsBULFMIpu0yqoGCoWVmjvepofWsWUUq9xHSEETlo7nPYKhIq9q1iapuduwIsO6s9kgphh3AXEDDOYIGbbQRjFd6MIYiDm9qYAKmb7DMIoYhSt4g4QAzG3l3EwirYdhFHEKFolijpiD3I9Sa/d5VtXAAAAAElFTkSuQmCC' />"}],"value":"[9 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x7e85cac5 \"comportexviz.server.notebook$eval28204$fn$reify__28208@7e85cac5\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>10</span>","value":"10"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEgAAAAzCAYAAAA0CE5FAAABVklEQVRoQ+2bPQ6CQBBGd2mMnbgNt9ETEG/gUdTKa3gE4wn0NjTQaWIDNoABjAPZXTTk0W1CNjDMY76dH51lWaHKqyiKg9Z653K9NKbe7xnHt9nlsqr277N+nE71/T6eT3xfDCQ4RJqm++qLBkFwzfN87XI9327r/VUUnVWSbKr9+6zvx+PC5fMMfT+NBwkehIFAzOqXAWJS1AYxEAMxl8K2IxxBDMRADMQ8HqbFwypnse9nT4QiQtEyvwViIGaXIUUoIhQRighFhGJpgQ9JfZL2Qp2Muti/GwihiFBEKHoN8yAGYiAGYqUFfHSfSN0eZBRbGUVjzLtdRymFgSQDEcWaUSwMw2vd4IUHdXsyQUyoYnQMBGIgNkgoghiI2bUxE8WGehB1sWZdjCgmTBJ4R8x2eKU97DJ20t57FJucgVwLRdvhlfawy9h1MRD7dRQDMSEKTA2xF7JXSa+oMenlAAAAAElFTkSuQmCC' />"}],"value":"[10 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3d48213b \"comportexviz.server.notebook$eval28204$fn$reify__28208@3d48213b\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>11</span>","value":"11"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEgAAAAzCAYAAAA0CE5FAAABXUlEQVRoQ+2bQQ6CMBBFWzbGndiNt9ETGG/gUdSV1/AIhhPobdjAThM34EZQa2IhYwuY566JsTDO4//ODDrP81I9PmVZ7rTWG5/rqTH179+Wy/MoSebVfk3W18Oh/n6I69UEyJEQWZZtq38wiqJTURQLn+vxel3vp2azo0rTVbVfk/Vlv5/4vD77/skg1yMFxEBM9MgAMRAT2hZU7LtqgxiIgZhIpVzGGMRADMRA7LUcQrnDUR8KHiCMIkZRViGl3EG5Q6RyGEWMIkZRhFDnZzFpH8zumwX3Qb5VbPAB8m0UpX0wu2/2d32xwWcQiHVsFEHMYbSkiAVXLft++o5Y5wHqu4qFVq3g0x1DQ8wY8xzPUUp5P6wOPkAg9l5RjOP4VA90kUGfM5kg5rAlHwECMRBrVbQHMRCTjTGjYm0ziLPYe7kDFXO8SQBiICbrg4EYiP04g1AxVKxVa9pWsTvltEmv6GCipgAAAABJRU5ErkJggg==' />"}],"value":"[11 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3985598c \"comportexviz.server.notebook$eval28204$fn$reify__28208@3985598c\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>12</span>","value":"12"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEgAAAAzCAYAAAA0CE5FAAABWklEQVRoQ+2bMQ6CMBSGWxbjJnbhNnoC4g04ijpxDY5gPIHexgU2TVzARTCUxNLUVjGfWxOTkp/35f/7+pBVVTXi+WuaZi+l3IZcL5Xq9run6Xl2PK7a/cesb0XR/d/L8yOQoSDKsty1byyKolNd1+uQ63mWdfuLJDmIy2XT7j9mfc3zhc/nlaEryBUpHcG/Q2xyAoVGzBUpHUEQM7gaiH1bIBB779qTczHvSOlBeWo2H1ygqSHm27X0oAxiprMniP3YWcw1KILYt3MPLmbZ78LFCIpuHVJcDBdz6pASFAmKlq6lC4aL4WK4mNeLThADMRADsVaBJDnQ7qDd8ePjLgRFu+kVzmJaxSilXuM4QggEMglEUOwjF8fxqRvgooKGM5kgZuj/DAQCMRCzOouBGIjZBUF9mgMXs60g7sX6TX1czPAlAYiBmNs9GIiB2IcrCBfDxaymO3QXewAZIEmvbj+J4gAAAABJRU5ErkJggg==' />"}],"value":"[12 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x22e30b09 \"comportexviz.server.notebook$eval28204$fn$reify__28208@22e30b09\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>13</span>","value":"13"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEgAAAAzCAYAAAA0CE5FAAABX0lEQVRoQ+2bQQ6CMBBFWzbGndiNt9ETGG/gUdSV1/AIhhPgbdjAThM34AYwFmIxpQ01z10Tk5LPPObPdCqLoqhE/auq6iSlPPhcL5Vq93tut7dZkqyb/YesH5dL+38nz49AhoDI8/zYvLEoitKyLDc+1/P9vt1frFZXkWW7Zv8h6/v5vHD5vJIIMkQQAoGY1ScDxExZG8RADMScGlsQAzEQA7Fagb7i1XmxSi32vfbEKGIULftbIAZidh1SjCJGEaOIUcQo1grQtO+eu3EuZnCqkxcIo4hRxCg6TfMgBmIgBmKNAj1G8u+mO2x9jz5gNXZPWin1HscRQnhvuQYnkO8sZjswpQ9YjY1YHMdpO8BFBHVrQxAz1IYdgUDs0ygGj9jYWUsfew4eMe8ChYbY2FlLH3sGsV8/0r7PxWyNIogZJu1BzHCZxXsEgdjn0XPwRhHEQGzaF+hAzHAfTjeKL1k6Sa/mZfzjAAAAAElFTkSuQmCC' />"}],"value":"[13 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x72e02e72 \"comportexviz.server.notebook$eval28204$fn$reify__28208@72e02e72\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>14</span>","value":"14"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEgAAAAzCAYAAAA0CE5FAAABYklEQVRoQ+2aMQ6CMBSGWxbjJnbhNnoC4w08ijpxDY9gPIHehgU2TVzARVBrYmlaLJDPrYmJ9fG+/v//qCyKohLPT1VVeynlNuR6rlTz+/fV6jI5nRb1ftqsb4dD830v/4cCGRoiz/Nd/YSiKDqXZbkMuZ5uNs1+RJIcRZat6/20WV/TdOZz/5IOMnQQBQIxpyMDxEyqDWIgBmLvRhKjaHDW3guEUfxtjIOrmGv20rOa9w4KrWK9L1BoxFyzl57VRpfFet9BINZzowhihqzjiph31dL3O3TEOi/Q0FXMt2rpE9XBG8XOOwjERq5iIPbv9I6KWb4YRcVGPu5AxUKfQSAGYm63VTCKGMVhvxfr27hDKfW6fiOEIKxqzvmrQKjYp4rFcXxuLmzRQd93MkHMMAIGMcMdTBCz7SCM4qdRBDEQsxyImXwQiIGYVRZDxVAxz2cQWYwsZjVRJIvZnkEg9huxB4yXSa/+RlQ1AAAAAElFTkSuQmCC' />"}],"value":"[14 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xedd98ec \"comportexviz.server.notebook$eval28204$fn$reify__28208@edd98ec\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>15</span>","value":"15"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEgAAAAzCAYAAAA0CE5FAAABX0lEQVRoQ+2bsQ3CMBBF7TSIjuCGbWACxAaMAlSswQgoE4RtaJIOJJqEhgTiSDjoEmzQo7OEhLnc8//nu+g8z0v1+JRludNab3yup8bUv39bLk+jJJlX++myvh4O9fd7+T8EyJEQWZZtqycURVFaFMXC53q8Xtf7UbPZUZ3Pq2o/XdaX/X7S5/41GeTIIAIEYqIjA8Rcqg1iIAZir0YSo+hw1r0HCKP43hijYqiYsPgGMRCTXd9gFDGKGEWM4iMCXq5cUTFUDBUb9AwCMRADMRCrItChT0Zf7NsXZtRi1GLUYoMe0iAGYiDWJ2LGmOf4jVKKrobV1WgFiFqsWYvFcZzWA1tkUHsmE8QcjUIQc8xggtinGYRRbBpFEPs1xKST9fYkvnQ+KDgVCz5Avo2idLLensSX3igGp2LBZ5BvFQs+QCAWeC0mzSCpatnvx/2dig0eoF9HTKpa9vtxtordAcyxSa/aoC4dAAAAAElFTkSuQmCC' />"}],"value":"[15 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1d87e982 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1d87e982\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>16</span>","value":"16"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEgAAAAzCAYAAAA0CE5FAAABZUlEQVRoQ+2bsQ3CMBBF7TSIjuCGbWACxAaMAlSskREQE8A2NEkHEk1CQRKEg3DMgXCiR2eJ4nK55//vYussywpV/oqi2GitVyGtx8bU8Vzn8+Ngv59W8bVZX5Kk/v9Hz0eCHAWRpum6eiNRFB3yPJ+FtB4ul3V8ajLZqdNpUcXXZn3ebkeS59FUkKOCSBCIgdg7VUPFHLIvThAq9l61UTGXMUbFUDFUDBUrM9CmObWbWVTM0ZvRi/3aB6FiqBgqhoqhYvcMvJpoMlG0eiHpkP7rPig0FQsuQaGNO6RI2UP93hnF4CoIxDpmFEHMMcGTIibu3u34+obY1xPUNxWTqpb96T24oT2IdW0PArGOfRf7N2LGmMdxG6UUe5CFfCNBIPaMWBzHh/qAFhXUPJMJYg4VBTHHGUwQ860gerHncQeIgZjsZD0q5rsHYRQxil6Xc0AMxGSXVzCKvhWEUcQoet1/sxG7AQAdSa8s771hAAAAAElFTkSuQmCC' />"}],"value":"[16 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x27861278 \"comportexviz.server.notebook$eval28204$fn$reify__28208@27861278\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>17</span>","value":"17"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEgAAAAzCAYAAAA0CE5FAAABVElEQVRoQ+2bMQ6CMBSGWxbjJnbhNnoC4g08ijp5DY9gOAHexgU2TVzARTCtCfVRSQh+bh1MyvN9/P/7W3VZlrV6feq6Pmitd2NaL41p9/NI08ssy1bN/vqs76dT+/2vnpcCeRqiKIp984tEUZRXVbUe03q+3bb7U0lyVtfrptlfn/XteFxInk/TQZ4OokAgZiEJYo7KiVXLVfGpIxZcoKmrmBQpV8Unr2LBHQRif65iIOaZzUBs6AKhYt2zJyrmi3dQMVRMFG9gFH1IMYsJI2VUDBWThfQgBmJBCSIqhorZB4vBwyoqhoqhYl1n9SDmOYomURw6MCPuIO4g7vjlS9oY875uo5QicnWc9UeBMIq2UYzjOG8vaNFBn3cyQcwzrIKY5w4miEk7CKNoG0UQA7GwwAwVk76DMIoYRVGiCGIgZv/ZRZooYhSlHYRRxCiKEkUXsSczekmv6AgX9AAAAABJRU5ErkJggg==' />"}],"value":"[17 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x54662b44 \"comportexviz.server.notebook$eval28204$fn$reify__28208@54662b44\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>18</span>","value":"18"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEgAAAAzCAYAAAA0CE5FAAABTUlEQVRoQ+2bPRKCMBBGExrHTkzDbfQEjjfwKGrlNTiCwwnwNjTQ6YwNsVBQLAgZ/iI+usxQhGUf37ebRWZZpsXr0lofpZR7l9dLpcr93TebyyyKVsV+m6xvYVje3+h5CZAhIdI0PRRvwPO8OM/ztcvr+W5X7lcEwVkkybbYb5P19XRa2DyfJIMMGUSAQKyCIIgZVA0V6zpAqFi9aqNiJmOMiqFiqBi12EcxPXixioqhYpV2CEYRo0i7o7ajOfhHGqOIUcQoYhQxis8INDk0oGlvaD+gYq4FiFqMWoxarFeZB7EfR6ytan1Pf0yu3TF6gFxHrK0x/J7+mNzR89AZpJR6j9cIIZw/OBw9QCBWVTHf9+NyIOsfMshWtf4OsdYBmjpitqoFYoZuAYjZBgjEUDGryXoQA7Hq5D0q1vGwAoiBGIj1+n8YiIFYt4g9AHOUSa+h03Z8AAAAAElFTkSuQmCC' />"}],"value":"[18 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x369ed3dc \"comportexviz.server.notebook$eval28204$fn$reify__28208@369ed3dc\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>19</span>","value":"19"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEgAAAAzCAYAAAA0CE5FAAABaklEQVRoQ+2bsQ3CMBBF7TSIjuCGbWACxAaMAlRZIyMgJgjb0CQdSDQJDQmKQb5YjpAML52lFM7p3v3z90VXVdWo59M0zUFrvYtpPTem2+99vT5PTqdlu/8h61ued+9//H4CJASoLMt9G/EkSYq6rlcxrafbbbd/tVgc1eWyafc/ZH3NspnrezUZBGLOmiTWIBADsbAMogahYk5V+zkVC20M7UZSLNKxIfb1AMWmYqGNod1IgphwNgOx0ACB2I81iqFFWkTKtnv+TcW8A/RviEmqZds90dkdICZYvl8PEIi5HVQQky4pUDHhFgfEQKznKHr3QSAGYj1HkUYx9PTOWUww6e0AoWKoGCo26rgLiLkdROwOYWDKGPMap1FKcVi1kHoLECrWV7E0TYtuAIsMep/JBDHfGgRiIObVKIIYiLnHgCW7AxXzzSAcxb6jSKMoTNaDGIi5//aRbjVADMRGziBUDBXzuhezVewBpvFJr3ciH1gAAAAASUVORK5CYII=' />"}],"value":"[19 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1477e774 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1477e774\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>20</span>","value":"20"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEgAAAAzCAYAAAA0CE5FAAABX0lEQVRoQ+2aPQ6CMBiGWxbjJnbxNnoC4g08ijp5DY9gOAHexgU2TVzAQcFQDZ9YxVQetyYO5U2f7/3+dJZlhbr9iqJYa62XPp3HxlT3PUfRfhDH0/L+r5xP2231/6ffj0CCQGmarkrFgyBI8jyf+XQeLhbV/dVkslOHw7y8/yvn42YzavpezQsCscaYJMYgEAMxtxdEDMLFGl0NFxMSRzFIgxiIgVib2suu1UDMNQaRKJIokih+NQaBGIiBGIjdFHjW8aSjKAwRaNr/WiBcDBfDxXAxXOyqwDujb1wMF3MbdZMo/voF0bSnaU/T/pN5kDHmvk6jlGL9xdowexCIWqxei4VhmFQLWLygaG/PxUDMmpOJAoHYnyHmWntJo2fvEetcIN8Qc6297DVhe4HKexfr/AX5Vot1LhCI4WK1pr20MNV7F2stUN8Qk9Z+e+9irV9Q31ystUAgVh8a2IhdAOcLSa+9ovk9AAAAAElFTkSuQmCC' />"}],"value":"[20 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3f52250 \"comportexviz.server.notebook$eval28204$fn$reify__28208@3f52250\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>21</span>","value":"21"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEgAAAAzCAYAAAA0CE5FAAABbklEQVRoQ+2bMW7CQBBF124iOpxtuA2cwMoNOAqh4hocIfIJzG1o7A6kNDYSwoYdSx6stYs1L91KKdjRvP2fP0NUlmVtHn91Xe+jKNqFfP62tv38/2l6+sqydXOfd87X47H9/3s9KJBbwE6BiqL4bSocx3FeVdUm5PNiu23vY1arP3M+/zT3eed8ORyWr/engwSCIKa8SSCmIAdiQzsIFUPFULFXY4tRVJz+6AXCKLpGEhVDxUb+sgpiIEbcManMg9jMEfP1PTJxnF3cMXmBQkfMN0GUiePsjOLkHRR63DF5gUAMFes1ih3VkmnBpyOmFujTEZOqJeeCwc/FfB9ptYNATMzi5RsEYu6oGcS0UTMqpszBQKzfGIKYgpi19rk+Y4xB5sVGWadAqJirYkmS5O3CFR2UnqRRBLGhbxCIgdiguAPEQMxv7RcVG9pBxB1u3IFRFGvAMlEEMRDz+3UPiIHYyB2EiqFivZv2mordABp3Sa/jw8jWAAAAAElFTkSuQmCC' />"}],"value":"[21 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x662c0561 \"comportexviz.server.notebook$eval28204$fn$reify__28208@662c0561\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>22</span>","value":"22"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEgAAAAzCAYAAAA0CE5FAAABU0lEQVRoQ+2bPQ6CQBBGWRq1M9pwGz0B8QYeRa1orak8guEEehsa6DSxYY1GNuySiONqgT666dhh3n7zhyrLUgePR2u9UUqt+mxPplPz/pc4Pg6ybFaf5x1b4aDnDlRFUaxrD4dheKiqat5ne7RcmvMEUbQP8nxRn+cdmwhyEDzvdgbJ+5UDYjZiLQeBmI3gKUnGzSuGCAIxmcyDWIeqgVhH4oiKSR2EiqFioloMxEDMrxZDxaQRRC1GLSZqd4AYiPm1XKnFqMU+HEGoGCqGijXneL5zMWoxqczT7qDdQbvjq3cQiIEYiIHYwwOvbJ/4bnfQ7ujIg4ZZZtZ/bt+F2bzjsJaDUDFbxYZpujULV0RQe4cRxKR3EIiBmChRBDEQ8xs9o2LSCKJpbzftSRQ75mIgBmJ+g0MQ+zfEfOdg7v9kP5coft1Bfa/FfDuI7v9kropdAf5XRvBHTKkFAAAAAElFTkSuQmCC' />"}],"value":"[22 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x26ec3a4b \"comportexviz.server.notebook$eval28204$fn$reify__28208@26ec3a4b\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>23</span>","value":"23"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEgAAAAzCAYAAAA0CE5FAAABOElEQVRoQ+2bPQ6CMBiGKYu4ysJt9ATEG3AU9QbOnML0BHobFthdqDGBJpSEUkuX+rB17Jvv6fv9IbquU8nwKaVuQohLTOdDnuv7vMvytZPyON5vzVkg0LJgom3b66homqbPvu9PMZ33VaXvlxTFI2ma83i/NWciyIIcAtkEArFl5IggWwThYrgYLraU2JIoWjJ/b4FwMVyMWizoGwRiIAZiIDYo8Es7hnZH6DyIWoxajFos6CMNYiAGYiDmMdik3WGZ45EoGomib8SYo+nophqbCxRbu8MXKXM0TQT921wMxAK7FohZkMqk1Os/3/SLN8gQbCYQLjZt2md1fdcLVkTQfGcRxFzfIBADMae1XxADMb+1X1zMNYJia9r71mIkipZNexADMb+/e0AMxDaOIFxsOnoGMUfEPvNQK3qTPAsYAAAAAElFTkSuQmCC' />"}],"value":"[23 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xd0b25a5 \"comportexviz.server.notebook$eval28204$fn$reify__28208@d0b25a5\"]]"}],"value":"([0 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x29901e4a \"comportexviz.server.notebook$eval28204$fn$reify__28208@29901e4a\"]] [1 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4cd94599 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4cd94599\"]] [2 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x68a7189d \"comportexviz.server.notebook$eval28204$fn$reify__28208@68a7189d\"]] [3 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x7e5382ad \"comportexviz.server.notebook$eval28204$fn$reify__28208@7e5382ad\"]] [4 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5718a163 \"comportexviz.server.notebook$eval28204$fn$reify__28208@5718a163\"]] [5 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x516bd394 \"comportexviz.server.notebook$eval28204$fn$reify__28208@516bd394\"]] [6 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1d85ad4 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1d85ad4\"]] [7 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x637cf6a8 \"comportexviz.server.notebook$eval28204$fn$reify__28208@637cf6a8\"]] [8 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x71fd401d \"comportexviz.server.notebook$eval28204$fn$reify__28208@71fd401d\"]] [9 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x7e85cac5 \"comportexviz.server.notebook$eval28204$fn$reify__28208@7e85cac5\"]] [10 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3d48213b \"comportexviz.server.notebook$eval28204$fn$reify__28208@3d48213b\"]] [11 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3985598c \"comportexviz.server.notebook$eval28204$fn$reify__28208@3985598c\"]] [12 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x22e30b09 \"comportexviz.server.notebook$eval28204$fn$reify__28208@22e30b09\"]] [13 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x72e02e72 \"comportexviz.server.notebook$eval28204$fn$reify__28208@72e02e72\"]] [14 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xedd98ec \"comportexviz.server.notebook$eval28204$fn$reify__28208@edd98ec\"]] [15 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1d87e982 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1d87e982\"]] [16 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x27861278 \"comportexviz.server.notebook$eval28204$fn$reify__28208@27861278\"]] [17 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x54662b44 \"comportexviz.server.notebook$eval28204$fn$reify__28208@54662b44\"]] [18 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x369ed3dc \"comportexviz.server.notebook$eval28204$fn$reify__28208@369ed3dc\"]] [19 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1477e774 \"comportexviz.server.notebook$eval28204$fn$reify__28208@1477e774\"]] [20 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x3f52250 \"comportexviz.server.notebook$eval28204$fn$reify__28208@3f52250\"]] [21 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x662c0561 \"comportexviz.server.notebook$eval28204$fn$reify__28208@662c0561\"]] [22 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x26ec3a4b \"comportexviz.server.notebook$eval28204$fn$reify__28208@26ec3a4b\"]] [23 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xd0b25a5 \"comportexviz.server.notebook$eval28204$fn$reify__28208@d0b25a5\"]])"}
;; <=

;; **
;;; Already we see a potential area of improvement: **wrap encodings around at the edges**. Hours 0:00 and 23:00 currently don't overlap. Neither do Day 0 and Day 6. Maybe some day we'll come back to this.
;; **

;; **
;;; ### Consumption rates
;; **

;; **
;;; There are a few ways of encoding numbers.
;;; 
;;; Two objectives for encoding:
;;; 
;;; 1. Similar values have similar encodings.
;;; 2. Different values have different encodings.
;;; 
;;; If you want both, you won't be able to beat a "moving window" encoding.
;;; 
;;; $$ capacity = total\\_bits - active\\_bits $$
;;; 
;;; If you're willing to sacrifice some of (2), you can use every [choice](https://en.wikipedia.org/wiki/Binomial_coefficient) of bits. At least, that's the theoretical limit. It will take a lot of effort to reach it. You'll probably have to build stateful encoders that store a list of values that they've encoded.
;;; 
;;; $$ capacity = {total\\_bits \choose active\\_bits} $$
;;; 
;;; If you're willing to sacrifice some of (1) and (2), and rely on probability instead, you can encode an arbitrary number of values, depending on how little you care.
;;; 
;;; $$ \lim\limits_{care \to 0}capacity = \infty $$
;;; 
;;; I'm going to explore both ends of the spectrum, but not the middle. The middle seems painful.
;; **

;; **
;;; #### Scalar encoding: Moving window
;; **

;; **
;;; This approach requires us to specify min and max values. We have those readily available.
;;; 
;;; We could have, say, 1000 bits, where the 415th bit means "41.5". To make this discussion easier, let's forget about the decimal point and just use these * 10 integers.
;; **

;; @@
(show-example-linear-encoder 0 1000 0 1 [10 20 30])
;; @@
;; ->
;;; Showing (e/linear-encoder [1000] 1 [0 1000])
;;; 
;; <-
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>10</span>","value":"10"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAACWCAYAAABJoA8EAAACR0lEQVR4Xu3dQU7CUBSF4ZYVOGQ3uoJuSRmxDZbgEmBBZabRCTUGcMDotYfk5eHnjMTGePN+zn97b6E/Ho9Td/mZpmnT9/2r1+cKFNVDAcMDM47j2/XErVar/el0evH6XIGSevROYHgCFTAsIISztywIp9YBYQhXtQYIQzhEMC2gFJbCdXt3KRy+BUAYwhBu+v4jhCEMYQhfKlByB/fR7njrhdNWjkgTabezmg4RCEMYwhD+z7s1euGwF5667m8763sYDp+73XPTSKViPPd6BQxT+GsY9tcT163X7x/b7ZPtLNtZ5QuSc5G9/X0iHSIshcMUdgLDE6iAYQEhDGFDpaY7HwhDGMIQNhee8XBd2js+2vVEmkgbKjUdIhCGMIQhbC5cPkR5tAXJ9P+xYJmKvRSWwlJYCkthKVz6QTu3qS2FpXCYomkBDZUMlQyVmtYYCEMYwhA2FzYX/q1A0YfO2pG+s3hLYSkshaWwFJbCUniphridlaWouXB6gtLrzYVDsSbS2VuAoRKEQwTTAkIYwnphvbBeWC+sF9YLl30NWdq73vt6Ip16oF44FHEiTaSJNJEm0kSaSBNpIr1suykV2drXE2ki7WHDpj0QwhCGMIQ9L+x5Yc8L12rlpLAUlsJSWApLYSkshRc+LVn7fqC5sLmwuXDTGgNhCEMYwlY7rHZY7ajViUhhKSyFpbAUlsJSWApbsLRguejz82rfEa7994k0kSbSRJpIE2kiTaSJNJEm0tOmnyvmRJpIE2kiTaSJNJEm0m2K9A9o3ymtKb4xkAAAAABJRU5ErkJggg==' />"}],"value":"[10 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x2ba5e4e7 \"comportexviz.server.notebook$eval28204$fn$reify__28208@2ba5e4e7\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>20</span>","value":"20"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAACWCAYAAABJoA8EAAACUElEQVR4Xu3dwW3CQBRFUZsKsqSbpAK3lLCiDUpICVAQ7BIlGztCQBasxn5I1piTHVIcKV/zuPf/Gdvt6XQamuvPMAybtm3ffb5UoKgeChgumOPx+HFbcavVat/3/ZvPlwqU1KO1AsMVqIBhAUU4+8oS4dQ6RFiEZ7UGERbhMIJpAVEYheft3VE4/AoQYREW4arnjyIswiIswtcKlExwlzbx1gunrRyRJtLGWVVDRIRFWIRF+JnP1uiF9cJ64aq/A0VYhEVYhI2zRpwJTsc/S7teL6wX1gtXDRERFmERFmHjrPI7c5a2r5v+P/aFU7FHYRRGYRRGYRQuvT/4ntoojMIhRdMC2lSyqWRTqWqNEWERFmERfuZ94aFp/p+d9dt1h+/d7rXqFZF63djrFTAU8Z+u299WXLNef35tty+enVU+G9ALj43s/e+bSIcRJtJEmkhXrU0iLMIiLMLP3AvzwNADFTAsIAqjMAqjMAq7U+lcgaK3NxhnhdS9LyAKozAKozAKozAKT9WQdFcOhVEYhVEYhVEYhVG47E2C6f29j77e6azUA20qheMtIk2kiTSRJtJEmkgTaSI9bV81Fdm5ryfSRNqDd6r2QBEWYREWYc/OKr8/9tET3dr/nol0KuIojMIojMIojMLnCkx5PzIKo3BI0bSA9oXtC9sXrlpjRFiERViEHe1wtMPRDkc7pvWic0+09cJpK2ciHfbSRJpIE2kiTaSJNJEm0kTaGelJD99KTb7264k0kSbSRJpIE2kiTaSJNJEm0sOmHSv2RDoT6T/pqSmtzKlA4QAAAABJRU5ErkJggg==' />"}],"value":"[20 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x793b5d6f \"comportexviz.server.notebook$eval28204$fn$reify__28208@793b5d6f\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>30</span>","value":"30"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAACWCAYAAABJoA8EAAACSUlEQVR4Xu3dvXHCQBRFYYkKHNKNXYFasologxJcAhQkMnvshHUAOCBacZnRLPqcKVDgNzo6729FfzweS3f5K6Vs+r5/d32OQFU8BDB8YMZx/Lg+cavVan86nd5cnyNQE4/eExg+gQIYBhDC2SsLwmnWAWEIz5o1QBjCIYJpAFmYheet3Vk4fAVAGMIQbrr/CGEIQxjClwjUdHCfreOtFk5LOYm0RFo7q2mJQBjCEIbwkndr1MJqYbVw0+9ACEMYwhDWzpqwE5y2f57tfrWwWlgt3LREIAxhCENYO6v+ZM6zzXXT/8dcOE3sWZiFWZiFWZiFa88H31qbhVk4tGgaQEMlQyVDpabTGAhDGMIQNhc2F67+VtZt3qidFSbiLMzCLMzCLMzCLFz7yc60fXV7PwuzMAuzMAuzMAuzcN13m9Ntqkffby6c5oXaWdpZtrOazgMhDGEIQ9iCpQVLC5ZzlXIszMIszMIszMJ3W7h03f8vG/4Ow+F7t3ttGqm0PTX1fgEMLfwzDPvrE9et159f2+3L9XqJ34We2rHWkZ6KrAXLEFmrHdkqh8OGKbIQhvCstS+EIfxgBNOAWrDMrCyRTp9AHenwlQBhCNuRbrr/CGEIQxjCjjk45uCYw1y7MSzMwizMwizMwizMwg4bdqWUTZ92eFu7X0daR3rWObGhUvrKgDCEIdx0KQdhCEMYwk4qOal090klEgklIoBhAA2VDJUMlZpOYyAMYQhD2FzYXNhc2FzYXNhceJGD8XQqJ5GWSEukJdILTqT/AGqCKa2w4nmhAAAAAElFTkSuQmCC' />"}],"value":"[30 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x57f72ea4 \"comportexviz.server.notebook$eval28204$fn$reify__28208@57f72ea4\"]]"}],"value":"([10 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x2ba5e4e7 \"comportexviz.server.notebook$eval28204$fn$reify__28208@2ba5e4e7\"]] [20 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x793b5d6f \"comportexviz.server.notebook$eval28204$fn$reify__28208@793b5d6f\"]] [30 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x57f72ea4 \"comportexviz.server.notebook$eval28204$fn$reify__28208@57f72ea4\"]])"}
;; <=

;; @@
(show-example-linear-encoder 0 1000 10 1 [10 20 30 35])
;; @@
;; ->
;;; Showing (e/linear-encoder [1020] 21 [0 1000])
;;; 
;; <-
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>10</span>","value":"10"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAACZCAYAAAC49r3RAAACUUlEQVR4Xu3dsVHDQBSEYUkVELobqEAtAZHbcAmUYBdkZzCQ6AhkE5BY0gaasz4yDzjZeb/27fp8tJfLpTTXn1LKe9u2r16PCkzSg4DhwJzP57fbxHVddxyG4cXrUYEperQmMJxAAoYCQjh7ZEE43TogDOFVtwYIQzhEMBWQC3PhdbM7Fw4fARCGMISr7h8hDGEIQ/iqwJQG99Eab1k4jXIWaYu0OqtqE4EwhCEM4S2frZGFwyxcmubvdNZP35++DofnqpFKF+O57ydg6MLffX+8TVyz23187vdPTmfNOJ1lAsMJJGAoIIS58LqNNIQhvG4WNoHhBBIwFJALc2EuXHV5AWEIQxjCWz7aYQ8M90AChgJyYS7MhbkwFx4V8LnwgltLuDAX1khXbSIQhjCEIbzlA5aysCwsC1f9DIQwhCEMYXWWOmvypbP/vwYhC8vCsnDVJgJhCEMYwuqsqwK+bDj7KvyWiYQmQsBQQHWWOkudVfUaA2EIQxjCGmmNtEZ66v+Bm3sz0b2/58JcmAtz4S27sDtUwzaGgKGAbrAMXdgEhhNIwFBACEN43UUawhBe92iHCQwnkIChgFyYC3PhqussCEMYwhDWSI8KlLLg4pl7H/s9+u8t0hZpWbhqE4EwhCEM4S1/U0kWloVl4aqfgRCGMIQhrM5SZy2u82RhWVgWrtpEIAxhCENYnTUq0HXd7ItnhiHLkrW/v2UioYkQMBRQnZU9giCcHj2BMIQt0hZpi7RFemmQ4MJcOHTRVECLtEXa58JVrzEQhjCEIbzhox2/CxLQS0kdzQoAAAAASUVORK5CYII=' />"}],"value":"[10 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x20f330d7 \"comportexviz.server.notebook$eval28204$fn$reify__28208@20f330d7\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>20</span>","value":"20"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAACZCAYAAAC49r3RAAACYElEQVR4Xu3dwU3DUBCEYdsVcEw3UIFbAk5pIyVQQigouYHgEiPkhENOzxmkpxd/3CKRQ0b7e3Zn/ez+eDxO3flvmqbXvu+ffZ4VKNKDgGHBHA6Hl0vFDcOwP51OTz7PCpTo0avAsAIJGAoI4eySBeG064AwhKt2DRCGcIhgKiAX5sJ1Z3cuHF4CIAxhCDedP0IYwhCG8FmBkgT33hJvs3A6ymmkNdLirKZNBMIQhjCE13xvjVnYLGwWbvoaCGEIQxjC4qwF9wSn8c+9fd8sbBY2CzdtIhCGMIQhLM4qP5lzb3vd9PfYC6eNPRfmwlyYC3NhLlx6PvjatbkwFw5dNBXQUslSyVKp6TYGwhCGMITXvBeeuu7v2Vnf4/j+uds9Nl0RaV+39PsEDBvxr3HcXyqu22zePrbbB8/OKs8GehUYViABQwEhHDbSKjCsQAKGAkIYwnVnYQhDuO5eWAWGFUjAUEAuzIW5cNP5I4QhDGEIWyrNClgq3fBCGo20Rtos3LSJQBjCEIbwmo85mIXNwmbhpq+BEIYwhCEszhJn/SpQ9D7h63uozcJmYbNw0yYCYQhDGMLirLMCTirtlz5Ly0mlpYcLNdKh614LKM4SZ4mzmm5jIAxhCENYIi2RlkjfGsmnkwgX5sJcmAtzYS7Mhblw1w3DsDgRXpog//f/e4502gd6m0OYUHsIbdhIq8CwAgkYCghhCNedhSEM4bp3Z6nAsAIJGArIhbkwF246kYYwhCEM4TUvlfSBYR9IwFBALsyFuTAX5sKzAjcdeU/3qq1/nwtzYYl00yYCYQhXRfgHnDrQS6TLiG4AAAAASUVORK5CYII=' />"}],"value":"[20 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x793994a3 \"comportexviz.server.notebook$eval28204$fn$reify__28208@793994a3\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>30</span>","value":"30"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAACZCAYAAAC49r3RAAACUklEQVR4Xu3dMVLCQBiG4SwnsOQ2eoJcSam4BkfwCHAg6HS0YS0AC6oNnzNx4bHLjCn8Z9+83/77J5bD4VCH80+tdVVKeXV9qkBTPRQwXDD7/f7tsuIWi8X2eDy+uD5VoKUexQoMV6AChgWEcPbIgnCaOiAM4VlTA4QhHCKYFpCFWXjevTsLh48ACEMYwl33HyEMYQhD+FyBlg7uvXW87YXTrZwgLUhrZ3UtEQhDGMIQfuTZGnthe2F74a6fgRCGMIQhrJ01YSY4bf/c2/32wvbC9sJdSwTCEIYwhLWz2t/Mubdz3fTvcS6cBnsWZmEWZmEWZuHW94Ovrc3CLBxaNC2gQyWHSg6Vuo4xEIYwhCHsXNi5cPO3sq5zo3ZWGMRZmIVZmIVZmIVZuPWTnWn76vp+FmZhFmZhFmZhFmbhtu82p9NUf32/c+E0F2pnaWeZzuo6B0IYwhCGsAFLA5YGLOfayrEwC7MwC7MwC99s4ToMv//Z8Hscd5+bzXPXSKXtqan3K2Bo4a9x3F5W3LBcvn+s10+X60f8LvTUjnWxAsMVqIBhASEcjnZYgeEKVMCwgBCG8LzTWRCG8LztLCswXIEKGBaQhVmYhbvuP0IYwhCG8CO/5iAHhjlQAcMCsjALszALs/CpAkY76qoY7QitOrWALMzCLMzCLMzCN7/yby8cWpuFWZiFWZiFWZiF53pbk4VZmIVZmIVZmIVZuNPP3+nGhN0YBQwLKEgL0oK0IC1IC9KCtCAtSA+13jDdNHWY57/9viAtSHtfuOscCGEIz4rwDy1x0EuZ6CPzAAAAAElFTkSuQmCC' />"}],"value":"[30 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x44aa785e \"comportexviz.server.notebook$eval28204$fn$reify__28208@44aa785e\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>35</span>","value":"35"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAACZCAYAAAC49r3RAAACmUlEQVR4Xu3dMVLjQBSEYcnZZtSS+DZwAl2Bo+xu5Gv4CBwBDmRnUEtibblsgUsbjZtimNJHppId0PX+6fdamnG/3+/H7vw3juOfn7e3v6brt2F4ft1u7y7v933/fv/4+cVfEzAsiN1u93uqsNVq9fTj4eH9uluvH182m5vL+4fD4d71SYGjXr0KDCuQgKGAEM6WJAinXQWEIVy1K4AwhEMEUwG5MBeuO5tz4XAJgPAnI7z4eKrUVOYIE7AQ6TnC4qoypP9rpFVgYQVCuFCw+RoJ4TJk50schEtdd/55CEP4e8VZXLiwIiFcKBgXzlyXC6euy4VDZCEM4boJNIQhXLVx5sJc+JMRTAUVZ2WuLM5KK9AsHC4JEIZw3cYawhCu2lhzYS4cIpgKyIW5MBduejcohCEMYQifFTjun13aC5oa6bQPNAuHjTgX5sJcmAtz4ZMCDtK54iAhLsyFJdJNmwiEIQxhCF+cP7i4fSZmYbOwWbjpNRDCEIYwhMVZ4qyr4zyzsFnYLNy0iUAYwhCGsDjr49cJlvZ2Vfr/ejvL21mhi6YCirPEWeKsptsYCEMYwhCWSEukJdK13rDlwlyYC3NhLsyFuTAXPv1eb5oQf/X3JdJpoOq5cJhoa6Q10hppjfSSG+mx68apAt6G4fl1u71ruiJSVy39PgFDF/47DE9TxXXr9ePLZnMzXbfY2H55I60CwwokYCgghMNGWgWGFUjAUEAIQ7juLAxhCNfd5qACwwokYCggF+bCXLjp/BHCEIYwhD1UOingodIVh9BqpDXSZuGmTQTCEIYwhJd8aodZ2CxsFm56DYQwhCEMYXGWOOvqzYZmYbOwWbhpE4EwhCEMYXHWWQE7lYqPHOiZSGgiBAwFFGdlcdY/QzXfS9esu8UAAAAASUVORK5CYII=' />"}],"value":"[35 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4af92f0 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4af92f0\"]]"}],"value":"([10 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x20f330d7 \"comportexviz.server.notebook$eval28204$fn$reify__28208@20f330d7\"]] [20 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x793994a3 \"comportexviz.server.notebook$eval28204$fn$reify__28208@793994a3\"]] [30 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x44aa785e \"comportexviz.server.notebook$eval28204$fn$reify__28208@44aa785e\"]] [35 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4af92f0 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4af92f0\"]])"}
;; <=

;; **
;;; Maybe give each number a 100 bit (10.0) radius. Then 20 and 35 share 50 bits in common. This means a single representations is 201 on-bits, and the total number of bits goes up to 1200.
;; **

;; @@
(show-example-linear-encoder 0 1000 100 1 [200 350])
;; @@
;; ->
;;; Showing (e/linear-encoder [1200] 201 [0 1000])
;;; 
;; <-
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>200</span>","value":"200"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAAC0CAYAAAADxKs5AAADMklEQVR4Xu3dQU4jMRCF4XQ2aHYINtyGOQHiBhyFYcU1OALiBMxt2CS7GWk26VmQjkSDAtXljrDzsYvoQGT8+716totuvV73i+1X3/d3XdfdRl6fnZ/vnv93dfX75Onpcnj/R6//Pjzsvj/l90U/3+zPG8DYhHn3B1mtVr+GGbNcLp83m83PyOsfNze79y8uLh4XLy/Xw/s/ev3n/v408vOjn+fQz3dmYHIGGsDkAEI4tmSNlwgIB13HOxGBMIRDrqG0SkMYwkkEswNIhalwqHYvXhtT4eQSAGEIQziSPxZfw6hwDsGssWakszOQClNhtXDVIgJhCEMYwtsROMqNdbVwzogz0ox0UkWzAwhhCIuzqrYxEIYwhCE8jIADls5Ih/dcxFlJI06FqTAVpsJU+HUEptwcclPJVa/cGkqFqXBuBmUDVUaakbYvXLUPhDCEIQxhRzteR8CV/3jLA0c7GOmkimYHUC2sFlYLV21jIHxghKMJ9LiX1tH3zjKAI9WPIhztlTXupXX0vbPMwNEMjMZZBhDCReOvcC1sBkK4bOVChRnpsjMqms5Q4WQcBmEIQ/io4iy18FvkGemo6o6fp8JUuGhtG+2lBWEIJxHMDiAjzUgz0oy0m0pfv6kkkZZIl10zqTAVLjujor5QLZw04hCGMIQZaUaakZ7a8kAeGLUt4+epMBWmwlSYClNhKjyx81K4XyAVzqlu+v8LS6Ql0mVtDyOdQ1oply3lJNISaaezqi7lIAxhCENY76yv985yzcE1h7KlHBWmwlSYClNhKjy1g6c4S5yVVNHsAEqkv3ki3Vy7u0PfF25+AOdGuLV2dwffF25+Bs5dCzc/gBCmwmXjqagvhHDSiEMYwhCuOg+EMIQhDOE91xzUwiNjGT0j3Xwpx0gz0m3vC0PYGrjfJjHSjDQjzUgz0vPdF6bCVJgKDyvMlP+f/Flf6dlPZ0EYwhCG8HYE+r6/SzeJiO77fva8OEucJc6qupSDMIQhDOE9N5War0Qk0hJpiXTVayCEIQxhCNtUsqn0bZvQMtKbtyoVbbzjgKU9kf0qL84SZ4mzqvaBEIYwhCEskZ6vd1brRvo/3/0HYKL629EAAAAASUVORK5CYII=' />"}],"value":"[200 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6ee075e8 \"comportexviz.server.notebook$eval28204$fn$reify__28208@6ee075e8\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>350</span>","value":"350"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAAC0CAYAAAADxKs5AAAC70lEQVR4Xu3dT04iQRSA8Wo2xt1EN97GOQHxBhxFXXkNj2A4wXgbNrCbSdzQxiiEoGFsXxGqun7uOjEdeKmvv3p/uuhWq1WfPv76vr/vuu425/XF5eX2fi/T6fPZfH69uf93rv89Pm7//xifL/x9BTC4YJbL5d1mRUwmkz/r9fp3zuvz2Wx7/3R19ZQWi5vN/b9z/ffh4VfOz5P7+3VWYHAFCmAwgBCOPbIgHN11QBjCWXcNQy0NYQgHEYwGkIVZOGvuPjg3ZuHgIwDCEIbwofpg8fVACEMYwhD+iMBXFW0V6WCqo6kkgIefsSzMwizMwiz8HoFjjE6wMAuz8DFHQzSVgoQJYDSANtIj20gXX0DdX7GlNZWqC2BpCJc+ULk/+lGcRKpbgRAeWV8YwnuWGpoLQ7i1ALKwjbSK9G5FurpnIIQhDGEIZ2wq2Ui3tg+UC8uFvalUtUQgDGEIQ/jAsSmjrwfKheXCcuGqn4EQhjCEIaycVc6MdHU9EbmwXFguXLVEIAxhCENYOes9AiUcQqsvrC8ctFJrAVTOUs5Szqp6GwNhCEMYwirSKtI/PrVEOSuYOLAwC7MwC7MwC7PwkQ5P++/Z+izMwizMwizMwizMwim9/VipvnBrbU3lLOUs01lV7wMhDGEIQ9iApQHLnw6YOoQ2epI5C7MwC7MwC7MwC5+qoMrCLMzCLMzCLMzCLJzSy3T6XN2xJ6azTGeZzqp6GwNhCEMYwgYsDVgasDxVJsLCLMzCLMzCLMzCLOw1h9T3/f3+66Gj/z0RfWF9YX3hqveBEIYwhCFstMNoh9GOU6VyLMzCLMzCLMzCxVh4aPnqbD6/hvAOws0FMHdf+Hw2u9usqLeDdNJicTPkuvmDd5pbgbk30s0FEMKFjXY0twIhXFguzMLBUywgLIDDxuVYmIWHrZggYZ9ONmdhFh5XRZqFg8+I5gLIwizMwlWX9CEMYQhDeOdNJeUs+8BhjwS5sFxYLly1RCAM4ZMi/ApxPgdgc+yJfgAAAABJRU5ErkJggg==' />"}],"value":"[350 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x19ffe6eb \"comportexviz.server.notebook$eval28204$fn$reify__28208@19ffe6eb\"]]"}],"value":"([200 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6ee075e8 \"comportexviz.server.notebook$eval28204$fn$reify__28208@6ee075e8\"]] [350 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x19ffe6eb \"comportexviz.server.notebook$eval28204$fn$reify__28208@19ffe6eb\"]])"}
;; <=

;; **
;;; Without this overlap, 20.0 would be as similar to 20.1 as it is to 99.
;; **

;; @@
(show-example-linear-encoder 0 1000 0 1 [200 201 990])
;; @@
;; ->
;;; Showing (e/linear-encoder [1000] 1 [0 1000])
;;; 
;; <-
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>200</span>","value":"200"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAACWCAYAAABJoA8EAAACXUlEQVR4Xu3dsXHCQBCFYYnMmcckdGNXQAuUYhPRBiW4BFwQZPaME+QATADRamWkHX/OmJGAOe6/997undweDoeuOf91Xbdu2/Y18vppPr9c/71cfnxtt8+R+6OfN7nrDWBswtz8gPv9/u13xsxms93xeHyJvH5YrS73N4vF++dm8xi5P/p5U7u+zc7AySEVXILS398Ajozw1JC69/eBcBZ5CEM45BqGRhzCEE4imB3ArJEeGolq7wfh7AykwsklAMKx7H+9xEAYwkkEswMIYQiHKujp8tX1jKXCySUAwhCGcOkuHoQhDGEIn0egT1+5WvlKFs5GN0Y6aZyvB5AKU2EqTIWp8GkE+mywHLw8NLRK/vX7KWclVZkKU2EqTIWpMBXu7UKoMBW2wbK0iEAYwhCGcOK0Z/lihCwsC8vCpddACEMYwhBWzlLOUs4aq6tIhakwFabCVJgKU2Eq3DQV91g7L5zdvaWppKmkqVTaB0IYwhCGsL7waQQqGtmxDysy0ox0UkWzA6girSKtIl3axkAYwhCGsKaSppKmkqZSzSwuC2ejnIp0Mksz0ow0I81IM9KMNCPNSDPS//PZW4w0I21rR2kfCGEIQxjCdmfZndV3d5qKtIp0UkWzA6girSKtIl3axkAYwhCGsKaSppKmkqaSppKmkv9o063DD8VlpBlpRpqRZqQZaUaakWakGWlGmpHe3fspHvrC2bam3VnJvrIsLAvLwrKwLCwLy8KysCwsC8vCsrAsHG4rZrNs9n5ZWBZ2Uql0lIMwhCEMYYcNHTZ02HCsagwVpsKjqvAPgiEsrUCkIfYAAAAASUVORK5CYII=' />"}],"value":"[200 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5001d894 \"comportexviz.server.notebook$eval28204$fn$reify__28208@5001d894\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>201</span>","value":"201"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAACWCAYAAABJoA8EAAACTklEQVR4Xu3dMW7CQBSEYUyXLgoNt0lOwBU4SkLFNThCjkAOZLpESoOjCEJBtTAR9uKPjsICPXl2/vdm1252u103OX66rls1TfPq+6ECRfVQwPCGadv27e+Om06n2/1+/+L7oQIl9WjSO/BpNjtJ/nux+PjabJ5HtQQoYM8SflguT0vAZD5//1yvH8e0BMQSHr1rpxIefQG5cEYdJJw2DiTcswuPHbxJmIRDCaYF5MJcuN/xGxcOlwASJmESrnp+SMIkTMIkfKxASQhzb72zXjht5YA0kO41xyZhEg4lmBYQSANpIA2kgfQFGwrTRffergfSoYtzYS7MhbkwF+bCxccazimCC3Nh46yqTYSESZiESXjMpz31wnphvXDVayAJkzAJk7BxlnGWcVbpY0r+OxXkwlyYC3NhLsyFuTAXLntW1dCOSdgjnXKhUEmoJFSqmgNJmIRJmITlwuUPXR0ayPb9f4A0kA5dNC2gibSJtIl01RhDwiRMwiQsVBIqCZWESkKlsncQpa3T0K43kQ57aSANpIE0kAbSQBpIA2kgDaSL3mY6tE6g7/8DpIE0kAbSQBpIA2kgDaSBNJDuVhe/LxlIA2kgDaSBNJAG0kAaSANpIA2kt7c+ueSkUpqp2J1ld5bzwlW3ciRMwiRMwo78O/L/W4FrXnEOpIF06KJpAYVKQiWhUtUYQ8IkTMIkLBeWC8uF5cLX9aK3DpHOf08vnLZyJtJhLw2kgTSQBtJAGkgDaSANpG2wtMHSBksbLC8+q5b2sun1emG9cK+7s34ARV0srd+krqgAAAAASUVORK5CYII=' />"}],"value":"[201 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x238c8972 \"comportexviz.server.notebook$eval28204$fn$reify__28208@238c8972\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>990</span>","value":"990"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAACWCAYAAABJoA8EAAACQ0lEQVR4Xu3dPW7CQBRFYdtd2tCwnKyALbCUwI6yBLIg6NMwKYAUUDDWteSM+dKhyJHyNMfn/dn0p9OpdNefUsq+7/tPny8RqIqHAIYH5ng87m4nbhiGw/l8/vD5EoGaePROYHgCBTAMIISzWxaE06wDwhCeNWuAMIRDBNMAsjALz1u7s3B4C4AwhCHcdP8RwhCGMISvEajp4C6t460WTks5ibREWjuraYlAGMIQhvAr79aohdXCauGm74EQhjCEIaydNWInOG3/LO16tbBaWC3ctEQgDGEIQ1g7q/7JnKXNddP/x1w4TexZmIVZmIVZmIVrnw++tzYLs3Bo0TSAhkqGSoZKTacxEIYwhCFsLmwuXP2urPu8UTsrTMRZmIVZmIVZmIVZuPaVnWn76v56FmZhFmZhFmZhFmbhuvc2p9tUU19vLpzmhdpZ2lm2s5rOAyEMYQhD2IKlBUsLlnOVcizMwizMwizMwizMwpVfxZj276a+3lzYXNhcuOk0BsIQhjCErXZY7bDaMVclwsIszMIszMIszMIsbMGyK6Xs+6kblv/975kLmwubCzedB0IYwhCGsNUOqx1WO+Yq5ViYhVm4JQu/r1Z/HfCfzebb05ojmxUPATRUGjdUettudzdiuvX6ywkceQIf2nUszMIs3JKFIZze87w7a5x1n73lg4XTE8nCLMzCLKwjrSOtI60j7WHDbhiGw7PEc2m/l0hLpMNEOA2gjnRWG0M4PYFq4fAWAGEIe1Kp6WYChCEMYQi/8MOGv5/lKe4bAjhwAAAAAElFTkSuQmCC' />"}],"value":"[990 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x36ad7fa3 \"comportexviz.server.notebook$eval28204$fn$reify__28208@36ad7fa3\"]]"}],"value":"([200 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5001d894 \"comportexviz.server.notebook$eval28204$fn$reify__28208@5001d894\"]] [201 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x238c8972 \"comportexviz.server.notebook$eval28204$fn$reify__28208@238c8972\"]] [990 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x36ad7fa3 \"comportexviz.server.notebook$eval28204$fn$reify__28208@36ad7fa3\"]])"}
;; <=

;; **
;;; In fact, this 201 bit representation makes the bold statement that 20 is equally different from 40 as it is from 90. A more information-rich approach would have 1000 bit representations within a total array of 2000 bits. It'd be equipped to handle all values between 0 and 100, to one decimal place.
;; **

;; @@
(show-example-linear-encoder 0 1000 1000 1 [400 900])
;; @@
;; ->
;;; Showing (e/linear-encoder [3000] 2001 [0 1000])
;;; 
;; <-
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>400</span>","value":"400"},{"type":"html","content":"<img src='data:image/png;base64,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' />"}],"value":"[400 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x29ba113c \"comportexviz.server.notebook$eval28204$fn$reify__28208@29ba113c\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>900</span>","value":"900"},{"type":"html","content":"<img src='data:image/png;base64,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' />"}],"value":"[900 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4f4cee21 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4f4cee21\"]]"}],"value":"([400 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x29ba113c \"comportexviz.server.notebook$eval28204$fn$reify__28208@29ba113c\"]] [900 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4f4cee21 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4f4cee21\"]])"}
;; <=

;; **
;;; ##### Settle down
;;; 
;;; You might be screaming right now. Numbers can be encoded much more effectively. Computers do it through Base 2 counting. Biology does it through spike rates.
;;; 
;;; Base 2 isn't appropriate for HTM encoders because it provides jittery overlaps in bits of nearby numbers. You'd be in a world where 1 is closer to 256 than it is to 2.
;;; 
;;; But you're totally right about spike rates. The neocortex uses spike rates to encode numbers like orientation (the measure of the angle) and color (e.g. the ratio of red to green). HTM intentionally doesn't model this. The reasoning: the neocortex uses spike rates, but not for everything, so let's see how far we can get without them. There's a risk of getting caught up in closely mimicking low-level details, never reaching the point of characterizing higher fundamental concepts like temporal pooling.
;; **

;; **
;;; #### Scalar encoding: Probabilistic
;; **

;; **
;;; With the moving window approach, we represented a value in terms of its nearby values. In some sense:
;;; 
;;; - 9 is @@ \\{7, 8, 9, 10, 11\\} @@
;;; - 10 is @@ \\{8, 9, 10, 11, 12\\} @@
;;; - etc.
;;; 
;;; 9 and 10 share 4 bits.
;;; 
;;; The moving window approach has two weaknesses:
;;; 
;;; 1. You have to know the range of numbers that you'll ever see.
;;; 2. The @@ active\\_bits @@ count grows with the radius.
;;;    - If an encoding illustrates the relation between 42 and 86, then _a lot_ of bits are going into representing 42. This means lots of synapses, which means longer training periods and slower HTMs.
;;; 
;;; You can "solve" both of these problems by relying on randomness / probability. ([Actually](https://twitter.com/Actually_HN), you don't have to use randomness to solve Weakness 2, but for this discussion we will because that's what the Coordinate Encoder does. For an arguably better sampling strategy, see the [next section](#a_third_way).)
;;; 
;;; The coordinate encoder addresses these problems as follows:
;;; 
;;; - To solve Weakness 2, sample the numbers within the radius, rather than use all of them.
;;;   - Strategy: Hash each number and take the top few results. That way, we're likely to choose many of the same numbers as other nearby encodings.
;;; - To solve Weakness 1, simply hash each number and coerce it into an integer modulo the number of bits.
;; **

;; @@
(def example-enc2
  (e/coordinate-encoder [100] 5 [1] [2]))

(for [v [9 10]]
  [v
   (into #{} (p/encode example-enc2 [v]))
   (viz example-enc2 [v])])
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>9</span>","value":"9"},{"type":"list-like","open":"<span class='clj-set'>#{</span>","close":"<span class='clj-set'>}</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-unkown'>65</span>","value":"65"},{"type":"html","content":"<span class='clj-unkown'>32</span>","value":"32"},{"type":"html","content":"<span class='clj-unkown'>56</span>","value":"56"},{"type":"html","content":"<span class='clj-unkown'>97</span>","value":"97"},{"type":"html","content":"<span class='clj-unkown'>67</span>","value":"67"}],"value":"#{65 32 56 97 67}"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAA00lEQVRYR+2WsQ3CQAxF4+vSIdJkmzBBVsgoQJU1MgIjwEB3HUg0dzSQwqmsb0cWuutOcvH1n/63KaVUmu8rpVyJ6Ozq715gjPHycyyEcM85nzz9yb2D7gX+HeJj160pf4/j47Usg2nqpYh3FyhF3E7Tmvqm72/PeT5Ypr6mGN5MUsR7F3lFXBGjDsC9KS1qqWBYoHWK0WJXTzHsGL/otRGrC9RGjCLli0AdMQ+R1FE+71+gNmKOSIqcz5s7KO3Nzbx2imFBvGasEaPnWUUMI/eO+ANxp+qq551nNQAAAABJRU5ErkJggg==' />"}],"value":"[9 #{65 32 56 97 67} #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4a89b09 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4a89b09\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>10</span>","value":"10"},{"type":"list-like","open":"<span class='clj-set'>#{</span>","close":"<span class='clj-set'>}</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-unkown'>65</span>","value":"65"},{"type":"html","content":"<span class='clj-unkown'>32</span>","value":"32"},{"type":"html","content":"<span class='clj-unkown'>97</span>","value":"97"},{"type":"html","content":"<span class='clj-unkown'>83</span>","value":"83"},{"type":"html","content":"<span class='clj-unkown'>67</span>","value":"67"}],"value":"#{65 32 97 83 67}"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAA1klEQVRYR+2WsQ2DMBBFsbt0UWjYJkzACoySpGINRmCEZCC7S6Q0dpqEgubrZH7uhKCzTPF0z8/gYoy5+j4555tz7mJqbR4whHD9Tcx7f08ptZbWzvwEzQNuTvGprufK3133eI3jmVq9VPHfAaWKD30/V181zfQchiOzevWKoRGp4rW/NBBQqnjtixwdGXXF0Ii2YgiorRgdGbpiGAH6vWMrLgZkK0aVqiuGEWgrLgZkK0YK0b65ipdR2QdkK5ZWvHyfPsHiSNgXdTEgWzGqFO3vijev+AP4Teqq5A90UQAAAABJRU5ErkJggg==' />"}],"value":"[10 #{65 32 97 83 67} #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4026c2ef \"comportexviz.server.notebook$eval28204$fn$reify__28208@4026c2ef\"]]"}],"value":"([9 #{65 32 56 97 67} #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4a89b09 \"comportexviz.server.notebook$eval28204$fn$reify__28208@4a89b09\"]] [10 #{65 32 97 83 67} #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x4026c2ef \"comportexviz.server.notebook$eval28204$fn$reify__28208@4026c2ef\"]])"}
;; <=

;; **
;;; Now:
;;; 
;;; - 9 is @@ \\{65, 32, 56, 97, 67\\} @@
;;; - 10 as @@ \\{65, 32, 97, 83, 67\\} @@
;;; 
;;; They still share 4 bits.
;;; 
;;; Humans can't make any sense of these encodings. And good luck writing code that decodes it. It uses the hash of the input, so you can't work backwards, so you'd have to brute-force the infinite domain of possible numbers.
;; **

;; **
;;; ##### <a name="the_random_casualties" /> The random casualties
;;; 
;;; For dramatic effect, we're only using 100 bits.
;;; 
;;; Let's start without sampling. Include all bits within the radius.
;;; 
;;; Without this substitution, overlap only happens when the values are nearby, +- 4. With the substitution, does that stay true?
;; **

;; @@
(defn spacer
  [v]
  (subs (str v "   ") 0 3))

(print (spacer ""))
(doseq [j (range 0 100 9)]
  (print (spacer j)))
(println)
(doseq [i (range 0 100 9)]
  (print (spacer i))
  (doseq [j (range 0 100 9)]
    (let [overlap (set/intersection (into #{} (p/encode example-enc2 [i]))
                                    (into #{} (p/encode example-enc2 [j])))]
      (if (pos? (count overlap))
        (print (spacer (count overlap)))
        (print (spacer "")))))
  (println))
;; @@
;; ->
;;;    0  9  18 27 36 45 54 63 72 81 90 99 
;;; 0  5        1  1        2  1           
;;; 9     5                 1  1  1     1  
;;; 18       5     1              1        
;;; 27 1        5  1           1           
;;; 36 1     1  1  5           1  2        
;;; 45                5                    
;;; 54                   5           1  1  
;;; 63 2  1                 5     1     1  
;;; 72 1  1     1  1           5           
;;; 81    1  1     2        1     4     2  
;;; 90                   1           5     
;;; 99    1              1  1     2     5  
;;; 
;; <-
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; **
;;; The world becomes a little bit weird. The overlap between 36 and 81 is the same as the overlap between 36 and 39. Meanwhile 36 and 41 correctly have no overlap.
;; **

;; @@
(let [i 36
      enc example-enc2]
   (doseq [j [39 41 81]]
     (let [i-bits (into #{} (p/encode example-enc2 [i]))
           j-bits (into #{} (p/encode example-enc2 [j]))]
       (println i i-bits "and" j j-bits "overlap" (set/intersection i-bits j-bits)))))
;; @@
;; ->
;;; 36 #{82 14 79 10 42} and 39 #{77 29 34 82 79} overlap #{82 79}
;;; 36 #{82 14 79 10 42} and 41 #{77 29 34 19 96} overlap #{}
;;; 36 #{82 14 79 10 42} and 81 #{62 14 10 67} overlap #{14 10}
;;; 
;; <-
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; @@
;; In case you want to see those.

(for [i [36 39 41 81]]
  (viz example-enc2 [i]))
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAA4ElEQVRYR2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGRkZ4fIg9TTnD3oHvn37tgEWQkxMTAc44+PhfAYJiQ1f29sFkOX//fvnQE8+I3oI0jzKSE0ig96B6FFM7ygkZB/FUUxprieon9IoJmgBgTRHUD+lUUxpriekn+IopnmupzSKae5ASqOYUC6kVB4jigkmWlILWkrVj9bFFNbdo7mY4lw+5HIxxT6mNNei6x8tqEdzMbXTFKnmjeZiUkNsNBdTmGvRm2d0r4tJbs7Ru6Am2YH0zsWEOknoUQwAHUrtaXzP5DYAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAA1klEQVRYR+2XsQ2DMBBFOXd0KDRsQyZgBUZJUrEGIzACGcjuiJTGTpOghOb0dZF9xdFZgPXl9/8/mUIIqXo/KaUbEV1UrdUL9N5fPyfmnFtjjGdNa1J/guoFGmKmFU5tu7fGcxjuj3nuf1qkNGJWYGnE9TjuLVJ13bJNU/PdIpZi8WQqjZgbDDBi1tT/nuVoirMLRBFzqeOQoe9hxGLToxZAEWcXiCJGEUm/N8RiSxjiQ2rhHs2dYlhgbsRo0VuKLcWwqaWXpOP/XIqLC+RSjKaOm73ofi/o0O1p7EDx5AAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAA30lEQVRYR2N89+7dfwYo+P//fyMjI2P9oOIPege+ffu2ARZiTExMB/79++cwmPiMgz4EB70DR6OY0lJhNIopLBVGczHFNdOwy8VCwsLwuvqnn9/BbwsW2NO07iY1F9PdgaRGMWd8PLzuZpCQ2PC1vV2AlnX3aC4ezcUUhwCJdTF6Jhx0aRDDgaTmYlq3uNFLCbqHIMnlKKkFNaVpkmQH0juKSS3o6R7FJMcAvaOYZAfSO4pJLQUwohg9EbNv2gRvTtGleYVesKNH8aBzIHoUo+cyhhcvAmDNKXo0r9CTAAAFwuqqnmOl+wAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAA0ElEQVRYR2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGRkZ4fIg9TTnD3oHvn37tgEWQkxMTAc44+PhfAYJiQ1f29sFkOX//fvnQE8+I3oI0jzKSE0ig96B6FFM7ygkZB/No5jiUoHWUUyxA2kdxZSWCjSPYopLBVpHMcUOpHUUE8qlhOQxopjiRE1qQUxI/WhdTGHdPZqLR14uptjHhHIlqfKjBfVgz8UUF/y0jmKKHUjruni0uTXgxQ6to5hQc4qQ/GhdTHESGexRDADoG+5prScvigAAAABJRU5ErkJggg==' />"}],"value":"(#object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6884248d \"comportexviz.server.notebook$eval28204$fn$reify__28208@6884248d\"] #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x112696cc \"comportexviz.server.notebook$eval28204$fn$reify__28208@112696cc\"] #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x415b1522 \"comportexviz.server.notebook$eval28204$fn$reify__28208@415b1522\"] #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x17d22820 \"comportexviz.server.notebook$eval28204$fn$reify__28208@17d22820\"])"}
;; <=

;; **
;;; Also, a single representation's bit count can be diminished when bits collide. Here are some numbers that are represented by 4 bits instead of 5.
;; **

;; @@
(dotimes [i 50]
  (let [bits (p/encode example-enc2 [i])]
    (when (not= 5 (count bits))
      (println i "is represented by" bits))))
;; @@
;; ->
;;; 11 is represented by (67 83 65 97)
;;; 12 is represented by (67 83 65 97)
;;; 13 is represented by (53 67 83 65)
;;; 14 is represented by (53 67 83 43)
;;; 33 is represented by (37 91 42 10)
;;; 34 is represented by (91 14 42 10)
;;; 35 is represented by (79 14 42 10)
;;; 43 is represented by (7 96 34 19)
;;; 
;; <-
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; **
;;; When we add in sampling, we usually get rid of this "diminished count" problem, but we create other quirks.
;;; 
;;; As shown below, with radius 4, the representation for 29 should overlap with 34, 35, 36, but if you only take 3 samples, it doesn't.
;; **

;; @@
(defn v->bits2 [v]
  (let [enc (e/coordinate-encoder [100] 3 [1] [4])]
    (into #{} (p/encode enc [v]))))

(print (spacer ""))
(doseq [j (range 25 40)]
  (print (spacer j)))
(println)
(doseq [i (range 25 40)]
  (print (spacer i))
  (doseq [j (range 25 40)]
    (let [overlap (set/intersection (v->bits2 i)
                                    (v->bits2 j))]
      (if (pos? (count overlap))
        (print (spacer (count overlap)))
        (print (spacer "")))))
  (println))
;; @@
;; ->
;;;    25 26 27 28 29 30 31 32 33 34 35 36 37 38 39 
;;; 25 3  3  3  2  2  2  2  2  1                    
;;; 26 3  3  3  2  2  2  2  2  1                    
;;; 27 3  3  3  2  2  2  2  2  1                    
;;; 28 2  2  2  3  3  2  2  2  1                    
;;; 29 2  2  2  3  3  2  2  2  1                    
;;; 30 2  2  2  2  2  3  3  3  2  1  1  1  1  1     
;;; 31 2  2  2  2  2  3  3  3  2  1  1  1  1  1     
;;; 32 2  2  2  2  2  3  3  3  2  1  1  1  1  1     
;;; 33 1  1  1  1  1  2  2  2  3  2  2  2  2  2  1  
;;; 34                1  1  1  2  3  2  2  2  2  1  
;;; 35                1  1  1  2  2  2  2  2  2  1  
;;; 36                1  1  1  2  2  2  2  2  2  1  
;;; 37                1  1  1  2  2  2  2  2  2  1  
;;; 38                1  1  1  2  2  2  2  2  3  2  
;;; 39                         1  1  1  1  1  2  3  
;;; 
;; <-
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; **
;;; This sampling also often causes nearby numbers' representations to be indistinguishable.
;; **

;; @@
(let [enc (e/coordinate-encoder [1000] 10 [1] [50])]
  (doseq [v [43 44]]
    (println (format "Representation for %s:" v)
             (into #{} (p/encode enc [v])))))
;; @@
;; ->
;;; Representation for 43: #{204 420 792 565 697 535 667 928 628}
;;; Representation for 44: #{204 420 792 565 697 535 667 928 628}
;;; 
;; <-
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; **
;;; See [the appendix](?path=hotgym_appendix_2015_09_19.clj), where I spend lots of time characterizing these random events through math and trials. I don't think the analysis is finished, but it's pretty far along.
;; **

;; **
;;; #### Whining
;; **

;; **
;;; So far, none of these are super-appealing.
;;; 
;;; - With a moving window, we have to decide between showing limited interrelations and having huge representations.
;;; - With a probabilistic encoding, we have to convince ourselves that everything will be okay.
;;; 
;;; Can we do better?
;; **

;; **
;;; #### <a name="a_third_way" /> A third way
;;; 
;;; I like sampling. I don't like unnecessary randomness. I'm okay with specifying a min and max.
;;; 
;;; Let's use a sampling moving window. Don't use the Coordinate Encoder's random sampling. Instead, use multiples of some known number. Something like:
;;; 
;;; - For linear similarity, select the current number and the nearest few multiples of @@ \frac{2 * radius}{active\\_bits - 1} @@.
;;; - For varying precision, choose multiples of larger and larger numbers as you move away.
;;; 
;;; I present: the Sampling Linear Encoder.
;; **

;; @@
(defn middle-out-range
  "By example:
  Given 7.2, returns (7, 8, 6, 9, 5, 10, ...),
  Given 7.7, returns (8, 7, 9, 6, 10, 5, ...)"
  [v]
  (let [start (-> v float Math/round long)
        rounded-down? (> v start)
        up (iterate inc start)
        down (iterate dec start)]
    (if rounded-down?
      (interleave down (drop 1 up))
      (interleave up (drop 1 down)))))

(defn multiples-within-radius
  [center radius multiples-of]
  (let [lower-bound (- center radius)
        upper-bound (+ center radius)]
    (->> (middle-out-range (/ center multiples-of))
         (map (partial * multiples-of))
         (take-while #(<= lower-bound
                          %
                          upper-bound)))))

(defn into-bounded
  "Move items from `from` to `coll` until its size reaches `max-size`
  or we run out of items. Specifically supports sets and maps, which don't
  always grow when an item is added."
  [coll max-size from]
  (loop [coll coll
         from from]
    (let [n-remaining (- max-size (count coll))]
      (if (and (pos? n-remaining)
               (not-empty from))
        (let [[taken untaken] (split-at n-remaining from)]
          (recur (into coll taken)
                 untaken))
        coll))))

(defn sampled-window
  "Place a bit in the center.
  Distribute bits around the center until we've used half of the remainder.
  Double the density. Distribute again until we've used half of the remainder.
  Double the density. ...
  Continue until all active bits are distributed or all bits are active.

  Strategically choose bit positions so that the intersections between
  various ranges will select the same bits."
  [center n-bits target-n-active bit-radius]
  (loop [chosen #{center}
         density (/ (- target-n-active (count chosen))
                    (* 2 bit-radius)
                    2)]
    (let [remaining (- target-n-active (count chosen))
          multiples-of (long (/ 1 density))]
      (if (and (pos? remaining)
               (pos? multiples-of))
        (let [half-remaining (quot remaining 2)
               n-take (if (or (odd? remaining)
                              (odd? half-remaining))
                        remaining
                        half-remaining)]
          (recur (->> (multiples-within-radius center bit-radius multiples-of)
                      (filter #(<= 0 % (dec n-bits)))
                      (into-bounded chosen (+ n-take (count chosen))))
                 (* density 2)))
        chosen))))

(defrecord SamplingLinearEncoder
  [topo n-active lower upper radius]
  p/PTopological
  (topology
    [_]
    topo)
  p/PEncoder
  (encode
    [_ x]
    (if x
      (let [n-bits (p/size topo)
            domain-width (- upper lower)
            center (-> x
                       (max lower)
                       (min upper)
                       (- lower)
                       (/ domain-width)
                       (* n-bits)
                       long)
            bit-radius (* radius 
                          (/ (p/size topo) domain-width))]
        (sampled-window center n-bits n-active bit-radius))
      (sequence nil)))
  (decode
    [this bit-votes n]
    (let [span (double (- upper lower))
          values (range lower upper (if (< 5 span 250)
                                      1
                                      (/ span 50)))]
      (->> (e/decode-by-brute-force this values bit-votes)
           (take n)))))

(defn sampling-linear-encoder
  "A linear encoder that samples the surrounding radius, rather than
  activating all of it. Sampling density decreases as distance increases.

  * `dimensions` is the size of the encoder in bits along one or more
    dimensions, a vector e.g. [500].

  * `n-active` is the number of bits to be active.

  * `[lower upper]` gives the numeric range to cover. The input number
    will be clamped to this range.

  * `radius` describes the range to sample.

  Recommendations:

  * `lower` and `upper` should be `radius` below and above the actual
    lower and upper bounds. Otherwise the radius will extend off the
    number line, creating representations that behave a bit differently
    from the rest."
  [dimensions n-active [lower upper] radius]
  (let [topo (topology/make-topology dimensions)]
    (map->SamplingLinearEncoder {:topo topo
                                 :n-active n-active
                                 :lower lower
                                 :upper upper
                                 :radius radius})))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;comportexviz.notebook.hotgym/sampling-linear-encoder</span>","value":"#'comportexviz.notebook.hotgym/sampling-linear-encoder"}
;; <=

;; @@
(for [i [0 1 2 3 41 42 43 44 127 128 129 130 200 201 202]]
  (viz (sampling-linear-encoder [256] 17 [0 256] 64) i
       {:display-mode :one-d}))
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA5klEQVRoQ+2aQQ6EIAxFgVN4Ks40cyOPoAeC9cyqJiomM2ykCQTMc0losPT/32+sFWPEnM/X+9VmCx/vl7TDTNOc7xg4JMZ4pS8ib5sthBBeKX3n3KJJv9tTFC+mCFFgTHFKmxByKeULdSm9MTBWemNtMPakU56EMeoCXw6LdtPCKQADX/rEGHXpsy4KihFCKWvLOBgDY2CsqlOCYlAMikGxet9iKAwKg8KgMCjMnf+VqCVqiVqilqglavk/7ZINt9AsaBY0C5oFzYJmQbNIc587G+iVv3OwWAWsAlYBq4BVwCoMaxU2gnMsYJ+0S6IAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA5ElEQVRoQ+3aSw7CIBAGYNpT9FScSW/kEeqB2rWuaKLWhSyKaIkxX5eEyRRm/kfIdCmEFB7fNcZzly1cYhzXHWEYTvmO7ZB5np9ZUkrHLluYpumwZun7fsx3bIdUnKXNj1VkcZYGDaMu72K/4sbahMALvNz1g76UyCu8wAu8lNPFP+GljSK3yaIuv8ljbaovi+rvrWJ6TI/psXKnBC/wAi/wst87DIbBMBgGw2CYkpdebIktsSW2xJbY8nUM5xtTN/SFvtAX+kJf6At9WWdYb2ggrx/P9HIX3AV3wV1wF9zFDu5iASolLGDJ1ytnAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA50lEQVRoQ+3aQQ6CMBCF4cIpOBVn0ht5BDwQrHVVExUT7aZOAqnmY9kw0Hbm/W8C7XJKOT2v6zieu2LgMo7TekcahlO3LMsrJOd8LAfmeT6sIX3fT+VDf/gtgeUHQvbJS2BigRBr+VYvgU0OhMiLvDwwXYlxNbaDVwY2ORBC+7RP+/Xo+ye9BHDRbIi8tMmxZgvGxBTM1sanxtSYGqtvruiFXuiFXrb7DIUwCIMwCIMwNb9r0RIt0RIt0RItPw/7FGd7mAWzYBbMglkwC2axHnu9q4FXvh8D1ipoFbQKWgWtglahjVbhBoJBLGAKHaE3AAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA2UlEQVRoQ+3aSw7CIBAGYOgpeirOpDfyCPVAdK0rTFRc2IUJSkPM1yXppND5X2knlhBKeF7XlM5xs3BJaal3hHk+xXVdXyWllON2Ied8qCXTNC0tJQ0bayjZ5ywNT3GWHTCmL2NyX1/05eEf/fwFxmAMxrpSTIYZM8M0SN+wJf+EsWFfso0h8oBeCZZgCZZdIxyKoRiKodhQ32GIElEiSkSJKHX690pgCSyBJbAElsC+D/v8YraHv/AX/sJf+At/4S91UvbOBvb69eSwdCFdSBfShXQhXXxKFzfaXSxgl+3jQAAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA1UlEQVRoQ+3aMRKCMBSE4ZBTcKqcSW/kEfBApNYqFhhnlCZiE/CjzPBmEh77v2XYoYRQwvO6p3QdVgu3lKZ6RxjHy5BzfpWUUs7rhXmeT7Ukxjgp+fqJ6Uuf75i+6MuCNuhrIf+R9PLnZ+n2+N1ubINTOtJZNhy/2xJ9MffN/XbbQy/0Qi/0svevJBzDMRxr51i3Bt7G+hSyvugLwALs3o0ijuEYjuEYjrX8FkRLtERLtERLtPyME0oPCk8us0F2VHRW2Lgmqc1Ks9KsfFMDq8Aq/GQVHnlILGD/Kp3fAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA3ElEQVRoQ+3asQ2DMBQEUJspmMozJRtlBDKQqZPKKQiREhpovizyKC0sy/98989wuaXU0vt5lnLPm4FHKdP6RhrHW57n+TOltXbdDtRaL+uUYRgmUw5XDC59njG4wGWRNtK3R/nPxBd7Ocr9mIrFrBJje860l5iKxawCl6Pch0ufFYMLXBYH676/x8HiC77gS28XXn7sn1kZo8lWccZ86+tN+bESK7ESK/1/6ef2SpNpMk2myTSZJv+mw4TBZOGW3iAKKAkpO7oGY/VKvVKv/GIDq8AqsAqcUpxRfAHRZCxgbcABNQAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA2klEQVRoQ+3aQQqDMBSE4egpPFXO1N6oR7AHStbtKi6shdaNBpQgn8vgA5Px/W8I05UQSvg87xif3WrhFeO4vBGG4dHlnL8lpZT7eiGldFtK+r4flew+Mbq0+Y/RhS4z2qBvC/mv1C/2ckLvVxxyRUmzHuZKe2n2kCs+jC4n9D5d2jxkutCF6T3U85svbbYYXegCfYeir8JdKGmzK+lCF7RES3fjW+7G0RIt0RIt0RIt/zM1IjQSRPNsEKCSH5O4W+KEZqVZaVb+dAOrwCqwCpwSo8goMor7jOIEKY8sYEVjTaUAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA0ElEQVRoQ+3asQ3DIBQEUPAUnoqZko0ygjMQ1ElFCodIiRunQELouUT+Bdy/+6evizWEGt7fM6V7PBw8UtraH2Fdb7GU8imptV6PBznnSytZlmVT8veLwWXMHoMLXHZpI31nlH8mvrgL7vfmvh7TY717bCYDPxNf4DIm9+ECl96aTMfG7DG4wKU392eaL+4yJl/gAhc6Zm1ra31ma00tqSW1pJbUklr+pl2EW2R79tkg2iTZJQvXgn5mpVlpVn6xgVVgFVgFTolRZBQZxRGN4gtCuCxgGsds3wAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA40lEQVRoQ+2ayw3DIAyGgSmYipmSjTpCMhCc05MjNaVSyqHIlawofDkiLMDwP8DxpRRx709EZt805Jyn2iOEsLQ9CDmnkIxxYA7EgBfoAoJFX14ZQF5xF99kgFXAKmAV/nBK4tzHfD9TWn3TsKW0VAVyMT64v3DjG/fCC1547jhOP3a0x46Cl2vihX1hX+Cxfhq/E15s1qIYxSZEUbJRTOyyoygmpgixyZhiYoOHsC/oPro/pu4PTn0s34D60BeDJHOSSTIi3i/i4AW8gBfwQsmmp2QDW8KWsCVsCVvClr/+t9wB7L4sYKuCBuwAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA3klEQVRoQ+2Zyw2AIBBEgSqsipq0I0vQguCspzXxlygXiHFj5HkkGMLMzuwANsYoZv9EpLPJQAihPWY454Z0Br9cIQQxCmZTDHrBLjBY+suKAO2VdHE3A6ICUYGo8CApiTFn+J69H20yMHk/HB3INE3P+YUTX70HXvTCdcdW/cTRnDiKXr6pF3iBF3ws38b/pBedvbBKqcPoIKbzYvWnveggprMKvJSqEl6+iRi81MyLDvusUlpj9JdSxKixmhFDLzWzj/Zhn1u4/Fs49IJe0At64e015+0Vt8QtX3bLBU0zLGBjioVgAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA4klEQVRoQ+2awQ3DIAxFgSkyFTO1G3WEZCA4tydHCiVSywX5gCx4OSIssJ3v/+3E55zFfR8RefpmIaX0qDtCCHu7A5PfEBIxXpiCGPBCuaDAwi9XBKBX1MV/MUAqIBWQCiglhCJCcYhQFOfuZvUT4+GbhXeMe1Vsbtte9PtMSNYdEIEXm+NB8kJeSlWire5pq2fCyxhfzJ6iuJjCRPFhaMwpiospTGbyReG+WRPyAu/D+/2yB7zYxIvZArv4xcALeIFf4BemCj1ThcXJAvchC8iinyzAC3gBL+AFcYW4svEn5AmtmSxgi2pyKwAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA2ElEQVRoQ+3ZQQ6CMBQE0NJTcKqeSW/kEfBAsNZVTURMtBu6IaR9LJsSwvyZ6fzfYVmWHD5Pzvk6FAvzPF+2HTHGqdzhlV8IIYYwq2LohV0wWOfLGwHHq3TxbwaigqggKkhKgqKgeEhQzCF8m9VnSvehWHikNG2JLYzjTb9vQtLvgIhezjkeVBd1WV1JW72nrW5JL8f8i6/UOgzEahFz+1aL2DEcUxd1aSNd0Estk2m/Z8Topefq035t9emlFjEcg1gb2RKTMRmTjaBN4PdM4Lklt+SWJ3PLF2ZDLGDqgvhuAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA1ElEQVRoQ+3asQ2DMBQEUJspmMozJRtlBDKQqZPKSCFECm5ojCz0KC1T8O/f+b6POM9zCd+nlHKP1ULO+bbtGIZhqnd45b+EKqZhVsbgC7kgsM6XTwUcr9zFXgxYBVaBVeCUGEVGkVFkFBlFPtktnEvI3aWsMcGYYEwwJhgTzhoTSgi/UOed0jNWC6+Ups2whnF8yMXOSBLh0mdcCxe4rGoo5jgSc+ALvuDLcbnAF3xpzZcr9ZhvwZfWfLnSv6NX4gtc+uQ+XODSWpPpWJ89BpcucVkAXDcsYFCcUZIAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA20lEQVRoQ+3awQ3DIAwFUMIUmYqZ2o06QjpQOLcnKjVNpSaXiEOFwssRgYTt+Pt/myHnXMLnK6Vch93CPM+XdUeMcdrvcOTXhTzmh1kyRr6ACwCrvrw9oLxiF1swQBVQBVQBU0IUEUVEEVFEFPFkXThNyE1TlkwgE8gEMoFMIBP6lgklhO+w7ZnSfdgtPFKaViERxvFmXvmPCa+4tDlGFxdxWdDQ+OnI+Em+yBf5chwuzpQvndvSrPnNXqzi8eSZbKkwv9kj4qLuq/t91v1mQaniYnAMjsGxwzj2ArydLGCld3E6AAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA3UlEQVRoQ+3aQQ6DIBAFUPQUnooztTfqEeyBcN2uMCm1SXWDG2L0sSQQwnz/8P+M3TRNOXxHzvnebSZSSrdlRd/343aFLf8hFDEfTGEMvkgXEqz35RMBzyt1sU4GpAKpQCpQSoQioUgoEoqEIp2sCqcIuSrKsglsApvAJrAJbMK1bUIO4ddse8f47DYTrxjHxUiEYXjoV7bo8MLlmG10uMClZEPtp5r2E77gC77Up4sz8cVd9nK/TcTanNLmT8gz3aVNxNqcApe93IfLMSMGF7gUBas+VuP48OXCfJkBFMgsYB2LnhAAAAAASUVORK5CYII=' />"}],"value":"(#object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x6a52056f \"comportexviz.server.notebook$eval28204$fn$reify__28208@6a52056f\"] #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x2e4576a8 \"comportexviz.server.notebook$eval28204$fn$reify__28208@2e4576a8\"] #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xac49fc7 \"comportexviz.server.notebook$eval28204$fn$reify__28208@ac49fc7\"] #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x5a3107bd \"comportexviz.server.notebook$eval28204$fn$reify__28208@5a3107bd\"] #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x203208e9 \"comportexviz.server.notebook$eval28204$fn$reify__28208@203208e9\"] #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x2547ca71 \"comportexviz.server.notebook$eval28204$fn$reify__28208@2547ca71\"] #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0xb5343b9 \"comportexviz.server.notebook$eval28204$fn$reify__28208@b5343b9\"] #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x598db181 \"comportexviz.server.notebook$eval28204$fn$reify__28208@598db181\"] #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x69488f19 \"comportexviz.server.notebook$eval28204$fn$reify__28208@69488f19\"] #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x72c7c403 \"comportexviz.server.notebook$eval28204$fn$reify__28208@72c7c403\"] #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x63980f69 \"comportexviz.server.notebook$eval28204$fn$reify__28208@63980f69\"] #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x57adbb22 \"comportexviz.server.notebook$eval28204$fn$reify__28208@57adbb22\"] #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x7a30c689 \"comportexviz.server.notebook$eval28204$fn$reify__28208@7a30c689\"] #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x369be210 \"comportexviz.server.notebook$eval28204$fn$reify__28208@369be210\"] #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x15db50ed \"comportexviz.server.notebook$eval28204$fn$reify__28208@15db50ed\"])"}
;; <=

;; **
;;; Whoa, it worked!
;;; 
;;; _(I ran some trials in [the appendix](?path=hotgym_appendix_2015_09_19.clj).)_
;;; 
;;; This "use multiples of known numbers" strategy would also work with 2D and 3D, so you could build a coordinate encoder on top of this.
;;; 
;;; Maybe our consumption encoder could have a radius of 25.6, because powers of 2 are good, and it's fair to say that numbers separated by 51.2 are completely dissimilar in this context.
;;; 
;;; We'll use a range of @@ [-25.6, 125.6] @@ with a little extra to allow for 1536 bits, and probably use 17 active bits.
;; **

;; @@
(defn show-example-consumption-encoder
  [n-buckets n-active-bits radius samples]
  (let [n-bits (+ n-buckets (* 2 10 radius))
        start (double (- 0 (/ radius 10)))
        end (double (+ 100 (/ radius 10)))
        enc (sampling-linear-encoder [n-bits] n-active-bits [start end] radius)]
    (println (format "Showing (sampling-linear-encoder [%s] %s [%s %s] %s)"
                     n-bits n-active-bits start end radius))
    (for [i samples]
      [i (viz enc i)])))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;comportexviz.notebook.hotgym/show-example-consumption-encoder</span>","value":"#'comportexviz.notebook.hotgym/show-example-consumption-encoder"}
;; <=

;; @@
(show-example-consumption-encoder 1000 17 25.6 [5 10 25 50 75 95])
;; @@
;; ->
;;; Showing (sampling-linear-encoder [1512.0] 17 [-2.56 102.56] 25.6)
;;; 
;; <-
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>5</span>","value":"5"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAADkCAYAAAAPSqPOAAAEeklEQVR4Xu2dMW7bQBBFSTVBuihqfJvkBEJu4KPErnwNHSHwCZTbuBG7GEkjBoYlGlnJGYz+rkjuPDeBQNuAf/bx/5nhLtu+afrm8PVnvf75vNl8OX7u+/6+bdvvfH5V4KweCCgukN/r9fa4wpqbmx+/Hh4+HT8vFovtfr//yudXBc7p0XZdNyAMshfcshBQRHi3292B6PuIWrcwN8KfV6vBlXHt/h4B1ZjmRfjj7e2APK692LpXIME6MR1cGBceNegXR7h61y6NcPUCel3YCpbp9dpduzjC1bt2aYSrF7A0wl7k5/b9IKyWciBMkK47SFdvIiAMwiA867kzCIPwtBGuvptSOkgjoIG4VQvX3o5Sa29q4dIIV19JqAJaCKtLvPafB2F1BRKkCdLTDtKYiLFCQRiEQZh21kGBkA+pE6S1bRwEaYK06KKqgCAMwuNuhiRIi7cAEM6MMDMQ54pMEUZAp4ApwgyRfEgTpNUciAs7kU0Fx4V9yKZDMhnh8KajIhxeQBXh6K4tIxx+6KQiHF5AFeHaH92w/j4QJkiLQVgVEIRHDtLhTQQXFm8BIAzCDJVm/WwMCIsIc4aqaCIIKArIIbQiwuRAcQUioCggLgzCBGmC9EGBiEfH05FWG6q4MC7MTqVZmwgIgzAIgzCbDS9/IYn16EPt1wnSBGnRRVUBaWfRzqKdNesYA8IgDMIgTEf67f256caZD4+Pw/uGeZPhmZeXWvuFEdB5CG2686h5evrG+4d5v/D7r/hWSzk60mItTZCeWJAOt/01N8LhBMyNcLT9w3SkcWHRRVUBcyNc+xAp+6kdbDbsun7W7SQVQfXnQXhiQToc0rmDdDgBQRiEGSrNOgWAcGaEw3VT1BxoDZWeN5thKhfybQ2WwBxCmxnhcDnOWmHWdYK02A7DhUGYIE2Q5umst6ezcGGnqeDCTsE4Cl5zXYZKVjD2XgdhEGa/8KxzIAiDMAiDMFv+2fL/osAlZ3+1nKEqmggCigJyCK1WG/OMtLf2Tb+fIC0izFDpygjz6EeyYr0II2AioBfhaDuRrG0buDAuLLqoKqAXYWtJR7sOwuoK9Lowg3fRhaMhav29IAzCuPCoQyELUes6CIMwCIPwrOe6IAzCIAzCkR/NUO+BtLMKD5WYgRgmZbWzENB5CG1a+zFE+j/i1MKqiVgI04EWEbbaOdGvgzAIUwtTC1MLUwtfftY+tXDhWpgcKNbCCEiQLuryBGmCNEG6KGKla3UQBmEQrgthZiDOFc0htE7BrLOzGCL5amNcGBcWEVQFpJ3lQzb78XfhXVudC4cXUEU4umvjwqqJqAiH71irCJduF03994MwCBOk62pnhTcFL9K4sHgLwIVHroWjIb9are6OT6O9/EuMcd7zTgQEYR/Cy+Vy+88K5AxVn4mcrEAEFAXkEFoRYXKguAIRUBQQFwbhq77ZkCDtDM5ppUWQ3vuQTTviJ0EaExFNBAFFAXFhH9IgnNtEQBiErzqUAmEQ9iGXe+RAkCZIT2wF4sK+/xAQzozwX+QV0UnlqTMWAAAAAElFTkSuQmCC' />"}],"value":"[5 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x20e964e5 \"comportexviz.server.notebook$eval28204$fn$reify__28208@20e964e5\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>10</span>","value":"10"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAADkCAYAAAAPSqPOAAAEwklEQVR4Xu2dMVIjMRREx842g3Xi27AnmCtwlIXI1/AROII5EGRQkHg2AHurBLtdn5awR/+RuTQO6PpP3fqSxotpGKbh/e91HO+ft9urw+dpmm4Xi8VvPr8p8KkeCGgWyMs47g4VNqzXd0+bzcXh83K53O33+198flPgMz0Wj4+PR4RB9gtTFgKaCD88PNyA6L8RVVNYGOGfq9XRlXHt6RYB3ZgWRfjH9fUReVx7uQtXIMG6MB1cGBc+adBvjnD3rt0a4e4FjLqwCpbleO+u3Rzh7l27NcLdC9ga4Sjyc3sehN2lHAgTpPsO0t2bCAiDMAjPet8ZhEH4vBHuvpvSOkgjoEBcrYV7b0e5a2/Wwq0R7n4l4QqoEHZLvPfvg7BbgQRpgvR5B2lMRFQoCIMwCNPOelcg5SF1grR3jYMgTZA2XdQVEIRB+LSXIQnS5hQAwpURZg8kWJElwggYFLBEmE2kGNIEaTcH4sJBZEvBceEYsuUmmY1wetNxEU4voItwdte2EU6/6eQinF5AF+Hej26o/w+ECdJmEHYFBOETB+n0JoILm1MACIMwm0qzPhsDwiAMwiD8rkDGl9bKtXD6hqla6qkgjYBcNmx6hloinH6t6yKMgCbCqiObfRyEFaJqXLkwCIMwLnzWSz0QpiPdFFGVMqQLs5Qz7wsjoOnC2U9f2QiTA02EEdBEWJVw9nHpwlQgCLfdtWNfmH3hthVGO8tc6yoBQbgxwizlTBdGQDNIsxb+P+IEaWUSapyOtOnSuHBjF46uhdOZTm2E0wlYG+Fsri1dOF1FKdctxxXCCEiQbrpvLBGOunC65xXC6QSJzoG1XTjbJhMIRysu6sIgbLpwNiSj/y8Ig7DZjnIFxIXPrJ2VznQI0uYUAMIgzNGOs74HolwahEEYhEGYt3a8KZDyV7mUSahxgjRBuummkWpv0c5SiKpxEAZhEJ51DgRhEAZhEOb3hd8UyPgGShWU1fhiGobpgNDrON4/b7dXs0ZKBd/a4whouvDLOO4OFTes13dPm83F4TNIL3cSYXKgWYEIaArIptI3bypxar+o2CjCCFgIGEU4200kXLh2cOaMtOm6/L6w57rVf19YHajs3nSiLqwEK8e7FzDqwsqVyvHeXZt9YdelWyMcRX52z7dGOIr83J4HYRCuHIyjgoKwF6xBOFpxrIUrIw/CIMw1h1lv5INwY4S776a0dmEENN/a0Xs7yl17E6RbIzy79pIrSPT7uHBjF6YCzVfBI6Dpwq5L9f59XDhqGrSzaGed9GbStx/t6N6E2Bc2kSZIE6TpSNOR5sU7vHjnyy8ewoVx4ZMG6w9rYfZAghVZIoyAQQHLIM0mUixY086inRVEzhWMm0oxRFVH3UY4vem4QTq9gG47K7tr2wh333FWpuMinF5AF2HlUr2Pg7BCVI2DsBnEQdgL1iCsEFXjIAzC59WRTp/rBLKr1erm+L7FYRiYA9UcV4x/EBAXjrnw5eXl3zd+UoHTbXTKAuEgsqXAILyPIVuu5UG4dgUSpGNBGoRBOFYxUZdVz+PCtedAgnTMlXHh2hWIC8fmVFwYF45VjHLV6DguXHsOxIVx4W+9qQTCIBxDrvbREoJ07QokSMdiEUG6cpD+A6hN10l5jh4MAAAAAElFTkSuQmCC' />"}],"value":"[10 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x2006b7a3 \"comportexviz.server.notebook$eval28204$fn$reify__28208@2006b7a3\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>25</span>","value":"25"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAADkCAYAAAAPSqPOAAAEmklEQVR4Xu2dQVIbMRBFbXbZxfGG2yQn4AocJbDiGhwhRyAHsndJVTaeFIWhCrGQm98aqdUvmxTlMVAfPb1WSzPeHo/HZXP+tyzL/Xa7/cnXLwlclAcBigPmcDjcvY64q6urp9Pp9IOvXxK4JI8tI1AcgQQoBgjC2pRlRvjbfv9m6X83N7//Pj5+T21tK8IEWCBvRfjL7e2btTfX17/+PDx8zWxtM8IU2sUItCJMgCLCFNrvrd0c4eml0xrh6QO0WtiK8OzWdkd4+hFXtvu8EU4XoDfCsyNbTmHuCKerE70RThegN8JWS0e/HoTVTTQQpiPddRMMhEFYRFANEAuvvKmUrs6rjVAsLE4BIAzCfU+TgTAIz1VIp2uoeiOcLkBvC9ORrhWOvP7e+t4Ip1upeCMcvcNs/f1pZ6lTEghTSM9VSKeTCAiDMAiHvk0ChEF4LITVbor6/nAWLxFWA1DfHy7Aci2stqPU91vXor2vZy3MWli0qBog7azO+8LppOF9m0P6AFWEs1mX2xxUaXgjHK7w9Q5QRbh3Idv7569eSE8nnbXbWdMFuDbCs1l7dYSnk87aCE8X4NoI97am988HYbUuBGGxHQbCndtZ00nBijQIg/BY+8LpkQRhEUlrgFgYC3OzYejjbSAMwiAMwucELvn0A+92Uu/v597Omm7Po1YXeq+F0wXobeHZNo1qU4Q7wunW0t4IpwvQG+HakJ/tdRCuWbb2OgiL7S8QZi3MWpi1MGthw4dx1qyU7XUsjIXH3hdO112xTkE1hAmwgnitkM7WnrKu1VkLW5HlTiXRumWANYStQzrb9e4Ip5NOzcLWDnO6AL0RzmZtd4StIzb89d4Ihw/EWtZ4I4yFrX8B8frw0umNcPgAeyMc3dpYWJyCCFANsDfC0a3NCFRHYG8Lhy+8QZijHRzt4GgHRzs42vGcwLIs92apYWFxkwkLY2EsjIWxMBbGwp8tQ9RmAhbGwlgYC2NhLIyFsfBmE/GpH2wqqXUg7SzaWWPfqWTu0KpIRHs/CIMwCIdeyoEwCMdCOPyRXG/LWxEmwAJ5a0c6+plm7wOdrIVVpK0IszIREfZGIPr3a47w9NJpjfD0AVotbEVudms3R3h66bRGePoAWyNsRT7a9cMhHE46oyEcLsDREI5m7eEQDied0RAOF+BoCGNhtT0U7f0gzKZSrE2lcJN86ykBhEEYhDnacU7gUzcst56jWn9/CunB7lQK101RR6i3hdMF6I1wtHaUuvamnTUawulWKt4Iq0hEez8Ig7C4llUDBOHBCul0EvEupNMFCMIgzFM7QvcDQRiEQRiEefAOD97hwTs8eCfpg3fS7YF4fzhp+gDVQjrbJlLZMacjrTZUaWeJHW0V4WibQN6/LwiDsIigGiAI086inUU7i3YW7axL21n7/f7ulZjn/yljjBb+ECAWtll4t9s9MQKFuwpA2IhseVQFhE82ZMu1Mwh7j0DaWba1NAiDsG3EeJ9XxMLecyCFtM3KWNh7BGJh25yKhbGwbcRgYXHOah4gFsbCq24qUUiLUwIWxsJYeNU5CwuLyHmfxirXwv8BxdLdSURBKYsAAAAASUVORK5CYII=' />"}],"value":"[25 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x111e4ffe \"comportexviz.server.notebook$eval28204$fn$reify__28208@111e4ffe\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>50</span>","value":"50"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAADkCAYAAAAPSqPOAAAEeklEQVR4Xu2dPVIbQRBGtcqcWVbCbfAJuAJHMURcgyP4CHAgKbOrnGhdKn6CJRi1WtPzdeuRUNSuoKqZt++b3l5p2u/38+r9a57nx2mafvHzWwVOqgcFdC6Y3W738LHi1uv1y+Fw+MnPbxU4pR4TK9C5Aimgs4Ag7LtkgbA3dYAwCA9NDSAMwk4EvQXEwuIW/rHdfu6t/93dvf59fr4ttdfubeHyBeyN8Lf7+8+99urm5vefp6fvlfbaWNgrkd4Il+8v9ka4entMDuF00lFDOF0B1RDOZm05hNNJRw3hdAVUQzibtdMjPFw62REeXsDsCI+2dnqEh0snO8LDC5gd4dHWlkd4uCRa7S51hOULqI7waMu2LhHyCA+XRHaE5QuojnALodHHQbiFaOu4uoVBuPjMNQi3EG0dB2HndBcWFp/OkpdAC9HWcRAGYWakUw9cgjAIgzAIX/MD2wRpgvTYtyjAwlgYC2NhLHz6u1SMvomj9vfpSLfaVa3jWBgLX5eF5YeFWsguj0cjXK6A0Xth9Wkrq+WxsBXZ0QiXu0sXjbAVEfXz5RGWl060ha0IyxdQHWF1a8sjbF2x4eerIxxeEGusUUcYC1v/o87zw6VTDeHwAlZDONraWNh5yaCA3gJWQzja2qxA7wqsZuHw4A3CTGcxnZV6NgaEQRiEQfi9Aqd8BlF00O399+SCdHg3pVqQTldANQtHt6O8iMshHL4Vq4ZwugKqIexFKvr1IAzCzhlnbwFBmL0we2H2wuyFDR+J7b3oVns9N5WcFsfCWBgLY2EsjIWPFZjn+dHcDcLCWPi6nhdeIpLuHojaw4bpCzg6SGe7ibTseNOR9u7NsTAWvm4LW4OrnHSyISxXwNEWtt7HVbM2FsbCTot6C5gNYSvyvc8vj3B36WSzsFzsqY5wb2uXR9i6Ys3nV0fYXBCrlasjjIWtKyL6fBB2BnEQZrSD0Q5GOxjtYLSD0Y5zZ1u8sQcLY2EsjIWxMBbGwlh4tcr4rh90pL05kHYW7Syms1LnQBAGYRAGYT6clA8nPVbgnCBPkCZIOy3qLSAdaTrSdKRTxxgQLoZw95lmrzTUnhdO/wC2GsK9Z5ovPXBJkPYiTTvLGcTVEL40Yr1/HwiDsBNBbwFBuFiQ7v5gjHfFqQfpdAUEYRCmnUU7iwFLBiwZsGTA8rz7sr33uq3fz17YG6xpZzn30gRpgjRBmiBNkCZIE6QJ0gTp8z4NwRtkR7+eIE2Q5jGH1DkQhEEYhEGYJ5V4UqnMk0rMSDt3BukKqNaRZkbauQLTDRcRpAnS1xWk00midUmKRrhcAaMtnM2yjHa0EPQej0Y4XUxpFTga4RYS2Y7LT2epSWe73T58tN+O3ylgC9HF8S8FVEdYzdqbzeYl1QpUk046hOULqI6wmpVB2CiNJQEgfOkCgrBtRhqEL70C2QvbOtLpgjQWNiKjtvdNZ+F0BVSzsNred3kJwcLGS0o6hNX2vs0CqiGsZl0QdiLbXIEEaYJ06GiHvIXT5UA1hNMVUM3C2YL0f/9n3UntA/sMAAAAAElFTkSuQmCC' />"}],"value":"[50 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1df4d34a \"comportexviz.server.notebook$eval28204$fn$reify__28208@1df4d34a\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>75</span>","value":"75"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAADkCAYAAAAPSqPOAAAEnklEQVR4Xu2cQU4bQRBF8eyyi+MNtyEn4AocJbDiGhwhRzAHsneJlI0nQgYkBsnl4nfNuKseG4Smx0Lf/fr/qu6Z1X6/H69ef8ZxfFitVr/4+6jAWXogoDhhdrvd/duMG4ZhezgcfvL3UYFz9FgxA8UZiICigCCsLVkgrKYOEAbhRVMDCIOwiKAqIC6MCy9bu+PC4hIAwiAMwl33H0EYhEEYhF8VOKeDm63jTS2slnIEaYI07SzFRH5sNu8u/O/29vnv09ON8nnuffHeEV5cwN6D9Le7u/d97avr699/Hh+/z7mvjQvjwqKLqgL2jvDSwbwcws1Np3cX9saO5gJWQ7i1a5dD2DtjzfHVEDYF8bpyNYRbuzYIGzPONB0QPh3ETQFB+PSmkuXaIOw1jel4EBZraRBmX5h94Vlb8OqaN70fhEEYhEGYox2O52NbL8K9fx5BmiDN0Y5IEzG7KeoSkh3hcAGzB2mrHaV2qGlngbDooqqA2RFWEbXuB2F1BmZ34eb7wLSztPbVFGkQBmFceNFa1nJZ6zoIgzAIg3BkOyo8x4EwCIMwCPMW3/Pf2kstTC188ihH+B5I9gOWswuYrSMdvYlEO0sNztkRnr1yyYaw1X5qfb18O0s2neqbSrKA1RFWXbs8wrLpVEdYFrA6wqorg7AzWE9NBwFVAUHY196aujYz0DkDP5kOLixuSoGwD2HaWSqytLNEZNlU0pAFYRBujKAqKC6sIU2QVmcgQVpcEkAYhHlrR9fH40AYhEEYhHnxDi/eeVFgHMcH9zYnQZogvehjEtTC1MIigqqABGmCNEGaIE2QJkgTpL9aSeDCmotywFKdQer91MJiECdIa0tA+lpYfpDGQjw7wuECZkdYfRLJcun0CLs7zBayHLAUXZcDlprrcsDSi6g1PrsLh6+B2V3YclH1Oi5sIWpdB2HRlUFYc2UQthC1roMwCHO0o+ttTRAGYRAG4covsSVIE6SXfSltdhdmU+mgIcamklWLLn09O8J0pEWE1Y6zdT/tLHUJAGFqYWphamFq4aMCwzBsLdfh+sdgn96Fw0u57C4cLmD2dha1sBp0o+/PjjC1MLWwWCpFI6h+PgiLX3B2F44O/t0H6fCcZyHeO8KLC9g7wtFB2VoCukc4POdlR3hxAXtH2EIs+no5hJubTu8u7EW4uYDVEG7t2uUQ9s5Yc3w1hE1BrNjC88Laaa9yzws3N41qj/yHC5jdhVu7bjmEm5tGNYTDBcyOMLWwN5fNPZ4gzaYSByw5YMkBSw5YfvWAKe0s1bVxYVwYF4504fBuTHaEwwXMXgvTzlJdMvr+7AjTzuKMtJizohFUPx+ExS84uwvTkVYRi74fhEGYWjiyFg7PgSAMwiB8CuHwborTpTebzf3b//vy++Jb+hcv4KUH6eh2lDdor9frbVczMNxFsyN88QJeOsJexKLHp0c42nS6c2EvwrMLmA3haNdOj7B3xnrHp0fYK4h3/CcBsyGMCzuDrXcGTcd7TQeEJ1+QLGB1hL2ujQuLSwQItxawOsJe1wbh1jOQTSXfphJBWjztBcIg7ENOrVSm94NwY4T/A3yA2op+n5KbAAAAAElFTkSuQmCC' />"}],"value":"[75 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x7651425d \"comportexviz.server.notebook$eval28204$fn$reify__28208@7651425d\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>95</span>","value":"95"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAADkCAYAAAAPSqPOAAAEaUlEQVR4Xu3dTU4bQRDFcexddiHecBtyAq7AURJWXMNHyBHIgWAXpGxwFPEhMUi0S6+NmekfG2R5bKGi//OqXlX3rO7u7nYnTz+73e5qtVr98PoxAnvFQwDDBXN7e/vzecWt1+ubh4eH714/RmCfeKyswHAFCmAYQAhntywIp1kHhCF81KxheIS/bTYvhcPfi4vf99vteamQGB3hOICjq/CXy8uXQuLk7OzXn+vrr5VCYniE49p/dITjAI6OcFr7Q7iYSE9FRwDTAEK4VgtPVdsKLK7AN6JDhdXCauFS7Zki1/vzEIYwhCE8cmNeIl1LpKe1s0Q6VWUqTIWp8EeqcNwDmSI/GsLdAziaCqdNJCqcqu7oCMdNpGkAR0M4bSJBuIhwU3RGU+Eqws0AQvj9Wril2mrhItKaSmnAqHBmX1Hh3iuQCrOz2FkfaWdV88Dm9RCGMIQhrC+8//7Y3nbQ3L9PLZwm1lSYClNhKkyFqfC+p3Swsxqq2+yB6Au/r7rlAGoqvXaoW00kCKeJM4TDxFlTSVPpcx3PpxYOkabCGdLsrFSVIQxhdhY7i53FzmJn7Xvuc6q6auFQddXCWeLMzoJwZwTTgKqFM6SHr4XLPRAq/PoWEAdwdISrTSQqnIoGhDur+OgIpwOew6twcxvDBFmH0Ib3wDcBhHAtkXYIbbgC7RfuHUAI1xCWSPdegfrCYWINYQgfd1oLwhA22mG0w2iH0Q6jHUY79nymee9KIv0+ibREWiI96zQGwhA+KMLNyQW1cLh/GMIOoT0ows2+MYTZWeysWeeBEIYwhCHMkeZIc6Q50hzpk/V6fZPOHM/t84ufkW7aUWlPZOmJ9MEDuHQ7K92J1LqlLB7hph0F4bCWTQO4dIRbCKbvQ7ixE+l+uz1/t1ZfugpX74Fl1YawQ2g1lWbtB0LYaAeEIfwUAXbWbndVzZuGv14iHdbSVJgKU2EqTIUfI7CjwvUshApTYdNZsxYRCEMYwsdEuNwDcYKlQ2i7IpuOfugLp5MJVJgKd0W6OuoB4SLCDqEtBmzasnAI7UPmQDuENlyBDqHtHUBNpQxpKpyuSIm0RFoifUw7Kx5NgTCEIQxhW/5t+bflf8/RlLiJNHpTqXsAR6uF0yaSB7Kkte/oCMeVh0eEZ/YVhIsIN0VHLewQ2oPWwi3V5kgXkdZUSgNGhanwcXcmSaRDBxrCEIbwrC18CEMYwhAeeH8wO6t3KcfOChPr0ZpK1W0MrevZWSnSEIbwQR1pCBcRbTaR2FlhE0ktnD3JUCJdRLo5yUCFqTAVnrU7A2EIQ3hOCG82m5/Pf+//38yEYlrzJoDsrFqT6fT09MYKDDYGQbiI7LQSgXB4asfiES7bUcUVuXiEPzyAS1Ph1kxzy2Fuvb94hJv2UxHZpoiohWu1MBWmwrUV00K6KjqLV+FWwFrH2bUeBQnhCcJV1abCVLjvPbCKPISp8CdbgRLp2j8EwhCurZiqSLSul0j3TmOWZme17Kj0/Wki/Q8IdtUMXneqzAAAAABJRU5ErkJggg==' />"}],"value":"[95 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x63ff489b \"comportexviz.server.notebook$eval28204$fn$reify__28208@63ff489b\"]]"}],"value":"([5 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x20e964e5 \"comportexviz.server.notebook$eval28204$fn$reify__28208@20e964e5\"]] [10 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x2006b7a3 \"comportexviz.server.notebook$eval28204$fn$reify__28208@2006b7a3\"]] [25 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x111e4ffe \"comportexviz.server.notebook$eval28204$fn$reify__28208@111e4ffe\"]] [50 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x1df4d34a \"comportexviz.server.notebook$eval28204$fn$reify__28208@1df4d34a\"]] [75 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x7651425d \"comportexviz.server.notebook$eval28204$fn$reify__28208@7651425d\"]] [95 #object[comportexviz.server.notebook$eval28204$fn$reify__28208 0x63ff489b \"comportexviz.server.notebook$eval28204$fn$reify__28208@63ff489b\"]])"}
;; <=

;; **
;;; ## <a name="choose_a_rubric" /> Choose a rubric
;;; 
;;; How do we know if an HTM is good?
;;; 
;;; We want to use the HTM to predict future energy consumption. There are a number of ways to do this, and we'll talk more about that [later](#read_the_crystal_ball) when we work on prediction strategies. But this presents a problem: we need to tune our HTM _and_ our prediction strategy. If we use predictions as the rubric for our HTM, we're going to have a bad time. We'll never know whether the HTM or the prediction strategy is the weak link. Our HTM might follow our prediction strategy off a cliff.
;;; 
;;; I think there's a clear best way to measure an HTM. Here, **scroll down and check out this plot:**
;; **

;; @@
(defn plot
  "`data` is a sequence: [[x y color opacity symbol-size z-order], [x y color opacity symbol-size z-order] ...]"
  ([data]
   (plot data false))
  ([data joined?]
   (apply plot/compose
          (let [xs (map first data)
                ys (map second data)
                x-range [(apply min xs) (apply max xs)]
                y-range [(apply min ys) (apply max ys)]]
            (for [[[color opacity
                    symbol-size z-order] subset] (->> data
                                                      (group-by (fn [[_ _ color opacity symbol-size z-order]]
                                                                  [color opacity symbol-size z-order]))
                                                      (sort-by (fn [[color opacity symbol-size z-order]]
                                                                 z-order)))
                  :when (and (pos? opacity)
                             (pos? symbol-size))
                  :let [pruned (->> subset
                                    (map (fn [[x y _ _ _]]
                                           [x y])))]]
              (plot/list-plot pruned
                              :plot-size 700
                              :plot-range [x-range y-range]
                              :joined joined?
                              :color color
                              :opacity opacity
                              :symbol-size symbol-size))))))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;comportexviz.notebook.hotgym/plot</span>","value":"#'comportexviz.notebook.hotgym/plot"}
;; <=

;; @@
(defn sensor-selector
  [htm sense-id]
  (let [[selector _] (get-in htm [:sensors sense-id])]
    selector))

(defn sensed-value
  [htm sense-id]
  (p/extract (sensor-selector htm sense-id) (:input-value htm)))

(defn toggle-learning
  [htm learn?]
  (->> (for [rgn-id (core/region-keys htm)
             :let [rgn (get-in htm [:regions rgn-id])]
             lyr-id (core/layers rgn)]
         [rgn-id lyr-id :spec :freeze?])
       (reduce #(assoc-in % %2 (not learn?)) htm)))

(defn consider-future
  [htm inval]
    (-> htm
        (toggle-learning false) ;; Makes it faster.
        (p/htm-step inval)
        core/region-seq
        first
        core/column-state-freqs))

(defn anomaly-score
  [{:keys [active active-predicted]}]
  (/ active (+ active active-predicted)))

(defn htm-anomaly-score
  [htm]
  (-> htm
      core/region-seq
      first
      core/column-state-freqs
      anomaly-score))

(defn htm-timeline
  [htm inputs]
  (reductions p/htm-step htm inputs))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;comportexviz.notebook.hotgym/htm-timeline</span>","value":"#'comportexviz.notebook.hotgym/htm-timeline"}
;; <=

;; @@
(defn grab-representative-sample
  [n fscore bottom top max-precision]
  (loop [scores (sorted-map bottom (fscore bottom)
                            top (fscore top))
         remaining n]
    (if (pos? remaining)
      (let [pairs (->> (partition 2 1 scores)
                       (filter (fn [[[left-v _] [right-v _]]]
                                 (>= (/ (- right-v left-v)
                                        2)
                                     max-precision))))]
        (if (empty? pairs)
          scores
          (let [[[left-v left-score]
                 [right-v right-score]] (apply max-key
                                               (fn [[[left-v left-score] [right-v right-score]]]
                                                 (+ (* (/ (- right-v left-v)
                                                          (- top bottom)))
                                                    (/ (- 2
                                                          (+ left-score right-score)) ;; favor low scores
                                                       4)))
                                               pairs)
                next-v (/ (+ left-v right-v)
                          2)]
            (recur (assoc scores next-v (fscore next-v))
                   (dec remaining)))))
      scores)))

(defn interpolate-scores [scores target-n bottom top]
  (reduce (fn [interpolations [[left-v left-score] [right-v right-score]]]
            (->> (range bottom (inc top) (/ (- top bottom)
                                            target-n))
                 (filter #(< left-v % right-v))
                 (map (fn [v]
                        (let [slope (/ (- right-score left-score)
                                       (- right-v left-v))]
                          [v
                           (+ left-score
                              (* slope
                                 (- v left-v)))])))
                 (into interpolations)))
          (sorted-map)
          (partition 2 1 scores)))

(defn plot-would-be-anomalies
  ([steps buckets n-samples bottom top]
   (plot-would-be-anomalies steps buckets n-samples bottom top identity))
  ([steps buckets n-samples bottom top xf]
   (plot (concat (for [t (range (count steps))
                       :let [v (sensed-value (nth steps t) :power-consumption)]]
                   [t v "black" 1 50 0])
                 
                 (apply concat
                        (for [t (range 1 (count steps))
                              :let [prev-htm (nth steps (dec t))
                                    htm (nth steps t)
                                    actual (:input-value htm)
                                    actual-consumption (:consumption actual)
                                    scores (grab-representative-sample n-samples
                                                                       #(-> prev-htm
                                                                            (consider-future (assoc actual
                                                                                               :consumption %))
                                                                            anomaly-score)
                                                                       bottom top (/ (- top bottom)
                                                                                     buckets))
                                    scores (assoc scores
                                             actual-consumption (htm-anomaly-score htm))]]
                          (concat (for [[v score] scores]
                                    [t v "red" 1 (* (xf score) 30) 1])
                                  (for [[v score] (interpolate-scores scores buckets bottom top)]
                                    [t v "red" 0.2 (* (xf score) 30) 1]))))))))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;comportexviz.notebook.hotgym/plot-would-be-anomalies</span>","value":"#'comportexviz.notebook.hotgym/plot-would-be-anomalies"}
;; <=

;; @@
(let [steps (->> (htm-timeline (core/region-network
                                 {:rgn-0 [:power-consumption :is-weekend? :hour-of-day]}
                                 (constantly core/sensory-region)
                                 {:rgn-0 cells/better-parameter-defaults}
                                 {:power-consumption [:consumption
                                                      (sampling-linear-encoder [1536] 17 [-25.6 125.6] 25.6)]
                                  :is-weekend? [:is-weekend?
                                                (e/category-encoder [10] [true false])]
                                  :hour-of-day [:hour-of-day
                                                (e/linear-encoder [56] 10 [0 24])]}
                                 nil)
                               inputs)
                 (drop 1500)
                 (take 100))]
  (plot-would-be-anomalies steps 64 (if be-really-slow? 64 7) 0 100))
;; @@
;; =>
;;; {"type":"html","content":"<img src='data:image/png;base64,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' />"}
;; <=

;; **
;;; **An HTM is doing a good job when the big red dots wrap tightly around the black dots, without touching them.**
;;; 
;;; Let's call this "Plotting the would-be anomalies".
;;; 
;;; Here we're using Numenta's "anomaly score" measurement:
;;; 
;;; $$ \text{anomaly score} = \frac{\left| \text{active columns} \right| - \left| \text{active columns} \cap \text{predictive columns}\right|}{\left| \text{active columns} \right|}
;;; $$
;;; 
;;; So, 0 is "totally predicted" and 1 is "totally unpredicted".
;;; 
;;; At each timestep, we're asking the HTM what it thinks of a set of candidate values. We're simply passing the value into the HTM and checking the anomaly score. The red dots represent these values, and their size represents the anomaly score.
;;; 
;;; _(Note that the HTM is not modified in-place. A new HTM step is created, read, then discarded. This is how timesteps work in Comportex, so we don't have to do any undo magic.)_
;;; 
;;; Using this approach, we can design our HTM independently of designing our prediction strategy. We're proving that [the predictions are in the HTM](https://www.youtube.com/watch?v=TV3Oncvz_cU), and later we'll figure out how to extract them. The steps are now disentangled.
;;; 
;;; It would also be possible to invert this and show low anomaly scores as big blue dots. That's totally valid. I prefer "bad red dots" over "good blue dots" because:
;;; 
;;; - It demonstrates that far-off values have been tested. Having a bunch of red dots at the limits is more informative / comforting than having a lack of blue dots.
;;; - It's more obvious when something is "slightly anomalous". Look at some of the tiny red dots that are near the black dots. Would you have noticed those if they'd been not-quite-full-sized blue dots?
;;; - It draws a clear line between tuning the HTM and tuning our prediction strategies. Save the blue dots for prediction.
;; **

;; **
;;; ## <a name="design_an_htm" /> Design an HTM
;; **

;; **
;;; Let's start with our encoders and the default spec.
;; **

;; @@
(def example-steps
  (->> (htm-timeline (core/region-network
                                 {:rgn-0 [:power-consumption :is-weekend? :hour-of-day]}
                                 (constantly core/sensory-region)
                                 {:rgn-0 cells/better-parameter-defaults}
                                 {:power-consumption [:consumption
                                                      (sampling-linear-encoder [1536] 17 [-25.6 125.6] 25.6)]
                                  :is-weekend? [:is-weekend?
                                                (e/category-encoder [10] [true false])]
                                  :hour-of-day [:hour-of-day
                                                (e/linear-encoder [56] 10 [0 24])]}
                                 nil)
                               inputs)
                 (drop 1500)
                 (take 40)))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;comportexviz.notebook.hotgym/example-steps</span>","value":"#'comportexviz.notebook.hotgym/example-steps"}
;; <=

;; @@
(viz example-steps
     {:drawing {:display-mode :two-d}})
;; @@
;; =>
;;; {"type":"html","content":"<img src='data:image/png;base64,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' /><img src='data:image/png;base64,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' />"}
;; <=

;; **
;;; Off the top of my head, I have a few questions:
;;; 
;;; - Are we gaining anything through overlapping hour encodings? What if we used just a simple category encoder?
;;;   - Alternately, what if we concatenated a category encoder to the current one, so that each hour gets some unique bits and some overlapping bits?
;;; - Does it help if we wrap the hour encodings, so that hour 23 overlaps with hour 0?
;;; - We're encoding power consumptions very precisely. Does it help to, for example, drop the decimal point?
;;; 
;;; But I think this is the wrong approach. We're not going to get very far if we act like apes, throwing different things at the HTM until we get lucky. We should seek to understand why the HTM is doing what it's doing, and then the proper response will become apparent.
;;; 
;;; Looking at the would-be-anomalies in the previous section, I see multiple places where I disagree with the HTM. Lots of the so-called anomalous events seem typical. Look at a step where everything was an anomaly.
;;; 
;;; At step 33, everything was anomalous. This means Step 32's set of active cells didn't depolarize very many other cells.
;;; 
;;; **TODO: Sometimes this happens and other times it doesn't. Call `set-seed!`, or something?.**
;; **

;; @@
(viz example-steps
     {:drawing {:display-mode :two-d}})
;; @@
;; =>
;;; {"type":"html","content":"<img src='data:image/png;base64,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' /><img src='data:image/png;base64,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' />"}
;; <=

;; @@
(:input-value (nth example-steps 32))
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-map'>{</span>","close":"<span class='clj-map'>}</span>","separator":", ","items":[{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:timestamp</span>","value":":timestamp"},{"type":"html","content":"<span class='clj-string'>&quot;9/3/10 19:00&quot;</span>","value":"\"9/3/10 19:00\""}],"value":"[:timestamp \"9/3/10 19:00\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:day-of-week</span>","value":":day-of-week"},{"type":"html","content":"<span class='clj-unkown'>5</span>","value":"5"}],"value":"[:day-of-week 5]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:hour-of-day</span>","value":":hour-of-day"},{"type":"html","content":"<span class='clj-unkown'>19</span>","value":"19"}],"value":"[:hour-of-day 19]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:is-weekend?</span>","value":":is-weekend?"},{"type":"html","content":"<span class='clj-unkown'>false</span>","value":"false"}],"value":"[:is-weekend? false]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:consumption</span>","value":":consumption"},{"type":"html","content":"<span class='clj-double'>40.0</span>","value":"40.0"}],"value":"[:consumption 40.0]"}],"value":"{:timestamp \"9/3/10 19:00\", :day-of-week 5, :hour-of-day 19, :is-weekend? false, :consumption 40.0}"}
;; <=

;; **
;;; In some sense, the HTM is saying, "A consumption of 40.0 at 19:00 on a weekday? I've never seen this before. No clue what comes next." Is it right?
;; **

;; @@
(plot/histogram (->> inputs
                     (take 1500)
                     (filter #(= 19 (:hour-of-day %)))
                     (map :consumption)))
;; @@
;; =>
;;; {"type":"html","content":"<img src='data:image/png;base64,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' />"}
;; <=

;; @@
(plot/histogram (->> inputs
                     (take 700)
                     (filter #(= 19 (:hour-of-day %)))
                     (filter #(= false (:is-weekend? %)))
                     (map :consumption)))
;; @@
;; =>
;;; {"type":"html","content":"<img src='data:image/png;base64,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' />"}
;; <=

;; **
;;; 40.0 seems pretty typical. So why is the HTM saying that? It must be objecting to the sequence, not to this value. Something about this sequence freaked it out:
;; **

;; @@
(pprint
  (for [t (range 28 34)]
    [t (select-keys (:input-value (nth example-steps t))
                    [:hour-of-day :is-weekend? :consumption])]))
;; @@
;; ->
;;; ([28 {:hour-of-day 15, :is-weekend? false, :consumption 34.8}]
;;;  [29 {:hour-of-day 16, :is-weekend? false, :consumption 36.3}]
;;;  [30 {:hour-of-day 17, :is-weekend? false, :consumption 40.2}]
;;;  [31 {:hour-of-day 18, :is-weekend? false, :consumption 41.8}]
;;;  [32 {:hour-of-day 19, :is-weekend? false, :consumption 40.0}]
;;;  [33 {:hour-of-day 20, :is-weekend? false, :consumption 18.5}])
;;; 
;; <-
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; **
;;; I think maybe it's giving too much attention to sequences of consumptions, and not enough attention to the timestamp. It should never be completely surprised when power consumption is typical. It gets bonus points for knowing to expect high demand on a busy day, but first it should learn to predict the typical.
;;; 
;;; If we give the "hour of day" significantly more input bits, then many columns will be devoted to just telling what time it is. Those columns will output predictions which tell the typical time.
;; **

;; @@
(let [steps (->> (htm-timeline (core/region-network
                                 {:rgn-0 [:power-consumption :is-weekend? :hour-of-day]}
                                 (constantly core/sensory-region)
                                 {:rgn-0 cells/better-parameter-defaults}
                                 {:power-consumption [:consumption
                                                      (sampling-linear-encoder [1536] 17 [-25.6 125.6] 25.6)]
                                  :is-weekend? [:is-weekend?
                                                (e/category-encoder [10] [true false])]
                                  :hour-of-day [:hour-of-day
                                                (e/category-encoder [(* 100 24)] (range 24))]}
                                 nil)
                               inputs)
                 (drop 1500)
                 (take 100))]
  [(viz (take 3 steps)
        {:drawing {:display-mode :two-d}})
   (plot-would-be-anomalies steps 64 (if be-really-slow? 64 7) 0 100 (partial * 5))])
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAA54AAAAaCAYAAADfRoO0AAAVrElEQVR4Xu1dbWxVRRp++wHU2tIqpXyIFAyULyl+EJVV2dUAhhgF4gcaUrUSfxl1Fw1SRRCDWTEUF2PUsI0WZS2KJFZR0QUBscEACSAfSoFSpKVYoECpLVDbu3kud+rheO+dOWde3WN5J7kx0nPnPueZmfd5n5k5cxJIijAgDAgDwoAwIAwIA8KAMCAMCAPCgDCgZ2AiEf2ViK6KfDIjX1lLRCeICP8tI6Iqd1UJ+rrlCmFAGBAGhAFhQBgQBoQBYUAYEAaEgQuYgYeIaDYR9TPkoISI5jgNqBhPQ+bkMmFAGBAGhAFhQBgQBoQBYUAYEAYuMAaworkmsrrp59YLiAgmlHTGM4mI8Dnr51dc3+lMRK2Rj011HR1TMhGFGHhC23ZiajtuTC2Re7TpB8CE8otNJZExAJ4EU3wiVX/qqDwhruAebfsTWESs4+CJG5PEX32wEJ3Sc6T6eND6E7dOceQ93Jg44opop1kfF57MeOLWKY4+zo0paLGuGxE1EVGzWRPFvOoiIkolomOaerCdFqZTbaf1+7P/IqJ/RDOeAIIPkvFER+3oDPg0GpoiNHxapB7UpUpbpB4EddRlUuJhQoc4ZZgw/pkwgZczEeNoylMXIro4StshmcbHlCf0i65R2s6J6WdD86gwgXsVyJVhBCbUYyLwChPqQZ3Oonj6M2NCwou2A0e2PKEe1BeNJ9UPMLmhK/EwYdyBb3CvK2i7eJhQV4Nhf/ojMKlYZ4oJ7ZUepe1UrDPlCTwiZkZrO25MECxT0VKYbDVB8YQxHE0T/l+YVMy0xSTaqYsE5/7+e+u5aOf57SDaaZ5jiHbqx3A8PefWKdHOc+2BfDw3EjszHE0EvpGnHCSian3Tha9APTCuKsdXXzsZqavClRtgS+0WBtOpfqfEaTyRDOCG3Mmq+16QTMHAwG3HKhi8SFacxjXatSDteBwja4oJdQNTPIPmBRMejI218gHOLjHgyQQTuMYMgglPOkyoJ8Wg4+l4+n9gQnDB4IlVTDGhzTB4YhlZtJ0pT8AErmIZNJgD9ANd25lggmlB/9QVE0wYw07jGq1OTkyIA/EMGngywYS4gj4ey8ii7Ux5Oh2pK1bbgR/0A6fRiMaTDhO+gzgHXLrS0THpNMGUJ7Q/xjAmI6IVaALGnUnb6TCZaoIJJtHOcxPTOp0S7TTjyYtOiXbqou+5SVKdnpvolGinaCd6G6ee63QKRhEfXcGq5dY4E8qY8BsZycfi1YU4DvO5P5In2Wyvjfo7yngCENyvLol2VgIxro9S66WGpkx9NVZyZ5rYuzHByLoTTgQULCd7KRBQ94oAElbMFHjhCY2IDiGY4rP/R/CEpEdnypwoITJHo7SdaRLtrCua4UcSjfHiFRPGnTsxDyImJPaIK14KjCyMh7MgTmHc6cyG8zuIKxh37gkkxDqv20UEk1kLcvKEOO6eQDKdgPojdSqImEQ7f9tfRc/NxnAsnvxo559Fp0Q7z+8bop1mY+X35gn5zg0GRtGJFnn05ihbZ3sS0QiPORTyp9siBwmZMWJ4FRI6P2ZKVe9Opv0MYNSFBqxzJPjAle3R4ClM7tUzP8mvwgTToRJ8YMryaBKCjAkzNkjuVEECBRPkpxxxJfjgyYtJUL8ZREzuCRYkdTBBfgoGsjOZ9jpJo34ziJgQ8DBeVEFc6e6HpMiElnPlE0mPyWq+++eCisk5EWUTf7HSgXinCky+ycq5myf3BAsmRDCGvUywqTqDiCmIOhVETKKdZgEriDoVRExB1KkgYgqqTgVRzy8k7RxGRP3NQtJ5V2ECeH3kkUb8AZPto33k5MglFvlYPNBCVmbKj0lQlSvTYZNooi5nQPCbaCpMKsG3SercmPwmdW5MNkkd6nIGKVtMmJEE75yY/E4+KJ4UJpvJB8WTClK2mFQybYvJmeD7nRBxJ/g2EyKoixOTSqZtMTknomwmRHB/zgTf74QI6uHE5EzwbTGpyTGbCRHcnxOT3wkRxRMXJmcybYMJuLh0yonJVqe4MIl2alOd8AWineY8iXbquRLt1HMk2mnGkVs7sbgxyvyrv7nycGTlE39APX4WS+4kokcsMMT8KhLEXpYVK9GzTQwAA9s8IA5+V0rUrSAgQNRtE4OgYoIRAu+2PCGZ/omJJ2BCUoaVapuiMPnZHu3+3SBighFCku93RV/dozJCMLAmzxjGaxP1nKYtJvwGdi5gddLrFls3PrVts4fPlTdnfVyYlOngwIT4BFNtyxM3JkxCet2K7G47hQnxycs28mh9FDxxYFI6xYFJnQDoR8yd9yjaaaYUnHou2mnGuWinGU+inWY8cetUh9XOUCh0qrCw8KHt27e3fPrpp2Gtuf/++3umpaW1a+mWLVtObN68ubGgoOCyHj16pNbX159etGgRDhiiK6+8MnX8+PG98vPzt+zatSulvLw8t6Sk5MdTp061XXfddekjRoxoP5yosbHxl9LS0sPqO/j+unXr6jZu3Ig89Z+oDv+Wnp5OCxcuxN9o8eLF4RafPn06de9+zoKEQiH65JNPaP369fTiiy9SXl4eVVRU0JNPPhn++zXXXEMzZsyg5uZmKi4uDr8+wNZ4ot5apnqQkCOhtk2kgQmOH/uabQsnJiTAfrewOe8DwQ5ibJu0ok5l0G0TRAQWfGyTVmDCYHOfuOWnHZHYof38biF2/iYnJhyCZZu0Kp6wmqs7EEzHHXjiwoQVa5hhW0zqtDYOnjChBTPsZ7uukzuMOWxPt53wQZ3AhFVKr8+eu9sS8RIrjLYTPqgXySbigC0mzvgLAcSWXz9biN1ccekUMIX1WDewDP7OhYlTp0Q7DRpOtNOMJGY959Ip6Llop74JRTv1HLFp58KFC3sXFBR0WbJkyW1VVVXHX3755b05OTldnnrqqeFJSUnth8Hu27evPjExkfr379+e29bU1Jx866239hcWFg7PyMi46Oqrr/65ubk5+7vvvjs9b968jXv37j09ffr0Af369cOCXLi0traGFi9e/MMDDzyQm5ycjJ2P4X9bunRpRXl5+X/w/w8//DCML40ZM4bKyspo4sSJdMUVV9D3339PnTsjhTlXli1bRsB01113tf/bqlWr6M0336QlS5ZQSsq59KutrY3NeKJzchgOJMA4hMc2QVSJHRcmPOfJkYxx8YSkHM8KciRjMAocxgxJOdqPAxMXT0jKkZBxJIhIyjmMPvomElcOTFz1qMDJcX/AhD7g5/lAd4jn4hyY8JyD7eQKZ1xBv0TUtnnMQfHFNV4woRXtdUVm0nv+VTDo7QLnp4LIdzCZBcG1ncjgbDv1/HHQMIl26juaaKeeo3B+KNppRlRkwk60U08Xl051SO0cMGBAUmlp6U1paWmJa9asGQRzWVRUtG/06NEZkydPHrhnz55j27dvP7569eoT6enpiXPnzh0Bkzht2rSt8+bNG56amtrl8OHDDZdddlkGTOrgwYMTWltbe9fU1LTMnj37v5WVlWfmzp07tGvXrillZWWVTU1NbRs2bGh47LHH+g0ZMqT71q1bDzc0NJwdPXp036+//jph6dKlz2Gl88CBA3TJJeekHOby3nvvpdtvv50+/vhjeu+99+ijjz6i5cuXh1dFq6ur6cyZM5SdnU379u2jyy+/nFasWEGTJk2il156iQ4ePEivvfYam/GECfrV+uo7YKwrEOzw4UgQkRxwJAbACrHiSBA5MXHVhcSOw+iDJy5MXP2JExPXvQET1/1x1cPJE9dY4eSJs+24xksQMXG2Hdf9YUKLQw84+xM0CoVjcoVrDIt2mucdXH2TKxZwxl+u/sSJiYtvzjEcRJ444y/X/XG2Hdd4CSImtrZbsGDBdSkpKdco45mfn9/7hhtu6J2QkBBe8Tx9+nTLpk2bDt900019GhoazhQWFm6fM2fOkKysrIthPpuams6sXLmyKhQKNebn508IhUK/PPPMMytgPGFQ09PTw/l+KBQK4TcyMzNTunXrlorvnDx5sgUm98svv+xdVlbW/nwnzOKjjz7abjyfe+45mjNnDkUgUV1dXXiL7YIFC2jPnj00ZMgQ2rBhA11//fVho1lfX0/PP/88lZaW0n333SfG01CruDoV54DhqosrGHAKFVfQ5MTExbeIp+GgYzTonG3HNV6CiIkrznGOOzGeZuNFjKcZT5x9kysWcGIS7TTrB0HkiTP+ct1fEHUqiJjY2s5tPB955JE+eXl5Pevq6hpTU1M7ZWRkpJw9e7a1U6dOiY2NjWeefvrp7S+88MJQmEdlPFetWnWgpaXl1JQpUyYkJCSEjeeRI0daZs2aNTw5OTmxtrb2VE5OTia2vba0tPzSpUuXZHzn2LFjZ02M5/z58+nxxx+nb7/9Nvyc5+DBg2nTpk3hZzkrKyspNzc3/P/XXntt+DqYzy+++ILGjRuH32MznlxL6LLV1ixoynYhM55ku5AZT7iKc1sr11ZbrrgiW23N+oFstTXjSbbamvMkj6nouZLHVPQcqSu4Hi/hqke007ztuPS8Q261VTSuXLlyQGVl5Ri14jl27NhLMjMzOy9btuynkSNHphUUFAzGgmVbW1sCVjedxtNRB1Y8Dz7xxBP3Hj16NPnZZ5/9DMZz3LhxWTCXa9euPaHMKp7LhAHFimdNTU3z1KlTh2zZsqVLcXHxDFWfe8XznnvuoT59+tArr7xCN998M61Zs4aqqqqob9++4a25AwcObDeiDz74IBUWFtLQoUPp0KFDNHbsWDbjKQckmA0+OVzIjCfOg3zkcCE953K4kJ4jXCGHC5nxhKu4DqiRw4XMOJfDhcx4koP5zHiSg/nMeeI68EgO5jPjvEMfzFdXV9fjww8/nKSM58yZMwf17t27a3V19QmsTHbv3j1t//799b169UrH/+/evfvYwIEDu7W1tbVWVFTUDxs2LPv48eNNQ4cO/bFnz55/2bt3b9Ls2bNXZ2ZmJsNUtra2tu3YsePI8OHDcTp/eLttbm5uFr7T0tLSmp2dnf7DDz8cffXVV1+NZTxXr15Nt956K3322WfUrVu38Jba999/n2655RbKysqid955h6ZMmUJNTU3hU3Cx6omTb7/55hvq1KkTi/GUI+HNBoscCW/Gk7xOxZwnTGTI61T0fMnrVPQc4QquV5fI61TM+BbtNONJtNOMJ9FOc55EO824Eu0044lNO0OhUK+SkpKJO3bs+BmHC914441dJ0+ePECdOtvc3Hz2jTfe2D1q1KhL1bOfeF5z586dde+++271zJkzh+E5zgkTJuxqaGjI++qrr1qLioq24RnPGTNmDOzbt2/40FV8B6b17bffPqC+g39H/fPnz99VW1v7gnqdSlFREU2bNq39Gc/x48fTBx98QGlpOBiaqLa2Nnzq7d13302zZs2ipKSksNF8/fXXw897wqQ6i3rRu83BOfISbLOOiVU87LvHqZE2r1SRl2Cb8Q2e5CXYeq7kJdh6jnCFem8qTg7FoWU2J0FjdQpbm1EQC/zGX/UqFWxnxOFuNq+ecWKyeSczJyZlYMGTDSZlqtXzojavw3Fisn1PtGin2dgT7TTjCStmmMxAXmfzTmbRTjO+RTvNeBLtNOcJr0aDnkPLRzm/dscdd2QlJiYmlJWVQTfCZdCgQRfl5eV1PXToUHN5ebnKKWjq1Kmh4uLinQkJCVWRetpzA7zLMycnJ7WioqJx27Zt0P1wQf2dO3dOXL58OSYbUO4kovYDhqLdAl61kpycTIsWLWr/M97hCaO5e/du+vzzz6PeOQIUTgwEKD8n9GEbFLYZqAL76+c1Ec6OibpsAqczgUJdWBHyc9S1M4FSmJAk+jlhMYiYnAkU7s8mmVYJlOoHfpPpIGJSqxLq3mwSfJVAqbr8JtNBxOScEMH9YZz4TfBVAqV48pvgBxWTmhBRPPmNvxCaduGIxDk/rzNyJlDAZDM5FkRMQdSpIGIKok4FEVMQdSqImIKoU0HEFFSdgglSJSh6fiFp5zAi6h/VucX/RzxOsD7yNg5ciVfJjfYxwY1c4t8+PV1chOqFpAAGc+bFfLoHsN9kGgYPe7bVgQ3OBB8Jp1dMeH8cViWcJcPHeziBqdlVjx+T7pw9FEyxu+MfwRP6k5eJA3dCbjPB4k40VYIP8+kVE4wZZsWcxc+kz++NCXHF67t03YmmmvSBOfOyMoi4ApFCGzqLH0wI5Cdd9SB2CqbzSYnFE8adl1dboe0Qx7FDxFlQB/qTV03AeHEXr5M+sXQqiJgwOSbaeX6Li57HTQXb/xiLJ9HO8/kT7TTrT6Kd/nlCvjPS404m5NGbI7mP85d7EtEIjzkU8qfbiGi22S2YX6WMp0qCYdB0CQIEGKuczpl29y/CKSMR1iUIMJtI6NxJtKoPM+8mmHC9e6bdLyY0HIKvO2FV9YEzJD8m776MFpz8JFKCySzhRJuhP7kTVmfbYYIl1WCIoH+j/dyTGOqrSO7QN3Wm0QQTdgmYrFQFEROEBWPPlqdYib2zqdB2Jjxh9h9jOBYmtBnGsM7ImmAyNfwdHRP6gHuiztl2pjxxaUIQdSqImEQ7Rc/RB0x0SrTTjCdTPRftNNsheaFrZy4R4aMrMIpb4+gwJt2vMjCy8BsVRLQ/kietjZhW3e8b/91pPNWXAA4fJGVO4wgw+MB0xjKKzh+GaUSygXqcCR7EF/WoU9NMwOowISGPZRS9YIJhcW4djoctHibwA9NiggltgIAeiycvmDBpgMTc3XbAgY9XTGhD90QE2s0PJtTlNGkKE9oullF08g+eENBRRyxMqCuW2XAbfphP1BUUTDCyaDsOTKgH9dnypMME4XTvVIg2ZtB28TCpvmnSdvEwYdyhD3BgQl3xDLXzPtFmqm+6+5MXTKgTMRP3GCv+esWEcRct/sIkxjOKbtMYD5OpJoAbFZ9sMaEe9O/fW6e88KTTKVOeRDtFOzH+lJ6LdsbPEnU65SXHEO3UZ+Q6PRft/JVDnZ6bagI8ArbdYrHD+eggvBRygoNEVK1vuvAVqAcroMp3qK+hHizcwHQ6c4N+EUOL3+Yoi6MZT7dRQ9AzMQY6QAgO6JAmpjVeXcATNEzgEQkQB09IzpCA2/J0IWBCPzEx9/H6k+IJA9jE+MSrS5kOwRQ/GnDxhF9BXOFoO8QU9AXbthNMOiX49e8dWROCqFNBxHQh6FQQ9Vy0Ux+nuHQqiDmG6JS+/dUVQcwx8JgPJv5NJ49j3S0mS7EIg9XSeAUrpVj5tDWfC4no7zrjad40cqUwIAwIA8KAMCAMCAPCgDAgDAgDwkBHYgCPJ9lsuy0gohIQIsazI3ULuRdhQBgQBoQBYUAYEAaEAWFAGBAG+Bl4iIieJ6Icw6oXR67Hq13CRYynIXNymTAgDAgDwoAwIAwIA8KAMCAMCAMXOAMTiehvkQOL8ByoMqLrIoc7fhRZIW03nIqv/wFdxx1Sk5faPQAAAABJRU5ErkJggg==' /><img src='data:image/png;base64,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