;; gorilla-repl.fileformat = 1

;; **
;;; # Graphing tests
;;; 
;;; A showcase of gorilla-plot graph types. Here I load both the vega and core namespaces. This is useful for development, but if you're just using the library then you only need to load the 'gorilla-plot.core namespace.
;; **

;; @@
(use 'gorilla-plot.vega :reload)
(use 'gorilla-plot.core :reload)
(use 'clojure.pprint)
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; **
;;; Some data to look at. And some gratuitous mathematics @@x = 5@@
;;; 
;;; $$a_i = \int_0^{l_i} \sin(x) dx$$
;; **

;; @@
(def d
  [1 3 5 7 9 5 4 6 9 8 3 5 6])
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;user/d</span>","value":"#'user/d"}
;; <=

;; **
;;; List plot plots data in the style of a scatter plot by default.
;; **

;; @@
(list-plot d)
;; @@
;; =>
;;; {"type":"html","content":"<img src='data:image/png;base64,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' />"}
;; <=

;; **
;;; There are a number of options that can be given to the plot. At the moment your best bet is to peak at the source code to see what they are.
;; **

;; @@
(list-plot d :joined true :plot-size 600 :aspect-ratio 6 :opacity 0.5 :plot-range [:all :all])
;; @@
;; =>
;;; {"type":"html","content":"<img src='data:image/png;base64,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' />"}
;; <=

;; **
;;; You can also plot bar charts. The first argument is the list of category names, the second the category values.
;; **

;; @@
(bar-chart (range (count d)) d)
;; @@
;; =>
;;; {"type":"html","content":"<img src='data:image/png;base64,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' />"}
;; <=

;; **
;;; It's interesting to look at the structure of a plot.
;; **

;; @@
(def p (list-plot d))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;user/p</span>","value":"#'user/p"}
;; <=

;; **
;;; It's nothing but a clojure value, which is interpreted by Gorilla as a Vega visualisation and rendered appropriately.
;; **

;; @@
(pprint p)
;; @@
;; ->
;;; {:content
;;;  {:axes [{:scale &quot;x&quot;, :type &quot;x&quot;} {:scale &quot;y&quot;, :type &quot;y&quot;}],
;;;   :scales
;;;   [{:name &quot;x&quot;,
;;;     :type &quot;linear&quot;,
;;;     :range &quot;width&quot;,
;;;     :zero false,
;;;     :domain
;;;     {:data &quot;0b280898-0849-49c0-904c-75d2c0619b66&quot;, :field &quot;data.x&quot;}}
;;;    {:name &quot;y&quot;,
;;;     :type &quot;linear&quot;,
;;;     :range &quot;height&quot;,
;;;     :nice true,
;;;     :zero false,
;;;     :domain
;;;     {:data &quot;0b280898-0849-49c0-904c-75d2c0619b66&quot;, :field &quot;data.y&quot;}}],
;;;   :marks
;;;   [{:type &quot;symbol&quot;,
;;;     :from {:data &quot;0b280898-0849-49c0-904c-75d2c0619b66&quot;},
;;;     :properties
;;;     {:enter
;;;      {:x {:scale &quot;x&quot;, :field &quot;data.x&quot;},
;;;       :y {:scale &quot;y&quot;, :field &quot;data.y&quot;},
;;;       :fill {:value &quot;steelblue&quot;},
;;;       :fillOpacity {:value 1}},
;;;      :update
;;;      {:shape &quot;circle&quot;,
;;;       :size {:value 70},
;;;       :stroke {:value &quot;transparent&quot;}},
;;;      :hover {:size {:value 210}, :stroke {:value &quot;white&quot;}}}}],
;;;   :data
;;;   [{:name &quot;0b280898-0849-49c0-904c-75d2c0619b66&quot;,
;;;     :values
;;;     ({:x 0, :y 1}
;;;      {:x 1, :y 3}
;;;      {:x 2, :y 5}
;;;      {:x 3, :y 7}
;;;      {:x 4, :y 9}
;;;      {:x 5, :y 5}
;;;      {:x 6, :y 4}
;;;      {:x 7, :y 6}
;;;      {:x 8, :y 9}
;;;      {:x 9, :y 8}
;;;      {:x 10, :y 3}
;;;      {:x 11, :y 5}
;;;      {:x 12, :y 6})}],
;;;   :width 400,
;;;   :height 247.2188,
;;;   :padding {:bottom 20, :top 10, :right 10, :left 50}}}
;;; 
;; <-
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; **
;;; This makes it easy to tweak plot properties 'after the fact'.
;; **

;; @@
(assoc-in p [:content "marks" 0 "properties" "enter" "fill" "value"] "red")
;; @@
;; =>
;;; {"type":"html","content":"<img src='data:image/png;base64,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' />"}
;; <=

;; **
;;; Another key feature of gorilla-plot, is that because plots are values, it's straighforward to compose them. Here's another plot
;; **

;; @@
(def p2 (list-plot d :joined true :plot-range [:all [1 5]]))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;user/p2</span>","value":"#'user/p2"}
;; <=

;; **
;;; And here we compose them to form a single plot. The axes and plot range are taken from the first plot given to compose.
;; **

;; @@
(compose p p2)
;; @@
;; =>
;;; {"type":"html","content":"<img src='data:image/png;base64,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' />"}
;; <=

;; **
;;; You can also plot functions.
;; **

;; @@
(plot (fn [x] (Math/sin x)) [0 10] :color "orange" :plot-points 50)
;; @@
;; =>
;;; {"type":"html","content":"<img src='data:image/png;base64,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' />"}
;; <=

;; **
;;; And, of course, these plots compose too.
;; **

;; @@
(compose
  (list-plot (map #(vector % (rand %)) (range 0 10 0.01)) :opacity 0.3 :symbol-size 50)
  (plot (fn [x] (* x (Math/pow (Math/sin x) 2))) [0 10]))
;; @@
;; =>
;;; {"type":"html","content":"<img src='data:image/png;base64,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' />"}
;; <=

;; **
;;; There's a histogram plot type as well. As above, it composes with list-plots well.
;; **

;; @@
(compose
  (histogram (repeatedly 1000 #(rand 10)) :color "steelblue" :bins 100 :normalize :probability-density)
  (plot (constantly 0.1) [0 10]))
;; @@
;; =>
;;; {"type":"html","content":"<img src='data:image/png;base64,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' />"}
;; <=

;; **
;;; Something a little fancier!
;; **

;; @@
(compose
  (histogram (repeatedly 10000 #(- (apply + (repeatedly 50 rand)) 25))
             :bins 100
             :normalise :probability-density)
  (plot #(gaussian % 2 0) [-10 10]))
;; @@

;; @@
(compose
  (histogram (repeatedly 10000 #(- (apply + (repeatedly 50 rand)) 25))
             :bins 100
             :normalise :probability-density)
  (histogram (repeatedly 10000 #(- (apply + (repeatedly 50 rand)) 23))
             :bins 100
             :normalise :probability-density
             :colour "green")
  (plot #(gaussian % 2 0) [-10 10])
  (plot #(gaussian % 2 2) [-10 10] :colour "orange"))
;; @@

;; @@

;; @@

;; @@

;; @@
