;; gorilla-repl.fileformat = 1

;; **
;;; # Predicting power consumptions with HTM
;; **

;; **
;;; _by Marcus Lewis_
;;; 
;;; ## Abstract
;;; 
;;; Comportex, an implementation of Hierarchical Temporal Memory, is used to stream Numenta's "hotgym" data set and predict subsequent values. The data is meticulously encoded into the language of HTM, and the HTM region is tuned and observed.
;;; 
;;; ## Jump around
;;; 
;;; This page is tall. Here's some help:
;;; 
;;; - [The data](#the_data)
;;; - [Encoding the data](#encoding_the_data)
;;;   - [Timestamps](#encoding_timestamps)
;;;   - [Consumptions](#encoding_consumptions)
;;;     - Scalar encoding: Moving window
;;;       - Settle down
;;;     - Scalar encoding: Probabilistic
;;;       - Doing the math
;;;       - That took a while
;;;       - [Doing the trials](#doing_the_trials)
;;;     - Whining
;;;     - A third way
;;;   - [Put them all together](#encoding_combined)
;;; - [Design an HTM region](#design_htm_region)
;;; - [Encoding, revisited](#encoding_revisited) (spoiler alert)
;;; - Errata from my talk
;;; 
;;; ## This is code
;;; 
;;; This page is just a [ComportexViz](https://github.com/nupic-community/comportexviz) worksheet. Feel free to [tweak](link-to-worksheet) it. You might enjoy generating your own images -- every image in this page is created by the Clojure code above it.
;;; 
;;; Let's get this out of the way:
;; **

;; @@
(ns comportexviz.notebook.hotgym
  (:require [cemerick.pprng :as random]
            [clj-time.core :as t]
            [clj-time.format :as f]
            [clj-time.predicates :as pr]
            [clojure.data.csv :as csv]
            [clojure.java.io :as io]
            [clojure.set :as set]
            [comportexviz.server.launchpad :refer [start-runner
                                                   stop-all-runners]]
            [comportexviz.server.notebook :refer [viz]]
            [gorilla-plot.core :as plot]
            [org.nfrac.comportex.core :as core]
            [org.nfrac.comportex.encoders :as e]
            [org.nfrac.comportex.protocols :as p]
            [org.nfrac.comportex.repl]
            [org.nfrac.comportex.topology :as topology])
  (:use [clojure.pprint]
        [clojure.stacktrace]))

(org.nfrac.comportex.repl/truncate-large-data-structures)
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; **
;;; ## <a name="the_data" /> The data
;; **

;; **
;;; Here's a glance at the dataset:
;; **

;; @@
(def the-data
  (with-open [in-file (io/reader "data/rec-center-hourly.csv")]
    (doall (->> (csv/read-csv in-file)
                (drop 3)
                (map (fn [[timestamp-str consumption-str]]
                       [timestamp-str
                        (Double/parseDouble consumption-str)]))))))

(pprint (take 10 the-data))
;; @@
;; ->
;;; ([&quot;7/2/10 0:00&quot; 21.2]
;;;  [&quot;7/2/10 1:00&quot; 16.4]
;;;  [&quot;7/2/10 2:00&quot; 4.7]
;;;  [&quot;7/2/10 3:00&quot; 4.7]
;;;  [&quot;7/2/10 4:00&quot; 4.6]
;;;  [&quot;7/2/10 5:00&quot; 23.5]
;;;  [&quot;7/2/10 6:00&quot; 47.5]
;;;  [&quot;7/2/10 7:00&quot; 45.4]
;;;  [&quot;7/2/10 8:00&quot; 46.1]
;;;  [&quot;7/2/10 9:00&quot; 41.5])
;;; 
;; <-
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; **
;;; Each row is a `[timestamp power-consumption]` pair. The power consumption is measured in kilowatts.
;;; 
;;; Here's more data, zoomed out:
;; **

;; @@
(plot/list-plot (map (fn [i]
                       (let [[timestamp consumption] (nth the-data i)]
                         [i consumption]))
                     (range 0 200))
                :joined true
                :color "red")
;; @@
;; =>
;;; {"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"top":10,"left":50,"bottom":20,"right":10},"data":[{"name":"70b678e0-10ee-4f73-bd72-2fe1d629d7b2","values":[{"x":0,"y":21.2},{"x":1,"y":16.4},{"x":2,"y":4.7},{"x":3,"y":4.7},{"x":4,"y":4.6},{"x":5,"y":23.5},{"x":6,"y":47.5},{"x":7,"y":45.4},{"x":8,"y":46.1},{"x":9,"y":41.5},{"x":10,"y":43.4},{"x":11,"y":43.8},{"x":12,"y":37.8},{"x":13,"y":36.6},{"x":14,"y":35.7},{"x":15,"y":38.9},{"x":16,"y":36.2},{"x":17,"y":36.6},{"x":18,"y":37.2},{"x":19,"y":38.2},{"x":20,"y":14.1},{"x":21,"y":5.1},{"x":22,"y":5.0},{"x":23,"y":5.9},{"x":24,"y":22.5},{"x":25,"y":22.3},{"x":26,"y":6.0},{"x":27,"y":4.7},{"x":28,"y":4.4},{"x":29,"y":4.8},{"x":30,"y":23.4},{"x":31,"y":42.4},{"x":32,"y":52.0},{"x":33,"y":50.8},{"x":34,"y":45.3},{"x":35,"y":43.6},{"x":36,"y":40.7},{"x":37,"y":38.4},{"x":38,"y":39.9},{"x":39,"y":42.8},{"x":40,"y":42.2},{"x":41,"y":28.6},{"x":42,"y":11.6},{"x":43,"y":5.3},{"x":44,"y":5.4},{"x":45,"y":5.4},{"x":46,"y":5.4},{"x":47,"y":13.7},{"x":48,"y":22.2},{"x":49,"y":13.5},{"x":50,"y":5.0},{"x":51,"y":4.9},{"x":52,"y":4.9},{"x":53,"y":5.0},{"x":54,"y":5.2},{"x":55,"y":5.6},{"x":56,"y":42.9},{"x":57,"y":50.3},{"x":58,"y":43.5},{"x":59,"y":37.1},{"x":60,"y":38.7},{"x":61,"y":37.1},{"x":62,"y":40.0},{"x":63,"y":44.1},{"x":64,"y":39.3},{"x":65,"y":26.1},{"x":66,"y":9.9},{"x":67,"y":5.4},{"x":68,"y":5.5},{"x":69,"y":5.3},{"x":70,"y":5.3},{"x":71,"y":17.8},{"x":72,"y":22.4},{"x":73,"y":7.8},{"x":74,"y":4.7},{"x":75,"y":4.7},{"x":76,"y":4.7},{"x":77,"y":22.9},{"x":78,"y":46.5},{"x":79,"y":37.4},{"x":80,"y":38.8},{"x":81,"y":41.8},{"x":82,"y":43.5},{"x":83,"y":44.1},{"x":84,"y":38.7},{"x":85,"y":38.9},{"x":86,"y":37.9},{"x":87,"y":36.6},{"x":88,"y":39.6},{"x":89,"y":38.5},{"x":90,"y":44.8},{"x":91,"y":46.9},{"x":92,"y":38.2},{"x":93,"y":29.8},{"x":94,"y":5.4},{"x":95,"y":10.7},{"x":96,"y":22.1},{"x":97,"y":12.8},{"x":98,"y":4.7},{"x":99,"y":4.6},{"x":100,"y":4.6},{"x":101,"y":22.3},{"x":102,"y":42.5},{"x":103,"y":39.0},{"x":104,"y":38.2},{"x":105,"y":40.2},{"x":106,"y":44.2},{"x":107,"y":41.1},{"x":108,"y":40.5},{"x":109,"y":40.0},{"x":110,"y":39.1},{"x":111,"y":39.1},{"x":112,"y":37.2},{"x":113,"y":40.8},{"x":114,"y":40.1},{"x":115,"y":43.5},{"x":116,"y":36.9},{"x":117,"y":33.4},{"x":118,"y":7.0},{"x":119,"y":6.6},{"x":120,"y":22.2},{"x":121,"y":22.9},{"x":122,"y":12.0},{"x":123,"y":4.7},{"x":124,"y":4.5},{"x":125,"y":22.6},{"x":126,"y":42.6},{"x":127,"y":42.9},{"x":128,"y":39.4},{"x":129,"y":45.6},{"x":130,"y":41.7},{"x":131,"y":41.6},{"x":132,"y":39.5},{"x":133,"y":37.2},{"x":134,"y":40.1},{"x":135,"y":37.0},{"x":136,"y":39.8},{"x":137,"y":40.1},{"x":138,"y":44.8},{"x":139,"y":44.3},{"x":140,"y":36.8},{"x":141,"y":31.2},{"x":142,"y":5.2},{"x":143,"y":15.2},{"x":144,"y":22.9},{"x":145,"y":20.6},{"x":146,"y":4.8},{"x":147,"y":4.8},{"x":148,"y":4.8},{"x":149,"y":22.6},{"x":150,"y":45.3},{"x":151,"y":44.4},{"x":152,"y":39.8},{"x":153,"y":43.2},{"x":154,"y":42.3},{"x":155,"y":44.1},{"x":156,"y":37.9},{"x":157,"y":39.4},{"x":158,"y":35.8},{"x":159,"y":38.2},{"x":160,"y":37.2},{"x":161,"y":40.2},{"x":162,"y":41.0},{"x":163,"y":38.1},{"x":164,"y":38.4},{"x":165,"y":13.8},{"x":166,"y":9.5},{"x":167,"y":22.0},{"x":168,"y":20.0},{"x":169,"y":11.5},{"x":170,"y":4.7},{"x":171,"y":4.7},{"x":172,"y":4.8},{"x":173,"y":22.8},{"x":174,"y":46.1},{"x":175,"y":39.0},{"x":176,"y":38.1},{"x":177,"y":43.7},{"x":178,"y":47.1},{"x":179,"y":43.3},{"x":180,"y":40.7},{"x":181,"y":37.3},{"x":182,"y":39.0},{"x":183,"y":36.5},{"x":184,"y":38.7},{"x":185,"y":38.5},{"x":186,"y":40.3},{"x":187,"y":35.3},{"x":188,"y":15.0},{"x":189,"y":5.1},{"x":190,"y":5.3},{"x":191,"y":4.8},{"x":192,"y":19.2},{"x":193,"y":14.4},{"x":194,"y":4.7},{"x":195,"y":4.7},{"x":196,"y":4.8},{"x":197,"y":4.8},{"x":198,"y":25.4},{"x":199,"y":43.9}]}],"marks":[{"type":"line","from":{"data":"70b678e0-10ee-4f73-bd72-2fe1d629d7b2"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"y":{"scale":"y","field":"data.y"},"stroke":{"value":"red"},"strokeWidth":{"value":2},"strokeOpacity":{"value":1}}}}],"scales":[{"name":"x","type":"linear","range":"width","zero":false,"domain":{"data":"70b678e0-10ee-4f73-bd72-2fe1d629d7b2","field":"data.x"}},{"name":"y","type":"linear","range":"height","nice":true,"zero":false,"domain":{"data":"70b678e0-10ee-4f73-bd72-2fe1d629d7b2","field":"data.y"}}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"70b678e0-10ee-4f73-bd72-2fe1d629d7b2\", :values ({:x 0, :y 21.2} {:x 1, :y 16.4} {:x 2, :y 4.7} {:x 3, :y 4.7} {:x 4, :y 4.6} {:x 5, :y 23.5} {:x 6, :y 47.5} {:x 7, :y 45.4} {:x 8, :y 46.1} {:x 9, :y 41.5} {:x 10, :y 43.4} {:x 11, :y 43.8} {:x 12, :y 37.8} {:x 13, :y 36.6} {:x 14, :y 35.7} {:x 15, :y 38.9} {:x 16, :y 36.2} {:x 17, :y 36.6} {:x 18, :y 37.2} {:x 19, :y 38.2} {:x 20, :y 14.1} {:x 21, :y 5.1} {:x 22, :y 5.0} {:x 23, :y 5.9} {:x 24, :y 22.5} {:x 25, :y 22.3} {:x 26, :y 6.0} {:x 27, :y 4.7} {:x 28, :y 4.4} {:x 29, :y 4.8} {:x 30, :y 23.4} {:x 31, :y 42.4} {:x 32, :y 52.0} {:x 33, :y 50.8} {:x 34, :y 45.3} {:x 35, :y 43.6} {:x 36, :y 40.7} {:x 37, :y 38.4} {:x 38, :y 39.9} {:x 39, :y 42.8} {:x 40, :y 42.2} {:x 41, :y 28.6} {:x 42, :y 11.6} {:x 43, :y 5.3} {:x 44, :y 5.4} {:x 45, :y 5.4} {:x 46, :y 5.4} {:x 47, :y 13.7} {:x 48, :y 22.2} {:x 49, :y 13.5} {:x 50, :y 5.0} {:x 51, :y 4.9} {:x 52, :y 4.9} {:x 53, :y 5.0} {:x 54, :y 5.2} {:x 55, :y 5.6} {:x 56, :y 42.9} {:x 57, :y 50.3} {:x 58, :y 43.5} {:x 59, :y 37.1} {:x 60, :y 38.7} {:x 61, :y 37.1} {:x 62, :y 40.0} {:x 63, :y 44.1} {:x 64, :y 39.3} {:x 65, :y 26.1} {:x 66, :y 9.9} {:x 67, :y 5.4} {:x 68, :y 5.5} {:x 69, :y 5.3} {:x 70, :y 5.3} {:x 71, :y 17.8} {:x 72, :y 22.4} {:x 73, :y 7.8} {:x 74, :y 4.7} {:x 75, :y 4.7} {:x 76, :y 4.7} {:x 77, :y 22.9} {:x 78, :y 46.5} {:x 79, :y 37.4} {:x 80, :y 38.8} {:x 81, :y 41.8} {:x 82, :y 43.5} {:x 83, :y 44.1} {:x 84, :y 38.7} {:x 85, :y 38.9} {:x 86, :y 37.9} {:x 87, :y 36.6} {:x 88, :y 39.6} {:x 89, :y 38.5} {:x 90, :y 44.8} {:x 91, :y 46.9} {:x 92, :y 38.2} {:x 93, :y 29.8} {:x 94, :y 5.4} {:x 95, :y 10.7} {:x 96, :y 22.1} {:x 97, :y 12.8} {:x 98, :y 4.7} {:x 99, :y 4.6} {:x 100, :y 4.6} {:x 101, :y 22.3} {:x 102, :y 42.5} {:x 103, :y 39.0} {:x 104, :y 38.2} {:x 105, :y 40.2} {:x 106, :y 44.2} {:x 107, :y 41.1} {:x 108, :y 40.5} {:x 109, :y 40.0} {:x 110, :y 39.1} {:x 111, :y 39.1} {:x 112, :y 37.2} {:x 113, :y 40.8} {:x 114, :y 40.1} {:x 115, :y 43.5} {:x 116, :y 36.9} {:x 117, :y 33.4} {:x 118, :y 7.0} {:x 119, :y 6.6} {:x 120, :y 22.2} {:x 121, :y 22.9} {:x 122, :y 12.0} {:x 123, :y 4.7} {:x 124, :y 4.5} {:x 125, :y 22.6} {:x 126, :y 42.6} {:x 127, :y 42.9} {:x 128, :y 39.4} {:x 129, :y 45.6} {:x 130, :y 41.7} {:x 131, :y 41.6} {:x 132, :y 39.5} {:x 133, :y 37.2} {:x 134, :y 40.1} {:x 135, :y 37.0} {:x 136, :y 39.8} {:x 137, :y 40.1} {:x 138, :y 44.8} {:x 139, :y 44.3} {:x 140, :y 36.8} {:x 141, :y 31.2} {:x 142, :y 5.2} {:x 143, :y 15.2} {:x 144, :y 22.9} {:x 145, :y 20.6} {:x 146, :y 4.8} {:x 147, :y 4.8} {:x 148, :y 4.8} {:x 149, :y 22.6} {:x 150, :y 45.3} {:x 151, :y 44.4} {:x 152, :y 39.8} {:x 153, :y 43.2} {:x 154, :y 42.3} {:x 155, :y 44.1} {:x 156, :y 37.9} {:x 157, :y 39.4} {:x 158, :y 35.8} {:x 159, :y 38.2} {:x 160, :y 37.2} {:x 161, :y 40.2} {:x 162, :y 41.0} {:x 163, :y 38.1} {:x 164, :y 38.4} {:x 165, :y 13.8} {:x 166, :y 9.5} {:x 167, :y 22.0} {:x 168, :y 20.0} {:x 169, :y 11.5} {:x 170, :y 4.7} {:x 171, :y 4.7} {:x 172, :y 4.8} {:x 173, :y 22.8} {:x 174, :y 46.1} {:x 175, :y 39.0} {:x 176, :y 38.1} {:x 177, :y 43.7} {:x 178, :y 47.1} {:x 179, :y 43.3} {:x 180, :y 40.7} {:x 181, :y 37.3} {:x 182, :y 39.0} {:x 183, :y 36.5} {:x 184, :y 38.7} {:x 185, :y 38.5} {:x 186, :y 40.3} {:x 187, :y 35.3} {:x 188, :y 15.0} {:x 189, :y 5.1} {:x 190, :y 5.3} {:x 191, :y 4.8} {:x 192, :y 19.2} {:x 193, :y 14.4} {:x 194, :y 4.7} {:x 195, :y 4.7} {:x 196, :y 4.8} {:x 197, :y 4.8} {:x 198, :y 25.4} {:x 199, :y 43.9})}], :marks [{:type \"line\", :from {:data \"70b678e0-10ee-4f73-bd72-2fe1d629d7b2\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :y {:scale \"y\", :field \"data.y\"}, :stroke {:value \"red\"}, :strokeWidth {:value 2}, :strokeOpacity {:value 1}}}}], :scales [{:name \"x\", :type \"linear\", :range \"width\", :zero false, :domain {:data \"70b678e0-10ee-4f73-bd72-2fe1d629d7b2\", :field \"data.x\"}} {:name \"y\", :type \"linear\", :range \"height\", :nice true, :zero false, :domain {:data \"70b678e0-10ee-4f73-bd72-2fe1d629d7b2\", :field \"data.y\"}}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}}"}
;; <=

;; **
;;; ## Encoding the data
;; **

;; **
;;; The data are:
;;; 
;;; - timestamps
;;; - power consumption rates
;;; 
;;; We need to feed this information into a region of cortex. To do this, we come up with an encoding scheme, converting each value into a set of bits. These bits should illustrate commonalities between values.
;; **

;; **
;;; ### Timestamps
;; **

;; **
;;; The timestamps contain a date and a time. They come in 1-hour intervals.
;; **

;; @@
(def timestamps
  (->> the-data
       (map (fn [[timestamp consumption]]
              timestamp))))

(take 10 timestamps)
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;7/2/10 0:00&quot;</span>","value":"\"7/2/10 0:00\""},{"type":"html","content":"<span class='clj-string'>&quot;7/2/10 1:00&quot;</span>","value":"\"7/2/10 1:00\""},{"type":"html","content":"<span class='clj-string'>&quot;7/2/10 2:00&quot;</span>","value":"\"7/2/10 2:00\""},{"type":"html","content":"<span class='clj-string'>&quot;7/2/10 3:00&quot;</span>","value":"\"7/2/10 3:00\""},{"type":"html","content":"<span class='clj-string'>&quot;7/2/10 4:00&quot;</span>","value":"\"7/2/10 4:00\""},{"type":"html","content":"<span class='clj-string'>&quot;7/2/10 5:00&quot;</span>","value":"\"7/2/10 5:00\""},{"type":"html","content":"<span class='clj-string'>&quot;7/2/10 6:00&quot;</span>","value":"\"7/2/10 6:00\""},{"type":"html","content":"<span class='clj-string'>&quot;7/2/10 7:00&quot;</span>","value":"\"7/2/10 7:00\""},{"type":"html","content":"<span class='clj-string'>&quot;7/2/10 8:00&quot;</span>","value":"\"7/2/10 8:00\""},{"type":"html","content":"<span class='clj-string'>&quot;7/2/10 9:00&quot;</span>","value":"\"7/2/10 9:00\""}],"value":"(\"7/2/10 0:00\" \"7/2/10 1:00\" \"7/2/10 2:00\" \"7/2/10 3:00\" \"7/2/10 4:00\" \"7/2/10 5:00\" \"7/2/10 6:00\" \"7/2/10 7:00\" \"7/2/10 8:00\" \"7/2/10 9:00\")"}
;; <=

;; **
;;; We're free to add outside information about those dates. You could add the day of week, information about holidays or weekends, etc.
;;; 
;;; One pragmatic approach:
;;; 
;;; - Compress the date down to the "day of week"
;;; - Compress the time down to the "hour"
;; **

;; @@
(def parsed-timestamps (->> timestamps
                            (map (partial f/parse (f/formatter "MM/dd/yy HH:mm")))))

(def input-days (->> parsed-timestamps
                     (map t/day-of-week)))

(def input-hours (->> parsed-timestamps
                      (map t/hour)))

(->> (map vector parsed-timestamps input-days input-hours)
     (take-nth 12)
     (take 8)
     (map (fn [[parsed d h]]
            [(str parsed) d h]))
     pprint)
;; @@
;; ->
;;; ([&quot;2010-07-02T00:00:00.000Z&quot; 5 0]
;;;  [&quot;2010-07-02T12:00:00.000Z&quot; 5 12]
;;;  [&quot;2010-07-03T00:00:00.000Z&quot; 6 0]
;;;  [&quot;2010-07-03T12:00:00.000Z&quot; 6 12]
;;;  [&quot;2010-07-04T00:00:00.000Z&quot; 7 0]
;;;  [&quot;2010-07-04T12:00:00.000Z&quot; 7 12]
;;;  [&quot;2010-07-05T00:00:00.000Z&quot; 1 0]
;;;  [&quot;2010-07-05T12:00:00.000Z&quot; 1 12])
;;; 
;; <-
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; **
;;; So let's choose an encoding strategy for "day of week" and "hour of day".
;;; 
;;; The two are similar. A range of numbers where order is meaningful. If we use a "moving window" encoding strategy, we have two decisions to make:
;;; 
;;; - Interrelation radius
;;;   - i.e. how far should the window reach?
;;;   - The encoding will illustrate relations between any two numbers whose distance is within 2 * this number.
;;; - Difference between adjacent numbers
;;;   - i.e. how far do we move the window on each increment?
;;;   - Make it easier for an HTM to see a difference between the numbers 4 and 5.
;;; 
;;; Assuming we want to use every bit, the total number of bits and the number-of-active bits can be calculated from these alone.
;; **

;; @@
(defn ranged-int-encoder
  [start end radius step-distance]
  (let [n-extra (long (* 2 radius step-distance))
        n-active (+ step-distance n-extra)
        n-bits (-> (- end start)
                   (* step-distance)
                   (+ n-extra)
                   Math/ceil
                   long)]
    ;; To avoid rounding errors in the linear encoder,
    ;; give an `end` that's one higher than the max.
    (e/linear-encoder [n-bits] n-active [start end])))

(def day-of-week-encoder
  (partial ranged-int-encoder 0 7))

(def hour-of-day-encoder
  (partial ranged-int-encoder 0 24))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;comportexviz.notebook.hotgym/hour-of-day-encoder</span>","value":"#'comportexviz.notebook.hotgym/hour-of-day-encoder"}
;; <=

;; **
;;; Later we can see how the HTM performs with different parameters. Here's a glance at the options.
;;; 
;;; Play with the `radius`:
;; **

;; @@
(for [i (range 24)]
  [i (viz (hour-of-day-encoder 1 1) i)])
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>0</span>","value":"0"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAhElEQVQ4T2P8z8DwnwEKfvr5Hfy2YIE9jP////9GRkbGeor4NLfgh5/fAZgLGSQkNnxtbxeA8ZmYmA78+/fPgRI+I7V9ICws3AB3MAMDA+0toHYQCQoKIoKcLj6geRzQPIjevXsHz2jUyFgYqYjmFrx9+xaebqmRsTBSEc19QHMLaB1EANIZVroPDSE9AAAAAElFTkSuQmCC' />"}],"value":"[0 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x4148bcd4 \"comportexviz.server.notebook$eval28199$fn$reify__28203@4148bcd4\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>1</span>","value":"1"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAhElEQVQ4T2P8z8DwnwEKfvr5Hfy2YIE9jP////9GRkbGeor4NLfgh5/fAZgLGSQkNnxtbxeA8ZmYmA78+/fPgRI+I7V9ICws3AB3MAMDA+0toHYQCQoKIoKcLj6geRzQPIjevXsHz2jUyFgYqYjmFrx9+xaebqmRsTBSEc19QHMLaB1EANIZVroPDSE9AAAAAElFTkSuQmCC' />"}],"value":"[1 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2f2bac74 \"comportexviz.server.notebook$eval28199$fn$reify__28203@2f2bac74\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>2</span>","value":"2"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAf0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Vfk0t+Dt27cNMBczMTEd+PfvnwM1+Yz/GRjgQfTTz+/gtwUL7CkJImFhYbiDQebQ3oIffn4HYC5mkJDY8LW9XYCSIBIUFESYRxcf0DwORoMIPedjJFOaBxG1iwoMH9DcAmqXRegZDQB261a69bxyCwAAAABJRU5ErkJggg==' />"}],"value":"[2 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2fb71435 \"comportexviz.server.notebook$eval28199$fn$reify__28203@2fb71435\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>3</span>","value":"3"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAeElEQVQ4T2N89+7dfwYo+P//fyMjI2M9Vfk0t+Dt27cNMBczMTEd+PfvnwM1+YzU9oGwsDDcwSCH0t4CageRoKDgAVgQg33wn4EBnop++vkd/LZggT0lqQgjiGhuwQ8/P4SXJCQ2fG1vF6AkFY0GEcGMOBpEJAcRABu9VrptHtpFAAAAAElFTkSuQmCC' />"}],"value":"[3 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x6536480b \"comportexviz.server.notebook$eval28199$fn$reify__28203@6536480b\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>4</span>","value":"4"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAnUlEQVQ4T2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGRkZ4fIg9QT5NLfg7du3DTAXMjExHeCMj4fzGSQkNnxtbxdAlv/3758DKXxGdB8Q9DKBIBIWFkY4kIGBgfYWoAcRqUGArl5QUPAALAhBNO19QPM4oHkQ/WdggGc0amQsjFREcwt++PkhYp0KGQsjFdHcBzS3gNZBBAAYNVm6wO8a5gAAAABJRU5ErkJggg==' />"}],"value":"[4 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x619ce665 \"comportexviz.server.notebook$eval28199$fn$reify__28203@619ce665\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>5</span>","value":"5"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAoUlEQVQ4T2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGRkZ4fIg9QT5NLfg7du3DTAXMjExHeCMj4fzGSQkNnxtbxdAlv/3758DKXxGavtAWFgY4UAGBgbaW0DtIBIUFDwAC0IQjeEDgqmCQCoiGERUtwA9iEhNJejqR4PoAKEgxAii/wwM8LKIGmUPRiqiuQU//PwQGYMKZQ96EAEAq3FculxT3PQAAAAASUVORK5CYII=' />"}],"value":"[5 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1326be2e \"comportexviz.server.notebook$eval28199$fn$reify__28203@1326be2e\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>6</span>","value":"6"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAkklEQVQ4T2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGRkZ4fIg9QT5NLfg7du3DTAXMjExHeCMj4fzGSQkNnxtbxdAlv/3758DKXxGavtAWFgY4UAGBgbaW0DtIBIUFDwAC0IQTXsf0DwO6B5EBDMOgYxGMBVR3QL0ICI1I6GrJ5iKqO4D9FREdQtoHUQA1YBfumDcJc4AAAAASUVORK5CYII=' />"}],"value":"[6 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x4a716c3 \"comportexviz.server.notebook$eval28199$fn$reify__28203@4a716c3\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>7</span>","value":"7"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAg0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Vfk0t+Dt27cNMBczMTEd+PfvnwM1+YzoPhASFoYH0U8/v4PfFiywJyXIhIWF4Q4G6aO9BehBxBkfj3CBhMSGr+3tAqQEmaCg4AGYevr4gOZxMBpE6DkfI5nSPYgoLYsIZjSqW0Dtsgg9owEAPrxfundxNV4AAAAASUVORK5CYII=' />"}],"value":"[7 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x5386604c \"comportexviz.server.notebook$eval28199$fn$reify__28203@5386604c\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>8</span>","value":"8"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAe0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Vfk0t+Dt27cNMBczMTEd+PfvnwM1+YzU9oGwsDDcwSCH0t4CageRoKDgAVgQY/WBkLAwPBX99PM7+G3BAntSUhXBIKK6BehBxBkfj4gkCYkNX9vbBUhJVaNBRDAjjgYRyUEEAKfpX7ptBRT5AAAAAElFTkSuQmCC' />"}],"value":"[8 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x552f1b97 \"comportexviz.server.notebook$eval28199$fn$reify__28203@552f1b97\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>9</span>","value":"9"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAnElEQVQ4T2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd/DbggX2+NSj62ekuQVv375tgLmIiYnpwL9//xyQ+Zzx8XB5BgmJDV/b2wXwqUfXT9AHGF5GC0J0eWFhYYSDGBgYaG8BoSBC9zIhvqCg4AFYEIJo2vuAUCqiOA7oHkSUZiyCqYjqFqAHEaUZi2AqoroP0FMR1S2gdRABAHHLX7opIBmpAAAAAElFTkSuQmCC' />"}],"value":"[9 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x73de18d2 \"comportexviz.server.notebook$eval28199$fn$reify__28203@73de18d2\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>10</span>","value":"10"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAm0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd/DbggX2+NSj62ekuQVv375tgLmIiYnpwL9//xyQ+Zzx8XB5BgmJDV/b2wXwqUfXT3UfCAsLIxzEwMBAewuoHUSCgoIHYEEIogn6ACNVoKUydHmSg4hiCwgFEXqqIMQfDSKMjIgeZASDiNKyh2AqoroF6KmI0rIHPYgAO61fuii1L5wAAAAASUVORK5CYII=' />"}],"value":"[10 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x4f731964 \"comportexviz.server.notebook$eval28199$fn$reify__28203@4f731964\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>11</span>","value":"11"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAl0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd/DbggX2+NSj62ekuQVv375tgLmIiYnpwL9//xyQ+Zzx8XB5BgmJDV/b2wXwqUfXT3UfCAsLIxzEwMBAewuoHUSCgoIHYEEIomnvA2qnIow4GPAgwsg4aBkRXZ7kVESxBYSCCD3jEOKTnIoo9gGhVESxBbQOIgAFj1+6Lh6Q2gAAAABJRU5ErkJggg==' />"}],"value":"[11 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2f1794c6 \"comportexviz.server.notebook$eval28199$fn$reify__28203@2f1794c6\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>12</span>","value":"12"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAhUlEQVQ4T2N89+7dfwYo+P//fyMjI2M9Vfk0t+Dt27cNMBczMTEd+PfvnwM1+YyEfCAkLAwPsp9+fge/LVhgjy8IhYWF4Q4GqaO9BYSCiDM+HuEiCYkNX9vbBfAFoaCg4AGYPH18QPM4GA0i9JyPkUwHPIhILZtIzmgUW0AoiEgtm9AzGgBuvF+6vYnnRgAAAABJRU5ErkJggg==' />"}],"value":"[12 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x249d110 \"comportexviz.server.notebook$eval28199$fn$reify__28203@249d110\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>13</span>","value":"13"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAe0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Vfk0t+Dt27cNMBczMTEd+PfvnwM1+YzU9oGwsDDcwSCH0t4CageRoKDgAVgQE+UDIWFheKr66ed38NuCBfb4UhnJQUSxBYSCiDM+HhFpEhIbvra3C+BLZaNBNBpEmGURpakIANfpX7qPQh6qAAAAAElFTkSuQmCC' />"}],"value":"[13 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1524103 \"comportexviz.server.notebook$eval28199$fn$reify__28203@1524103\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>14</span>","value":"14"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAkElEQVQ4T2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef38FvCxbY49VPcwvevn3bAHMBExPTgX///jng43PGx8PVM0hIbPja3i6ATz0jqT4gFITCwsIIBzAwMNDeAlKDiFAQCgoKHoAFGYimvQ9oHgcDHkSkZiySUxHFFhAKIlIzFsmpiGIfEEpFFFtA6yACAKHLX7ojCTEGAAAAAElFTkSuQmCC' />"}],"value":"[14 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x563dc074 \"comportexviz.server.notebook$eval28199$fn$reify__28203@563dc074\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>15</span>","value":"15"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAj0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef38FvCxbY49VPcwvevn3bAHMBExPTgX///jng43PGx8PVM0hIbPja3i6ATz0jtX0gLCyMcAADAwPtLaB2EAkKCh6ABRmIJtkHhFIZxUFEsgWkBhGhVDYaRAQzIslBRGrZQ3IqotgCQqmI1LIHPYgAa61futjTxx0AAAAASUVORK5CYII=' />"}],"value":"[15 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x459fcfd1 \"comportexviz.server.notebook$eval28199$fn$reify__28203@459fcfd1\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>16</span>","value":"16"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAi0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef38FvCxbY49VPcwvevn3bAHMBExPTgX///jng43PGx8PVM0hIbPja3i6ATz0jtX0gLCyMcAADAwPtLaB2EAkKCh6ABRmIpr0PaB4Hgy6ICGVEilMRyRaQGkSEMiLFqYhkH5Caiki2gNZBBAA1j1+6IrL/8gAAAABJRU5ErkJggg==' />"}],"value":"[16 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x9124723 \"comportexviz.server.notebook$eval28199$fn$reify__28203@9124723\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>17</span>","value":"17"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAgklEQVQ4T2N89+7dfwYo+P//fyMjI2M9Vfk0t+Dt27cNMBczMTEd+PfvnwM1+Yyk+kBIWBgehD/9/A5+W7DAHjlIhYWF4Q4GidPeAlKDiDM+HuFCCYkNX9vbBZCDVFBQ8ACMTx8f0DwORoOIYDIddEFEqKyiOKORbAGpQUSorELPaACevF+6jPVARwAAAABJRU5ErkJggg==' />"}],"value":"[17 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1adb555b \"comportexviz.server.notebook$eval28199$fn$reify__28203@1adb555b\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>18</span>","value":"18"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAfElEQVQ4T2N89+7dfwYo+P//fyMjI2M9Vfk0t+Dt27cNMBczMTEd+PfvnwM1+YzU9oGwsDDcwSCH0t4CageRoKDgAVgQk+UDIWFheCr76ed38NuCBfbIqY7iICLZAlKDiDM+HhGJEhIbvra3CyCnutEgOjAaRATLIlKDCAAH+F+6hH9iegAAAABJRU5ErkJggg==' />"}],"value":"[18 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x22dd0814 \"comportexviz.server.notebook$eval28199$fn$reify__28203@22dd0814\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>19</span>","value":"19"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAj0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fge/LVhgj6Kf5ha8ffu2AWYjExPTgX///jmQwueMj4frZ5CQ2PC1vV0AWT8jpT5AD1JhYWGEhQwMDLS3gNIgQg9SQUHBA7AgAtG09wHN42DQBRGhjEVxKiLZAlKDiFDGojgVkewDUlMRyRbQOogA0ctfuiv4O7MAAAAASUVORK5CYII=' />"}],"value":"[19 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1907aad9 \"comportexviz.server.notebook$eval28199$fn$reify__28203@1907aad9\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>20</span>","value":"20"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAl0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fge/LVhgj6Kf5ha8ffu2AWYjExPTgX///jmQwueMj4frZ5CQ2PC1vV0AWT8jtX3AsWkT3IEgi2hvAbWDiGPWrAmwIKKKD9BTnbCwMCJOqBFEBC2gNIjQU52goOCB0SDCmxEpDiJCZQ/FqYhkC0hNRYTKHvQgAgB1nl+6QiNMIAAAAABJRU5ErkJggg==' />"}],"value":"[20 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x237dd85 \"comportexviz.server.notebook$eval28199$fn$reify__28203@237dd85\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>21</span>","value":"21"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAkklEQVQ4T2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fge/LVhgj6Kf5ha8ffu2AWYjExPTgX///jmQwueMj4frZ5CQ2PC1vV0AWT8jtX3AsWkT3IEgi2hvAbWDiGPWrAmwIKKPD2geB4M+iNAzprCwMCLZUiMVEbSA0iBCz5iCgoIHqJqKCPqA0lRE0AJaBxEAGXFfuo6EysAAAAAASUVORK5CYII=' />"}],"value":"[21 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x42862115 \"comportexviz.server.notebook$eval28199$fn$reify__28203@42862115\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>22</span>","value":"22"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAgUlEQVQ4T2N89+7dfwYo+P//fyMjI2M9Vfk0t+Dt27cNMBczMTEd+PfvnwM1+YyU+kBIWBgepD/9/A5ybNoEdyDIobS3gNIg4oyPhwcxg4TEBo5ZsybAgpg+PqB5HIwGEcFkOuiDCL3sEhYWRiRbamQ0ghZQGkToZZegoOAB5IwGAFyPX7qCcwXsAAAAAElFTkSuQmCC' />"}],"value":"[22 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x8d10cf9 \"comportexviz.server.notebook$eval28199$fn$reify__28203@8d10cf9\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>23</span>","value":"23"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAPCAYAAAD+pA/bAAAAd0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Vfk0t+Dt27cNMBczMTEd+PfvnwM1+YzU9oGwsDDcwSCH0t4CageRoKDgAVgQU8UHQsLC8FT308/vIMemTfA4pI8FlAYRZ3w8IlIlJDZwzJo1YTSIUDLiaBARLJsIBREAxbxfumG5eAsAAAAASUVORK5CYII=' />"}],"value":"[23 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2ca43e58 \"comportexviz.server.notebook$eval28199$fn$reify__28203@2ca43e58\"]]"}],"value":"([0 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x4148bcd4 \"comportexviz.server.notebook$eval28199$fn$reify__28203@4148bcd4\"]] [1 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2f2bac74 \"comportexviz.server.notebook$eval28199$fn$reify__28203@2f2bac74\"]] [2 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2fb71435 \"comportexviz.server.notebook$eval28199$fn$reify__28203@2fb71435\"]] [3 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x6536480b \"comportexviz.server.notebook$eval28199$fn$reify__28203@6536480b\"]] [4 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x619ce665 \"comportexviz.server.notebook$eval28199$fn$reify__28203@619ce665\"]] [5 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1326be2e \"comportexviz.server.notebook$eval28199$fn$reify__28203@1326be2e\"]] [6 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x4a716c3 \"comportexviz.server.notebook$eval28199$fn$reify__28203@4a716c3\"]] [7 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x5386604c \"comportexviz.server.notebook$eval28199$fn$reify__28203@5386604c\"]] [8 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x552f1b97 \"comportexviz.server.notebook$eval28199$fn$reify__28203@552f1b97\"]] [9 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x73de18d2 \"comportexviz.server.notebook$eval28199$fn$reify__28203@73de18d2\"]] [10 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x4f731964 \"comportexviz.server.notebook$eval28199$fn$reify__28203@4f731964\"]] [11 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2f1794c6 \"comportexviz.server.notebook$eval28199$fn$reify__28203@2f1794c6\"]] [12 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x249d110 \"comportexviz.server.notebook$eval28199$fn$reify__28203@249d110\"]] [13 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1524103 \"comportexviz.server.notebook$eval28199$fn$reify__28203@1524103\"]] [14 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x563dc074 \"comportexviz.server.notebook$eval28199$fn$reify__28203@563dc074\"]] [15 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x459fcfd1 \"comportexviz.server.notebook$eval28199$fn$reify__28203@459fcfd1\"]] [16 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x9124723 \"comportexviz.server.notebook$eval28199$fn$reify__28203@9124723\"]] [17 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1adb555b \"comportexviz.server.notebook$eval28199$fn$reify__28203@1adb555b\"]] [18 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x22dd0814 \"comportexviz.server.notebook$eval28199$fn$reify__28203@22dd0814\"]] [19 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1907aad9 \"comportexviz.server.notebook$eval28199$fn$reify__28203@1907aad9\"]] [20 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x237dd85 \"comportexviz.server.notebook$eval28199$fn$reify__28203@237dd85\"]] [21 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x42862115 \"comportexviz.server.notebook$eval28199$fn$reify__28203@42862115\"]] [22 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x8d10cf9 \"comportexviz.server.notebook$eval28199$fn$reify__28203@8d10cf9\"]] [23 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2ca43e58 \"comportexviz.server.notebook$eval28199$fn$reify__28203@2ca43e58\"]])"}
;; <=

;; @@
(for [i (range 24)]
  [i (viz (hour-of-day-encoder 4 1) i)])
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>0</span>","value":"0"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAiklEQVQ4T2P8z8DwnwEKfvr5HWTftMkemf9twQI4/////42MjIz1MHmi+DS34Ief3wGYixgkJDYwvHgRgMz/2t4uAOMzMTEd+PfvnwMpfEaa+4DmFgy5IBIWFm6AxyEDAwPV4wDDAmoHkaCgICJVYvMBpRmLYBBR3QL0IKI0Y40GEcGyaDSISA4iAElCllhcQnLLAAAAAElFTkSuQmCC' />"}],"value":"[0 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1e062204 \"comportexviz.server.notebook$eval28199$fn$reify__28203@1e062204\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>1</span>","value":"1"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAiklEQVQ4T2P8z8DwnwEKfvr5HWTftMkemf9twQI4/////42MjIz1MHmi+DS34Ief3wGYixgkJDYwvHgRgMz/2t4uAOMzMTEd+PfvnwMpfEaa+4DmFgy5IBIWFm6AxyEDAwPV4wDDAmoHkaCgICJVYvMBpRmLYBBR3QL0IKI0Y40GEcGyaDSISA4iAElCllhcQnLLAAAAAElFTkSuQmCC' />"}],"value":"[1 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2a11e6f2 \"comportexviz.server.notebook$eval28199$fn$reify__28203@2a11e6f2\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>2</span>","value":"2"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAsElEQVQ4T2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGRkZ4fIg9QT5NLfg7du3DTAXMjExHeCMj4fzGSQkNnxtbxdAlv/3758DKXzG/wwM8CACBQn7pk3wIKFKENHcgh9+fgdgXgYFCcOLFwHI/EEXRMLCwog4ZGBgoHocYFhA7SASFBREBDldfEDtVDTwQURp2UMwFVHdAvRURGnGIpiKqO4D9FREdQtoHUQAMKqZWLQbiIcAAAAASUVORK5CYII=' />"}],"value":"[2 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x6701b3ab \"comportexviz.server.notebook$eval28199$fn$reify__28203@6701b3ab\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>3</span>","value":"3"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAArElEQVQ4T2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGRkZ4fIg9QT5NLfg7du3DTAXMjExHeCMj4fzGSQkNnxtbxdAlv/3758DKXxGmvuA5hbQPIj+MzDAUxEo1bBv2gRPNeSkImFhYUQcMjAwMNLcgh9+fgdgqQKUahhevAhA5pOaigQFBRHm0cUHo0GEXhaNgFREafFMMIiobgF6RiM1Y6EX3+gZDQCu1pxYSlwzBQAAAABJRU5ErkJggg==' />"}],"value":"[3 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x385bbdbd \"comportexviz.server.notebook$eval28199$fn$reify__28203@385bbdbd\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>4</span>","value":"4"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAp0lEQVQ4T2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGRkZ4fIg9QT5NLfg7du3DTAXMjExHeCMj4fzGSQkNnxtbxdAlv/3758DKXxGmvuA5hYMuSASFhZGxCEDAwPV4wDDAmoHkaCg4AFYKgPRjP8ZGOAZDZSx2DdtgmcscjIahg9obsEPPz+ElyQkNjC8eBEA9yIZGW00iA4QKotGg4jkIAIAw9WfWFfNsDIAAAAASUVORK5CYII=' />"}],"value":"[4 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x17e2aeb0 \"comportexviz.server.notebook$eval28199$fn$reify__28203@17e2aeb0\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>5</span>","value":"5"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAtElEQVQ4T2N89+7dfwYo+P//f6OQsHA9jP/Tz+8g+6ZN9sj8bwsWwPkg9YyMjHD1WPk0t+Dt27cNMBcyMTEd4IyPh/MZJCQ2MLx4EQCTB/G/trcLIKv/9++fAz4+IyEfkBwk6EFGcwsIBRGpQYIeZFQPImFhYUQcMjAw0N4CageRoKDgAXiqo4sPqJ2KMOKA5kH0n4EBXhZRo+zB8AHNLfjh54eIdSqUPRipiOY+oLkFtA4iAK0+olhTSnmlAAAAAElFTkSuQmCC' />"}],"value":"[5 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x3261ecc8 \"comportexviz.server.notebook$eval28199$fn$reify__28203@3261ecc8\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>6</span>","value":"6"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAqElEQVQ4T2N89+7dfwYo+P//f6OQsHA9jP/Tz+8g+6ZN9sj8bwsWwPkg9YyMjHD1WPk0t+Dt27cNMBcyMTEd4IyPh/MZJCQ2MLx4EQCTB/G/trcLIKv/9++fAz4+I819QHMLBjyISE01wsLCiDhkYGAgGAcUW0AoiEhNNYKCggfgqY4uPiCUikaDaBikov8MDPDimhrFM0ZGo7kFP/z8EBmDCsUzekYDAC16pVhBaNtnAAAAAElFTkSuQmCC' />"}],"value":"[6 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x58179dc4 \"comportexviz.server.notebook$eval28199$fn$reify__28203@58179dc4\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>7</span>","value":"7"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAmklEQVQ4T2N89+7dfwYo+P//f6OQsHA9jP/Tz+8g+6ZN9sj8bwsWwPkg9YyMjHD1WPk0t+Dt27cNMBcyMTEd4IyPh/MZJCQ2MLx4EQCTB/G/trcLIKv/9++fAz4+I819QHMLhlwQCQsLI+KQgYGB6nGAYQG1g0hQUPAAPNUR4wNSMxbJQUSxBYSCiNSMNRpEBwiVPaNBRHIQAQBEeqhYjtiAHAAAAABJRU5ErkJggg==' />"}],"value":"[7 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x3d44652e \"comportexviz.server.notebook$eval28199$fn$reify__28203@3d44652e\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>8</span>","value":"8"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAr0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd/DbggX2+NSj62ekuQVv375tgLmIiYnpwL9//xyQ+Zzx8XB5BgmJDV/b2wXwqUfXj+ED9CBh37QJHiRUCSKqW4AeROhBwvDiRQAsSAZFEAkLCyPijIGBgepxgGEBtYNIUFDwADxI6eID9IxGaSoafEFEatlDciqi2AJCqYjUsofkVESxDwilIootoHUQAQC9p6hYQJAYHgAAAABJRU5ErkJggg==' />"}],"value":"[8 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x3f51db18 \"comportexviz.server.notebook$eval28199$fn$reify__28203@3f51db18\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>9</span>","value":"9"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAoUlEQVQ4T2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd/DbggX2+NSj62ekuQVv375tgLmIiYnpwL9//xyQ+Zzx8XB5BgmJDV/b2wXwqUfXT3sfDL8gQk817Js2wVMNMalIWFgYEWcMDAwYcUB1C9BTEXqqYXjxIgCWaohJRYKCggfg6uniA/RUNBpEIzAVkVo8kxxEFFtAKKORWjyjZzQANuOoWAq1xpAAAAAASUVORK5CYII=' />"}],"value":"[9 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x26b6796 \"comportexviz.server.notebook$eval28199$fn$reify__28203@26b6796\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>10</span>","value":"10"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAnklEQVQ4T2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd/DbggX2+NSj62ekuQVv375tgLmIiYnpwL9//xyQ+Zzx8XB5BgmJDV/b2wXwqUfXT3sfjAYReqoRFhZGxBkDAwPV4wDDAmqnIkFBwQOwVAaiMXyAnrHYN22CZyxiMhrBIKK6BehBhJ6xGF68CIB7mYiMNhpEBMui0SAiOYgAsBCoWGQCxl8AAAAASUVORK5CYII=' />"}],"value":"[10 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x66f633a2 \"comportexviz.server.notebook$eval28199$fn$reify__28203@66f633a2\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>11</span>","value":"11"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAsUlEQVQ4T2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDyI/23BAjgfm35Gmlvw9u3bBpiLmJiYDvz7988Bmc8ZHw+XZ5CQ2MDw4kUATB7E/9reLoBPP8k+IBgkaEFMewtIDSJCQYIexFT3gbCwMCLOGBgYaG8BtYNIUFDwADyV0cUHpGY0QskUIw7oHkSUlj0EUxHVLUAPIkrLHoKpiOo+QE9FVLeA1kEEAGbjqFjGX3pGAAAAAElFTkSuQmCC' />"}],"value":"[11 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x78855747 \"comportexviz.server.notebook$eval28199$fn$reify__28203@78855747\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>12</span>","value":"12"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAn0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDyI/23BAjgfm35Gmlvw9u3bBpiLmJiYDvz7988Bmc8ZHw+XZ5CQ2MDw4kUATB7E/9reLoBPP+19MPKCiFCqERYWRsQZAwMDyXFAsgWkpiJCqUZQUPAAPJXRxQekpqLRICJY2A3+IKK0eCaY0ahuAXpGo7R4Rs9oAB22qFg4ua3UAAAAAElFTkSuQmCC' />"}],"value":"[12 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x493dda54 \"comportexviz.server.notebook$eval28199$fn$reify__28203@493dda54\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>13</span>","value":"13"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAmElEQVQ4T2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDyI/23BAjgfm35Gmlvw9u3bBpiLmJiYDvz7988Bmc8ZHw+XZ5CQ2MDw4kUATB7E/9reLoBPP+19MBpE6KlIWFgYEWcMDAxUjwMMC6idigQFBQ/AUxk5PiCUsSgOIpItIDWICGWs0SAiWPaMBhHJQQQA1HqoWPrp8tQAAAAASUVORK5CYII=' />"}],"value":"[13 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x10b34332 \"comportexviz.server.notebook$eval28199$fn$reify__28203@10b34332\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>14</span>","value":"14"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAqElEQVQ4T2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef38FvCxbY49VPcwvevn3bAHMBExPTgX///jng43PGx8PVM0hIbPja3i6ATz0jIR+gBwn7pk3wIKFKEFFsAaEgQg8ShhcvAmBBMiBBJCwsjIgjBgYGqscBhgXUDiJBQcED8CCkiw+onUwHfxARKnsoTkUkW0BqKiJU9lCcikj2AampiGQLaB1EAE22qFi5wC70AAAAAElFTkSuQmCC' />"}],"value":"[14 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x478f09a5 \"comportexviz.server.notebook$eval28199$fn$reify__28203@478f09a5\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>15</span>","value":"15"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAoElEQVQ4T2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef38FvCxbY49VPcwvevn3bAHMBExPTgX///jng43PGx8PVM0hIbPja3i6ATz0jzX1AcwsGPIjQUw37pk3wVIMtFQkLCyPiiIGBgWAcUGwBoSBCTzUML14EwFINtlQkKCh4AC5PFx8QSkWjQTQCUxGh4pnijEayBaRmNELFM3pGAwDG46hYz43bEQAAAABJRU5ErkJggg==' />"}],"value":"[15 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1562517c \"comportexviz.server.notebook$eval28199$fn$reify__28203@1562517c\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>16</span>","value":"16"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAnElEQVQ4T2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef38FvCxbY49VPcwvevn3bAHMBExPTgX///jng43PGx8PVM0hIbPja3i6ATz0jzX1AcwuGXBAJCwsj4oiBgYHqcYBhAbWDSFBQ8AAsVYFogj5Az1jsmzbBMxa2jEZyEFFsAaEgQs9YDC9eBMCDAEtGGw2iDaNBhFFcU5qKAEAfqFgzFRHUAAAAAElFTkSuQmCC' />"}],"value":"[16 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x32aad6fe \"comportexviz.server.notebook$eval28199$fn$reify__28203@32aad6fe\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>17</span>","value":"17"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAsElEQVQ4T2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef30H2TZvsYepB/G8LFsD5YPNobsHbt28bYC5gYmI68O/fPwd8fM74eLh6BgmJDQwvXgTA1IP4X9vbBZD1U+wDjCBBD2JKg4igBZQGEXqQoAcx1YNIWFgYEUcMDAy0t4DaQSQoKHgAnqro4gNqpyKMOBjwICK17CE5FVFsAaEgIrXsITkVUewDQqmIYgtoHUQA9uOoWGrEyo8AAAAASUVORK5CYII=' />"}],"value":"[17 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x388ab94a \"comportexviz.server.notebook$eval28199$fn$reify__28203@388ab94a\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>18</span>","value":"18"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAoUlEQVQ4T2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef30H2TZvsYepB/G8LFsD5YPNobsHbt28bYC5gYmI68O/fPwd8fM74eLh6BgmJDQwvXgTA1IP4X9vbBZD1094Ho0GEnmqEhYURccTAwEBxHBC0gNJUhJ5qBAUFD8BTFV18QGkqGg0ijMIPvbAb/EFEavFMckaj2AJCGY3U4hk9owEArbaoWPxGAYYAAAAASUVORK5CYII=' />"}],"value":"[18 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7775ddf \"comportexviz.server.notebook$eval28199$fn$reify__28203@7775ddf\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>19</span>","value":"19"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAo0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef30H2TZvsYepB/G8LFsD5YPNobsHbt28bYC5gYmI68O/fPwd8fM74eLh6BgmJDQwvXgTA1IP4X9vbBZD1094Ho0GEnoo4Nm2CxyEoLqgeBxgWUDsVccyaNQGeqqjhA/SMJSwsjEjGdLGA0iBCz1iCgoIHRoMIpSwaDSKM4hu9uCYURAA+eqhYPQvgIgAAAABJRU5ErkJggg==' />"}],"value":"[19 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1905fc \"comportexviz.server.notebook$eval28199$fn$reify__28203@1905fc\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>20</span>","value":"20"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAsUlEQVQ4T2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fge/LVhgj6Kf5ha8ffu2AWYjExPTgX///jmQwueMj4frZ5CQ2PC1vV0AWT8jqT5ADxL2TZvgQUKVICLZAlKDCD1IGF68CIAFCV2CiGPTJngcgiymehxgWEDtIOKYNWsCPMjo4gNqJ9OhF0ToZY+wsDAiZ1MjDghaQGkqQi97BAUFD1A1FRH0AaWpiKAFtA4iAJGYqFhBP+evAAAAAElFTkSuQmCC' />"}],"value":"[20 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x64e02e27 \"comportexviz.server.notebook$eval28199$fn$reify__28203@64e02e27\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>21</span>","value":"21"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAArElEQVQ4T2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fge/LVhgj6Kf5ha8ffu2AWYjExPTgX///jmQwueMj4frZ5CQ2PC1vV0AWT8jzX1AcwsGXRChpxr2TZvgqQaUijg2bYLHISguSI4Dki0gNYjQUw3DixcBsFQDSkUcs2ZNgPPp4gNSU9FoEBEs7IZfEKEXz8LCwojCjxrJlKAFlGY09OJZUFDwAHJGAwDktqhYEhtYvAAAAABJRU5ErkJggg==' />"}],"value":"[21 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x9cd8b18 \"comportexviz.server.notebook$eval28199$fn$reify__28203@9cd8b18\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>22</span>","value":"22"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAm0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fge/LVhgj6Kf5ha8ffu2AWYjExPTgX///jmQwueMj4frZ5CQ2PC1vV0AWT8jzX1AcwuGXBBxbNoEj0NQXFA9DjAsoHYQccyaNQGWisjyAXrGYt+0CZ6xQBmN4iAi2QJSgwg9YzG8eBEADxIJiQ2jQXRgNIgIFtekBhEAN+OoWHLfulEAAAAASUVORK5CYII=' />"}],"value":"[22 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x34e9b65b \"comportexviz.server.notebook$eval28199$fn$reify__28203@34e9b65b\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>23</span>","value":"23"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAqUlEQVQ4T2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fgfZN22yh+kH8RlpbsHbt28bYDYyMTEd+PfvnwMpfM74eLh+BgmJDQwvXgTA9IP4VPfBtwUL4EEEDnJqBxGGBdQOoq/t7QLIQUx1H3Bs2gSPQ5BFtLeA2kHEMWvWBHgqoosPqJ2KMOJg0AURobKH4lREsgWkBhGhsofiVESyD0hNRSRbQOsgAgAbI6WZd7zl8QAAAABJRU5ErkJggg==' />"}],"value":"[23 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7c9773df \"comportexviz.server.notebook$eval28199$fn$reify__28203@7c9773df\"]]"}],"value":"([0 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1e062204 \"comportexviz.server.notebook$eval28199$fn$reify__28203@1e062204\"]] [1 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2a11e6f2 \"comportexviz.server.notebook$eval28199$fn$reify__28203@2a11e6f2\"]] [2 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x6701b3ab \"comportexviz.server.notebook$eval28199$fn$reify__28203@6701b3ab\"]] [3 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x385bbdbd \"comportexviz.server.notebook$eval28199$fn$reify__28203@385bbdbd\"]] [4 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x17e2aeb0 \"comportexviz.server.notebook$eval28199$fn$reify__28203@17e2aeb0\"]] [5 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x3261ecc8 \"comportexviz.server.notebook$eval28199$fn$reify__28203@3261ecc8\"]] [6 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x58179dc4 \"comportexviz.server.notebook$eval28199$fn$reify__28203@58179dc4\"]] [7 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x3d44652e \"comportexviz.server.notebook$eval28199$fn$reify__28203@3d44652e\"]] [8 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x3f51db18 \"comportexviz.server.notebook$eval28199$fn$reify__28203@3f51db18\"]] [9 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x26b6796 \"comportexviz.server.notebook$eval28199$fn$reify__28203@26b6796\"]] [10 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x66f633a2 \"comportexviz.server.notebook$eval28199$fn$reify__28203@66f633a2\"]] [11 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x78855747 \"comportexviz.server.notebook$eval28199$fn$reify__28203@78855747\"]] [12 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x493dda54 \"comportexviz.server.notebook$eval28199$fn$reify__28203@493dda54\"]] [13 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x10b34332 \"comportexviz.server.notebook$eval28199$fn$reify__28203@10b34332\"]] [14 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x478f09a5 \"comportexviz.server.notebook$eval28199$fn$reify__28203@478f09a5\"]] [15 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1562517c \"comportexviz.server.notebook$eval28199$fn$reify__28203@1562517c\"]] [16 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x32aad6fe \"comportexviz.server.notebook$eval28199$fn$reify__28203@32aad6fe\"]] [17 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x388ab94a \"comportexviz.server.notebook$eval28199$fn$reify__28203@388ab94a\"]] [18 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7775ddf \"comportexviz.server.notebook$eval28199$fn$reify__28203@7775ddf\"]] [19 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1905fc \"comportexviz.server.notebook$eval28199$fn$reify__28203@1905fc\"]] [20 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x64e02e27 \"comportexviz.server.notebook$eval28199$fn$reify__28203@64e02e27\"]] [21 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x9cd8b18 \"comportexviz.server.notebook$eval28199$fn$reify__28203@9cd8b18\"]] [22 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x34e9b65b \"comportexviz.server.notebook$eval28199$fn$reify__28203@34e9b65b\"]] [23 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7c9773df \"comportexviz.server.notebook$eval28199$fn$reify__28203@7c9773df\"]])"}
;; <=

;; @@
(for [i (range 24)]
  [i (viz (hour-of-day-encoder 10 1) i)])
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>0</span>","value":"0"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAiElEQVRIS2P8z8DwnwEKfvr5HWTftMkeH//bggVw+f///zcyMjLWw9QTxae7hT/8/A7AXMggIbGB4cWLAHz8r+3tAjB5JiamA//+/XMghc9Idx/S3cJhF6TCwsIN8DTBwMBA8zjEsJDWQSooKIjIBQPiQ1qn0tEgpXrhPRqko0F6kNQaf+SVpQATfyBqXM9v6gAAAABJRU5ErkJggg==' />"}],"value":"[0 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x78ea60a \"comportexviz.server.notebook$eval28199$fn$reify__28203@78ea60a\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>1</span>","value":"1"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAiElEQVRIS2P8z8DwnwEKfvr5HWTftMkeH//bggVw+f///zcyMjLWw9QTxae7hT/8/A7AXMggIbGB4cWLAHz8r+3tAjB5JiamA//+/XMghc9Idx/S3cJhF6TCwsIN8DTBwMBA8zjEsJDWQSooKIjIBQPiQ1qn0tEgpXrhPRqko0F6kNQaf+SVpQATfyBqXM9v6gAAAABJRU5ErkJggg==' />"}],"value":"[1 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7ef13ccb \"comportexviz.server.notebook$eval28199$fn$reify__28203@7ef13ccb\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>2</span>","value":"2"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAArUlEQVRIS2N89+7dfwYo+P//f6OQsHA9jP/Tz+8g+6ZN9vj43xYsgMuD9DMyMsL1Y+XT3cK3b982wHzAxMR0gDM+Hs5nkJDYwPDiRQBMHhv/a3u7ALL+f//+OeDjM/5nYIAHKU2CED2I6W7hDz+/A/iCDD1ISQ1C9CCmeZAKCwsj0gQDAwP9LaR1kAoKCiKibEB8SOtUihGHo0GKXniTWliPBinV67/RIB1+QQoACyMjao98JPUAAAAASUVORK5CYII=' />"}],"value":"[2 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x18f3a79c \"comportexviz.server.notebook$eval28199$fn$reify__28203@18f3a79c\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>3</span>","value":"3"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAp0lEQVRIS2N89+7dfwYo+P//f6OQsHA9jP/Tz+8g+6ZN9vj43xYsgMuD9DMyMsL1Y+XT3cK3b982wHzAxMR0gDM+Hs5nkJDYwPDiRQBMHhv/a3u7ALL+f//+OeDjM9Ldh3S3kO5B+p+BAZ5KaZEqhYWFEWmCgYGBke4W/vDzO4AvFaKnUlJTpaCgIML8AfHhaJCil6Wklp2jqZTq1dFokI4G6QZKy1IACYsmampY3rsAAAAASUVORK5CYII=' />"}],"value":"[3 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x5893820 \"comportexviz.server.notebook$eval28199$fn$reify__28203@5893820\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>4</span>","value":"4"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAq0lEQVRIS2N89+7dfwYo+P//f6OQsHA9jP/Tz+8g+6ZN9vj43xYsgMuD9DMyMsL1Y+XT3cK3b982wHzAxMR0gDM+Hs5nkJDYwPDiRQBMHhv/a3u7ALL+f//+OeDjM9Ldh3S3cNgFqbCwMCJNMDAw0DwOMSykdZAKCgoegKdykA//MzDAMz4tMjqGD+lu4Q8/P4SXaZDRR4OU6mXnaJCOBulBUpsUGCUNvTM+AA63KWp1iWcvAAAAAElFTkSuQmCC' />"}],"value":"[4 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x77c5606d \"comportexviz.server.notebook$eval28199$fn$reify__28203@77c5606d\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>5</span>","value":"5"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAtElEQVRIS2N89+7dfwYo+P//f6OQsHA9jP/Tz+8g+6ZN9vj43xYsgMuD9DMyMsL1Y+XT3cK3b982wHzAxMR0gDM+Hs5nkJDYwPDiRQBMHhv/a3u7ALL+f//+OeDjM9Ldh3S3cNgFqbCwMCJNMDAw0DwOMSykdZAKCgoegKfyAfEhrVPpwAfpfwYGeFlKi7ITw4d0t/CHnx8iFdGg7MRIpXT3Id0tHA1S9CYGqU0KjGxB7yAFABqnLGr8Tc6wAAAAAElFTkSuQmCC' />"}],"value":"[5 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x251d3a \"comportexviz.server.notebook$eval28199$fn$reify__28203@251d3a\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>6</span>","value":"6"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAArklEQVRIS2N89+7dfwYo+P//f6OQsHA9jP/Tz+8g+6ZN9vj43xYsgMuD9DMyMsL1Y+XT3cK3b982wHzAxMR0gDM+Hs5nkJDYwPDiRQBMHhv/a3u7ALL+f//+OeDjM9Ldh3S3cNgFqbCwMCJNMDAw0DwOMSykdZAKCgoegKfyAfEhrVPpaJBSvfAe+CD9z8AAr55oUR1h+JDuFv7w80NkTBpURxgZn+4+pLuF9A5SAC1bL2qFkg44AAAAAElFTkSuQmCC' />"}],"value":"[6 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x56f85e5 \"comportexviz.server.notebook$eval28199$fn$reify__28203@56f85e5\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>7</span>","value":"7"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAqUlEQVRIS2N89+7dfwYo+P//f6OQsHA9jP/Tz+8g+6ZN9vj43xYsgMuD9DMyMsL1Y+XT3cK3b982wHzAxMR0gDM+Hs5nkJDYwPDiRQBMHhv/a3u7ALL+f//+OeDjM9Ldh3S3cNgFqbCwMCJNMDAw0DwOMSykdZAKCgoegKfyAfEhrVPpaJBSvfAeDdJhGKT/GRjgNT4taniMREN3C3/4+SHKOhrU8OhlKQBG0zJq4PJFHQAAAABJRU5ErkJggg==' />"}],"value":"[7 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x49ebf550 \"comportexviz.server.notebook$eval28199$fn$reify__28203@49ebf550\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>8</span>","value":"8"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAmElEQVRIS2N89+7dfwYo+P//f6OQsHA9jP/Tz+8g+6ZN9vj43xYsgMuD9DMyMsL1Y+XT3cK3b982wHzAxMR0gDM+Hs5nkJDYwPDiRQBMHhv/a3u7ALL+f//+OeDjM9Ldh3S3cNgFqbCwMCJNMDAw0DwOMSykdZAKCgoegKfyAfEhrVPpaJBSvfAeDdLRIMVoARCq8UdeWQoAZw81avuk9k0AAAAASUVORK5CYII=' />"}],"value":"[8 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x15712208 \"comportexviz.server.notebook$eval28199$fn$reify__28203@15712208\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>9</span>","value":"9"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAqElEQVRIS2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDw2/rcFC+Dy2MxjpLuFb9++bYC5mImJ6cC/f/8ckPmc8fFweQYJiQ0ML14EwOSx8b+2twvgMw/Dh1QPQrQoor+F6EFK7SBEjyKa+1BYWBiRBhgYGOhvIa2DVFBQ8AA8VQ+ID9EzPrWzBUYcjgYppYX1aJA2jqZS9EREqIYfeWUpAJcPNWqMFXs9AAAAAElFTkSuQmCC' />"}],"value":"[9 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x3d153151 \"comportexviz.server.notebook$eval28199$fn$reify__28203@3d153151\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>10</span>","value":"10"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAn0lEQVRIS2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDw2/rcFC+Dy2MxjpLuFb9++bYC5mImJ6cC/f/8ckPmc8fFweQYJiQ0ML14EwOSx8b+2twvgM4/+Phx5QUrtVCksLIxIAwwMDBhxSHML0VMptVOloKDgAXiqHhAfoqfS0SBFL0sJlZ2jqbSR5olmNJVSWsOPvFQKAMcPNWp8iRNJAAAAAElFTkSuQmCC' />"}],"value":"[10 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x442cdbbb \"comportexviz.server.notebook$eval28199$fn$reify__28203@442cdbbb\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>11</span>","value":"11"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAApUlEQVRIS2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDw2/rcFC+Dy2MxjpLuFb9++bYC5mImJ6cC/f/8ckPmc8fFweQYJiQ0ML14EwOSx8b+2twvgM4/+PhwNUkpTqbCwMCINMDAw0DwOMSykdSoVFBQ8AE/V2HxIaRCiFxwEg5TmFqIHKbUz+miQHhgNUvT6kFD9RzDjjwbpkAtSAPcPNWoXX/WGAAAAAElFTkSuQmCC' />"}],"value":"[11 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x52fbe71b \"comportexviz.server.notebook$eval28199$fn$reify__28203@52fbe71b\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>12</span>","value":"12"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAArklEQVRIS2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDw2/rcFC+Dy2MxjpLuFb9++bYC5mImJ6cC/f/8ckPmc8fFweQYJiQ0ML14EwOSx8b+2twvgM4/+PhwNUkpTqbCwMCINMDAw0DwOMSykdSoVFBQ8AE/VA+JDWqfSwReklKZK9LKYYCqluYXoqZTaZSfBVEpzH6KnUppbOBqklDYpCGZ8WqdSACceNWrkmTopAAAAAElFTkSuQmCC' />"}],"value":"[12 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x4e86fe2e \"comportexviz.server.notebook$eval28199$fn$reify__28203@4e86fe2e\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>13</span>","value":"13"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAq0lEQVRIS2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDw2/rcFC+Dy2MxjpLuFb9++bYC5mImJ6cC/f/8ckPmc8fFweQYJiQ0ML14EwOSx8b+2twvgM4/+PhwNUkpTqbCwMCINMDAw0DwOMSykdSoVFBQ8AE/VA+JDWqfS0SCleuE9+IKU0oyOXr0RzPg0txA941O7OiKY8WnuQ/SMT3ML6R2kAFceNWq5FBDlAAAAAElFTkSuQmCC' />"}],"value":"[13 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x5a8dc116 \"comportexviz.server.notebook$eval28199$fn$reify__28203@5a8dc116\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>14</span>","value":"14"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAo0lEQVRIS2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDw2/rcFC+Dy2MxjpLuFb9++bYC5mImJ6cC/f/8ckPmc8fFweQYJiQ0ML14EwOSx8b+2twvgM4/+PhwNUkpTqbCwMCINMDAw0DwOMSykdSoVFBQ8AE/VA+JDWqfS0SCleuE9GqQjIEgpLTvRWwwEy1KaW4hellK7hkcvSwGHHjVqWx5rBAAAAABJRU5ErkJggg==' />"}],"value":"[14 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x785e38ff \"comportexviz.server.notebook$eval28199$fn$reify__28203@785e38ff\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>15</span>","value":"15"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAlUlEQVRIS2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDw2/rcFC+Dy2MxjpLuFb9++bYC5mImJ6cC/f/8ckPmc8fFweQYJiQ0ML14EwOSx8b+2twvgM4/+PhwNUkpTqbCwMCINMDAw0DwOMSykdSoVFBQ8AE/VA+JDWqfS0SCleuE9GqSjQUpyjT/yylIAtx41aiimJp8AAAAASUVORK5CYII=' />"}],"value":"[15 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7590cbd0 \"comportexviz.server.notebook$eval28199$fn$reify__28203@7590cbd0\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>16</span>","value":"16"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAqUlEQVRIS2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef30H2TZvsYeqx8b8tWACXB5tPdwvfvn3bAHMhExPTgX///jng43PGx8PVM0hIbGB48SIAph4b/2t7uwCyeQR9SHEQokcRoSCluoWEgpTSIESPIpoHqbCwMCLOGRgY6G8hrYNUUFDwADwVD4gPaZ1KMeJwNEgpzfijQYpRH44GKXqNj17Dj7yyFADnHjVqykjYLAAAAABJRU5ErkJggg==' />"}],"value":"[16 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0xc1e3cc2 \"comportexviz.server.notebook$eval28199$fn$reify__28203@c1e3cc2\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>17</span>","value":"17"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAApElEQVRIS2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef30H2TZvsYeqx8b8tWACXB5tPdwvfvn3bAHMhExPTgX///jng43PGx8PVM0hIbGB48SIAph4b/2t7uwCyefT34WiQUpoqhYWFEXHOwMBAMA6pbiGhVEppqhQUFDwAT8UD4kNCqXQ0SNHLUvSyczSVYlRPVE80o6l0NEjRa3z0Gh69LAUAFy01amnKQf8AAAAASUVORK5CYII=' />"}],"value":"[17 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x38afd926 \"comportexviz.server.notebook$eval28199$fn$reify__28203@38afd926\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>18</span>","value":"18"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAsElEQVRIS2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef30H2TZvsYeqx8b8tWACXB5tPdwvfvn3bAHMhExPTgX///jng43PGx8PVM0hIbGB48SIAph4b/2t7uwCyefT34WiQoqdaUlMpx6ZN8DQBikuaxyGGhbROpRyzZk2Ap2JifEhqEKJndGFhYUQ2GhALCQUppRldUFDwwGiQopSlo0GKXh8SzBajqZTqJQ29gxQA8Q81an1+lzkAAAAASUVORK5CYII=' />"}],"value":"[18 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x31bf970c \"comportexviz.server.notebook$eval28199$fn$reify__28203@31bf970c\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>19</span>","value":"19"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAtElEQVRIS2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef30H2TZvsYeqx8b8tWACXB5tPdwvfvn3bAHMhExPTgX///jng43PGx8PVM0hIbGB48SIAph4b/2t7uwCyefT34WiQoqdaUlMpx6ZN8DQBikuaxyGGhbROpRyzZk2Ap+IB8SGtU+ngD1JSUyV62SksLIwomYiJQ6pbSCiVUlp2CgoKHiAplVLdh4RSKdUtHA3SIR+kAMsANWo4gK/fAAAAAElFTkSuQmCC' />"}],"value":"[19 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x428214a0 \"comportexviz.server.notebook$eval28199$fn$reify__28203@428214a0\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>20</span>","value":"20"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAt0lEQVRIS2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef30H2TZvsYeqx8b8tWACXB5tPdwvfvn3bAHMhExPTgX///jng43PGx8PVM0hIbGB48SIAph4b/2t7uwCyefT34WiQoqdaUlMpx6ZN8DQBikuaxyGGhbROpRyzZk2Ap+IB8SGtU+lokFK98B78QUpqRkevjoSFhRGFPTHZguoWEsr4lFZHgoKCB0jK+FT3IaGMT3UL6R2kAKTxNWrwEPKmAAAAAElFTkSuQmCC' />"}],"value":"[20 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x3f07d81e \"comportexviz.server.notebook$eval28199$fn$reify__28203@3f07d81e\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>21</span>","value":"21"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAr0lEQVRIS2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef30H2TZvsYeqx8b8tWACXB5tPdwvfvn3bAHMhExPTgX///jng43PGx8PVM0hIbGB48SIAph4b/2t7uwCyefT34WiQoqdaUlMpx6ZN8DQBikuaxyGGhbROpRyzZk2Ap+IB8SGtU+lokFK98B4N0hEYpKSWneg1vLCwMKL+JKakobqFhMpSSmt4QUHBA8hlKQB+4jVqCtpx+AAAAABJRU5ErkJggg==' />"}],"value":"[21 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x79a74031 \"comportexviz.server.notebook$eval28199$fn$reify__28203@79a74031\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>22</span>","value":"22"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAAmklEQVRIS2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef30H2TZvsYeqx8b8tWACXB5tPdwvfvn3bAHMhExPTgX///jng43PGx8PVM0hIbGB48SIAph4b/2t7uwCyefT34WiQoqdaUlMpx6ZN8DQBikuaxyGGhbROpRyzZk2Ap+IB8SGtU+lokFK98B4N0tEgbRwtS9FbAOhlKQBY0zVqfYPB3AAAAABJRU5ErkJggg==' />"}],"value":"[22 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x346dbffe \"comportexviz.server.notebook$eval28199$fn$reify__28203@346dbffe\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>23</span>","value":"23"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABwAAAAVCAYAAABVAo5cAAAApklEQVRIS2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fgfZN22yh+nHxmeku4Vv375tgLmIiYnpwL9//xxI4XPGx8P1M0hIbGB48SIAph8bn2QfkhqE3xYsgAcxOMpIDVKKLSQ1SEkNwq/t7QLIUURzH3Js2gRPEyCL6W8hrYOUY9asCfBUOyA+pHUqxYjD0SBFL8xJzfijQUqwPhwNUvQaf+SVpQAhWTKrnD6GLwAAAABJRU5ErkJggg==' />"}],"value":"[23 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x66989ac7 \"comportexviz.server.notebook$eval28199$fn$reify__28203@66989ac7\"]]"}],"value":"([0 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x78ea60a \"comportexviz.server.notebook$eval28199$fn$reify__28203@78ea60a\"]] [1 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7ef13ccb \"comportexviz.server.notebook$eval28199$fn$reify__28203@7ef13ccb\"]] [2 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x18f3a79c \"comportexviz.server.notebook$eval28199$fn$reify__28203@18f3a79c\"]] [3 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x5893820 \"comportexviz.server.notebook$eval28199$fn$reify__28203@5893820\"]] [4 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x77c5606d \"comportexviz.server.notebook$eval28199$fn$reify__28203@77c5606d\"]] [5 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x251d3a \"comportexviz.server.notebook$eval28199$fn$reify__28203@251d3a\"]] [6 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x56f85e5 \"comportexviz.server.notebook$eval28199$fn$reify__28203@56f85e5\"]] [7 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x49ebf550 \"comportexviz.server.notebook$eval28199$fn$reify__28203@49ebf550\"]] [8 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x15712208 \"comportexviz.server.notebook$eval28199$fn$reify__28203@15712208\"]] [9 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x3d153151 \"comportexviz.server.notebook$eval28199$fn$reify__28203@3d153151\"]] [10 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x442cdbbb \"comportexviz.server.notebook$eval28199$fn$reify__28203@442cdbbb\"]] [11 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x52fbe71b \"comportexviz.server.notebook$eval28199$fn$reify__28203@52fbe71b\"]] [12 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x4e86fe2e \"comportexviz.server.notebook$eval28199$fn$reify__28203@4e86fe2e\"]] [13 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x5a8dc116 \"comportexviz.server.notebook$eval28199$fn$reify__28203@5a8dc116\"]] [14 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x785e38ff \"comportexviz.server.notebook$eval28199$fn$reify__28203@785e38ff\"]] [15 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7590cbd0 \"comportexviz.server.notebook$eval28199$fn$reify__28203@7590cbd0\"]] [16 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0xc1e3cc2 \"comportexviz.server.notebook$eval28199$fn$reify__28203@c1e3cc2\"]] [17 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x38afd926 \"comportexviz.server.notebook$eval28199$fn$reify__28203@38afd926\"]] [18 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x31bf970c \"comportexviz.server.notebook$eval28199$fn$reify__28203@31bf970c\"]] [19 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x428214a0 \"comportexviz.server.notebook$eval28199$fn$reify__28203@428214a0\"]] [20 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x3f07d81e \"comportexviz.server.notebook$eval28199$fn$reify__28203@3f07d81e\"]] [21 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x79a74031 \"comportexviz.server.notebook$eval28199$fn$reify__28203@79a74031\"]] [22 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x346dbffe \"comportexviz.server.notebook$eval28199$fn$reify__28203@346dbffe\"]] [23 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x66989ac7 \"comportexviz.server.notebook$eval28199$fn$reify__28203@66989ac7\"]])"}
;; <=

;; **
;;; Play with the `step-distance`:
;; **

;; @@
(for [i (range 7)]
  [i (viz (day-of-week-encoder 0.5 1) i)])
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>0</span>","value":"0"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAwAAAAJCAYAAAAGuM1UAAAAbklEQVQYV2P8z8DwnwEKfvr5Hfy2YIE9jP////9GRkbGehQ+yRp++PkdgJnAICGx4Wt7uwCMz8TEdODfv38OyHxGkm0gWQPJTnr37h08lLCFirCwcAPcjwwMDIwka3j79i3cBGyhIigoiAhFoFUATYhuQGFckK0AAAAASUVORK5CYII=' />"}],"value":"[0 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7da508cc \"comportexviz.server.notebook$eval28199$fn$reify__28203@7da508cc\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>1</span>","value":"1"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAwAAAAJCAYAAAAGuM1UAAAAbklEQVQYV2P8z8DwnwEKfvr5Hfy2YIE9jP////9GRkbGehQ+yRp++PkdgJnAICGx4Wt7uwCMz8TEdODfv38OyHxGkm0gWQPJTnr37h08lLCFirCwcAPcjwwMDIwka3j79i3cBGyhIigoiAhFoFUATYhuQGFckK0AAAAASUVORK5CYII=' />"}],"value":"[1 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x680b007f \"comportexviz.server.notebook$eval28199$fn$reify__28203@680b007f\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>2</span>","value":"2"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAwAAAAJCAYAAAAGuM1UAAAAaElEQVQYV2N89+7dfwYo+P//fyMjI2M9Xj7JGt6+fdsAM5GJienAv3//HPDxGf8zMMCd9NPP7+C3BQvs8TqJZA0//PwOwExkkJDY8LW9XYAiJwkLC8P9CDKIoB8wNBBykqCgIMLJQCsA5cRuQAMLWzQAAAAASUVORK5CYII=' />"}],"value":"[2 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x321ed4d7 \"comportexviz.server.notebook$eval28199$fn$reify__28203@321ed4d7\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>3</span>","value":"3"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAwAAAAJCAYAAAAGuM1UAAAAgElEQVQYV2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGUnW8Pbt2waYCUxMTAc44+PhfAYJiQ1f29sFkOUxbGBkZIQ7CeREDD66kwhqQHfSv3//HJCdgM5n/M/AAA8lbKEiLCyM8BMDAwPpGn74+R2AOQFbqAgKCiLkgQoBy/FxQCdUH5kAAAAASUVORK5CYII=' />"}],"value":"[3 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0xcb9c650 \"comportexviz.server.notebook$eval28199$fn$reify__28203@cb9c650\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>4</span>","value":"4"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAwAAAAJCAYAAAAGuM1UAAAAbElEQVQYV2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGUnW8Pbt2waYCUxMTAc44+PhfAYJiQ1f29sFkOVJt4H+TmJkZISHEijUhIWFEX5iYGDA8ANBDeih9O/fPwfkUBEUFDwA44NoAPkAdEB9mCx3AAAAAElFTkSuQmCC' />"}],"value":"[4 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x598f4c63 \"comportexviz.server.notebook$eval28199$fn$reify__28203@598f4c63\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>5</span>","value":"5"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAwAAAAJCAYAAAAGuM1UAAAAbUlEQVQYV2N89+7dfwYo+P//fyMjI2M9Xj7JGt6+fdsAM5GJienAv3//HPDxGdFtEBIWhjvpp5/fwW8LFtijOJFkDehO4oyPhzuRQUJiw9f2dgFkJxJ0EsemTXA/gTSSroGQkzhmzZoAcxKIBgCJD3RAzikKcQAAAABJRU5ErkJggg==' />"}],"value":"[5 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x6937a7d5 \"comportexviz.server.notebook$eval28199$fn$reify__28203@6937a7d5\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>6</span>","value":"6"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAwAAAAJCAYAAAAGuM1UAAAAaElEQVQYV2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5CRZA1v375tgJnIxMR04N+/fw7IfM74eLg8g4TEBoI2oDuRdA2EnITuRAwb0EOFY9MmuJ9AfiNdA7qT0EOFY9asCbBQA9EAwCZxgSSzhbwAAAAASUVORK5CYII=' />"}],"value":"[6 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2dcb7e48 \"comportexviz.server.notebook$eval28199$fn$reify__28203@2dcb7e48\"]]"}],"value":"([0 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7da508cc \"comportexviz.server.notebook$eval28199$fn$reify__28203@7da508cc\"]] [1 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x680b007f \"comportexviz.server.notebook$eval28199$fn$reify__28203@680b007f\"]] [2 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x321ed4d7 \"comportexviz.server.notebook$eval28199$fn$reify__28203@321ed4d7\"]] [3 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0xcb9c650 \"comportexviz.server.notebook$eval28199$fn$reify__28203@cb9c650\"]] [4 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x598f4c63 \"comportexviz.server.notebook$eval28199$fn$reify__28203@598f4c63\"]] [5 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x6937a7d5 \"comportexviz.server.notebook$eval28199$fn$reify__28203@6937a7d5\"]] [6 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2dcb7e48 \"comportexviz.server.notebook$eval28199$fn$reify__28203@2dcb7e48\"]])"}
;; <=

;; @@
(for [i (range 7)]
  [i (viz (day-of-week-encoder 0.5 4) i)])
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>0</span>","value":"0"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAhUlEQVQ4T2P8z8DwnwEKfvr5HWTftMkemf9twQI4/////42MjIz1MHmi+DS34Ief3wGYixgkJDYwvHgRgMz/2t4uAOMzMTEd+PfvnwMpfEaa+4DmFtA9iChNNcLCwg3wOGRgYMCIA6pbgB5ElKYaQUFBRKqkiw/QU9FoEI2mIoLF9fALIgAT5pdYK2JI7gAAAABJRU5ErkJggg==' />"}],"value":"[0 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1200a837 \"comportexviz.server.notebook$eval28199$fn$reify__28203@1200a837\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>1</span>","value":"1"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAtUlEQVQ4T2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGRkZ4fIg9QT5NLfg7du3DTAXMjExHeCMj4fzGSQkNnxtbxdAlv/3758DKXxGmvuA5hYMuSASFhZGxCEDAwPV4wDDAmoHkaCg4AFYKgPRjP8ZGOAZDZSx2DdtgmcscjIahg9obsEPPz+ElyQkNjC8eBEA9yIZGW00iA4QKosIBhGlxTPBVER1C9BTEaXFM3oQAQAlTKBYipYWWwAAAABJRU5ErkJggg==' />"}],"value":"[1 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x425b4c41 \"comportexviz.server.notebook$eval28199$fn$reify__28203@425b4c41\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>2</span>","value":"2"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAlElEQVQ4T2N89+7dfwYo+P//f6OQsHA9jP/Tz+8g+6ZN9sj8bwsWwPkg9YyMjHD1WPk0t+Dt27cNMBcyMTEd4IyPh/MZJCQ2MLx4EQCTB/G/trcLIKv/9++fAz4+I819QHMLBjyISE01wsLCiDhkYGAgGAcUW0AoiEhNNYKCggfgqY4uPiCUikaDaDQVYRTXwy+jAQD/D6lYQZFkBwAAAABJRU5ErkJggg==' />"}],"value":"[2 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7c6a14a6 \"comportexviz.server.notebook$eval28199$fn$reify__28203@7c6a14a6\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>3</span>","value":"3"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAqUlEQVQ4T2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd/DbggX2+NSj62ekuQVv375tgLmIiYnpwL9//xyQ+Zzx8XB5BgmJDV/b2wXwqUfXT3sfjAYReqoRFhZGxBkDAwPV4wDDAmqnIkFBwQOwVAaiae8DaqeigQ8i9LKHfdMmeNlDTFlEMBVR3QL0VIRe9jC8eBEATxVElEUEUxHVfYCeiqhuAa2DCABXpalYOpVk3wAAAABJRU5ErkJggg==' />"}],"value":"[3 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0xc7a4571 \"comportexviz.server.notebook$eval28199$fn$reify__28203@c7a4571\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>4</span>","value":"4"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAnklEQVQ4T2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef38FvCxbY49VPcwvevn3bAHMBExPTgX///jng43PGx8PVM0hIbPja3i6ATz0jIR+gBwn7pk3wIKFKEFFsAaEgQg8ShhcvAmBBMiBBJCwsjIgjBgYGqscBhgXUDiJBQcED8CAkxweEMhbFQUSyBaQGEaGMNRpEBMue0SAiOYgA1TypWAfpFJ8AAAAASUVORK5CYII=' />"}],"value":"[4 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1b2c15d9 \"comportexviz.server.notebook$eval28199$fn$reify__28203@1b2c15d9\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>5</span>","value":"5"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAq0lEQVQ4T2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef30H2TZvsYepB/G8LFsD5YPNobsHbt28bYC5gYmI68O/fPwd8fM74eLh6BgmJDQwvXgTA1IP4X9vbBZD1U+wDjCBBD2JKg4igBZQGEXqQoAcx1YNIWFgYEUcMDAy0t4DaQSQoKHgAnqro4gNqpyKMOBgNIvSMNviCiNTimeSMRrEFhFIRqcUzekYDAGtpqVhhryS5AAAAAElFTkSuQmCC' />"}],"value":"[5 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x38d6bb9 \"comportexviz.server.notebook$eval28199$fn$reify__28203@38d6bb9\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>6</span>","value":"6"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAASCAYAAABB7B6eAAAAsUlEQVQ4T2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fge/LVhgj6Kf5ha8ffu2AWYjExPTgX///jmQwueMj4frZ5CQ2PC1vV0AWT8jzX1AcwsGXRChpxr2TZvgqQaUijg2bYLHISguSI4Dki0gNYjQUw3DixcBsFQDSkUcs2ZNgPPp4gNSU9HwCyL0skdYWBiRs6kRBwQtoDQVoZc9goKCB6iaigj6gNJURNACWgcRAF9LqVjn1WB2AAAAAElFTkSuQmCC' />"}],"value":"[6 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2f6e904a \"comportexviz.server.notebook$eval28199$fn$reify__28203@2f6e904a\"]]"}],"value":"([0 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1200a837 \"comportexviz.server.notebook$eval28199$fn$reify__28203@1200a837\"]] [1 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x425b4c41 \"comportexviz.server.notebook$eval28199$fn$reify__28203@425b4c41\"]] [2 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7c6a14a6 \"comportexviz.server.notebook$eval28199$fn$reify__28203@7c6a14a6\"]] [3 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0xc7a4571 \"comportexviz.server.notebook$eval28199$fn$reify__28203@c7a4571\"]] [4 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1b2c15d9 \"comportexviz.server.notebook$eval28199$fn$reify__28203@1b2c15d9\"]] [5 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x38d6bb9 \"comportexviz.server.notebook$eval28199$fn$reify__28203@38d6bb9\"]] [6 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2f6e904a \"comportexviz.server.notebook$eval28199$fn$reify__28203@2f6e904a\"]])"}
;; <=

;; @@
(for [i (range 7)]
  [i (viz (day-of-week-encoder 0.5 10) i)])
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>0</span>","value":"0"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACQAAAAbCAYAAAAULC3gAAAAtElEQVRIS2P8z8DwnwEKfvr5HWTftMkeH//bggVw+f///zcyMjLWw9RThT/oHPTDz+8AzIcMEhIbGF68CMDH/9reLgCTZ2JiOvDv3z8HavIZB10IDToHDbkoo3muQs+lhKKM7g4iFGW0zlXouZRgLqN7CI1GGYGCdDTKCFbGo7mMUHNlNJeN5jJCaYSQ/GguIxRCo7lssOcyYWHhBngvh4GBYcArVwwHDXQuExQURPQLgUEFAJJJzyHVY0hZAAAAAElFTkSuQmCC' />"}],"value":"[0 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7660fdaa \"comportexviz.server.notebook$eval28199$fn$reify__28203@7660fdaa\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>1</span>","value":"1"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACQAAAAbCAYAAAAULC3gAAAAyUlEQVRIS2N89+7dfwYo+P//f6OQsHA9jP/Tz+8g+6ZN9vj43xYsgMuD9DMyMsL1k8UfdA56+/ZtAywEmJiYDnDGx8P5DBISGxhevAiAyWPjf21vF0DW/+/fPwdK+IykhhDVowg9igedg0iNMmpHEXoUj0YZwWJhNMoIFZyjuYxAwTmay0ZzGcEQGM1lo80PajdZ0dPUaF02mssozWX/GRjg/bKB6IcJCwsjejkMDAyMg85BP/z8DuDrd6H3y6jdyBcUFETYD3QIAJks5yHDsqeMAAAAAElFTkSuQmCC' />"}],"value":"[1 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x668d3024 \"comportexviz.server.notebook$eval28199$fn$reify__28203@668d3024\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>2</span>","value":"2"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACQAAAAbCAYAAAAULC3gAAAAyklEQVRIS2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDw2/rcFC+Dy2MxDNx+DP+gc9Pbt2waYj5mYmA78+/fPAZnPGR8Pl2eQkNjA8OJFAEweG/9re7sAPvPQzUfnM1IaQhRHEVoSGXwOojTKKI2i0SgjmM3R09BolKGHyGguQytoRwtGknPVaC4jlKtGc9loLqO0yTqay0ZzGYWNeoItRnr3w4SFhRG9GgYGBoxG/oA7CL3FSO9+mKCg4AF4Pw/IAADphuohzlwU4gAAAABJRU5ErkJggg==' />"}],"value":"[2 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x35c23b40 \"comportexviz.server.notebook$eval28199$fn$reify__28203@35c23b40\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>3</span>","value":"3"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACQAAAAbCAYAAAAULC3gAAAAuklEQVRIS2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef30H2TZvsYepB/G8LFsD5xJiHYd+gc9Dbt28bYD5kYmI68O/fPwd8fM74eLh6BgmJDQwvXgTA1IP4X9vbBUgxD90+xkEXQoPOQaNRRiAXD740NBplo1E2WjDSunIdzWWjuWw0l43mMrRcQKhRT/dGPt0dRKhgJNSop3sjn9YhJCwsjOjFMDAwEGwx0t1BAx1lgoKCB+D9OiADAN9J6iEwgTtkAAAAAElFTkSuQmCC' />"}],"value":"[3 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x16728174 \"comportexviz.server.notebook$eval28199$fn$reify__28203@16728174\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>4</span>","value":"4"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACQAAAAbCAYAAAAULC3gAAAAu0lEQVRIS2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fgfZN22yh+kH8b8tWADnE2X+oHPQ27dvG2A+YmJiOvDv3z8HUvic8fFw/QwSEhsYXrwIgOkH8b+2twuQYh7joAuhQeeg0SgjlItHo4xALh7NZQRrgtFcNprLRusytDRA9fbQaC4bzWX0zmWEEjHdG/k0dxCpuYxQo57ujXxqh5CwsDCi18LAwEBye4jmDhroKBMUFDwA78cBGQAfZ+ohQ4nKqAAAAABJRU5ErkJggg==' />"}],"value":"[4 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x678eb3bd \"comportexviz.server.notebook$eval28199$fn$reify__28203@678eb3bd\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>5</span>","value":"5"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACQAAAAbCAYAAAAULC3gAAAAwklEQVRIS2N89+7dfwYo+P//fyMjI2M9JXwhYWG4/p9+fgfZN22yh5kH4n9bsADOx2rfoHPQ27dvG2A+YGJiOvDv3z8HSvic8fFw8xgkJDYwvHgRADMPxP/a3i6Az3zGQRdCg85Bo1E2mstGcxmBkpzikno0l43mstFcNprLSGwxEip36N5ipLqDKC0YCbUQ6d5ipHoIUdpipLqDRlyUEeqHCQsLI3opDAwMFPc6CEUZyQ6idZQR6ocJCgoegPfbgAwA2TrqIYqMaSIAAAAASUVORK5CYII=' />"}],"value":"[5 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x688db378 \"comportexviz.server.notebook$eval28199$fn$reify__28203@688db378\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>6</span>","value":"6"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACQAAAAbCAYAAAAULC3gAAAAxklEQVRIS2N89+7dfwYo+P//fyMjI2M9NflCwsJw8376+R1k37TJHmY+iP9twQI4H2z/oHPQ27dvG2AuZmJiOvDv3z8HavI54+Ph5jNISGxgePEiAGY+iP+1vV0A2b7BF0KjUTYaZWjFBsnZfjSXjRaMaAXraMFIqDIezWVUD6HRuozedRmhREz3FiPFDqJ2XUaoIKR7i5HiEKJ2LqPYQSM+ytD7YcLCwoheCQMDA9V7HYSijKCD6B1l6P0wQUHBA/B+GpABABlY6iGOcTKeAAAAAElFTkSuQmCC' />"}],"value":"[6 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x4cb18d12 \"comportexviz.server.notebook$eval28199$fn$reify__28203@4cb18d12\"]]"}],"value":"([0 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7660fdaa \"comportexviz.server.notebook$eval28199$fn$reify__28203@7660fdaa\"]] [1 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x668d3024 \"comportexviz.server.notebook$eval28199$fn$reify__28203@668d3024\"]] [2 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x35c23b40 \"comportexviz.server.notebook$eval28199$fn$reify__28203@35c23b40\"]] [3 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x16728174 \"comportexviz.server.notebook$eval28199$fn$reify__28203@16728174\"]] [4 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x678eb3bd \"comportexviz.server.notebook$eval28199$fn$reify__28203@678eb3bd\"]] [5 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x688db378 \"comportexviz.server.notebook$eval28199$fn$reify__28203@688db378\"]] [6 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x4cb18d12 \"comportexviz.server.notebook$eval28199$fn$reify__28203@4cb18d12\"]])"}
;; <=

;; **
;;; Play with both:
;; **

;; @@
(for [i (range 7)]
  [i (viz (day-of-week-encoder 3 10) i)])
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>0</span>","value":"0"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAADAAAAAhCAYAAACfiCi5AAAA00lEQVRYR2P8z8DwnwEKfvr5HWTftMmelvxvCxbAzf///38jIyNjPcw+svhD3gM//PwOwEKAQUJiA8OLFwG05H9tbxeAmc/ExHTg379/DpTwGYd8DAx5D4wmIRJLrdFSCL3YHU1Co0mIwpp4NAkRSEJUL3XQMzGtKzKae4DWSYjabR/0thPN20I0j4HRJESg+T2ahAj12EaT0GgSorApMZqERpPQaBKi7bjQaEWGVpEJCws3wAfeGBgYhlxjDsMDQ605LSgoiBgKHRYxMNQqsmGXhACl8v7gFLY+4AAAAABJRU5ErkJggg==' />"}],"value":"[0 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7db48dbd \"comportexviz.server.notebook$eval28199$fn$reify__28203@7db48dbd\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>1</span>","value":"1"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAADAAAAAhCAYAAACfiCi5AAAA8ElEQVRYR+2YMQ7CMAxFnSyIjZClt4ETRNygRylMvUaPgHqCcpsuyQaCJUFIJIVMRGkqGbnbV6VYrl9t5zNjjIP345w7baVsvH4odVn1/a6kvnVdOP8VnzEW4v+k0SegtT76L8w5H9Z1HTRU1RnG8eDfl9DXtt18xrfW7lM0W7oC2cjEiKFPYGmEcpGJESOEUtvs7P8AIZQ46GavAHWhxEFHXYgGWSIy8S5FCBFChFDmDYwQIoQIoW/XAv067QCCL4TCB4q7EPoE7koNJX2f4rsQtgpIKSfjDQAY+gSwISSEmJD/iwoQQom7Ue6NLEboCSPIFu+GsNSgAAAAAElFTkSuQmCC' />"}],"value":"[1 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x3c855aec \"comportexviz.server.notebook$eval28199$fn$reify__28203@3c855aec\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>2</span>","value":"2"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAADAAAAAhCAYAAACfiCi5AAAA80lEQVRYR2N89+7dfwYo+P//fyMjI2M9Ml9IWBjO/+nnd5B90yZ7mDw1+N8WLICbh81+dPdg8Ie8B96+fdsAC1EmJqYD//79c0Dmc8bHw+UZJCQ2MLx4EQCTpwb/a3u7AD770d2Dzmcc8jEw5D0wmoQoLJVGS6HRJDSahCisiUeT0GgSGk1Co81pivoHFNfEw64UonUPDL1HR3EMoDenh5wH0JMQrXtg6D06qvfIhlwMjCYhCjv5o0loNAmNJiHSWqPCwsKIgTYGBgaMga3BXoxieGCoVWSCgoIH4EObAxEDlLZ9BjwJUd0D9E5ClFZc6EkIABrFH+9kERb1AAAAAElFTkSuQmCC' />"}],"value":"[2 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x4e8300aa \"comportexviz.server.notebook$eval28199$fn$reify__28203@4e8300aa\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>3</span>","value":"3"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAADAAAAAhCAYAAACfiCi5AAAA90lEQVRYR2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef30H2TZvsYeqpwf+2YAHcPGLcwzjkPfD27dsGWAgyMTEd+PfvnwM+Pmd8PFw9g4TEBoYXLwJg6qnB/9reLkCKe4Z+DIwmodEkhFqKjZZCw64UonXFhV4RUj0JDXoPEKrIaF1xoVeEVK/IBn0MEKrIBr0HRpPQYG9OjyYhEttKo6UQepdyNAkN9yREctsHfdBhoCsyij0w0BUZqaUOenOfYKee1pmY4hgYTUIkljKUNp+HfBISFhZGDKwxMDAMuTyA4YGhVgoJCgoegA9lDosYGGql0LBLQgA1XR/vPehw7AAAAABJRU5ErkJggg==' />"}],"value":"[3 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x328aab57 \"comportexviz.server.notebook$eval28199$fn$reify__28203@328aab57\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>4</span>","value":"4"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAADAAAAAhCAYAAACfiCi5AAAA+klEQVRYR2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fgfZN22yh+mnBv/bggVw87C6b8h74O3btw2wEGNiYjrw798/B1L4nPHxcP0MEhIbGF68CIDppwb/a3u7AD73MJIaA7ROMuhJkOpJaNB5gNQkROskg54ER5MQerE6moRILKVGk9BoEiIxyYyWQoQac6OlEIlJasiVQgTbPujN/cHWmCPZA4OtLUQoyaA39wddc5rkGBhNQiSWKoR6bKNJaKArMpLzwGgpRO9xoNGKbIDHgYZ8RSYsLIwYSGNgYBhyNTGGB4ZaKSQoKHgAPnQ5LGJgqLWFhl0SAgC8fB/vmSFVmgAAAABJRU5ErkJggg==' />"}],"value":"[4 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x6b0a01de \"comportexviz.server.notebook$eval28199$fn$reify__28203@6b0a01de\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>5</span>","value":"5"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAADAAAAAhCAYAAACfiCi5AAAA9ElEQVRYR2N89+7dfwYo+P//fyMjI2M9KXwhYWG4+p9+fgfZN22yh+mnBf/bggVw88HuHfIeePv2bQMsxJiYmA78+/fPgRQ+Z3w8XD+DhMQGhhcvAmD6acH/2t4ugOy+oR8Do0loNAnhL7UwSh30UnKwJyGCHhjspRB6qYNeSg76UohgDIwmIRqXQqNJiFDjjtaNOYJ5YLQUonFzmmAMjJZCo6XQaFtocPfIRiuy0YqMwk7+aBIaTUKDLQnRuvWJPjSJ3hYSFhZGDKQxMDCQ3KUcdB4gtTk90EOJgoKCB+BDl8MiBkhtTo8mIRJLJfSKDD0JAQB8Ah/viCN71gAAAABJRU5ErkJggg==' />"}],"value":"[5 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7dce7469 \"comportexviz.server.notebook$eval28199$fn$reify__28203@7dce7469\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>6</span>","value":"6"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAADAAAAAhCAYAAACfiCi5AAAA70lEQVRYR2N89+7dfwYo+P//fyMjI2M9JXwhYWG4/p9+fgfZN22yh5lHDf63BQvg5oHdO+Q98Pbt2wZYCDExMR349++fAyV8zvh4uHkMEhIbGF68CICZRw3+1/Z2AWT3Df0YGE1CaEluNAkRKLVGSyFCpdRoEhpNQqMV2WhbCG/jD6MYHW0LoTW/R5vTpBajo0loNAmR1uUcLYXQO/20LnXQBwWo3pwecA9QWgrRuvmMPihA9U79gMcApZ36AffAaBKi8VAieinEsWkTfOANNMBF8cAWvZMQhgeGWhLimDVrAnyocljEwFArhYZdEgIAG9Mf75+FRWUAAAAASUVORK5CYII=' />"}],"value":"[6 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2b9ead2a \"comportexviz.server.notebook$eval28199$fn$reify__28203@2b9ead2a\"]]"}],"value":"([0 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7db48dbd \"comportexviz.server.notebook$eval28199$fn$reify__28203@7db48dbd\"]] [1 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x3c855aec \"comportexviz.server.notebook$eval28199$fn$reify__28203@3c855aec\"]] [2 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x4e8300aa \"comportexviz.server.notebook$eval28199$fn$reify__28203@4e8300aa\"]] [3 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x328aab57 \"comportexviz.server.notebook$eval28199$fn$reify__28203@328aab57\"]] [4 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x6b0a01de \"comportexviz.server.notebook$eval28199$fn$reify__28203@6b0a01de\"]] [5 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7dce7469 \"comportexviz.server.notebook$eval28199$fn$reify__28203@7dce7469\"]] [6 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2b9ead2a \"comportexviz.server.notebook$eval28199$fn$reify__28203@2b9ead2a\"]])"}
;; <=

;; **
;;; Already we see a potential area of improvement: **wrap encodings around at the edges**. Hours 0:00 and 23:00 currently don't overlap. Neither do Day 0 and Day 6. Maybe some day we'll come back to this.
;; **

;; **
;;; ### Consumption rates
;; **

;; **
;;; The "consumption" is a power consumption rate, measured in kilowatts.
;; **

;; @@
(def input-consumptions
  (->> the-data
       (map (fn [[timestamp consumption]]
              consumption))))

(take 10 input-consumptions)
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-double'>21.2</span>","value":"21.2"},{"type":"html","content":"<span class='clj-double'>16.4</span>","value":"16.4"},{"type":"html","content":"<span class='clj-double'>4.7</span>","value":"4.7"},{"type":"html","content":"<span class='clj-double'>4.7</span>","value":"4.7"},{"type":"html","content":"<span class='clj-double'>4.6</span>","value":"4.6"},{"type":"html","content":"<span class='clj-double'>23.5</span>","value":"23.5"},{"type":"html","content":"<span class='clj-double'>47.5</span>","value":"47.5"},{"type":"html","content":"<span class='clj-double'>45.4</span>","value":"45.4"},{"type":"html","content":"<span class='clj-double'>46.1</span>","value":"46.1"},{"type":"html","content":"<span class='clj-double'>41.5</span>","value":"41.5"}],"value":"(21.2 16.4 4.7 4.7 4.6 23.5 47.5 45.4 46.1 41.5)"}
;; <=

;; **
;;; There are a few ways of encoding numbers.
;;; 
;;; Two objectives for encoding:
;;; 
;;; 1. Similar values have similar encodings.
;;; 2. Different values have different encodings.
;;; 
;;; If you want both, you won't be able to beat a "moving window" encoding.
;;; 
;;; $$ capacity = total\\_bits - active\\_bits $$
;;; 
;;; If you're willing to sacrifice some of (2), you can use every [choice](https://en.wikipedia.org/wiki/Binomial_coefficient) of bits. At least, that's the theoretical limit. It will take a lot of effort to reach it.
;;; 
;;; $$ capacity = {total\\_bits \choose active\\_bits} $$
;;; 
;;; If you're willing to sacrifice some of (1) and (2), and rely on probability instead, you can encode an arbitrary number of values, depending on how little you care.
;;; 
;;; $$ \lim\limits_{care \to 0}capacity = \infty $$
;;; 
;;; I'm going to explore both ends of the spectrum, but not the middle. The middle seems hard.
;; **

;; **
;;; #### Scalar encoding: Moving window
;; **

;; **
;;; This approach requires us to specify min and max values. We have those readily available.
;; **

;; @@
[(apply min input-consumptions)
 (apply max input-consumptions)]
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-double'>4.4</span>","value":"4.4"},{"type":"html","content":"<span class='clj-double'>90.9</span>","value":"90.9"}],"value":"[4.4 90.9]"}
;; <=

;; **
;;; We could have, say, 1000 bits, where the 415th bit means "41.5". To make this discussion easier, let's forget about the decimal point and just use these * 10 integers.
;; **

;; @@
(def consumption-encoder
  (partial ranged-int-encoder 0 1000))

(for [c [10 20 30]]
  [c (viz (consumption-encoder 0 1) c)])
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>10</span>","value":"10"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAACWCAYAAABJoA8EAAACUUlEQVR4Xu3dMW7CQBSEYZsTpOQ2yQl8pYSKa3CEHIEcyO4SJQ1OYZOCap1BMgsfHRKW0NOO53+zb+12GIaxmT/jOO7atn31fapAUT0UMFwwfd+/nVfcZrM5nk6nF9+nCpTUo7UCwxWogGEBSTi7ZZFwSh0kTMKrUgMJk3AowbSAXJgLr9u7c+HwFkDCJEzCVeePJEzCJEzCcwVKEtx7S7z1wmkrB6SBtDirahMhYRJeV8Jj0/yNdvx03cfX4fBctaRSV116vQKGEv7uuuN5xTXb7fvnfv9ktMNoR/l01VLJXv6eC4cSFmeJs8RZVWMTCZMwCZOwRHrBWH8Knvd2PZAG0uvGWVZguAIVMCwgkAbSQBpIA2kgXXw+WCIduu5lAbkwF+bCXJgLc2EuXPqYkmvnkVyYC3NhLsyFuTAX5sJlz6q6tZNOTiqlXGhTKYy3gDSQBtJAGkgDaSANpIF02XOXU3C9teuBNJA2YFk1B5IwCZMwCc8VKHp9xK1hSPp/xFniLHFW1fdAEiZhEiZhibREWiItkZZIS6QfspfVC2cgnE57mc5KV6BEWiItka66lSNhEiZhErapNFXgEV+oAqRTEE6v58JcmAtzYS7Mhf9LIeIsLhy6aFpAox1Zok3C6QoE0uEtgIRJ2HRW1Z0ICZMwCZOwAUsDlgYsDVjWuamlF9YLh71sWkAgDaSBNJAG0kAaSANpIO2kkpNK465dCtZAGkgDaSANpIE0kAbSQBpIA2kgfUxPHi293qbS0tbN+4WvvAmlF9YL64X1wg/cC/8CW/4praizZKgAAAAASUVORK5CYII=' />"}],"value":"[10 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x54a6fe78 \"comportexviz.server.notebook$eval28199$fn$reify__28203@54a6fe78\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>20</span>","value":"20"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAACWCAYAAABJoA8EAAACSUlEQVR4Xu3dwU3DQBRFUdsVsEw3UIFbgqzSRkqghFCQswPBJkYoCYusxn6RrHEOOySMxNfc3Pf/zJj2eDyOzeVrHMdt27avvj9XoKgeChgumGEY3q4rruu6w+l0evH9uQIl9WitwHAFKmBYQAhnH1kQTlMHhCG8aGqAMIRDBNMCsjALL9u7s3D4EQBhCEO46vkjhCEMYQhfKlAywV3bxFsvnLZygrQgbZxVtUQgDGEIQ/iRz9bohfXCeuGqPwMhDGEIQ9g4a8KZ4HT8s7bn9cJ6Yb1w1RKBMIQhDGHjrPKbOWvb103/HvvCabBnYRZmYRZmYRYuvR98a20WZuHQomkBbSrZVLKpVHWMgXCI8Ng0/y/e+en7j6/9/rnqFZFKYerzChha/LvvD9cV12w275+73ZMX75Q3FoL0VGRvf944K0SYhUMLW4HhClTAsIAQhrBeuOrOB8IQhjCEnZF2RvqvAkXvjdYLh8H5toAszMIszMIszMIsPDeGpCN9FmZhFmZhFmZhFmbhsv9hlN4suvfzjnakOdC+cDjeEqQFaUFakBakBWlBWpAWpOftq6ZBdunnBWlB2pX/qnMghCEMYQh7a0f55bp7T3Rr/30m0mkQZ2EWZmEWZmEW9u6spQaqLMzCLMzCLMzCLMzCM29LpsOA9Hmns5zOcjqr6hgDYQhDGMIOWDpg6YDlUp0IC7MwC7MwC7MwC7Owaw6uOcx6f1460a39eUFakBakBWlBWpAWpAVpQVqQFqTHbTs12AvSgrQgLUg/cJD+BdzIKa0BrE91AAAAAElFTkSuQmCC' />"}],"value":"[20 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x3d600904 \"comportexviz.server.notebook$eval28199$fn$reify__28203@3d600904\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>30</span>","value":"30"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAACWCAYAAABJoA8EAAACQklEQVR4Xu3dvXHCQBRFYYkKHNKNXYFasologxJcAhQkMnvshHUAOCBazWVmZ+FzpkCB3+jovL8V4/F4LMPlr5SyGcfx3fU5AlXxEMDwgZnn+eP6xK1Wq/3pdHpzfY5ATTxGT2D4BApgGEAIZ68sCKdZB4Qh3DRrgDCEQwTTALIwC7et3Vk4fAVAGMIQ7rr/CGEIQxjClwjUdHAfreOtFk5LOYm0RFo7q2uJQBjCEIbwM+/WqIXVwmrhrt+BEIYwhCGsnbVgJzht/zza/WphtbBauGuJQBjCEIawdlb9yZxHm+um/4+5cJrYszALszALszAL154PvrU2C7NwaNE0gIZKhkqGSl2nMRCGMIQhbC5sLlz9razbvFE7K0zEWZiFWZiFWZiFWbj2k51p++r2fhZmYRZmYRZmYRZm4brvNqfbVPe+31w4zQu1s7SzbGd1nQdCGMIQhrAFSwuWFixblXIszMJtLVyG4f9n0X6n6fC92712bcW0tl16vwCGCP9M0/76xA3r9efXdvtyvX7Gj8oubXdpZy1F1nZWiKy5cDYHdlIpRRbCEG6aOEMYwndGMA2o7azMyhLp9AnUzgpfCRCGsAXLrvuPEIYwhCFsR9qOtB3pVoN1FmZhFmZhFmZhFmZhJ5WGUspmTDu8vd2vI60j3XRObKiUvjIgDGEId13KQRjCEIaww4YOGzps2Kobw8IszMIszMIszMIsPPQ5ErDaYbXDakfXaQyEIQxhCNvOsp1lO6tVJcLCLMzCLPzEFv4DXaEprSNOhTYAAAAASUVORK5CYII=' />"}],"value":"[30 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x226f3223 \"comportexviz.server.notebook$eval28199$fn$reify__28203@226f3223\"]]"}],"value":"([10 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x54a6fe78 \"comportexviz.server.notebook$eval28199$fn$reify__28203@54a6fe78\"]] [20 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x3d600904 \"comportexviz.server.notebook$eval28199$fn$reify__28203@3d600904\"]] [30 #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x226f3223 \"comportexviz.server.notebook$eval28199$fn$reify__28203@226f3223\"]])"}
;; <=

;; @@
(for [c [10 20 30 35]]
  [c (viz (consumption-encoder 10 1) c)])
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>10</span>","value":"10"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAACZCAYAAAC49r3RAAACUElEQVR4Xu3csVHDUBCEYckVELobqEAtAZHbcAmUYAqyMxhIJMYjm4BI0gaaQ58zBUp27n97tz699nK5DM3tNwzDa9u2z55HBSbpQcCwYM7n88u94na73anv+yfPowJT9GhVYFiBBAwFhHB2ZEE47TogDOFVuwYIQzhEMBWQC3PhdWd3LhweARCGMIRL548QhjCEIXxTYEqC+98Sb7NwOspppDXS4qzSJgJhCK+L8NA0v6sd3133/nk8PpZGKnXVue8TMET4q+tO94pr9vu3j8PhwWrHjNUOFRhWIAFDASEcxlkqMKxAAoYCQhjC6ybSEIawWbj07A1hCEMYwlv+VEwjrZHWSJc+AyEMYQhDeMvbWUa5cJQjYCggF+bCXJgLc+FRAdtZCy4e4sJcWCJd2kQgDGEIQ9ifSjcF7EjPvv6vZSKhiRAwFFCcJc4SZ5VuYyAMYQhDWCItkb4qMOnq97/fE2ukNdLirNImAmEIQxjCEmmJ9FWBJdf3SaTnXrSjkQ5d96+A4ixxljirdBsD4RBhFzCGpkLAUEBXwUN4XReGMITXjbNUYFiBBAwF5MJcmAuXnoUhDGEIQ3jL21n6wLAPJGAoIBfmwlyYC3PhUYFFO8LpakT197kwF5ZIlzYRCEMYwhDe8o60WdgsbBYufQZCGMIQhrA4S5y1OM4zC5uFzcKlTQTCEIYwhMVZowJLPnnv+2yWrP5+y0RCEyFgKKA4KzuCIJyulkAYwhppjbRGWiO9dJDgwlw4dNFUQI20Rtr/wqXbGAhDGMIQ3vBqxw/8f9BLUHKbNgAAAABJRU5ErkJggg==' />"}],"value":"[10 #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0x5254ee51 \"comportexviz.server.notebook$eval28215$fn$reify__28219@5254ee51\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>20</span>","value":"20"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAACZCAYAAAC49r3RAAACUUlEQVR4Xu3dsXHCQBSEYYkKHNKNXYFash3RBiW4BFwQZPbYCfJ4hB0QndhA89BHRqCAnfffvrenO/rT6TR2l884jq993z/7PinQpAcBw4I5Ho8vfxW32WwO5/P5yfdJgRY9ehUYViABQwEhnC1ZEE67DghDeNGuAcIQDhFMBeTCXHjZ2Z0Lh0sAhCEM4dL5I4QhDGEIXxRoSXDvLfE2C6ejnEZaIy3OKm0iEIYwhCG85ndrzMJmYbNw6TUQwhCGMITFWTPeCU7jn3t73ixsFjYLlzYRCEMYwhAWZ7WfzLm3fd3099gXTht7LsyFuTAX5sJcuPV88LVrc2EuHLpoKqBNJZtKNpVKtzEQDhEeu+7/4p3vYXj/3O8fS1dEagpznydg6OJfw3D4q7huu3372O0eXLzTPlj0KjCsQAKGAkKYCy/bSEMYwssm0iowrEAChgJyYS7MhUuHFxCGMIQhvOZjDvrAsA8kYCggF+bCXJgLc+FJAfvCN/whDRfmwhLp0iYCYQhDGMJrPuZgFjYLm4VLr4EQhjCEISzOEmf9KtD0f8LXxyDMwmZhs3BpE4EwhCEMYXHWRQGHDQ9z79Jy2HDu+WCNdOi61wKKs8RZ4qzSbQyEIQxhCEukJdIS6Vsj+XQS4cJcmAtz4TW7sJvMw3SGgKGAbrAMXVgFhhVIwFBACEN42UYawhBe9tUOFRhWIAFDAbkwF+bCpeMsCEMYwhCWSE8K3HRSJ90WrP68RlojbRYubSIQhjCEIbzmk0pmYbOwWbj0GghhCEMYwiuOs34AjbbQS/L6kvYAAAAASUVORK5CYII=' />"}],"value":"[20 #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0x795b30e4 \"comportexviz.server.notebook$eval28215$fn$reify__28219@795b30e4\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>30</span>","value":"30"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAACZCAYAAAC49r3RAAACV0lEQVR4Xu3dMVICQRBG4V1OYMht9AR7JSXiGhzBI+iBINPShDEADIhm+a0ah/3MqHIDuubt6+7pGcbD4VCG818pZTOO47PPpwhUxUMAwwWz3+9fLitutVq9HY/HJ59PEaiJx2gFhitQAMMAQjh7ZUE4zTogDOGmWQOEIRwimAaQhVm4be3OwuErAMIQhnDX/UcIQxjCED5HoKaDe28db7VwWspJpCXS2lldSwTCEIYwhJc8W6MWVgurhbt+B0IYwhCGsHbWjJngtP1zb8+rhdXCauGuJQJhCEMYwtpZ9Sdz7m1fN/0+9oXTxJ6FWZiFWZiFWbj2fPC1tVmYhUOLpgG0qWRTyaZS12kMhCEMYQjbF7YvXH1X1nXeqJ0VJuIszMIszMIszMIsXHtlZ9q+un6ehVmYhVmYhVmYhVm47t7mdJrqr5+3L5zmhdpZ2lmms7rOAyEMYQhD2IClAUsDlq1KORZm4bYWLsPw+7No39P0/rnbPXZtxbS2nfu8AIYIf03T22XFDev168d2+3D5vMRLZee2u0YrMFyBAhgGEMLhvrAVGK5AAQwDCGEItx3tgDCE1cJd194QhjCEIbzkTSWJtERaIt31OxDCEIYwhJd8zEEpF5ZyAhgGkIVZmIVZmIVPETCdVTaj6azQqnMDyMIszMIszMIsfPORf7VwaG0WZmEWZmEWZmEWbnVak4VZmIVZmIVZmIVZuNPr73Rjwm6MAIYBlEhLpCXSEmmJtERaIi2RlkgPpdww3TR3mOe//b9EWiLd9rywCxjDFSiAYQD9plJWC/8AHu3QS1nPQGUAAAAASUVORK5CYII=' />"}],"value":"[30 #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0x7211249a \"comportexviz.server.notebook$eval28215$fn$reify__28219@7211249a\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>35</span>","value":"35"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAACZCAYAAAC49r3RAAACm0lEQVR4Xu3dMU7rUBBGYTsdHYImu4EVeAssBaiyjSyBJcCCkg4ETYyixBCZ6uZHmCt/r7Pi14zm+MzMHZt2u932zfFf3/ePV9fX98P1R9e9vK3XN6e/t2379fv+/tlfC2CYEJvN5mHIsMVi8Xxxd/d13SyXT6+r1eXp77vd7tb1IQL7eLUyMMxAAQwDCOHskQThtKqAMIQnrQogDOEQwTSALPzLFp59b1uakWMLC2DhI2GMsF63DOkfFpaBhRkI4cKAjZ+REC5DdvyIg3Cpdcf3QxjC/6sXZuHCjIRwYcBYOLMuC6fWZeEQWQhD+H+tlrBwiLReOENaL5xaGcIQ1gtXvcAJYQhDGMInS+qzmycqpBXS0/bGLMzCLMzCLPz95o3trDIrG2cZZ4UWTQOokC5D1rlwmnHOhX8ZeQhDWC9cdScCYQhDGMLHCOy/YjG3XlovnBbWJtJhYc3CLMzCLMzChwj4nN0Zn/NjYRZ2Lly1RCAMYQhD2GqH1Y7ho7Kl4zjjLOOs0KJpAI2zjLOMs6ouYyAMYQhD2ETaRPrsibxxVliIszALszALszALs/BUiwEszMIszMIszMIszMKHP/ZZei479f3OhdNjTeMs4yzbWVXXgRCG8LQI903TDwh9dN3L23p9UzVSqVVL/78Ahgi/d93zkHHNcvn0ulpdDtc1FrZ/XVi3MjDMQAEMAwjhcCItA8MMFMAwgBCG8LSHShCGsF646t4bwhCGMITn/L6wQlohrZCu+hkIYQhDGMJzXrDUyoWtnACGAWRhFmZhFmbhQwRsZ53xEVoWZmET6aolAmEIQxjCDpWOEbAjXfyyox3p0tcaxvezMAuzMAuzMAvvI3DOm1kszMKhRdMAOlRyqORQqeoyBsIZwp84lNxL3hRObQAAAABJRU5ErkJggg==' />"}],"value":"[35 #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0x1a3e0a89 \"comportexviz.server.notebook$eval28215$fn$reify__28219@1a3e0a89\"]]"}],"value":"([10 #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0x5254ee51 \"comportexviz.server.notebook$eval28215$fn$reify__28219@5254ee51\"]] [20 #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0x795b30e4 \"comportexviz.server.notebook$eval28215$fn$reify__28219@795b30e4\"]] [30 #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0x7211249a \"comportexviz.server.notebook$eval28215$fn$reify__28219@7211249a\"]] [35 #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0x1a3e0a89 \"comportexviz.server.notebook$eval28215$fn$reify__28219@1a3e0a89\"]])"}
;; <=

;; **
;;; Maybe give each number a 100 bit (10.0) radius. Then 20 and 35 share 50 bits in common. This means a single representations is 201 on-bits, and the total number of bits goes up to 1200.
;; **

;; @@
(for [c [200 350]]
  [c (viz (consumption-encoder 100 1) c)])
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>200</span>","value":"200"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAAC0CAYAAAADxKs5AAADJ0lEQVR4Xu3dTTIDURTF8e6eKDPFJLthBcoOLAUj28gSlBWwG5NkRpVJYkBSpalwc1+HfvmZpXRIXe//zrnnfWjn8/my+fhaLpc3bdteRV4fn5ysn389P388uL8/Xb3/u9cv0+n6+9v8vujnG/x5BYwNmC9/kNlsdr0aMV3XPSwWi7PI68PLy/X7m8nkrnl6uli9/7vXz7e3R5GfH/08u36+NQKTI1ABkwWEcGzK6k8REA66ji8iAmEIh1xDaZWGMISTCGYLSIWpcKh3L94bU+HkFABhCEM4kj8Wn8OocA7BrLFmpLMjkApTYb3wqEUEwhCGMIQ/KrCXC+t64ZwRZ6QZ6aSKZgsIYQiLs0ZtYyAMYQhDeFUBGyztkQ6vuYizkkacClNhKkyFqfB7BbY5OeSkkqNeuTmUClPh3AjKBqqMNCNtXXjUPhDCEIYwhG3teK+AI//xKw9s7WCkkyqaLaBeeMe9cDS+6l/Es/cX7yhgb8qIIhy9aKd/Ec/eX7xjBPZGYLQXVkAIF+2dw0baCIRw2SUAKsxIlx1R0daOCid7aQhDGMKjjvQhDGEIQ9gGy99vsNQL64XLzplUmAqXHVF64WRvGy0ghCEMYUaakWaktz2pZVUuqrr956kwFabCVJgKU2EqvOWB8fA1J1Q4p7rpf4smkZZIl7U9jHQOaa1ctpWzOyuZYEMYwmVFIYo0hCFc9NhC9H8sUeEosv3nIQxhCI86UIUwhCEMYRfv/P7iHUf+P/fOjDQjnVTRbAHFWeIscdaobQyEIQxhCNudNdzurOquu9t1Il19AYdW4dquu9v57qzqR+DQiXT1BYTwPzfS1Y9ACCfjMAhDuO5euPo5EMIQhrA4a0OcpRfuxTfRk0rViwgjzUjXvTsLwubAzTaJkWakGWlGmpG2Lvxvb+2o3sZQYSpMhakwFabCVLjWu7Mk0tKYzSovkZZIS6RH7QMhDGEIQ3jDkX9x1uJz3BM98s9IM9KM9KBzrERaIi2RHhSx7K0cP70fwhCGMIQtKllUsqhkUalpuq57iN7j/NfPD379XfVpjERaIi2RHrUPhDCE/xThN6xcB2BDj2vUAAAAAElFTkSuQmCC' />"}],"value":"[200 #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0x45316b84 \"comportexviz.server.notebook$eval28215$fn$reify__28219@45316b84\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>350</span>","value":"350"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAAC0CAYAAAADxKs5AAAC+UlEQVR4Xu3dQU4iQRSA4W42xt1EN97GOQHxBhxFXXkNj2A4wXgbNrCbSdzQxiiEoGEsXxGquj53nZgOvNTf/3uvXhf9arUauo+/YRju+76/zXl9cXm5vd/LdPp8Np9fb+7/net/j4/b/z/G5wt/XwEMLpjlcnm3WRGTyeTPer3+nfP6fDbb3r+7unrqFoubzf2/c/334eFXzs+T+/v1VmBwBQpgMIAQjj2yIBzNOiAM4axZQ6qlIQzhIILRALIwC2et3ZNrYxYOPgIgDGEIH+oPFt8PhDCEIQzhjwh81dHWkQ6WOjaVBPDwM5aFWZiFWZiF3yNwjNEJFmZhFj7maEhxm0rFd1/2iSytI11dAEtLpEufxtrfN4ZwUJICGA0ghEdWC1cnERYe2b4wC+89lFNrYQi3FkAWZmEd6d2OdHXPQAhDGMIQzripJJFuLQ9UC6uFvalUtUQgDGEIQ/jAsSmj7weqhdXCauGqn4EQhjCEIaydVc6MdHV7ImphtbBauGqJQBjCEIawdtZ7BEo4hNa+sH3hoJVaC6B2lnaWdlbVaQyEIQxhCOtI60j/+NQS7axg4cDCLMzCLMzCLMzCRzo87b9n67MwC7MwC7MwC7MwC3fd24+V2hdubVtTO0s7y3RW1XkghCEMYQgbsDRg+dMBU8cgR49BZmEWZmEWZmEWZuFTNVRZmIVZmIVZmIVZmIW77mU6fa7u2BPTWaazTGdVncZAGMIQhrABSwOWBixPVYmwMAuzMAuzMAuzMAt7zaEbhuF+//XQ0f+eiH1h+8L2havOAyEMYQhD2GiH0Q6jHacq5ViYhcdl4dTa92w+v2bhHQs3F8Dcm0rns9ndZkW9ncLRLRY3KdfNn9rR3ArMbeHmAgjhwvaFm1uBEC4skWbh4CvwEBbAtFkbFmbhtBUTJOzTscgszMLaWdpZ2ln5NpUk0kFLSaQFMC0tkkhLpNNWTJAwiXTuAEIYwhCuuhKBMIQhDOGdlw3VwsE8SS0sgGnPVBaOWfgV7swHYEr8hC4AAAAASUVORK5CYII=' />"}],"value":"[350 #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0x47484fee \"comportexviz.server.notebook$eval28215$fn$reify__28219@47484fee\"]]"}],"value":"([200 #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0x45316b84 \"comportexviz.server.notebook$eval28215$fn$reify__28219@45316b84\"]] [350 #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0x47484fee \"comportexviz.server.notebook$eval28215$fn$reify__28219@47484fee\"]])"}
;; <=

;; **
;;; Without this overlap, 20.0 would be as similar to 20.1 as it is to 99.
;; **

;; @@
(for [c [200 201 990]]
  [c (viz (consumption-encoder 0 1) c)])
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>200</span>","value":"200"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAACWCAYAAABJoA8EAAACSElEQVR4Xu3dvW3CUBRAYdtduig0bJNMwAqMEqhYgxEyAhnI7hIpjV8KIIWrZ10r5tlfOgoj5crH5/49U3ddl6rbX0rpWNf1u8/XCGTFQwCDN0zbtof7Hdc0zaXv+zefrxHIiUftDgzegQIYDCCEY48sCEezDghDeNasAcIQDiIYDSALs/C8tTsLBx8BEIYwhIvuP0IYwhCG8C0COR3cpXW81cLRUk4iLZHWzipaIhCGMIQhvObdGrWwWlgtXPQzEMIQhjCEtbNG7ARH2z9Lu14trBZWCxctEQhDGMIQ1s7KP5mztLlu9P8xF44m9izMwizMwizMwrnng4fWZmEWDlo0GkBDJUMlQ6Wi0xgIQxjCEDYXNhfOflfWMG/Uzgom4izMwizMwizMwiyc+8rOaPtqeD0LszALszALszALs3Dee5uj21RTX28uHM0LtbO0s2xnFZ0HQhjCEIawBUsLlhYs5yrlWJiFWZiFWZiFWZiFM3+KMdq/m/p6c2FzYXPhotMYCEMYwhC22mG1w2rHXJUIC7MwC7MwC7MwC7OwBcsqpXSsp25YPvr3mQubC5sLF50HQhjCEIaw1Q6rHVY75irlWJiFWZiFWZiFWZiFLVhWTdNcpj5S/+jf58h/tOMtkZZIS6Ql0hJpibREWiItkZZIr3I1QyIdTISjAbQjbUfajnTRlQiEIQxhCDvm4JiDYw5zNRNY+J8t/LLZ/FnvZ7f7/D6fX4uWQLSUGztUEsBB7T0W4af9/nC/46rt9uPrdHq+f17jXPgXz+ksrb1PyGUAAAAASUVORK5CYII=' />"}],"value":"[200 #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0x2625ff26 \"comportexviz.server.notebook$eval28215$fn$reify__28219@2625ff26\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>201</span>","value":"201"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAACWCAYAAABJoA8EAAACXUlEQVR4Xu3dsXHCQBCFYYnMmcckdGNXQAuUYhPRBiW4BFwQZPaME+QATADRamWkHX/OmJGAOe6/997undweDoeuOf91Xbdu2/Y18vppPr9c/71cfnxtt8+R+6OfN7nrDWBswtz8gPv9/u13xsxms93xeHyJvH5YrS73N4vF++dm8xi5P/p5U7u+zc7AySEVXILS398Ajozw1JC69/eBcBZ5CEM45BqGRhzCEE4imB3ArJEeGolq7wfh7AykwsklAMKx7H+9xEAYwkkEswMIYQiHKujp8tX1jKXCySUAwhCGcOkuHoQhDGEIn0egT1+5WvlKFs5GN0Y6aZyvB5AKU2EqTIWp8GkE+mywHLw8NLRK/vX7KWclVZkKU2EqTIWpMBXu7UKoMBW2wbK0iEAYwhCGcOK0Z/lihCwsC8vCpddACEMYwhBWzlLOUs4aq6tIhakwFabCVJgKU2Eq3DQV91g7L5zdvaWppKmkqVTaB0IYwhCGsL7waQQqGtmxDysy0ox0UkWzA6girSKtIl3axkAYwhCGsKaSppKmkqZSzSwuC2ejnIp0Mksz0ow0I81IM9KMNCPNSDPS//PZW4w0I21rR2kfCGEIQxjCdmfZndV3d5qKtIp0UkWzA6girSKtIl3axkAYwhCGsKaSppKmkqaSppKmkv9o063DD8VlpBlpRpqRZqQZaUaakWakGWlGmpHe3fspHvrC2bam3VnJvrIsLAvLwrKwLCwLy8KysCwsC8vCsrAsHG4rZrNs9n5ZWBZ2Uql0lIMwhCEMYYcNHTZ02HCsagwVpsKjqvAPgiEsrUCkIfYAAAAASUVORK5CYII=' />"}],"value":"[201 #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0x92640b6 \"comportexviz.server.notebook$eval28215$fn$reify__28219@92640b6\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>990</span>","value":"990"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAACWCAYAAABJoA8EAAACR0lEQVR4Xu3dMW7CQBRFUdtd2tCwnKyALbCUwI6yBLIg6NMwKYAUUDDWQzhjTjoUOVJGc33f/39s+sPhULrzTyll2/f9p8+nFahaDwsYbpj9fr+57LhhGHbH4/HD59MK1KxHbweGO9AChgsI4eyWBeE0dUAYwpOmBghDOEQwXUAWZuFpa3cWDm8BEIYwhJvuP0IYwhCG8HkFajq4c+t4q4XTUk6QFqS1s5qWCIQhDGEIv/LZGrWwWlgt3PQ9EMIQhjCEtbNGnAlO2z9zu14trBZWCzctEQhDGMIQ1s6qfzJnbnPd9P8xF06DPQuzMAuzMAuzcO3zwdfWZmEWDi2aLqChkqGSoVLTMQbCEIYwhM2FzYWr35V1nRu1s8IgzsIszMIszMIszMK1r+xM21fX17MwC7MwC7MwC7MwC9e9tzk9TfXo682F01yonaWd5XRW0zkQwhCGMIQdsHTA0gHLqUo5FmZhFmZhFmZhFmbhyq9iTPt3j77eXNhc2Fy46RgDYQhDGMKOdjja4WjHVJUIC7MwC7MwC7MwC7OwA5ZdKWXbP7ph+d//nrmwubC5cNM5EMIQhjCEHe1wtMPRjieVcu+LxV/77Ge1+vao18hK52YBdaTHdaTf1uvNRXrdcvllB47cgTe1viAtSAvSgrQgLUgL0k8K0iycxhbvzhoXnO+95UOQTnekIC1IC9KCtCAtSAvSgrSHDbthGHb3gufcfi9IC9JhEE4X0FApq40hnO5AtXB4C4AwhD2p1HQzAcIQhjCEX/hhw1+HlinulJUH8wAAAABJRU5ErkJggg==' />"}],"value":"[990 #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0x6d57aa44 \"comportexviz.server.notebook$eval28215$fn$reify__28219@6d57aa44\"]]"}],"value":"([200 #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0x2625ff26 \"comportexviz.server.notebook$eval28215$fn$reify__28219@2625ff26\"]] [201 #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0x92640b6 \"comportexviz.server.notebook$eval28215$fn$reify__28219@92640b6\"]] [990 #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0x6d57aa44 \"comportexviz.server.notebook$eval28215$fn$reify__28219@6d57aa44\"]])"}
;; <=

;; **
;;; In fact, this 200 bit representation makes the bold statement that 20 is equally different from 40 as it is from 90. A more information-rich approach would have 1000 bit representations within a total array of 2000 bits. It'd be equipped to handle all values between 0 and 100, to one decimal place.
;; **

;; @@
(for [c [400 900]]
  [c (viz (consumption-encoder 1000 1) c)])
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>400</span>","value":"400"},{"type":"html","content":"<img src='data:image/png;base64,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' />"}],"value":"[400 #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0x3e692dbc \"comportexviz.server.notebook$eval28215$fn$reify__28219@3e692dbc\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>900</span>","value":"900"},{"type":"html","content":"<img src='data:image/png;base64,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' />"}],"value":"[900 #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0xc089c38 \"comportexviz.server.notebook$eval28215$fn$reify__28219@c089c38\"]]"}],"value":"([400 #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0x3e692dbc \"comportexviz.server.notebook$eval28215$fn$reify__28219@3e692dbc\"]] [900 #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0xc089c38 \"comportexviz.server.notebook$eval28215$fn$reify__28219@c089c38\"]])"}
;; <=

;; **
;;; ##### Settle down
;;; 
;;; You might be screaming right now. Numbers can be encoded much more effectively. Computers do it through Base 2 counting. Biology does it through spike rates.
;;; 
;;; Base 2 isn't appropriate for HTM encoders because it provides jittery overlaps in bits of nearby numbers. You'd be in a world where 1 is closer to 256 than it is to 2.
;;; 
;;; But you're totally right about spike rates. The neocortex uses spike rates to encode numbers like orientation (the measure of the angle) and color (e.g. the ratio of red to green). HTM intentionally doesn't model this. The reasoning: the neocortex uses spike rates, but not for everything, so let's see how far we can get without them. There's a risk of getting caught up in closely mimicking low-level details, never reaching the point of characterizing higher fundamental concepts like temporal pooling.
;; **

;; **
;;; #### Scalar encoding: Probabilistic
;; **

;; **
;;; With the moving window approach, we represented a value in terms of its nearby values. In some sense:
;;; 
;;; - 9 is @@ \\{7, 8, 9, 10, 11\\} @@
;;; - 10 is @@ \\{8, 9, 10, 11, 12\\} @@
;;; - etc.
;;; 
;;; 9 and 10 share 4 bits.
;;; 
;;; This approach has two weaknesses:
;;; 
;;; 1. You have to know the range of numbers that you'll ever see.
;;; 2. The @@ active\\_bits @@ count grows with the radius.
;;;    - If an encoding illustrates the relation between 42 and 86, then _a lot_ of bits are going into representing 42. This means lots of synapses, which means longer training periods and slower HTMs.
;;; 
;;; You can "solve" both of these problems by relying on randomness / probability. Actually, you don't have to use randomness to solve Weakness 2, but for this discussion we will because that's what the Coordinate Encoder does.
;;; 
;;; The coordinate encoder addresses these problems as follows:
;;; 
;;; - To solve Weakness 2, sample the numbers within the radius, rather than use all of them. 
;;;   - Strategy: Hash each number and take the top few results. That way, we're likely to choose many of the same numbers as other nearby encodings.
;;;   - <a name="alternate_strategy" /> Alternate Strategy: Select the current number and the nearest few multiples of @@ \frac{2 * radius}{active\\_bits - 1} @@. This would give more consistent / predictable results.
;;;     - On top of this: choose multiples of larger and larger numbers as you move away, enabling varying precision.
;;;       - Alternate Strategy: Actually finish this essay.
;;; - To solve Weakness 1, simply hash each number and coerce it into an integer modulo the number of bits.
;; **

;; @@
(def example-enc2
  (e/coordinate-encoder [100] 5 [1] [2]))

(for [v [9 10]]
  [v 
   (into #{} (p/encode example-enc2 [v]))
   (viz example-enc2 [v])])
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>9</span>","value":"9"},{"type":"list-like","open":"<span class='clj-set'>#{</span>","close":"<span class='clj-set'>}</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-unkown'>65</span>","value":"65"},{"type":"html","content":"<span class='clj-unkown'>32</span>","value":"32"},{"type":"html","content":"<span class='clj-unkown'>56</span>","value":"56"},{"type":"html","content":"<span class='clj-unkown'>97</span>","value":"97"},{"type":"html","content":"<span class='clj-unkown'>67</span>","value":"67"}],"value":"#{65 32 56 97 67}"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAA00lEQVRYR+2WsQ3CQAxF4+vSIdJkmzBBVsgoQJU1MgIjwEB3HUg0dzSQwqmsb0cWuutOcvH1n/63KaVUmu8rpVyJ6Ozq715gjPHycyyEcM85nzz9yb2D7gX+HeJj160pf4/j47Usg2nqpYh3FyhF3E7Tmvqm72/PeT5Ypr6mGN5MUsR7F3lFXBGjDsC9KS1qqWBYoHWK0WJXTzHsGL/otRGrC9RGjCLli0AdMQ+R1FE+71+gNmKOSIqcz5s7KO3Nzbx2imFBvGasEaPnWUUMI/eO+ANxp+qq551nNQAAAABJRU5ErkJggg==' />"}],"value":"[9 #{65 32 56 97 67} #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0x278a78e \"comportexviz.server.notebook$eval28215$fn$reify__28219@278a78e\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-long'>10</span>","value":"10"},{"type":"list-like","open":"<span class='clj-set'>#{</span>","close":"<span class='clj-set'>}</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-unkown'>65</span>","value":"65"},{"type":"html","content":"<span class='clj-unkown'>32</span>","value":"32"},{"type":"html","content":"<span class='clj-unkown'>97</span>","value":"97"},{"type":"html","content":"<span class='clj-unkown'>83</span>","value":"83"},{"type":"html","content":"<span class='clj-unkown'>67</span>","value":"67"}],"value":"#{65 32 97 83 67}"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAA1klEQVRYR+2WsQ2DMBBFsbt0UWjYJkzACoySpGINRmCEZCC7S6Q0dpqEgubrZH7uhKCzTPF0z8/gYoy5+j4555tz7mJqbR4whHD9Tcx7f08ptZbWzvwEzQNuTvGprufK3133eI3jmVq9VPHfAaWKD30/V181zfQchiOzevWKoRGp4rW/NBBQqnjtixwdGXXF0Ii2YgiorRgdGbpiGAH6vWMrLgZkK0aVqiuGEWgrLgZkK0YK0b65ipdR2QdkK5ZWvHyfPsHiSNgXdTEgWzGqFO3vijev+AP4Teqq5A90UQAAAABJRU5ErkJggg==' />"}],"value":"[10 #{65 32 97 83 67} #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0x279588af \"comportexviz.server.notebook$eval28215$fn$reify__28219@279588af\"]]"}],"value":"([9 #{65 32 56 97 67} #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0x278a78e \"comportexviz.server.notebook$eval28215$fn$reify__28219@278a78e\"]] [10 #{65 32 97 83 67} #object[comportexviz.server.notebook$eval28215$fn$reify__28219 0x279588af \"comportexviz.server.notebook$eval28215$fn$reify__28219@279588af\"]])"}
;; <=

;; **
;;; Now:
;;; 
;;; - 9 is @@ \\{65, 32, 56, 97, 67\\} @@
;;; - 10 as @@ \\{65, 32, 97, 83, 67\\} @@
;;; 
;;; They still share 4 bits.
;;; 
;;; Humans can't make any sense of these encodings. And good luck writing code that decodes it. It uses the hash of the input, so you can't work backwards, so you'd have to brute-force the infinite domain of possible numbers.
;; **

;; **
;;; **The random casualties**
;;; 
;;; For dramatic effect, we're only using 100 bits.
;;; 
;;; Let's start without sampling. Include all bits within the radius.
;;; 
;;; Without this substitution, overlap only happens when the values are nearby, +- 4. With the substitution, does that stay true?
;; **

;; @@
(defn spacer
  [v]
  (subs (str v "   ") 0 3))

(print (spacer ""))
(doseq [j (range 0 100 9)]
  (print (spacer j)))
(println)
(doseq [i (range 0 100 9)]
  (print (spacer i))
  (doseq [j (range 0 100 9)]
    (let [overlap (set/intersection (into #{} (p/encode example-enc2 [i]))
                                    (into #{} (p/encode example-enc2 [j])))]
      (if (pos? (count overlap))
        (print (spacer (count overlap)))
        (print (spacer "")))))
  (println))
;; @@
;; ->
;;;    0  9  18 27 36 45 54 63 72 81 90 99 
;;; 0  5        1  1        2  1           
;;; 9     5                 1  1  1     1  
;;; 18       5     1              1        
;;; 27 1        5  1           1           
;;; 36 1     1  1  5           1  2        
;;; 45                5                    
;;; 54                   5           1  1  
;;; 63 2  1                 5     1     1  
;;; 72 1  1     1  1           5           
;;; 81    1  1     2        1     4     2  
;;; 90                   1           5     
;;; 99    1              1  1     2     5  
;;; 
;; <-
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; **
;;; The world becomes a little bit weird. The overlap between 36 and 81 is the same as the overlap between 36 and 39. Meanwhile 36 and 41 correctly have no overlap.
;; **

;; @@
(let [i 36
      enc example-enc2]
   (doseq [j [39 41 81]]
     (let [i-bits (into #{} (p/encode example-enc2 [i]))
           j-bits (into #{} (p/encode example-enc2 [j]))]
       (println i i-bits "and" j j-bits "overlap" (set/intersection i-bits j-bits)))))
;; @@
;; ->
;;; 36 #{82 14 79 10 42} and 39 #{77 29 34 82 79} overlap #{82 79}
;;; 36 #{82 14 79 10 42} and 41 #{77 29 34 19 96} overlap #{}
;;; 36 #{82 14 79 10 42} and 81 #{62 14 10 67} overlap #{14 10}
;;; 
;; <-
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; @@
;; In case you want to see those.

(for [i [36 39 41 81]]
  (viz example-enc2 [i]))
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAA4ElEQVRYR2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGRkZ4fIg9TTnD3oHvn37tgEWQkxMTAc44+PhfAYJiQ1f29sFkOX//fvnQE8+I3oI0jzKSE0ig96B6FFM7ygkZB/FUUxprieon9IoJmgBgTRHUD+lUUxpriekn+IopnmupzSKae5ASqOYUC6kVB4jigkmWlILWkrVj9bFFNbdo7mY4lw+5HIxxT6mNNei6x8tqEdzMbXTFKnmjeZiUkNsNBdTmGvRm2d0r4tJbs7Ru6Am2YH0zsWEOknoUQwAHUrtaXzP5DYAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAA1klEQVRYR+2XsQ2DMBBFOXd0KDRsQyZgBUZJUrEGIzACGcjuiJTGTpOghOb0dZF9xdFZgPXl9/8/mUIIqXo/KaUbEV1UrdUL9N5fPyfmnFtjjGdNa1J/guoFGmKmFU5tu7fGcxjuj3nuf1qkNGJWYGnE9TjuLVJ13bJNU/PdIpZi8WQqjZgbDDBi1tT/nuVoirMLRBFzqeOQoe9hxGLToxZAEWcXiCJGEUm/N8RiSxjiQ2rhHs2dYlhgbsRo0VuKLcWwqaWXpOP/XIqLC+RSjKaOm73ofi/o0O1p7EDx5AAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAA30lEQVRYR2N89+7dfwYo+P//fyMjI2P9oOIPege+ffu2ARZiTExMB/79++cwmPiMgz4EB70DR6OY0lJhNIopLBVGczHFNdOwy8VCwsLwuvqnn9/BbwsW2NO07iY1F9PdgaRGMWd8PLzuZpCQ2PC1vV2AlnX3aC4ezcUUhwCJdTF6Jhx0aRDDgaTmYlq3uNFLCbqHIMnlKKkFNaVpkmQH0juKSS3o6R7FJMcAvaOYZAfSO4pJLQUwohg9EbNv2gRvTtGleYVesKNH8aBzIHoUo+cyhhcvAmDNKXo0r9CTAAAFwuqqnmOl+wAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAeCAYAAABe3VzdAAAA0ElEQVRYR2N89+7dfwYo+P//f6OQsHA9jP/Tz+/gtwUL7JHlGRkZ4fIg9TTnD3oHvn37tgEWQkxMTAc44+PhfAYJiQ1f29sFkOX//fvnQE8+I3oI0jzKSE0ig96B6FFM7ygkZB/No5jiUoHWUUyxA2kdxZSWCjSPYopLBVpHMcUOpHUUE8qlhOQxopjiRE1qQUxI/WhdTGHdPZqLR14uptjHhHIlqfKjBfVgz8UUF/y0jmKKHUjruni0uTXgxQ6to5hQc4qQ/GhdTHESGexRDADoG+5prScvigAAAABJRU5ErkJggg==' />"}],"value":"(#object[comportexviz.server.notebook$eval28152$fn$reify__28156 0x7ee7f8bb \"comportexviz.server.notebook$eval28152$fn$reify__28156@7ee7f8bb\"] #object[comportexviz.server.notebook$eval28152$fn$reify__28156 0x38b7fd6d \"comportexviz.server.notebook$eval28152$fn$reify__28156@38b7fd6d\"] #object[comportexviz.server.notebook$eval28152$fn$reify__28156 0xa0be6b0 \"comportexviz.server.notebook$eval28152$fn$reify__28156@a0be6b0\"] #object[comportexviz.server.notebook$eval28152$fn$reify__28156 0x1cd195b9 \"comportexviz.server.notebook$eval28152$fn$reify__28156@1cd195b9\"])"}
;; <=

;; **
;;; Also, a single representation's bit count can be diminished when bits collide.
;; **

;; @@
(dotimes [i 50]
  (let [bits (v->bits i)]
    (when (not= 5 (count bits))
      (println i "is represented by" bits))))
;; @@
;; ->
;;; 11 is represented by #{65 97 83 67}
;;; 12 is represented by #{65 97 83 67}
;;; 13 is represented by #{65 83 53 67}
;;; 14 is represented by #{43 83 53 67}
;;; 33 is represented by #{91 10 42 37}
;;; 34 is represented by #{91 14 10 42}
;;; 35 is represented by #{14 79 10 42}
;;; 43 is represented by #{7 34 19 96}
;;; 
;; <-
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; **
;;; When we add in sampling, we usually get rid of this "diminished count" problem, but we create other quirks.
;;; 
;;; As shown below, with radius 4, the representation for 29 should overlap with 34, 35, 36, but if you only take 3 samples, it doesn't.
;; **

;; @@
(defn v->bits2 [v]
  (let [enc (e/coordinate-encoder [100] 3 [1] [4])]
    (into #{} (p/encode enc [v]))))

(print (spacer ""))
(doseq [j (range 25 40)]
  (print (spacer j)))
(println)
(doseq [i (range 25 40)]
  (print (spacer i))
  (doseq [j (range 25 40)]
    (let [overlap (set/intersection (v->bits2 i)
                                    (v->bits2 j))]
      (if (pos? (count overlap))
        (print (spacer (count overlap)))
        (print (spacer "")))))
  (println))
;; @@
;; ->
;;;    25 26 27 28 29 30 31 32 33 34 35 36 37 38 39 
;;; 25 3  3  3  2  2  2  2  2  1                    
;;; 26 3  3  3  2  2  2  2  2  1                    
;;; 27 3  3  3  2  2  2  2  2  1                    
;;; 28 2  2  2  3  3  2  2  2  1                    
;;; 29 2  2  2  3  3  2  2  2  1                    
;;; 30 2  2  2  2  2  3  3  3  2  1  1  1  1  1     
;;; 31 2  2  2  2  2  3  3  3  2  1  1  1  1  1     
;;; 32 2  2  2  2  2  3  3  3  2  1  1  1  1  1     
;;; 33 1  1  1  1  1  2  2  2  3  2  2  2  2  2  1  
;;; 34                1  1  1  2  3  2  2  2  2  1  
;;; 35                1  1  1  2  2  2  2  2  2  1  
;;; 36                1  1  1  2  2  2  2  2  2  1  
;;; 37                1  1  1  2  2  2  2  2  2  1  
;;; 38                1  1  1  2  2  2  2  2  3  2  
;;; 39                         1  1  1  1  1  2  3  
;;; 
;; <-
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; **
;;; Let's characterize the effect of all of this randomness.
;; **

;; **
;;; #### Doing the math
;; **

;; **
;;; Before we use an encoding strategy like this, we ought to know the answers to:
;;; 
;;; 1. **Given the distance between two coordinates, how much do their representations overlap?**
;;; 2. **Given the overlap of two representations, what is the distance between the two coordinates?**
;;;   - A less general version: This bit is shared. Is it legit?
;;; 
;;; 
;;; Let's say we use an @@ n @@-bit, @@ w@@-on-bit representation. The answers should come in the form:
;;; 
;;; **0 with probability \_, 1 with probability \_, ..., @@ w @@ with probability \_.**
;;; 
;;; Let's start with cases where @@ distance > 2 * radius @@. Explore how collisions can cause unexpected overlaps.
;;; 
;;; Notation:
;;; 
;;; - @@ a, b, ... @@
;;;   - pre-hash sequences that describe a number
;;;   - unbounded
;;;   - Example: @@ \\{7, 8, 9, 10, 11\\} @@
;;; - @@ a', b', ... @@
;;;   - random sequences generated from their associated @@ a, b, ... @@
;;;   - bounded by the number of bits
;;;   - Example: @@ \\{65, 32, 56, 97, 67\\} @@
;;; - @@ distinct @@
;;;   - a function that returns set of elements in a sequence. Duplicates are removed.
;;;   - Example: @@distinct((1,1,2)) = \\{1,2\\}@@
;; **

;; **
;;; Given two numbers described by @@ a @@ and @@ b @@,
;;; 
;;; $$ \begin{align} P(\left|a' \cap b'\right| = m \Big| \left|a \cap b\right| = 0) & =
;;; \frac{\left| \begin{array}{c} \text{candidates for} \\\
;;; a',b' \\\
;;; \text{such that} \\\
;;; \left|a' \cap b' \right| = m \end{array} \right|}
;;; {\left| \begin{array}{c} \text{candidates for} \\\
;;; a', b' \end{array} \right|} \\\\[2ex]
;;; & = \frac{
;;; \left| \begin{array}{c} \text{candidates for} \\\
;;; a' \cap b' \\\
;;; \text{such that} \\\
;;; \left| a' \cap b' \right| = m \end{array} \right|
;;; \left| \begin{array}{c} \text{candidates for} \\\
;;; a', b' \\\
;;; \text{given} \\\
;;; a' \cap b' \end{array} \right|
;;; }
;;; {\left| \begin{array}{c} \text{candidates for} \\\
;;; a', b' \end{array} \right|}
;;; \end{align}$$
;; **

;; **
;;; These @@ a', b' @@ candidates are intertwined because @@ a' @@ and @@ b' @@ can't contain any of the same values, except for those specified in the selected @@ a' \cap b' @@.
;;; 
;;; It's difficult to directly tackle the problem of "sequences containing each element of a set" because the sequence's other elements might be duplicates, so a "position the elements, select the remainder" strategy will double-count sequences. Instead, break it into cases based on the total number of distinct elements in the sequence.
;;; 
;;; $$ = \frac{
;;; \left| \begin{array}{c} \text{candidates for} \\\
;;; a' \cap b' \\\
;;; \text{such that} \\\
;;; \left| a' \cap b' \right| = m \end{array} \right|
;;; \sum \limits\_{i=m}^{w}
;;; \sum \limits\_{j=m}^{w}
;;; \left| \begin{array}{c} \text{candidates for} \\\
;;; a', b' \\\
;;; \text{given} \\\
;;; a' \cap b' \\\
;;; \text{such that} \\\
;;; \left|distinct(a')\right|=i \\\
;;; \left|distinct(b')\right|=j \end{array} \right|
;;; }
;;; {\left| \begin{array}{c} \text{candidates for} \\\
;;; a', b' \end{array} \right|} $$
;; **

;; **
;;; Now select the distinct number sets, times the number of sequences containing only those elements. At this point we've disentangled the pairs and can count @@ a' @@ and @@ b' @@ separately.
;;; 
;;; $$ = \frac{
;;; \left| \begin{array}{c} \text{candidates for} \\\
;;; a' \cap b' \\\
;;; \text{such that} \\\
;;; \left| a' \cap b' \right| = m \end{array} \right|
;;; \sum \limits\_{i=m}^{w}
;;; \sum \limits\_{j=m}^{w}
;;; \left| \begin{array}{c} \text{candidates for} \\\ 
;;; distinct(a'), distinct(b') \\\
;;; \text{given} \\\
;;; a' \cap b' \\\
;;; \text{such that } \\\ \left|distinct(a')\right|=i, \\\
;;; \left|distinct(b')\right|=j \end{array} \right|
;;; \left| \begin{array}{c} \text{candidates for} \\\ 
;;; a' \\\
;;; \text{given} \\\
;;; distinct(a') \end{array} \right|
;;; \left| \begin{array}{c} \text{candidates for} \\\
;;; b' \\\
;;; \text{given} \\\
;;; distinct(b') \end{array} \right|
;;; }
;;; {\left| \begin{array}{c} \text{candidates for} \\\
;;; a', b' \end{array} \right|} $$
;; **

;; **
;;; Let's solve the easy parts.
;;; 
;;; $$ = \frac{
;;; {n \choose m} 
;;; \sum \limits\_{i=m}^{w}
;;; \sum \limits\_{j=m}^{w}
;;; {n - m \choose i - m} {n - i \choose j - m}
;;; \left| \begin{array}{c} \text{candidates for} \\\ 
;;; a' \\\
;;; \text{given} \\\
;;; distinct(a') \end{array} \right|
;;; \left| \begin{array}{c} \text{candidates for} \\\
;;; b' \\\
;;; \text{given} \\\
;;; distinct(b') \end{array} \right|
;;; }
;;; {n^w n^w} $$
;; **

;; **
;;; **Subproblem: How many w-length sequences have a given set of distinct numbers? **
;;; 
;;; This one is fun.
;;; 
;;; The answer depends on the count of distinct numbers, not the distinct numbers themselves, so let's use set @@ \\{1, 2, \ldots , i\\} @@ to make this discussion easier.
;;; 
;;; $$
;;; \left| \begin{array}{c} \text{candidates for} \\\
;;; a' \\\
;;; \text{given} \\\
;;; distinct(a') \end{array} \right| =
;;; \left| \begin{array}{c} w \text{-length sequences of} \\\
;;; \\{1, 2, \ldots , i\\} \end{array} \right| -
;;; \left| \begin{array}{c} w \text{-length sequences of} \\\
;;; \\{1, 2, \ldots , i\\} \\\
;;; \text{that don't contain every number} \end{array} \right|
;;; $$
;; **

;; **
;;; Break this up into "numbers that aren't contained" cases.
;;; 
;;; $$ \begin{aligned} = \left| \begin{array}{c} w \text{-length sequences of} \\\
;;; \\{1, 2, \ldots , i\\} \end{array} \right| & - (\left| \text{exclude } 1 \right| + \left| \text{exclude } 2 \right| + \cdots + \left| \text{exclude } i \right|) \\\
;;; & + (\left| \text{exclude } 1,2 \right| + \left| \text{exclude } 1,3 \right| + \cdots + \left| \text{exclude } i-1,i \right|) \\\
;;; & - (\left| \text{exclude } 1,2,3 \right| + \left| \text{exclude } 1,2,4 \right| + \cdots + \left| \text{exclude } i-2,i-1,i \right|) \\\
;;; & + (\left| \text{exclude } 1,2,3,4 \right| + \left| \text{exclude } 1,2,3,5 \right| + \cdots + \left| \text{exclude } i-3,i-2,i-1,i \right|) \\\
;;; & - \cdots \\\
;;; & \pm \left| \text{exclude } 1,2, \ldots ,i \right| \end{aligned}
;;; $$
;; **

;; **
;;; Here's how to think of this:
;;; 
;;; 1. Remove all sequences that are missing a number.
;;; 2. Compensate for everything that got over-removed. i.e. all sequences missing two numbers.
;;; 3. Compensate for everything that got over-re-added, i.e. all sequences missing three numbers.  
;;; ...  
;;; 
;;; 
;;; This process goes on until you reach "all sequences missing @@ i @@ numbers". This set is empty, so the recursive compensation can stop.
;;; 
;;; $$
;;; = i^w - {i \choose 1} (i-1)^w + {i \choose 2} (i-2)^w - {i \choose 3}(i-3)^w + {i \choose 4}(i-4)^w - \cdots
;;; $$
;; **

;; **
;;; Now, capture it.
;;; 
;;; $$
;;; = i^w - \sum\limits_{k=1}^i (-1)^{k-1} {i \choose k} (i - k)^w
;;; $$
;; **

;; **
;;; **Bring everything together**
;;; 
;;; $$ = \frac{
;;; {n \choose m} 
;;; \sum \limits\_{i=m}^{w}
;;; \sum \limits\_{j=m}^{w}
;;; {n - m \choose i - m} {n - i \choose j - m}
;;; \Big( i^w - \sum\limits\_{k=1}^i (-1)^{k-1} {i \choose k} (i - k)^w \Big)
;;; \Big( j^w - \sum\limits\_{k=1}^j (-1)^{k-1} {j \choose k} (j - k)^w \Big)
;;; }
;;; {n^w n^w} $$
;; **

;; **
;;; **Test it**
;;; 
;;; Check our math and our hashing strategy.
;;; 
;;; Plug in the numbers.
;; **

;; @@
;; Don't use Math/pow. It does bad things to large integers.
(defn pow [x n]
  (loop [r (bigint 1)
         n n]
    (if (zero? n)
      r
      (recur (* r x) (dec n)))))

(defn choose [n k]
  (/ (reduce * (bigint 1) (range (inc (- n k)) (inc n)))
     (reduce * (bigint 1) (range 1 (inc k)))))

(defn count-sequences-with-distinct
  [n-distinct w]
  (- (pow n-distinct w)
     (reduce +
             (for [k (range 1 (inc n-distinct))]
               (* (pow -1 (dec k))
                  (choose n-distinct k)
                  (pow (- n-distinct k) w))))))

(defn probability-of-intersection
  [n w m]
  (/ (* (choose n m)
        (reduce +
                (for [i (range m (inc w))
                      j (range m (inc w))]
                  (* (choose (- n m) (- i m))
                     (choose (- n i) (- j m))
                     (count-sequences-with-distinct i w)
                     (count-sequences-with-distinct j w)))))
     (pow n (+ w w))))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;comportexviz.notebook.hotgym/probability-of-intersection</span>","value":"#'comportexviz.notebook.hotgym/probability-of-intersection"}
;; <=

;; @@
(let [n 100
      w 5]
  (println "Given two non-intersecting numbers... (based on math)")
  (doseq [i (range 0 (inc w))]
    (println (format "P(intersection = %d) = %.6f"
                     i
                     (float (probability-of-intersection n w i))))))
;; @@
;; ->
;;; Given two non-intersecting numbers... (based on math)
;;; P(intersection = 0) = 0.777903
;;; P(intersection = 1) = 0.204533
;;; P(intersection = 2) = 0.017032
;;; P(intersection = 3) = 0.000526
;;; P(intersection = 4) = 0.000005
;;; P(intersection = 5) = 0.000000
;;; 
;; <-
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; **
;;; Now run some experiments.
;; **

;; @@
(defn sample-pair-overlaps
  [encoder pair-distance n-trials]
  (let [rng (random/rng 42)]
    (->> (repeatedly #(random/int rng))
         (map (fn [v1]
                [v1 (+ v1 pair-distance)]))
         (map (fn [pair]
                (->> pair
                     (map #(p/encode encoder [%]))
                     (map (partial into #{}))
                     (apply set/intersection)
                     count)))
         (take n-trials))))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;comportexviz.notebook.hotgym/sample-pair-overlaps</span>","value":"#'comportexviz.notebook.hotgym/sample-pair-overlaps"}
;; <=

;; @@
(let [radius 2
      n-active-bits (inc (* 2 radius))
      n-trials 100000
      enc (e/coordinate-encoder [100] n-active-bits [1] [radius])
      freqs (frequencies (sample-pair-overlaps enc n-active-bits n-trials))]
  (println (format "Given two non-intersecting numbers... (based on %d trials)" n-trials))
  (doseq [i (range (inc n-active-bits))]
    (println (format "P(intersection = %d) = %.6f"
                     i
                     (-> (get freqs i 0)
                         (/ n-trials)
                         float)))))
;; @@
;; ->
;;; Given two non-intersecting numbers... (based on 100000 trials)
;;; P(intersection = 0) = 0.777460
;;; P(intersection = 1) = 0.204840
;;; P(intersection = 2) = 0.017210
;;; P(intersection = 3) = 0.000490
;;; P(intersection = 4) = 0.000000
;;; P(intersection = 5) = 0.000000
;;; 
;; <-
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; **
;;; I'm convinced. The math correctly predicts the outcome.
;; **

;; **
;;; #### That took a while
;;; 
;;; We've shown two ways of getting these numbers:
;;; 
;;; - Math (combinatorics)
;;; - Trials
;;; 
;;; From here on out, we'll just use trials. The math is interesting and useful, but not essential. I'm hoping to come back and finish it some day, but feel free to beat me to it.
;; **

;; **
;;; #### <a name="doing_the_trials" /> Doing the trials
;;; 
;;; So far we've assumed that the pre-hash overlap @@ a \cap b @@ is @@ 0 @@, i.e. that @@ distance > 2 * radius @@. Let's remove that assumption.
;; **

;; @@
(def enc-example1
  (e/coordinate-encoder [1000] 10 [1] [50]))

(defn plot-overlap-probabilities
  [encoder distance]
  (let [n-trials 1000
        freqs (frequencies (sample-pair-overlaps encoder distance n-trials))
        categories (range (inc (apply max (keys freqs))))]
    (plot/bar-chart categories (->> categories
                                    (map #(get freqs % 0))
                                    (map #(/ % n-trials)))
                    :plot-range [categories [0 1]])))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;comportexviz.notebook.hotgym/plot-overlap-probabilities</span>","value":"#'comportexviz.notebook.hotgym/plot-overlap-probabilities"}
;; <=

;; **
;;; When sampling 10 bits from a radius of 50, the distribution of overlaps of pairs separated by 30 is:
;; **

;; @@
(plot-overlap-probabilities enc-example1 30)
;; @@
;; =>
;;; {"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"top":10,"left":50,"bottom":20,"right":10},"data":[{"name":"c494a199-703e-43ac-a4de-496ea33da0b3","values":[{"x":0,"y":0},{"x":1,"y":0},{"x":2,"y":0.002},{"x":3,"y":0.014},{"x":4,"y":0.061},{"x":5,"y":0.155},{"x":6,"y":0.283},{"x":7,"y":0.302},{"x":8,"y":0.151},{"x":9,"y":0.031},{"x":10,"y":0.001}]}],"marks":[{"type":"rect","from":{"data":"c494a199-703e-43ac-a4de-496ea33da0b3"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"width":{"scale":"x","band":true,"offset":-1},"y":{"scale":"y","field":"data.y"},"y2":{"scale":"y","value":0}},"update":{"fill":{"value":"steelblue"},"opacity":{"value":1}},"hover":{"fill":{"value":"#FF29D2"}}}}],"scales":[{"name":"x","type":"ordinal","range":"width","domain":[0,1,2,3,4,5,6,7,8,9,10]},{"name":"y","range":"height","nice":true,"domain":[0,1]}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"c494a199-703e-43ac-a4de-496ea33da0b3\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 1/500} {:x 3, :y 7/500} {:x 4, :y 61/1000} {:x 5, :y 31/200} {:x 6, :y 283/1000} {:x 7, :y 151/500} {:x 8, :y 151/1000} {:x 9, :y 31/1000} {:x 10, :y 1/1000})}], :marks [{:type \"rect\", :from {:data \"c494a199-703e-43ac-a4de-496ea33da0b3\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}}"}
;; <=

;; **
;;; For 1, it's:
;; **

;; @@
(plot-overlap-probabilities enc-example1 1)
;; @@
;; =>
;;; {"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"top":10,"left":50,"bottom":20,"right":10},"data":[{"name":"aec48fd8-86ad-41f1-9063-29201185756e","values":[{"x":0,"y":0},{"x":1,"y":0},{"x":2,"y":0},{"x":3,"y":0},{"x":4,"y":0},{"x":5,"y":0},{"x":6,"y":0},{"x":7,"y":0},{"x":8,"y":0.007},{"x":9,"y":0.209},{"x":10,"y":0.784}]}],"marks":[{"type":"rect","from":{"data":"aec48fd8-86ad-41f1-9063-29201185756e"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"width":{"scale":"x","band":true,"offset":-1},"y":{"scale":"y","field":"data.y"},"y2":{"scale":"y","value":0}},"update":{"fill":{"value":"steelblue"},"opacity":{"value":1}},"hover":{"fill":{"value":"#FF29D2"}}}}],"scales":[{"name":"x","type":"ordinal","range":"width","domain":[0,1,2,3,4,5,6,7,8,9,10]},{"name":"y","range":"height","nice":true,"domain":[0,1]}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"aec48fd8-86ad-41f1-9063-29201185756e\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 0} {:x 3, :y 0} {:x 4, :y 0} {:x 5, :y 0} {:x 6, :y 0} {:x 7, :y 0} {:x 8, :y 7/1000} {:x 9, :y 209/1000} {:x 10, :y 98/125})}], :marks [{:type \"rect\", :from {:data \"aec48fd8-86ad-41f1-9063-29201185756e\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}}"}
;; <=

;; **
;;; When it's outside the radius, it's:
;; **

;; @@
(plot-overlap-probabilities enc-example1 101)
;; @@
;; =>
;;; {"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"top":10,"left":50,"bottom":20,"right":10},"data":[{"name":"8f59d376-a907-43db-9407-3f9d60621bf1","values":[{"x":0,"y":0.903},{"x":1,"y":0.094},{"x":2,"y":0.003}]}],"marks":[{"type":"rect","from":{"data":"8f59d376-a907-43db-9407-3f9d60621bf1"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"width":{"scale":"x","band":true,"offset":-1},"y":{"scale":"y","field":"data.y"},"y2":{"scale":"y","value":0}},"update":{"fill":{"value":"steelblue"},"opacity":{"value":1}},"hover":{"fill":{"value":"#FF29D2"}}}}],"scales":[{"name":"x","type":"ordinal","range":"width","domain":[0,1,2]},{"name":"y","range":"height","nice":true,"domain":[0,1]}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"8f59d376-a907-43db-9407-3f9d60621bf1\", :values ({:x 0, :y 903/1000} {:x 1, :y 47/500} {:x 2, :y 3/1000})}], :marks [{:type \"rect\", :from {:data \"8f59d376-a907-43db-9407-3f9d60621bf1\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}}"}
;; <=

;; **
;;; When we're within the radius, increasing the number of bits doesn't dramatically affect the distribution:
;; **

;; @@
(for [n-bits [100 1000 10000]]
  (plot-overlap-probabilities (e/coordinate-encoder [n-bits] 10 [1] [50]) 30))
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"top":10,"left":50,"bottom":20,"right":10},"data":[{"name":"06e19992-5e59-4beb-ab6b-b47bfa964f8f","values":[{"x":0,"y":0},{"x":1,"y":0},{"x":2,"y":0.002},{"x":3,"y":0.013},{"x":4,"y":0.063},{"x":5,"y":0.161},{"x":6,"y":0.297},{"x":7,"y":0.29},{"x":8,"y":0.149},{"x":9,"y":0.024},{"x":10,"y":0.001}]}],"marks":[{"type":"rect","from":{"data":"06e19992-5e59-4beb-ab6b-b47bfa964f8f"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"width":{"scale":"x","band":true,"offset":-1},"y":{"scale":"y","field":"data.y"},"y2":{"scale":"y","value":0}},"update":{"fill":{"value":"steelblue"},"opacity":{"value":1}},"hover":{"fill":{"value":"#FF29D2"}}}}],"scales":[{"name":"x","type":"ordinal","range":"width","domain":[0,1,2,3,4,5,6,7,8,9,10]},{"name":"y","range":"height","nice":true,"domain":[0,1]}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"06e19992-5e59-4beb-ab6b-b47bfa964f8f\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 1/500} {:x 3, :y 13/1000} {:x 4, :y 63/1000} {:x 5, :y 161/1000} {:x 6, :y 297/1000} {:x 7, :y 29/100} {:x 8, :y 149/1000} {:x 9, :y 3/125} {:x 10, :y 1/1000})}], :marks [{:type \"rect\", :from {:data \"06e19992-5e59-4beb-ab6b-b47bfa964f8f\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}}"},{"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"top":10,"left":50,"bottom":20,"right":10},"data":[{"name":"46a7a732-83ac-4aea-a46c-262ec3db44c1","values":[{"x":0,"y":0},{"x":1,"y":0},{"x":2,"y":0.002},{"x":3,"y":0.014},{"x":4,"y":0.061},{"x":5,"y":0.155},{"x":6,"y":0.283},{"x":7,"y":0.302},{"x":8,"y":0.151},{"x":9,"y":0.031},{"x":10,"y":0.001}]}],"marks":[{"type":"rect","from":{"data":"46a7a732-83ac-4aea-a46c-262ec3db44c1"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"width":{"scale":"x","band":true,"offset":-1},"y":{"scale":"y","field":"data.y"},"y2":{"scale":"y","value":0}},"update":{"fill":{"value":"steelblue"},"opacity":{"value":1}},"hover":{"fill":{"value":"#FF29D2"}}}}],"scales":[{"name":"x","type":"ordinal","range":"width","domain":[0,1,2,3,4,5,6,7,8,9,10]},{"name":"y","range":"height","nice":true,"domain":[0,1]}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"46a7a732-83ac-4aea-a46c-262ec3db44c1\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 1/500} {:x 3, :y 7/500} {:x 4, :y 61/1000} {:x 5, :y 31/200} {:x 6, :y 283/1000} {:x 7, :y 151/500} {:x 8, :y 151/1000} {:x 9, :y 31/1000} {:x 10, :y 1/1000})}], :marks [{:type \"rect\", :from {:data \"46a7a732-83ac-4aea-a46c-262ec3db44c1\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}}"},{"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"top":10,"left":50,"bottom":20,"right":10},"data":[{"name":"eb07c221-3294-43e7-bdfc-ea699f175063","values":[{"x":0,"y":0},{"x":1,"y":0},{"x":2,"y":0.002},{"x":3,"y":0.016},{"x":4,"y":0.058},{"x":5,"y":0.156},{"x":6,"y":0.281},{"x":7,"y":0.303},{"x":8,"y":0.152},{"x":9,"y":0.031},{"x":10,"y":0.001}]}],"marks":[{"type":"rect","from":{"data":"eb07c221-3294-43e7-bdfc-ea699f175063"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"width":{"scale":"x","band":true,"offset":-1},"y":{"scale":"y","field":"data.y"},"y2":{"scale":"y","value":0}},"update":{"fill":{"value":"steelblue"},"opacity":{"value":1}},"hover":{"fill":{"value":"#FF29D2"}}}}],"scales":[{"name":"x","type":"ordinal","range":"width","domain":[0,1,2,3,4,5,6,7,8,9,10]},{"name":"y","range":"height","nice":true,"domain":[0,1]}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"eb07c221-3294-43e7-bdfc-ea699f175063\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 1/500} {:x 3, :y 2/125} {:x 4, :y 29/500} {:x 5, :y 39/250} {:x 6, :y 281/1000} {:x 7, :y 303/1000} {:x 8, :y 19/125} {:x 9, :y 31/1000} {:x 10, :y 1/1000})}], :marks [{:type \"rect\", :from {:data \"eb07c221-3294-43e7-bdfc-ea699f175063\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}}"}],"value":"(#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"06e19992-5e59-4beb-ab6b-b47bfa964f8f\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 1/500} {:x 3, :y 13/1000} {:x 4, :y 63/1000} {:x 5, :y 161/1000} {:x 6, :y 297/1000} {:x 7, :y 29/100} {:x 8, :y 149/1000} {:x 9, :y 3/125} {:x 10, :y 1/1000})}], :marks [{:type \"rect\", :from {:data \"06e19992-5e59-4beb-ab6b-b47bfa964f8f\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}} #gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"46a7a732-83ac-4aea-a46c-262ec3db44c1\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 1/500} {:x 3, :y 7/500} {:x 4, :y 61/1000} {:x 5, :y 31/200} {:x 6, :y 283/1000} {:x 7, :y 151/500} {:x 8, :y 151/1000} {:x 9, :y 31/1000} {:x 10, :y 1/1000})}], :marks [{:type \"rect\", :from {:data \"46a7a732-83ac-4aea-a46c-262ec3db44c1\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}} #gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"eb07c221-3294-43e7-bdfc-ea699f175063\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 1/500} {:x 3, :y 2/125} {:x 4, :y 29/500} {:x 5, :y 39/250} {:x 6, :y 281/1000} {:x 7, :y 303/1000} {:x 8, :y 19/125} {:x 9, :y 31/1000} {:x 10, :y 1/1000})}], :marks [{:type \"rect\", :from {:data \"eb07c221-3294-43e7-bdfc-ea699f175063\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}})"}
;; <=

;; **
;;; But when we're outside the radius, it does. It has a huge effect on collisions, as expected:
;; **

;; @@
(for [n-bits [100 1000 10000]]
  (plot-overlap-probabilities (e/coordinate-encoder [n-bits] 10 [1] [50]) 101))
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"top":10,"left":50,"bottom":20,"right":10},"data":[{"name":"5f0aef1b-9299-4eea-a98c-59f2173a14c7","values":[{"x":0,"y":0.395},{"x":1,"y":0.37},{"x":2,"y":0.185},{"x":3,"y":0.044},{"x":4,"y":0.006}]}],"marks":[{"type":"rect","from":{"data":"5f0aef1b-9299-4eea-a98c-59f2173a14c7"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"width":{"scale":"x","band":true,"offset":-1},"y":{"scale":"y","field":"data.y"},"y2":{"scale":"y","value":0}},"update":{"fill":{"value":"steelblue"},"opacity":{"value":1}},"hover":{"fill":{"value":"#FF29D2"}}}}],"scales":[{"name":"x","type":"ordinal","range":"width","domain":[0,1,2,3,4]},{"name":"y","range":"height","nice":true,"domain":[0,1]}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"5f0aef1b-9299-4eea-a98c-59f2173a14c7\", :values ({:x 0, :y 79/200} {:x 1, :y 37/100} {:x 2, :y 37/200} {:x 3, :y 11/250} {:x 4, :y 3/500})}], :marks [{:type \"rect\", :from {:data \"5f0aef1b-9299-4eea-a98c-59f2173a14c7\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}}"},{"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"top":10,"left":50,"bottom":20,"right":10},"data":[{"name":"018d7a63-d8a9-4475-9566-38523cf269fe","values":[{"x":0,"y":0.903},{"x":1,"y":0.094},{"x":2,"y":0.003}]}],"marks":[{"type":"rect","from":{"data":"018d7a63-d8a9-4475-9566-38523cf269fe"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"width":{"scale":"x","band":true,"offset":-1},"y":{"scale":"y","field":"data.y"},"y2":{"scale":"y","value":0}},"update":{"fill":{"value":"steelblue"},"opacity":{"value":1}},"hover":{"fill":{"value":"#FF29D2"}}}}],"scales":[{"name":"x","type":"ordinal","range":"width","domain":[0,1,2]},{"name":"y","range":"height","nice":true,"domain":[0,1]}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"018d7a63-d8a9-4475-9566-38523cf269fe\", :values ({:x 0, :y 903/1000} {:x 1, :y 47/500} {:x 2, :y 3/1000})}], :marks [{:type \"rect\", :from {:data \"018d7a63-d8a9-4475-9566-38523cf269fe\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}}"},{"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"top":10,"left":50,"bottom":20,"right":10},"data":[{"name":"0edcf21b-e1e3-4182-a224-7d3ab3641bec","values":[{"x":0,"y":0.988},{"x":1,"y":0.012}]}],"marks":[{"type":"rect","from":{"data":"0edcf21b-e1e3-4182-a224-7d3ab3641bec"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"width":{"scale":"x","band":true,"offset":-1},"y":{"scale":"y","field":"data.y"},"y2":{"scale":"y","value":0}},"update":{"fill":{"value":"steelblue"},"opacity":{"value":1}},"hover":{"fill":{"value":"#FF29D2"}}}}],"scales":[{"name":"x","type":"ordinal","range":"width","domain":[0,1]},{"name":"y","range":"height","nice":true,"domain":[0,1]}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"0edcf21b-e1e3-4182-a224-7d3ab3641bec\", :values ({:x 0, :y 247/250} {:x 1, :y 3/250})}], :marks [{:type \"rect\", :from {:data \"0edcf21b-e1e3-4182-a224-7d3ab3641bec\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}}"}],"value":"(#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"5f0aef1b-9299-4eea-a98c-59f2173a14c7\", :values ({:x 0, :y 79/200} {:x 1, :y 37/100} {:x 2, :y 37/200} {:x 3, :y 11/250} {:x 4, :y 3/500})}], :marks [{:type \"rect\", :from {:data \"5f0aef1b-9299-4eea-a98c-59f2173a14c7\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}} #gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"018d7a63-d8a9-4475-9566-38523cf269fe\", :values ({:x 0, :y 903/1000} {:x 1, :y 47/500} {:x 2, :y 3/1000})}], :marks [{:type \"rect\", :from {:data \"018d7a63-d8a9-4475-9566-38523cf269fe\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}} #gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"0edcf21b-e1e3-4182-a224-7d3ab3641bec\", :values ({:x 0, :y 247/250} {:x 1, :y 3/250})}], :marks [{:type \"rect\", :from {:data \"0edcf21b-e1e3-4182-a224-7d3ab3641bec\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}})"}
;; <=

;; **
;;; As the number of active bits increases, collisions become the norm.
;; **

;; @@
(for [n-active-bits [10 20 50 100]]
  (plot-overlap-probabilities (e/coordinate-encoder [1000] n-active-bits [1] [50]) 101))
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"top":10,"left":50,"bottom":20,"right":10},"data":[{"name":"56eb83ad-f9db-4526-875c-072e868676e3","values":[{"x":0,"y":0.903},{"x":1,"y":0.094},{"x":2,"y":0.003}]}],"marks":[{"type":"rect","from":{"data":"56eb83ad-f9db-4526-875c-072e868676e3"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"width":{"scale":"x","band":true,"offset":-1},"y":{"scale":"y","field":"data.y"},"y2":{"scale":"y","value":0}},"update":{"fill":{"value":"steelblue"},"opacity":{"value":1}},"hover":{"fill":{"value":"#FF29D2"}}}}],"scales":[{"name":"x","type":"ordinal","range":"width","domain":[0,1,2]},{"name":"y","range":"height","nice":true,"domain":[0,1]}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"56eb83ad-f9db-4526-875c-072e868676e3\", :values ({:x 0, :y 903/1000} {:x 1, :y 47/500} {:x 2, :y 3/1000})}], :marks [{:type \"rect\", :from {:data \"56eb83ad-f9db-4526-875c-072e868676e3\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}}"},{"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"top":10,"left":50,"bottom":20,"right":10},"data":[{"name":"021e5b07-b365-42aa-a9e7-b4dac829da77","values":[{"x":0,"y":0.679},{"x":1,"y":0.258},{"x":2,"y":0.06},{"x":3,"y":0.003}]}],"marks":[{"type":"rect","from":{"data":"021e5b07-b365-42aa-a9e7-b4dac829da77"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"width":{"scale":"x","band":true,"offset":-1},"y":{"scale":"y","field":"data.y"},"y2":{"scale":"y","value":0}},"update":{"fill":{"value":"steelblue"},"opacity":{"value":1}},"hover":{"fill":{"value":"#FF29D2"}}}}],"scales":[{"name":"x","type":"ordinal","range":"width","domain":[0,1,2,3]},{"name":"y","range":"height","nice":true,"domain":[0,1]}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"021e5b07-b365-42aa-a9e7-b4dac829da77\", :values ({:x 0, :y 679/1000} {:x 1, :y 129/500} {:x 2, :y 3/50} {:x 3, :y 3/1000})}], :marks [{:type \"rect\", :from {:data \"021e5b07-b365-42aa-a9e7-b4dac829da77\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}}"},{"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"top":10,"left":50,"bottom":20,"right":10},"data":[{"name":"c4a1d89a-5f12-4eed-8fd1-93fcf8a0e555","values":[{"x":0,"y":0.088},{"x":1,"y":0.222},{"x":2,"y":0.26},{"x":3,"y":0.214},{"x":4,"y":0.137},{"x":5,"y":0.057},{"x":6,"y":0.016},{"x":7,"y":0.006}]}],"marks":[{"type":"rect","from":{"data":"c4a1d89a-5f12-4eed-8fd1-93fcf8a0e555"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"width":{"scale":"x","band":true,"offset":-1},"y":{"scale":"y","field":"data.y"},"y2":{"scale":"y","value":0}},"update":{"fill":{"value":"steelblue"},"opacity":{"value":1}},"hover":{"fill":{"value":"#FF29D2"}}}}],"scales":[{"name":"x","type":"ordinal","range":"width","domain":[0,1,2,3,4,5,6,7]},{"name":"y","range":"height","nice":true,"domain":[0,1]}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"c4a1d89a-5f12-4eed-8fd1-93fcf8a0e555\", :values ({:x 0, :y 11/125} {:x 1, :y 111/500} {:x 2, :y 13/50} {:x 3, :y 107/500} {:x 4, :y 137/1000} {:x 5, :y 57/1000} {:x 6, :y 2/125} {:x 7, :y 3/500})}], :marks [{:type \"rect\", :from {:data \"c4a1d89a-5f12-4eed-8fd1-93fcf8a0e555\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}}"},{"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"top":10,"left":50,"bottom":20,"right":10},"data":[{"name":"56ae2e10-ca66-4aaf-a1dd-039a76314119","values":[{"x":0,"y":0},{"x":1,"y":0},{"x":2,"y":0.004},{"x":3,"y":0.015},{"x":4,"y":0.019},{"x":5,"y":0.038},{"x":6,"y":0.086},{"x":7,"y":0.118},{"x":8,"y":0.148},{"x":9,"y":0.132},{"x":10,"y":0.145},{"x":11,"y":0.107},{"x":12,"y":0.077},{"x":13,"y":0.057},{"x":14,"y":0.031},{"x":15,"y":0.01},{"x":16,"y":0.007},{"x":17,"y":0.003},{"x":18,"y":0.002},{"x":19,"y":0.001}]}],"marks":[{"type":"rect","from":{"data":"56ae2e10-ca66-4aaf-a1dd-039a76314119"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"width":{"scale":"x","band":true,"offset":-1},"y":{"scale":"y","field":"data.y"},"y2":{"scale":"y","value":0}},"update":{"fill":{"value":"steelblue"},"opacity":{"value":1}},"hover":{"fill":{"value":"#FF29D2"}}}}],"scales":[{"name":"x","type":"ordinal","range":"width","domain":[0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19]},{"name":"y","range":"height","nice":true,"domain":[0,1]}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"56ae2e10-ca66-4aaf-a1dd-039a76314119\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 1/250} {:x 3, :y 3/200} {:x 4, :y 19/1000} {:x 5, :y 19/500} {:x 6, :y 43/500} {:x 7, :y 59/500} {:x 8, :y 37/250} {:x 9, :y 33/250} {:x 10, :y 29/200} {:x 11, :y 107/1000} {:x 12, :y 77/1000} {:x 13, :y 57/1000} {:x 14, :y 31/1000} {:x 15, :y 1/100} {:x 16, :y 7/1000} {:x 17, :y 3/1000} {:x 18, :y 1/500} {:x 19, :y 1/1000})}], :marks [{:type \"rect\", :from {:data \"56ae2e10-ca66-4aaf-a1dd-039a76314119\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}}"}],"value":"(#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"56eb83ad-f9db-4526-875c-072e868676e3\", :values ({:x 0, :y 903/1000} {:x 1, :y 47/500} {:x 2, :y 3/1000})}], :marks [{:type \"rect\", :from {:data \"56eb83ad-f9db-4526-875c-072e868676e3\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}} #gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"021e5b07-b365-42aa-a9e7-b4dac829da77\", :values ({:x 0, :y 679/1000} {:x 1, :y 129/500} {:x 2, :y 3/50} {:x 3, :y 3/1000})}], :marks [{:type \"rect\", :from {:data \"021e5b07-b365-42aa-a9e7-b4dac829da77\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}} #gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"c4a1d89a-5f12-4eed-8fd1-93fcf8a0e555\", :values ({:x 0, :y 11/125} {:x 1, :y 111/500} {:x 2, :y 13/50} {:x 3, :y 107/500} {:x 4, :y 137/1000} {:x 5, :y 57/1000} {:x 6, :y 2/125} {:x 7, :y 3/500})}], :marks [{:type \"rect\", :from {:data \"c4a1d89a-5f12-4eed-8fd1-93fcf8a0e555\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}} #gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"56ae2e10-ca66-4aaf-a1dd-039a76314119\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 1/250} {:x 3, :y 3/200} {:x 4, :y 19/1000} {:x 5, :y 19/500} {:x 6, :y 43/500} {:x 7, :y 59/500} {:x 8, :y 37/250} {:x 9, :y 33/250} {:x 10, :y 29/200} {:x 11, :y 107/1000} {:x 12, :y 77/1000} {:x 13, :y 57/1000} {:x 14, :y 31/1000} {:x 15, :y 1/100} {:x 16, :y 7/1000} {:x 17, :y 3/1000} {:x 18, :y 1/500} {:x 19, :y 1/1000})}], :marks [{:type \"rect\", :from {:data \"56ae2e10-ca66-4aaf-a1dd-039a76314119\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}})"}
;; <=

;; **
;;; As radius increases, nearby pairs become closer and closer to being identical.
;; **

;; @@
(for [radius [50 100 200 400]]
  (plot-overlap-probabilities (e/coordinate-encoder [1000] 10 [1] [radius]) 30))
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"top":10,"left":50,"bottom":20,"right":10},"data":[{"name":"e121ed1c-8e57-4e39-ae85-fee38083e4ee","values":[{"x":0,"y":0},{"x":1,"y":0},{"x":2,"y":0.002},{"x":3,"y":0.014},{"x":4,"y":0.061},{"x":5,"y":0.155},{"x":6,"y":0.283},{"x":7,"y":0.302},{"x":8,"y":0.151},{"x":9,"y":0.031},{"x":10,"y":0.001}]}],"marks":[{"type":"rect","from":{"data":"e121ed1c-8e57-4e39-ae85-fee38083e4ee"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"width":{"scale":"x","band":true,"offset":-1},"y":{"scale":"y","field":"data.y"},"y2":{"scale":"y","value":0}},"update":{"fill":{"value":"steelblue"},"opacity":{"value":1}},"hover":{"fill":{"value":"#FF29D2"}}}}],"scales":[{"name":"x","type":"ordinal","range":"width","domain":[0,1,2,3,4,5,6,7,8,9,10]},{"name":"y","range":"height","nice":true,"domain":[0,1]}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"e121ed1c-8e57-4e39-ae85-fee38083e4ee\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 1/500} {:x 3, :y 7/500} {:x 4, :y 61/1000} {:x 5, :y 31/200} {:x 6, :y 283/1000} {:x 7, :y 151/500} {:x 8, :y 151/1000} {:x 9, :y 31/1000} {:x 10, :y 1/1000})}], :marks [{:type \"rect\", :from {:data \"e121ed1c-8e57-4e39-ae85-fee38083e4ee\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}}"},{"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"top":10,"left":50,"bottom":20,"right":10},"data":[{"name":"7b37eda1-2aa5-4268-8097-7d8595b93ecf","values":[{"x":0,"y":0},{"x":1,"y":0},{"x":2,"y":0},{"x":3,"y":0},{"x":4,"y":0},{"x":5,"y":0.008},{"x":6,"y":0.06},{"x":7,"y":0.239},{"x":8,"y":0.404},{"x":9,"y":0.242},{"x":10,"y":0.047}]}],"marks":[{"type":"rect","from":{"data":"7b37eda1-2aa5-4268-8097-7d8595b93ecf"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"width":{"scale":"x","band":true,"offset":-1},"y":{"scale":"y","field":"data.y"},"y2":{"scale":"y","value":0}},"update":{"fill":{"value":"steelblue"},"opacity":{"value":1}},"hover":{"fill":{"value":"#FF29D2"}}}}],"scales":[{"name":"x","type":"ordinal","range":"width","domain":[0,1,2,3,4,5,6,7,8,9,10]},{"name":"y","range":"height","nice":true,"domain":[0,1]}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"7b37eda1-2aa5-4268-8097-7d8595b93ecf\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 0} {:x 3, :y 0} {:x 4, :y 0} {:x 5, :y 1/125} {:x 6, :y 3/50} {:x 7, :y 239/1000} {:x 8, :y 101/250} {:x 9, :y 121/500} {:x 10, :y 47/1000})}], :marks [{:type \"rect\", :from {:data \"7b37eda1-2aa5-4268-8097-7d8595b93ecf\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}}"},{"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"top":10,"left":50,"bottom":20,"right":10},"data":[{"name":"a05a4fa2-a110-40a9-b648-c55f2ed30312","values":[{"x":0,"y":0},{"x":1,"y":0},{"x":2,"y":0},{"x":3,"y":0},{"x":4,"y":0.001},{"x":5,"y":0},{"x":6,"y":0.008},{"x":7,"y":0.07},{"x":8,"y":0.238},{"x":9,"y":0.464},{"x":10,"y":0.219}]}],"marks":[{"type":"rect","from":{"data":"a05a4fa2-a110-40a9-b648-c55f2ed30312"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"width":{"scale":"x","band":true,"offset":-1},"y":{"scale":"y","field":"data.y"},"y2":{"scale":"y","value":0}},"update":{"fill":{"value":"steelblue"},"opacity":{"value":1}},"hover":{"fill":{"value":"#FF29D2"}}}}],"scales":[{"name":"x","type":"ordinal","range":"width","domain":[0,1,2,3,4,5,6,7,8,9,10]},{"name":"y","range":"height","nice":true,"domain":[0,1]}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"a05a4fa2-a110-40a9-b648-c55f2ed30312\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 0} {:x 3, :y 0} {:x 4, :y 1/1000} {:x 5, :y 0} {:x 6, :y 1/125} {:x 7, :y 7/100} {:x 8, :y 119/500} {:x 9, :y 58/125} {:x 10, :y 219/1000})}], :marks [{:type \"rect\", :from {:data \"a05a4fa2-a110-40a9-b648-c55f2ed30312\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}}"},{"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"top":10,"left":50,"bottom":20,"right":10},"data":[{"name":"51dea5ff-3c0f-4838-83e8-c953396fb886","values":[{"x":0,"y":0},{"x":1,"y":0},{"x":2,"y":0},{"x":3,"y":0},{"x":4,"y":0},{"x":5,"y":0},{"x":6,"y":0},{"x":7,"y":0.012},{"x":8,"y":0.096},{"x":9,"y":0.414},{"x":10,"y":0.478}]}],"marks":[{"type":"rect","from":{"data":"51dea5ff-3c0f-4838-83e8-c953396fb886"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"width":{"scale":"x","band":true,"offset":-1},"y":{"scale":"y","field":"data.y"},"y2":{"scale":"y","value":0}},"update":{"fill":{"value":"steelblue"},"opacity":{"value":1}},"hover":{"fill":{"value":"#FF29D2"}}}}],"scales":[{"name":"x","type":"ordinal","range":"width","domain":[0,1,2,3,4,5,6,7,8,9,10]},{"name":"y","range":"height","nice":true,"domain":[0,1]}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"51dea5ff-3c0f-4838-83e8-c953396fb886\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 0} {:x 3, :y 0} {:x 4, :y 0} {:x 5, :y 0} {:x 6, :y 0} {:x 7, :y 3/250} {:x 8, :y 12/125} {:x 9, :y 207/500} {:x 10, :y 239/500})}], :marks [{:type \"rect\", :from {:data \"51dea5ff-3c0f-4838-83e8-c953396fb886\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}}"}],"value":"(#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"e121ed1c-8e57-4e39-ae85-fee38083e4ee\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 1/500} {:x 3, :y 7/500} {:x 4, :y 61/1000} {:x 5, :y 31/200} {:x 6, :y 283/1000} {:x 7, :y 151/500} {:x 8, :y 151/1000} {:x 9, :y 31/1000} {:x 10, :y 1/1000})}], :marks [{:type \"rect\", :from {:data \"e121ed1c-8e57-4e39-ae85-fee38083e4ee\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}} #gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"7b37eda1-2aa5-4268-8097-7d8595b93ecf\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 0} {:x 3, :y 0} {:x 4, :y 0} {:x 5, :y 1/125} {:x 6, :y 3/50} {:x 7, :y 239/1000} {:x 8, :y 101/250} {:x 9, :y 121/500} {:x 10, :y 47/1000})}], :marks [{:type \"rect\", :from {:data \"7b37eda1-2aa5-4268-8097-7d8595b93ecf\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}} #gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"a05a4fa2-a110-40a9-b648-c55f2ed30312\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 0} {:x 3, :y 0} {:x 4, :y 1/1000} {:x 5, :y 0} {:x 6, :y 1/125} {:x 7, :y 7/100} {:x 8, :y 119/500} {:x 9, :y 58/125} {:x 10, :y 219/1000})}], :marks [{:type \"rect\", :from {:data \"a05a4fa2-a110-40a9-b648-c55f2ed30312\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}} #gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"51dea5ff-3c0f-4838-83e8-c953396fb886\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 0} {:x 3, :y 0} {:x 4, :y 0} {:x 5, :y 0} {:x 6, :y 0} {:x 7, :y 3/250} {:x 8, :y 12/125} {:x 9, :y 207/500} {:x 10, :y 239/500})}], :marks [{:type \"rect\", :from {:data \"51dea5ff-3c0f-4838-83e8-c953396fb886\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}})"}
;; <=

;; **
;;; That raises a question: does this sampling cause nearby numbers to become indistinguishable?
;; **

;; @@
(let [enc (e/coordinate-encoder [1000] 10 [1] [50])]
  (doseq [v [43 44]]
    (println (format "Representation for %s:" v)
             (into #{} (p/encode enc [v])))))
;; @@
;; ->
;;; Representation for 43: #{204 420 792 565 697 535 667 928 628}
;;; Representation for 44: #{204 420 792 565 697 535 667 928 628}
;;; 
;; <-
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; **
;;; Yup.
;; **

;; **
;;; #### Whining
;; **

;; **
;;; So far, none of these are super-appealing.
;;; 
;;; - With a moving window, we have to decide between showing limited interrelations and having huge representations.
;;; - With a probabilistic encoding, we have to convince ourselves that everything will be okay.
;;; 
;;; Can we do better?
;; **

;; **
;;; #### A third way
;;; 
;;; I like sampling. I don't like unnecessary randomness. I'm okay with specifying a min and max.
;;; 
;;; We already stumbled on the answer. Remember the [alternate strategy](#alternate\_strategy) for coordinate encoder sampling? It's also compatible with a moving window.
;;; 
;;; Let's use a sampling moving window.
;; **

;; @@
(defn middle-out-range
  "By example:
  Given 7.2, returns (7, 8, 6, 9, 5, 10, ...),
  Given 7.7, returns (8, 7, 9, 6, 10, 5, ...)"
  [v]
  (let [start (-> v float Math/round long)
        rounded-down? (> v start)
        up (iterate inc start)
        down (iterate dec start)]
    (if rounded-down?
      (interleave down (drop 1 up))
      (interleave up (drop 1 down)))))

(defn multiples-within-radius
  [center radius multiples-of]
  (let [lower-bound (- center radius)
        upper-bound (+ center radius)]
    (->> (middle-out-range (/ center multiples-of))
         (map (partial * multiples-of))
         (take-while #(<= lower-bound
                          %
                          upper-bound)))))

(defn into-bounded
  "Move items from `from` to `coll` until its size reaches `max-size`
  or we run out of items. Specifically supports sets and maps, which don't
  always grow when an item is added."
  [coll max-size from]
  (loop [coll coll
         from from]
    (let [n-remaining (- max-size (count coll))]
      (if (and (pos? n-remaining)
               (not-empty from))
        (let [[taken untaken] (split-at n-remaining from)]
          (recur (into coll taken)
                 untaken))
        coll))))

(defn sampled-window
  "Place a bit in the center.
  Distribute bits around the center until we've used half of the remainder.
  Double the density. Distribute again until we've used half of the remainder.
  Double the density. ...
  Continue until all active bits are distributed or all bits are active.

  Strategically choose bit positions so that the intersections between
  various ranges will select the same bits."
  [center n-bits target-n-active radius]
  (loop [chosen #{center}
         density (/ (- target-n-active (count chosen))
                    (* 2 radius)
                    2)]
    (let [remaining (- target-n-active (count chosen))
          multiples-of (long (/ 1 density))]
      (if (and (pos? remaining)
               (pos? multiples-of))
        (let [half-remaining (quot remaining 2)
               n-take (if (or (odd? remaining)
                              (odd? half-remaining))
                        remaining
                        half-remaining)]
          (recur (->> (multiples-within-radius center radius multiples-of)
                      (filter #(<= 0 % (dec n-bits)))
                      (into-bounded chosen (+ n-take (count chosen))))
                 (* density 2)))
        chosen))))

(defrecord SampledLinearEncoder
  [topo n-active lower upper radius]
  p/PTopological
  (topology
    [_]
    topo)
  p/PEncoder
  (encode
    [_ x]
    (if x
      (let [n-bits (p/size topo)
            center (-> x
                       (max lower)
                       (min upper)
                       (- lower)
                       (/ (- upper lower))
                       (* n-bits)
                       long)]
        (sampled-window center n-bits n-active radius))
      (sequence nil)))
  (decode
    [this bit-votes n]
    (let [span (double (- upper lower))
          values (range lower upper (if (< 5 span 250)
                                      1
                                      (/ span 50)))]
      (->> (e/decode-by-brute-force this values bit-votes)
           (take n)))))

(defn sampled-linear-encoder
  "A linear encoder that samples the surrounding radius, rather than
  activating all of it. Sampling density decreases as distance increases.

  * `dimensions` is the size of the encoder in bits along one or more
    dimensions, a vector e.g. [500].

  * `n-active` is the number of bits to be active.

  * `[lower upper]` gives the numeric range to cover. The input number
    will be clamped to this range.
  
  * `radius` describes the range to sample.
  
  Recommendations:
  
  * `radius` should be a power of 2, e.g. 32, 256
  
  * `n-active` should be one above a power of 2, e.g. 33, 257
  
  * `lower` and `upper` should be `radius` below and above the actual
    lower and upper bounds. Otherwise the radius will extend off the
    number line, creating representations that behave a bit differently
    from the rest."
  [dimensions n-active [lower upper] radius]
  (let [topo (topology/make-topology dimensions)]
    (map->SampledLinearEncoder {:topo topo
                                :n-active n-active
                                :lower lower
                                :upper upper
                                :radius radius})))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;comportexviz.notebook.hotgym/sampled-linear-encoder</span>","value":"#'comportexviz.notebook.hotgym/sampled-linear-encoder"}
;; <=

;; @@
(for [i [0 1 2 3 41 42 43 44 127 128 129 130 200 201 202]]
  (viz (sampled-linear-encoder [256] 17 [0 256] 64) i
       {:display-mode :one-d}))
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA5klEQVRoQ+2aQQ6EIAxFgVN4Ks40cyOPoAeC9cyqJiomM2ykCQTMc0losPT/32+sFWPEnM/X+9VmCx/vl7TDTNOc7xg4JMZ4pS8ib5sthBBeKX3n3KJJv9tTFC+mCFFgTHFKmxByKeULdSm9MTBWemNtMPakU56EMeoCXw6LdtPCKQADX/rEGHXpsy4KihFCKWvLOBgDY2CsqlOCYlAMikGxet9iKAwKg8KgMCjMnf+VqCVqiVqilqglavk/7ZINt9AsaBY0C5oFzYJmQbNIc587G+iVv3OwWAWsAlYBq4BVwCoMaxU2gnMsYJ+0S6IAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA5ElEQVRoQ+3aSw7CIBAGYNpT9FScSW/kEeqB2rWuaKLWhSyKaIkxX5eEyRRm/kfIdCmEFB7fNcZzly1cYhzXHWEYTvmO7ZB5np9ZUkrHLluYpumwZun7fsx3bIdUnKXNj1VkcZYGDaMu72K/4sbahMALvNz1g76UyCu8wAu8lNPFP+GljSK3yaIuv8ljbaovi+rvrWJ6TI/psXKnBC/wAi/wst87DIbBMBgGw2CYkpdebIktsSW2xJbY8nUM5xtTN/SFvtAX+kJf6At9WWdYb2ggrx/P9HIX3AV3wV1wF9zFDu5iASolLGDJ1ytnAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA50lEQVRoQ+3aQQ6CMBCF4cIpOBVn0ht5BDwQrHVVExUT7aZOAqnmY9kw0Hbm/W8C7XJKOT2v6zieu2LgMo7TekcahlO3LMsrJOd8LAfmeT6sIX3fT+VDf/gtgeUHQvbJS2BigRBr+VYvgU0OhMiLvDwwXYlxNbaDVwY2ORBC+7RP+/Xo+ye9BHDRbIi8tMmxZgvGxBTM1sanxtSYGqtvruiFXuiFXrb7DIUwCIMwCIMwNb9r0RIt0RIt0RItPw/7FGd7mAWzYBbMglkwC2axHnu9q4FXvh8D1ipoFbQKWgWtglahjVbhBoJBLGAKHaE3AAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA2UlEQVRoQ+3aSw7CIBAGYOgpeirOpDfyCPVAdK0rTFRc2IUJSkPM1yXppND5X2knlhBKeF7XlM5xs3BJaal3hHk+xXVdXyWllON2Ied8qCXTNC0tJQ0bayjZ5ywNT3GWHTCmL2NyX1/05eEf/fwFxmAMxrpSTIYZM8M0SN+wJf+EsWFfso0h8oBeCZZgCZZdIxyKoRiKodhQ32GIElEiSkSJKHX690pgCSyBJbAElsC+D/v8YraHv/AX/sJf+At/4S91UvbOBvb69eSwdCFdSBfShXQhXXxKFzfaXSxgl+3jQAAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA1UlEQVRoQ+3aMRKCMBSE4ZBTcKqcSW/kEfBApNYqFhhnlCZiE/CjzPBmEh77v2XYoYRQwvO6p3QdVgu3lKZ6RxjHy5BzfpWUUs7rhXmeT7Ukxjgp+fqJ6Uuf75i+6MuCNuhrIf+R9PLnZ+n2+N1ubINTOtJZNhy/2xJ9MffN/XbbQy/0Qi/0svevJBzDMRxr51i3Bt7G+hSyvugLwALs3o0ijuEYjuEYjrX8FkRLtERLtERLtPyME0oPCk8us0F2VHRW2Lgmqc1Ks9KsfFMDq8Aq/GQVHnlILGD/Kp3fAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA3ElEQVRoQ+3asQ2DMBQEUJspmMozJRtlBDKQqZPKKQiREhpovizyKC0sy/98989wuaXU0vt5lnLPm4FHKdP6RhrHW57n+TOltXbdDtRaL+uUYRgmUw5XDC59njG4wGWRNtK3R/nPxBd7Ocr9mIrFrBJje860l5iKxawCl6Pch0ufFYMLXBYH676/x8HiC77gS28XXn7sn1kZo8lWccZ86+tN+bESK7ESK/1/6ef2SpNpMk2myTSZJv+mw4TBZOGW3iAKKAkpO7oGY/VKvVKv/GIDq8AqsAqcUpxRfAHRZCxgbcABNQAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA2klEQVRoQ+3aQQqDMBSE4egpPFXO1N6oR7AHStbtKi6shdaNBpQgn8vgA5Px/W8I05UQSvg87xif3WrhFeO4vBGG4dHlnL8lpZT7eiGldFtK+r4flew+Mbq0+Y/RhS4z2qBvC/mv1C/2ckLvVxxyRUmzHuZKe2n2kCs+jC4n9D5d2jxkutCF6T3U85svbbYYXegCfYeir8JdKGmzK+lCF7RES3fjW+7G0RIt0RIt0RIt/zM1IjQSRPNsEKCSH5O4W+KEZqVZaVb+dAOrwCqwCpwSo8goMor7jOIEKY8sYEVjTaUAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA0ElEQVRoQ+3asQ3DIBQEUPAUnoqZko0ygjMQ1ElFCodIiRunQELouUT+Bdy/+6evizWEGt7fM6V7PBw8UtraH2Fdb7GU8imptV6PBznnSytZlmVT8veLwWXMHoMLXHZpI31nlH8mvrgL7vfmvh7TY717bCYDPxNf4DIm9+ECl96aTMfG7DG4wKU392eaL+4yJl/gAhc6Zm1ra31ma00tqSW1pJbUklr+pl2EW2R79tkg2iTZJQvXgn5mpVlpVn6xgVVgFVgFTolRZBQZxRGN4gtCuCxgGsds3wAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA40lEQVRoQ+2ayw3DIAyGgSmYipmSjTpCMhCc05MjNaVSyqHIlawofDkiLMDwP8DxpRRx709EZt805Jyn2iOEsLQ9CDmnkIxxYA7EgBfoAoJFX14ZQF5xF99kgFXAKmAV/nBK4tzHfD9TWn3TsKW0VAVyMT64v3DjG/fCC1547jhOP3a0x46Cl2vihX1hX+Cxfhq/E15s1qIYxSZEUbJRTOyyoygmpgixyZhiYoOHsC/oPro/pu4PTn0s34D60BeDJHOSSTIi3i/i4AW8gBfwQsmmp2QDW8KWsCVsCVvClr/+t9wB7L4sYKuCBuwAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA3klEQVRoQ+2Zyw2AIBBEgSqsipq0I0vQguCspzXxlygXiHFj5HkkGMLMzuwANsYoZv9EpLPJQAihPWY454Z0Br9cIQQxCmZTDHrBLjBY+suKAO2VdHE3A6ICUYGo8CApiTFn+J69H20yMHk/HB3INE3P+YUTX70HXvTCdcdW/cTRnDiKXr6pF3iBF3ws38b/pBedvbBKqcPoIKbzYvWnveggprMKvJSqEl6+iRi81MyLDvusUlpj9JdSxKixmhFDLzWzj/Zhn1u4/Fs49IJe0At64e015+0Vt8QtX3bLBU0zLGBjioVgAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA4klEQVRoQ+2awQ3DIAxFgSkyFTO1G3WEZCA4tydHCiVSywX5gCx4OSIssJ3v/+3E55zFfR8RefpmIaX0qDtCCHu7A5PfEBIxXpiCGPBCuaDAwi9XBKBX1MV/MUAqIBWQCiglhCJCcYhQFOfuZvUT4+GbhXeMe1Vsbtte9PtMSNYdEIEXm+NB8kJeSlWire5pq2fCyxhfzJ6iuJjCRPFhaMwpiospTGbyReG+WRPyAu/D+/2yB7zYxIvZArv4xcALeIFf4BemCj1ThcXJAvchC8iinyzAC3gBL+AFcYW4svEn5AmtmSxgi2pyKwAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA2ElEQVRoQ+3ZQQ6CMBQE0NJTcKqeSW/kEfBAsNZVTURMtBu6IaR9LJsSwvyZ6fzfYVmWHD5Pzvk6FAvzPF+2HTHGqdzhlV8IIYYwq2LohV0wWOfLGwHHq3TxbwaigqggKkhKgqKgeEhQzCF8m9VnSvehWHikNG2JLYzjTb9vQtLvgIhezjkeVBd1WV1JW72nrW5JL8f8i6/UOgzEahFz+1aL2DEcUxd1aSNd0Estk2m/Z8Topefq035t9emlFjEcg1gb2RKTMRmTjaBN4PdM4Lklt+SWJ3PLF2ZDLGDqgvhuAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA1ElEQVRoQ+3asQ2DMBQEUJspmMozJRtlBDKQqZPKSCFECm5ojCz0KC1T8O/f+b6POM9zCd+nlHKP1ULO+bbtGIZhqnd45b+EKqZhVsbgC7kgsM6XTwUcr9zFXgxYBVaBVeCUGEVGkVFkFBlFPtktnEvI3aWsMcGYYEwwJhgTzhoTSgi/UOed0jNWC6+Ups2whnF8yMXOSBLh0mdcCxe4rGoo5jgSc+ALvuDLcbnAF3xpzZcr9ZhvwZfWfLnSv6NX4gtc+uQ+XODSWpPpWJ89BpcucVkAXDcsYFCcUZIAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA20lEQVRoQ+3awQ3DIAwFUMIUmYqZ2o06QjpQOLcnKjVNpSaXiEOFwssRgYTt+Pt/myHnXMLnK6Vch93CPM+XdUeMcdrvcOTXhTzmh1kyRr6ACwCrvrw9oLxiF1swQBVQBVQBU0IUEUVEEVFEFPFkXThNyE1TlkwgE8gEMoFMIBP6lgklhO+w7ZnSfdgtPFKaViERxvFmXvmPCa+4tDlGFxdxWdDQ+OnI+Em+yBf5chwuzpQvndvSrPnNXqzi8eSZbKkwv9kj4qLuq/t91v1mQaniYnAMjsGxwzj2ArydLGCld3E6AAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAMACAYAAAADiMb1AAAA3UlEQVRoQ+3aQQ6DIBAFUPQUnooztTfqEeyBcN2uMCm1SXWDG2L0sSQQwnz/8P+M3TRNOXxHzvnebSZSSrdlRd/343aFLf8hFDEfTGEMvkgXEqz35RMBzyt1sU4GpAKpQCpQSoQioUgoEoqEIp2sCqcIuSrKsglsApvAJrAJbMK1bUIO4ddse8f47DYTrxjHxUiEYXjoV7bo8MLlmG10uMClZEPtp5r2E77gC77Up4sz8cVd9nK/TcTanNLmT8gz3aVNxNqcApe93IfLMSMGF7gUBas+VuP48OXCfJkBFMgsYB2LnhAAAAAASUVORK5CYII=' />"}],"value":"(#object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x8352652 \"comportexviz.server.notebook$eval28199$fn$reify__28203@8352652\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x4df993b7 \"comportexviz.server.notebook$eval28199$fn$reify__28203@4df993b7\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x796bf8bd \"comportexviz.server.notebook$eval28199$fn$reify__28203@796bf8bd\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x5c65f76f \"comportexviz.server.notebook$eval28199$fn$reify__28203@5c65f76f\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7dc35d07 \"comportexviz.server.notebook$eval28199$fn$reify__28203@7dc35d07\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7c261d23 \"comportexviz.server.notebook$eval28199$fn$reify__28203@7c261d23\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x5a2ebc0a \"comportexviz.server.notebook$eval28199$fn$reify__28203@5a2ebc0a\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2903e6af \"comportexviz.server.notebook$eval28199$fn$reify__28203@2903e6af\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x4e4fbded \"comportexviz.server.notebook$eval28199$fn$reify__28203@4e4fbded\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x24c6e02 \"comportexviz.server.notebook$eval28199$fn$reify__28203@24c6e02\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2dad4cad \"comportexviz.server.notebook$eval28199$fn$reify__28203@2dad4cad\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1b2de129 \"comportexviz.server.notebook$eval28199$fn$reify__28203@1b2de129\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x283e8774 \"comportexviz.server.notebook$eval28199$fn$reify__28203@283e8774\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x4963c00f \"comportexviz.server.notebook$eval28199$fn$reify__28203@4963c00f\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0xeec25aa \"comportexviz.server.notebook$eval28199$fn$reify__28203@eec25aa\"])"}
;; <=

;; **
;;; Cool!
;;; 
;;; Does it solve all of our problems? Let's try a few different distances and see how consistent the overlaps are.
;; **

;; @@
(defn pair-overlaps2
  [encoder pair-distance start-vals]
  (->> start-vals
       (map (fn [v1]
              [v1 (+ v1 pair-distance)]))
       (map (fn [pair]
              (->> pair
                   (map (partial p/encode encoder ))
                   (map (partial into #{}))
                   (apply set/intersection)
                   count)))))

(defn plot-overlap-probabilities2
  [encoder distance start-vals]
  (let [freqs (frequencies (pair-overlaps2 encoder distance start-vals))
        categories (range (inc (apply max (keys freqs))))]
    (plot/bar-chart categories (->> categories
                                    (map #(get freqs % 0))
                                    (map #(/ % (count start-vals))))
                    :plot-range [categories [0 1]])))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;comportexviz.notebook.hotgym/plot-overlap-probabilities2</span>","value":"#'comportexviz.notebook.hotgym/plot-overlap-probabilities2"}
;; <=

;; @@
(for [d [30 10 1]]
  (plot-overlap-probabilities2 (sampled-linear-encoder [1024] 17
                                                       [(- 0 0.12) (+ 100 0.12)]
                                                       256)
                               d (range 0 (- 100 d) 0.1)))
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"top":10,"left":50,"bottom":20,"right":10},"data":[{"name":"91e9e58a-ab39-44c8-9063-1df1ffb9861d","values":[{"x":0,"y":0},{"x":1,"y":0},{"x":2,"y":0},{"x":3,"y":0.2785714285714286},{"x":4,"y":0.4371428571428571},{"x":5,"y":0.2842857142857143}]}],"marks":[{"type":"rect","from":{"data":"91e9e58a-ab39-44c8-9063-1df1ffb9861d"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"width":{"scale":"x","band":true,"offset":-1},"y":{"scale":"y","field":"data.y"},"y2":{"scale":"y","value":0}},"update":{"fill":{"value":"steelblue"},"opacity":{"value":1}},"hover":{"fill":{"value":"#FF29D2"}}}}],"scales":[{"name":"x","type":"ordinal","range":"width","domain":[0,1,2,3,4,5]},{"name":"y","range":"height","nice":true,"domain":[0,1]}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"91e9e58a-ab39-44c8-9063-1df1ffb9861d\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 0} {:x 3, :y 39/140} {:x 4, :y 153/350} {:x 5, :y 199/700})}], :marks [{:type \"rect\", :from {:data \"91e9e58a-ab39-44c8-9063-1df1ffb9861d\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}}"},{"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"top":10,"left":50,"bottom":20,"right":10},"data":[{"name":"1149c121-06e9-4c4c-9e3e-acc532c42104","values":[{"x":0,"y":0},{"x":1,"y":0},{"x":2,"y":0},{"x":3,"y":0},{"x":4,"y":0},{"x":5,"y":0},{"x":6,"y":0},{"x":7,"y":0},{"x":8,"y":0.07880133185349612},{"x":9,"y":0.4406215316315205},{"x":10,"y":0.1587125416204218},{"x":11,"y":0.1897891231964484},{"x":12,"y":0.09988901220865705},{"x":13,"y":0.03107658157602664},{"x":14,"y":0.001109877913429523}]}],"marks":[{"type":"rect","from":{"data":"1149c121-06e9-4c4c-9e3e-acc532c42104"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"width":{"scale":"x","band":true,"offset":-1},"y":{"scale":"y","field":"data.y"},"y2":{"scale":"y","value":0}},"update":{"fill":{"value":"steelblue"},"opacity":{"value":1}},"hover":{"fill":{"value":"#FF29D2"}}}}],"scales":[{"name":"x","type":"ordinal","range":"width","domain":[0,1,2,3,4,5,6,7,8,9,10,11,12,13,14]},{"name":"y","range":"height","nice":true,"domain":[0,1]}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"1149c121-06e9-4c4c-9e3e-acc532c42104\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 0} {:x 3, :y 0} {:x 4, :y 0} {:x 5, :y 0} {:x 6, :y 0} {:x 7, :y 0} {:x 8, :y 71/901} {:x 9, :y 397/901} {:x 10, :y 143/901} {:x 11, :y 171/901} {:x 12, :y 90/901} {:x 13, :y 28/901} {:x 14, :y 1/901})}], :marks [{:type \"rect\", :from {:data \"1149c121-06e9-4c4c-9e3e-acc532c42104\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10 11 12 13 14)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}}"},{"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"top":10,"left":50,"bottom":20,"right":10},"data":[{"name":"6e2060d1-988e-4f1e-b139-0b2505d9b9ce","values":[{"x":0,"y":0},{"x":1,"y":0},{"x":2,"y":0},{"x":3,"y":0},{"x":4,"y":0},{"x":5,"y":0},{"x":6,"y":0},{"x":7,"y":0},{"x":8,"y":0},{"x":9,"y":0},{"x":10,"y":0},{"x":11,"y":0},{"x":12,"y":0.01614530776992936},{"x":13,"y":0.02926337033299697},{"x":14,"y":0.1180625630676085},{"x":15,"y":0.6125126135216953},{"x":16,"y":0.2240161453077699}]}],"marks":[{"type":"rect","from":{"data":"6e2060d1-988e-4f1e-b139-0b2505d9b9ce"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"width":{"scale":"x","band":true,"offset":-1},"y":{"scale":"y","field":"data.y"},"y2":{"scale":"y","value":0}},"update":{"fill":{"value":"steelblue"},"opacity":{"value":1}},"hover":{"fill":{"value":"#FF29D2"}}}}],"scales":[{"name":"x","type":"ordinal","range":"width","domain":[0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16]},{"name":"y","range":"height","nice":true,"domain":[0,1]}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"6e2060d1-988e-4f1e-b139-0b2505d9b9ce\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 0} {:x 3, :y 0} {:x 4, :y 0} {:x 5, :y 0} {:x 6, :y 0} {:x 7, :y 0} {:x 8, :y 0} {:x 9, :y 0} {:x 10, :y 0} {:x 11, :y 0} {:x 12, :y 16/991} {:x 13, :y 29/991} {:x 14, :y 117/991} {:x 15, :y 607/991} {:x 16, :y 222/991})}], :marks [{:type \"rect\", :from {:data \"6e2060d1-988e-4f1e-b139-0b2505d9b9ce\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}}"}],"value":"(#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"91e9e58a-ab39-44c8-9063-1df1ffb9861d\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 0} {:x 3, :y 39/140} {:x 4, :y 153/350} {:x 5, :y 199/700})}], :marks [{:type \"rect\", :from {:data \"91e9e58a-ab39-44c8-9063-1df1ffb9861d\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}} #gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"1149c121-06e9-4c4c-9e3e-acc532c42104\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 0} {:x 3, :y 0} {:x 4, :y 0} {:x 5, :y 0} {:x 6, :y 0} {:x 7, :y 0} {:x 8, :y 71/901} {:x 9, :y 397/901} {:x 10, :y 143/901} {:x 11, :y 171/901} {:x 12, :y 90/901} {:x 13, :y 28/901} {:x 14, :y 1/901})}], :marks [{:type \"rect\", :from {:data \"1149c121-06e9-4c4c-9e3e-acc532c42104\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10 11 12 13 14)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}} #gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"6e2060d1-988e-4f1e-b139-0b2505d9b9ce\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 0} {:x 3, :y 0} {:x 4, :y 0} {:x 5, :y 0} {:x 6, :y 0} {:x 7, :y 0} {:x 8, :y 0} {:x 9, :y 0} {:x 10, :y 0} {:x 11, :y 0} {:x 12, :y 16/991} {:x 13, :y 29/991} {:x 14, :y 117/991} {:x 15, :y 607/991} {:x 16, :y 222/991})}], :marks [{:type \"rect\", :from {:data \"6e2060d1-988e-4f1e-b139-0b2505d9b9ce\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}})"}
;; <=

;; **
;;; Well that's underwhelming.
;;; 
;;; This encoder has a quirk that you should know about. It doesn't like when the radius goes outside the min and max. So now let's actually obey the guidelines in the `sampled-linear-encoder` documentation above. Give it enough space on each end to make up for the radius.
;; **

;; @@
(let [radius 256]
  (for [d [30 10 1 0.1]]
    (plot-overlap-probabilities2 (sampled-linear-encoder [(+ 1024 (* 2 radius))] 17
                                                         [(- 0 (/ radius 10))
                                                          (+ 100 (/ radius 10))]
                                                         radius)
                                 d (range 0 (- 100 d) 0.1))))
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"top":10,"left":50,"bottom":20,"right":10},"data":[{"name":"c20e216e-78d6-41b1-946c-1e6bbfb7eb6d","values":[{"x":0,"y":0},{"x":1,"y":0},{"x":2,"y":0},{"x":3,"y":0.7414285714285714},{"x":4,"y":0.2585714285714286}]}],"marks":[{"type":"rect","from":{"data":"c20e216e-78d6-41b1-946c-1e6bbfb7eb6d"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"width":{"scale":"x","band":true,"offset":-1},"y":{"scale":"y","field":"data.y"},"y2":{"scale":"y","value":0}},"update":{"fill":{"value":"steelblue"},"opacity":{"value":1}},"hover":{"fill":{"value":"#FF29D2"}}}}],"scales":[{"name":"x","type":"ordinal","range":"width","domain":[0,1,2,3,4]},{"name":"y","range":"height","nice":true,"domain":[0,1]}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"c20e216e-78d6-41b1-946c-1e6bbfb7eb6d\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 0} {:x 3, :y 519/700} {:x 4, :y 181/700})}], :marks [{:type \"rect\", :from {:data \"c20e216e-78d6-41b1-946c-1e6bbfb7eb6d\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}}"},{"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"top":10,"left":50,"bottom":20,"right":10},"data":[{"name":"ada9e1a3-b399-44c3-82f7-11c768082fae","values":[{"x":0,"y":0},{"x":1,"y":0},{"x":2,"y":0},{"x":3,"y":0},{"x":4,"y":0},{"x":5,"y":0},{"x":6,"y":0},{"x":7,"y":0},{"x":8,"y":0.144284128745838},{"x":9,"y":0.855715871254162}]}],"marks":[{"type":"rect","from":{"data":"ada9e1a3-b399-44c3-82f7-11c768082fae"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"width":{"scale":"x","band":true,"offset":-1},"y":{"scale":"y","field":"data.y"},"y2":{"scale":"y","value":0}},"update":{"fill":{"value":"steelblue"},"opacity":{"value":1}},"hover":{"fill":{"value":"#FF29D2"}}}}],"scales":[{"name":"x","type":"ordinal","range":"width","domain":[0,1,2,3,4,5,6,7,8,9]},{"name":"y","range":"height","nice":true,"domain":[0,1]}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"ada9e1a3-b399-44c3-82f7-11c768082fae\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 0} {:x 3, :y 0} {:x 4, :y 0} {:x 5, :y 0} {:x 6, :y 0} {:x 7, :y 0} {:x 8, :y 130/901} {:x 9, :y 771/901})}], :marks [{:type \"rect\", :from {:data \"ada9e1a3-b399-44c3-82f7-11c768082fae\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}}"},{"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"top":10,"left":50,"bottom":20,"right":10},"data":[{"name":"f2b12b3e-6a71-4453-8a0e-5fe547175ee6","values":[{"x":0,"y":0},{"x":1,"y":0},{"x":2,"y":0},{"x":3,"y":0},{"x":4,"y":0},{"x":5,"y":0},{"x":6,"y":0},{"x":7,"y":0},{"x":8,"y":0},{"x":9,"y":0},{"x":10,"y":0},{"x":11,"y":0},{"x":12,"y":0},{"x":13,"y":0},{"x":14,"y":0.1473259334006054},{"x":15,"y":0.8526740665993946}]}],"marks":[{"type":"rect","from":{"data":"f2b12b3e-6a71-4453-8a0e-5fe547175ee6"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"width":{"scale":"x","band":true,"offset":-1},"y":{"scale":"y","field":"data.y"},"y2":{"scale":"y","value":0}},"update":{"fill":{"value":"steelblue"},"opacity":{"value":1}},"hover":{"fill":{"value":"#FF29D2"}}}}],"scales":[{"name":"x","type":"ordinal","range":"width","domain":[0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15]},{"name":"y","range":"height","nice":true,"domain":[0,1]}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"f2b12b3e-6a71-4453-8a0e-5fe547175ee6\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 0} {:x 3, :y 0} {:x 4, :y 0} {:x 5, :y 0} {:x 6, :y 0} {:x 7, :y 0} {:x 8, :y 0} {:x 9, :y 0} {:x 10, :y 0} {:x 11, :y 0} {:x 12, :y 0} {:x 13, :y 0} {:x 14, :y 146/991} {:x 15, :y 845/991})}], :marks [{:type \"rect\", :from {:data \"f2b12b3e-6a71-4453-8a0e-5fe547175ee6\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}}"},{"type":"vega","content":{"width":400,"height":247.2187957763672,"padding":{"top":10,"left":50,"bottom":20,"right":10},"data":[{"name":"a4b817ec-8208-4a91-859a-701f204c433d","values":[{"x":0,"y":0},{"x":1,"y":0},{"x":2,"y":0},{"x":3,"y":0},{"x":4,"y":0},{"x":5,"y":0},{"x":6,"y":0},{"x":7,"y":0},{"x":8,"y":0},{"x":9,"y":0},{"x":10,"y":0},{"x":11,"y":0},{"x":12,"y":0},{"x":13,"y":0},{"x":14,"y":0},{"x":15,"y":0.005},{"x":16,"y":0.995}]}],"marks":[{"type":"rect","from":{"data":"a4b817ec-8208-4a91-859a-701f204c433d"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"width":{"scale":"x","band":true,"offset":-1},"y":{"scale":"y","field":"data.y"},"y2":{"scale":"y","value":0}},"update":{"fill":{"value":"steelblue"},"opacity":{"value":1}},"hover":{"fill":{"value":"#FF29D2"}}}}],"scales":[{"name":"x","type":"ordinal","range":"width","domain":[0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16]},{"name":"y","range":"height","nice":true,"domain":[0,1]}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"a4b817ec-8208-4a91-859a-701f204c433d\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 0} {:x 3, :y 0} {:x 4, :y 0} {:x 5, :y 0} {:x 6, :y 0} {:x 7, :y 0} {:x 8, :y 0} {:x 9, :y 0} {:x 10, :y 0} {:x 11, :y 0} {:x 12, :y 0} {:x 13, :y 0} {:x 14, :y 0} {:x 15, :y 1/200} {:x 16, :y 199/200})}], :marks [{:type \"rect\", :from {:data \"a4b817ec-8208-4a91-859a-701f204c433d\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}}"}],"value":"(#gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"c20e216e-78d6-41b1-946c-1e6bbfb7eb6d\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 0} {:x 3, :y 519/700} {:x 4, :y 181/700})}], :marks [{:type \"rect\", :from {:data \"c20e216e-78d6-41b1-946c-1e6bbfb7eb6d\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}} #gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"ada9e1a3-b399-44c3-82f7-11c768082fae\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 0} {:x 3, :y 0} {:x 4, :y 0} {:x 5, :y 0} {:x 6, :y 0} {:x 7, :y 0} {:x 8, :y 130/901} {:x 9, :y 771/901})}], :marks [{:type \"rect\", :from {:data \"ada9e1a3-b399-44c3-82f7-11c768082fae\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}} #gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"f2b12b3e-6a71-4453-8a0e-5fe547175ee6\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 0} {:x 3, :y 0} {:x 4, :y 0} {:x 5, :y 0} {:x 6, :y 0} {:x 7, :y 0} {:x 8, :y 0} {:x 9, :y 0} {:x 10, :y 0} {:x 11, :y 0} {:x 12, :y 0} {:x 13, :y 0} {:x 14, :y 146/991} {:x 15, :y 845/991})}], :marks [{:type \"rect\", :from {:data \"f2b12b3e-6a71-4453-8a0e-5fe547175ee6\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}} #gorilla_repl.vega.VegaView{:content {:width 400, :height 247.2188, :padding {:top 10, :left 50, :bottom 20, :right 10}, :data [{:name \"a4b817ec-8208-4a91-859a-701f204c433d\", :values ({:x 0, :y 0} {:x 1, :y 0} {:x 2, :y 0} {:x 3, :y 0} {:x 4, :y 0} {:x 5, :y 0} {:x 6, :y 0} {:x 7, :y 0} {:x 8, :y 0} {:x 9, :y 0} {:x 10, :y 0} {:x 11, :y 0} {:x 12, :y 0} {:x 13, :y 0} {:x 14, :y 0} {:x 15, :y 1/200} {:x 16, :y 199/200})}], :marks [{:type \"rect\", :from {:data \"a4b817ec-8208-4a91-859a-701f204c433d\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :width {:scale \"x\", :band true, :offset -1}, :y {:scale \"y\", :field \"data.y\"}, :y2 {:scale \"y\", :value 0}}, :update {:fill {:value \"steelblue\"}, :opacity {:value 1}}, :hover {:fill {:value \"#FF29D2\"}}}}], :scales [{:name \"x\", :type \"ordinal\", :range \"width\", :domain (0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16)} {:name \"y\", :range \"height\", :nice true, :domain [0 1]}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}]}})"}
;; <=

;; **
;;; It's consistent! In other words:
;;; 
;;; - When a pair is separated by 30, they overlap by 3 or 4 bits every time.
;;; - When a pair is separated by 1, they overlap by either 14 or 15 bits every time.
;;; - When a pair is separated by 0.1, they overlap on all 16 bits almost every time. Remember, there are 17 bits total.
;;; 
;;; So maybe our consumption encoder could have a radius of 25.6, because powers of 2 are good, and it's fair to say that numbers separated by 51.2 are completely dissimilar in this context.
;;; 
;;; We'll use a range of @@ [-25.6, 125.6] @@ with a little extra to allow for 1536 bits, and probably use 17 active bits.
;; **

;; @@
(defn consumption-encoder
  [n-active-bits]
  (let [radius 256]
    (sampled-linear-encoder [(+ 1024 (* 2 radius))]
                            n-active-bits [(- 0 (/ radius 10))
                                           (+ 100 (/ radius 10))] radius)))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;comportexviz.notebook.hotgym/consumption-encoder</span>","value":"#'comportexviz.notebook.hotgym/consumption-encoder"}
;; <=

;; @@
(for [i [5 10 25 50 75 95]]
  (viz (consumption-encoder 17)  i
       {:display-mode :two-d}))
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAADnCAYAAACJ3tFgAAAEqUlEQVR4Xu2dMVIbQRBFV8qcgZXoNvgEugJHMY50DR2BI8CBIIMyCXIAwuUlaLXfLNJOP2fUIox+zdv/+88sWjw+Pu6H93/7/f7XYrH46ddvChylhwLCBfPw8HBzWHHL5fLu9fX1h1+/KXCMHgtXIFyBCggFFGF2y8IIf1+tPlz7ZbO5f97trkq5OEW4vIAU4W/X1x8uPqzXt0/b7UUlF8cIlw/eFOHyAlKEqwdvjHB5E6EIlxeQIqwLW2ex+o4irAtbZ6H6Drtw+RUowtZZCEE6CIgw3UQTYREW4Vk32CIswiIswpXP1tA6iwbRub/eIG2Qhi5KBRThEx/tsM6y0meVvgifGOHy25p0Fi4vIEXYfWFNhJkIRbh8jKEIz32Wpb+/szAd5UQYztIifOIgXd5ERFiE3VRyU8lNpeOfj6XBs7fXG6QN0tBFqYAGaYM0q6PoCjRIw1uACIuwCM96EhFhERZhEX5X4Ji/NeUsTINnb683SBukbaRnbSIiLMIiLMJuKrmpdOwfnR0PAm4q0WCvC+vC83Lh8qfyx8hnEVbAEfLZRrr6Yw26MHVdinD5E6ljAbMI99Yo0/djkKZIZ11YhKEL0yXf2+sxwuVzIUW4vIDUhasHa4xweVOhCJcXkCLcm6tm348IG6Rho0wFFOEzO2BZLhe2duFyArZGuFqw1oWpibRGuFywbo1wNojO/ftFWIQN0ic9mkFvISIswiIswrM+4yzCIizCIlz5MQV6D3QWPrNGulwbY50FXVyEJ0a43B5H1lQihBUwQDxCuNomUbbess7KIutjDtB1fcyBua5PKlFkRViEz6r+0oUp0lGQLlcOZAWNgnQ2WFb7fhHOrjhdWBfWhbvahNKFIdK6MJuNsQuXL1wpwuUFpAhXb6wxwuVHPYpweQEpwtVmXxtpOvs6C8Pg7KYSC84iLMKNEaSC6sIMaYM0XYEGaXhLEGER9mPRZl3xi/CZI9x94Tq1C3cv4NQI995YfwrS3a8YGpyjOksBk8F6jHDvyLVu0J2FKdJTu3D3m05Tu3BrZM7t54mwCCddkwrmphKbfd1Uar0CdWF4C9CFGdK6MEVahEX4pI89iLAIQwSpgLrwF7uwhetoxWddWAHhJxvaWP+LvC5MTSSLcPcNc1ZQXfiLXdgVCF1YAaELn9umzql/H104axrR0Q4RTc7WurAu7Blpz0i/K7BcLu9O7Ypf/f/rwrpw0jWpYB7tYK47+dGOcoVr6zqrnICtg3S1xnqxH4b9IYe9bDb3z7vd1axzWWuTiH6eAkIX/73Z3B1W3LBe3z5ttxeHrysG42wQTwfpciYRIZx1YQWEjXQ1l42QTiNs4eqmUtv+sHWQjpZ8b9dFOHLZ6HrWhb0HQhfuDUH6fkQ4QjS6LsJwFtaFWUMtwhGi0XURFuHzflLJ9iVYoRHCCujnC0+KuC4cuWx0PULY2RciTGfF3l8vwhGi0XURNkhP6rLRLSiNcPVcuFqtbj4OIgzDoIDRPW50/ZOA2Tqr+r7w5eXl36Mw/7MCq+dCjLACju6BWYQjV+r9uggnTWNMnAi3FlCEc5tKzRGuFqybI1xeQIpwtWA9RvgP2Lz3YnRdcgYAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAADnCAYAAACJ3tFgAAAEfUlEQVR4Xu2dsXHbQBBFRWbOTDNRN3YFakGlWIrUhkpQCXJBZGbPOCE8HooMTsFy+Rc83O5zpgHMkf7cw//7DyBW+/1+uvv4N03T82q1+snPRwUu0gMBxQWz2+2eTituvV6/Hw6HH/x8VOASPVasQHEFIqAoIAhrlywQVlMHCINw19QAwiAsIqgKiAvjwn1nd1xYvASAMAiD8ND9IwiDMAiD8IcClzS42RpvZmF1lCNIE6Sps4Y2ERAGYRAG4cr31jALMwszCw99DQRhEAZhEKbOctwT3NQ/37bbM0J/Hx5+/Xl9/T70ivLWW+osXF5A1YW/PD6e77G+u79/+/3y8rXSPdY00l5k2/NVhMs/V6IinG2TyPv3gDAIi42yKiAIMwszCw89uYAwCIMwCFNnXV9nlR/lmIXFII4LL9yF0xeucyOcXsC5Ec7eWFNnqW3M3AinjzlzI+xteEc7H4RBWAzCqoAgvPAgnd5EcGHxEgDCIEwjTSNNI00jffGXznJ3lui6rYC4MC6MC+PCuPDlLpx+j8PbznhnYQRsXNzrwtk3ibyNOI20F1mCNEG667d0tIiDMAgHI+kV1OvCXpfKfr6McPlc6A3S7S5beQFVhKsHaxnh9Pu+lqmoCJcXUEU4u8tafx8IW4hax0FYDOIgvLBNpXK5MBrhcgJGI1wtWOPClstax6MRLhesoxG2gme24yBsIWodB2GCdNdNJhC2ELWOgzAIg/DQ9weCMAiDMAjzQpajAhXfTKjO5jcP0ukK11u7cDoBb11nZWusTYTTrRhrtvUetxBGQCNoWwhnQ051Xe6R9iJqnW8hXG6TyBKM54W1WzlA2LvCrPNBmDqLOos6izqLOuvaOs+chcmBhsngwrgwLowL48K4cDcXLl+4qi5cXkCrkbYa3OqNNUHaqqus4yrC5ScVFWEL8ezHQdhC1DoOwszCzMLMwszCzMLdZuHyORAXxoVxYVwYF8aFF+vC6QvXuV04vYBz11nZG+tPdVb6FWPVU97jLcII6AzWLcLZkYtuyOVGuvyKVV24vICqC1dHHoS9rhv9QhYQ3u2eTm3INY/8g/B+Pw1dJ6kIqv9fdeHylb7qwtHBdLTPk124/AoEYefsy/PCPC/c933C0UG6/DUQF9aQdrtw+dFNRRgBeb9w6K0gboTLm4aKMAKKCI82q879+4IwdZY4y6oCEqRvHKR5PW6z4tU6q3ywVhFmU4lNJa0eUxEuH6xVhOcOqkv/fIK0mgNBWAziINw5SJcP1tEIlwvW0QhXC9a4MC4suqgqYDTCSw++0b8fCKsrMNqFy83GILywIF1uBYKw6OIgDMJao4wLiwiqAoJwZ4TLtS/Rd2eVF1BFuFp9xWvRVNOIRrjc5MGTSprrgjAIdw7OIAzCfWdfXDj4EqAG6ehNmtE+j00l1ZVppEWkQVhzZRB2Irzdbs/f9PT/C4sQUBUQhH0Ibzab99NXZbECp+nZ2yaBsBPZVuBPAoIwCN90NgZhEPYhFz1bh7twtW3NcITLC6i6cLV94Rbhf/2T+mLxh+1dAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAADnCAYAAACJ3tFgAAAEhUlEQVR4Xu2dPW4bMRSEJXXp4qjxbZIT+Ao+SuzK1/ARcgTnQFKXAGm0QeCfYlNQxPCRfLOfG0NYrSQM+O0Mh9Rqfz6fl93b37Isj/v9/juPXxW4Sg8EFAfM6XR6eB9xh8Ph5XK5fOPxqwLX6LFnBIojEAFFAUFYu2SBsJo6QBiEh6YGEAZhEUFVQFwYFx47d8eFxUsACCdD+Mvx+IHcn7u7n7+fn7+mrs96I2wnYG+EP93ff9Rnu9vbH7+enj5nrs8I0moO7I2w3ZJBb4TdGm8QBmExCKsCgnCyIG1nIriweAkAYRCmzkpdJoAwCIMwCL8pcM1uJubC6tSncH66wnW2IJ1OwNlcOFtjLddZ6UZM60uQivDmBVQRzoZc6xQgI2xXT9UiriK8eQFVhFsjke31QLgW2fXzQZhGmj3SqdsYEAZhEAbhLX9hmyDNohKLSqmvgSA8OcL2hWt0kLYXMBph98aaOos6S5zLqgJGI5ytYa79vCCsjsBoF7ZfdALhyYO0/QgEYdHFQRiEqbOos9gjXXHTVTV4up2PC+PC214XTl+4jkY4vYCjg3T2xpo6S00FoxFOXzaMRri2AZ7t+SAMwmIQVgUEYeos6izqLOos6qyrf0OJbyo1dm1cGBfGhXFhXBgXxoWv/SlHde67Ph8XnsyF069x1I7Q1o305gRsjXD2RaLaxptGuhZZ5sLMhYdu5VgjDsIg3BjJWkFbu3Cti2V/fjjC9rmwdZBeb9WwFzAaYfdgHY5w+s1DJVOJRthewGiEs7ts6fODcAnR0nEQFoM4CE/WSJdMwy4X9kbYTsDeCLsFa1y45LKl470RLl0j0x3vjXApmGY7DsIlREvHQZggPXSRCYRLiJaOgzAIg3DqLb4gDMLeCNu1L713Z9kLGD0Xdquv2J1VCsa1x3FhXNjbhdMVpCAsIlkrYLQLZ2uYaz8vdVbtiOsdpO2vgSCcbGuH3YgkSIuuDcIgzF07Ulf6IAzCIAzC3HiHG+9w4x1uvLPbHQ6Hl9o6afTzw+ss+2XN6LmwvYDRQZp1YbWxdT8/GmG7/o/b32lz3+5bO+xNJBphewFxYQ3p8CBtbyLRCNsLGI3w6Llq9PuDsBr0QZh1YTZYpl4TAWEQBmEQflNgWZZH++BMnaXNfbvXWfYjEhfGhb1d2L5QjUbYXsDoOot1YbXucT8/GmH7GBONcHQjPPr1aaTVSwwIE6S9g7S9iYAwCIMwjTSN9KsCGb9mYBek7duX6Lt2bE7A1nNh9/qKRSV17huNsP3Mg3Vh1oXHfsUfhMW5LwgnQ9g+F0bXWfYCtg7S66DpHqxZVFKDdTTC9sE6GuHRdVP0+4MwCDcOxrWCgrAWrLsjbJcLe7uwnYC9EXYL1t0RtsuFvRG2E7A3wtHBtvfrg3Bt7qORbhy8QThZkLYzEVxYRBqEQbjrOvHxeHx43xD67z8xpjLG/CcgCNchfHNz88IIFLYkg3AlsuvYBcKXOmTXc+vpEZ69cJ0e4XQCzubCszfWzRGefcS0nns3R3jzAqoIz45c64Z6jfBfzOf6YjJDEbMAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAADnCAYAAACJ3tFgAAAEmklEQVR4Xu2dQVLbQBREkXbs4mjDbZITcAWOElhxDY7AEZwD2bukKhsrlbKhgrKQJz2e33/8sqEoE6Da89T9e0Zi2O/3883p3zzPT8MwfOPzowJn6YGA4oLZ7XaPbytuHMft4XD4yudHBc7RY2AFiisQAUUBQVi7ZIGwmjpAGIRDUwMIg7CIoCogLowLx87uuLB4CQBhEAbh1P0jCHeO8Odpekf01/39958vL1+sVqy7C9sL6I7w7cPDe2N+c3f3+uP5+ZNTY84srI5y7gjb7xK6I+y+yQXCICzOsqqAINx5kLY3EVxYvASAMAhTZ1mVA6WuDMKdIWzfvixXqJsLpxPQDWH3+mo5mzMLl5qGO8L2k8dSQDeE3esrEFaRBWFx9gVhLTinQ9g+F7oF6aUL2wvo7sLuwZogrbqyO8L2wdodYfdgDcIgXDkYlwoKwlqwToewXS7M5sJ2AmZD2C1Yp0PYLhdmQ9hOwGwIuwVrEC7NfTTSlYM3CF9ZkLYzEVxYRBqEQZgDlhywPClwzuPi3IKw+vsQpAnSoouqAuLCuPAHF25euPYWpJsL2BvCrRvrcBduvmJU03Crs9ILGI1wa+TU4JzugKVdfeWGsL1Aa9fMaIRrI9X6+4W7cPoV2FuQbv6GgDCzMI00jTSNdMHfEFrLRdf2Oi4sNtq4MC6MC+PCuLCvC9sXru4ubC+guwu7N9bUWWrwd0e4eT1VKqg7wq0b5tKfB8KlK45NJXH25cE72uzLvrCKLAiDcOgfIwVhEK6MoCooQVpzZYK0ugKZhcVLAghfOcLhhWt2hMMFzI5wdGPd3IXDV4zqutGzcHcCtkY4GrnSxnnt65sjbL/HUYp4dhcOf0NaI7yGRLbXQbgU2WgXDkdOFYxNJW32pZGuvQJxYeqs0E0mXFhFGoRBGIRTn9IHYRAG4b8RTle4uiGcTkC3OitbY02QJkiLLqoK6IYwjbT6jmb7/24unK6xBmGtocaF1UsGCIsuDsIgzFM7UveBIAzCIAzCPHjH98E7y9HOrnDNFqTtBMzmwm6N9cVnYbsVo86+rQ9Ydi/gpRF2Q652431xhNMVpKWIZ3Nhuzfk0gjXRsbt+4FwKbKtXdgOOVUwbnPQ6ituc6i9AnFhNpU4YJm6UAVhEAZhED4pMM/zU/fBmSDdWZBOX7hGu3B6AaPrrOyNNXWWOhtHI5zetaMRdmuYS38fEAZhcZZVBQRhLVjbIZwuF7q5cDoB3RDOFqztEE6XC90QTiegG8KlQTb660FYzYEgLAZxEO4sSNvf1pDtdJZ9sHZH2D1Y48K4sOiiqoDuCEcH5bWfD8LqCiRIi5cAEO48SNu3MyAMwhyw5IAlByyPCozjuF0Ljrz+0bXtgrR9++JeZ6UT0C1Iu9dX3Gyozr7uCNtPHtzmoM2+IAzC4uxaW0A3F84W1O2CdDoToc4SLwkgrLkyCKumAsIgTCNNI00jTSP9v408LlzowtM0Pb5dcv58REBVQIJ0WZDebDZbVqBgeiBciOyy3PhHQBDuDGH3bU17hNMJ6Iaw+74wLlzbRKizyuosXPhQ5rrLTa4lwr8BXwT6Yk71R+YAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAADnCAYAAACJ3tFgAAAEjElEQVR4Xu2dTU4bQRCFGe+yi+MNt0lOwBU4SvCKa3CEHME5kL1LpGw8UWSDxCCl3XrV1V01HxuE5kdQ9Nfv1eummU6n03x3/ZjneT9N03e+vlTgpnpQQHHAHI/Hp9cRt9lsDufz+RtfXypwSz0mRqA4AimgWEAQ1qYsEFZdBwiDcFfXAMIgLCKoFhAVXrkKf9nt3nr3Pw8PP3+/vHx17eWjq3D3AkZH+NPj41svf3d//+PX8/Nnz14eFVZFJDrC3fPL6Aj3jt9AGIQx0l17WXUKSIewuy/MpsLuBcymwt7GOh3C7r4wG8LuBcyGsKqqtc+DMEYaI42Rdk2AK5Fr7guzq3DzAmZX4dbGGhWunBI++MzsCDc31tkRrjXGtfeDMAhjpDHSIxvp5iKCCotTACq88t1ZzREtqTQIg3BXFU9npJunL0uksyHsXsBsKtw6vlr2yukQdlflbAi7FzAbwrVxlHo/CJeMcuk6CGOkMdKh4zAQBmEQBuE1n3yEkSaRfnfWl3sak02F3QuYDWHirFLvOdr1bAgTZwU7fo84S50SQJhemF7Yshd294HZEHYvIEaaXrjvudfZEMZIY6RFX6UaW+/nQVj8hWdTYXWrRu3z9MIq8iAMwvTClr2wuw8EYRAGYRBma8ft/0Oo1mhmvx8jjZEWVVQtIL0wiTSJdGgbA8IgzAZLS4RZFxYXldhgqfoy7+eJs0QjjgqjwhhpSxV2j/RBGIRBGISvFdhsNofsCTQH71gbbYw0Rpp1YUVE3NOXbMffdS9gdCPtHV+hwqiwqJrWBYyOcG/jztYOdURipMUpAYSJs1gXVjqRZQLtbqyzIextrFFhVFhUUbWA2RD2NtYgrI5AjLQ4BYAwRvq/68LNfWF2hJsXMDvCrY01KowKiyqqFjA7wq2NNQirIzC7CjffcAnCGGk2WFom0s2RXc6ZIAzCIAzC7JG+VGCe5737JKwa2d7PY6TFXhoVRoVRYVQYFY6jws3XOGpVPZoKD1fAaCrcepGoNsEmka5FNttfKnXvnKIhXItY6/tBGITFXlYtIAjTC9ML0wvTC8fphbv7Poy0sWqjwqgwKowKo8J5VNg9cI2WSJdsjHsBs6mwd2JNnKWmMdkQLiFufj0bwq0TaM7OUpGlF6YX7npiJQiDsDGCakFRYeIs4iziLOKsPHGWeatWEhl6YVHVUWFUuEqFzQPXtSFsXsC1IWydWA+fSJuPmJKq1l4fHeHhCzg6wtbIWSfWwyPsboyzITx8AUdH2Bo56/eBcC2yLCqJvS8H72i9L4tKKrIgDMKsC4deAwFhEAZhEL5WgMPK5v1EL6wZ6/C9cPfAdfREuhRndS9gdIR7J9bhES6N0ObXoyPcvECl8CE6wtYJc+37QLg0wkrXQVjsjUF45Ua6u4iAMAh3jbdQ4ZLKlq6DMAiDcKSEerfbPb1+v/8+MweW5rjF9Q8FxEjXGentdntY1Qi0DlxXh3DzAmZH2DqxHh5h6xFj3SsPj3C4Ao6GsDVytQlz6f7hEbZGzvp9wyNs/QNbvw8jfa4zzkuklwj/BdXU+mJfbykBAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAADnCAYAAACJ3tFgAAAEc0lEQVR4Xu2dQU4bQRBFZ7zLDscbbkNOwBU4SmDla/gIOYJzIHuXSNl4oigYiUFyUfwaD9P/sUGobSSX+vX/9bvb0x+Px6F7/hmG4anv++9Of3/dbF4+75/7+5+/d7u71OengGIBD4fD47niq9Vqfzqdvjn9/eXh4eXzd7e3P35ttzeZz9+7z0B5yaKA4prvjrC6ZIGw6jpAGIRndQ0gPEI4baxB+DXC6QKiwq8bh6yxBmFUWFRRtYAgrPX+IKzOQFRYXAJAGITnTdBBGITphVObOKpqVr8fhEEYhOdEOJ2+jJcAd4TlArob6Wx8Nd6EohdOqvJ4xlJAtYDuCGf3hcfINz8DZZGIZmjrKjx5AVtHWFXZCPHmEZZPX7kjPHkBW0c4QlAdB+EI0Wi8dRUG4U9+ZhuEI0SjcRAmkSaRnjORlkUGhEEYhEHY+LZnTy/M6SzpdJYcuLqrsFxAd4TVxJpeOOp1o3F3hOVOxB1hEukIsanHQZhemF6YXphe+H8FHL94BxWeWmWj/48Ko8KoMCqMCqPCH3UhxFmByoaBKyp8WYXDAhJnXd5UihLr5hAOZ0xkjLPjrSF89QK2hnCEnNr7ctkwi2j0+tYQljeJooKNx1tDuBrR6P81p8JXn4EgTJxFnEWcRZxFnEWc1XXdLA+UQYVRYVQYFUaFUWFU+J0qXB64uqlweQHd4qzqxJo4Kxug8g2WonEmkdYuF7KppCILwiA8a+8LwiBcjKBaUDcjHe3zZscx0uoMdOuFyzfeQVgz1iAMwjOrMgiDsPTFO2NRSQeuqDDPFy7tjbOJNSqMCqPCpQhme1n19SAMwiAMwos+XQXCIAzCIOx8QFJdA4mziLNK46z0nglxlqjiIAzCILxoGwTCIHwR4fQmUdYXtq7CkxewdYSzm0TZhJpEOossZ6RF48w1B011OSOtIgvCIPypEmw7FS73ha0bafn4WrRmtm6kx6pZbaztEE4nztEMdEO4vIBuCGd73ej1IBwhGo2DsGisQVjrje0Rlo21O8JyAd0RVo21PcKyL3RHWC6gO8KRUY7GQTgyytE4CGOkZ02oQThCNBoHYRAG4UWfDwRhEAZhEHa+6UQvTCLNNYdFr4EgDMIgDMLPFVitVvsowW1tnDgriquicXphemF64UWLCAiDMAiDMHEWTzbkyYbvfLKhfJiIy4ai6nJfWIuvuC8c9bbZcYy0iDSJtIa0XZwln8p3V+HyArohrF5rQIWzKhu9HhVGhYmziLOIs4iziLOIs7puGIan8rwusiHquJuRrj5aYtcLl89wjDRGGiONkcZIY6Qx0hhpjHTHGekltlJqK6a+HyONkcZIX9NIl29ruiFcXkC3OIt9YVE1y2cgCO/upDUYhLc35wJ+pBEgkRaXBAqoFtANYTaV1BlT/X43FS7flQNhzkhf9Vs7NpvN49n2/PuNCifXxDcFBOEcwuv1em81A6t7XzuEJy9g6whXx1d2CFf7PjuEJy9g6whX974gnPR94xkMwtUFBGHNSP8F0uT3ozdop3QAAAAASUVORK5CYII=' />"}],"value":"(#object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x736b13ba \"comportexviz.server.notebook$eval28199$fn$reify__28203@736b13ba\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x63d70e71 \"comportexviz.server.notebook$eval28199$fn$reify__28203@63d70e71\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x30fed774 \"comportexviz.server.notebook$eval28199$fn$reify__28203@30fed774\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1109c7cb \"comportexviz.server.notebook$eval28199$fn$reify__28203@1109c7cb\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x3ad4e093 \"comportexviz.server.notebook$eval28199$fn$reify__28203@3ad4e093\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x61abd53e \"comportexviz.server.notebook$eval28199$fn$reify__28203@61abd53e\"])"}
;; <=

;; **
;;; ### Put them all together
;; **

;; **
;;; So we have:
;;; 
;;; - An `hour-of-day-encoder` that we need give a `radius` and `step-distance`
;;; - A `day-of-week-encoder` that we need give a `radius` and `step-distance`
;;; - A `consumption-encoder` that we need to give a `n-active-bits`
;;;   - 17 is working pretty well
;;; 
;;; As we mentioned above, the design constraints for our `radius` and `step-distance` are:
;;; 
;;; - Interrelation radius
;;;   - i.e. how far should the window reach?
;;;   - The encoding will illustrate relations between any two numbers whose distance is within 2 * this number.
;;; - Difference between adjacent numbers
;;;   - i.e. how far do we move the window on each increment?
;;;   - Make it easier for an HTM to see a difference between the numbers 4 and 5.
;;;   
;;; There's a third design constraint:
;;; 
;;; - The number of active bits is a way of hinting to the HTM the importance of the input relative to other inputs.
;; **

;; @@
(for [step-distance [1 2 3]]
  (for [i (range 7)]
    (viz (day-of-week-encoder 0.5 step-distance) i
         {:display-mode :one-d})))
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAAYCAYAAADd5VyeAAAAQklEQVQYV2P8z8DwnwEKfvr5HWTEEPjh53cApoJBQmIDpgrCWt69ewe35f///42MGAJv375tgNnCxMR0AFPFiNICANNah3M5pFwhAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAAYCAYAAADd5VyeAAAAQklEQVQYV2P8z8DwnwEKfvr5HWTEEPjh53cApoJBQmIDpgrCWt69ewe35f///42MGAJv375tgNnCxMR0AFPFiNICANNah3M5pFwhAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAAYCAYAAADd5VyeAAAAQElEQVQYV2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOMP5nYIBr+enndxBT4Ief3wGYFgYJiQ3kaCHDYcNICwCDoYdzWVxOFgAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAAYCAYAAADd5VyeAAAAR0lEQVQYV2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKogrOU/AwPclp9+fgcZMQR++PkdgNnCICGxAVMFYS308QtdbAEAM+iHcxLN8tkAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAAYCAYAAADd5VyeAAAARUlEQVQYV2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKqgiZb/DAxwh/308zvIiCHww8/vAMxhDBISGzBVENYyWL0PAOQgh3OxNkfmAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAAYCAYAAADd5VyeAAAARUlEQVQYV2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoYLFr+MzDA/fLTz+8gI4bADz+/AzC/MEhIbMBUQVgLXUIMAJRnh3P3xDeDAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAAYCAYAAADd5VyeAAAARUlEQVQYV2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoYwlr+MzDAvf/Tz+8gI4bADz+/AzDvM0hIbMBUQVgL6YEMAESuh3OaDOrAAAAAAElFTkSuQmCC' />"}],"value":"(#object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x530638b8 \"comportexviz.server.notebook$eval28199$fn$reify__28203@530638b8\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1c4d8b23 \"comportexviz.server.notebook$eval28199$fn$reify__28203@1c4d8b23\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1966911b \"comportexviz.server.notebook$eval28199$fn$reify__28203@1966911b\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x5314cb5a \"comportexviz.server.notebook$eval28199$fn$reify__28203@5314cb5a\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x3135fa78 \"comportexviz.server.notebook$eval28199$fn$reify__28203@3135fa78\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x41d16bee \"comportexviz.server.notebook$eval28199$fn$reify__28203@41d16bee\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x6c12c486 \"comportexviz.server.notebook$eval28199$fn$reify__28203@6c12c486\"])"},{"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAAwCAYAAAA2GtvFAAAASklEQVQoU2P8z8DwnwEKfvr5HWTEEPjh53cApoJBQmIDporBouXdu3dwv/z//7+REUPg7du3DTC/MDExHcBUMaoFNQhHQwwlwQAAb48O9AqCIKcAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAAwCAYAAAA2GtvFAAAARklEQVQoU2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOMP5nYIBr+enndxBT4Ief3wGYFgYJiQ2DVwsZ3h/VQmqCGUYhBgDQDg707n456AAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAAwCAYAAAA2GtvFAAAAS0lEQVQoU2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKqgiZb/DAxwh/308zvIiCHww8/vAMxhDBISGzBVDBYtgzaQRx1GYuIHAJEbDvT1zn1/AAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAAwCAYAAAA2GtvFAAAASElEQVQoU2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoYwlr+MzDAvf/Tz+8gI4bADz+/AzDvM0hIbMBUMVi0jPCopIv3AVIoDvSAM+bJAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAAwCAYAAAA2GtvFAAAASElEQVQoU2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoYWVr+MzDAQ+ynn99BRgyBH35+B2AhxiAhsQFTxWDRMhr7JCZ+ALKnDvR8JoSfAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAAwCAYAAAA2GtvFAAAAS0lEQVQoU2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahNQIsf8MDHBDf/r5HWTEEPjh53cAFi8MEhIbMFUMFi3DKI0BAHO0DvSdiUFIAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAAAwCAYAAAA2GtvFAAAAS0lEQVQoU2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahEMnxP4zMMCd/tPP7yAjhsAPP78DsNhnkJDYgKlisGihS0oGADTBDvR/zxocAAAAAElFTkSuQmCC' />"}],"value":"(#object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x19a99c45 \"comportexviz.server.notebook$eval28199$fn$reify__28203@19a99c45\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7515332 \"comportexviz.server.notebook$eval28199$fn$reify__28203@7515332\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x4d15d62d \"comportexviz.server.notebook$eval28199$fn$reify__28203@4d15d62d\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x37d91dfd \"comportexviz.server.notebook$eval28199$fn$reify__28203@37d91dfd\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x6247f4f7 \"comportexviz.server.notebook$eval28199$fn$reify__28203@6247f4f7\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x6f82f34e \"comportexviz.server.notebook$eval28199$fn$reify__28203@6f82f34e\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0xcc5e42d \"comportexviz.server.notebook$eval28199$fn$reify__28203@cc5e42d\"])"},{"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABICAYAAADRa1RpAAAAT0lEQVQ4T2P8z8DwnwEKfvr5HWTEEPjh53cApoJBQmIDpoqRpeXdu3fwEPv//38jI4bA27dvG2AhxsTEdABTxagW1CAcDbHRBAPJMWTmFwDUgZZm4M2KMgAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABICAYAAADRa1RpAAAAT0lEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKogrOU/AwPclp9+fgcZMQR++PkdgNnCICGxAVPFyNJCn3gZtYUOiX80kGkfyAD2DZZmvS1wwQAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABICAYAAADRa1RpAAAATElEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoYwlr+MzDAvf/Tz+8gI4bADz+/AzDvM0hIbMBUMbK0jPAEM+r9QVlcAAAobpZmTqWiNgAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABICAYAAADRa1RpAAAAUklEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahBgh9p+BAa7ip5/fQUYMgR9+fgdggcwgIbEBU8XI0jKaLEdzJSQ/EFcoAQBJ+pZmpllNHAAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABICAYAAADRa1RpAAAATklEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahEMnxP4zMMCd/tPP7yAjhsAPP78DsNhnkJDYgKliZGkZzS+DsrgAAHxMlmbRg6cwAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABICAYAAADRa1RpAAAAUElEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhhppg/jMwwAPop5/fQUYMgR9+fgdgaYxBQmIDpoqRpWU0V5JYKAEAndiWZl63xM0AAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABICAYAAADRa1RpAAAATklEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNkQSzH8GBnjM/fTzO8iIIfDDz+8ALPEzSEhswFQxsrQM1uICANAqlmbdCJjqAAAAAElFTkSuQmCC' />"}],"value":"(#object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x738c36bd \"comportexviz.server.notebook$eval28199$fn$reify__28203@738c36bd\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2e318477 \"comportexviz.server.notebook$eval28199$fn$reify__28203@2e318477\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x56510a7c \"comportexviz.server.notebook$eval28199$fn$reify__28203@56510a7c\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x75167817 \"comportexviz.server.notebook$eval28199$fn$reify__28203@75167817\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x67bd91d6 \"comportexviz.server.notebook$eval28199$fn$reify__28203@67bd91d6\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x458da36a \"comportexviz.server.notebook$eval28199$fn$reify__28203@458da36a\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1bd1534d \"comportexviz.server.notebook$eval28199$fn$reify__28203@1bd1534d\"])"}],"value":"((#object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x530638b8 \"comportexviz.server.notebook$eval28199$fn$reify__28203@530638b8\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1c4d8b23 \"comportexviz.server.notebook$eval28199$fn$reify__28203@1c4d8b23\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1966911b \"comportexviz.server.notebook$eval28199$fn$reify__28203@1966911b\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x5314cb5a \"comportexviz.server.notebook$eval28199$fn$reify__28203@5314cb5a\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x3135fa78 \"comportexviz.server.notebook$eval28199$fn$reify__28203@3135fa78\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x41d16bee \"comportexviz.server.notebook$eval28199$fn$reify__28203@41d16bee\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x6c12c486 \"comportexviz.server.notebook$eval28199$fn$reify__28203@6c12c486\"]) (#object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x19a99c45 \"comportexviz.server.notebook$eval28199$fn$reify__28203@19a99c45\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7515332 \"comportexviz.server.notebook$eval28199$fn$reify__28203@7515332\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x4d15d62d \"comportexviz.server.notebook$eval28199$fn$reify__28203@4d15d62d\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x37d91dfd \"comportexviz.server.notebook$eval28199$fn$reify__28203@37d91dfd\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x6247f4f7 \"comportexviz.server.notebook$eval28199$fn$reify__28203@6247f4f7\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x6f82f34e \"comportexviz.server.notebook$eval28199$fn$reify__28203@6f82f34e\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0xcc5e42d \"comportexviz.server.notebook$eval28199$fn$reify__28203@cc5e42d\"]) (#object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x738c36bd \"comportexviz.server.notebook$eval28199$fn$reify__28203@738c36bd\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2e318477 \"comportexviz.server.notebook$eval28199$fn$reify__28203@2e318477\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x56510a7c \"comportexviz.server.notebook$eval28199$fn$reify__28203@56510a7c\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x75167817 \"comportexviz.server.notebook$eval28199$fn$reify__28203@75167817\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x67bd91d6 \"comportexviz.server.notebook$eval28199$fn$reify__28203@67bd91d6\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x458da36a \"comportexviz.server.notebook$eval28199$fn$reify__28203@458da36a\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1bd1534d \"comportexviz.server.notebook$eval28199$fn$reify__28203@1bd1534d\"]))"}
;; <=

;; @@
(for [step-distance [1 2]]
  (for [i (range 24)]
    (viz (hour-of-day-encoder 2 step-distance) i
         {:display-mode :one-d})))
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABUCAYAAAClf5aJAAAAUklEQVQ4T2P8z8DwnwEKfvr5HWTEEPjh53cApoJBQmIDpoohrOXdu3dw7////7+REUPg7du3DTDvMzExHcBUMaoFNQhHQ2w0wUByzGh+Iae4AAD9Cued90ppowAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABUCAYAAAClf5aJAAAAUklEQVQ4T2P8z8DwnwEKfvr5HWTEEPjh53cApoJBQmIDpoohrOXdu3dw7////7+REUPg7du3DTDvMzExHcBUMaoFNQhHQ2w0wUByzGh+Iae4AAD9Cued90ppowAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABUCAYAAAClf5aJAAAAUUlEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOMP5nYIBr+enndxBT4Ief3wGYFgYJiQ3DSgsZITaqhdQ0NhpioyEGKUGILJSGUYIBALW0551nrnNhAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABUCAYAAAClf5aJAAAAVUlEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKogrOU/AwPclp9+fgcZMQR++PkdgNnCICGxAVPFENZCn0AetYUOKXk0kEcDGVJO0ay0BABuXuedhGLYwQAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABUCAYAAAClf5aJAAAAUUlEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKqgiZb/DAxwh/308zvIiCHww8/vAMxhDBISGzBVDGEtgzZeRh02OPPLaLwMl3gBACcI553G4NakAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABUCAYAAAClf5aJAAAAUUlEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoYLFr+MzDA/fLTz+8gI4bADz+/AzC/MEhIbMBUMYS1DKeoHPXLsMmVo1FJYlQCAN+j5535mFwWAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABUCAYAAAClf5aJAAAATklEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoYwlr+MzDAvf/Tz+8gI4bADz+/AzDvM0hIbMBUMYS1jPDYH/X+SM77gzX2AZhN5522xolfAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABUCAYAAAClf5aJAAAAUElEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoYWVr+MzDAQ+ynn99BRgyBH35+B2AhxiAhsQFTxRDWMppgRvMLJHGPFhe0KS0BUPfnndDUMSMAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABUCAYAAAClf5aJAAAAVElEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahBgh9p+BAa7ip5/fQUYMgR9+fgdggcwgIbEBU8UQ1jKaxkazGCRxj5YwRBSwAAmh5527ePJWAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABUCAYAAAClf5aJAAAAUklEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahNQIsf8MDHBDf/r5HWTEEPjh53cAFi8MEhIbMFUMYS2jyXI0V0IS98grlADCPOedanLingAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABUCAYAAAClf5aJAAAAU0lEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahAMUYv8ZGODu+Onnd5ARQ+CHn98BWFQySEhswFQxhLWMpuTRjAxJ3EO8HAMAeubnnZFjlF4AAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABUCAYAAAClf5aJAAAAU0lEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahEMnxP4zMMCd/tPP7yAjhsAPP78DsNhnkJDYgKliCGsZTfyjeR+SuAdT0QcAM5Dnnf37bJ4AAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABUCAYAAAClf5aJAAAAU0lEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahMM6xP4zMMB9+9PP7yAjhsAPP78DsATDICGxAVPFENYyml9GiwtI4qZZaQkA7Cvnnc4Ix7sAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABUCAYAAAClf5aJAAAAVUlEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhhppg/jMwwAPop5/fQUYMgR9+fgdgaYxBQmIDpoohrGU0i42WMJDETVwBCwCk1eedJ9lMCAAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABUCAYAAAClf5aJAAAAUElEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhRnGC+c/AAA/Tn35+BxkxBH74+R2AJUsGCYkNmCqGsJbRXDmCCyUAXX/nnYT3fAAAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABUCAYAAAClf5aJAAAAUElEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNhAJ5j8DAzwafvr5HWTEEPjh53cAlpIZJCQ2YKoYwlpGM/JwKccAFinnnQpjd+gAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABUCAYAAAClf5aJAAAAUUlEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNkQSzH8GBnjM/fTzO8iIIfDDz+8ALPEzSEhswFQxhLWM5v1BWfQBAM7E553YrETGAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABUCAYAAAClf5aJAAAAUElEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNnwTzH8GBnhk//TzO8iIIfDDz+8ALL8wSEhswFQxhLWMFhe0Ly0Bh27nnel51c0AAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABUCAYAAAClf5aJAAAAUklEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDkGnF/+MzDA08dPP7+DmAI//PwOwLIYg4TEhmGlZbSEIbGABQBAGOedKtx7nAAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABUCAYAAAClf5aJAAAAU0lEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDmG3Pzyn4EBnqR++vkdZMQQ+OHndwCWKxkkJDZgqhjCWkZ2oQQA+LPnnfFkex0AAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABUCAYAAAClf5aJAAAAU0lEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDmGjvnlPwMDPBX+9PM7yIgh8MPP7wAsIzNISGzAVDGEtQyjcgwAsV3nnWl/OKMAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABUCAYAAAClf5aJAAAAU0lEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDlmcOeX/wwM8IT708/vICOGwA8/vwOwvM8gIbEBU8UQ1jJYiz4AagfnnYcTVVYAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABUCAYAAAClf5aJAAAAU0lEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDlm2OWX/wwM8LT+08/vICOGwA8/vwOw4oJBQmIDpoohrIUupSUAIrHnnZyIjNkAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAABUCAYAAAClf5aJAAAAUUlEQVQ4T2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDlmNL8cYPzPwADPHj/9/A5iCvzw8zsAK2EYJCQ2DCstpBewANtM550s9gQgAAAAAElFTkSuQmCC' />"}],"value":"(#object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x30361f0c \"comportexviz.server.notebook$eval28199$fn$reify__28203@30361f0c\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x794f8641 \"comportexviz.server.notebook$eval28199$fn$reify__28203@794f8641\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1b9ef5e8 \"comportexviz.server.notebook$eval28199$fn$reify__28203@1b9ef5e8\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x15d4b9ad \"comportexviz.server.notebook$eval28199$fn$reify__28203@15d4b9ad\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x151a1a64 \"comportexviz.server.notebook$eval28199$fn$reify__28203@151a1a64\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1062fb08 \"comportexviz.server.notebook$eval28199$fn$reify__28203@1062fb08\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7c3f6d59 \"comportexviz.server.notebook$eval28199$fn$reify__28203@7c3f6d59\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x4c16baa2 \"comportexviz.server.notebook$eval28199$fn$reify__28203@4c16baa2\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x99a507e \"comportexviz.server.notebook$eval28199$fn$reify__28203@99a507e\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0xea740fe \"comportexviz.server.notebook$eval28199$fn$reify__28203@ea740fe\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x9f5b1d6 \"comportexviz.server.notebook$eval28199$fn$reify__28203@9f5b1d6\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x704fd9e0 \"comportexviz.server.notebook$eval28199$fn$reify__28203@704fd9e0\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2b0d6e37 \"comportexviz.server.notebook$eval28199$fn$reify__28203@2b0d6e37\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1f997b18 \"comportexviz.server.notebook$eval28199$fn$reify__28203@1f997b18\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x3b46023c \"comportexviz.server.notebook$eval28199$fn$reify__28203@3b46023c\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x35c84c6d \"comportexviz.server.notebook$eval28199$fn$reify__28203@35c84c6d\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x75f4f035 \"comportexviz.server.notebook$eval28199$fn$reify__28203@75f4f035\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x21611428 \"comportexviz.server.notebook$eval28199$fn$reify__28203@21611428\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2abda5da \"comportexviz.server.notebook$eval28199$fn$reify__28203@2abda5da\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x5de13fc4 \"comportexviz.server.notebook$eval28199$fn$reify__28203@5de13fc4\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0xfc61915 \"comportexviz.server.notebook$eval28199$fn$reify__28203@fc61915\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x262f8c91 \"comportexviz.server.notebook$eval28199$fn$reify__28203@262f8c91\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x61fb6efb \"comportexviz.server.notebook$eval28199$fn$reify__28203@61fb6efb\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x35b0621f \"comportexviz.server.notebook$eval28199$fn$reify__28203@35b0621f\"])"},{"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAACoCAYAAADHL0/rAAAAZ0lEQVRIS2P8z8DwnwEKfvr5HWTEEPjh53cApoJBQmIDpopRLahBOHRC7N27d3Cn////v5ERQ+Dt27cNsNhnYmI6gKliVAtqEI6G2GiCgeSY0fwyWlyMFrCw2mO0fhmtXtFzw5CpKwGhKs9Iei2oSAAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAACoCAYAAADHL0/rAAAAY0lEQVRIS2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOMP5nYIBr+enndxBT4Ief3wGYFgYJiQ2jWoZPiJGRYEa1kJrFRkNsNMQgBSiRZfJoghlNMKMJZjS/jBYXxLRgSS8tARJ+z0jK8q58AAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAACoCAYAAADHL0/rAAAAZklEQVRIS2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKqgiZb/DAxwh/308zvIiCHww8/vAMxhDBISGzBVjGpBDcKhE2KDNlmOOmxwFhej8TIaL5DKYFDVYqPJcjRZjibLoZ8rAfUIz0jYomsZAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAACoCAYAAADHL0/rAAAAZElEQVRIS2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoYwlr+MzDAvf/Tz+8gI4bADz+/AzDvM0hIbMBUMaoFNQiHToiN8MQ/6v2RXPSNxv5o7EMqthFY748m/tHET0riBwDXoc9I03vIJAAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAACoCAYAAADHL0/rAAAAa0lEQVRIS2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahBgh9p+BAa7ip5/fQUYMgR9+fgdggcwgIbEBU8WoFtQgHDohNprFRksYSN4eLWBH6xf0+na0eh1tXUBKh9HG1eBtWwIAujrPSE2BQGsAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAACoCAYAAADHL0/rAAAAaUlEQVRIS2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahAMUYv8ZGODu+Onnd5ARQ+CHn98BWFQySEhswFQxqgU1CIdOiI1m5NFyDJK3R4vx0VoMvVYfoCpptFAaLZSGRaEEAJzTz0gVx9TAAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAACoCAYAAADHL0/rAAAAZ0lEQVRIS2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahMM6xP4zMMB9+9PP7yAjhsAPP78DsATDICGxAVPFqBbUIBw6ITZaXIyWlpC8PVpZjNaV6G2HYV3xjRZ9I7joAwB/bM9Iz7FswQAAAABJRU5ErkJggg==' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAACoCAYAAADHL0/rAAAAa0lEQVRIS2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhRnGC+c/AAA/Tn35+BxkxBH74+R2AJUsGCYkNmCpGtaAG4dAJsdFCabRMhuTt0SpptEZGb6GMVq8UV6+jBSyJBSwAYgXPSG7NWUwAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAACoCAYAAADHL0/rAAAAaUlEQVRIS2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNkQSzH8GBnjM/fTzO8iIIfDDz+8ALPEzSEhswFQxqgU1CIdOiI0WfaMlPyRvj1Z8o/U+ejtotBIfIpU4XYpxAESez0gW48qvAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAACoCAYAAADHL0/rAAAAZUlEQVRIS2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDkGnF/+MzDA08dPP7+DmAI//PwOwLIYg4TEhlEtwyfERgvY0foFkrdHq9fR1gV6a2u0qTA0mgoAJzfPSBjMV5sAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAACoCAYAAADHL0/rAAAAa0lEQVRIS2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDmGjvnlPwMDPBX+9PM7yIgh8MPP7wAsIzNISGzAVDGqBTUIh06IjRbjo7UYJG+PVuKjbRj0Nt1og4TSBgkACdDPSJaPo2YAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAACoCAYAAADHL0/rAAAAaUlEQVRIS2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDlm2OWX/wwM8LT+08/vICOGwA8/vwOw4oJBQmIDpopRLahBOHRCbLSyGK0rIXl7tKkw2lJCbzkO52YPAOxaz0h6aWb2AAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAACoCAYAAADHL0/rAAAAa0lEQVRIS2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDlmNL+QVVz8Z2CA56iffn4HGTEEfvj5HYAVSgwSEhswVYxqQQ3CoRNio1XSaI0MydujDZLR9hh6+3RgGlcAzvPPSBX0TkEAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAACoCAYAAADHL0/rAAAAa0lEQVRIS2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDlmNL/Qq7j4z8AAz4Q//fwOMmII/PDzOwArxxgkJDZgqhjVghqEQyfERmux0UockrdH2zCjTTj0Ji0V2mMAQEfPSGuVzw8AAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAACoCAYAAADHL0/rAAAAZklEQVRIS2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDlmNL8Mr+LiPwMDPKv/9PM7yIgh8MPP7wCstGSQkNiAqWJUC2oQDp0QG60rR5sKkLw92lIaYQ1FACLgz0i2Hv3RAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAACoCAYAAADHL0/rAAAAaUlEQVRIS2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDlmNL+MFhekF7D/GRjgBcpPP7+DjBgCP/z8DsDKZAYJiQ2YKka1oAbh0Amx0Rp5tEECyduj7bFB0xwFAAV5z0jZX71MAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAACoCAYAAADHL0/rAAAAaUlEQVRIS2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDlmNL+MFheDtYD9z8AAL7Z++vkdZMQQ+OHndwBW8jNISGzAVDGqBTUIh06Ijdb7o80eSN4ebfUR0egFAOgDz0jeCctJAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAACoCAYAAADHL0/rAAAAZklEQVRIS2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDlmNL+MFhcju4D9z8AALxx/+vkdZMQQ+OHndwBWvzBISGzAVDGqBTUIh06IjbYuRhtXkLw9xNuWAMqcz0hgyiarAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAACoCAYAAADHL0/rAAAAaElEQVRIS2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDlmNL+MFhejBSzt65f/DAzwIvinn99BRgyBH35+B2DuYJCQ2ICpYlQLahAOnRAbbcOMNuEgeZtmLVgArTXPSJFcJYEAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAACoCAYAAADHL0/rAAAAY0lEQVRIS2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDlmNL+MFhejBexwqV/+MzDAC/qffn4HGTEEfvj5HYD5lkFCYgOmilEtqEE4dEJstKU0ghuKAI/Oz0i1YSOgAAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAACoCAYAAADHL0/rAAAAaElEQVRIS2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDlmNL+MFhejBexo/QIOASKr1/8MDPDq5Kef30FGDIEffn4HYGHKICGxAVPFqBbUIBw6ITbaHhuUzVEAcmfPSGEhQkUAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAACoCAYAAADHL0/rAAAAaElEQVRIS2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDlmNL+MFhejBexo/QIOgUFVvf5nYIBXWj/9/A4yYgj88PM7AIs5BgmJDZgqRrWgBuHQCbHRVh+JjV4AVQDPSEG07ZwAAAAASUVORK5CYII=' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAACoCAYAAADHL0/rAAAAZklEQVRIS2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDlmNL+MFhejBexo/QIOgRFYvf5nYIBXjT/9/A4yYgj88PM7AEsfDBISGzBVjGpBDcKhE2LDqG0JADeZz0hY0YW8AAAAAElFTkSuQmCC' />"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAQAAACoCAYAAADHL0/rAAAAaElEQVRIS2N89+7dfwYo+P//fyMjhsDbt28bYCqYmJgOYKoY1YIahKMhNppgIDlmNL+MFhejBexo/QIOgdHqlUati/8MDPAK+Kef30FGDIEffn4HYKmQQUJiA6aKUS2oQTh0QowuLVgAGjLPSEjsNRAAAAAASUVORK5CYII=' />"}],"value":"(#object[comportexviz.server.notebook$eval28199$fn$reify__28203 0xa81db0d \"comportexviz.server.notebook$eval28199$fn$reify__28203@a81db0d\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x509095c8 \"comportexviz.server.notebook$eval28199$fn$reify__28203@509095c8\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x6aa6a597 \"comportexviz.server.notebook$eval28199$fn$reify__28203@6aa6a597\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x5cb08246 \"comportexviz.server.notebook$eval28199$fn$reify__28203@5cb08246\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x58ed636e \"comportexviz.server.notebook$eval28199$fn$reify__28203@58ed636e\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0xc07836e \"comportexviz.server.notebook$eval28199$fn$reify__28203@c07836e\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x53520564 \"comportexviz.server.notebook$eval28199$fn$reify__28203@53520564\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0xb281ffa \"comportexviz.server.notebook$eval28199$fn$reify__28203@b281ffa\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x38edebe2 \"comportexviz.server.notebook$eval28199$fn$reify__28203@38edebe2\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x8411362 \"comportexviz.server.notebook$eval28199$fn$reify__28203@8411362\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x4bfb4e2b \"comportexviz.server.notebook$eval28199$fn$reify__28203@4bfb4e2b\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0xe74f61 \"comportexviz.server.notebook$eval28199$fn$reify__28203@e74f61\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x57820664 \"comportexviz.server.notebook$eval28199$fn$reify__28203@57820664\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x5758c5f5 \"comportexviz.server.notebook$eval28199$fn$reify__28203@5758c5f5\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x46c34323 \"comportexviz.server.notebook$eval28199$fn$reify__28203@46c34323\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7c6580dd \"comportexviz.server.notebook$eval28199$fn$reify__28203@7c6580dd\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x3b130813 \"comportexviz.server.notebook$eval28199$fn$reify__28203@3b130813\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2a7515e9 \"comportexviz.server.notebook$eval28199$fn$reify__28203@2a7515e9\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x767391df \"comportexviz.server.notebook$eval28199$fn$reify__28203@767391df\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x26c084a4 \"comportexviz.server.notebook$eval28199$fn$reify__28203@26c084a4\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x53043ff \"comportexviz.server.notebook$eval28199$fn$reify__28203@53043ff\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x768d4ed \"comportexviz.server.notebook$eval28199$fn$reify__28203@768d4ed\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x6dff40b6 \"comportexviz.server.notebook$eval28199$fn$reify__28203@6dff40b6\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x4f380c09 \"comportexviz.server.notebook$eval28199$fn$reify__28203@4f380c09\"])"}],"value":"((#object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x30361f0c \"comportexviz.server.notebook$eval28199$fn$reify__28203@30361f0c\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x794f8641 \"comportexviz.server.notebook$eval28199$fn$reify__28203@794f8641\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1b9ef5e8 \"comportexviz.server.notebook$eval28199$fn$reify__28203@1b9ef5e8\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x15d4b9ad \"comportexviz.server.notebook$eval28199$fn$reify__28203@15d4b9ad\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x151a1a64 \"comportexviz.server.notebook$eval28199$fn$reify__28203@151a1a64\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1062fb08 \"comportexviz.server.notebook$eval28199$fn$reify__28203@1062fb08\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7c3f6d59 \"comportexviz.server.notebook$eval28199$fn$reify__28203@7c3f6d59\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x4c16baa2 \"comportexviz.server.notebook$eval28199$fn$reify__28203@4c16baa2\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x99a507e \"comportexviz.server.notebook$eval28199$fn$reify__28203@99a507e\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0xea740fe \"comportexviz.server.notebook$eval28199$fn$reify__28203@ea740fe\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x9f5b1d6 \"comportexviz.server.notebook$eval28199$fn$reify__28203@9f5b1d6\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x704fd9e0 \"comportexviz.server.notebook$eval28199$fn$reify__28203@704fd9e0\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2b0d6e37 \"comportexviz.server.notebook$eval28199$fn$reify__28203@2b0d6e37\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x1f997b18 \"comportexviz.server.notebook$eval28199$fn$reify__28203@1f997b18\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x3b46023c \"comportexviz.server.notebook$eval28199$fn$reify__28203@3b46023c\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x35c84c6d \"comportexviz.server.notebook$eval28199$fn$reify__28203@35c84c6d\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x75f4f035 \"comportexviz.server.notebook$eval28199$fn$reify__28203@75f4f035\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x21611428 \"comportexviz.server.notebook$eval28199$fn$reify__28203@21611428\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2abda5da \"comportexviz.server.notebook$eval28199$fn$reify__28203@2abda5da\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x5de13fc4 \"comportexviz.server.notebook$eval28199$fn$reify__28203@5de13fc4\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0xfc61915 \"comportexviz.server.notebook$eval28199$fn$reify__28203@fc61915\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x262f8c91 \"comportexviz.server.notebook$eval28199$fn$reify__28203@262f8c91\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x61fb6efb \"comportexviz.server.notebook$eval28199$fn$reify__28203@61fb6efb\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x35b0621f \"comportexviz.server.notebook$eval28199$fn$reify__28203@35b0621f\"]) (#object[comportexviz.server.notebook$eval28199$fn$reify__28203 0xa81db0d \"comportexviz.server.notebook$eval28199$fn$reify__28203@a81db0d\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x509095c8 \"comportexviz.server.notebook$eval28199$fn$reify__28203@509095c8\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x6aa6a597 \"comportexviz.server.notebook$eval28199$fn$reify__28203@6aa6a597\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x5cb08246 \"comportexviz.server.notebook$eval28199$fn$reify__28203@5cb08246\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x58ed636e \"comportexviz.server.notebook$eval28199$fn$reify__28203@58ed636e\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0xc07836e \"comportexviz.server.notebook$eval28199$fn$reify__28203@c07836e\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x53520564 \"comportexviz.server.notebook$eval28199$fn$reify__28203@53520564\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0xb281ffa \"comportexviz.server.notebook$eval28199$fn$reify__28203@b281ffa\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x38edebe2 \"comportexviz.server.notebook$eval28199$fn$reify__28203@38edebe2\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x8411362 \"comportexviz.server.notebook$eval28199$fn$reify__28203@8411362\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x4bfb4e2b \"comportexviz.server.notebook$eval28199$fn$reify__28203@4bfb4e2b\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0xe74f61 \"comportexviz.server.notebook$eval28199$fn$reify__28203@e74f61\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x57820664 \"comportexviz.server.notebook$eval28199$fn$reify__28203@57820664\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x5758c5f5 \"comportexviz.server.notebook$eval28199$fn$reify__28203@5758c5f5\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x46c34323 \"comportexviz.server.notebook$eval28199$fn$reify__28203@46c34323\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x7c6580dd \"comportexviz.server.notebook$eval28199$fn$reify__28203@7c6580dd\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x3b130813 \"comportexviz.server.notebook$eval28199$fn$reify__28203@3b130813\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x2a7515e9 \"comportexviz.server.notebook$eval28199$fn$reify__28203@2a7515e9\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x767391df \"comportexviz.server.notebook$eval28199$fn$reify__28203@767391df\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x26c084a4 \"comportexviz.server.notebook$eval28199$fn$reify__28203@26c084a4\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x53043ff \"comportexviz.server.notebook$eval28199$fn$reify__28203@53043ff\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x768d4ed \"comportexviz.server.notebook$eval28199$fn$reify__28203@768d4ed\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x6dff40b6 \"comportexviz.server.notebook$eval28199$fn$reify__28203@6dff40b6\"] #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x4f380c09 \"comportexviz.server.notebook$eval28199$fn$reify__28203@4f380c09\"]))"}
;; <=

;; @@
(count (p/encode (day-of-week-encoder 0.5 3) 3))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-unkown'>6</span>","value":"6"}
;; <=

;; @@
(count (p/encode (hour-of-day-encoder 2 2) 3))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-unkown'>10</span>","value":"10"}
;; <=

;; **
;;; If we use:
;;; 
;;; - hour of day
;;;   - radius 2
;;;   - step distance 2
;;; - day of week
;;;   - radius 0.5
;;;   - step distance 3
;;; 
;;; Then we put a total of 16 active bits into showing time, and 17 into showing consumption.
;;; 
;;; Let's go with that.
;; **

;; @@
(def sensors
  {:day-of-week [:day (day-of-week-encoder 0.5 3)]
   :hour-of-day [:hour (hour-of-day-encoder 2 2)]
   :power-consumption [:consumption (consumption-encoder 17)]})

(def inputs
  (->> (map vector input-days input-hours input-consumptions)
       (map (partial zipmap [:day :hour :consumption]))))

(let [v (nth inputs 42)]
  (println "Encoding for" v)
  (for [[sense [k enc]] sensors]
    [sense (viz enc (get v k))]))
;; @@
;; ->
;;; Encoding for {:day 6, :hour 18, :consumption 11.6}
;;; 
;; <-
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:day-of-week</span>","value":":day-of-week"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABQAAAAPCAYAAADkmO9VAAAAjklEQVQ4T2N89+7dfwYo+P//fyMjI2M9Pr6QsDBc/qef30H2TZvsYepBfEaqG/j27dsGmA1MTEwH/v3754CPzxkfD1fPICGxgeHFiwCYehCfYhd+W7AA7mVwkFHqZQwDKfXy1/Z2AeQgor4LR71Mck4Z/LHMsWkTPGeBkg/FyQbDQEoTNsesWRPgeRnIAAB6nTrMhTd+cgAAAABJRU5ErkJggg==' />"}],"value":"[:day-of-week #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x289c383f \"comportexviz.server.notebook$eval28199$fn$reify__28203@289c383f\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:hour-of-day</span>","value":":hour-of-day"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAVCAYAAAAnzezqAAAAnElEQVRIS2N89+7dfwYo+P//fyMjI2M9JXwhYWG4/p9+fgfZN22yh5kH4n9bsADOB9s34A54+/ZtA8yFTExMB/79++dACZ8zPh5uHoOExAaGFy8CYOaB+F/b2wWQzR/4EBiNAlqHAEaqR89lA+4AWucC9FSPnstongtGo2A0CgimgdFcMFoQURoCJLeAqF0XUOwASnMBqS0g9LoAAFrUxHGi4WsfAAAAAElFTkSuQmCC' />"}],"value":"[:hour-of-day #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x4c7d5e5d \"comportexviz.server.notebook$eval28199$fn$reify__28203@4c7d5e5d\"]]"},{"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:power-consumption</span>","value":":power-consumption"},{"type":"html","content":"<img src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFAAAADnCAYAAACJ3tFgAAAEm0lEQVR4Xu2dMW4bUQxELXXpoqjxbZIT+Ao+SqzK1/ARcgTnQFKXAGn0g0CygawLihju/v3kc2MIKwnWgG9nSO7Km9Pp1O6uP621w2az+c7jiwI36YGAYsEcj8ent4rbbrev5/P5G48vCtyix4YKFCsQAUUBQVg7ZYGwmjpAGIS7pgYQBmERQVVAXBgX7tu748LiKQCEQRiEh54/gjAIgzAIXxW4ZYKbbeId3gt/2e/fkfrz8PDz98vL16ErzGr1ooN0OQGjXfjT4+P7kuru/v7Hr+fnz5mXVOEIl9srRyNcTsBohLO5rPV5QNhyWes4CDPOYqk0dNAGYRAGYRCufIUrQbrzRLrc9GUarFUXLi+ginC18dW0N6YXtnpd67iKcLnx1VRQFWFr3JP9OAhbiFrHQZheOFcvXC4XRiNcTsBoF64WrHFhy2Wt49EIlwvW0QhnD870whaS3uMgTJDOFaSnJpI+F86NcHoB53bh7MGaIO113eilUrngzERa2wMTpFVkQVgMziAMwn1vLgRhEO7a++LCuHAwgqqgc/fC2SfU9MJqBc49zkrfK4OwFqxBGIQ7uzIIL4xw+h2HF2mvCyPg5JThRTj7ksgb/HFhL7KMs4Jd24uwt8SzPx+EQTgYSa+gILxwkE4/XfFWoDdII6AYpLO7qvfz4cJeZAnSwa6NC+PCfS/1wIVFpEF4ZQiXG7hGI1xOwGiEq02sCdIEadFFVQGjEfb2kqM/H4TVCox24XLjLhBeWZAuV4EgLLo4CIMw46yhv4gbhEEYhEGY/2x4UaC1dpg7CKcbuC4dpNMJuLQLZ5tYm+OsdBWjjq+8l3YgoNErWwhnQy56Am4iPLcrD//+S7vw8ILxxTta78u3diztwumQixbQcuFo18r2friwWpG4MEulrt+lBcIgLCKoCogLa8EahNUKxIXFUwAId0a4/MBVRbi8gCrC1SfWuDAuLLqoKqCKcLbxlPfzgLBagaoLl59Yg3DnIF2+AkFYdHEQBmGukeYaaa6RXu4a6XSujQvjwrn3wukHrnMjnF7AuYN09on1h3FW+opRx1fWbQ4I6HTlKcLZkfNOnK3nM5FWkZ7bhdN1HtzmoI2vuM1BRdZy4fTIRQs4d5C2XGz047iwWpG4sDM448K4cN8lEi4sIgvCIAzCQ++BQbgzwgxcJybkDdIIKP5bNCbW/58C6IXphYODsVdQxlmaK4Owt+LohYORB+HOCJfPhd4gPV06lRdQRbh6sMaFceFgV/UKqiI8+l5X/ftB2FtxBOlg5EG4c5AunwvVIF1ewGiEqwVrXBgXDnZVr6DRCKvBdLTXg7C34gjSwciD8MqCdLmr/KODdDkBQRiE+15wCcKiK4NwZ4TLrzVVhMsLqCJcbXwVfr9w+QoE4c4uDMKnU0t144s6nvK+XkW4XO/LnUpacA534fIVCMKdXXi0JVD038tSyeu6LJVEZHFhXLjrBHq/3z+9NR7/fnMOdJ4DPwiojrOiXW3t77fb7V6pwKsCrbWDtxEAYSeyU4FB+Ky5MAhHVyC9sC9Yg/DaEK62VAp34fICqkG62lJp6sJ/ARZg+mJO4Ey4AAAAAElFTkSuQmCC' />"}],"value":"[:power-consumption #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0xb9488c7 \"comportexviz.server.notebook$eval28199$fn$reify__28203@b9488c7\"]]"}],"value":"([:day-of-week #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x289c383f \"comportexviz.server.notebook$eval28199$fn$reify__28203@289c383f\"]] [:hour-of-day #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0x4c7d5e5d \"comportexviz.server.notebook$eval28199$fn$reify__28203@4c7d5e5d\"]] [:power-consumption #object[comportexviz.server.notebook$eval28199$fn$reify__28203 0xb9488c7 \"comportexviz.server.notebook$eval28199$fn$reify__28203@b9488c7\"]])"}
;; <=

;; **
;;; ## <a name="design_htm_region" /> Design a HTM region
;; **

;; **
;;; Let's get our feet wet.
;; **

;; @@
(defn make-model [spec]
  (core/region-network {:rgn-0 (keys sensors)}
                                   (constantly core/sensory-region)
                                   {:rgn-0 spec}
                                   sensors
                                   nil))

(defn htm-timeline
  [spec]
  (reductions p/htm-step (make-model spec) inputs))

(def baseline-spec
  {:column-dimensions [1024]
   :global-inhibition? false
   :depth 4
   :max-segments 5
   :seg-learn-threshold 7})

(def baseline-example
  (-> (htm-timeline baseline-spec)
      (nth 400)))

(viz baseline-example
     {:columns {:predictive false}})
;; @@
;; =>
;;; {"type":"html","content":"<img src='data:image/png;base64,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' />"}
;; <=

;; **
;;; First let's look at the columns. We'll ignore cells, we'll ignore predicted bits.
;;; 
;;; Here's a messy look at interconnections.
;; **

;; @@
(viz baseline-example
     {:columns {:predictive false}
      :ff-synapses {:inactive true}})
;; @@
;; =>
;;; {"type":"html","content":"<img src='data:image/png;base64,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' />"}
;; <=

;; @@
(def example-steps
  (->> (htm-timeline baseline-spec)
       (drop 770)
       (take 50)))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;comportexviz.notebook.hotgym/example-steps</span>","value":"#'comportexviz.notebook.hotgym/example-steps"}
;; <=

;; @@
(viz example-steps
     {:drawing {:display-mode :two-d}
      :ff-synapses {:permanences true}})
;; @@
;; =>
;;; {"type":"html","content":"<img src='data:image/png;base64,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' /><img src='data:image/png;base64,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' />"}
;; <=

;; @@
(defn actual-consumption [steps t]
  (get-in (nth steps t) [:input-value :consumption]))

(defn predicted-consumption [steps t]
  (when (> t 0)
    (-> (nth steps (dec t))
        (core/predictions :power-consumption 1)
        first
        :value
        double)))

(defn plot-predictions [steps]
  (let [r (range (count steps))]
    (plot/compose
     (plot/list-plot (->> r
                          (map (fn [i]
                                 [i (actual-consumption steps i)])))
                     :plot-size 700
                     :joined true
                     :color "red"
                     :opacity 0.8)
     (plot/list-plot (->> r
                          (map (fn [i]
                                 [i (predicted-consumption steps i)])))
                     :plot-size 700
                     :joined false
                     :color "blue"
                     :opacity 0.8))))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;comportexviz.notebook.hotgym/plot-predictions</span>","value":"#'comportexviz.notebook.hotgym/plot-predictions"}
;; <=

;; @@
(plot-predictions example-steps)
;; @@
;; =>
;;; {"type":"vega","content":{"width":700,"height":432.63287353515625,"padding":{"top":10,"left":50,"bottom":20,"right":10},"scales":[{"name":"x","type":"linear","range":"width","zero":false,"domain":{"data":"0312df94-313f-42f2-be9c-9e4f516dff52","field":"data.x"}},{"name":"y","type":"linear","range":"height","nice":true,"zero":false,"domain":{"data":"0312df94-313f-42f2-be9c-9e4f516dff52","field":"data.y"}}],"axes":[{"type":"x","scale":"x"},{"type":"y","scale":"y"}],"data":[{"name":"0312df94-313f-42f2-be9c-9e4f516dff52","values":[{"x":0,"y":15.0},{"x":1,"y":5.3},{"x":2,"y":5.2},{"x":3,"y":5.3},{"x":4,"y":18.8},{"x":5,"y":40.2},{"x":6,"y":37.4},{"x":7,"y":41.2},{"x":8,"y":47.5},{"x":9,"y":46.7},{"x":10,"y":40.7},{"x":11,"y":39.7},{"x":12,"y":39.7},{"x":13,"y":36.5},{"x":14,"y":38.2},{"x":15,"y":38.3},{"x":16,"y":39.0},{"x":17,"y":43.7},{"x":18,"y":41.0},{"x":19,"y":37.7},{"x":20,"y":32.2},{"x":21,"y":5.3},{"x":22,"y":4.9},{"x":23,"y":9.4},{"x":24,"y":4.8},{"x":25,"y":14.7},{"x":26,"y":22.5},{"x":27,"y":8.3},{"x":28,"y":19.5},{"x":29,"y":41.6},{"x":30,"y":37.0},{"x":31,"y":38.6},{"x":32,"y":43.2},{"x":33,"y":48.0},{"x":34,"y":43.6},{"x":35,"y":41.3},{"x":36,"y":41.1},{"x":37,"y":44.8},{"x":38,"y":45.2},{"x":39,"y":39.9},{"x":40,"y":41.5},{"x":41,"y":46.6},{"x":42,"y":46.0},{"x":43,"y":39.8},{"x":44,"y":30.2},{"x":45,"y":5.1},{"x":46,"y":16.2},{"x":47,"y":23.7},{"x":48,"y":20.4},{"x":49,"y":4.9}]},{"name":"1e009718-eb18-41ff-80aa-e73edb10f929","values":[{"x":0,"y":null},{"x":1,"y":21.4},{"x":2,"y":6.4},{"x":3,"y":4.4},{"x":4,"y":6.4},{"x":5,"y":37.4},{"x":6,"y":43.4},{"x":7,"y":41.4},{"x":8,"y":41.4},{"x":9,"y":41.4},{"x":10,"y":41.4},{"x":11,"y":41.4},{"x":12,"y":41.4},{"x":13,"y":41.4},{"x":14,"y":41.4},{"x":15,"y":37.4},{"x":16,"y":37.4},{"x":17,"y":37.4},{"x":18,"y":41.4},{"x":19,"y":37.4},{"x":20,"y":37.4},{"x":21,"y":6.4},{"x":22,"y":6.4},{"x":23,"y":6.4},{"x":24,"y":15.4},{"x":25,"y":4.4},{"x":26,"y":22.4},{"x":27,"y":37.4},{"x":28,"y":21.4},{"x":29,"y":41.4},{"x":30,"y":43.4},{"x":31,"y":41.4},{"x":32,"y":41.4},{"x":33,"y":41.4},{"x":34,"y":41.4},{"x":35,"y":41.4},{"x":36,"y":41.4},{"x":37,"y":41.4},{"x":38,"y":41.4},{"x":39,"y":41.4},{"x":40,"y":37.4},{"x":41,"y":37.4},{"x":42,"y":37.4},{"x":43,"y":37.4},{"x":44,"y":37.4},{"x":45,"y":6.4},{"x":46,"y":6.4},{"x":47,"y":21.4},{"x":48,"y":15.4},{"x":49,"y":22.4}]}],"marks":[{"type":"line","from":{"data":"0312df94-313f-42f2-be9c-9e4f516dff52"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"y":{"scale":"y","field":"data.y"},"stroke":{"value":"red"},"strokeWidth":{"value":2},"strokeOpacity":{"value":0.8}}}},{"type":"symbol","from":{"data":"1e009718-eb18-41ff-80aa-e73edb10f929"},"properties":{"enter":{"x":{"scale":"x","field":"data.x"},"y":{"scale":"y","field":"data.y"},"fill":{"value":"blue"},"fillOpacity":{"value":0.8}},"update":{"shape":"circle","size":{"value":70},"stroke":{"value":"transparent"}},"hover":{"size":{"value":210},"stroke":{"value":"white"}}}}]},"value":"#gorilla_repl.vega.VegaView{:content {:width 700, :height 432.63287, :padding {:top 10, :left 50, :bottom 20, :right 10}, :scales [{:name \"x\", :type \"linear\", :range \"width\", :zero false, :domain {:data \"0312df94-313f-42f2-be9c-9e4f516dff52\", :field \"data.x\"}} {:name \"y\", :type \"linear\", :range \"height\", :nice true, :zero false, :domain {:data \"0312df94-313f-42f2-be9c-9e4f516dff52\", :field \"data.y\"}}], :axes [{:type \"x\", :scale \"x\"} {:type \"y\", :scale \"y\"}], :data ({:name \"0312df94-313f-42f2-be9c-9e4f516dff52\", :values ({:x 0, :y 15.0} {:x 1, :y 5.3} {:x 2, :y 5.2} {:x 3, :y 5.3} {:x 4, :y 18.8} {:x 5, :y 40.2} {:x 6, :y 37.4} {:x 7, :y 41.2} {:x 8, :y 47.5} {:x 9, :y 46.7} {:x 10, :y 40.7} {:x 11, :y 39.7} {:x 12, :y 39.7} {:x 13, :y 36.5} {:x 14, :y 38.2} {:x 15, :y 38.3} {:x 16, :y 39.0} {:x 17, :y 43.7} {:x 18, :y 41.0} {:x 19, :y 37.7} {:x 20, :y 32.2} {:x 21, :y 5.3} {:x 22, :y 4.9} {:x 23, :y 9.4} {:x 24, :y 4.8} {:x 25, :y 14.7} {:x 26, :y 22.5} {:x 27, :y 8.3} {:x 28, :y 19.5} {:x 29, :y 41.6} {:x 30, :y 37.0} {:x 31, :y 38.6} {:x 32, :y 43.2} {:x 33, :y 48.0} {:x 34, :y 43.6} {:x 35, :y 41.3} {:x 36, :y 41.1} {:x 37, :y 44.8} {:x 38, :y 45.2} {:x 39, :y 39.9} {:x 40, :y 41.5} {:x 41, :y 46.6} {:x 42, :y 46.0} {:x 43, :y 39.8} {:x 44, :y 30.2} {:x 45, :y 5.1} {:x 46, :y 16.2} {:x 47, :y 23.7} {:x 48, :y 20.4} {:x 49, :y 4.9})} {:name \"1e009718-eb18-41ff-80aa-e73edb10f929\", :values ({:x 0, :y nil} {:x 1, :y 21.4} {:x 2, :y 6.4} {:x 3, :y 4.4} {:x 4, :y 6.4} {:x 5, :y 37.4} {:x 6, :y 43.4} {:x 7, :y 41.4} {:x 8, :y 41.4} {:x 9, :y 41.4} {:x 10, :y 41.4} {:x 11, :y 41.4} {:x 12, :y 41.4} {:x 13, :y 41.4} {:x 14, :y 41.4} {:x 15, :y 37.4} {:x 16, :y 37.4} {:x 17, :y 37.4} {:x 18, :y 41.4} {:x 19, :y 37.4} {:x 20, :y 37.4} {:x 21, :y 6.4} {:x 22, :y 6.4} {:x 23, :y 6.4} {:x 24, :y 15.4} {:x 25, :y 4.4} {:x 26, :y 22.4} {:x 27, :y 37.4} {:x 28, :y 21.4} {:x 29, :y 41.4} {:x 30, :y 43.4} {:x 31, :y 41.4} {:x 32, :y 41.4} {:x 33, :y 41.4} {:x 34, :y 41.4} {:x 35, :y 41.4} {:x 36, :y 41.4} {:x 37, :y 41.4} {:x 38, :y 41.4} {:x 39, :y 41.4} {:x 40, :y 37.4} {:x 41, :y 37.4} {:x 42, :y 37.4} {:x 43, :y 37.4} {:x 44, :y 37.4} {:x 45, :y 6.4} {:x 46, :y 6.4} {:x 47, :y 21.4} {:x 48, :y 15.4} {:x 49, :y 22.4})}), :marks ({:type \"line\", :from {:data \"0312df94-313f-42f2-be9c-9e4f516dff52\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :y {:scale \"y\", :field \"data.y\"}, :stroke {:value \"red\"}, :strokeWidth {:value 2}, :strokeOpacity {:value 0.8}}}} {:type \"symbol\", :from {:data \"1e009718-eb18-41ff-80aa-e73edb10f929\"}, :properties {:enter {:x {:scale \"x\", :field \"data.x\"}, :y {:scale \"y\", :field \"data.y\"}, :fill {:value \"blue\"}, :fillOpacity {:value 0.8}}, :update {:shape \"circle\", :size {:value 70}, :stroke {:value \"transparent\"}}, :hover {:size {:value 210}, :stroke {:value \"white\"}}}})}}"}
;; <=

;; @@
;; Display the last exception's stack trace
(print-stack-trace *e)
;; @@
;; ->
;;; clojure.lang.Compiler$CompilerException: java.util.concurrent.ExecutionException: java.lang.IllegalArgumentException: Key must be integer, compiling:(form-init8918245791403204953.clj:2:12)
;;;  at clojure.lang.Compiler$StaticMethodExpr.eval (Compiler.java:1699)
;;;     clojure.lang.Compiler$DefExpr.eval (Compiler.java:439)
;;;     clojure.lang.Compiler.eval (Compiler.java:6787)
;;;     clojure.lang.Compiler.eval (Compiler.java:6745)
;;;     clojure.core$eval.invoke (core.clj:3081)
;;;     clojure.main$repl$read_eval_print__7099$fn__7102.invoke (main.clj:240)
;;;     clojure.main$repl$read_eval_print__7099.invoke (main.clj:240)
;;;     clojure.main$repl$fn__7108.invoke (main.clj:258)
;;;     clojure.main$repl.doInvoke (main.clj:258)
;;;     clojure.lang.RestFn.invoke (RestFn.java:1523)
;;;     clojure.tools.nrepl.middleware.interruptible_eval$evaluate$fn__16116.invoke (interruptible_eval.clj:58)
;;;     clojure.lang.AFn.applyToHelper (AFn.java:152)
;;;     clojure.lang.AFn.applyTo (AFn.java:144)
;;;     clojure.core$apply.invoke (core.clj:630)
;;;     clojure.core$with_bindings_STAR_.doInvoke (core.clj:1868)
;;;     clojure.lang.RestFn.invoke (RestFn.java:425)
;;;     clojure.tools.nrepl.middleware.interruptible_eval$evaluate.invoke (interruptible_eval.clj:56)
;;;     clojure.tools.nrepl.middleware.interruptible_eval$interruptible_eval$fn__16158$fn__16161.invoke (interruptible_eval.clj:191)
;;;     clojure.tools.nrepl.middleware.interruptible_eval$run_next$fn__16153.invoke (interruptible_eval.clj:159)
;;;     clojure.lang.AFn.run (AFn.java:22)
;;;     java.util.concurrent.ThreadPoolExecutor.runWorker (ThreadPoolExecutor.java:1142)
;;;     java.util.concurrent.ThreadPoolExecutor$Worker.run (ThreadPoolExecutor.java:617)
;;;     java.lang.Thread.run (Thread.java:745)
;;; 
;; <-
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; @@
(start-runner (make-model (assoc baseline-spec
                            :ff-potential-radius 1.0)) inputs)
;; @@
;; ->
;;; Started server on port 11308
;;; Navigate to http://localhost:11308/demos/runner.html
;;; 
;; <-
;; =>
;;; {"type":"html","content":"<span class='clj-unkown'>#object[comportexviz.server.runner$start$reify__12711 0x72da96a &quot;comportexviz.server.runner$start$reify__12711@72da96a&quot;]</span>","value":"#object[comportexviz.server.runner$start$reify__12711 0x72da96a \"comportexviz.server.runner$start$reify__12711@72da96a\"]"}
;; <=

;; @@
(stop-all-runners)
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-map'>{</span>","close":"<span class='clj-map'>}</span>","separator":", ","items":[],"value":"{}"}
;; <=
