(ns cloud-clj.etl.vlacs
  (:require
    [clojure.set :refer [rename-keys]]
    [cheshire.core :as json]
    [taoensso.timbre :as timbre]
    [manifold.stream :as s]
    [manifold.deferred :as d]
    [cloud-clj.api :as api]))

(timbre/refer-timbre)

(defn extract-data [resp]
  (map :data (:data resp)))

(defn make-map
  "Takes sequential data and turns it into a map by mapping over it, using the
   key-fn to extract a key for the map and val-fn to extract the value for the
   map."
  [data key-fn val-fn]
  (apply hash-map (mapcat #(vector (key-fn %) (val-fn %)) data)))

;;; Database Select Functions
(defn get-persona-map
  "Queries all of the personas out of the cloud api and turns it into a map of
   persona name to its unique UUID."
  [session]
  (make-map
    (extract-data (api/select-data session "platform" "persona" nil))
    :name :id))

(defn get-existing-users
  "Query all existing VLACS users based on their internal sis_user_idstr."
  [session vlacs-users]
  (extract-data
    (api/select-data
      session "platform" "user"
      {:where {:vlacs__sis_user_idstr {:$in (map :sis_user_idstr vlacs-users)}}})))

(defn get-existing-user-personas
  "Queries existing user personas for a given list of platformn users."
  [session platform-users]
  (extract-data
    (api/select-data
      session "platform" "user_persona"
      {:where {:user_id {:$in (map :id platform-users)}}})))

(defn get-existing-identities
  "Queries existing identities for VLACS' sis_user_idstr values based on a
   provided list of platform users."
  [session users]
  (extract-data
    (api/select-data
      session "platform" "identity"
      {:where {:$and
               [{:name {:$in (map :vlacs__sis_user_idstr users)}}
                {:type {:$eq "vlacs-identity"}}]}})))

(defn get-existing-user-identities
  "Query user identity records based on a list of platform users."
  [session users]
  (extract-data
    (api/select-data
      session "platform" "user_identity"
      {:where {:user_id {:$in (map :id users)}}})))

(defn get-existing-courses
  "Query existing courses based on a list of VLACS courses from their
   PostgreSQL database."
  [session vlacs-courses]
  (extract-data
    (api/select-data
      session "lrm" "course"
      {:where {:vlacs__master_course_idstr {:$in (map :master_course_idstr vlacs-courses)}}})))

(defn get-existing-course-versions
  "Query existing course version using VLACS' internal course versions from
   their PostgreSQL database."
  [session vlacs-course-versions]
  (extract-data
    (api/select-data
      session "lrm" "course_version"
      {:where {:vlacs__master_course_version_id {:$in (map :id vlacs-course-versions)}}})))

(defn get-related-course-versions
  "Queries lrm courses with related course versions based on a list of
   classrooms/offerings for the VLACS PostgreSQL database using
   master_course_idstr as an external id."
  [session vlacs-offerings]
  (:data
    (api/select-data
      session "lrm" "course"
      {:where {:vlacs__master_course_idstr {:$in (set (map :master_course_idstr vlacs-offerings))}}
       :related ["lrm__course_version"]})))

(defn get-existing-offerings
  "Queries existing offerings based on a list of classrooms/offerings from
   VLACS' PostgreSQL database using classroom_idstr as an external id."
  [session vlacs-offerings]
  (:data
    (api/select-data
      session "lrm" "academic_offering"
      {:where {:vlacs__classroom_idstr {:$in (map :classroom_idstr vlacs-offerings)}}
       :related ["lrm__offering_subscription"]})))

(defn get-offering-instructors
  "Queries all instructor users based on the instructors list from a list of
   classrooms/offerings from the VLACS database."
  [session vlacs-offerings]
  (extract-data
    (api/select-data
      session "platform" "user"
      {:where {:vlacs__sis_user_idstr {:$in (set
                                              (mapcat
                                                #(clojure.string/split
                                                   (:sis_user_idstr_list %)
                                                   #"\|") vlacs-offerings))}}})))

(defn get-existing-enrollments
  "Queries enrollments based on data from VLACS' PostgreSQL database, using the
   enrolment_idstr as an external id."
  [session vlacs-enrollments]
  (extract-data
    (api/select-data
      session "lrm" "enrollment"
      {:where {:vlacs__enrollment_idstr {:$in (map :enrolment_idstr vlacs-enrollments)}}})))

(defn get-existing-competencies
  "Queries existing competencies from a list of internal VLACS asmt_pod_ids
   which is an external id for VLACS' competency data."
  [session asmt-pod-ids]
  (extract-data
    (api/select-data
      session "lrm" "competency"
      {:where {:vlacs__asmt_pod_id {:$in asmt-pod-ids}}})))

(defn get-existing-competency-transcripts
  "Queries existing competency transcripts from lrm enrollments, lrm
   competencies, and vlacs e2ap or e2cg records."
  [session user-enrollment-map competency-map vlacs-e2cgs]
  (extract-data
    (api/select-data
      session "lrm" "competency_transcript"
      {:where
       {:$or
        (set
          (mapv
            (fn [e2cg]
              (let [user-id (get user-enrollment-map (:enrolment_idstr e2cg))
                    competency-id (get competency-map (:asmt_pod_id e2cg))]
                {:$and
                 [{:student {:$eq user-id}}
                  {:competency {:$eq competency-id}}]}))
            vlacs-e2cgs))}})))

(defn privilege->persona
  "Creates a VLACS privlege to persona map for loading user information."
  [session]
  (let [persona-map (get-persona-map session)]
    {"STUDENT" (get persona-map "student")
     "TEACHER" (get persona-map "instructor")
     "ADMIN" (get persona-map "administrator")}))

(defn translate-vlacs-users
  "Takes a VLACS user and transforms it into a platform user."
  [vlacs-users]
  (map
    #(-> %
         (select-keys [:sis_user_idstr :firstname :lastname :username])
         (clojure.set/rename-keys {:sis_user_idstr :vlacs__sis_user_idstr
                                   :firstname :first_name
                                   :lastname :last_name
                                   :username :name}))
    vlacs-users))

(defn make-user-persona-dml
  "Using the persona map, user persona map, vlacs users, and platform users,
   generate a series of DML operations to insert and update users in the
   cloud api."
  [persona-map user-persona-map vlacs-users platform-users]
  (let [vlacs-user-privilege-map
        (make-map vlacs-users :sis_user_idstr :privilege)]
    {:insert
     (filter
       identity
       (mapcat
         (fn [user]
           (let [privilege (get vlacs-user-privilege-map (:vlacs__sis_user_idstr user))
                 persona (get persona-map privilege)]
             [(when
                (and
                  (not (nil? persona))
                  (not
                    (some
                      #(= persona (:persona %))
                      (get user-persona-map (:id user)))))
                {:user_id (:id user)
                 :persona persona
                 :active true})])) platform-users))
     ;;; TODO: Implement this later.
     :delete []}))

(defn load-user-personas!
  "Takes the session, vlacs users, and platform users and migrates vlacs users
   into the cloud api."
  [session vlacs-users platform-users]
  (let [persona-map (privilege->persona session)
        user-personas (get-existing-user-personas session platform-users)
        user-persona-map (group-by :user_id user-personas)
        {:keys [insert delete]}
        (make-user-persona-dml
          persona-map user-persona-map vlacs-users platform-users)]
    (when (not (empty? delete))
      (api/delete-data! session "platform" "user_persona" delete))
    (when (not (empty? insert))
      (api/insert-data! session "platform" "user_persona" insert))))

(defn make-new-identities
  "Takes platform users and a map of vlacs identities and creates new identies
   for those that don't already exist."
  [platform-users identity-map]
  (not-empty
    (filter
      not-empty
      (for [{sis-user-idstr :vlacs__sis_user_idstr :as user} platform-users]
        (when (and (not (nil? sis-user-idstr))
                   (nil? (get identity-map sis-user-idstr)))
          {:name sis-user-idstr :type "vlacs-identity"})))))

(defn load-user-identities!
  "Takes a session and platform-users and provisions user identities based on
   VLACS' sis_user_idstrs."
  [session platform-users]
  ;;; Get all identities that have the "vlacs-identity" type and match
  ;;; vlacs__sis_user_idstr. If the identity and relationship don't already
  ;;; exist, make them.
  (let [identity-list (get-existing-identities
                        session platform-users)
        identity-map (make-map identity-list :name :id)
        user-identity-list (get-existing-user-identities
                             session platform-users)
        user-identities-map (group-by :user_id user-identity-list)
        ;;; All identities we care about.
        identity-map
        (merge
          identity-map
          (when-let
            [new-identities (make-new-identities platform-users identity-map)]
            (make-map 
              (api/insert-data! session "platform" "identity" new-identities)
              :name :id)))]
    (when-let
      [new-user-identities
       (not-empty
         (filter
           not-empty
           (for [{sis-user-idstr :vlacs__sis_user_idstr
                  :keys [id] :as user} platform-users]
             (let [identity-id (get identity-map sis-user-idstr)]
               (when (not
                       (contains?
                         (set (map :identity_id (get user-identities-map id)))
                         identity-id))
                 {:user_id id
                  :identity_id identity-id
                  :instance_id (get-in session [:instance :id])})))))]
      (api/insert-data! session "platform" "user_identity" new-user-identities))))

(defn load-users!
  "User loading function to be provided to #'run-import!. Take the current
   session, a count (agent,) and VLACS users from their PostgreSQL database
   and imports them into the cloud api. Returns a deferred value representing
   the completion of the import."
  [session count-agent vlacs-users]
  (d/future
    (let [instance-id (get-in session [:instance :id])
          platform-users (get-existing-users session vlacs-users)
          platform-user-id-map (make-map platform-users :vlacs__sis_user_idstr :id)
          translated-users (translate-vlacs-users vlacs-users)
          upsert-users (map
                         (fn [user]
                           (assoc
                             (if-let [existing-id (get platform-user-id-map
                                                       (:vlacs__sis_user_idstr user))]
                               (assoc user :id existing-id) user)
                             :instance_id instance-id))
                         translated-users)
          platform-users 
          (when-let [upsert-users (not-empty upsert-users)]
            (api/upsert-data! session "platform" "user" upsert-users))]
      (load-user-identities! session platform-users)
      (load-user-personas! session vlacs-users platform-users)
      (send count-agent #(+ % (count vlacs-users)))
      nil)))

(defn load-student-data!
  "Student loading function to be run by #'run-import!"
  [session count-agent vlacs-students]
  (d/future
    (let [platform-users (get-existing-users session vlacs-students)
          platform-user-map (make-map platform-users :vlacs__sis_user_idstr identity)
          update-data
          (filter
            identity
            (map
              (fn [vlacs-student]
                (when-let
                  [platform-user (platform-user-map (:sis_user_idstr vlacs-student))]
                  (merge
                    platform-user
                    (when (:email vlacs-student)
                      {:email (:email vlacs-student)})
                    {:vlacs__is_fulltime (= "FULL_TIME" (:ptft vlacs-student))
                     :vlacs__has_iep (= "1" (:iep vlacs-student))
                     :vlacs__has_section_504 (= "1" (:section504 vlacs-student))
                     :vlacs__dob (not-empty (:dob vlacs-student))
                     :vlacs__grade_level
                     (when-let [grade-level (Integer/parseInt (:grade_level vlacs-student 0))]
                       (if (< 0 grade-level) grade-level nil))})))
              vlacs-students))]
      (let [result (when (not (empty? update-data))
                     (api/update-data! session "platform" "user" update-data))]
        (send count-agent #(+ % (count vlacs-students)))
        result))))

(defn form-courses
  "Takes courses from VLACS' PostgreSQL database and converts them into a
   format consumable by the cloud api."
  [vlacs-courses]
  (map
    #(-> %
         (select-keys [:master_course_idstr :name])
         (rename-keys {:master_course_idstr :vlacs__master_course_idstr}))
    vlacs-courses))

(defn load-courses!
  "Course loading function to be run by #'run-import!"
  [session count-agent vlacs-courses]
  (d/future
    (let [platform-courses (get-existing-courses session vlacs-courses)
          platform-course-map (make-map platform-courses :vlacs__master_course_idstr :id)
          vlacs-courses (form-courses vlacs-courses)]
      (let [rval (api/upsert-data!
                   session "lrm" "course"
                   (map
                     #(if-let [course-id (get platform-course-map (:vlacs__master_course_idstr %))]
                        (assoc % :id course-id) %)
                     vlacs-courses))]
        (send count-agent #(+ % (count vlacs-courses)))
        rval))))

(defn form-course-versions
  "Transforms VLACS course versions from their PostgreSQL database and platform
   courses and create course versions consumable by the cloud api."
  [vlacs-course-versions platform-courses]
  (let [platform-course-map (make-map platform-courses
                                      :vlacs__master_course_idstr
                                      :id)]
    (map
      #(-> %
           (select-keys [:master_course_idstr :version :id])
           (update-in [:master_course_idstr] (fn [id] (get platform-course-map id)))
           (rename-keys {:version :name
                         :id :vlacs__master_course_version_id
                         :master_course_idstr :course}))
      vlacs-course-versions)))

(defn load-course-versions!
  "Course version loading function to be run by #'run-import!"
  [session count-agent vlacs-course-versions]
  (d/future
    (let [platform-course-versions (get-existing-course-versions
                                     session vlacs-course-versions)
          platform-courses (get-existing-courses
                             session vlacs-course-versions)
          platform-cv-map (make-map platform-course-versions
                                    :vlacs__master_course_version_id :id)
          formed-course-versions (form-course-versions
                                  vlacs-course-versions
                                  platform-courses)
          rval (api/upsert-data!
                 session "lrm" "course_version"
                 (map
                   #(if-let [course-version-id (get platform-cv-map (:vlacs__master_course_version_id %))]
                      (assoc % :id course-version-id) %)
                   formed-course-versions))]
      (send count-agent #(+ % (count vlacs-course-versions)))
      rval)))

(defn attach-offering-lookups
  "Takes a vlacs offering and a platform course version map and attaches course
   version ids if the mapping exists."
  [offering platform-cv-map]
  (if-let [platform-cv-id 
           (get-in platform-cv-map
                   [(:master_course_idstr offering)
                    (:version offering)])]
    (assoc offering :course_version platform-cv-id) offering))

(defn form-offerings
  "Takes a list of VLACS offerings from their PostgreSQL database as well as a
   platform course version map and creates offerings consumable by the cloud
   api."
  [vlacs-offerings platform-cv-map]
  (map
    #(-> %
         (select-keys [:classroom_idstr :name :master_course_idstr :version])
         (attach-offering-lookups platform-cv-map)
         (dissoc :master_course_idstr :version)
         (rename-keys {:classroom_idstr :vlacs__classroom_idstr}))
    vlacs-offerings))

(defn make-platform-cv-map
  "Takes a list of platform courses with related version records and creates a
   lookup map for relating offerings to the appropriate course versions."
  [platform-courses]
  (make-map
    platform-courses
    #(get-in % [:data :vlacs__master_course_idstr])
    #(make-map
       (get-in % [:related :lrm__course_version])
       (fn [i] (get-in i [:data :name]))
       (fn [i] (get-in i [:data :id])))))

(defn form-offering-subscriptions
  "Create offerings to be consumed by the cloud API using VLACS offerings from
   their PostgreSQL database, a map of instructors (sis_user_idstr to cloud
   api id,) and an offering map (classroom_idstr to cloud api id.)"
  [vlacs-offerings instructor-map offering-map]
  (filter
    identity
    (mapcat
      (fn [offering]
        (let [instructor-ids (set
                               (clojure.string/split
                                 (:sis_user_idstr_list offering) #"\|"))]
          (map
            (fn [sis-user-idstr]
              (let [platform-offering (get offering-map (:classroom_idstr offering))
                    instructor-id (get instructor-map sis-user-idstr)
                    subscription-map (make-map (map
                                                 :data
                                                 (get-in
                                                   platform-offering
                                                   [:related :lrm__offering_subscription]))
                                               :user :id)]
                (when (and platform-offering instructor-id)
                  (merge
                    {:user instructor-id
                     :academic_offering (:id platform-offering)}
                    (when-let [platform-subscription-id (get subscription-map instructor-id)]
                      {:id platform-subscription-id})))))
            instructor-ids)))
      vlacs-offerings)))

(defn load-offerings!
  "Offering loading function to be run by #'run-import!"
  [session count-agent vlacs-offerings]
  (d/future
    (let [platform-courses (get-related-course-versions session vlacs-offerings)
          platform-offerings (make-map
                               (get-existing-offerings session vlacs-offerings)
                               #(get-in % [:data :vlacs__classroom_idstr])
                               #(get-in % [:data :id]))
          platform-instructors (get-offering-instructors session vlacs-offerings)
          instructor-map (make-map platform-instructors :vlacs__sis_user_idstr :id)
          platform-cv-map (make-platform-cv-map platform-courses)
          vlacs-offerings (form-offerings vlacs-offerings platform-cv-map)
          offering-id-map (make-map vlacs-offerings :vlacs__classroom_idstr :id)
          rval
          (api/upsert-data!
            session "lrm" "academic_offering"
            (map
              #(if-let [id (get platform-offerings (:vlacs__classroom_idstr %))]
                 (assoc % :id id) %)
              vlacs-offerings))]
      ;;; Sync offering subscriptions based on the initial offerings fetched.
      ;;; New offerings won't have any subscriptions yet.
      (form-offering-subscriptions
        vlacs-offerings instructor-map platform-offerings)
      (send count-agent #(+ % (count vlacs-offerings)))
      rval)))

(defn load-offering-subscriptions!
  "Offering subscription loading function to be run by #'run-import!"
  [session count-agent vlacs-offerings]
  (d/future
    (let [existing-offerings (get-existing-offerings session vlacs-offerings)
          platform-offerings (make-map
                               existing-offerings
                               #(get-in % [:data :vlacs__classroom_idstr]) identity)
          platform-instructors (get-offering-instructors session vlacs-offerings)
          instructor-map (make-map platform-instructors :vlacs__sis_user_idstr :id)
          formed-offering-subscriptions
          (form-offering-subscriptions
            vlacs-offerings instructor-map platform-offerings)
          rval
          (api/upsert-data!
            session "lrm" "offering_subscription"
            formed-offering-subscriptions)]
      (send count-agent #(+ % (count vlacs-offerings)))
      ;;; TODO: Handle delete later.
      )))

(defn form-enrollments
  "Takes VLACS enrollments from their PostgreSQL database, platform
   enrollments, platform users, and platform offerings and forms enrollments
   to be consumed by the cloud api."
  [vlacs-enrollments platform-enrollments platform-users platform-offerings]
  (let [user-map (make-map platform-users :vlacs__sis_user_idstr :id)
        offering-map (make-map platform-offerings :vlacs__classroom_idstr :id)
        enrollment-map (make-map platform-enrollments :vlacs__enrollment_idstr :id)]
    (filter
      identity
      (map
        (fn [vlacs-enrollment]
          (let [user-id (get user-map (:sis_user_idstr vlacs-enrollment))
                offering-id (get offering-map (:classroom_idstr vlacs-enrollment))]
            (when (and user-id offering-id)
              (merge
                {:user user-id
                 :academic_offering offering-id
                 :is_active (and
                              (= (:status_idstr vlacs-enrollment) "ACTIVE")
                              (contains?
                                #{"ENABLED" "CONTACT_INSTRUCTOR"}
                                (:activation_status_idstr vlacs-enrollment)))
                 :vlacs__enrollment_idstr (:enrolment_idstr vlacs-enrollment)}
                (when-let [enrollment-id (get enrollment-map (:enrolment_idstr vlacs-enrollment))]
                  {:id enrollment-id})))))
        vlacs-enrollments))))

(defn load-enrollments!
  "Enrollment loading function to be run by #'run-import."
  [session count-agent vlacs-enrollments]
  (d/future
    (let [result (api/upsert-data!
                   session "lrm" "enrollment"
                   (form-enrollments
                     vlacs-enrollments
                     (get-existing-enrollments session vlacs-enrollments)
                     (get-existing-users session vlacs-enrollments)
                     (get-existing-offerings session vlacs-enrollments)))]
      (send count-agent #(+ % (count vlacs-enrollments)))
      result)))

(defn load-competency-grades!
  "Competency transcript loading function to be run by #'run-import!"
  [session count-agent vlacs-e2cgs]
  (d/future
    (let [platform-enrollments (get-existing-enrollments session vlacs-e2cgs)
          platform-competencies (get-existing-competencies session (map :asmt_pod_id vlacs-e2cgs))
          enrollment-map (make-map platform-enrollments :vlacs__enrollment_idstr :user)
          competency-map (make-map platform-competencies :vlacs__asmt_pod_id :id)
          platform-competency-transcripts (get-existing-competency-transcripts
                                            session enrollment-map competency-map vlacs-e2cgs)
          ct-map (make-map platform-competency-transcripts
                           #(vector (:student %) (:competency %)) identity)]
      (when-let [data
                 (not-empty
                   (vals
                     (reduce
                       (fn [ct-map e2cg]
                         (let [student-id (get enrollment-map (:enrolment_idstr e2cg))
                               competency-id (get competency-map (:asmt_pod_id e2cg))]
                           (if (not (and student-id competency-id))
                             ct-map
                             (update-in
                               ct-map [[student-id competency-id]]
                               (fn [ct]
                                 (update-in
                                   (or ct
                                       {:student student-id
                                        :competency competency-id})
                                   [:vlacs__enrollment_idstr_list]
                                   #(conj (set %) (:enrolment_idstr e2cg))))))))
                       ct-map vlacs-e2cgs)))]
        (api/upsert-data! session "lrm" "competency_transcript" data))
      (send count-agent #(+ % (count vlacs-e2cgs))))))

(defn read-json-file
  "Takes a path, reads the contents of that file, and parsed the JSON."
  [path] (json/parse-string (slurp path) true))

(defn run-import!
  "Imports data using a session, loading function, and json data path into the
   cloud api. Immediately returns a map with a finished? key which is a
   deferred value that becomes realized when loading is complete or errors out,
   a load-fn! which is used to tell the importer how to load the data and
   handles counting processed records, and json-data-path which is a path to
   the data to be loaded."
  [session load-fn! json-data-path]
  (let [count-agent (agent 0)
        data (read-json-file json-data-path)
        data-stream (s/batch 200 (s/->source data))
        pending-buffer (s/stream 2)
        completion-deferred (d/deferred)]
    (s/connect
      (s/map (partial load-fn! session count-agent) data-stream)
      pending-buffer)
    (s/consume
      (fn [_] true)
      (s/realize-each pending-buffer))
    (s/on-closed
      pending-buffer
      (fn [] (d/success! completion-deferred true)))
    {:finished? completion-deferred
     :total (count data)
     :processed count-agent}))

(defn import-student-info! [session student-json-path]
  (run-import! session load-student-data! student-json-path))

(defn import-competency-transcripts! [session competency-grade-json-path]
  (run-import! session load-competency-grades! competency-grade-json-path))

(defn import-enrollments! [session enrollment-json-path]
  (run-import! session load-enrollments! enrollment-json-path))

(defn import-offering-subscriptions! [session offering-json-path]
  (run-import! session load-offering-subscriptions! offering-json-path))

(defn import-offerings! [session offering-json-path]
  (run-import! session load-offerings! offering-json-path))

(defn import-course-versions! [session course-version-json-path]
  (run-import! session load-course-versions! course-version-json-path))

(defn import-courses! [session course-json-path]
  (run-import! session load-courses! course-json-path))

(defn import-users! [session user-json-path]
  (run-import! session load-users! user-json-path))

