(ns monkey.zmq.events
  "Functionality for setting up an event server and poster.  This uses ZeroMQ's
   push/pull socket types in the background."
  (:require [clojure.tools.logging :as log]
            [com.stuartsierra.component :as co]
            [manifold.stream :as ms]
            [monkey.zmq.common :as mc]
            [zeromq.zmq :as z]))

(defn event-server
  "Starts an event server that can only receive events, it cannot send out any of
   itself."
  [ctx addr handler & [opts]]
  (letfn [(receiver [s]
            (let [evt (z/receive s)]
              (handler (mc/parse-edn evt))))]
    (-> (mc/->Server ctx addr receiver :pull)
        (merge opts)
        (co/start))))

(defn event-poster
  "Creates an event poster that sends to the given address.  Cannot receive any events."
  [ctx addr]
  (let [s (doto (z/socket ctx :push)
            (z/connect addr))]
    (mc/->Client s #(z/send-str s (pr-str %2)))))

;; Request types
(def req-event 0)
(def req-register 1)
(def req-unregister 2)
(def req-disconnect 3)

(defn register-client
  "Registers a new client with filter in the state.  The client is identified by the
   socket and its id."
  [state sock id evt-filter _]
  (log/info "Registering client" id "for filter" evt-filter)
  (update-in state [:listeners evt-filter sock] (fnil conj #{}) id))

(defn unregister-client
  "Removes a client registration from the filter.  It removes the id from the
   registrations for the same filter."
  [state sock id evt-filter _]
  (log/info "Unregistering client" id "for filter" evt-filter)
  (update-in state [:listeners evt-filter sock] disj id))

(defn dispatch-event [matches-filter? state sock id evt raw]
  (log/info "Dispatching event from" id ":" evt)
  ;; Find all socket/id pairs where the event matches the filter
  (let [socket-ids (->> (:listeners state)
                        (filter (comp (partial matches-filter? evt) first))
                        (map second))]
    (log/debug "Found" (count socket-ids) "matching socket/ids pairs")
    ;; TODO Eliminate duplicates (from multiple matching filters)
    (update state :replies (fnil conj []) [[req-event raw] socket-ids])))

(defn disconnect-client
  "Handles client disconnect request, by removing it from all registrations.
   A reply is sent back to the client to indicate the request was processed.
   This allows clients to wait until it has been unregistered."
  [state sock id _ _]
  (letfn [(remove-from-socket [[f s]]
            [f (update s sock disj id)])
          (remove-from-filters [l]
            (->> (map remove-from-socket l)
                 (filter (comp empty? second))
                 (into {})))]
    (log/info "Disconnecting client" id)
    (update state :listeners remove-from-filters)))

(defn- handle-incoming
  "Handles incoming request on the broker"
  [state req-handlers sock]
  (let [[id req payload] (z/receive-all sock)]
    (try
      (let [id (String. id)
            parsed (mc/parse-edn payload)
            req (aget req 0)
            h (get req-handlers req)]
        (log/debug "Handling incoming request from id" id ":" req)
        (if h
          (h state sock id parsed payload)
          (do
            (log/warn "Got invalid request type:" req)
            state)))
      (catch Exception ex
        (log/error "Unable to handle incoming request.  Id: " id ", req:" req ", payload:" (String. payload))))))

(defn post-pending
  "Posts pending events in the state to the specified socket/client id."
  [state]
  (let [{:keys [replies]} state]
    (when (not-empty replies)
      (log/debug "Posting" (count replies) "outgoing replies")
      (doseq [[[req e] dest] replies]
        (doseq [sock-ids dest]
          (log/debug "Posting to:" (vals sock-ids))
          (doseq [[sock ids] sock-ids]
            ;; TODO Only send when socket is able to process the event
            (doseq [id ids]
              (z/send-str sock id z/send-more)
              ;; Send request type
              (z/send sock (byte-array 1 [req]) z/send-more)
              ;; Event is raw payload
              (z/send sock e))))))
    (dissoc state :replies)))

(defn- run-broker-server [{:keys [context address running? poll-timeout matches-filter? state-stream
                                  linger close-context?]
                           :or {poll-timeout 500
                                linger 0
                                close-context? false}}]
  ;; TODO Add support for multiple addresses (e.g. tcp and inproc)
  (let [socket (doto (z/socket context :router)
                 (z/set-linger linger)
                 (z/bind address))
        poller (doto (z/poller context 1)
                 (z/register socket :pollin))
        matches-filter? (or matches-filter? (constantly true))

        req-handlers {req-event (partial dispatch-event matches-filter?)
                      req-register register-client
                      req-unregister unregister-client
                      req-disconnect disconnect-client}

        recv-incoming
        (fn [state]
          (z/poll poller poll-timeout)
          (cond-> state
            (z/check-poller poller 0 :pollin)
            (handle-incoming req-handlers socket)))

        publish-state
        (fn [state]
          (ms/put! state-stream state)
          state)]
    (try
      (reset! running? true)
      ;; State keeps track of registered clients
      (loop [state {}]
        (when (and @running?
                   (not (.. Thread currentThread isInterrupted)))
          ;; TODO Auto-unregister dead clients (use a ping system)
          (-> state
              (post-pending)
              (recv-incoming)
              (publish-state)
              (recur))))
      (catch Exception ex
        (log/error "Server error:" ex))
      (finally
        (reset! running? false)
        (z/close socket)
        (ms/close! state-stream)
        (when close-context?
          (.close context))))
    (log/info "Server terminated")))

(defrecord ThreadComponent [run-fn]
  co/Lifecycle
  (start [this]
    (let [t (assoc this :running? (atom false))]
      (assoc t :thread (doto (Thread. #(run-fn t))
                         (.start)))))
  
  (stop [{:keys [thread running?] :as this}]
    (when thread
      (reset! running? false)
      (.interrupt thread)
      (.join thread)))

  java.lang.AutoCloseable
  (close [this]
    (co/stop this)))

(defn- component-running? [{:keys [running?]}]
  (true? (some-> running? deref)))

(defn broker-server
  "Starts an event broker that can receive incoming events, but also dispatches outgoing
   events back to the clients.  Clients must register for events with a filter.  The filter
   is a user-defined object, that is matched against the events using the `matches-filter?`
   option.  If no matcher is specified, all events are always matched.  The server also
   provides a `state-stream` that holds the latest state, useful for metrics and inspection."
  [ctx addr & [{:keys [autostart?]
                :as opts
                :or {autostart? true}}]]
  (cond-> (map->ThreadComponent (assoc opts
                                       :context ctx
                                       :address addr
                                       :state-stream (ms/sliding-stream 1)
                                       :run-fn run-broker-server))
    autostart? (co/start)))

(defn- run-sync-client
  [{:keys [id context address handler stream running? poll-timeout linger close-context?]
    :or {poll-timeout 500 linger 0 close-context? false}}]
  ;; Sockets are not thread save so we must use them in the same thread
  ;; where we create them.
  (let [socket (doto (z/socket context :dealer)
                 (z/set-identity (.getBytes id))
                 (z/set-linger linger)
                 (z/connect address))
        poller (doto (z/poller context 1)
                 (z/register socket :pollin))
        take-next
        (fn [] @(ms/try-take! stream 0))

        send-request
        (fn [[req payload :as m]]
          (log/debug "Posting request:" m)
          (z/send socket (byte-array 1 [req]) z/send-more)
          (z/send-str socket (pr-str payload)))

        receive-evt
        (fn []
          (log/debug "Received events at" id)
          (let [[_ recv] (z/receive-all socket)]
            ;; TODO Instead of passing all events to a single handler, allow a handler per
            ;; registered event filter.  This would mean the broker will have to send back
            ;; the filter (or some id?) that passed the event.
            (-> recv
                (mc/parse-edn)
                (handler))))

        check-running
        (fn []
          (and @running?
               (not (.. Thread currentThread isInterrupted))))]
    (try
      (reset! running? true)
      (loop [continue? (check-running)]
        ;; Send pending outgoing requests
        (loop [m (take-next)]
          (when m
            (send-request m)
            (recur (take-next))))
        ;; When stopped, add a disconnect request
        (when (not continue?)
          (log/debug "Sending disconnect request")
          (ms/close! stream) ; Stop accepting more requests
          (send-request [req-disconnect {}]))
        ;; Check for incoming data
        (z/poll poller poll-timeout)
        (when (z/check-poller poller 0 :pollin)
          (receive-evt))
        (when continue?
          (recur (check-running))))
      (catch Exception ex
        (log/error "Socket error:" ex))
      (finally
        (reset! running? false)
        (z/close socket)
        (when close-context?
          (.close context))))
    (log/info "Client" id "terminated")))

(defn send-request
  "Sends a raw request to the broker.  Returns a deferred that realizes when the request
   has been accepted by the subsystem (not necessarily when it's transmitted)."
  [client req]
  (ms/put! (get-in client [:component :stream]) req))

(defn post-event
  "Posts event, returns a deferred that realizes when the event has been accepted
   by the background thread."
  [client evt]
  (send-request client [req-event evt]))

(defrecord BrokerClient [component]
  co/Lifecycle
  (start [this]
    (log/info "Connecting client to" (:address component))
    (assoc this
           :component
           (-> component
               (assoc :run-fn run-sync-client
                      :stream (ms/stream))
               (co/start))))
  
  (stop [{:keys [component] :as this}]
    (if (component-running? component)
      (do 
        (log/debug "Stopping client" (:id component))
        (assoc this :component (co/stop component)))
      this))

  clojure.lang.IFn
  (invoke [this evt]
    (post-event this evt))
  
  java.lang.AutoCloseable
  (close [this]
    (co/stop this)))

(defn broker-client
  "Connects to an event broker.  Can post and receive events."
  [ctx addr handler & [{:keys [autostart?] :or {autostart? true} :as opts}]]
  (cond->
      (-> (map->ThreadComponent {:context ctx
                                 :address addr
                                 :handler handler
                                 :id (str (random-uuid))})
          (merge opts)
          (->BrokerClient))
    autostart? (co/start)))

(defn register
  "Registers the client to receive events matching the filter.  A client can
   register with multiple filters.  If any of the filters match, the event will
   be received."
  [client evt-filter]
  (send-request client [req-register evt-filter]))

(defn unregister
  "Unregisters to no longer receive events matching the filter."
  [client evt-filter]
  (send-request client [req-unregister evt-filter]))
