(ns monkey.scw.core
  "Core Scaleway client namespace.  It provides functions for creating clients for
   the various Scaleway services using their provided OpenAPI definitions."
  (:require [camel-snake-kebab.core :as csk]
            [martian
             [core :as mc]
             [interceptors :as mi]
             [openapi :as mo]]
            [monkey.martian.aleph :as mma]
            [monkey.scw.openapi :as openapi]))

(def scw-url "https://api.scaleway.com")

(def default-version "v1beta1")

(def api-config
  {:instance
   {:version "v1"}
   :nats
   {:id "mnq"
    :api "NatsApi"}
   :serverless-jobs
   {:version "v1alpha1"}
   :domain
   {:version "v2beta1"}
   :key-manager
   {:version "v1alpha1"}})

(defn openapi-url [api version]
  (let [version (or version (get-in api-config [api :version] default-version))
        api-name (get-in api-config [api :api] "Api")
        id (get-in api-config [api :id] (.replaceAll (name api) "-" "_"))]
    (format "https://www.scaleway.com/en/developers/static/scaleway.%s.%s.%s.yml"
            id version api-name)))

(defn add-auth-header
  "Interceptor that adds the `X-Auth-Token` header for authentication"
  [token]
  {:name ::add-auth-header
   :enter (fn [ctx]
            (assoc-in ctx [:request :headers "X-Auth-Token"] token))})

(defn default-interceptors
  "Creates a list of default interceptors that are used by the context.  You can use
   this to build your own interceptor list."
  [conf]
  (concat mc/default-interceptors
          [mi/default-encode-body
           (mi/coerce-response (mma/make-encoders csk/->kebab-case-keyword))
           mma/perform-request
           (add-auth-header (:secret-key conf))]))

(defn default-ctx
  "Loads api definition and applies default transformations to create a Martian
   context."
  [api conf]
  (mma/bootstrap-openapi
   (openapi-url api (get conf :api-version (:version conf)))
   {:server-url scw-url
    :interceptors (or (:interceptors conf) (default-interceptors conf))}))

(defn patched-ctx
  "Loads the openapi spec using given configuration, the applies the patchers to
   it in order.  The result is then used to create a Martian context."
  [api conf patchers]
  (let [patch (apply comp (reverse patchers))
        defs (-> (openapi/load-definition (openapi-url api (:version conf)))
                 deref
                 (patch))
        interceptors (or (:interceptors conf) (default-interceptors conf))
        handlers (mo/openapi->handlers defs (mi/supported-content-types interceptors))]
    (mma/bootstrap scw-url handlers {:interceptors interceptors})))

(def secrets-ctx
  "Creates a secret manager api client by downloading the openapi spec from Scaleway."
  (partial default-ctx :secret-manager))

(def containers-ctx
  "Creates a serverless containers api client by downloading the openapi spec from Scaleway."
  (partial default-ctx :containers))

(def registry-ctx
  "Creates context for container registry"
  (partial default-ctx :registry))

(defn instance-ctx
  "Creates context for compute instances"
  [conf]
  ;; Here we need to override the default functionality because the spec contains some
  ;; errors regarding defaults, which we need to patch before we can proceed.
  (patched-ctx :instance conf [openapi/fix-default-types]))

(def nats-ctx
  "Context for the NATS messaging api"
  (partial default-ctx :nats))

(def functions-ctx
  "Context for the serverless functions api"
  (partial default-ctx :functions))

(def jobs-ctx
  "Context for the serverless jobs api"
  (partial default-ctx :serverless-jobs))

(def domain-ctx
  "Context for the domain and dns api"
  (partial default-ctx :domain))

(defn key-mgr-ctx
  "Context for the key manager api"
  [conf]
  (patched-ctx :key-manager conf [(openapi/remove-value-at
                                   ["paths"
                                    "/key-manager/v1alpha1/regions/{region}/keys"
                                    "post"
                                    "requestBody"
                                    "content"
                                    "application/json"
                                    "schema"
                                    "properties"
                                    "usage"
                                    "default"])]))

;; TODO Add more contexts
