(ns monkey.ci.plugin.pushover
  "Pushover plugin for MonkeyCI.  It allows you to create a job that sends a
   message to Pushover."
  (:require [monkey.ci.api :as api]
            [monkey.pushover.core :as p]))

(defn- make-msg [msg ctx]
  (if (fn? msg)
    (msg ctx)
    (str msg)))

(defn- get-creds [{:keys [user token user-param token-param]
                   :or {user-param "pushover-user"
                        token-param "pushover-token"}}
                  ctx]
  (if (and user token)
    {:user user
     :token token}
    (let [params (api/build-params ctx)]
      {:user (or user (get params user-param))
       :token (or token (get params token-param))})))

(def pushover-opts [:device :sound :title :url :url-title :html :priority :timestamp :ttl])

(defn default-poster
  "Default post function, sends a message to pushover using the postman api"
  [conf]
  @(p/post-message (p/make-client {}) conf))

(defn pushover-action
  "Job action fn that actually sends the message.  This serves as the job action,
   but can be called directly, or integrated in other action jobs."
  [{:keys [msg] :as config} ctx]
  (let [poster (get config :poster default-poster)
        r (poster (-> (get-creds config ctx)
                      (assoc :message (make-msg msg ctx))
                      (merge (select-keys config pushover-opts))))]
    (if (>= (:status r) 400)
      api/failure
      api/success)))

(defn pushover-msg
  "Creates an action job that sends the configured message to Pushover, using
   configured credentials.  These are either passed in using `user` and `token`,
   or read from build parameters (default `pushover-user` and `pushover-token`)."
  [{:keys [id] :or {id "pushover"} :as config}]
  (api/action-job
   id
   (partial pushover-action config)
   (apply dissoc config :id :msg :user :token pushover-opts)))
