(ns monkey.ci.plugin.terraform
  "Allows basic manipulation of terraform var files"
  (:require [clojure.string :as cs]))

(defn- apply-replace [replacements line]
  (let [[k :as parts] (cs/split line #"=")]
    (if (and parts (contains? replacements k))
      (str k "=\"" (get replacements k) "\"")
      line)))

(defn replace-vars
  "Given the contents of a tfvars file, applies given replacements."
  [contents replacements]
  (->> (cs/split-lines contents)
       (mapv (partial apply-replace replacements))
       (cs/join "\n")))

(defn patch-vars
  "Applies replacements on given file"
  [path repl]
  (-> (slurp path)
      (replace-vars repl)
      (as-> c (spit path c))))

(def versions-path "monkeyci/versions.auto.tfvars")

(def id-map
  {:prod
   {"website" "monkeyci_web"
    "monkeyci-gui" "monkeyci_gui"
    "monkeyci-api" "monkeyci_api"}})

(defn update-versions [fc env images]
  (->> images
       (reduce-kv (fn [r k v]
                    (let [rk (get-in id-map [env k])]
                      (cond-> r
                        rk (assoc (str rk "_version") v))))
                  {})
       (replace-vars fc)))

(defn patch-versions
  "Patches the `versions.auto.tfvars` in the monkeyci dir for the given env."
  [updater env images]
  (updater versions-path #(update-versions % env images)))
