(ns monkey.ci.vault.oci
  "Vault implementation for Oracle cloud"
  (:require [clojure.tools.logging :as log]
            [com.stuartsierra.component :as co]
            [monkey.ci.protocols :as p]
            [monkey.ci.vault.common :as vc]
            [monkey.oci.vault :as vault]
            [monkey.oci.vault.b64 :as b]))

(def key-shape {:algorithm "AES"
                :length 32})

(defn- load-enc-key
  "Loads encryption key from the configured vault secret"
  [{:keys [client config]}]
  (log/debug "Loading secret" (:secret-name config) "from vault" (:vault-id config))
  (-> (vault/get-secret-bundle-by-name client (select-keys config [:secret-name :vault-id :version-number]))
      :secret-bundle-content
      :content
      (b/b64->)))

(defn- generate-new-key [{:keys [client config]}]
  (log/debug "Generating new encryption key using master key" (:key-id config))
  (let [gen (-> (vault/generate-data-encryption-key
                 client
                 (-> (select-keys config [:key-id])
                     (assoc :key-shape key-shape
                            :include-plaintext-key true)))
                :plaintext)]
    (log/debug "Storing generated key in secret" (:secret-name config))
    (vault/create-secret client (-> (select-keys config [:secret-name :compartment-id :vault-id :key-id])
                                    (assoc :description "Encryption key generated by MonkeyCI"
                                           :secret-content
                                           {:content-type "BASE64"
                                            :content gen})))
    (b/b64-> gen)))

(defn- load-or-generate-key [v]
  (try
    (load-enc-key v)
    (catch Exception ex
      (if (= 404 (:status (ex-data ex)))
        (generate-new-key v)
        (throw ex)))))

(defrecord OciVault [client config]
  p/Vault
  (encrypt [this iv txt]
    (vc/encrypt (:encryption-key this) iv txt))

  (decrypt [this iv enc]
    (vc/decrypt (:encryption-key this) iv enc))

  co/Lifecycle
  (start [this]
    (assoc this :encryption-key (load-or-generate-key this)))

  (stop [this]
    this))

(defn make-oci-vault [config]
  (->OciVault (vault/make-client config)
              (select-keys config [:compartment-id :vault-id :key-id :secret-name])))
