(ns monkey.ci.containers.podman
  "Functions for running containers using Podman.  We don't use the api here, because
   it requires a socket, which is not always available.  Instead, we invoke the podman
   command as a child process and communicate with it using the standard i/o streams."
  (:require [babashka.fs :as fs]
            [cheshire.core :as json]
            [clojure.java.io :as io]
            [clojure.string :as cs]
            [clojure.tools.logging :as log]
            [monkey.ci
             [artifacts :as art]
             [build :as b]
             [cache :as cache]
             [containers :as mcc]
             [jobs :as j]
             [process :as proc]
             [protocols :as p]
             [utils :as u]
             [workspace :as ws]]
            [monkey.ci.build.core :as bc]
            [monkey.ci.containers.common :as cc]
            [monkey.ci.events.mailman :as em]
            [monkey.ci.events.mailman.interceptors :as emi]))

;;; Process commandline configuration

(defn- make-script-cmd [script sd]
  (->> (range (count script))
       (map str)
       (into [(str sd "/" cc/job-script)])))

(defn- make-cmd [job sd]
  (if-let [cmd (mcc/cmd job)]
    cmd
    ;; When no command is given, run the script
    (make-script-cmd (:script job) sd)))

(defn- mounts [job]
  (mapcat (fn [[h c]]
            ;; TODO Mount options
            ["-v" (str h ":" c)])
          (mcc/mounts job)))

(defn- env-vars [env]
  (mapcat (fn [[k v]]
            ["-e" (str k "=" v)])
          env))

(defn- platform [job conf]
  (when-let [p (or (mcc/platform job)
                   (:platform conf))]
    ["--platform" p]))

(defn- entrypoint [job]
  (let [ep (mcc/entrypoint job)]
    (cond
      ep
      ["--entrypoint" (str "'" (json/generate-string ep) "'")]
      (nil? (mcc/cmd job))
      ["--entrypoint" "/bin/sh"])))

(defn- get-job-id
  "Creates a string representation of the job sid"
  [job-sid]
  (cs/join "-" job-sid))

(defn- vol-mnt [from to]
  (str from ":" to ":Z"))

(defn- script->files [script dest]
  (fs/create-dirs dest)
  (->> script
       (map-indexed (fn [idx l]
                      (spit (fs/file (fs/path dest (str idx))) l)))
       (doall)))

(defn build-cmd-args
  "Builds command line args for the podman executable"
  [{:keys [job sid] base :work-dir sd :script-dir :as opts}]
  (let [cn (get-job-id sid)
        cwd "/home/monkeyci"
        ext-dir "/opt/monkeyci"
        csd (str ext-dir "/script")
        cld (str ext-dir "/logs")
        wd (if-let [jwd (j/work-dir job)]
             (str (fs/path cwd jwd))
             cwd)
        start "start"
        base-cmd (cond-> ["/usr/bin/podman" "run"
                          "-t"
                          "--name" cn
                          "-v" (vol-mnt base cwd)
                          "-v" (vol-mnt sd csd)
                          "-v" (vol-mnt (:log-dir opts) cld)
                          "-w" wd]
                   ;; Do not delete container in dev mode
                   (not (:dev-mode opts)) (conj "--rm"))
        env {"MONKEYCI_WORK_DIR" wd
             "MONKEYCI_SCRIPT_DIR" csd
             "MONKEYCI_LOG_DIR" cld
             "MONKEYCI_START_FILE" (str csd "/" start)
             "MONKEYCI_ABORT_FILE" (str csd "/abort")
             "MONKEYCI_EVENT_FILE" (str csd "/events.edn")}]
    (when-let [s (:script job)]
      (script->files s sd)
      (io/copy (slurp (io/resource cc/job-script)) (fs/file (fs/path sd cc/job-script)))
      ;; Auto start, so touch the start file immediately
      (let [sf (fs/path sd start)]
        (when-not (fs/exists? sf)
          (fs/create-file sf))))
    (concat
     base-cmd
     (mounts job)
     (env-vars (merge (mcc/env job) env))
     (platform job opts)
     (entrypoint job)
     [(mcc/image job)]
     (make-cmd job csd))))

;;; Mailman event handling

;;; Context management

(def build-sid (comp :sid :event))

(defn get-job
  ([ctx id]
   (some-> (emi/get-state ctx)
           (get-in [:jobs (build-sid ctx) id])))
  ([ctx]
   (get-job ctx (get-in ctx [:event :job-id]))))

(defn set-job [ctx job]
  (emi/update-state ctx assoc-in [:jobs (build-sid ctx) (:id job)] job))

(def get-job-dir
  "The directory where files for this job are put"
  (comp ::job-dir emi/get-state))

(defn set-job-dir [ctx wd]
  (emi/update-state ctx assoc ::job-dir wd))

(def get-work-dir
  "The directory where the container process is run"
  (comp #(fs/path % "work") get-job-dir))

(def get-log-dir
  "The directory where container output is written to"
  (comp #(fs/path % "logs") get-job-dir))

(def get-script-dir
  "The directory where script files are stored"
  (comp #(fs/path % "script") get-job-dir))

(defn job-work-dir [ctx job]
  (let [wd (j/work-dir job)]
    (cond-> (str (get-work-dir ctx))
      wd (u/abs-path wd))))

;;; Interceptors

(defn add-job-dir
  "Adds the directory for the job files in the event to the context"
  [wd]
  {:name ::add-job-dir
   :enter (fn [ctx]
            (->> (conj (build-sid ctx) (get-in ctx [:event :job-id]))
                 (apply fs/path wd)
                 (str)
                 (set-job-dir ctx)))})

(defn restore-ws
  "Prepares the job working directory by restoring the files from the workspace."
  [workspace]
  {:name ::restore-ws
   :enter (fn [ctx]
            (let [dest (fs/create-dirs (get-work-dir ctx))
                  ws (ws/->BlobWorkspace workspace (str dest))]
              (log/debug "Restoring workspace to" dest)
              (assoc ctx ::workspace @(p/restore-workspace ws (build-sid ctx)))))})

(def filter-container-job
  "Interceptor that terminates when the job in the event is not a container job"
  (emi/terminate-when ::filter-container-job
                      #(nil? (mcc/image (get-in % [:event :job])))))

(def save-job
  "Saves job to state for future reference"
  {:name ::save-job
   :enter (fn [ctx]
            (set-job ctx (get-in ctx [:event :job])))})

(def require-job
  "Terminates if no job is present in the state"
  (emi/terminate-when ::require-job #(nil? (get-job % (get-in % [:event :job-id])))))

(defn add-job-ctx
  "Adds the job context to the event context, and adds the job from state.  Also
   updates the build in the context so the checkout dir is the workspace dir."
  [initial-ctx]
  {:name ::add-job-ctx
   :enter (fn [ctx]
            (-> ctx
                (emi/set-job-ctx (-> initial-ctx
                                     (assoc :job (get-job ctx)
                                            :sid (build-sid ctx)
                                            :checkout-dir (str (get-work-dir ctx)))))))})

;;; Event handlers

(def job-executed-evt
  "Creates an internal job-executed event, specifically for podman containers.  This is used
   as an intermediate step to save artifacts."
  (partial j/job-status-evt :podman/job-executed))

(defn prepare-child-cmd
  "Prepares podman command to execute as child process"
  [ctx]
  (let [job (get-in ctx [:event :job])
        log-file (comp fs/file (partial fs/path (fs/create-dirs (get-log-dir ctx))))
        {:keys [job-id sid]} (:event ctx)]
    {:cmd (build-cmd-args {:job job
                           :sid (conj sid job-id)
                           :work-dir (get-work-dir ctx)
                           :log-dir (get-log-dir ctx)
                           :script-dir (get-script-dir ctx)})
     :dir (job-work-dir ctx job)
     :out (log-file "out.log")
     :err (log-file "err.log")
     :exit-fn (proc/exit-fn
               (fn [{:keys [exit]}]
                 (log/info "Container job exited with code" exit)
                 (try
                   (em/post-events (emi/get-mailman ctx)
                                   [(job-executed-evt job-id sid (if (= 0 exit) bc/success bc/failure))])
                   (catch Exception ex
                     (log/error "Failed to post job/executed event" ex)))))}))

(defn job-queued [conf ctx]
  (let [{:keys [job-id sid]} (:event ctx)]
    ;; Podman runs locally, so no credits consumed
    [(j/job-initializing-evt job-id sid (:credit-multiplier conf))]))

(defn job-init [ctx]
  (let [{:keys [job-id sid]} (:event ctx)]
    ;; Ideally the container notifies us when it's running by means of a script,
    ;; similar to the oci sidecar.
    [(j/job-start-evt job-id sid)]))

(defn job-exec
  "Invoked after the podman container has exited.  Posts a job/executed event."
  [{{:keys [job-id sid status result]} :event}]
  [(j/job-executed-evt job-id sid (assoc result :status status))])

(defn- make-job-ctx [conf]
  (-> (select-keys conf [:artifacts :cache])
      (assoc :checkout-dir (b/checkout-dir (:build conf)))))

(defn make-routes [{:keys [workspace work-dir mailman] :as conf}]
  (let [state (emi/with-state (atom {}))
        job-ctx (make-job-ctx conf)
        wd (or work-dir (str (fs/create-temp-dir)))]
    (log/info "Creating podman container routes using work dir" wd)
    [[:container/job-queued
      [{:handler prepare-child-cmd
        :interceptors [emi/handle-job-error
                       state
                       save-job
                       (add-job-dir wd)
                       (restore-ws workspace)
                       (emi/add-mailman mailman)
                       (add-job-ctx job-ctx)
                       (cache/restore-interceptor emi/get-job-ctx)
                       (art/restore-interceptor emi/get-job-ctx)
                       emi/start-process]}
       {:handler (partial job-queued conf)}]]

     [:job/initializing
      ;; TODO Start polling for events from events.edn
      [{:handler job-init
        :interceptors [emi/handle-job-error
                       state
                       require-job]}]]

     [:podman/job-executed
      [{:handler job-exec
        :interceptors [emi/handle-job-error
                       state
                       (add-job-dir wd)
                       (add-job-ctx job-ctx)
                       (art/save-interceptor emi/get-job-ctx)
                       (cache/save-interceptor emi/get-job-ctx)]}]]]))
