(ns monkey.ci.vault
  "Functions related to encryption/decryption of data using a vault"
  (:require [clojure.tools.logging :as log]
            [buddy.core
             [codecs :as codecs]
             [crypto :as bcc]
             [nonce :as bcn]]
            [com.stuartsierra.component :as co]
            [monkey.ci.protocols :as p]
            [monkey.oci.vault :as vault]
            [monkey.oci.vault.b64 :as b]))

(def key-shape {:algorithm "AES"
                :length 32})
;; Key size determines algorithm and iv length
(def algo {:algo :aes-256-gcm})
(def iv-size 16)

(defn- load-enc-key
  "Loads encryption key from the configured vault secret"
  [{:keys [client config]}]
  (log/debug "Loading secret" (:secret-name config) "from vault" (:vault-id config))
  (-> (vault/get-secret-bundle-by-name client (select-keys config [:secret-name :vault-id :version-number]))
      :secret-bundle-content
      :content
      (b/b64->)))

(defn- generate-new-key [{:keys [client config]}]
  (log/debug "Generating new encryption key using master key" (:key-id config))
  (let [gen (-> (vault/generate-data-encryption-key
                 client
                 (-> (select-keys config [:key-id])
                     (assoc :key-shape key-shape)))
                :plaintext)]
    (log/debug "Storing generated key in secret" (:secret-name config))
    (vault/create-secret client (-> (select-keys config [:secret-name :compartment-id :vault-id :key-id])
                                    (assoc :description "Encryption key generated by MonkeyCI"
                                           :secret-content
                                           {:content-type "BASE64"
                                            :content gen})))
    (b/b64-> gen)))

(defn- load-or-generate-key [v]
  (try
    (load-enc-key v)
    (catch Exception ex
      (if (= 404 (:status (ex-data ex)))
        (generate-new-key v)
        (throw ex)))))

(defn- encrypt [enc-key iv txt]
  (-> (bcc/encrypt (codecs/str->bytes txt)
                   enc-key
                   iv
                   algo)
      (codecs/bytes->b64-str)))

(defn- decrypt [enc-key iv enc]
  (-> (bcc/decrypt (codecs/b64->bytes enc)
                   enc-key
                   iv
                   algo)
      (codecs/bytes->str)))

(defrecord OciVault [client config]
  p/Vault
  (encrypt [this iv txt]
    (encrypt (:encryption-key this) iv txt))

  (decrypt [this iv enc]
    (decrypt (:encryption-key this) iv enc))

  co/Lifecycle
  (start [this]
    (assoc this :encryption-key (load-or-generate-key this)))

  (stop [this]
    this))

(defn make-oci-vault [config]
  (->OciVault (vault/make-client config)
              (select-keys config [:compartment-id :vault-id :key-id :secret-name])))

;; Fixed key vault, that uses a preconfigured key.  Useful for testing or developing.
(defrecord FixedKeyVault [encryption-key]
  p/Vault
  (encrypt [_ iv txt]
    (encrypt encryption-key iv txt))
  
  (decrypt [_ iv enc]
    (decrypt encryption-key iv enc)))

(defn generate-key
  "Generates random encryption key"
  []
  (bcn/random-nonce 32))

(defn make-fixed-key-vault [config]
  (->FixedKeyVault (or (:encryption-key config) (generate-key))))

(defmulti make-vault :type)

(defmethod make-vault :oci [config]
  (make-oci-vault config))

(defmethod make-vault :fixed [config]
  (make-fixed-key-vault config))

(defn generate-iv
  "Generates a random initialization vector for AES encryption"
  []
  (bcn/random-nonce iv-size))
