(ns monkey.ci.runners.controller
  "Functions for running the application as a controller."
  (:require [babashka.fs :as fs]
            [clojure.tools.logging :as log]
            [monkey.ci
             [build :as b]
             [runners :as r]
             [script :as script]]
            [monkey.ci.events.core :as ec]))

(def run-path (comp :run-path :config))
(def abort-path (comp :abort-path :config))

(defn- post-init-evt [{:keys [build] :as rt}]
  (ec/post-events (:events rt) (script/script-init-evt build (b/script-dir build)))
  rt)

(defn- create-run-file [rt]
  (log/debug "Creating run file at:" (run-path rt))
  (some-> (run-path rt)
          (fs/create-file))
  rt)

(defn- create-abort-file [rt]
  (log/debug "Creating abort file at:" (abort-path rt))
  (some-> (abort-path rt)
          (fs/create-file))
  rt)

(defn- download-and-store-src [rt]
  (assoc rt :build (-> (:build rt)
                       (r/download-src rt)
                       (r/store-src rt))))

(defn- restore-build-cache [rt]
  ;; TODO
  rt)

(defn- save-build-cache [rt]
  ;; TODO
  rt)

(defn- wait-until-run-file-deleted [rt]
  (let [rp (run-path rt)]
    (log/debug "Waiting until run file has been deleted at:" rp)
    (while (fs/exists? rp)
      (Thread/sleep 500))
    (log/debug "Run file deleted, build script finished."))
  rt)

(defn- post-end-evt [{:keys [build] :as rt}]
  ;; TODO Get script exit code somehow (through a file?)
  (ec/post-events (:events rt) (b/build-end-evt build 0))
  rt)

(defn run-controller [rt]
  (try
    (-> rt
        (post-init-evt)
        (download-and-store-src)
        (restore-build-cache)
        (create-run-file)
        (wait-until-run-file-deleted)
        (post-end-evt)
        (save-build-cache))
    (catch Throwable ex
      (log/error "Failed to run controller" ex)
      (create-abort-file rt)
      1)))
