(ns msprandom.keyboard
  (:gen-class)
  (:import (ConsoleReader.)
           (jline.console ConsoleReader))
  (:require [msprandom.crypto]))

(defn read-scan-code
  "This function performs unbuffered single character reading from a console.
  Returns integer value (scan code) of pressed button."
  []
  (flush)
  (let [console-reader (ConsoleReader.)]
    (.readCharacter console-reader)))

(defn get-current-timestamp-nanoseconds
  "This function returns current timestamp in nanoseconds."
  []
  (System/nanoTime))

(defn counter-runner
  "This is smart function runs infinite cycle and increments given (atom) counter every tic.
  When man press key button counter value should be read by external function to get random byte value.
  If Atom go-flag? = false, it breaks infinite cycle to stop the thread."
  [counter
   go-flag?]
  (while @go-flag?
    (swap! counter inc)
    (if (>= @counter 255)
      (reset! counter 0))))

(defn get-random-bytes-from-keyboard
  "This function returns String with random hex bytes produced from console user input.
  This function can produce max 32 bytes per call. (you can request from 1..32)
  Suppose that every user button press can give us 3 bits of random.
  So, to get true random 32 bytes we need ~ 32*3 button press from user input.
  At the final step collected values are hashed with strong 256 bit function GOST 3411-94 to
  produce the final output."
  [bytes-needed
   need-print-progress?]

  (let [counter (atom 0)    ;this counter is incremented every tic
        go-flag? (atom true)]

    ;;run seaprate thread where counter increments every tic
    (future (counter-runner counter go-flag?))

    (loop [collected-values-vec [(str (get-current-timestamp-nanoseconds))]
           collected-bits-number 0]

      ;if we collected required amount of random data then do final step -> hash them to guarantee that probability
      ; 1 and 0 bits will be 0.5
      (if (>= collected-bits-number (* bytes-needed 8))
        (let [ ;_ (println (apply str collected-values-vec))

               ;stop runner thread
               _ (reset! go-flag? false)

               ;;call GOST3411 to mix bits in collected random data.
               random-string (msprandom.crypto/get-hash-from-string (apply str collected-values-vec))]

          (if (>= bytes-needed 32)
            random-string
            (subs random-string 0 (* 2 bytes-needed))))

        ;else collect new random values
        (do
          (when need-print-progress?
            (print (format "=====>%.2f%% done.   \r" (* 100 (/ collected-bits-number (* bytes-needed 8.0))))))
          (do
            (let [
                  ;;wait for user input and read scan code
                  next-keyboard-scancode (read-scan-code)
                  ;;when user press button read current counter value which changes in separate
                  ;; thread in 0..255 in infinite loop
                  next-random-tic-value @counter

                  ;;read current value of nanoseconds from start of Epoch and take mod 256
                  ;;to convert its value to random byte
                  current-nanosec-value (mod (get-current-timestamp-nanoseconds) 256)

                  ;xor random values to produce final result = random byte.
                  ;but we suppose that we got only 3 true random bits to guarantee that final result will have required amount of random bits.
                  result (bit-xor next-keyboard-scancode next-random-tic-value current-nanosec-value)
                  ;_ (println "result:" result " next scan code:" next-keyboard-scancode " next tic value:" next-random-tic-value " current nanosec:" current-nanosec-value)
                  ]
              (recur
                (conj collected-values-vec result)
                (+ 3 collected-bits-number)))))))))