(ns msprandom.crypto
  (:gen-class)
  (:import (org.bouncycastle.util.encoders Hex)
           (org.bouncycastle.crypto.digests GOST3411Digest)
           (org.bouncycastle.jce.provider BouncyCastleProvider)
           (java.security Security)
           (org.bouncycastle.crypto.macs HMac)
           (org.bouncycastle.crypto.params KeyParameter)
           (org.bouncycastle.crypto.generators PKCS5S1ParametersGenerator)
           (javax.crypto.spec SecretKeySpec IvParameterSpec)
           (javax.crypto Cipher CipherOutputStream CipherInputStream)
           (java.io ByteArrayOutputStream ByteArrayInputStream DataInputStream)))

(defn break-jce-policy-limit
  "This function breaks JCE crypto limits. Should be run once, primarily at the begining of the program
  to avoid JCE policy limit if JDK/JRE runtime has no installed files for break crypto limit. Returns nil."
  []
  (let [field (-> (Class/forName "javax.crypto.JceSecurity")
                  (.getDeclaredField "isRestricted"))]
    (.setAccessible field true)
    (.set field nil java.lang.Boolean/FALSE)))


(defn get-hash-from-string
  "This function returns String that represents hash value in hex form for a given input string."
  [^String s]
  (break-jce-policy-limit)
  (Security/addProvider (BouncyCastleProvider.))
  (let [digest (GOST3411Digest.)
        s-bytes (.getBytes s)
        _ (.update digest s-bytes 0 (alength s-bytes))
        hash-buffer (byte-array (.getDigestSize digest))
        _ (.doFinal digest hash-buffer 0)]
    (Hex/toHexString hash-buffer)))

(defn get-hash-from-bytes
  "This function returns bytes[] that represents hash value for a given input byte []."
  [data]
  (break-jce-policy-limit)
  (Security/addProvider (BouncyCastleProvider.))
  (let [digest (GOST3411Digest.)
        s-bytes data
        _ (.update digest s-bytes 0 (alength s-bytes))
        hash-buffer (byte-array (.getDigestSize digest))
        _ (.doFinal digest hash-buffer 0)]
    hash-buffer))

(defn get-hmac-from-string
  "This function calculates a HMAC using GOST3411-94 using given String seed value .
  Seed and data should be in String form.
  Returns String of hmac value in hex representation of bytes."
  [^String hmac-seed
   ^String data]
  (break-jce-policy-limit)
  (let [hmac-fn (HMac. (GOST3411Digest.))
        key-param (KeyParameter. (PKCS5S1ParametersGenerator/PKCS5PasswordToUTF8Bytes (.toCharArray hmac-seed)))
        _ (.init hmac-fn key-param)
        data-bytes (.getBytes data)
        _ (.update hmac-fn data-bytes 0 (alength data-bytes))
        hmac-byte-array (byte-array (.getMacSize hmac-fn))
        _ (.doFinal hmac-fn hmac-byte-array 0)]
    (Hex/toHexString hmac-byte-array)))

(defn get-hmac-from-bytes
  "This function calculates a HMAC using GOST3411-94 using given byte [] seed.
  Seed and data should be in byte [] form.
  Returns byte[] of hmac value, length 32 bytes."
  [hmac-seed
   data]
  (break-jce-policy-limit)
  (let [hmac-fn (HMac. (GOST3411Digest.))
        key-param (KeyParameter. hmac-seed)
        _ (.init hmac-fn key-param)
        _ (.update hmac-fn data 0 (alength data))
        hmac-byte-array (byte-array (.getMacSize hmac-fn))
        _ (.doFinal hmac-fn hmac-byte-array 0)]
    hmac-byte-array))

(defn encrypt-string-in-cfb-mode
  "This function encrypts input String using String key in hex and String iv in hex.
  Returns String - encrypted text in hex."
  [^String keyString
   ^String iv
   ^String data]
  (break-jce-policy-limit)
  (let [data-bytes (.getBytes data)
        key (SecretKeySpec. (Hex/decode keyString) "GOST28147")
        cipher (Cipher/getInstance "GOST28147/CFB8/NoPadding" "BC")
        _ (.init cipher Cipher/ENCRYPT_MODE key (IvParameterSpec. (Hex/decode iv)))
        bOut (ByteArrayOutputStream.)
        cOut (CipherOutputStream. bOut cipher)
        _ (.write cOut data-bytes 0 (alength data-bytes))
        _ (.close cOut)]
    (Hex/toHexString (.toByteArray bOut))))

(defn decrypt-string-in-cfb-mode
  "This function decrypts input String in hex using String key in hex and String iv in hex.
  Returns ecrypted String"
  [^String keyString
   ^String iv
   ^String data]
  (break-jce-policy-limit)
  (let [data-bytes (Hex/decode data)
        decrypted-data-array (byte-array (alength data-bytes))
        key (SecretKeySpec. (Hex/decode keyString) "GOST28147")
        cipher (Cipher/getInstance "GOST28147/CFB8/NoPadding" "BC")
        _ (.init cipher Cipher/DECRYPT_MODE key (IvParameterSpec. (Hex/decode iv)))
        bIn (ByteArrayInputStream. data-bytes)
        cIn (CipherInputStream. bIn cipher)
        dIn (DataInputStream. cIn)
        _ (.readFully dIn decrypted-data-array )
        _ (.close bIn)]
    (String. decrypted-data-array)))

(defn encrypt-bytes-in-cfb-mode
  "This function encrypts input bytes[] using bytes[32] key in hex and bytes[8] iv in hex.
  Returns bytes[] - encrypted data."
  [key-data
   iv
   data]
  (break-jce-policy-limit)
  (let [key (SecretKeySpec. key-data "GOST28147")
        cipher (Cipher/getInstance "GOST28147/CFB8/NoPadding" "BC")
        _ (.init cipher Cipher/ENCRYPT_MODE key (IvParameterSpec. iv))
        bOut (ByteArrayOutputStream.)
        cOut (CipherOutputStream. bOut cipher)
        _ (.write cOut data 0 (alength data))
        _ (.close cOut)]
    (.toByteArray bOut)))

(defn decrypt-bytes-in-cfb-mode
  "This function decrypts input bytes[] using bytes[32] key and bytes[8] iv.
  Returns decrypted data in a bytes[] form."
  [key-data
   iv
   data]
  (break-jce-policy-limit)
  (let [decrypted-data-array (byte-array (alength data))
        key (SecretKeySpec. key-data "GOST28147")
        cipher (Cipher/getInstance "GOST28147/CFB8/NoPadding" "BC")
        _ (.init cipher Cipher/DECRYPT_MODE key (IvParameterSpec. iv))
        bIn (ByteArrayInputStream. data)
        cIn (CipherInputStream. bIn cipher)
        dIn (DataInputStream. cIn)
        _ (.readFully dIn decrypted-data-array )
        _ (.close bIn)]
    decrypted-data-array))
