(ns msprandom.core
  (:gen-class)
  (:import (org.bouncycastle.util.encoders Hex))
  (:require [msprandom.crypto]
            [msprandom.keyboard]
            [clojure.edn :as edn]

            ;;logging features
            [clojure.tools.logging :as log]))


(defmacro safe [bindings? & forms]
  "This macro is used to execute any function inside try-catch block."
  (let [bindings (if (and (even? (count bindings?)) (vector? bindings?))
                   bindings? nil)
        forms (if bindings forms (cons bindings? forms))
        except `(catch Exception e# e#
                       (log/error (.getMessage e#)))]
    (if bindings
      `(let ~bindings (try ~@forms ~except))
      `(try ~@forms ~except))))

;low level
(defn read-edn-file
  "This function reads (safely)  file in edn format and returns its content as clojure object."
  [file-name]
  (let [o (safe (edn/read-string (slurp file-name)))]
    (if-not (nil? o)
      o)))

;low level function
(defn get-true-random-bytes-as-string
  "This function collects true random data. This function works only in console mode.
  Returns String (length N*2 bytes ) with hex values of random bytes."
  [random-bytes-num
   print-progress?]
  (if (> random-bytes-num 32)
    (do
      (loop [out-vec []
             bytes-remain random-bytes-num]
        (if (<= bytes-remain 0)
          (apply str out-vec)
          (do
            (when print-progress?
              (println "next 32 bytes step."))
            (if (<= bytes-remain 32)
              (recur
                (conj out-vec (msprandom.keyboard/get-random-bytes-from-keyboard bytes-remain print-progress?))
                0)
              (recur
                (conj out-vec (msprandom.keyboard/get-random-bytes-from-keyboard 32 print-progress?))
                (- bytes-remain 32)))))))
    (do
      (let [result (msprandom.keyboard/get-random-bytes-from-keyboard random-bytes-num print-progress?)]
        (subs result 0 (* 2 random-bytes-num))))))

;low level function
(defn get-true-random-bytes-as-bytes
  "This function collects true random data. This function works only in console mode.
  Returns bytes[] array of random values."
  [random-bytes-num
   print-progress?]
  (let [rand-from-user (get-true-random-bytes-as-string random-bytes-num print-progress?)]
    (Hex/decode rand-from-user)))

;this is high level function
(defn create-true-random-numbers-vault
  "This function generates 512 true random bytes and save them to a local file with given filename.
  Random data is encrypted with String password.
  This function works only in console mode.
  Before saving random data a HMAC is calculated to protect an encrypted random data using given String password.
  512 bytes of random data + HMAC are saved to a local file."
  [filename
   string-password]
  (let [random-data (get-true-random-bytes-as-bytes 64 true)
        key (msprandom.crypto/get-hash-from-bytes (.getBytes string-password))
        sync (Hex/decode (subs (msprandom.crypto/get-hash-from-string (Hex/toHexString key)) 0 16))
        encrypted-random-data (msprandom.crypto/encrypt-bytes-in-cfb-mode key sync random-data)
        hmac (msprandom.crypto/get-hmac-from-bytes key encrypted-random-data)
        result (str {:data (Hex/toHexString encrypted-random-data) :hmac (Hex/toHexString hmac)})]
    (safe (spit filename result :append false))))

;low level function.
(defn update-true-random-numbers-vault
  "This function updates random data  in a vault. This function should be run every start of program which uses this vault.
  Updating random data is necessary to avoid usage the same random data twice.
  Before saving random data a HMAC is calculated to protect an encrypted random data using given String password.
  512 bytes of random data + HMAC are saved to a local file."
  [filename
   string-password
   random-data]
  (let [
         nanosec1 (msprandom.keyboard/get-current-timestamp-nanoseconds)
         random-as-string (Hex/toHexString random-data)
         half1 (subs random-as-string 0 64)
         half2 (subs random-as-string 64 128)
         hash1-string (msprandom.crypto/get-hash-from-string (str half1 nanosec1))
         nanosec2 (msprandom.keyboard/get-current-timestamp-nanoseconds)
         hash2-string (msprandom.crypto/get-hash-from-string (str half2 nanosec2))
         new-random-data-string (str hash1-string hash2-string)
         new-random (Hex/decode new-random-data-string)
         key (msprandom.crypto/get-hash-from-bytes (.getBytes string-password))
         sync (Hex/decode (subs (msprandom.crypto/get-hash-from-string (Hex/toHexString key)) 0 16))
         encrypted-random-data (msprandom.crypto/encrypt-bytes-in-cfb-mode key sync new-random)
         hmac (msprandom.crypto/get-hmac-from-bytes key encrypted-random-data)
         result (str {:data (Hex/toHexString encrypted-random-data) :hmac (Hex/toHexString hmac)})]
    (safe (spit filename result :append false))))

;this is high level function
(defn load-true-random-numbers-vault
  "This function loads 512 true random bytes from a local file with given filename.
Random data is decrypted with String password.
Before loading random data a HMAC is calculated and compared with HMAC in file to detect changes.
If HMACs are equal return random data else nil. After successful reading of random data random vault is updated to
to avoid usage the same random data twice."
  [filename
   string-password]
  (let [key-bytes (msprandom.crypto/get-hash-from-bytes (.getBytes string-password))
        sync-bytes (Hex/decode (subs (msprandom.crypto/get-hash-from-string (Hex/toHexString key-bytes)) 0 16))
        data (read-edn-file filename)]
    (if data
      (do
        (let [encrypted-random-data-bytes (Hex/decode (data :data))
              hmac-from-file-bytes (Hex/decode (data :hmac))
              hmac-bytes (msprandom.crypto/get-hmac-from-bytes key-bytes encrypted-random-data-bytes)]
          (if (= (Hex/toHexString hmac-bytes) (Hex/toHexString hmac-from-file-bytes))
            (do
              (let [result (msprandom.crypto/decrypt-bytes-in-cfb-mode key-bytes sync-bytes encrypted-random-data-bytes)
                    _ (when result
                        (update-true-random-numbers-vault filename string-password result))]
                result))
            (do
              (log/error "Vault is corrupted and cannon be used or wrong password!")))))
      (do
        (log/error "Can't load random vault.")))))

;this is high level function
(defn secure-rand
  "This function generates strong random sequence of bytes using PRNG based on GOST28147-89 in CFB mode.
  As a seed of this function should be used true random number 512 bit.
  Returns array of bytes filled by random data sizeof random-num-required."
  [random-data-bytes
   random-num-required]
  (let [half-selector (rand-int 2)
        nanosec (msprandom.keyboard/get-current-timestamp-nanoseconds)
        random-as-string (Hex/toHexString random-data-bytes)
        temp-hash (msprandom.crypto/get-hash-from-string (str random-as-string nanosec))
        derrived-key-string (if (= half-selector 1)
                              (msprandom.crypto/get-hash-from-string (str temp-hash (subs random-as-string 64 128)))
                              (msprandom.crypto/get-hash-from-string (str temp-hash (subs random-as-string 0 64))))
        empty-byte-array (byte-array random-num-required)]
    (msprandom.crypto/encrypt-bytes-in-cfb-mode (Hex/decode derrived-key-string)
                                                (Hex/decode (subs temp-hash 0 16))
                                                empty-byte-array)))
