# msprandom

This library demonstrates a technique which allows to solve the problem of getting random numbers on 
computers and small devices for cryptographic purposes. So, if you need encrypt and sign data on a small device or 
computer without hardware RNG do the following:

1. Create true random numbers vault using this library on your computer or notebook.
2. Put this vault on your device, computer or server where you need encrypt and sign data.
3. Load the vault once at the start of the program when you need encrypt or sign data.
4. Call secure-rand function to get random bytes as many times as you need.

Vault is encrypted and secured with HMAC.
Random data in a vault is updated every time you load random seed with unpredictable way, so HMAC is recalculated too.

## Gathering a true random data

To get a true random data a human input is used. Algorithm of collecting a random data:

1. Run separate thread where atomic counter increments every tic from 0..255 with a very high speed.
2. Wait for unbuffered key press by human and get a scan code of pressed button.
3. Take current nanoseconds value from start of Epoch and take mod 256 to convert its value to a random byte.
4. Xor values between each other: scan-code-byte ^ current-counter-value ^ nanoseconds to produce random byte.
5. Add random byte to output vector. We suppose that only 3 bits has true randomness in this random byte. So, to get 
   true random 32 bytes we need ~ 32*3 button press from user input.
6  Repeat steps 2-5 until we get required amount of random bytes. 
7. If we collected required amount of random data then do final step -> hash output vector with cryptographically strong 
   hash function GOST 3411-94 to guarantee that probability 1 and 0 bits in output vector will be 0.5. 
   Note, that hash function used here only to mix random bits and do not influence to the quality of random data. 
   So hash(random data) = random data. Hash will produce vector of 32 bytes containing a random data.
8. Repeat steps 1..7 if we need more than 32 bytes of random data.

Using this algorithm we collect a true 512 random bits. Why 512? 
Well, every PRNG needs a true random seed. If an attacker knows a seed then you can't protect your data. 
256 bit length is far enough to keep millitary grade secrets.
I did 512 to close the security question of random seed. 
My opinion, 512 bit of true random data is enough to use in PRNG: generating keys, signatures, etc.

## Usage

Get ready msprandom-0.1.jar or compile the library msprandom with `lein uberjar` command. 
Then, go to target/ubarjar folder and take ready msprandom-0.1.jar file for usage. (don't use uberjar file from target folder)

1. Create new project `lein new app rand-tester`

2. Add to your project.clj the following lines:
```
:dependencies [[org.clojure/clojure "1.6.0"]

                 [com.middlesphere/msprandom "0.1"]

                 ;;;msprandom dependencies
                 ;;encryption engine
                 [org.bouncycastle/bcprov-jdk15on "1.50"]
                 ;keyboard library
                 [jline "2.11"]
                 ]
                 
  :manifest { "Class-Path" "lib/bcprov-jdk15on-1.50.jar lib/clojure-1.6.0.jar lib/jline-2.11.jar lib/msprandom-0.1.jar"}
```
 Bouncycastle library should be as a separate jar file, not inside jar. 
 It cannot be placed in uberjar due to jar signing (it will not work).
 So, in manifest section of rand-tester jar we tell that bouncycastle will be in a ./lib folder, and also jline 
 and msprandom libraries too.
* Note, that msprandom library should be in one folder with  bouncycastle and jline libraries. (see manifest.mf in msprandom jar)*
 
### Create random data vault
 
 Before you can get random numbers with good quality you should create a true random numbers and put them in a vault.
 Here is the code snippet how to create true random numbers and put them to a secured vault.
``` 
 (ns rand-tester.core
   (:gen-class)
   (:import (org.bouncycastle.util.encoders Hex))
   (:require [msprandom.core]))
 
 (defn -main
   "Entry point."
   [& args]
   (println "Please press some buttons on keyboard.")
   (msprandom.core/create-true-random-numbers-vault "random-vault.edn" "VeryStrongPassword129")
   (println "Random data vault created successfuly!")
   (System/exit 0))
```
   
This code should be run in console mode due to unbuffered console input is used to collect random data.
Generated random data 512 bit length is encrypted with given password and HMAC is added to detect any changes.

### Get random data

To get good random data you should:

1. Load true random seed from vault. 

* Warning, call this function only 1 time at the start of the program. *

2. To produce random data call msprandom.core/secure-rand function to get random data based on seed 512 bits length.
This function generates strong random sequence of bytes using PRNG based on GOST28147-89 in CFB mode.
Here is the code snippet how to get 1000 bytes of random data.

```
(ns rand-tester.core
  (:gen-class)
  (:import (org.bouncycastle.util.encoders Hex))
  (:require [msprandom.core]))

(defn -main
  "Entry point."
  [& args]
  (let [ r (msprandom.core/load-true-random-numbers-vault "random-vault.edn" "VeryStrongPassword129")]
    (if r
      (do
        (println (Hex/toHexString (msprandom.core/secure-rand r 1000))))
      (println "Can't load random data.")))
  (System/exit 0))
```
You can call msprandom.core/secure-rand as many times as you need.

** After successful reading of random data a random vault is updated to to avoid usage the same random data twice.
Strong cryptography hash function is used to derive new random data.
So, next time  you load  random data from a vault you will get different value of true random seed. **  
 

## License

Copyright © 2014 Middlepshere LLC.

Distributed under the Eclipse Public License either version 1.0 or any later version.

