(ns clj-audio.pcm-audio-recorder
  (:gen-class)

  ;; audio low level functions
  (:use [clj-audio.low-level]
        [clj-audio.common])

  ;;logging features
  (:require [clojure.tools.logging :as log])

  ;;io streams
  (:use [clojure.java.io :only [output-stream]]))


(defn get-audio-params
  "This function reads config file in edn format. Audio parameters in config file should be in a map form.
  Example of config: {:sample-rate 16000 :sample-size-bits 16 :channels-number 2 :signed? true :bigEndian? false }.
  Return: initialized audio format object or nil"
  [config-file-name]
  (if-not (nil? config-file-name)
    (create-audio-format (let [params (read-config-file config-file-name)]
                           (if-not (nil? params)
                             params
                             (log/warn "Can't initialize an audio object."))))))

(defn init-audio-format
  "This function initialize audio-format object with given audio parameters.
  Audio parameters should be map. Example {:sample-rate 16000 :sample-size-bits 16 :channels-number 2 :signed? true :bigEndian? false }.
  Return: initialized audio format object or nil."
  [params]
  (if-not (nil? params)
    (create-audio-format params)
    (log/warn "Can't initialize an audio object.")))

(defn init-target-data-line
  "This function perform initialization of target data line object using audio parameters (see get-audio-params).
  Retrun: target data line object (microphone interface)  or nil"
  [audio-params]
  (let [line (create-data-line audio-params)]
    (if-not (check-data-line-supported? line)
      (log/warn "Warning! Audio parameters are not supported by this hardware. Try to change audio parameters in config file.")
      (get-target-data-line line))))

(defn start-record-audio-to-rawpcm-file
  "This function perfoms recording of raw audio stream from microphone to a binary (PCM) file.
  Stop flag is used to stop recording. After stop - output file will be closed and sound resources will be relesed."
  [config-audio-filename ;;configuration file with audio parameters in edn format
   output-filename ;;name of output binary file name
   stop-flag?] ;;boolean atom to stop audio recording
  (safe
    (with-open [out-stream (output-stream output-filename)]
      (let [params (get-audio-params config-audio-filename)
            target-data-line (init-target-data-line params)]
        (start-capture-audio stop-flag? target-data-line params out-stream)))))
