(ns clj-audio.low-level
  (:gen-class)
  ;;logging features
  (:require [clojure.tools.logging :as log]))

(defn create-audio-format
  "This function creates audio format object which can be used for audio recording functions.
  Parameter p is a parameters map, example: {:sample-rate 44100 :sample-size-bits 16 :channels-number 2 :signed? true :bigEndian? false }
  is used for sound parameter initializaion.
  Return: audio format object (^javax.sound.sampled.AudioFormat) or nil."
  [p]
  (if-not (nil? p)
    (javax.sound.sampled.AudioFormat.
      (float (p :sample-rate))
      (int (p :sample-size-bits))
      (int (p :channels-number))
      (boolean (p :signed?))
      (boolean (p :bigEndian?)))))

(defn create-data-line
  "This function creates data line object from the specified audio format object, which includes a set of supported
  audio formats and a range for the buffer size. This object is typically used by mixer implementations when returning
  information about a supported line.
  Return: data line object or nil."
  [audio-format]
  (if-not (nil? audio-format)
    (javax.sound.sampled.DataLine$Info. javax.sound.sampled.TargetDataLine audio-format)))

(defn check-data-line-supported?
  "This function checks that line supports audio-format specified in create-data-line function.
  Return: true - line is supports requested audio format, false - format is not supported."
  [data-line]
  (if-not (nil? data-line)
    (javax.sound.sampled.AudioSystem/isLineSupported data-line)
    false))

(defn get-target-data-line
  "This function cast data line object to a target data line object created by create-data-line function.
  It is interface for recording device.
  Return target data line object or nil"
  [data-line]
  (if-not (nil? data-line)
    (cast javax.sound.sampled.TargetDataLine (javax.sound.sampled.AudioSystem/getLine data-line))))

(defn open-target-data-line
  "This function opens a target data line (TargetDataLine) object for recording with given audio format.
  Return: void"
  [^javax.sound.sampled.TargetDataLine target-data-line
   ^javax.sound.sampled.AudioFormat audio-format]
  (if (and (not (nil? target-data-line))
        (not (nil? target-data-line)))
    (. target-data-line (open) audio-format)))

(defn target-data-line-is-open?
  "This function checks a target data line for open.
  Return: true - line is opened, false - not opened."
  [^javax.sound.sampled.TargetDataLine data-line]
  (if-not (nil? data-line)
    (.isOpen data-line)
    false))

(defn close-target-data-line
  "This function closes an opened target data line object.
  Return: void"
  [^javax.sound.sampled.TargetDataLine data-line]
  (if-not (nil? data-line)
    (.close data-line)))

(defn start-target-data-line
  "This function starts a line (TargetDataLine) object for recording.
  Return: void"
  [^javax.sound.sampled.TargetDataLine target-data-line]
  (if-not (nil? target-data-line)
    (.start target-data-line)))

(defn stop-target-data-line
  "This function stops previously started a target data line object for recording.
  Return: void"
  [^javax.sound.sampled.TargetDataLine target-data-line]
  (if-not (nil? target-data-line)
    (.stop target-data-line)))


(defn get-audio-input-stream
  "Constructs an audio input stream object that can read data from the target data line.
  Return: AudioInputStream object."
  [^javax.sound.sampled.TargetDataLine data-line]
  (if-not (nil? data-line)
    (javax.sound.sampled.AudioInputStream. data-line)))

(defn close-audio-input-stream
  "Closes given audio input stream (created by get-audio-input-stream) and releases any system resources associated with the stream.
  Return: void"
  [^javax.sound.sampled.AudioInputStream audio-stream]
  (if-not (nil? audio-stream)
    (.close audio-stream)))

(defn calculate-audio-buffer-length
  "Calculate optimal buffer length for captured chunk of data from  audio stream.
  Return: optimal value of buffer length in bytes or nil."
  [^javax.sound.sampled.AudioFormat audio-format
   ^javax.sound.sampled.TargetDataLine target-data-line]
  (if (and
        (not (nil? audio-format))
        (not (nil? target-data-line)))
    (* (/ (.getBufferSize target-data-line) 8) (.getFrameSize audio-format))))

(defn read-audio-stream
  "This function reads a chunk of audio data from stream (target data line which represents a microphone)
  but not more than buffer-length bytes.
  Return: integer value the number of bytes actually read"
  [^javax.sound.sampled.TargetDataLine data-line
   audio-buffer
   buffer-length]
  (if (and
        (not (nil? data-line))
        (not (nil? audio-buffer))
        (not (nil? buffer-length)))
    (.read data-line audio-buffer 0 buffer-length)))

(defn write-audio-data-to-byte-stream
  "This function writes a chunk of captured audio data to output byte stream.
   Return: nil"
  [^java.io.OutputStream output-byte-stream
   audio-buffer
   bytes-captured-length]
  (if (and
        (instance? java.io.OutputStream output-byte-stream)
        (not (nil? audio-buffer))
        (not (nil? bytes-captured-length)))
    (do
      (.write output-byte-stream audio-buffer 0 bytes-captured-length)
      (log/debug (str "Bytes written " bytes-captured-length)))))


(defn stop-capture-audio
  "This functions stops capture audio data. It stops data-line and closes target data line.
  This is internal function and should not be called directly (call only from start-capture-audio)
  Return: nil"
  [^javax.sound.sampled.TargetDataLine data-line]
  (if-not (nil? data-line)
    (do
      (stop-target-data-line data-line)
      (close-target-data-line data-line)
      (log/debug "Capture audio - stop."))))


(defn start-capture-audio
  "This functions starts capture audio data and writes audio data chunks to an output byte stream.
  Stop flag (atom stop-flag?) is used to stop infinite loop of sound capturing and should be triggered from another stream.
  Target data line may be open or not. After event <stop-flag? is true> data-line will be stopped and closed automatically.
  Output byte stream should be closed outside of this function.
  Return: total captured size in bytes"
  [stop-flag?
   data-line
   audio-format
   output-byte-stream]
  (if (and
        (not (nil? stop-flag?))
        (not (nil? data-line))
        (not (nil? audio-format))
        (not (nil? output-byte-stream)))
    (do
      (if-not (target-data-line-is-open? data-line)
        (open-target-data-line data-line audio-format))

      (start-target-data-line data-line)
      (let [audio-buffer-length (calculate-audio-buffer-length audio-format data-line)
            audio-buffer (byte-array audio-buffer-length)
            total-captured-size (atom 0)]
        (log/debug "Capture audio - start.")

        (while (not @stop-flag?)
          (do
            (let [bytes-captured-length (read-audio-stream data-line audio-buffer audio-buffer-length)]
              (swap! total-captured-size #(+ % bytes-captured-length))
              (log/debug (str "Total captured size " @total-captured-size " bytes of raw audio data."))
              (write-audio-data-to-byte-stream output-byte-stream audio-buffer bytes-captured-length))))

        (stop-capture-audio data-line)
        @total-captured-size))
    (log/warn "Capturing audio data is not started. Some of the parameters are nil.")))


