(ns clj-audio.spx-audio-recorder
  (:gen-class)
  (:use [clj-audio.low-level]
        [clj-audio.pcm-audio-recorder]
        [clj-audio.common])
  ;;logging features
  (:require [clojure.tools.logging :as log]
            ;i/o
            [clojure.java.io :refer [file input-stream]])
  ;;Speex Encoder
  (:import [org.xiph.speex.SpeexEncoder]
           [org.xiph.speex.OggSpeexWriter]
           [java.io File])
  (:use [clojure.java.io :only [input-stream]])
  (:import java.nio.ByteBuffer [java.nio ByteOrder]))

(defn init-audio
  "This is internal function for initialization of target data line (prepares microphone for 'ready to recording' state).
  Audio parameters should be map. Example {:sample-rate 16000 :sample-size-bits 16 :channels-number 2 :signed? true :bigEndian? false }.
  Returns: target data line in open state"
  [audio-params]
  (safe
    (let [audio-format (init-audio-format audio-params)
          target-data-line (init-target-data-line audio-format)]
      (do
        (if-not (target-data-line-is-open? target-data-line)
          (open-target-data-line target-data-line audio-format))
        (start-target-data-line target-data-line))
      target-data-line)))


;;audio-params (read-edn-config-file config-audio-filename)
;;spx-params (read-edn-config-file config-spx-filename)

(defn get-spx-encoder
  "This is internal function for initialization of SPX encoder.
  Audio parameters should be map. Example {:sample-rate 16000 :sample-size-bits 16 :channels-number 2 :signed? true :bigEndian? false }.
  SPX parameters should be map. Example: { :mode 1 :quality 10 :complexity 2 :vad true :vbr false :nframes 1 }
  Returns: SPX encoder object is ready for raw audio processig."
  [audio-params
   spx-params]
  (safe
    (let [e (org.xiph.speex.SpeexEncoder.)
          _ (.init e (spx-params :mode) (spx-params :quality) (audio-params :sample-rate) (audio-params :channels-number))]
      (do
        (-> (.getEncoder e) (.setComplexity (spx-params :complexity)))
        (-> (.getEncoder e) (.setVad (spx-params :vad))))
      e)))


(defn get-ready-spx-writer
  "This is internal function prepares ogg/spx writer for data writing on disk.
  Audio parameters should be map. Example {:sample-rate 16000 :sample-size-bits 16 :channels-number 2 :signed? true :bigEndian? false }.
  SPX parameters should be map. Example: { :mode 1 :quality 10 :complexity 3 :vad true :vbr false :nframes 1 }
  Returns: initialized ogg/spx writer ready for data writing. New file is created on disk with ogg header."
  [audio-params
   spx-params
   output-filename] ;;name of output binary file name
  (safe
    (let [ogg-writer (org.xiph.speex.OggSpeexWriter. (spx-params :mode) (audio-params :sample-rate) (audio-params :channels-number)
                       (spx-params :nframes) (spx-params :vbr))]
      (do
        (.open ogg-writer output-filename)
        (.writeHeader ogg-writer "Encoded with: Java Speex Encoder v0.9.7")
        )
      ogg-writer)))


(defn start-record-audio-fixed-length-to-spx-files
  "This function perfoms record of audio stream from microphone and write encoded stream to a SPX/OGG files fixed time length.
After period of time output file will be closed and sound resources will be relesed.
SPX configuration file should be map.
Example: { :mode 1 :quality 10 :complexity 2 :vad true :vbr false :nframes 1}
Audio parameters file should be map.
Example: {:sample-rate 16000 :sample-size-bits 16 :channels-number 2 :signed? true :bigEndian? false }."
  [config-audio-filename ;;configuration file with audio parameters in edn format
   config-spx-filename ;;configuration file with spx parameters in edn format
   output-filename-template ;;template for name of output binary files ( somename: somename-1, somename-2, ...)
   output-file-length-sec ;;maxium
   output-filenames-vector ;;atom - this vector is updated by this function during recording
   stop-flag?] ;;boolean atom to stop audio recording
  (safe
    (let [audio-params (read-config-file config-audio-filename)
          spx-params (read-config-file config-spx-filename)
          target-data-line (init-audio audio-params)
          spx-encoder (get-spx-encoder audio-params spx-params)
          pcm-packet-size (* 2 (audio-params :channels-number) (.getFrameSize spx-encoder))
          bb (byte-array (* 10 pcm-packet-size))
          file-counter (atom 0)]
      (while (not @stop-flag?)
        (do
          (let [out-file-name (str output-filename-template @file-counter ".ogg")
                ogg-writer (get-ready-spx-writer audio-params spx-params out-file-name)
                start-time (System/currentTimeMillis)
                stop-time (+ start-time
                            (* output-file-length-sec 1000))]
            (do
              (swap! file-counter inc)
              (while (and
                       (not @stop-flag?)
                       (< (System/currentTimeMillis) stop-time))
                (let [bytes-captured-length (read-audio-stream target-data-line bb pcm-packet-size)
                      foo (.processData spx-encoder bb 0 pcm-packet-size)
                      bytes-converted (.getProcessedData spx-encoder bb 0)]
                  (.writePacket ogg-writer bb 0 bytes-converted)))
              (.close ogg-writer)
              (swap! output-filenames-vector #(conj % out-file-name))))))
      (stop-capture-audio target-data-line))
    nil))

(defn start-record-audio-to-spx-files-with-pause-support
  "This function perfoms record of audio stream from microphone and write encoded stream to a SPX/OGG files.
Boolean (atom) pause? flag indicates a pause during recording. You can freely start/stop recording using pause? flag.
SPX configuration file should be map.
Example: { :mode 1 :quality 10 :complexity 2 :vad true :vbr false :nframes 1}
Audio parameters file should be map.
Example: {:sample-rate 16000 :sample-size-bits 16 :channels-number 2 :signed? true :bigEndian? false }."
  [config-audio-filename ;;configuration file with audio parameters in edn format
   config-spx-filename ;;configuration file with spx parameters in edn format
   output-filename-template ;;template for name of output binary files ( somename: somename-1, somename-2, ...)
   output-file-length-sec ;;maxium
   output-filenames-vector ;;atom - this vector is updated by this function during recording
   stop-flag? ;;boolean atom to stop audio recording
   pause?]
  (safe
    (let [audio-params (read-config-file config-audio-filename)
          spx-params (read-config-file config-spx-filename)
          target-data-line (init-audio audio-params)
          spx-encoder (get-spx-encoder audio-params spx-params)
          pcm-packet-size (* 2 (audio-params :channels-number) (.getFrameSize spx-encoder))
          bb (byte-array (* 10 pcm-packet-size))
          file-counter (atom 0)]
      (while (not @stop-flag?)
        (if-not @pause?
          (do
            (let [out-file-name (str output-filename-template @file-counter ".ogg")
                  ogg-writer (get-ready-spx-writer audio-params spx-params out-file-name)]
              (do
                (swap! file-counter inc)
                (while (and
                         (not @stop-flag?)
                         (not @pause?))
                  (let [bytes-captured-length (read-audio-stream target-data-line bb pcm-packet-size)
                        foo (.processData spx-encoder bb 0 pcm-packet-size)
                        bytes-converted (.getProcessedData spx-encoder bb 0)]
                    (.writePacket ogg-writer bb 0 bytes-converted)))
                (.close ogg-writer)
                (swap! output-filenames-vector #(conj % out-file-name)))))
          (Thread/sleep 10)))
      (stop-capture-audio target-data-line))
    nil))

(defn start-record-audio-to-spx-file
  "This function perfoms record of audio stream from microphone and write encoded stream to a SPX/OGG file.
  Stop flag is used to stop recording. After stop - output file will be closed and sound resources will be relesed.
  SPX configuration file should be map.
  Example: { :mode 1 :quality 10 :complexity 2 :vad true :vbr false :nframes 1}
  Audio parameters file should be map.
  Example: {:sample-rate 16000 :sample-size-bits 16 :channels-number 2 :signed? true :bigEndian? false }."
  [config-audio-filename ;;configuration file with audio parameters in edn format
   config-spx-filename ;;configuration file with spx parameters in edn format
   output-filename ;;name of output binary file name
   stop-flag?] ;;boolean atom to stop audio recording
  (safe
    (let [audio-format (get-audio-params config-audio-filename)
          target-data-line (init-target-data-line audio-format)
          audio-params (read-config-file config-audio-filename)
          spx-params (read-config-file config-spx-filename)
          e (org.xiph.speex.SpeexEncoder.)
          _ (.init e (spx-params :mode) (spx-params :quality) (audio-params :sample-rate) (audio-params :channels-number))
          ogg-writer (org.xiph.speex.OggSpeexWriter. (spx-params :mode) (audio-params :sample-rate) (audio-params :channels-number)
                       (spx-params :nframes) (spx-params :vbr))
          pcm-packet-size (* 2 (audio-params :channels-number) (.getFrameSize e))
          bb (byte-array (* 10 pcm-packet-size))
          processed-bytes-size (atom 0)]
      (do
        (if-not (target-data-line-is-open? target-data-line)
          (open-target-data-line target-data-line audio-format))
        (start-target-data-line target-data-line)

        (-> (.getEncoder e) (.setComplexity (spx-params :complexity)))
        (-> (.getEncoder e) (.setVad (spx-params :vad)))

        (.open ogg-writer output-filename)
        (.writeHeader ogg-writer "Encoded with: Java Speex Encoder v0.9.7")

        (while (not @stop-flag?)
          (do
            (let [bytes-captured-length (read-audio-stream target-data-line bb pcm-packet-size)
                  foo (.processData e bb 0 pcm-packet-size)
                  bytes-converted (.getProcessedData e bb 0)]
              (swap! processed-bytes-size #(+ % bytes-converted))
              (.writePacket ogg-writer bb 0 bytes-converted))))

        (stop-capture-audio target-data-line)
        (.close ogg-writer)))))



(defn start-record-audio-from-wav-to-spx-files-with-pause-support
  "This function reads audio stream from wav file  and write SPX/OGG encoded stream to a files.
Boolean (atom) pause? flag indicates a pause during recording. You can freely start/stop recording using pause? flag.
SPX configuration file should be map.
Example: { :mode 1 :quality 10 :complexity 2 :vad true :vbr false :nframes 1}
Audio parameters file should be map.
Example: {:input-wav-filename \"filename\" :sample-rate 16000 :sample-size-bits 16 :channels-number 2 :signed? true :bigEndian? false }."
  [config-audio-filename ;;configuration file with audio parameters in edn format
   config-spx-filename ;;configuration file with spx parameters in edn format
   output-filename-template ;;template for name of output binary files ( somename: somename-1, somename-2, ...)
   output-file-length-sec ;;maxium
   output-filenames-vector ;;atom - this vector is updated by this function during recording
   stop-flag? ;;boolean atom to stop audio recording
   pause?]
  (safe
    (let [audio-params (read-config-file config-audio-filename)
          spx-params (read-config-file config-spx-filename)
          spx-encoder (get-spx-encoder audio-params spx-params)
          pcm-packet-size (* 2 (audio-params :channels-number) (.getFrameSize spx-encoder))
          bb (byte-array (* 10 pcm-packet-size))
          file-counter (atom 0)

          infile (input-stream (File. (audio-params :input-wav-filename)))
          _ (.read infile bb 0 44)                          ;pass wav-header
          ]
      (while (not @stop-flag?)
        (if-not @pause?
          (do
            (let [out-file-name (str output-filename-template @file-counter ".ogg")
                  ogg-writer (get-ready-spx-writer audio-params spx-params out-file-name)]
              (do
                (swap! file-counter inc)
                (while (and
                         (not @stop-flag?)
                         (not @pause?))
                  (let [bytes-captured-length (.read infile bb 0 pcm-packet-size)
                        _ (when (= bytes-captured-length -1)
                            (reset! stop-flag? true))
                        foo (.processData spx-encoder bb 0 pcm-packet-size)
                        bytes-converted (.getProcessedData spx-encoder bb 0)]
                    (.writePacket ogg-writer bb 0 bytes-converted)))
                (.close ogg-writer)
                (swap! output-filenames-vector #(conj % out-file-name)))))
          (Thread/sleep 10)))
      (.close infile))
    nil))