(ns com.michaelgaare.clj-util.functional
  "Functional programming utility functions")

(defn ajuxt
  "Like juxt combined with apply. Takes a set of functions, and returns
  a function of the same arity as the number of functions that will
  call each function with the arg that matches it positionally, and
  return a vector of the results.

  ((ajuxt a b) x y) => [(a x) (b y)]"
  ([f]
   (fn [x] [(f x)]))
  ([f g]
   (fn [x y] [(f x) (g y)]))
  ([f g h]
   (fn [x y z] [(f x) (g y) (h z)]))
  ([f g h & fs]
   (fn [x y z & args]
     (when-not (= (count args) (count fs))
       (throw (IllegalArgumentException. "Number of args must match number of functions in ajuxt")))
     (mapv (fn [fun val] (fun val))
           (into [f g h] fs)
           (into [x y z] args)))))

(defmacro get-in*
  "Faster version of get-in as a macro. Does not support default value."
  [m ks]
  (let [gets (map (fn [k]
                    (list 'clojure.core/get k))
                  ks)]
    (concat (list 'clojure.core/-> m)
            gets)))

(defmacro cond-let
  "Like cond, but each test should be in the form of [binding test]. If
  test evaluates logical true, then the associated binding will be
  available in the corresponding expression."
  [& clauses]
  (when clauses
    (list 'if-let (first clauses)
          (if (next clauses)
            (second clauses)
            (throw (IllegalArgumentException.
                    "cond-let requires an even number of forms")))
          (cons 'com.michaelgaare.clj-util.functional/cond-let
                (next (next clauses))))))

(defmacro cond-let->
  "Like cond->, but each test should be in the form of [binding
  test]. If test evaluates logical true, then the associated binding
  will be available in the corresponding expression."
  [expr & clauses]
  (assert (even? (count clauses)))
  (let [g (gensym)
        steps (map (fn [[test step]]
                     `(if-let ~test (-> ~g ~step) ~g))
                   (partition 2 clauses))]
    `(let [~g ~expr
           ~@(interleave (repeat g) (butlast steps))]
       ~(if (empty? steps)
          g
          (last steps)))))

(defmacro cond-let->>
  "Like cond->>, but each test should be in the form of [binding
  test]. If test evaluates logical true, then the associated binding
  will be available in the corresponding expression."
  [expr & clauses]
  (assert (even? (count clauses)))
  (let [g (gensym)
        steps (map (fn [[test step]]
                     `(if-let ~test (->> ~g ~step) ~g))
                   (partition 2 clauses))]
    `(let [~g ~expr
           ~@(interleave (repeat g) (butlast steps))]
       ~(if (empty? steps)
          g
          (last steps)))))
