(ns com.michaelgaare.clj-util.random
  "Seeded random functions, mirroring those from clojure core. Each take a
  seed, a java.util.Random rng, as first arg. This will be mutated."
  (:refer-clojure :exclude [rand rand-int rand-nth shuffle]))

(defprotocol ToSeed
  (make-seed [x]
    "Creates a seeded random generator from given seed value"))

(extend-protocol ToSeed

  java.util.UUID
  (make-seed [uuid]
    (java.util.Random. (unchecked-multiply (.getLeastSignificantBits uuid) (.getMostSignificantBits uuid))))

  java.lang.Long
  (make-seed [long]
    (java.util.Random. long))

  java.lang.Integer
  (make-seed [integer]
    (java.util.Random. (long integer))))

(defn rand
  "Like clojure.core/rand but taking a seed, returns a random floating
  point number between 0 (inclusive) and n (default 1) (exclusive)."
  ([^java.util.Random seed]
   (.nextFloat seed))
  ([seed n]
   (* n (rand seed))))

(defn rand-int
  "Like clojure.core/rand-int but taking a seed, returns a random
  integer between 0 (inclusive) and n (exclusive)."
  [seed n]
  (int (rand seed n)))

(defn rand-nth
  "Like clojure.core/rand-nth but taking a seed, return a random
  element of the (sequential) collection. Will have the same
  performance characteristics as nth for the given collection."
  [seed coll]
  (nth coll (rand-int seed (count coll))))

(defn shuffle
  "Like clojure.core/shuffle but taking a seed, return a random
  permutation of coll"
  [^java.util.Random seed ^java.util.Collection coll]
  (let [al (java.util.ArrayList. coll)]
    (java.util.Collections/shuffle al seed)
    (clojure.lang.RT/vector (.toArray al))))
