(ns com.michaelgaare.clj-util.transform
  "Transformation utility functions.
    - `restructure` to rewrite maps
    - `val-map` map functions over map values")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; restructure
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn- get-value
  "Calls `get`, or `get-in` if key-or-path is a vector. Optional default."
  ([m key-or-path]
   (if (vector? key-or-path)
     (get-in m key-or-path)
     (get m key-or-path)))
  ([m key-or-path default]
   (if (vector? key-or-path)
     (get-in m key-or-path default)
     (get m key-or-path default))))

(defn literal
  "For use in `restructure`, marks a value as one that should be
   preserved literally in the output."
  [x]
  [::literal x])

;; the target arg is regrettably necessary only for helpful exception
;; messages
(defn- single-extract
  [input extractor xf-or-val default target]
  (if (= ::literal extractor)
    xf-or-val
    (let [val (get-value input extractor)
          xformed (if xf-or-val
                    (try
                      (xf-or-val val)
                      (catch Exception e
                        (throw
                         (ex-info (str "error in transform: " (.getMessage e))
                                  {:extractor extractor
                                   :input-val val
                                   :target target}
                                  e))))
                    val)]
      (if (nil? xformed)
        default
        xformed))))

(defn- multi-extract
  [input extractor xf default target]
  (let [extracted (map (partial get-value input)
                       (:ks extractor)
                       (concat (:defaults extractor)
                               (repeat nil)))
        xformed (if xf
                  (try
                    (apply xf extracted)
                    (catch Exception e
                      (throw
                       (ex-info (str "error in transform: " (.getMessage e))
                                {:extractor extractor
                                 :input-val extracted
                                 :target target}
                                e))))
                  extracted)]
    (if (or (nil? default)
            (and (not (coll? xformed))
                 (some? xformed))
            (some some? xformed))
      xformed
      default)))

(defn restructure
  "Takes an input map and 'restructures' it according to re-map.
   Restructuring could be thought of as a sort of complement to
   associative destructuring. It allows you to reshape and transform
   the data in the input map.
   This is accomplished by building a map (in re-map) with the desired
   structure, with values that describe where to get the desired data
   from the input map, along with optional transforms. Keys and nested
   maps in re-map will be preserved. Non-map values are extractors -
   they specify how to retrieve and optionally transform the data from
   the input map.
   Extractors can be:
    * non-collection values, which will be treated as keys in the
      input map and replaced with the associated value
    * vectors, which are treated as tuples of the following-form:
      [extraction xform-or-val :or? default?]
      * If extraction is a vector, it will retrieve the associated
        value from the input map as with `get-in`.
      * If extraction is a map, it can be used to extract multiple
        values to be passed to the transform, as described below.
      * If extraction is the special-case keyword ::literal (as is
        the case when one calls `literal`) then the value of
        xform-or-val will be used as the literal value.
      * nil values will be passed to transforms.
      * After extraction and any xform, an :or keyword followed by
        a default value will be used in case of nil.
      * Otherwise it will be treated as a key, as in an extractor.
      The map version of extraction takes this form:
        {:ks [key-names-or-nested-vectors]
         :defaults [optional-defaults]}
      The values of the extracted keys will be passed, in the order
      specified, to the transform function. Any missing keys will
      have nil values.
   If xform function is provided, it will be called on the value to
   transform. If a key is missing from the input map or the supplied
   transformation function returns nil, it will not be included in the
   output map.
   ex:
   (restructure {:a 1 :b 2 :c 3 :d {:e 4}}
                {:bb {:cc [:a inc]
                      :dd (literal \"hi\")}
                 :ee [{:ks [:b :c]} max]
                 :ff [{:ks [:f] :defaults [5]} inc]
                 :gg [[:d :e]]
                 :hh [:h :or 12]
                 :ii [{:ks [:h]} :or 13]})
    => {:bb {:cc 2, :dd \"hi\"} :ee 3 :ff 6 :gg 4 :hh 12 :ii 13}"
  [m re-map]
  (reduce
   (fn [out [k ext]]
     (cond
       (vector? ext)
       (let [[extractor & remaining] ext
             xf-or-val (if (= :or (first remaining))
                         nil
                         (first remaining))
             [& {default :or}] (if xf-or-val
                                 (rest remaining)
                                 remaining)
             multi-extract? (map? extractor)
             literal? (= ::literal extractor)
             val (if multi-extract?
                   (multi-extract m extractor xf-or-val default k)
                   (single-extract m extractor xf-or-val default k))]
         (cond-> out
           (some? val) (assoc k val)))

       (map? ext)
       (assoc out k (restructure m ext))

       :else
       (let [v (get m ext)]
         (cond-> out
           (some? v) (assoc k v)))))
   {}
   re-map))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Map transformation
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn val-map
  "Maps function f over the values of map m"
  [f m]
  (persistent!
   (reduce-kv (fn [m' k v]
                (assoc! m' k (f v)))
              (transient {})
              m)))
