(ns com.michaelgaare.clj-util.transform
  "Transformation utility functions.
    - `restructure` to rewrite maps
    - `val-map` map functions over map values")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; restructure
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn- get-value
  "Calls `get`, or `get-in` if key-or-path is a vector. Optional default."
  ([m key-or-path]
   (if (vector? key-or-path)
     (get-in m key-or-path)
     (get m key-or-path)))
  ([m key-or-path default]
   (if (vector? key-or-path)
     (get-in m key-or-path default)
     (get m key-or-path default))))

(defn literal
  "For use in `restructure`, marks a value as one that should be
   preserved literally in the output."
  [x]
  [::literal x])

(defn restructure
  "Takes an input map and 'restructures' it according to re-map.
   Restructuring could be thought of as a sort of complement to
   associative destructuring. It allows you to reshape and transform
   the data in the input map.
   This is accomplished by building a map (in re-map) with the desired
   structure, with values that describe where to get the desired data
   from the input map, along with optional transforms. Keys and nested
   maps in re-map will be preserved. Non-map values are extractors -
   they specify how to retrieve and optionally transform the data from
   the input map.
   Extractors can be:
    * non-collection values, which will be treated as keys in the
      input map and replaced with the associated value
    * vectors, which are treated as tuples of the following-form:
      [extraction xform-or-val]
      * If extraction is a vector, it will retrieve the associated
        value from the input map as with `get-in`.
      * If extraction is a map, it can be used to extract multiple
        values to be passed to the transform, as described below.
      * If extraction is the special-case keyword ::literal (as is
        the case when one calls `literal`) then the value of
        xform-or-val will be used as the literal value.
      * Otherwise it will be treated as a key, as in an extractor.
      The map version of extraction takes this form:
        {:ks [key-names-or-nested-vectors]
         :defaults [optional-defaults]}
      The values of the extracted keys will be passed, in the order
      specified, to the transform function. Any missing keys will
      have nil values.
   If xform function is provided, it will be called on the value to
   transform. If a key is missing from the input map or the supplied
   transformation function returns nil, it will not be included in the
   output map.
   ex:
   (restructure {:a 1 :b 2 :c 3 :d {:e 4}}
                {:bb {:cc [:a inc]
                      :dd (literal \"hi\")}
                 :ee [{:ks [:b :c]} max]
                 :ff [{:ks [:f] :defaults [5]} inc]
                 :gg [[:d :e]]})
    => {:bb {:cc 2, :dd \"hi\"} :ee 3 :ff 6 :gg 4}"
  [m re-map]
  (reduce
   (fn [out [k ext]]
     (cond
       (vector? ext)
       (let [[extractor xf-or-val] ext
             multi-extract? (map? extractor)
             literal? (= ::literal extractor)
             extracted (cond multi-extract?
                             (seq
                              (map (partial get-value m)
                                   (:ks extractor)
                                   (concat (:defaults extractor)
                                           (repeat nil))))
                             literal?
                             xf-or-val

                             :else
                             (get-value m extractor))
             new-v (if (and xf-or-val (not literal?))
                     (try
                       (if multi-extract?
                         (apply xf-or-val extracted)
                         (xf-or-val extracted))
                       (catch Exception e
                         (throw
                          (ex-info (str "error in transform: " (.getMessage e))
                                   {:extractor extractor
                                    :input-val extracted
                                    :target k}
                                   e))))
                     extracted)]
         (cond-> out
           (some? new-v) (assoc k new-v)))

       (map? ext)
       (assoc out k (restructure m ext))

       :else
       (let [v (get m ext)]
         (cond-> out
           (some? v) (assoc k v)))))
   {}
   re-map))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Map transformation
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn val-map
  "Maps function f over the values of map m"
  [f m]
  (persistent!
   (reduce-kv (fn [m' k v]
                (assoc! m' k (f v)))
              (transient {})
              m)))
