(ns com.michaelgaare.clj-util.transform
  "Transformation utility functions.
    - `restructure` to rewrite maps
    - `val-map` map functions over map values"
  (:require [com.michaelgaare.clj-util.functional :as f]))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; restructure
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn literal
  "For use in `restructure`, marks a value as one that should be
   preserved literally in the output."
  [x]
  [::literal x])

;; the target arg is regrettably necessary only for helpful exception
;; messages
(defn- single-extract
  [input extractor xf-or-val default target]
  (if (= ::literal extractor)
    xf-or-val
    (let [val (f/get-value input extractor)
          xformed (if xf-or-val
                    (try
                      (xf-or-val val)
                      (catch Exception e
                        (throw
                         (ex-info (str "error in transform: " (.getMessage e))
                                  {:extractor extractor
                                   :input-val val
                                   :target target}
                                  e))))
                    val)]
      (if (nil? xformed)
        default
        xformed))))

(defn- multi-extract
  [input extractor xf default target]
  (let [extracted (map (partial f/get-value input)
                       (:ks extractor)
                       (concat (:defaults extractor)
                               (repeat nil)))
        xformed (if xf
                  (try
                    (apply xf extracted)
                    (catch Exception e
                      (throw
                       (ex-info (str "error in transform: " (.getMessage e))
                                {:extractor extractor
                                 :input-val extracted
                                 :target target}
                                e))))
                  extracted)]
    (if (or (nil? default)
            (and (not (coll? xformed))
                 (some? xformed))
            (some some? xformed))
      xformed
      default)))

(defn restructure*
  "Runtime-only version of restructure."
  [m re-map]
  (reduce
   (fn [out [k ext]]
     (cond
       (vector? ext)
       (let [[extractor & remaining] ext
             xf-or-val (if (= :or (first remaining))
                         nil
                         (first remaining))
             [& {default :or}] (if xf-or-val
                                 (rest remaining)
                                 remaining)
             multi-extract? (map? extractor)
             val (if multi-extract?
                   (multi-extract m extractor xf-or-val default k)
                   (single-extract m extractor xf-or-val default k))]
         (cond-> out
           (some? val) (assoc k val)))

       (map? ext)
       (assoc out k (restructure* m ext))

       :else
       (let [v (get m ext)]
         (cond-> out
           (some? v) (assoc k v)))))
   {}
   re-map))

(declare restructure)

(defn extract*
  "Runtime value extraction for restructure"
  [m ext target]
  (cond
    (vector? ext)
    (let [[extractor & remaining] ext
          xf-or-val (if (= :or (first remaining))
                      nil
                      (first remaining))
          [& {default :or}] (if xf-or-val
                              (rest remaining)
                              remaining)
          multi-extract? (map? extractor)]
      (if multi-extract?
        (multi-extract m extractor xf-or-val default target)
        (single-extract m extractor xf-or-val default target)))

    (map? ext)
    (restructure* m ext)

    :else
    (get m ext)))

(defn multi-extract-form
  [input-sym extract-m xf or-val target]
  (let [{:keys [ks defaults]} extract-m
        gets (map (fn [k d]
                    `(f/get-value ~input-sym ~k ~d))
                  ks
                  (concat defaults (repeat nil)))
        val-form (if xf
                   `(try
                      (~xf ~@gets)
                      (catch Exception e#
                        (throw
                         (ex-info (str "error in transform: " (.getMessage e#))
                                  {:extractor ~extract-m
                                   :input-val (list ~@gets)
                                   :target ~target}
                                  e#))))
                   `(let [v# (list ~@gets)]
                      (when (and (seq? v#)
                                 (some some? v#))
                        v#)))]
    (if or-val
      `(let [v# ~val-form]
         (if (some? v#)
           v#
           ~or-val))
      val-form)))

(defn- single-extract-form
  [input-sym extract-v xf or-val target]
  (let [get-f `(f/get-value ~input-sym ~extract-v)
        val-form (if xf
                   `(try
                      (~xf ~get-f)
                      (catch Exception e#
                        (throw
                         (ex-info (str "error in transform: " (.getMessage e#))
                                  {:extractor ~extract-v
                                   :input-val ~get-f
                                   :target ~target}
                                  e#))))
                   get-f)]
    (if or-val
      `(let [v# ~val-form]
         (if (some? v#)
           v#
           ~or-val))
      val-form)))

(defn- extractor-form
  [input-sym extractor target]
  (let [xf? (some-fn symbol? list? map? #(and (keyword? %) (not= ::literal %)))
        extractable? (some-fn vector? keyword? string?)]
    (cond
      (symbol? extractor)
    ;; symbols need to be handled at runtime
      `(extract* ~input-sym ~extractor ~target)

      (not (coll? extractor))
      (list 'get input-sym extractor)

      (vector? extractor)
      (cond

        (= ::literal (first extractor))
        (second extractor)

        ((f/pred-pattern map? xf? nil?) extractor)
        (let [[ex-m xf] extractor]
          (multi-extract-form input-sym ex-m xf nil target))

        ((f/pred-pattern map? xf? #{:or} some?) extractor)
        (let [[ex-m xf _ or-val] extractor]
          (multi-extract-form input-sym ex-m xf or-val target))

        ((f/pred-pattern map? #{:or} some?) extractor)
        (let [[ex-m _ or-val] extractor]
          (multi-extract-form input-sym ex-m nil or-val target))

        ((f/pred-pattern extractable? nil?) extractor)
        (single-extract-form input-sym (first extractor) nil nil target)

        ((f/pred-pattern extractable? xf? nil?) extractor)
        (let [[ex xf] extractor]
          (single-extract-form input-sym ex xf nil target))

        ((f/pred-pattern extractable? xf? #{:or} some?) extractor)
        (let [[ex xf _ or-val] extractor]
          (single-extract-form input-sym ex xf or-val target))

        ((f/pred-pattern extractable? #{:or} some?) extractor)
        (let [[ex _ or-val] extractor]
          (single-extract-form input-sym ex nil or-val target))

        :else
        ;; Something we can't handle at compile time
        `(extract* ~input-sym ~extractor ~target))

      (map? extractor)
      `(restructure ~input-sym ~extractor)

      :else
      `(extract* ~input-sym ~extractor ~target))))

(defmacro restructure
  "Takes an input map and 'restructures' it according to re-map.
   Restructuring could be thought of as a sort of complement to
   associative destructuring. It allows you to reshape and transform
   the data in the input map.
   This is accomplished by building a map (in re-map) with the desired
   structure, with values that describe where to get the desired data
   from the input map, along with optional transforms. Keys and nested
   maps in re-map will be preserved. Non-map values are extractors -
   they specify how to retrieve and optionally transform the data from
   the input map.
   Extractors can be:
    * non-collection values, which will be treated as keys in the
      input map and replaced with the associated value
    * vectors, which are treated as tuples of the following-form:
      [extraction xform-or-val :or? default?]
      * If extraction is a vector, it will retrieve the associated
        value from the input map as with `get-in`.
      * If extraction is a map, it can be used to extract multiple
        values to be passed to the transform, as described below.
      * If extraction is the special-case keyword ::literal (as is
        the case when one calls `literal`) then the value of
        xform-or-val will be used as the literal value.
      * nil values will be passed to transforms.
      * After extraction and any xform, an :or keyword followed by
        a default value will be used in case of nil.
      * Otherwise it will be treated as a key, as in an extractor.
      The map version of extraction takes this form:
        {:ks [key-names-or-nested-vectors]
         :defaults [optional-defaults]}
      The values of the extracted keys will be passed, in the order
      specified, to the transform function. Any missing keys will
      have nil values.
   If xform function is provided, it will be called on the value to
   transform. If a key is missing from the input map or the supplied
   transformation function returns nil, it will not be included in the
   output map.
   ex:
   (restructure {:a 1 :b 2 :c 3 :d {:e 4}}
                {:bb {:cc [:a inc]
                      :dd (literal \"hi\")}
                 :ee [{:ks [:b :c]} max]
                 :ff [{:ks [:f] :defaults [5]} inc]
                 :gg [[:d :e]]
                 :hh [:h :or 12]
                 :ii [{:ks [:h]} :or 13]})
    => {:bb {:cc 2, :dd \"hi\"} :ee 3 :ff 6 :gg 4 :hh 12 :ii 13}

  Note that restructure is significantly faster when the re-map is
  availalbe at compile time."
  [m re-map]
  (if (map? re-map)
    (let [msym (gensym "m")
          val (gensym "v")
          key-exprs (mapcat (fn [[k v]]
                              `[[~val ~(extractor-form msym v k)]
                                (~(if (> (count re-map) 4)
                                    `assoc!
                                    `assoc) ~k ~val)])
                            re-map)]
      `(let [~msym ~m]
         (-> ~(if (> (count re-map) 4)
                `(transient {})
                `{})
             (f/cond-let-some->
              ~@key-exprs)
             ~(if (> (count re-map) 4)
                `(persistent!)
                `(identity)))))
    `(restructure* ~m ~re-map)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Map transformation
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn val-map
  "Maps function f over the values of map m"
  [f m]
  (persistent!
   (reduce-kv (fn [m' k v]
                (assoc! m' k (f v)))
              (transient {})
              m)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Sequences
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn distinct-by
  "Like distinct, but calls f on each item to determine the distinct
   value, while returning the original un-transformed item. Returns a
   transducer if called without a collection argument."
  ([f]
   (fn [rf]
     (let [seen (volatile! #{})]
       (fn
         ([] (rf))
         ([result] (rf result))
         ([result input]
          (let [d-val (f input)]
            (if (contains? @seen d-val)
              result
              (do (vswap! seen conj d-val)
                  (rf result input)))))))))
  ([f coll]
   (let [step (fn step [xs seen]
                (lazy-seq
                 ((fn [[x :as xs] seen]
                    (when-let [s (seq xs)]
                      (let [d-val (f x)]
                        (if (contains? seen d-val)
                          (recur (rest s) seen)
                          (cons x (step (rest s) (conj seen d-val)))))))
                  xs seen)))]
     (step coll #{}))))

(defn dedupe-by
  "Like dedupe, but calls f on each item to determine the item to dedupe
  by, while returning the original item. Returns a transducer when no
  collection is provided."
  ([f]
   (fn [rf]
     (let [pv (volatile! ::none)]
       (fn
         ([] (rf))
         ([result] (rf result))
         ([result input]
          (let [prior @pv
                dedupe-val (f input)]
            (vreset! pv dedupe-val)
            (if (= prior dedupe-val)
              result
              (rf result input))))))))
  ([f coll]
   (sequence (dedupe-by f) coll)))
