(ns com.michaelgaare.clj-util.random.selection
  "Make selections from different sources, using either weighted or
  non-weighted random selection, with optional repeatability using
  controlled seeds."
  (:require [com.michaelgaare.clj-util.random :as random]))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Weighted random
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn weighted-selector
  "Creates a weighted random selector from a map of {source weight}. Call
  `weighted-select` on the result."
  [weights]
  (assert (map? weights))
  (assert (every? number? (vals weights)))
  (let [weight-vals (reductions + (vals weights))
        choices (map vector (keys weights) weight-vals)]
    {::choices choices
     ::range (last weight-vals)
     ::weights weights}))

(defn- selector?
  "True if arg is a selector as returned from `weighted-selector`."
  [x]
  (some? (::choices x)))

(defn remove-selector-source
  "Takes a selector (as from `weighted-selector`) and removes the source."
  [selector source]
  (weighted-selector (dissoc (::weights selector) source)))

(defn weight-count
  "Returns a count of weighted items in selector."
  [selector]
  (count (keys (::weights selector))))

(defn weighted-select
  "Returns one item from selector based on weighted random
  selection. The selector is either the return value of
  `weighted-selector` or a map of {item weight}."
  [seed selector]
  (let [sel (if (selector? selector)
              selector
              (weighted-selector selector))
        choice (random/rand-int seed (::range sel))]
    (loop [[[item w] & more] (::choices sel)]
      (when w
        (if (< choice w)
          item
          (recur more))))))

(defn weighted-selections
  "Returns a lazy seq of items weighted random selected from
  sources. Takes a map of {source weight} and a map of
  {source [item]}. Each returned item will be a map of
   * :source
   * :item"
  ([seed source-weights source-lists]
   (let [selector (if (selector? source-weights)
                    source-weights
                    (weighted-selector source-weights))]
     ((fn next-items [sel sources]
        (when (pos? (weight-count sel))
          (lazy-seq
          ;; special casing when there's only 1 thing left
           (let [src (weighted-select seed sel)
                 items (get sources src)]
             (if (= 1 (weight-count sel))
               (map (fn [item]
                      {:source src
                       :item item})
                    (get sources src))
               (if (seq items)
                 (cons {:source src
                        :item (first items)}
                       (next-items sel (update sources src rest)))
                 (next-items (remove-selector-source sel src) sources)))))))
      selector source-lists)))
  ([source-weights source-lists]
   (weighted-selections (random/random-seed) source-weights source-lists)))

(defn weighted-select-iteration
  "Does a single \"iteration\" of weighted list selection. Takes a of
  inputs with:

   * :sources  map of {source list}
   * :weights  either a selector or map of {source weight}

  Returns a vector of selection and updated inputs, or nil if there
  are no more items.

  [{:source source :item item} updated-inputs]"
  ([seed {:keys [sources weights] :as input}]
   (let [selector (if (selector? weights)
                    weights
                    (weighted-selector weights))
         selected-source (weighted-select seed selector)
         item-list (when selected-source (get sources selected-source))]
     (when selected-source
       (if (seq item-list)
         [{:source selected-source :item (first item-list)}
          (-> input
              (update-in [:sources selected-source] rest)
              (assoc :weights selector))]
         (recur seed (assoc input :weights (remove-selector-source selector selected-source)))))))
  ([input]
   (weighted-select-iteration (random/random-seed) input)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Non-weighted random
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn- random-select-iteration
  "Takes a map of {source [item]}, and picks the first item from a
  random source. Returns nil if no more items. Returns a vector of:

  [{:source source :item item} updated-sources]"
  ([seed sources]
   (when (seq sources)
     (let [selected-source (random/rand-nth seed (keys sources))
           item-list (get sources selected-source)]
       (if (seq item-list)
         [{:source selected-source :item (first item-list)}
          (update sources selected-source rest)]
         (recur seed (dissoc sources selected-source))))))
  ([sources]
   (random-select-iteration (random/random-seed) sources)))

(defn random-selections
  "Takes a map of {source [item]}, and returns a lazy seq of {:source
  source :item item} with the items randomly selected from the lists."
  ([seed sources]
   (when (seq sources)
     (if (= 1 (count sources))
      ;; special case 1 source left for performance
       (let [[source items] (first sources)]
         (map (partial hash-map :source source :item) items))
       (lazy-seq
        (when-let [[selection updated-sources] (random-select-iteration seed sources)]
          (cons selection (random-selections seed updated-sources)))))))
  ([sources]
   (random-selections (random/random-seed) sources)))
