(ns com.markusaschl.imap-observer.extras
  (:require [clojure.string :as str])
  (:import [jakarta.mail Flags$Flag Folder Message$RecipientType]))

(def ^:constant folder-modes {:read-only Folder/READ_ONLY
                   :read-write Folder/READ_WRITE})


(def ^:constant message-flags {:answered Flags$Flag/ANSWERED
                    :deleted Flags$Flag/DELETED
                    :draft Flags$Flag/DRAFT
                    :flagged Flags$Flag/FLAGGED
                    :recent Flags$Flag/RECENT
                    :seen Flags$Flag/SEEN
                    :user Flags$Flag/USER})


(def ^:constant recipient-types {:bcc Message$RecipientType/BCC
                      :cc Message$RecipientType/CC
                      :to Message$RecipientType/TO})


(defn filter-by [flag message]
  (.isSet message flag))


(defn modify-flag [flag set? message]
  (.setFlag message flag set?))


(defn get-address [address]
  [(.getPersonal address)
   (.getAddress address)])


(defn get-addresses [message]
  (into {}
        (map (fn [type]
               [type (->> (seq (.getRecipients message (type recipient-types)))
                          (map #'get-address))]) [:bcc :cc :to])))


(defn input-stream->byte-array [in]
  (with-open [out (java.io.ByteArrayOutputStream.)]
    (clojure.java.io/copy in out)
    (.toByteArray out)))


(defn get-bodyparts--helper [bodypart acc]
  (loop [bodypart bodypart acc acc]
    (cond
      (.isMimeType bodypart "multipart/*")
      (let [mp (.getContent bodypart)]
        (apply #'conj
               (mapcat #(get-bodyparts--helper % acc)
                       (map #(.getBodyPart mp %) (range (.getCount mp)))) acc))

      (.isMimeType bodypart "message/rfc822")
      (recur (.getContent bodypart) acc)
          
      (.isMimeType bodypart "application/*")
      (conj acc [(str/lower-case (.getContentType bodypart))
                 (input-stream->byte-array (.getInputStream bodypart))])

      true
      (conj acc [(str/lower-case (.getContentType bodypart))
                 (.getContent bodypart)]))))


(defn get-bodyparts [message]
  (get-bodyparts--helper message ()))


(defn get-headers [message]
  (into {} (map #(hash-map (.getName %) (.getValue %))
                   (-> (.getAllHeaders message) enumeration-seq))))


(defn get-message [message]
  {:from (->> (.getFrom message) first get-address)
   :recipients (get-addresses message)
   :subject (.getSubject message)
   :message-id (.getMessageID message)
   :in-reply-to (.getInReplyTo message)
   :received-date (.getReceivedDate message)
   :sent-date (.getSentDate message)
   :headers (get-headers message)
   :bodyparts (get-bodyparts message)})
