(ns com.markusaschl.imap-observer
  (:require [clojure.string :as str]
            [com.stuartsierra.component :as component])
  (:import com.sun.mail.imap.IdleManager
           [jakarta.mail.event MessageCountEvent MessageCountListener]
           jakarta.mail.Session
           java.util.concurrent.Executors))

(defn as-properties--helper [m p]
  (doseq [[k v] m]
    (.setProperty p (str k) (str v)))
  p)


(defn as-properties [m]
  (as-properties--helper m (java.util.Properties.)))


(defn session [properties]
  (Session/getInstance (as-properties properties)))


(defrecord Store [protocol host user pass]
  component/Lifecycle

  (start [component]
    (let [{:keys [session]} component
          store--obj (.getStore session protocol)]

      (.connect store--obj host user pass)
      (assoc component :obj store--obj)))

  (stop [component]
    (.close (:obj component))
    (assoc component :obj nil)))


(defrecord Idle-Manager []
  component/Lifecycle

  (start [component]
    (as-properties--helper {"mail.imap.usesocketchannels" "true"
                            "mail.imaps.usesocketchannels" "true"}
                           (.getProperties (:session component)))
    (assoc component
           :obj
           (IdleManager. (:session component) (Executors/newCachedThreadPool))))

  (stop [component]
    (.stop (:obj component))
    (assoc component :obj nil)))


(defrecord IMAP-Folder [name mode on-folder-opened
                        on-message-added on-message-removed]
  component/Lifecycle

  (start [component]
    (let [store--obj (:obj (:store component))
          idle-manager--obj (:obj (:idle-manager component))

          executor (Executors/newCachedThreadPool)
          folder--obj (.getFolder store--obj name)]

      (.open folder--obj mode)

      (when on-folder-opened
        (on-folder-opened folder--obj))

      (let [listener--obj
            (.addMessageCountListener
             folder--obj
             (reify MessageCountListener
               (^void messagesAdded [this ^MessageCountEvent event]
                (when on-message-added
                  (on-message-added (-> event .getMessages seq))
                  (.watch idle-manager--obj folder--obj)))

               (^void messagesRemoved [this ^MessageCountEvent event]
                (when on-message-removed
                  (on-message-removed (-> event .getMessages seq))
                  (.watch idle-manager--obj folder--obj)))))]

        (.watch idle-manager--obj folder--obj)
        (assoc component :obj folder--obj :listener listener--obj))))

  (stop [component]
    (let [{:keys [obj listener]} component]
      (.removeMessageCountListener obj listener)
      (.close obj)
      (assoc component :obj nil :listener nil))))


(defn make-observer [config-options folder-options]
  (let [make-folder-name #(symbol (format ":%s-%s"
                                          (str/lower-case %)
                                          (gensym "folder-")))
        {:keys [imap-protocol host username password
                session-properties]} config-options]

    (->>
     (merge
      {:session (session session-properties)
       :store (component/using (->Store imap-protocol host username password)
                               [:session])
       :idle-manager (component/using (->Idle-Manager)
                                      [:session])}

      (mapcat (fn [properties]
                {(make-folder-name (:name properties))
                 (component/using (map->IMAP-Folder properties)
                                  [:store :idle-manager])})
              folder-options))

     (reduce into [])
     (apply component/system-map))))
