# clj-imap-observer

'clj-imap-observer' is a Clojure library which lets you create a
```com.sun.mail.imap.IdleManager``` in a more functional way.


## Usage

### 1. Add project dependency

``` clojure
[com.markusaschl/clj-imap-observer "0.3.0"]
```


### 2. Import the namespace

``` clojure
(ns com.example.your-application
  (:require [com.markusaschl.imap-observer :as imap-observer]))
```


### 3. Place your configuration into a file

Write something like this into
```<your-project>/ressources/mail-properties.edn```:

``` clojure
{:host "example.com",
 :username "user@example.com",
 :password "password",
 :imap-protocol "imaps",
 :session-properties
 {"mail.event.scope" "application"}}
```


### 4. Make an observer

``` clojure
(def observer (imap-observer/make-observer
                    (->> (clojure.java.io/resource "mail-properties.edn")
                         slurp
                         clojure.edn/read-string)
                    [{:name "INBOX"
                      :mode (:read-write imap-observer/folder-mode)
                      :on-folder-opened
                      (fn [folder-obj] (def ^:dynamic *folder*  folder-obj))
                      :on-message-added
                      (fn [messages]
                        (def ^:dynamic *messages*  messages))}]))
```


## Options

The function ```imap-observer/make-observer``` takes two arguments,
```config-options``` and ```folder-options.```

'config-options' is a map, which uses these keys:

- ```:imap-protocol```
- ```:host```
- ```:username```
- ```:password```
- and ```:session-properties```

':session-properties' is also a map, which is then converted into a
```java.util.Property``` object, which in turn is passed to
```jakarta.mail.Session/getDefaultInstance```. That happens in ```imap-observer/session```

For a list of all the properties you can use, see the
[com.sun.mail.imap](https://javaee.github.io/javamail/docs/api/com/sun/mail/imap/package-summary.html)
documentation.

':folder-options' is an array consisting of a map made of:

- ```:name```
- ```:mode```
- ```:on-message-added```
- ```:on-message-removed```
- ```:on-folder-opened```

For example:

``` clojure
[{:name "INBOX"
  :mode (:read-write com.markusaschl.imap-observer/folder-mode)
  :on-message-added (fn [messages])}]
```


### Notes

1. Posible ```:mode(s)``` are ```jakarta.mail.Folder/READ_ONLY``` or
   ```jakarta.mail.Folder/READ_WRITE```. For convenience reasons, those two are
   exported as a map in ```imap-observer/folder-mode```
2. ```:name``` is the name of an existing mailbox folder
3. The functions ```:on-messsage-added``` and ```:on-message-removed``` are
   called with an array of ```jakarta.mail.Message```. See also
   [jakarta.mail.Message](https://eclipse-ee4j.github.io/mail/docs/api//jakarta.mail/jakarta/mail/Message.html)
4. The function ```:on-folder-openend``` is called with the appropriate
   ```jakarta.mail.Folder``` object. See also
   [jakarta.mail.Folder](https://javaee.github.io/javamail/docs/api/com/sun/mail/imap/IMAPFolder.html)


## Tips

### Timeout

After a certain period of inactivity, the server closes the connection to the
client. To fix a possible ```jakarta.mail.FolderClosedException``` when reading
a folder or email, I offer this solution:

``` clojure
(let
    [ten-minutes (->> 1000 (* 60) (* 10))
     idle-agent (agent (->> (imap-observer/make-observer
                             (->> (clojure.java.io/resource "mail-properties.edn")
                                  slurp
                                  clojure.edn/read-string)
                             [{:name "INBOX"
                               :mode (:read-write imap-observer/folder-mode)
                               :on-message-added (fn [messages])}])))]

    (prn "starting execution")
    (while true
      (send idle-agent component/start)
      (Thread/sleep ten-minutes)
      (send idle-agent component/stop)
      (prn "restarting execution")))
```


## Bugs


## TODOs


- [ ] test connection success with ```GreenMail```
- [ ] test function calls (```:on-folder-opened```, ```:on-message-added``` and
      ```:on-message-removed```) with ```GreenMail```


## License

The MIT License (MIT)

Copyright © 2021 Markus Aschl

Permission is hereby granted, free of charge, to any person obtaining a copy of
this software and associated documentation files (the "Software"), to deal in
the Software without restriction, including without limitation the rights to
use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
the Software, and to permit persons to whom the Software is furnished to do so,
subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
