# clj-imap-observer

'IMAP-Observer' is a library which lets you create an ```IdleManager``` in a
functional way.


## Usage

### 1. Add project dependency

``` clojure
[com.markusaschl/clj-imap-observer "0.2.1"]
```


### 2. Import the namespace

``` clojure
(ns com.example.your-application
  (:require [com.markusaschl.imap-observer :as observer]))
```


### 3. Place your configuration into a file

Write something like this into
```<your-project>/ressources/mail-properties.edn```:

``` clojure
{:host "example.com",
 :username "user@example.com",
 :password "password",
 :imap-protocol "imaps",
 :session-properties
 {"mail.event.scope" "application"}}
```


### 4. Make an observer

``` clojure
(def observer (->> (imap-observer/make-observer
                  (->> (clojure.java.io/resource "mail-properties.edn")
                       slurp
                       clojure.edn/read-string)
                  [{:name "INBOX"
                    :mode Folder/READ_WRITE
                    :on-message-added (fn [messages])}])))
```


## Options

The function ```imap-observer/make-observer``` takes two arguments,
```config-options``` and ```folder-options.```

'config-options' is a map, which uses these keys:

- ```:imap-protocol```
- ```:host```
- ```:username```
- ```:password```
- and ```:session-properties```

':session-properties' is also a map, which is then converted into a
```Property``` object, which in turn is passed to
```Session/getDefaultInstance```. That happens in ```imap-observer/session```.

For a list of all the properties you can use, see the
[com.sun.mail.imap](https://javaee.github.io/javamail/docs/api/com/sun/mail/imap/package-summary.html)
documentation.

':folder-options' is an array consisting of a map made of:

- ```:name```
- ```:mode```
- ```:on-message-added```
- ```:on-message-removed```

For example:

``` clojure
[{:name "INBOX"
  :mode Folder/READ_WRITE
  :on-message-added (fn [messages])}]
```


### Notes

1. Both event functions ```:on-message-added``` and ```:on-message-removed```
   are optional
2. Posible ```:mode(s)``` are ```Folder/READ_ONLY``` or ```Folder/READ_WRITE```
3. ```:name``` is the name of an existing mailbox folder
4. The functions ```:on-messsage-added``` and ```:on-message-removed``` are
   called with an array of ```javax.mail.Message ```. See also [MessageCountEvent/getMessages](https://docs.oracle.com/javaee/7/api/javax/mail/event/MessageCountEvent.html)


## Tips

### Timeout

After a certain period of inactivity, the server considers the connection to the
client as inactive. To fix a possible ```FolderClosedException``` I offer this
solution:

``` clojure
(let
    [ten-minutes (->> 1000 (* 60) (* 10))
     idle-agent (agent (->> (imap-observer/make-observer
                             (->> (clojure.java.io/resource "mail-properties.edn")
                                  slurp
                                  clojure.edn/read-string)
                             [{:name "INBOX"
                               :mode Folder/READ_WRITE
                               :on-message-added (fn [messages])}])))]

    (prn "starting execution")
    (while true
      (send idle-agent component/start)
      (Thread/sleep ten-minutes)
      (send idle-agent component/stop)
      (prn "restarting execution")))
```


## Bugs

- [ ] As of version 0.2, the test fails, as I can't manage to establish a
      connection with ```GreenMail```.


## TODOs


- [ ] test connection success with ```GreenMail```

## License

The MIT License (MIT)

Copyright © 2021 Markus Aschl

Permission is hereby granted, free of charge, to any person obtaining a copy of
this software and associated documentation files (the "Software"), to deal in
the Software without restriction, including without limitation the rights to
use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
the Software, and to permit persons to whom the Software is furnished to do so,
subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
