(ns com.manigfeald.machinate.platform
  (:require [com.manigfeald.machinate.protocols :as p]))

(defn unique-key []
  (js/Object.))

;; TODO can thens and catchs be canceld/removed?

(extend-type js/Promise
  p/Event
  (try-event [^js/Promise event resume resume-with-error control nack-group]
    (let [action (volatile! (fn [k result exception]
                              (when (p/check-condition-state control :synced)
                                (p/remove-condition-listener control k))
                              (when (p/change-condition-state control :waiting :synced)
                                (if exception
                                  (resume-with-error event exception nack-group)
                                  (resume event result nack-group)))))]
      (.then event
             (fn [result]
               (let [k (js/Object.)
                     f (fn [& _]
                         (when-let [f @action]
                           (vreset! action nil)
                           (f k result nil)))]
                 (p/add-condition-listener control k f)
                 (f))))
      (.catch event
              (fn [result]
                (let [k (js/Object.)
                      f (fn [& _]
                          (when-let [f @action]
                            (vreset! action nil)
                            (f k nil result)))]
                  (p/add-condition-listener control k f)
                  (f)))))))

(defn promise-like-type []
  (let [res (atom nil)
        rej (atom nil)
        p (js/Promise.
           (fn [resolve reject]
             (reset! res resolve)
             (reset! rej reject)))]
    (aset p "resolve" @res)
    (aset p "reject" @rej)
    p))

(defn complete-plt [^js/Promise plt value]
  ((aget plt "resolve") value))

(defn barrier
  [f]
  (let [cf (promise-like-type)]
    (f cf #(complete-plt cf true))))

(defn now []
  (.now js/Date))

(defn run-after [^long start ^long delay f]
  (js/setTimeout f (- delay (- (now) start))))

(defn cancel-run-after [handle]
  (js/clearTimeout handle))

(def the-lock
  (reify
    p/ILock
    (lock [_this])
    (unlock [_this])))

(def lock (constantly the-lock))

(defn complete-plt-error [^js/Promise plt error]
  ((aget plt "reject") error))

(extend-type Atom
  p/ConditionVariable
  (get-condition-state [this]
    @this)
  (check-condition-state [this value]
    (identical? @this value))
  (change-condition-state [this old-value new-value]
    (compare-and-set! this old-value new-value))
  (add-condition-listener [this id listener]
    (add-watch this id (fn [& _] (listener id)))
    (listener id))
  (remove-condition-listener [this id]
    (remove-watch this id)))

(defn condition-variable [init-condition-state]
  (atom init-condition-state))

(defn fixed-size-buffer [^long n]
  (let [lst (object-array 0)]
    (reify
      p/Buffer
      (full? [_]
        (<= n (count lst)))
      (add-item [_ item]
        (.push lst item))
      (peek-item [_]
        (aget lst 0))
      (pop-item [_]
        (.shift lst)))))

(defn dropping-buffer [^long n]
  (let [lst (object-array 0)]
    (reify
      p/Buffer
      (full? [_]
        (<= n (count lst)))
      (add-item [_ item]
        (when (> n (count lst))
          (.push lst item))
        item)
      (peek-item [_]
        (aget lst 0))
      (pop-item [_]
        (.shift lst)))))

(defn sliding-buffer [^long n]
  (let [lst (object-array 0)]
    (reify
      p/Buffer
      (full? [_]
        (<= n (count lst)))
      (add-item [_ item]
        (when (> n (count lst))
          (.shift lst))
        (.push lst item)
        item)
      (peek-item [_]
        (aget lst 0))
      (pop-item [_]
        (.shift lst)))))

(defn unhandled-exception [exception]
  (.log js/console
        (str
         (print-str "unhandled error returned\n")
         (prn-str exception))))
