(ns com.manigfeald.machinate
  (:refer-clojure :exclude [send
                            constantly])
  (:require
   [com.manigfeald.machinate.platform :as plt]
   [com.manigfeald.machinate.protocols :as p]
   [clojure.core :as cc]
   [com.manigfeald.machinate.queue :as q]))

#?(:clj (set! *warn-on-reflection* true))

#?(:clj (set! *unchecked-math* :warn-on-boxed))

;; TODO clojureclr?

;; TODO clojured, compiler blows up on ^objects type hint :/

;; TODO io, netty? nio?

;; At the lowest level, this is a library for synchronizing on
;; events. synchronizing on an event means registering a callback to
;; be invoked after the event occurs.

;; acts as never
(extend-type nil
  p/Event
  (try-event [_event _resume _resume-with-error _control _nack-group])
  p/Nackable
  (check-nack-group [_event _nack-group])
  p/QuasiEvent
  (push-down [event ctor lst]
    (when-let [evt (ctor event)]
      (aset (q/add-entry lst) 0 evt)))
  p/Syncable
  (-sync [_event]))

(extend-type #?(:clj Object :cljs default)
  p/QuasiEvent
  (push-down [event ctor lst]
    (aset (q/add-entry lst) 0 (ctor event)))
  p/Syncable
  (-sync [event]
    ;; wrap list access in a volatile to ensure ordering, may not be
    ;; required.
    (let [lst (volatile! (q/new-q 1))
          control (plt/condition-variable :waiting)]
      (p/push-down event identity @lst)
      (letfn [(resume-f [_event _value nack-group]
                ;; resume-f could end up invoked on another thread,
                ;; hence the volatile
                (doseq [^objects x (q/seq @lst)
                        :let [e (aget x 0)]]
                  (p/check-nack-group e nack-group)))
              (unhandled-error [event value nack-group]
                (plt/unhandled-exception value)
                (resume-f event value nack-group))]
        (doseq [^objects e (q/seq @lst)
                :let [evt (aget e 0)]
                :when evt]
          (p/try-event evt resume-f unhandled-error control #{})))))
  p/Nackable
  (check-nack-group [_event _nack-group]))

(defn ^:no-doc constantly* [g value]
  (reify
    p/Event
    (try-event [event resume resume-with-error control nack-group]
      (let [k (plt/unique-key)
            f (fn this-fn [k]
                (when (p/change-condition-state control :waiting :synced)
                  (p/remove-condition-listener control k)
                  ((g resume resume-with-error) event value nack-group)))]
        (p/add-condition-listener control k f)
        (f k)))))

(defn constantly
  "Returns an event that is immediatly synchronizable with the value value"
  [value]
  (constantly* (fn [resume _resume-with-error] resume) value))

(defn constantly-error
  "Returns an event that is immediatly synchronizable with the error value"
  [value]
  (constantly* (fn [_resume resume-with-error] resume-with-error) value))

(def always
  "Event that has always already occured. Synchronization always succeeds
  immediately. Value of the event is true."
  (constantly true))

(declare barrier)

(defn watch-reference
  "Return an event that syncs when the given watchable clojure reference
  type (refs,atoms,agents) has the given value. Event has the value
  true."
  [r value]
  (barrier
   (fn [e b!]
     (let [k (plt/unique-key)
           f (fn [k r _os ns]
               (when (= ns value)
                 (remove-watch r k)
                 ;; could get called more than once in theory, should
                 ;; be fine
                 (b!)))]
       (add-watch r k f)
       (f k r nil @r))
     e)))

(declare middleware)

(deftype Middleware [try-event-transform
                     check-nack-group-transform
                     push-down-transform
                     target]
  p/Event
  (try-event [_ resume resume-with-error control nack-group]
    ((try-event-transform p/try-event) target resume resume-with-error control nack-group))
  p/Nackable
  (check-nack-group [_ nack-group]
    ((check-nack-group-transform p/check-nack-group) target nack-group))
  p/QuasiEvent
  (push-down [_event ctor lst]
    ((push-down-transform p/push-down)
     target
     (comp ctor #(middleware try-event-transform
                             check-nack-group-transform
                             push-down-transform
                             %))
     lst)))

(defn middleware
  "Middleware smart constructor. Middleware wraps an event, passing
  protocols calls down with the opportunity to intercept and make
  changes. The middleware smart constructor can collapse multiple
  middlewares into a single middleware via function composition."
  [try-event-transform check-nack-group-transform push-down-transform target]
  (if (instance? Middleware target)
    (let [^Middleware target target]
      (middleware (comp try-event-transform (.-try-event-transform target))
                  (comp check-nack-group-transform (.-check-nack-group-transform target))
                  (comp push-down-transform (.-push-down-transform target))
                  (.-target target)))
    (->Middleware try-event-transform check-nack-group-transform push-down-transform target)))


(defn barrier
  "Given a function f, applies f to a function g and an event e. e becomes
  occurs when g is invoked and has the value `true`. returns the result of invoking f."
  [f]
  (plt/barrier f))

;; events end up forming a tree, before synchronization that tree gets
;; flattened into something like disjuntive normal form, then at
;; synchronization one of the possible events is chosen.


(defn ^:no-doc choice*
  [evts ordered?]
  (middleware
   identity
   identity
   (fn [push-down]
     (fn [_event ctor lsg]
       (doseq [e (if-not ordered?
                   (shuffle evts)
                   evts)]
         (push-down e ctor lsg))))
   nil))

(defn ordered-choice
  "Takes a collection of events, and attempts to select
event to synchronize on in order."
  [evts]
  (choice* evts true))


(defn choice
  "Takes a collection of events, and non-deterministically selects one on
synchronization by defaults."
  ([evts]
   (choice* evts false)))

(defn nack
  "Like guard, but f is passed an event G. G will occur if on
  synchronization, the chosen event is not the one returned by f."
  [f]
  (middleware
   identity
   identity
   (fn [push-down]
     (fn [_event ctor lsg]
       (barrier
        (fn [b break]
          (let [e (f b)
                id (plt/unique-key)]
            (push-down
             e
             (comp ctor
                   (fn g [evt]
                     (middleware
                      (fn [try-event]
                        (fn [event resume resume-with-error control nack-group]
                          (try-event event resume resume-with-error control (conj nack-group id))))
                      (fn [check-nack-group]
                        (fn [event nack-group]
                          (when-not (contains? nack-group id)
                            (break))
                          (check-nack-group event nack-group)))
                      identity
                      evt)))
             lsg))))))
   nil))

(defn guard
  "Takes a function of no arguments f, returns an event E, before E is
  synchronized on, f will be invoked, and E replaced with the event
  returned by f."
  [f]
  (middleware
   identity
   identity
   (fn [push-down]
     (fn [_event ctor lst]
       (push-down (f) ctor lst)))
   nil))

(defn wrap
  "Wraps the post synchronization action f around the event evt"
  [evt f]
  (middleware
   (fn [try-event]
     (fn [evt resume resume-with-error control nack-group]
       (try-event evt
                  (fn [event value nack-group]
                    (try
                      (resume event (f value) nack-group)
                      (catch #?(:clj Throwable :cljs :any) t
                        (resume-with-error event t nack-group))))
                  resume-with-error
                  control
                  nack-group)))
   identity
   identity
   evt))

(defn wrap-handler
  "Wrap an error handler around an event. Handlers errors coming out of
the event itself, or any errors arising from other wrappings of the
event."
  [evt f]
  (middleware
   (fn [try-event]
     (fn [evt resume resume-with-error control nack-group]
       (try-event evt
                  resume
                  (fn [event value nack-group]
                    (try
                      (resume event (f value) nack-group)
                      (catch #?(:clj Throwable :cljs :any) t
                        (resume-with-error event t nack-group))))
                  control
                  nack-group)))
   identity
   identity
   evt))

(defn alts
  "A work-a-like for clojure.core.async/alts!. Given a collection of
events returns a new event that when sync'ed chooses one event and
returns [event-value event]."
  [events]
  (choice
   (for [event events]
     (wrap event (fn [result] [result event])))))

(defn timeout
  "Given a delay in milliseconds, returns an event that doesn't
  synchronize until after that delay has elapsed. Value of the event
  is true."
  [^long delay]
  (let [at (plt/now)]
    (barrier
     (fn [b break]
       (nack
        (fn [neg]
          (let [handle (plt/run-after at delay break)]
            (p/-sync (wrap neg (fn [_] (plt/cancel-run-after handle))))
            b)))))))

(declare duplex-channel-event
         channel-close!)

;; The default rendevous channel, unbuffered, no transforms, pure send
;; and receive
(deftype DuplexChannel [lock is-closed left-head right-head]
  p/AsymmetricExchange
  (send-left [channel value]
    (duplex-channel-event channel left-head value))
  (send-right [channel value]
    (duplex-channel-event channel right-head value))
  p/SendChannel
  (send-it [this it]
    (p/send-left this it))
  p/ReceiveChannel
  (receive-it [this]
    (p/send-right this true))
  p/Channel
  (close! [this]
    (channel-close! this)))

(defn channel
  "Returns full duplex channel, both a SendChannel and a ReceiveChannel"
  []
  (let [lock (doto (plt/lock)
               (p/lock))
        c (DuplexChannel.
           lock
           (plt/condition-variable :waiting)
           (q/new-q 6)
           (q/new-q 6))]
    (p/unlock lock)
    c))

(declare sync!
         close!
         full-duplex)

(defn fixed-size-buffer
  "A fixed size buffer for buffered-channel. Accepts inputs until full."
  [^long capacity]
  (plt/fixed-size-buffer capacity))

(defn dropping-buffer
  "A dropping buffer for buffered-channel. Always accepts inputs, but
inputs are silently dropped if the buffer is full."
  [^long capacity]
  (plt/dropping-buffer capacity))

(defn sliding-buffer
  "A sliding buffer for buffered-channel. Always accepts inputs, but
drops the oldest item if the buffer is full."
  [^long capacity]
  (plt/sliding-buffer capacity))

;; TODO linked list buffer, volatile field of most recently consumed entry

;; buffered channel is two channels internally with a loop copying
;; between them with a buffer in the middle.
(defn buffered-channel
  "Create a buffered full duplex channel. The buffer allows for senders
  to add to the channel without waiting for a receiver until the
  buffer's capacity is reached. Buffered channels optional can have a
  transducer (xform) which transforms values that flow through the
  channel. If a transducer is specified, optional an exception halder
  can be specified to handle any exceptions that may arise."
  ([buffer-or-n]
   (buffered-channel buffer-or-n identity))
  ([buffer-or-n xform]
   (buffered-channel buffer-or-n xform (comp (constantly nil) plt/unhandled-exception)))
  ([buffer-or-n xform ex-handler]
   (assert (or (not (number? buffer-or-n))
               (pos? (long buffer-or-n))))
   (let [b (volatile!
            (if (number? buffer-or-n)
              (fixed-size-buffer buffer-or-n)
              buffer-or-n))
         in-c (channel)
         out-c (channel)
         add! (xform (fn
                       ([b] b)
                       ([b item]
                        (assert (not (nil? item)))
                        (p/add-item b item))))]
     ;; TODO maybe needs some kind of trampoline?
     ((fn copy-loop [open?]
        (let [item (p/peek-item @b)]
          (sync!
           (choice
            [(if (some? item)
               (wrap (p/send-it out-c item)
                     (fn [_sent-result]
                       (p/pop-item @b)
                       (copy-loop open?)))
               (when-not open?
                 (p/close! out-c)
                 nil))
             (if open?
               (if (p/full? @b)
                 nil
                 (wrap (p/receive-it in-c)
                       (fn [new-item]
                         (if (some? new-item)
                           (do
                             (when (reduced?
                                    (try
                                      (add! @b new-item)
                                      (catch #?(:clj Throwable :cljs :any) t
                                        (if-some [r (ex-handler t)]
                                          (p/add-item @b r)
                                          @b))))
                               (close! in-c))
                             (copy-loop open?))
                           (do
                             (try
                               (add! @b)
                               (catch #?(:clj Throwable :cljs :any) t
                                 (if-some [r (ex-handler t)]
                                   (p/add-item @b r)
                                   @b)))
                             (copy-loop false))))))
               nil)]))))
      true)
     (full-duplex in-c out-c))))

(defn ^:no-doc channel-close! [^DuplexChannel channel]
  (let [lock (.-lock channel)
        _ (p/lock lock)
        is-closed (.-is-closed channel)]
    (p/unlock lock)
    (p/change-condition-state is-closed :waiting :synced)
    nil))

(def ^:no-doc the-empty (plt/unique-key))

(def ^:no-doc locked (plt/unique-key))

(defn ^:no-doc duplex-channel-event
  [^DuplexChannel channel' write-head value]
  (let [lock      (.-lock channel')
        result    (plt/condition-variable the-empty)
        is-closed (.-is-closed channel')]
    (middleware
     (fn [_try-event]
       (fn [event resume _resume-with-error control nack-group]
         (let [v (p/get-condition-state result)]
           (if (and (not (identical? v the-empty))
                    (not (identical? v locked))
                    (p/change-condition-state control :waiting :synced))
             (resume event v nack-group)
             (do
               (p/lock lock)
               (let [entry    (doto (q/add-entry write-head)
                                (aset 0 resume)
                                (aset 1 control)
                                (aset 2 nack-group)
                                (aset 3 value)
                                (aset 4 result)
                                (aset 5 event))
                     k        (plt/unique-key)
                     do-stuff (volatile!
                               (fn [r v']
                                 (vreset! r nil)
                                 (resume event v' nack-group)))
                     f        (fn [k]
                                (when (p/check-condition-state control :synced)
                                  (p/remove-condition-listener control k)
                                  (p/remove-condition-listener result k)
                                  (p/lock lock)
                                  (q/unlink write-head entry)
                                  (p/unlock lock))
                                (let [v' (p/get-condition-state result)]
                                  (when (and (not (identical? v' the-empty))
                                             (not (identical? v' locked)))
                                    (when (p/change-condition-state control :waiting :synced)
                                      (p/remove-condition-listener result k)
                                      (@do-stuff do-stuff v')))))]
                 (p/add-condition-listener result k f)
                 (p/add-condition-listener control k f)
                 (p/add-condition-listener
                  is-closed
                  k
                  (fn [k]
                    (when (p/check-condition-state is-closed :synced)
                      (p/lock lock)
                      (loop []
                        (if-let [^objects left (q/peek (.-left-head channel'))]
                          (if-let [^objects right (q/peek (.-right-head channel'))]
                            nil
                            (let [[left-resume left-control left-ng _left-value left-result left-event] left]
                              (if (p/change-condition-state left-control :waiting :synced)
                                (do 
                                  (p/unlock lock)
                                  (left-resume left-event nil left-ng)
                                  (p/change-condition-state left-result locked nil)
                                  (p/lock lock)
                                  (recur))
                                (do
                                  (p/unlock lock)
                                  (p/lock lock)
                                  (recur)))))
                          (if-let [^objects right (q/peek (.-right-head channel'))]
                            (let [[right-resume right-control right-ng _right-value right-result right-event] right]
                              (if (p/change-condition-state right-control :waiting :synced)
                                (do 
                                  (p/unlock lock)
                                  (right-resume right-event nil right-ng)
                                  (p/change-condition-state right-result locked nil)
                                  (p/lock lock)
                                  (recur))
                                (do
                                  (p/unlock lock)
                                  (p/lock lock)
                                  (recur))))
                            (p/unlock lock))))
                      (p/remove-condition-listener is-closed k))))
                 (loop []
                   (if-let [^objects left (q/peek (.-left-head channel'))]
                     (if-let [^objects right (q/peek (.-right-head channel'))]
                       (let [[left-resume left-control left-ng left-value left-result left-event]       left
                             [right-resume right-control right-ng right-value right-result right-event] right]
                         (if-not (p/check-condition-state left-control :synced)
                           (if-not (p/check-condition-state right-control :synced)
                             (if (p/change-condition-state left-control :waiting :claimed)
                               (if (p/change-condition-state left-result the-empty locked)
                                 (if (p/change-condition-state right-control :waiting :claimed)
                                   (if (p/change-condition-state right-result the-empty locked)
                                     (do
                                       (p/change-condition-state right-control :claimed :synced)
                                       (p/change-condition-state left-control :claimed :synced)
                                       (p/unlock lock)
                                       ;; don't hold any locks by the time callbacks are called
                                       (left-resume left-event right-value left-ng)
                                       (right-resume right-event left-value right-ng)
                                       (p/change-condition-state left-result locked right-value)
                                       (p/change-condition-state right-result locked left-value)
                                       (p/lock lock)
                                       (recur))
                                     (do
                                       (p/change-condition-state right-control :claimed :waiting)
                                       (p/change-condition-state left-result locked the-empty)
                                       (p/change-condition-state left-control :claimed :waiting)
                                       (p/unlock lock)
                                       (p/lock lock)
                                       (recur)))
                                   (do
                                     (p/change-condition-state left-result locked the-empty)
                                     (p/change-condition-state left-control :claimed :waiting)
                                     (p/unlock lock)
                                     (p/lock lock)
                                     (recur)))
                                 (do
                                   (p/change-condition-state left-control :claimed :waiting)
                                   (p/unlock lock)
                                   (p/lock lock)
                                   (recur)))
                               (do
                                 (p/unlock lock)
                                 (p/lock lock)
                                 (recur)))
                             (do
                               (q/unlink (.-right-head channel') right)
                               (p/unlock lock)
                               (p/lock lock)
                               (recur)))
                           (do
                             (q/unlink (.-left-head channel') left)
                             (p/unlock lock)
                             (p/lock lock)
                             (recur))))
                       (if (p/check-condition-state is-closed :synced)
                         (let [[left-resume left-control left-ng _left-value left-result left-event] left]
                           (if (p/change-condition-state left-control :waiting :synced)
                             (do 
                               (p/unlock lock)
                               (left-resume left-event nil left-ng)
                               (p/change-condition-state left-result locked nil)
                               (p/lock lock)
                               (recur))
                             (do
                               (p/unlock lock)
                               (p/lock lock)
                               (recur))))
                         (p/unlock lock)))
                     (if-let [^objects right (q/peek (.-right-head channel'))]
                       (if (p/check-condition-state is-closed :synced)
                         (let [[right-resume right-control right-ng _right-value right-result right-event] right]
                           (if (p/change-condition-state right-control :waiting :synced)
                             (do 
                               (p/unlock lock)
                               (right-resume right-event nil right-ng)
                               (p/change-condition-state right-result locked nil)
                               (p/lock lock)
                               (recur))
                             (do
                               (p/unlock lock)
                               (p/lock lock)
                               (recur))))
                         (p/unlock lock))
                       (p/unlock lock))))))))))
     identity
     identity
     nil)))

(defn send
  "Creates an event where synchronization completes after the given value
  has been sent via the given channel, or after the given channel
  closes. Returns true or false"
  [channel value]
  (assert (some? value))
  (p/send-it channel value))

(defn receive
  "Creates an event where synchronization completes after receivng a value
  via the given channel or the channel is closed. Returns the received
  value or nil."
  [channel]
  (p/receive-it channel))

(defn close!
  "Close a channel."
  [channel]
  (p/close! channel))

(defn full-duplex
  "Given a SendChannel and a ReceiveChannel return a single channel that
is both send and receivable."
  [send-channel receive-channel]
  (reify
    p/SendChannel
    (send-it [_ it]
      (p/send-it send-channel it))
    p/ReceiveChannel
    (receive-it [_]
      (p/receive-it receive-channel))
    p/Channel
    (close! [_]
      (close! send-channel))))

(defn send-only
  "Given a full duplex channel return a channel that is only a
SendChannel"
  [channel]
  (reify
    p/SendChannel
    (send-it [_ it]
      (p/send-it channel it))
    p/Channel
    (close! [_]
      (close! channel))))

(defn receive-only
  "Given a full duplex channel return a channel that is only a
ReceiveChannel"
  [channel]
  (reify
    p/ReceiveChannel
    (receive-it [_]
      (p/receive-it channel))
    p/Channel
    (close! [_]
      (close! channel))))

(defn sync!
  "Run a callback after the given event occurs. Returns a
CompletableFuture that will be completed after the event occurs."
  ([evt]
   (let [cf (plt/promise-like-type)]
     (p/-sync
      (-> evt
          (wrap (fn [value] (plt/complete-plt cf value)))
          (wrap-handler (fn [exception] (plt/complete-plt-error cf exception)))))
     cf))
  ([evt callback]
   (sync! (wrap evt callback))))

#?(:clj
   (defn sync!!
     "Stop and wait for the given event to occur"
     [evt]
     @(sync! evt)))

(defn pubsub
  "A pubsub allows messages to be sent to topics. input is a channel of
messages, topic-fn is message -> topic. All listeners to a topic are
sent messages in parallel. The pubsub will not move on to the next
message until all listeners have recieved a given message."
  [input topic-fn]
  (letfn [(mass-send [value outputs]
            (barrier
             (fn [evt b!]
               (let [c (count outputs)
                     results (atom outputs)
                     e (plt/condition-variable 0)]
                 (p/add-condition-listener e
                                           (plt/unique-key)
                                           (fn [_]
                                             (when (= c (p/get-condition-state e))
                                               (b!))))
                 (doseq [[channel _] outputs]
                   (sync! (wrap (send channel value)
                                (fn [send-result]
                                  (when-not send-result
                                    (swap! results dissoc channel))
                                  (loop []
                                    (let [n (long (p/get-condition-state e))]
                                      (when-not (p/change-condition-state e n (inc n))
                                        (recur))))))))
                 (wrap evt (fn [_] @results))))))
          (main-loop [commands outputs]
            (sync!
             (choice
              [(wrap (receive commands)
                     (fn [command]
                       (case (nth command 0)
                         :subscribe (main-loop commands
                                               (assoc-in outputs [(nth command 1)
                                                                  (nth command 2)]
                                                         (nth command 3)))
                         :unsubscribe (let [o (dissoc (get outputs (nth command 1)) (nth command 2) )]
                                        (if (seq o)
                                          (main-loop commands (assoc outputs (nth command 1) o))
                                          (main-loop commands (dissoc outputs (nth commands 1))))))))
               (wrap (receive input)
                     (fn [new-msg]
                       (if new-msg
                         (let [t (topic-fn new-msg)
                               topic-outputs (get outputs t)]
                           (sync! (wrap (mass-send new-msg topic-outputs)
                                        (fn [topic-outputs]
                                          (if (seq topic-outputs)
                                            (main-loop commands
                                                       (assoc outputs t topic-outputs))
                                            (main-loop commands
                                                       (dissoc outputs t)))))))
                         (doseq [[_topic topic-outputs] outputs
                                 output topic-outputs]
                           (when (:close? output)
                             (p/close! (:ch output)))))))])))]
    (let [commands (buffered-channel 1024)]
      (main-loop commands {})
      (reify
        p/PubSub
        (subscribe [_ topic channel close?]
          (wrap (send commands [:subscribe topic channel close?]) (cc/constantly channel)))
        (unsubscribe [_ topic channel]
          (wrap (send commands [:unsubscribe topic channel])
                (cc/constantly channel)))
        p/SendChannel
        (send-it [_ it]
          (p/send-it input it))))))

(defn sub
  "Given a pubsub add a channel as a listener to a given topic. Returns
an event that must be synchronized on. After that event occurs the
  listener has been added to the pubsub. If close? is true then the
  listener channel will be closed if the input to the pubsub is
  closed."
  ([ps topic channel]
   (sub ps topic channel true))
  ([ps topic channel close?]
   (p/subscribe ps topic channel close?)))

(def sub! (comp sync! sub))

(defn unsub
  "Remove a channel as a listener for the given topic from a
pubsub. Returns an event that must be synchronized on."
  [ps topic channel]
  (p/unsubscribe ps topic channel))

(def unsub! (comp sync! unsub))
