(ns com.manigfeald.machinate.platform
  (:require [com.manigfeald.machinate.protocols :as p])
  (:import (java.util LinkedList
                      List)
           (java.util.concurrent CompletableFuture DelayQueue TimeUnit Delayed
                                 ConcurrentHashMap
                                 LinkedBlockingQueue)
           (java.util.function BiFunction)
           (java.util.concurrent.locks ReentrantLock)
           (java.util.concurrent.atomic AtomicReference)))

(set! *warn-on-reflection* true)

(set! *unchecked-math* :warn-on-boxed)

(defn print-to-error [message]
  (binding [*out* *err*]
    (print message)
    (flush)))

(defn unique-key []
  (Object.))

;; TODO check resource usage, is cancel on cleanup actually cleaning stuff up

;; TODO CompletionStage doesn't come with a cancel method, maybe a new
;; protocol so consumer can add one?
(extend-type CompletableFuture
  p/Event
  (try-event [^CompletableFuture event resume resume-with-error control nack-group]
    (let [action (volatile! (fn [result exception]
                              (if exception
                                (resume-with-error event exception nack-group)
                                (resume event result nack-group))
))
          ^CompletableFuture x
          (.handle event
                   (reify
                     BiFunction
                     (apply [_ result exception]
                       (let [k (Object.)
                             f (fn [k]
                                 (if-let [f @action]
                                   (when (p/change-condition-state control :waiting :synced)
                                     (p/remove-condition-listener control k)
                                     (f result exception))
                                   (p/remove-condition-listener control k)))]
                         (p/add-condition-listener control (Object.) f)))))]
      (p/add-condition-listener control
                                (Object.)
                                (fn [k]
                                  (when-not @action
                                    ;; TODO never gets called?
                                    (.cancel x false))
                                  (when (p/check-condition-state control :synced)
                                    (vreset! action nil)
                                    (p/remove-condition-listener control k)))))))



(defn barrier
  "Given a function f, applies f to a function g and an event e. e becomes
  enabled when g is invoked. returns the result of invoking f."
  [f]
  (let [cf (CompletableFuture.)]
    (f cf (fn [] (.complete cf true)))))


(def ^DelayQueue ^:no-doc delay-queue (DelayQueue.))

(deftype D [action ^long delay ^long at]
  Delayed
  (getDelay [_ timeunit]
    (.convert timeunit
              (- delay (- (System/currentTimeMillis) at))
              TimeUnit/MILLISECONDS))
  Comparable
  (compareTo [_ other]
    (let [now (System/currentTimeMillis)]
      (compare (- delay (- now at))
               (- (.-delay ^D other) (- now (.-at ^D other)))))))

(def ^:no-doc process-delay-queue
  (delay
    (doto (Thread.
           (fn []
             (while true
               (try
                 ((.-action ^D (.take delay-queue)))
                 (catch Throwable _
                   ;; TODO report in some way?
                   )))))
      (.setDaemon true)
      (.setName "machinate-process-delay-queue")
      (.start))))

(defn now []
  (System/currentTimeMillis))

(defn run-after [start delay f]
  @process-delay-queue
  (let [d (->D f delay start)]
    (.add delay-queue d)
    d))

(defn cancel-run-after [handle]
  (.remove delay-queue handle))

(defn lock []
  (ReentrantLock.))

(extend-type ReentrantLock
  p/ILock
  (lock [this]
    (.lock this))
  (unlock [this]
    (.unlock this)))

(defn promise-like-type []
  (CompletableFuture.))

(defn complete-plt [^CompletableFuture plt value]
  (.complete plt value))

(defn complete-plt-error [^CompletableFuture plt error]
  (.completeExceptionally plt error))

;; TODO clean up, figure out clojurescript

;; TODO notifyAll

;; TODO clean up old listeners?

(deftype AConditionVariable [^AtomicReference state
                             ^ConcurrentHashMap waiters]
  p/ConditionVariable
  (get-condition-state [_]
    (.get state))
  (check-condition-state [_ value]
    (identical? (.get state) value))
  (change-condition-state [_this old-value new-value]
    (if (.compareAndSet state old-value new-value)
      (do
        (doseq [[k v] waiters]
          (v k))
        true)
      false))
  (add-condition-listener [_ id listener]
    (.put waiters id listener)
    (listener id)
    nil)
  (remove-condition-listener [_ id]
    (.remove waiters id)
    nil)
  Object
  (toString [_]
    (format "CondVar(%s)" (.get state))))

(defn condition-variable [init-condition-state]
  (->AConditionVariable (AtomicReference. init-condition-state)
                        (ConcurrentHashMap.)))


(defn fixed-size-buffer [^long n]
  (let [lst (LinkedList.)]
    (reify
      p/Buffer
      (full? [_]
        (<= n (count lst)))
      (add-item [_ item]
        (.addLast lst item))
      (peek-item [_]
        (.peekFirst lst))
      (pop-item [_]
        (.pop lst)))))
