(ns com.manigfeald.machinate
  (:refer-clojure :exclude [send
                            constantly])
  (:require
   [com.manigfeald.machinate.platform :as plt]
   [com.manigfeald.machinate.protocols :as p]
   [clojure.core :as cc]
   [com.manigfeald.machinate.queue :as q]))

#?(:clj (set! *warn-on-reflection* true))

#?(:clj (set! *unchecked-math* :warn-on-boxed))

;; TODO manifold interop

;; TODO other hosts? clojureclr clojured?

;; TODO io, netty? nio?

;; TODO nack power of 2 choices demo

;; At the lowest level, this is a library for synchronizing on
;; events. synchronizing on an event means registering a callback to
;; be invoked after the event occurs.

;; acts as never
(extend-type nil
  p/Event
  (try-event [_event _resume _resume-with-error _control _nack-group])
  p/Nackable
  (check-nack-group [_event _nack-group])
  p/QuasiEvent
  (push-down [event ctor lst]
    (when-let [evt (ctor event)]
      (aset (q/add-entry lst) 0 evt)))
  p/Syncable
  (-sync [_event]))

(extend-type #?(:clj Object :cljs default)
  p/QuasiEvent
  (push-down [event ctor lst]
    (aset (q/add-entry lst) 0 (ctor event)))
  p/Syncable
  (-sync [event]
    ;; wrap list access in a volatile to ensure ordering, may not be
    ;; required.
    (let [lst (volatile! (q/new-q 1))
          control (plt/condition-variable :waiting)]
      (p/push-down event identity @lst)
      (letfn [(resume-f [_event _value nack-group]
                ;; resume-f could end up invoked on another thread,
                ;; hence the volatile
                (doseq [^objects x (q/seq @lst)
                        :let [e (aget x 0)]]
                  (p/check-nack-group e nack-group)))
              (unhandled-error [event value nack-group]
                (plt/print-to-error
                 (str
                  (print-str "unhandled error returned\n")
                  (prn-str value)))
                (resume-f event value nack-group))]
        (doseq [^objects e (q/seq @lst)
                :let [evt (aget e 0)]
                :when evt]
          (p/try-event evt resume-f unhandled-error control #{})))))
  p/Nackable
  (check-nack-group [_event _nack-group]))

(defn ^:no-doc constantly* [g value]
  (reify
    p/Event
    (try-event [event resume resume-with-error control nack-group]
      (let [k (plt/unique-key)
            f (fn this-fn [k]
                (when (p/change-condition-state control :waiting :synced)
                  (p/remove-condition-listener control k)
                  ((g resume resume-with-error) event value nack-group)))]
        (p/add-condition-listener control k f)
        (f k)))))

(defn constantly
  "Returns an event that is immediatly synchronizable with the value value"
  [value]
  (constantly* (fn [resume _resume-with-error] resume) value))

(defn constantly-error
  "Returns an event that is immediatly synchronizable with the error value"
  [value]
  (constantly* (fn [_resume resume-with-error] resume-with-error) value))

(def always
  "Event that has always already occured. Synchronization always succeeds
  immediately. Value of the event is true."
  (constantly true))

(declare barrier)

(defn watch-reference
  "Return an event that syncs when the given watchable clojure reference
  type (refs,atoms,agents) has the given value. Event has the value
  true."
  [r value]
  (barrier
   (fn [e b!]
     (let [k (plt/unique-key)
           f (fn [k r _os ns]
               (when (= ns value)
                 (remove-watch r k)
                 ;; could get called more than once in theory, should
                 ;; be fine
                 (b!)))]
       (add-watch r k f)
       (f k r nil @r))
     e)))

(declare middleware)

(deftype Middleware [try-event-transform
                     check-nack-group-transform
                     push-down-transform
                     target]
  p/Event
  (try-event [_ resume resume-with-error control nack-group]
    ((try-event-transform p/try-event) target resume resume-with-error control nack-group))
  p/Nackable
  (check-nack-group [_ nack-group]
    ((check-nack-group-transform p/check-nack-group) target nack-group))
  p/QuasiEvent
  (push-down [_event ctor lst]
    ((push-down-transform p/push-down)
     target
     (comp ctor #(middleware try-event-transform
                             check-nack-group-transform
                             push-down-transform
                             %))
     lst)))

(defn middleware
  "Middleware smart constructor. Middleware wraps an event, passing
  protocols calls down with the opportunity to intercept and make
  changes. The middleware smart constructor can collapse multiple
  middlewares into a single middleware via function composition."
  [try-event-transform check-nack-group-transform push-down-transform target]
  (if (instance? Middleware target)
    (let [^Middleware target target]
      (middleware (comp try-event-transform (.-try-event-transform target))
                  (comp check-nack-group-transform (.-check-nack-group-transform target))
                  (comp push-down-transform (.-push-down-transform target))
                  (.-target target)))
    (->Middleware try-event-transform check-nack-group-transform push-down-transform target)))


(defn barrier
  "Given a function f, applies f to a function g and an event e. e becomes
  occurs when g is invoked and has the value `true`. returns the result of invoking f."
  [f]
  (plt/barrier f))

#_(defn barrier
    "Given a function f, applies f to a function g and an event e. e becomes
  enabled when g is invoked. returns the result of invoking f."
  [f]
  (let [root (AtomicReference.)
        done (Object.)
        event (middleware
               (fn [event-transform]
                 (fn [event resume control nack-group cleanup]
                   (listen-to-state-change control (Object.)
                                           (fn [k]
                                             (when (check-state control :synced)
                                               (loop [^AtomicReferenceArray r (.get root)]
                                                 (when r
                                                   (when-not (identical? r done)
                                                     (if (identical? control (.get r 1))
                                                       (doto r
                                                         (.set 0 nil)
                                                         (.set 1 nil)
                                                         (.set 2 nil))
                                                       (recur (.get r 3))))))
                                               (remove-watch control k))))
                   (let [cell (doto (AtomicReferenceArray. 4)
                                (.set 0 resume)
                                (.set 1 control)
                                (.set 2 nack-group))]
                     (loop [r (.get root)]
                       (if (identical? r done)
                         (loop []
                           (if (change-state control :waiting :synced)
                             (resume event nil nack-group)
                             (when-not (check-state control :synced)
                               (recur))))
                         (do
                           (.set cell 3 r)
                           (when-not (.compareAndSet root r cell)
                             (recur (.get root)))))))))
               (fn [check-nack-group]
                 (fn [event nack-group]))
               identity
               nil)]
    (f event
       (fn []
         (loop [^AtomicReferenceArray r (.get root)]
           (when-not (identical? r done)
             (if-not (.compareAndSet root r done)
               (recur (.get root))
               (loop [r r]
                 (when r
                   (let [resume (.get r 0)
                         control (.get r 1)
                         nack-group (.get r 2)]
                     (when (and resume control nack-group)
                       (loop []
                         (if (change-state control :waiting :synced)
                           (resume event nil nack-group)
                           (when-not (check-state control :synced)
                             (recur)))))
                     (recur (.get r 3))))))))))))


;; events end up forming a tree, before synchronization that tree gets
;; flattened into something like disjuntive normal form, then at
;; synchronization one of the possible events is chosen.


(defn ^:no-doc choice*
  [evts ordered?]
  (middleware
   identity
   identity
   (fn [push-down]
     (fn [_event ctor lsg]
       (doseq [e (if-not ordered?
                   (shuffle evts)
                   evts)]
         (push-down e ctor lsg))))
   nil))

(defn ordered-choice
  "Takes a collection of events, and attempts to select
event to synchronize on in order."
  [evts]
  (choice* evts true))


(defn choice
  "Takes a collection of events, and non-deterministically selects one on
synchronization by defaults."
  ([evts]
   (choice* evts false)))

(defn nack
  "Like guard, but f is passed an event G. G will occur if on
  synchronization, the chosen event is not the one returned by f."
  [f]
  (middleware
   identity
   identity
   (fn [push-down]
     (fn [_event ctor lsg]
       (barrier
        (fn [b break]
          (let [e (f b)
                id (plt/unique-key)]
            (push-down
             e
             (comp ctor
                   (fn g [evt]
                     (middleware
                      (fn [try-event]
                        (fn [event resume resume-with-error control nack-group]
                          (try-event event resume resume-with-error control (conj nack-group id))))
                      (fn [check-nack-group]
                        (fn [event nack-group]
                          (when-not (contains? nack-group id)
                            (break))
                          (check-nack-group event nack-group)))
                      identity
                      evt)))
             lsg))))))
   nil))

(defn guard
  "Takes a function of no arguments f, returns an event E, before E is
  synchronized on, f will be invoked, and E replaced with the event
  returned by f."
  [f]
  (middleware
   identity
   identity
   (fn [push-down]
     (fn [_event ctor lst]
       (push-down (f) ctor lst)))
   nil))

(defn wrap
  "Wraps the post synchronization action f around the event evt"
  [evt f]
  (middleware
   (fn [try-event]
     (fn [evt resume resume-with-error control nack-group]
       (try-event evt
                  (fn [event value nack-group]
                    (try
                      (resume event (f value) nack-group)
                      (catch #?(:clj Throwable :cljs :any) t
                        (resume-with-error event t nack-group))))
                  resume-with-error
                  control
                  nack-group)))
   identity
   identity
   evt))

(defn wrap-handler
  "Wrap an error handler around an event. Handlers errors coming out of
the event itself, or any errors arising from other wrappings of the
event."
  [evt f]
  (middleware
   (fn [try-event]
     (fn [evt resume resume-with-error control nack-group]
       (try-event evt
                  resume
                  (fn [event value nack-group]
                    (try
                      (resume event (f value) nack-group)
                      (catch #?(:clj Throwable :cljs :any) t
                        (resume-with-error event t nack-group))))
                  control
                  nack-group)))
   identity
   identity
   evt))

(defn alts
  "A work-a-like for clojure.core.async/alts!. Given a collection of
events returns a new event that when sync'ed chooses one event and
returns [event-value event]."
  [events]
  (choice
   (for [event events]
     (wrap event (fn [result] [result event])))))

;; (defonce scheduler
;;   (delay
;;     (Executors/newScheduledThreadPool
;;      1
;;      (reify
;;        ThreadFactory
;;        (newThread [_ r]
;;          (doto (Thread. r)
;;            (.setDaemon true)))))))




;; canceling a scheduled future doesn't actually remove it from the
;; queue, annoyingly

;; (defn timeout [delay]
;;   (let [now (System/nanoTime)
;;         n (* 1000000 (long delay))
;;         deadline (+ n now)]
;;     (barrier
;;      (fn [b break]
;;        (nack
;;         (fn [neg]
;;           (let [now (System/nanoTime)
;;                 delay (- deadline now)]
;;             (if (pos? delay)
;;               (let [f (.schedule ^ScheduledExecutorService @scheduler
;;                                  ^Runnable break
;;                                  delay
;;                                  TimeUnit/NANOSECONDS)]
;;                 (p/-sync (wrap neg (fn [_]
;;                                      (.cancel f false))))
;;                 b)
;;               always))))))))


(defn timeout
  "Given a delay in milliseconds, returns an event that doesn't
  synchronize until after that delay has elapsed. Value of the event
  is true."
  [^long delay]
  (let [at (plt/now)]
    (barrier
     (fn [b break]
       (nack
        (fn [neg]
          (let [handle (plt/run-after at delay break)]
            (p/-sync (wrap neg (fn [_] (plt/cancel-run-after handle))))
            b)))))))

(declare duplex-channel-event
         channel-close!)

;; The default rendevous channel, unbuffered, no transforms, pure send
;; and receive
(deftype DuplexChannel [fields]
  p/AsymmetricExchange
  (send-left [channel value]
    #_(duplex-channel-event channel 3 5 value)
    (duplex-channel-event channel 3 value)
    )
  (send-right [channel value]
    #_(duplex-channel-event channel 5 3 value)
    (duplex-channel-event channel 5 value))
  p/SendChannel
  (send-it [this it]
    (p/send-left this it))
  p/ReceiveChannel
  (receive-it [this]
    (p/send-right this true))
  p/Channel
  (close! [this]
    (channel-close! this)))

;; (defmacro def-mutable-struct [name- & fields]
;;   (let [n (count fields)]
;;     `(do
;;        (defn ~(symbol (str (name name-) "-new")) []
;;          (object-array ~n))
;;        ~@(for [[idx field] (map-indexed vector fields)
;;                :let [obj (with-meta (gensym 'obj) {:tag 'objects})]]
;;            `(defn ~(symbol (str (name name-) "-" (name field)))
;;               ([~obj] (aget ~obj ~idx))
;;               ([~obj value#] (aset ~obj ~idx value#))))
;;        ~@(for [[idx field] (map-indexed vector fields)]
;;            `(def ~(symbol (str (name name-) "-" (name field) "-n")) ~idx)))))

;; (def-mutable-struct achannel lock closed writers-head writers-tail readers-head readers-tail)

(defn channel
  "Returns full duplex channel, both a SendChannel and a ReceiveChannel"
  []
  (let [lock (doto (plt/lock)
               (p/lock))
        ;; TODO doesn't need mutable fields any more
        c (->DuplexChannel 
           (doto (object-array 7)
             (aset 0 lock)
             (aset 1 (plt/condition-variable :waiting))
             (aset 2 nil) ; unused
             ;; TODO 9 is too large here
             (aset 3 (q/new-q 6)) ; writers head
             (aset 4 nil) ; writers tail
             (aset 5 (q/new-q 6)) ; readers head
             (aset 6 nil) ; readers tail
             ))]
    (p/unlock lock)
    c))

(declare sync!)

;; TODO docs

(defn fixed-size-buffer [^long n]
  (plt/fixed-size-buffer n))

;; TODO exception handling for xform

;; TODO docs

;; TODO buffer seems off by one?

;; TODO ring buffer of expandable entires for xforms

;; buffered channel is two channels internally with a loop copying
;; between them with a buffer in the middle.
(defn buffered-channel
  ([buffer-or-n]
   (buffered-channel buffer-or-n identity))
  ([buffer-or-n xform]
   (let [b (volatile!
            (if (number? buffer-or-n)
              (fixed-size-buffer buffer-or-n)
              buffer-or-n))
         in-c (channel)
         out-c (channel)
         ;; TODO nil item check
         add! (xform (fn
                       ([b] b)
                       ([b item] (p/add-item b item))))]
     ;; TODO maybe needs some kind of trampoline?
     ((fn copy-loop [open?]
        (let [item (p/peek-item @b)]
          (sync!
           (choice
            [(if (some? item)
               (wrap (p/send-it out-c item)
                     (fn [_sent-result]
                       (p/pop-item @b)
                       (copy-loop open?)))
               (when-not open?
                 (p/close! out-c)
                 nil))
             (if open?
               (if (p/full? @b)
                 nil
                 (wrap (p/receive-it in-c)
                       (fn [new-item]
                         (if (some? new-item)
                           (do
                             ;; TODO figure out handling of reduced
                             (add! @b new-item)
                             (copy-loop open?))
                           (copy-loop false)))))
               nil)]))))
      true)
     (reify
       p/SendChannel
       (send-it [_ it]
         (p/send-it in-c it))
       p/ReceiveChannel
       (receive-it [_]
         (p/receive-it out-c))
       p/Channel
       (close! [_]
         (p/close! in-c))))))

(defn ^:no-doc channel-close! [^DuplexChannel channel]
  (let [^objects channel (.-fields channel)
        lock (aget channel 0)
        _ (p/lock lock)
        is-closed (aget channel 1)]
    (p/unlock lock)
    (p/change-condition-state is-closed :waiting :synced)
    nil))

;; TODO exchange with a buffer to fill up?



(def ^:no-doc the-empty ::the-empty #_(plt/unique-key)) ;; TODO
(def ^:no-doc locked ::locked #_(plt/unique-key)) ;;TODO


(defn ^:no-doc duplex-channel-event
  [^DuplexChannel channel' writer-index value]
  (let [^objects channel (.-fields channel')
        write-head       (aget channel writer-index)
        lock             (aget channel 0)
        result           (plt/condition-variable the-empty)
        is-closed        (aget channel 1)]
    (middleware
     (fn [_try-event]
       (fn [event resume _resume-with-error control nack-group]
         (let [v (p/get-condition-state result)]
           (if (and (not (identical? v the-empty))
                    (not (identical? v locked))
                    (p/change-condition-state control :waiting :synced))
             (resume event v nack-group)
             (do
               (p/lock lock)
               ;; TODO shrink entries?
               (let [entry (doto (q/add-entry write-head)
                             (aset 0 resume)
                             (aset 1 control)
                             (aset 2 nack-group)
                             (aset 3 value)
                             (aset 4 result)
                             (aset 5 event))
                     k     (plt/unique-key)
                     f     (fn [k]
                             (let [v' (p/get-condition-state result)]
                               (when (and (not (identical? v' the-empty))
                                          (not (identical? v' locked)))
                                 (when (p/change-condition-state control :waiting :synced)
                                   (p/remove-condition-listener result k)
                                   (resume event v' nack-group))))
                             (when (p/check-condition-state control :synced)
                               (p/remove-condition-listener control k)
                               (p/lock lock)
                               (q/unlink write-head entry)
                               (p/unlock lock)))]
                 (p/add-condition-listener result k f)
                 (p/add-condition-listener control k f)
                 (f k)
                 (loop []
                   (if-let [^objects left (q/peek (aget channel 3))]
                     (if-let [^objects right (q/peek (aget channel 5))]
                       (let [[left-resume left-control left-ng left-value left-result left-event]       left
                             [right-resume right-control right-ng right-value right-result right-event] right]
                         (if-not (p/check-condition-state left-control :synced)
                           (if-not (p/check-condition-state right-control :synced)
                             (if (p/change-condition-state left-control :waiting :claimed)
                               (if (p/change-condition-state left-result the-empty locked)
                                 (if (p/change-condition-state right-control :waiting :claimed)
                                   (if (p/change-condition-state right-result the-empty locked)
                                     (do
                                       (p/change-condition-state right-control :claimed :synced)
                                       (p/change-condition-state left-control :claimed :synced)
                                       (p/unlock lock)
                                       (left-resume left-event right-value left-ng)
                                       (right-resume right-event left-value right-ng)
                                       (p/change-condition-state left-result locked right-value)
                                       (p/change-condition-state right-result locked left-value)
                                       (p/lock lock)
                                       (recur))
                                     (do
                                       (p/change-condition-state right-control :claimed :waiting)
                                       (p/change-condition-state left-result locked the-empty)
                                       (p/change-condition-state left-control :claimed :waiting)
                                       (p/unlock lock)))
                                   (do
                                     (p/change-condition-state left-result locked the-empty)
                                     (p/change-condition-state left-control :claimed :waiting)
                                     (recur)))
                                 (do
                                   (p/change-condition-state left-control :claimed :waiting)
                                   (p/unlock lock)))
                               (recur))
                             (do
                               (q/unlink (aget channel 5) right)
                               (recur)))
                           (do
                             (q/unlink (aget channel 3) left)
                             (recur))))
                       (if (p/check-condition-state is-closed :synced)
                         (let [[left-resume left-control left-ng _left-value left-result left-event] left]
                           (if (p/change-condition-state left-control :waiting :synced)
                             (do 
                               (p/unlock lock)
                               (left-resume left-event nil left-ng)
                               (p/change-condition-state left-result locked nil)
                               (p/lock lock)
                               (recur))
                             (recur)))
                         (p/unlock lock)))
                     (if-let [^objects right (q/peek (aget channel 5))]
                       (if (p/check-condition-state is-closed :synced)
                         (let [[right-resume right-control right-ng _right-value right-result right-event] right]
                           (if (p/change-condition-state right-control :waiting :synced)
                             (do 
                               (p/unlock lock)
                               (right-resume right-event nil right-ng)
                               (p/change-condition-state right-result locked nil)
                               (p/lock lock)
                               (recur))
                             (recur)))
                         (p/unlock lock))
                       (p/unlock lock))))))))))
     identity
     identity
     nil)))


#_(defn ^:no-doc duplex-channel-event
  [^DuplexChannel channel' writer-index' reader-index' value]
  (let [^objects channel (.-fields channel')
        write-head (aget channel writer-index')
        ;; TODO only need index of write head
        read-head (aget channel reader-index')
        lock             (aget channel 0)
        result2          (plt/condition-variable :waiting nil)
        is-closed        (aget channel 1)]
    ;; TODO this comment is no longer accurate
    
    ;; a channel ends up being basically a pair of doubly linked
    ;; lists, and a hash map. The linked lists are where waiters are
    ;; parked (waiting writers in one, waiting readers inthe
    ;; other). The hash map is a map of control words to linked list
    ;; entries. The mapping is used to effeciently find and remove
    ;; entries from the lists.
    (middleware
     (fn [_try-event]
       (fn [event resume _resume-with-error control nack-group]
         (if (and (p/check-condition-state result2 :synced)
                  (p/change-condition-state control :waiting :synced))
           (resume event (p/get-condition-value result2) nack-group)
           (do
             (p/lock lock)
             ;; TODO shrink entries?
             (let [q (java.util.HashSet.)
                   entry (doto (-new write-head)
                           (aset 0 resume)
                           (aset 1 control)
                           (aset 2 nack-group)
                           (aset 3 value)
                           (aset 4 result2))]
               ;; splitting locking and unlocking across function calls: terrifying
               (letfn [(remove-synched-control [k]
                         (when (p/check-condition-state control :synced)
                           (p/lock lock)
                           (-unlink write-head entry)
                           (p/remove-condition-listener control k)
                           (p/remove-condition-listener result2 k)
                           (p/remove-condition-listener is-closed k)
                           (p/unlock lock)))
                       (event-has-already-occurred [q]
                         (when (and (p/check-condition-state result2 :synced)
                                    (p/change-condition-state control :waiting :synced q))
                           (run-q q)
                           (resume event (p/get-condition-value result2) nack-group)))
                       (do-exchange [left-control
                                     right-control
                                     left-r2
                                     right-r2
                                     left-resume
                                     right-resume
                                     left-value
                                     right-value
                                     left-nack-group
                                     right-nack-group]
                         (p/unlock lock)
                         (p/change-condition-state left-control :claimed :synced q)
                         (p/change-condition-state right-control :claimed :synced q)
                         (p/set-condition-value left-r2 right-value)
                         (p/set-condition-value right-r2 left-value)
                         (p/change-condition-state left-r2 :claimed :synced q)
                         (p/change-condition-state right-r2 :claimed :synced q)
                         (run-q q)
                         (left-resume event right-value left-nack-group)
                         (right-resume event left-value right-nack-group))
                       (handle-closed [^objects left]
                         (let [left-resume     (aget left 0)
                               left-control    (aget left 1)
                               left-nack-group (aget left 2)
                               left-r2         (aget left 4)]
                           (if (p/change-condition-state left-control :waiting :claimed q)
                             (if (p/change-condition-state left-r2 :waiting :claimed q)
                               (do
                                 (p/unlock lock)
                                 (p/change-condition-state left-control :claimed :synced q)
                                 (p/set-condition-value left-r2 nil)
                                 (p/change-condition-state left-r2 :claimed :synced q)
                                 (run-q q)
                                 (left-resume event nil left-nack-group))
                               (do
                                 (p/change-condition-state left-control :claimed :waiting q)
                                 (p/unlock lock)))
                             (p/unlock lock))))
                       (attempt-exchange [q]
                         (loop []
                           (p/lock lock)
                           (let [^objects left  (-peek (aget channel 3))
                                 ^objects right (-peek (aget channel 5))]
                             (if left
                               (if right
                                 (let [left-resume      (aget left 0)
                                       left-control     (aget left 1)
                                       left-nack-group  (aget left 2)
                                       left-value       (aget left 3)
                                       left-r2          (aget left 4)
                                       right-resume     (aget right 0)
                                       right-control    (aget right 1)
                                       right-nack-group (aget right 2)
                                       right-value      (aget right 3)
                                       right-r2         (aget right 4)]
                                   (if (identical? left-control right-control)
                                     (do
                                       (p/unlock lock)) ; TODO error?
                                     ;; in order to complete an
                                     ;; exchange we need to grab
                                     ;; several locks. the
                                     ;; channel lock is already
                                     ;; held here. next lock the
                                     ;; left-control,
                                     ;; right-control, for the
                                     ;; given operations, then
                                     ;; the left and right
                                     ;; results for those
                                     ;; operations. if taking
                                     ;; all those locks
                                     ;; succeeds, then left and
                                     ;; right can
                                     ;; exchange. otherwise we
                                     ;; unlock any locks we hold
                                     ;; and let someone else
                                     ;; attempt it.
                                     (do 
                                       (if (p/change-condition-state left-control :waiting :claimed q)
                                         (if (p/change-condition-state right-control :waiting :claimed q)
                                           (if (p/change-condition-state left-r2 :waiting :claimed q)
                                             (if (p/change-condition-state right-r2 :waiting :claimed q)
                                               (do-exchange left-control
                                                            right-control
                                                            left-r2
                                                            right-r2
                                                            left-resume
                                                            right-resume
                                                            left-value
                                                            right-value
                                                            left-nack-group
                                                            right-nack-group)
                                               (do
                                                 (p/change-condition-state right-r2 :claimed :waiting q)
                                                 (p/change-condition-state right-control :claimed :waiting q)
                                                 (p/change-condition-state left-control :claimed :waiting q)
                                                 (p/unlock lock)))
                                             (do
                                               (p/change-condition-state right-control :claimed :waiting q)
                                               (p/change-condition-state left-control :claimed :waiting q)
                                               (p/unlock lock)))
                                           (do
                                             (p/change-condition-state left-control :claimed :waiting q)
                                             (p/unlock lock)))
                                         (do
                                           (p/unlock lock))))))
                                 (if (p/get-condition-value is-closed)
                                   (handle-closed left)
                                   (p/unlock lock)))
                               (if right
                                 (if (p/get-condition-value is-closed)
                                   (handle-closed right)
                                   (p/unlock lock))
                                 (if (p/get-condition-value is-closed)
                                   (if (p/change-condition-state control :waiting :claimed q) 
                                     (if (p/change-condition-state result2 :waiting :claimed q)
                                       (do
                                         (p/unlock lock)
                                         (p/change-condition-state control :claimed :synced q)
                                         (p/set-condition-value result2 nil)
                                         (p/change-condition-state result2 :claimed :synced q)
                                         (resume event nil nack-group))
                                       (do
                                         (p/change-condition-state control :claimed :waiting q)
                                         (p/unlock lock)))
                                     (p/unlock lock))
                                   (p/unlock lock)))))))
                       (control-listener [k]
                         (let [q (plt/mutable-list)]
                           (remove-synched-control k)
                           (event-has-already-occurred q)
                           (attempt-exchange q)
                           (run-q q)))]
                 (let [k (plt/unique-key)]
                   (p/add-condition-listener control k control-listener)
                   (p/add-condition-listener result2 k control-listener)
                   (p/add-condition-listener is-closed k control-listener)
                   (p/unlock lock)
                   (control-listener k))))))))
     identity
     identity
     nil)))

(defn send
  "Creates an event where synchronization completes after the given value
  has been sent via the given channel, or after the given channel
  closes. Returns true or false"
  [channel value]
  (assert (some? value))
  ;; (exchange channel 3 5 value)
  (p/send-it channel value))

(defn receive
  "Creates an event where synchronization completes after receivng a value
  via the given channel or the channel is closed. Returns the received
  value or nil."
  [channel]
  #_(exchange channel 5 3 true)
  (p/receive-it channel))

(defn close!
  "Close a channel."
  [channel]
  (p/close! channel))

;; TODO pubsub/mult like

;; TODO turn a full duplex channel into just a Send or Receive channel

(defn sync!
  "Run a callback after the given event occurs. Returns a
CompletableFuture that will be completed after the event occurs."
  ([evt]
   (let [cf (plt/promise-like-type)]
     (p/-sync
      (-> evt
          (wrap (fn [value] (plt/complete-plt cf value)))
          (wrap-handler (fn [exception] (plt/complete-plt-error cf exception)))))
     cf))
  ([evt callback]
   (sync! (wrap evt callback))))

#?(:clj
   (defn sync!!
     "Stop and wait for the given event to occur"
     [evt]
     @(sync! evt)))

;; systemd-run --user --slice surmise-tests --pty --same-dir --wait --collect -p MemoryMax=1G -p MemorySwapMax=1G -p CPUQuota=80% --service-type=exec clojure -X:codox

;; systemd-run --user --slice surmise-tests --pty --same-dir --wait --collect -p MemoryMax=1G -p MemorySwapMax=1G -p CPUQuota=80% -p LimitRTTIME=1000 --service-type=exec clojure -T:build test


;; TODO figure this out, docs, tests

;; TODO move to protocols
(defprotocol PubSub
  (-subscribe [_ topic channel close?])
  (-unsubscribe [_ topic channel]))

(defn pubsub [input topic-fn]
  (letfn [(mass-send [value outputs]
            (barrier
             (fn [evt b!]
               (let [c (count outputs)
                     results (atom outputs)
                     e (plt/condition-variable 0)]
                 (p/add-condition-listener e
                                           (plt/unique-key)
                                           (fn [_]
                                             (when (= c (p/get-condition-state e))
                                               (b!))))
                 (doseq [[channel _] outputs]
                   (sync! (wrap (send channel value)
                                (fn [send-result]
                                  (when-not send-result
                                    (swap! results dissoc channel))
                                  (loop []
                                    (let [n (long (p/get-condition-state e))]
                                      (when-not (p/change-condition-state e n (inc n))
                                        (recur))))))))
                 (wrap evt (fn [_] @results))))))
          (main-loop [commands outputs]
            (sync!
             (choice
              [(wrap (receive commands)
                     (fn [command]
                       (case (nth command 0)
                         :subscribe (main-loop commands
                                               (assoc-in outputs [(nth command 1)
                                                                  (nth command 2)]
                                                         (nth command 3)))
                         :unsubscribe (let [o (dissoc (get outputs (nth command 1)) (nth command 2) )]
                                        (if (seq o)
                                          (main-loop commands (assoc outputs (nth command 1) o))
                                          (main-loop commands (dissoc outputs (nth commands 1))))))))
               (wrap (receive input)
                     (fn [new-msg]
                       (if new-msg
                         (let [t (topic-fn new-msg)
                               topic-outputs (get outputs t)]
                           (sync! (wrap (mass-send new-msg topic-outputs)
                                        (fn [topic-outputs]
                                          (if (seq topic-outputs)
                                            (main-loop commands
                                                       (assoc outputs t topic-outputs))
                                            (main-loop commands
                                                       (dissoc outputs t)))))))
                         (doseq [[_topic topic-outputs] outputs
                                 output topic-outputs]
                           (when (:close? output)
                             (p/close! (:ch output)))))))])))]
    (let [commands (buffered-channel 1024)]
      (main-loop commands {})
      (reify
        PubSub
        (-subscribe [_ topic channel close?]
          (wrap (send commands [:subscribe topic channel close?]) (cc/constantly channel)))
        (-unsubscribe [_ topic channel]
          (wrap (send commands [:unsubscribe topic channel])
                (cc/constantly channel)))
        p/SendChannel
        (send-it [_ it]
          (p/send-it input it))))))
