(ns com.manigfeald.machinate.platform
  (:require [com.manigfeald.machinate.protocols :as p])
  (:import (java.util LinkedList
                      List
                      HashMap)
           (java.util.concurrent CompletableFuture DelayQueue TimeUnit Delayed)
           (java.util.function BiFunction)
           (java.util.concurrent.locks ReentrantLock)))

(set! *warn-on-reflection* true)

(set! *unchecked-math* :warn-on-boxed)

(defn mutable-list ^List []
  (LinkedList.))

(defn peek-first [^LinkedList lst]
  (.peekFirst lst))

(defn pop-list [^LinkedList lst]
  (.pop lst))

(defn add-last [^LinkedList lst item]
  (.addLast lst item))


(defn print-to-error [message]
  (binding [*out* *err*]
    (print message)
    (flush)))

(defn unique-key []
  (Object.))


;; TODO check resource usage, is cancel on cleanup actually cleaning stuff up

;; TODO CompletionStage doesn't come with a cancel method, maybe a new
;; protocol so consumer can add one?
(extend-type CompletableFuture
  p/Event
  (try-event [^CompletableFuture event resume resume-with-error control nack-group cleanup]
    (let [^CompletableFuture x
          (.handle event
                   (reify
                     BiFunction
                     (apply [_ result exception]
                       (let [k (Object.)
                             f (fn [& _]
                                 (when (p/check-state control :synced)
                                   (p/remove-state-change-listener control k))
                                 (when (p/change-state control :waiting :synced)
                                   (if exception
                                     (resume-with-error event exception nack-group)
                                     (resume event result nack-group))))]
                         (p/listen-to-state-change control k f)
                         (f)))))]
      ;; TODO cleanup here ends up causing a circular thing that can
      ;; blow the stack
      (p/try-event cleanup
                   (fn [_event _value _nack-group]
                     (.cancel x false))
                   (fn [_event _value _nack-group]
                     (.cancel x false))
                   (atom :waiting)
                   #{}
                   nil))))



(defn barrier
  "Given a function f, applies f to a function g and an event e. e becomes
  enabled when g is invoked. returns the result of invoking f."
  [f]
  (let [cf (CompletableFuture.)]
    (f cf (fn [] (.complete cf true)))))


(def ^DelayQueue ^:no-doc delay-queue (DelayQueue.))

(deftype D [action ^long delay ^long at]
  Delayed
  (getDelay [_ timeunit]
    (.convert timeunit
              (- delay (- (System/currentTimeMillis) at))
              TimeUnit/MILLISECONDS))
  Comparable
  (compareTo [_ other]
    (let [now (System/currentTimeMillis)]
      (compare (- delay (- now at))
               (- (.-delay ^D other) (- now (.-at ^D other)))))))

(def ^:no-doc process-delay-queue
  (delay
    (doto (Thread.
           (fn []
             (while true
               (try
                 ((.-action ^D (.take delay-queue)))
                 (catch Throwable _
                   ;; TODO report in some way?
                   )))))
      (.setDaemon true)
      (.setName "machinate-process-delay-queue")
      (.start))))

(defn now []
  (System/currentTimeMillis))

(defn run-after [start delay f]
  @process-delay-queue
  (let [d (->D f delay start)]
    (.add delay-queue d)
    d))

(defn cancel-run-after [handle]
  (.remove delay-queue handle))

(defn lock []
  (ReentrantLock.))

(extend-type ReentrantLock
  p/ILock
  (lock [this]
    (.lock this))
  (unlock [this]
    (.unlock this)))

(defn promise-like-type []
  (CompletableFuture.))

(defn complete-plt [^CompletableFuture plt value]
  (.complete plt value))

(defn complete-plt-error [^CompletableFuture plt error]
  (.completeExceptionally plt error))

(defn mutable-hash-map ^HashMap []
  (HashMap.))

(defn hash-map-values [^HashMap m]
  (vals m))

(defn hash-map-get [^HashMap m k]
  (.get m k))

(defn hash-map-put [^HashMap m k v]
  (.put m k v))

(defn hash-map-remove [^HashMap m k]
  (.remove m k))
