(ns com.manigfeald.machinate
  (:refer-clojure :exclude [send
                            constantly])
  (:require [com.manigfeald.machinate.protocols :as p]
            [com.manigfeald.machinate.platform :as plt])
  #?(:clj (:import (clojure.lang Atom))))

#?(:clj (set! *warn-on-reflection* true))

#?(:clj (set! *unchecked-math* :warn-on-boxed))

;; TODO manifold interop

;; TODO io, netty? nio?

;; At the lowest level, this is a library for synchronizing on
;; events. synchronizing on an event means registering a callback to
;; be invoked when the event occurs.

;; acts as never
(extend-type nil
  p/Event
  (try-event [_event _resume _resume-with-error _control _nack-group _cleanup])
  p/Nackable
  (check-nack-group [_event _nack-group])
  p/QuasiEvent
  (push-down [event ctor lst]
    (when-let [evt (ctor event)]
      (plt/add-last lst evt)))
  p/Syncable
  (-sync [_event]))

(extend-type Atom
  p/Control
  (change-state [this current new]
    (compare-and-set! this current new))
  (check-state [this check]
    (= @this check))
  (listen-to-state-change [this key callback]
    (add-watch this key (fn [& _] (callback key))))
  (remove-state-change-listener [this key]
    (remove-watch this key)))

(declare barrier)

(extend-type #?(:clj Object :cljs default)
  p/QuasiEvent
  (push-down [event ctor lst]
    (plt/add-last lst (ctor event)))
  p/Syncable
  (-sync [event]
    ;; wrap list access in a volatile to ensure ordering, may not be
    ;; required.
    (let [lst (volatile! (plt/mutable-list))
          control (atom :waiting)]
      (p/push-down event identity @lst)
      (vreset! lst @lst)
      (barrier
       (fn [cleanup break]
         (letfn [(resume-f [_event _value nack-group]
                   ;; resume-f could end up invoked on another thread,
                   ;; hence the volatile
                   (doseq [e @lst]
                     (p/check-nack-group e nack-group))
                   (break))
                 (unhandled-error [event value nack-group]
                   (plt/print-to-error
                    (str
                     (print-str "unhandled error returned\n")
                     (prn-str value)))
                   (resume-f event value nack-group))]
           (doseq [evt @lst
                   :when evt]
             (p/try-event evt resume-f unhandled-error control #{} cleanup)))))))
  p/Nackable
  (check-nack-group [_event _nack-group]))

(defn ^:no-doc constantly* [g value]
  (reify
    p/Event
    (try-event [event resume resume-with-error control nack-group _cleanup]
      (let [k (plt/unique-key)
            f (fn [& _]
                (when (p/check-state control :synced)
                  (p/remove-state-change-listener control k))
                (when (p/change-state control :waiting :synced)
                  ((g resume resume-with-error) event value nack-group)))]
        (p/listen-to-state-change control k f)
        (f)))))

(defn constantly
  "Returns an event that is immediatly synchronizable with the value value"
  [value]
  (constantly* (fn [resume _resume-with-error] resume) value))

(defn constantly-error
  "Returns an event that is immediatly synchronizable with the error value"
  [value]
  (constantly* (fn [_resume resume-with-error] resume-with-error) value))

(def always
  "Event that has always already occured. Synchronization always succeeds
  immediately. Value of the event is true."
  (constantly true))

(defn watch-reference
  "Return an event that syncs when the given watchable clojure reference
  type (refs,atoms,agents) has the given value. Event has the value
  true."
  [r value]
  (barrier
   (fn [e b!]
     (let [k (plt/unique-key)
           f (fn [k r _os ns]
               (when (= ns value)
                 (remove-watch r k)
                 (b!)))]
       (add-watch r k f)
       (f k r nil @r))
     e)))

(declare middleware)

(deftype Middleware [try-event-transform
                     check-nack-group-transform
                     push-down-transform
                     target]
  p/Event
  (try-event [_ resume resume-with-error control nack-group cleanup]
    ((try-event-transform p/try-event) target resume resume-with-error control nack-group cleanup))
  p/Nackable
  (check-nack-group [_ nack-group]
    ((check-nack-group-transform p/check-nack-group) target nack-group))
  p/QuasiEvent
  (push-down [_event ctor lst]
    ((push-down-transform p/push-down)
     target
     (comp ctor #(middleware try-event-transform
                             check-nack-group-transform
                             push-down-transform
                             %))
     lst)))

(defn middleware
  "Middleware smart constructor. Middleware wraps an event, passing
  protocols calls down with the opportunity to intercept and make
  changes. The middleware smart constructor can collapse multiple
  middlewares into a single middleware via function composition."
  [try-event-transform check-nack-group-transform push-down-transform target]
  (if (instance? Middleware target)
    (let [^Middleware target target]
      (middleware (comp try-event-transform (.-try-event-transform target))
                  (comp check-nack-group-transform (.-check-nack-group-transform target))
                  (comp push-down-transform (.-push-down-transform target))
                  (.-target target)))
    (->Middleware try-event-transform check-nack-group-transform push-down-transform target)))


(defn barrier
  "Given a function f, applies f to a function g and an event e. e becomes
  enabled when g is invoked. returns the result of invoking f."
  [f]
  (plt/barrier f))

#_(defn barrier
    "Given a function f, applies f to a function g and an event e. e becomes
  enabled when g is invoked. returns the result of invoking f."
  [f]
  (let [root (AtomicReference.)
        done (Object.)
        event (middleware
               (fn [event-transform]
                 (fn [event resume control nack-group cleanup]
                   (listen-to-state-change control (Object.)
                                           (fn [k]
                                             (when (check-state control :synced)
                                               (loop [^AtomicReferenceArray r (.get root)]
                                                 (when r
                                                   (when-not (identical? r done)
                                                     (if (identical? control (.get r 1))
                                                       (doto r
                                                         (.set 0 nil)
                                                         (.set 1 nil)
                                                         (.set 2 nil))
                                                       (recur (.get r 3))))))
                                               (remove-watch control k))))
                   (let [cell (doto (AtomicReferenceArray. 4)
                                (.set 0 resume)
                                (.set 1 control)
                                (.set 2 nack-group))]
                     (loop [r (.get root)]
                       (if (identical? r done)
                         (loop []
                           (if (change-state control :waiting :synced)
                             (resume event nil nack-group)
                             (when-not (check-state control :synced)
                               (recur))))
                         (do
                           (.set cell 3 r)
                           (when-not (.compareAndSet root r cell)
                             (recur (.get root)))))))))
               (fn [check-nack-group]
                 (fn [event nack-group]))
               identity
               nil)]
    (f event
       (fn []
         (loop [^AtomicReferenceArray r (.get root)]
           (when-not (identical? r done)
             (if-not (.compareAndSet root r done)
               (recur (.get root))
               (loop [r r]
                 (when r
                   (let [resume (.get r 0)
                         control (.get r 1)
                         nack-group (.get r 2)]
                     (when (and resume control nack-group)
                       (loop []
                         (if (change-state control :waiting :synced)
                           (resume event nil nack-group)
                           (when-not (check-state control :synced)
                             (recur)))))
                     (recur (.get r 3))))))))))))


;; events end up forming a tree, before synchronization that tree gets
;; flattened into something like disjuntive normal form, then at
;; synchronization one of the possible events is chosen.


(defn ^:no-doc choice*
  [evts ordered?]
  (middleware
   identity
   identity
   (fn [push-down]
     (fn [_event ctor lsg]
       (doseq [e (if-not ordered?
                   (shuffle evts)
                   evts)]
         (push-down e ctor lsg))))
   nil))

(defn ordered-choice
  "Takes a collection of events, and attempts to select
event to synchronize on in order."
  [evts]
  (choice* evts true))


(defn choice
  "Takes a collection of events, and non-deterministically selects one on
synchronization by defaults."
  ([evts]
   (choice* evts false)))

(defn nack
  "Like guard, but f is passed an event G. G will occur if on
  synchronization, the chosen event is not the one returned by f."
  [f]
  (middleware
   identity
   identity
   (fn [push-down]
     (fn [_event ctor lsg]
       (barrier
        (fn [b break]
          (let [e (f b)
                id (plt/unique-key)]
            (push-down
             e
             (comp ctor
                   (fn g [evt]
                     (middleware
                      (fn [try-event]
                        (fn [event resume resume-with-error control nack-group cleanup]
                          (try-event event resume resume-with-error control (conj nack-group id) cleanup)))
                      (fn [check-nack-group]
                        (fn [event nack-group]
                          (when-not (contains? nack-group id)
                            (break))
                          (check-nack-group event nack-group)))
                      identity
                      evt)))
             lsg))))))
   nil))

;; TODO should guard just go away, it kind of sucks

;; TODO should f be invoked everytime?
(defn guard
  "Takes a function of no arguments f, returns an event E, before E is
  synchronized on, f will be invoked, and E replaced with the event
  returned by f."
  [f]
  (middleware
   identity
   identity
   (fn [push-down]
     (fn [_event ctor lst]
       (push-down (f) ctor lst)))
   nil))

(defn wrap
  "Wraps the post synchronization action f around the event evt"
  [evt f]
  (middleware
   (fn [try-event]
     (fn [evt resume resume-with-error control nack-group cleanup]
       (try-event evt
                  (fn [event value nack-group]
                    (try
                      (resume event (f value) nack-group)
                      (catch #?(:clj Throwable :cljs :any) t
                        (resume-with-error event t nack-group))))
                  resume-with-error
                  control
                  nack-group
                  cleanup)))
   identity
   identity
   evt))

(defn wrap-handler
  "Wrap an error handler around an event. Handlers errors coming out of
the event itself, or any errors arising from other wrappings of the
event."
  [evt f]
  (middleware
   (fn [try-event]
     (fn [evt resume resume-with-error control nack-group cleanup]
       (try-event evt
                  resume
                  (fn [event value nack-group]
                    (try
                      (resume event (f value) nack-group)
                      (catch #?(:clj Throwable :cljs :any) t
                        (resume-with-error event t nack-group))))
                  control
                  nack-group
                  cleanup)))
   identity
   identity
   evt))

(defn alts
  "A work-a-like for clojure.core.async/alts!. Given a collection of
events returns a new event that when sync'ed chooses one event and
returns [event-value event]."
  [events]
  (choice
   (for [event events]
     (wrap event (fn [result] [result event])))))

;; (defonce scheduler
;;   (delay
;;     (Executors/newScheduledThreadPool
;;      1
;;      (reify
;;        ThreadFactory
;;        (newThread [_ r]
;;          ;; TODO name
;;          (doto (Thread. r)
;;            (.setDaemon true)))))))




;; canceling a scheduled future doesn't actually remove it from the
;; queue, annoyingly

;; (defn timeout [delay]
;;   (let [now (System/nanoTime)
;;         n (* 1000000 (long delay))
;;         deadline (+ n now)]
;;     (barrier
;;      (fn [b break]
;;        (nack
;;         (fn [neg]
;;           (let [now (System/nanoTime)
;;                 delay (- deadline now)]
;;             (if (pos? delay)
;;               (let [f (.schedule ^ScheduledExecutorService @scheduler
;;                                  ^Runnable break
;;                                  delay
;;                                  TimeUnit/NANOSECONDS)]
;;                 (p/-sync (wrap neg (fn [_]
;;                                      (.cancel f false))))
;;                 b)
;;               always))))))))


(defn timeout
  "Given a delay in milliseconds, returns an event that doesn't
  synchronize until after that delay has elapsed. Value of the event
  is true."
  [^long delay]
  (let [at (plt/now)]
    (barrier
     (fn [b break]
       (nack
        (fn [neg]
          (let [handle (plt/run-after at delay break)]
            (p/-sync (wrap neg (fn [_] (plt/cancel-run-after handle))))
            b)))))))

(declare duplex-channel-event
         channel-close!)

;; The default rendevous channel, unbuffered, no transforms, pure send
;; and receive
(deftype DuplexChannel [fields]
  p/Exchange
  (send-left [channel value]
    (duplex-channel-event channel 3 5 value))
  (send-right [channel value]
    (duplex-channel-event channel 5 3 value))
  p/SendChannel
  (send-it [this it]
    (p/send-left this it))
  p/ReceiveChannel
  (receive-it [this]
    (p/send-right this true))
  p/Channel
  (close! [this]
    (channel-close! this)))

(defn channel
  "Returns full duplex channel, both a SendChannel and a ReceiveChannel"
  []
  (->DuplexChannel
   (doto (object-array 7)
     (aset 0 (plt/lock))
     (aset 1 false)
     (aset 2 (plt/mutable-hash-map))
     (aset 3 nil) ; writers head
     (aset 4 nil) ; writers tail
     (aset 5 nil) ; readers head
     (aset 6 nil) ; readers tail
     )))

(declare sync!)

;; TODO docs

(defn fixed-size-buffer [^long n]
  (let [lst (plt/mutable-list)]
    (reify
      p/Buffer
      (full? [_]
        (<= n (count lst)))
      (add-item [_ item]
        (plt/add-last lst item))
      (peek-item [_]
        (plt/peek-first lst))
      (pop-item [_]
        (plt/pop-list lst)))))

;; TODO test closing

;; TODO exception handling for xform

;; TODO docs

;; TODO buffer seems off by one

;; buffered channel is two channels internally with a loop copying
;; between them with a buffer in the middle.
(defn buffered-channel
  ([buffer-or-n]
   (buffered-channel buffer-or-n identity))
  ([buffer-or-n xform]
   (let [b (volatile!
            (if (number? buffer-or-n)
              (fixed-size-buffer buffer-or-n)
              buffer-or-n))
         in-c (channel)
         out-c (channel)
         ;; TODO nil item check
         add! (xform (fn
                       ([b] b)
                       ([b item] (p/add-item b item))))]
     ;; TODO maybe needs some kind of trampoline?
     ((fn copy-loop [open?]
        (let [item (p/peek-item @b)]
          (sync!
           (choice
            [(if (some? item)
               (wrap (p/send-it out-c item)
                     (fn [_sent-result]
                       (p/pop-item @b)
                       (copy-loop open?)))
               (when-not open?
                 (p/close! out-c)
                 nil))
             (if open?
               (if (p/full? @b)
                 nil
                 (wrap (p/receive-it in-c)
                       (fn [new-item]
                         (if (some? new-item)
                           (do
                             ;; TODO figure out handling of reduced
                             (add! @b new-item)
                             (copy-loop open?))
                           (copy-loop false)))))
               nil)]))))
      true)
     (reify
       p/SendChannel
       (send-it [_ it]
         (p/send-it in-c it))
       p/ReceiveChannel
       (receive-it [_]
         (p/receive-it out-c))
       p/Channel
       (close! [_]
         (p/close! in-c))))))

;; TODO make it a kind of event?
(defn ^:no-doc channel-close! [^DuplexChannel channel]
  (let [^objects channel (.-fields channel)
        lock (aget channel 0)
        _ (p/lock lock)
        controls (aget channel 2)]
    (if-not (aget channel 1)
      (do
        (aset channel 1 true)
        (let [items (into [] (plt/hash-map-values controls))]
          (p/unlock lock)
          (doseq [^objects item items
                  :let [r (aget item 0)
                        c (aget item 1)
                        n (aget item 2)
                        _v (aget item 3)]
                  :when (p/change-state c :waiting :claimed)]
            (p/change-state c :claimed :synced)
            (r nil nil n))))
      (p/unlock lock))))

;; TODO exchange with a buffer to fill up?

(def ^:no-doc incomplete (plt/unique-key))

(defn ^:no-doc duplex-channel-event
  [^DuplexChannel channel writer-index reader-index value]
  (let [^objects channel (.-fields channel)
        writer-index (long writer-index)
        reader-index (long reader-index)
        lock (aget channel 0)
        controls (aget channel 2)
        result2 (atom incomplete)]
    ;; a channel ends up being basically a pair of doubly linked
    ;; lists, and a hash map. The linked lists are where waiters are
    ;; parked (waiting writers in one, waiting readers inthe
    ;; other). The hash map is a map of control words to linked list
    ;; entries. The mapping is used to effeciently find and remove
    ;; entries from the lists.
    (middleware
     (fn [_try-event]
       (fn [event resume _resume-with-error control nack-group _cleanup]
         ;; TODO rename clean-up
         ;; TODO can the cleanup event get rid of the need for the hash map?
         (letfn [(clean-up [k]
                   (when (p/check-state control :waiting)
                     (p/lock lock)
                     (let [^objects item (aget channel reader-index)]
                       (if item
                         (let [r (aget item 0)
                               c (aget item 1)
                               n (aget item 2)
                               v (aget item 3)]
                           (if (identical? c control)
                             (p/unlock lock)
                             (if (p/change-state c :waiting :claimed)
                               (if (p/change-state control :waiting :claimed)
                                 (do
                                   (reset! result2 v)
                                   (p/unlock lock)
                                   (p/change-state c :claimed :synced)
                                   (p/change-state control :claimed :synced)
                                   (r event value n)
                                   (resume event v nack-group))
                                 (do
                                   (p/change-state c :claimed :waiting)
                                   (p/unlock lock)))
                               (p/unlock lock))))
                         (if (aget channel 1)
                           (do
                             (p/unlock lock)
                             (when (p/change-state control :waiting :synced)
                               (reset! result2 nil)
                               (resume event nil nack-group)))
                           (p/unlock lock)))))
                   (when (p/check-state control :synced)
                     (p/lock lock)
                     (loop [^objects item (plt/hash-map-get controls control)]
                       (when item
                         (when (aget item 4)
                           (aset ^objects (aget item 4) 5 (aget item 5)))
                         (when (aget item 5)
                           (aset ^objects (aget item 5) 4 (aget item 4)))
                         (when (aget item 6)
                           (aset ^objects (aget item 6) 7 (aget item 7)))
                         (when (aget item 7)
                           (aset ^objects (aget item 7) 6 (aget item 6)))
                         (when (identical? (plt/hash-map-get controls control) item)
                           (let [n (aget item 6)]
                             (if n
                               (plt/hash-map-put controls control n)
                               (plt/hash-map-remove controls control))))
                         (doseq [writer-index [writer-index reader-index]]
                           (when (identical? (aget channel writer-index) item)
                             (aset channel writer-index (aget item 4)))
                           (when (identical? (aget channel (inc (long writer-index))) item)
                             (aset channel writer-index (aget item 5)))
                           (when (nil? (aget channel writer-index))
                             (aset channel (inc (long writer-index)) nil)))
                         (recur (aget item 6))))
                     (p/unlock lock)
                     (p/remove-state-change-listener control k)))]
           (if (not= incomplete @result2)
             (resume event @result2 nack-group)
             (do
               (p/lock lock)
               (if (not= incomplete @result2)
                 (do
                   (p/unlock lock)
                   (resume event @result2 nack-group))
                 (let [k (plt/unique-key)
                       entry (doto (object-array 8)
                               (aset 0 resume)
                               (aset 1 control)
                               (aset 2 nack-group)
                               (aset 3 value)
                               (aset 4 nil) ; next
                               (aset 5 (aget channel (inc writer-index))) ; prev
                               (aset 6 (plt/hash-map-get controls control)) ; lst next
                               (aset 7 nil) ; lst prev
                               )]
                   (when (nil? (aget channel writer-index))
                     (aset channel writer-index entry))
                   (when (aget channel (inc writer-index))
                     (aset ^objects (aget channel (inc writer-index)) 4 entry))
                   (aset channel (inc writer-index) entry)
                   (when (some? (plt/hash-map-get controls control))
                     (aset ^objects (plt/hash-map-get controls control) 7 entry))
                   (plt/hash-map-put controls control entry)
                   (p/listen-to-state-change control k clean-up)
                   (p/unlock lock)
                   (clean-up k))))))))
     identity
     identity
     nil)))

(defn send
  "Creates an event where synchronization completes after the given value
  has been sent via the given channel, or after the given channel
  closes. Returns true or false"
  [channel value]
  (assert (some? value))
  ;; (exchange channel 3 5 value)
  (p/send-it channel value))

(defn receive
  "Creates an event where synchronization completes after receivng a value
  via the given channel or the channel is closed. Returns the received
  value or nil."
  [channel]
  #_(exchange channel 5 3 true)
  (p/receive-it channel))

(defn close!
  "Close a channel."
  [channel]
  (p/close! channel))

;; TODO turn a full duplex channel into just a Send or Receive channel

(defn sync!
  "Run a callback after the given event occurs. Returns a
CompletableFuture that will be completed after the event occurs."
  ([evt]
   (let [cf (plt/promise-like-type)]
     (p/-sync
      (-> evt
          (wrap (fn [value] (plt/complete-plt cf value)))
          (wrap-handler (fn [exception] (plt/complete-plt-error cf exception)))))
     cf))
  ([evt callback]
   (sync! (wrap evt callback))))

#?(:clj
   (defn sync!!
     "Stop and wait for the given event to occur"
     [evt]
     @(sync! evt)))

;; systemd-run --user --slice surmise-tests --pty --same-dir --wait --collect -p MemoryMax=1G -p MemorySwapMax=1G -p CPUQuota=80% --service-type=exec clojure -X:codox

;; systemd-run --user --slice surmise-tests --pty --same-dir --wait --collect -p MemoryMax=1G -p MemorySwapMax=1G -p CPUQuota=80% -p LimitRTTIME=1000 --service-type=exec clojure -T:build test
