(ns com.manigfeald.machinate.platform
  (:require [com.manigfeald.machinate.protocols :as p])
  (:import (java.util LinkedList
                      List
                      ArrayList)
           (java.util.concurrent CompletableFuture DelayQueue TimeUnit Delayed
                                 ConcurrentHashMap
                                 LinkedBlockingQueue)
           (java.util.function BiFunction)
           (java.util.concurrent.locks ReentrantLock
                                       Condition
                                       LockSupport
                                       Lock)
           (java.util.concurrent.atomic AtomicReference)))

(set! *warn-on-reflection* true)

(set! *unchecked-math* :warn-on-boxed)

(defn print-to-error [message]
  (binding [*out* *err*]
    (print message)
    (flush)))

(defn unique-key []
  (Object.))

(definterface CV
  (getConditionState [])
  (checkConditionState [value])
  (changeConditionState [old-value new-value])
  (addConditionListener [id listener])
  (removeConditionListener [id])
  (doSignal []))

(defmacro get-condition-state [cv]
  `(.getConditionState ~(with-meta cv {:tag `CV})))

(defmacro check-condition-state [cv value]
  `(.checkConditionState ~(with-meta cv {:tag `CV}) ~value))

(defmacro change-condition-state [cv old-value new-value]
  `(.changeConditionState ~(with-meta cv {:tag `CV}) ~old-value ~new-value))

(defmacro add-condition-listener [cv id listener]
  `(.addConditionListener ~(with-meta cv {:tag `CV}) ~id ~listener))

(defmacro remove-condition-listener [cv id]
  `(.removeConditionListener ~(with-meta cv {:tag `CV}) ~id))

(defmacro signal [cv]
  `(.doSignal ~(with-meta cv {:tag `CV})))

;; TODO check resource usage, is cancel on cleanup actually cleaning stuff up

;; TODO CompletionStage doesn't come with a cancel method, maybe a new
;; protocol so consumer can add one?
(extend-type CompletableFuture
  p/Event
  (try-event [^CompletableFuture event resume resume-with-error control nack-group]
    (let [^CompletableFuture x
          (.handle event
                   (reify
                     BiFunction
                     (apply [_ result exception]
                       (let [f (fn [k]
                                 (when (change-condition-state control :waiting :synced)
                                   (remove-condition-listener control k)
                                   (signal control)
                                   (if exception
                                     (resume-with-error event exception nack-group)
                                     (resume event result nack-group)))
                                 (remove-condition-listener control k))]
                         (add-condition-listener control (Object.) f)))))]
      (add-condition-listener control
                              (Object.)
                              (fn [k]
                                (when (check-condition-state control :synced)
                                  (.cancel x false)
                                  (remove-condition-listener control k)))))))



(defn barrier
  "Given a function f, applies f to a function g and an event e. e becomes
  enabled when g is invoked. returns the result of invoking f."
  [f]
  (let [cf (CompletableFuture.)]
    (f cf (fn [] (.complete cf true)))))


;; TODO replace delay queue processing with something hash wheel timer based?

(def ^DelayQueue ^:no-doc delay-queue (DelayQueue.))

(deftype D [action ^long delay ^long at]
  Delayed
  (getDelay [_ timeunit]
    (.convert timeunit
              (- delay (- (System/currentTimeMillis) at))
              TimeUnit/MILLISECONDS))
  Comparable
  (compareTo [_ other]
    (let [now (System/currentTimeMillis)]
      (compare (- delay (- now at))
               (- (.-delay ^D other) (- now (.-at ^D other)))))))

(def ^:no-doc process-delay-queue
  (delay
    (doto (Thread.
           (fn []
             (while true
               (try
                 ((.-action ^D (.take delay-queue)))
                 (catch Throwable _
                   ;; TODO report in some way?
                   )))))
      (.setDaemon true)
      (.setName "machinate-process-delay-queue")
      (.start))))

(defn now []
  (System/currentTimeMillis))

(defn run-after [start delay f]
  @process-delay-queue
  (let [d (->D f delay start)]
    (.add delay-queue d)
    d))

;; TODO what is the big-O of remove on a delay queue?
(defn cancel-run-after [handle]
  (.remove delay-queue handle))


;; (definterface Q
;;   (pushlock [data])
;;   (pushunlock [data]))

;; (deftype TrackingLock [^ReentrantLock l
;;                        ^:volatile-mutable locks
;;                        ^:volatile-mutable unlocks]
;;   Lock
;;   (lock [_]
;;     (.lock l))
;;   (unlock [_]
;;     (.unlock l))
;;   Q
;;   (pushlock [_ data]
;;     (set! locks (cons [data (.getName (Thread/currentThread)) (.getId (Thread/currentThread))] locks)))
;;   (pushunlock [_ data]
;;     (set! unlocks (cons [data (first locks) (.getName (Thread/currentThread)) (.getId (Thread/currentThread))] unlocks)))
;;   Object
;;   (toString [_] (pr-str [l locks unlocks])))

(defn lock ^Lock []
  (ReentrantLock.))

(defmacro do-lock [l]
  `(.lock ~(with-meta l {:tag `Lock})))

(defmacro unlock [l]
  `(.unlock ~(with-meta l {:tag `Lock})))

(defn promise-like-type []
  (CompletableFuture.))

(defn complete-plt [^CompletableFuture plt value]
  (.complete plt value))

(defn complete-plt-error [^CompletableFuture plt error]
  (.completeExceptionally plt error))

(defn table-grow [^objects waiters id listener table-insert]
  (let [new-table (object-array (* 2 (alength waiters)))]
    (loop [i 0]
      (when (> (alength waiters) i)
        (let [nk (nth waiters i)
              nv (nth waiters (inc i))]
          (table-insert new-table nk nv))
        (recur (+ 2 i))))
    (table-insert new-table id listener)))

(defn table-insert [^objects table ik iv]
  (let [start (long (* 2 (long (mod (hash ik) (long (/ (alength table) 2))))))]
    (loop [i start]
      (let [k (aget table i)]
        (if (= k ik)
          (do
            (aset table (inc i) iv)
            table)
          (if (nil? k)
            (do
              (aset table i ik)
              (aset table (inc i) iv)
              table)
            (let [nxt (long (mod (+ 2 i) (alength table)))]
              (if (== start nxt)
                (table-grow table ik iv table-insert)
                (recur nxt)))))))))

(defn table-remove [^objects table ik]
  (let [start (long (* 2 (long (mod (hash ik) (long (/ (alength table) 2))))))]
    (loop [i (long start)]
      (let [k (aget table i)]
        (if (= k ik)
          (do
            (aset table i nil)
            (aset table (inc i) nil))
          (let [nxt (long (mod (+ 2 i) (alength table)))]
            (when-not (== nxt start)
              (recur nxt))))))))

(defn table-lookup [^objects table ik not-found]
  (let [start (long (* 2 (long (mod (hash ik) (long (/ (alength table) 2))))))]
    (loop [i (long start)]
      (let [k (aget table i)]
        (if (= k ik)
          (aget table (inc i))
          (let [nxt (long (mod (+ 2 i) (alength table)))]
            (if-not (== nxt start)
              (recur nxt)
              not-found)))))))

(comment
  (let [N 300
        ^objects t (reduce
                    (fn [t i]
                      (table-insert t i i))
                    (object-array 2)
                    (range N))]
    (assert (> (alength t) N))
    (doseq [i (range N)]
      (assert (= i (table-lookup t i nil)) i))
    (reduce
     (fn [removed i]
       (table-remove t i)
       (assert (nil? (table-lookup t i nil)))
       (let [removed (conj removed i)]
         (doseq [i (remove removed (range 129))]
           (assert (= i (table-lookup t i nil)) i))
         (assert (not (some #{i} t)))
         removed))
     #{}
     (shuffle (range N)))))


(def X (class (object-array 0)))

(deftype AConditionVariable [^ReentrantLock lock
                             ^:unsynchronized-mutable state
                             ^:unsynchronized-mutable waiter-id
                             ^objects ^:unsynchronized-mutable waiters]
  CV
  (getConditionState [this]
    (.lock lock)
    (let [s state]
      (.unlock lock)
      s))
  (checkConditionState [this value]
    (.lock lock)
    (if (identical? state value)
      (do
        (.unlock lock)
        true)
      (do
        (.unlock lock)
        false)))
  (changeConditionState [this old-value new-value]
    (.lock lock)
    (if (identical? old-value state)
      (do
        (set! state new-value)
        (.unlock lock)
        true)
      (do
        (.unlock lock)
        false)))
  (addConditionListener [this id listener]
    (.lock lock)
    (if (nil? waiter-id)
      (do
        (set! waiter-id id)
        (set! waiters listener)
        (.unlock lock)
        (listener id))
      (if (identical? X waiter-id)
        (do
          (set! waiters (table-insert waiters id listener))
          (.unlock lock)
          (listener id))
        (let [a waiter-id
              b waiters]
          (set! waiter-id X)
          (set! waiters
                (-> (object-array 8)
                    (table-insert a b)
                    (table-insert id listener)))
          (.unlock lock)
          (listener id))))
    nil)
  (removeConditionListener [this id]
    (.lock lock)
    (if (nil? waiter-id)
      (.unlock lock)
      (if (identical? X waiter-id)
        (do
          (table-remove waiters id)
          (.unlock lock))
        (if (identical? waiter-id id)
          (do
            (set! waiter-id nil)
            (set! waiters nil)
            (.unlock lock))
          (.unlock lock))))
    nil)
  (doSignal [_]
    (.lock lock)
    (if (nil? waiter-id)
      (.unlock lock)
      (if (identical? X waiter-id)
        (loop [i 0]
          (if (< i (alength waiters))
            (let [k (aget waiters i)]
              (if k
                (let [f (aget waiters (inc i))
                      old-waiters waiters]
                  (.unlock lock)
                  (f k)
                  (.lock lock)
                  (if (identical? old-waiters waiters)
                    (recur (+ 2 i))
                    (recur 0)))
                (recur (+ 2 i))))
            (.unlock lock)))
        (let [a waiter-id
              b waiters]
          (.unlock lock)
          (b a))))
    nil)
  Object
  (toString [_]
    (format "ACondVar(%s)" state)))



(deftype BConditionVariable [^:unsynchronized-mutable state]
  CV
  (getConditionState [this]
    state)
  (checkConditionState [this value]
    (identical? state value))
  (changeConditionState [this old-value new-value]
    (if (identical? old-value state)
      (do
        (set! state new-value)
        true)
      (do
        false)))
  (addConditionListener [this id listener])
  (removeConditionListener [this id])
  (doSignal [_])
  Object
  (toString [_]
    (format "BCondVar(%s)" state)))

(defn condition-variable ^CV [init-condition-state]
  (->AConditionVariable (ReentrantLock.)
                        init-condition-state
                        nil
                        nil))

(defn null-condition-variable ^CV [init-condition-state]
  (->BConditionVariable init-condition-state))

;; (deftype BConditionVariable [^:volatile-mutable state
;;                              ^objects ^:volatile-mutable waiters]
;;   p/ConditionVariable
;;   (get-condition-state [_]
;;     state)
;;   (check-condition-state [_ value]
;;     (identical? state value))
;;   (change-condition-state [_this old-value new-value]
;;     (if (identical? old-value state)
;;       (do
;;         (set! state new-value)
;;         true)
;;       false))
;;   (add-condition-listener [_ id listener]
;;     (set! waiters (table-insert waiters id listener))
;;     (listener id)
;;     nil)
;;   (remove-condition-listener [_ id]
;;     (table-remove waiters id)
;;     nil)
;;   (signal [_]
;;     (loop [i 0]
;;       (when (< i (alength waiters))
;;         (let [k (aget waiters i)]
;;           (if k
;;             (let [f (aget waiters (inc i))]
;;               (f k)
;;               (recur (+ 2 i)))
;;             (recur (+ 2 i)))))))
;;   Object
;;   (toString [_]
;;     (format "BCondVar(%s)" state)))

;; (defn uncoordinated-condition-variable [init-condition-state]
;;   (->BConditionVariable 
;;    init-condition-state
;;    (object-array 2)))


(defn fixed-size-buffer [^long n]
  (let [lst (LinkedList.)]
    (reify
      p/Buffer
      (full? [_]
        (<= n (count lst)))
      (add-item [this item]
        (.addLast lst item)
        this)
      (peek-item [_]
        (.peekFirst lst))
      (pop-item [_]
        (.pop lst)))))

(defn dropping-buffer [^long n]
  (let [lst (LinkedList.)]
    (reify
      p/Buffer
      (full? [_] false)
      (add-item [this item]
        (when (> n (count lst))
          (.addLast lst item))
        this)
      (peek-item [_]
        (.peekFirst lst))
      (pop-item [_]
        (.pop lst)))))

(defn sliding-buffer [^long n]
  (let [lst (LinkedList.)]
    (reify
      p/Buffer
      (full? [_] false)
      (add-item [this item]
        (when-not (> n (count lst))
          (.pop lst))
        (.addLast lst item)
        this)
      (peek-item [_]
        (.peekFirst lst))
      (pop-item [_]
        (.pop lst)))))

(defn unhandled-exception [^Throwable exception]
  (.uncaughtException
   (.getUncaughtExceptionHandler
    (Thread/currentThread))
   (Thread/currentThread)
   exception))

;; (deftype SPSC [^ReentrantLock lock 
;;                ^Condition condition
;;                ^long ^:unsynchronized-mutable selector
;;                ^:unsynchronized-mutable value]
;;   clojure.lang.IDeref
;;   (deref [_]
;;     (.lock lock)
;;     (loop []
;;       (case selector
;;         0 (do
;;             (.await condition)
;;             (recur))
;;         1 (let [v value]
;;             (.unlock lock)
;;             v)
;;         2 (let [v value]
;;             (.unlock lock)
;;             (throw v)))))
;;   clojure.lang.IFn
;;   (invoke [this r e]
;;     (.apply this r e))
;;   java.util.function.BiFunction
;;   (apply [_ r e]
;;     (.lock lock)
;;     (if (some? e)
;;       (do
;;         (set! selector 2)
;;         (set! value e))
;;       (do
;;         (set! selector 1)
;;         (set! value r)))
;;     (.signalAll condition)
;;     (.unlock lock)))

;; (defn spsc ^java.util.function.BiFunction []
;;   (let [lock (ReentrantLock.)]
;;     (->SPSC lock (.newCondition lock) 0 nil)))

;; (deftype Box [value]
;;   clojure.lang.IDeref
;;   (deref [_] value))

;; (def boxed-nil (Box. nil))

;; (defprotocol TStack
;;   (push-transformer [_ t]))

;; (defn t [tstack result exception]
;;   (if (seq tstack)
;;     (let [f (first tstack)
;;           tstack (rest tstack)
;;           r (try
;;               (if (some? exception)
;;                 (f nil exception)
;;                 (f result nil))
;;               (catch Throwable t
;;                 (Box. t)))]
;;       (if (instance? Box r)
;;         (recur tstack nil @r)
;;         (recur tstack r nil)))
;;     (if (some? exception)
;;       (throw exception)
;;       result)))

;; (deftype SPSC [thread
;;                value
;;                transformer-stack]
;;   TStack
;;   (push-transformer [_ t]
;;     (SPSC. thread value (cons t transformer-stack)))
;;   clojure.lang.IDeref
;;   (deref [_]
;;     (loop []
;;       (let [v @value]
;;         (if (some? v)
;;           (if (instance? Box v)
;;             (if (identical? v boxed-nil)
;;               (t transformer-stack nil nil)
;;               (t transformer-stack nil (.-value ^Box v)))
;;             (t transformer-stack v nil))
;;           (do
;;             (LockSupport/park)
;;             (recur))))))
;;   clojure.lang.IFn
;;   (invoke [this r e]
;;     (if (some? e)
;;       (set! value (Box. e))
;;       (if (some? r)
;;         (set! value r)
;;         (set! value boxed-nil)))
;;     (LockSupport/unpark thread)))

;; (defn spsc ^java.util.function.BiFunction []
;;   (->SPSC (Thread/currentThread) nil identity))



(definterface IResultBox
  (isFull [])
  (getValue [])
  (setValue [new-value])
  (addListener [l])
  (removeListener [n])
  (runListeners []))


(defmacro is-full [cv]
  `(.isFull ~(with-meta cv {:tag `IResultBox})))

(defmacro get-value [cv]
  `(.getValue ~(with-meta cv {:tag `IResultBox})))

(defmacro set-value [cv new-value]
  `(.setValue ~(with-meta cv {:tag `IResultBox}) ~new-value))

(defmacro add-listener [cv l]
  `(.addListener ~(with-meta cv {:tag `IResultBox}) ~l))

(defmacro remove-listener [cv l]
  `(.removeListener ~(with-meta cv {:tag `IResultBox}) ~l))

(defmacro run-listeners [cv]
  `(.runListeners ~(with-meta cv {:tag `IResultBox})))

(def null-sentinel (Object.))

(defn bag-grow [^objects bag item bag-insert]
  (let [new-table (object-array (* 2 (alength bag)))]
    (loop [i 0]
      (when (> (alength bag) i)
        (let [ni (nth bag i)]
          (bag-insert new-table ni))
        (recur (inc i))))
    (bag-insert new-table item)))

(defn bag-insert [^objects bag item]
  (let [start (long (mod (hash item) (alength bag)))]
    (loop [i start]
      (let [k (aget bag i)]
        (if (= k item)
          bag
          (if (nil? k)
            (do
              (aset bag i item)
              bag)
            (let [nxt (long (mod (inc i) (alength bag)))]
              (if (== start nxt)
                (bag-grow bag item bag-insert)
                (recur nxt)))))))))

(defn bag-remove [^objects bag item]
  (let [start (long (mod (hash item) (alength bag)))]
    (loop [i (long start)]
      (let [k (aget bag i)]
        (if (= k item)
          (do
            (aset bag i nil)
            bag)
          (let [nxt (long (mod (inc i) (alength bag)))]
            (if-not (== nxt start)
              (recur nxt)
              bag)))))))

(deftype ResultBox [^:volatile-mutable result
                    ^:unsynchronized-mutable ^objects listeners]
  IResultBox
  (isFull [_]
    (some? result))
  (getValue [_]
    (let [v result]
      (if (identical? null-sentinel v)
        nil
        v)))
  (setValue [_ new-value]
    (if (some? new-value)
      (set! result new-value)
      (set! result null-sentinel)))
  (addListener [_ l]
    (if (nil? listeners)
      (set! listeners l)
      (if (instance? X listeners)
        (set! listeners (bag-insert listeners l))
        (set! listeners
              (-> (object-array 2)
                  (bag-insert listeners)
                  (bag-insert l))))))
  (removeListener [_ n]
    (let [l listeners]
      (if (nil? l)
        nil
        (if (instance? X l)
          (set! listeners (bag-remove listeners n))
          (when (identical? l n)
            (set! listeners nil))))))
  (runListeners [this]
    (let [v (.getValue this)
          l listeners]
      (when l
        (if (instance? X l)
          (dotimes [i (alength l)]
            (when-let [f (aget l i)]
              (f v)))
          (l v))))))

(deftype NullResultBox []
  IResultBox
  (isFull [_] false)
  (getValue [_] nil)
  (setValue [_ new-value] nil)
  (addListener [_ l] nil)
  (removeListener [_ n] nil)
  (runListeners [this] nil))

(def the-null-result-box (NullResultBox.))

(defn result-box ^IResultBox []
  (ResultBox. nil nil))

(defn null-result-box ^IResultBox []
  (ResultBox. nil nil))
