(ns com.manigfeald.machinate
  (:refer-clojure :exclude [send
                            constantly])
  (:require
   [com.manigfeald.machinate.platform :as plt]
   [com.manigfeald.machinate.protocols :as p]
   [clojure.core :as cc]
   [com.manigfeald.machinate.queue :as q]))

#?(:clj (set! *warn-on-reflection* true))

#?(:clj (set! *unchecked-math* :warn-on-boxed))

;; TODO clojureclr?

;; TODO clojured, compiler blows up on ^objects type hint :/

;; TODO io, netty? nio?

;; At the lowest level, this is a library for synchronizing on
;; events. synchronizing on an event means registering a callback to
;; be invoked after the event occurs.

;; acts as never
(extend-type nil
  p/Event
  (try-event [_event _resume _resume-with-error _control _nack-group])
  p/Nackable
  (check-nack-group [_event _nack-group])
  p/QuasiEvent
  (push-down [event ctor lst]
    (if lst
      (doto (q/add-entry1 lst)
        (aset 0 (ctor event)))
      (ctor event)))
  p/Syncable
  (-sync [_event]))

(extend-type #?(:clj Object :cljs default)
  p/QuasiEvent
  (push-down [event ctor lst]
    (if lst
      (doto (q/add-entry1 lst)
        (aset 0 (ctor event)))
      (ctor event)))
  p/Syncable
  (-sync [event]
    (let [control (plt/condition-variable :waiting)
          pdr (p/push-down event identity nil)]
      (letfn [(resume-f [_evt _value nack-group]
                (if-not (volatile? pdr)
                  (p/check-nack-group event nack-group)
                  (let [q @pdr]
                    (loop [e (q/peek1 q)]
                      (when e
                        (p/check-nack-group (aget e 0) nack-group)
                        (recur (q/next1 q e)))))))
              (unhandled-error [event value nack-group]
                (plt/unhandled-exception value)
                (resume-f event value nack-group))]
        (if-not (volatile? pdr)
          (p/try-event pdr resume-f unhandled-error control #{})
          (let [q @pdr]
            (loop [e (q/peek1 q)]
              (when e
                (p/try-event (aget e 0) resume-f unhandled-error control #{})
                (recur (q/next1 q e)))))))))
  p/Nackable
  (check-nack-group [_event _nack-group]))

(defn ^:no-doc constantly* [g value]
  (reify
    p/Event
    (try-event [event resume resume-with-error control nack-group]
      (let [k (plt/unique-key)
            f (fn this-fn [k]
                (when (plt/change-condition-state control :waiting :synced)
                  (plt/signal control)
                  (plt/remove-condition-listener control k)
                  ((g resume resume-with-error) event value nack-group)))]
        (plt/add-condition-listener control k f)
        (f k)))))

(defn constantly
  "Returns an event that is immediatly synchronizable with the value value"
  [value]
  (constantly* (fn [resume _resume-with-error] resume) value))

(defn constantly-error
  "Returns an event that is immediatly synchronizable with the error value"
  [value]
  (constantly* (fn [_resume resume-with-error] resume-with-error) value))

(def always
  "Event that has always already occured. Synchronization always succeeds
  immediately. Value of the event is true."
  (constantly true))

(declare barrier)

(defn watch-reference1
  "Return an event that occurs when the given watchable clojure
reference's value is such that (test value) is truthy. Event has the
value true."
  [r test]
  (barrier
   (fn [e b!]
     (let [k (plt/unique-key)
           f (fn [k r _os ns]
               (when (test ns)
                 (remove-watch r k)
                 ;; could get called more than once in theory, should
                 ;; be fine
                 (b!)))]
       (add-watch r k f)
       (f k r nil @r))
     e)))

(defn watch-reference
  "Return an event that syncs when the given watchable clojure reference
  type (refs,atoms,agents) has the given value. Event has the value
  true."
  [r value]
  (watch-reference1 r #(= % value)))

(declare middleware)

(deftype Middleware [try-event-transform
                     check-nack-group-transform
                     push-down-transform
                     target]
  p/Event
  (try-event [_ resume resume-with-error control nack-group]
    ((try-event-transform p/try-event) target resume resume-with-error control nack-group))
  p/Nackable
  (check-nack-group [_ nack-group]
    ((check-nack-group-transform p/check-nack-group) target nack-group))
  p/QuasiEvent
  (push-down [_event ctor lst]
    ((push-down-transform p/push-down)
     target
     (comp ctor #(middleware try-event-transform
                             check-nack-group-transform
                             push-down-transform
                             %))
     lst)))

(defn middleware
  "Middleware smart constructor for building events. Middleware wraps an
  event, passing protocol calls down with the opportunity to intercept
  and make changes. The middleware smart constructor can collapse
  multiple middlewares into a single middleware via function
  composition."
  [try-event-transform check-nack-group-transform push-down-transform target]
  (if (instance? Middleware target)
    (let [^Middleware target target]
      (middleware (comp try-event-transform (.-try-event-transform target))
                  (comp check-nack-group-transform (.-check-nack-group-transform target))
                  (comp push-down-transform (.-push-down-transform target))
                  (.-target target)))
    (->Middleware try-event-transform check-nack-group-transform push-down-transform target)))

(defn barrier
  "Given a function f, applies f to a function g and an event e. e becomes
  occurs when g is invoked and has the value `true`. returns the result of invoking f."
  [f]
  (plt/barrier f))

;; events end up forming a tree, before synchronization that tree gets
;; flattened into something like disjuntive normal form, then at
;; synchronization one of the possible events is chosen.


(defn ^:no-doc choice*
  [evts ordered?]
  (middleware
   identity
   identity
   (fn [push-down]
     (fn [_event ctor lsg]
       (if (nil? lsg)
         (let [lst (q/new-q1)]
           (doseq [e (if-not ordered?
                       (shuffle evts)
                       evts)]
             (push-down e ctor lst))
           (volatile! lst))
         (doseq [e (if-not ordered?
                     (shuffle evts)
                     evts)]
           (push-down e ctor lsg)))))
   nil))

(defn ordered-choice
  "Takes a collection of events, and attempts to select
  event to synchronize on in order."
  [evts]
  (choice* evts true))


(defn choice
  "Takes a collection of events, and non-deterministically selects one on
  synchronization by defaults."
  ([evts]
   (choice* evts false)))

(defn nack
  "Like guard, but f is passed an event G. G will occur if on
  synchronization, the chosen event is not the one returned by f."
  [f]
  (middleware
   identity
   identity
   (fn [push-down]
     (fn [_event ctor lsg]
       (barrier
        (fn [b break]
          (let [e (f b)
                id (plt/unique-key)]
            (push-down
             e
             (comp ctor
                   (fn g [evt]
                     (middleware
                      (fn [try-event]
                        (fn [event resume resume-with-error control nack-group]
                          (try-event event resume resume-with-error control (conj nack-group id))))
                      (fn [check-nack-group]
                        (fn [event nack-group]
                          (when-not (contains? nack-group id)
                            (break))
                          (check-nack-group event nack-group)))
                      identity
                      evt)))
             lsg))))))
   nil))

(defn guard
  "Takes a function of no arguments f, returns an event E, before E is
  synchronized on, f will be invoked, and E replaced with the event
  returned by f."
  [f]
  (middleware
   identity
   identity
   (fn [push-down]
     (fn [_event ctor lst]
       (push-down (f) ctor lst)))
   nil))

(defn wrap
  "Wraps the post synchronization action f around the event evt"
  [evt f]
  (reify
    p/Event
    (try-event [_ resume resume-with-error control nack-group]
      (p/try-event evt
                   (fn [event value nack-group]
                     (try
                       (resume event (f value) nack-group)
                       (catch #?(:clj Throwable :cljs :any) t
                         (resume-with-error event t nack-group))))
                   resume-with-error
                   control
                   nack-group))
    p/QuasiEvent
    (push-down [event ctor lst]
      (p/push-down evt
                   (comp ctor #(if (identical? evt %) event (wrap % f)))
                   lst))
    p/Nackable
    (check-nack-group [_ nack-group]
      (p/check-nack-group evt nack-group))))

(defn wrap-handler
  "Wrap an error handler around an event. Handlers errors coming out of
the event itself, or any errors arising from other wrappings of the
event."
  [evt f]
  (reify
    p/Event
    (try-event [_ resume resume-with-error control nack-group]
      (p/try-event evt
                   resume
                   (fn [event value nack-group]
                     (try
                       (resume event (f value) nack-group)
                       (catch #?(:clj Throwable :cljs :any) t
                         (resume-with-error event t nack-group))))
                   control
                   nack-group))
    p/QuasiEvent
    (push-down [event ctor lst]
      (p/push-down evt
                   (comp ctor #(if (identical? evt %) event (wrap-handler % f)))
                   lst))
    p/Nackable
    (check-nack-group [_ nack-group]
      (p/check-nack-group evt nack-group))))

(defn alts
  "A work-a-like for clojure.core.async/alts!. Given a collection of
events returns a new event that when sync'ed chooses one event and
returns [event-value event]."
  [events]
  (choice
   (for [event events]
     (wrap event (fn [result] [result event])))))

(defn timeout
  "Given a delay in milliseconds, returns an event that doesn't
  synchronize until after that delay has elapsed. Value of the event
  is true."
  [^long delay]
  (let [at (plt/now)]
    (barrier
     (fn [b break]
       (nack
        (fn [neg]
          (let [handle (plt/run-after at delay break)]
            (p/-sync (wrap neg (fn [_] (plt/cancel-run-after handle))))
            b)))))))

(declare duplex-channel-event
         channel-close!)

;; The default rendevous channel, unbuffered, no transforms, pure send
;; and receive
(deftype DuplexChannel [lock is-closed left-head right-head]
  p/AsymmetricExchange
  (send-left [channel value]
    (duplex-channel-event channel left-head value))
  (send-right [channel value]
    (duplex-channel-event channel right-head value))
  p/SendChannel
  (send-it [this it]
    (duplex-channel-event this left-head it))
  p/ReceiveChannel
  (receive-it [this]
    (duplex-channel-event this right-head true))
  p/Channel
  (close! [this]
    (channel-close! this)))

(defn channel
  "Returns full duplex channel, both a SendChannel and a ReceiveChannel"
  []
  (let [lock (doto (plt/lock)
               (plt/do-lock))
        c (DuplexChannel.
           lock
           (plt/condition-variable :waiting)
           (q/new-q6)
           (q/new-q6))]
    (plt/unlock lock)
    c))

(declare sync!
         close!
         full-duplex)

(defn fixed-size-buffer
  "A fixed size buffer for buffered-channel. Accepts inputs until full."
  [^long capacity]
  (plt/fixed-size-buffer capacity))

(defn dropping-buffer
  "A dropping buffer for buffered-channel. Always accepts inputs, but
inputs are silently dropped if the buffer is full."
  [^long capacity]
  (plt/dropping-buffer capacity))

(defn sliding-buffer
  "A sliding buffer for buffered-channel. Always accepts inputs, but
drops the oldest item if the buffer is full."
  [^long capacity]
  (plt/sliding-buffer capacity))

;; TODO linked list buffer, volatile field of most recently consumed entry

;; buffered channel is two channels internally with a loop copying
;; between them with a buffer in the middle.
(defn buffered-channel
  "Create a buffered full duplex channel. The buffer allows for senders
  to add to the channel without waiting for a receiver until the
  buffer's capacity is reached. Buffered channels optional can have a
  transducer (xform) which transforms values that flow through the
  channel. If a transducer is specified, optional an exception halder
  can be specified to handle any exceptions that may arise."
  ([buffer-or-n]
   (buffered-channel buffer-or-n identity))
  ([buffer-or-n xform]
   (buffered-channel buffer-or-n xform (comp (constantly nil) plt/unhandled-exception)))
  ([buffer-or-n xform ex-handler]
   (assert (or (not (number? buffer-or-n))
               (pos? (long buffer-or-n))))
   (let [b (volatile!
            (if (number? buffer-or-n)
              (fixed-size-buffer buffer-or-n)
              buffer-or-n))
         in-c (channel)
         out-c (channel)
         add! (xform (fn
                       ([b] b)
                       ([b item]
                        (assert (not (nil? item)))
                        (p/add-item b item))))]
     ;; TODO maybe needs some kind of trampoline?
     ((fn copy-loop [open?]
        (let [item (p/peek-item @b)]
          (sync!
           (choice
            [(if (some? item)
               (wrap (p/send-it out-c item)
                     (fn [_sent-result]
                       (p/pop-item @b)
                       (copy-loop open?)))
               (when-not open?
                 (p/close! out-c)
                 nil))
             (if open?
               (if (p/full? @b)
                 nil
                 (wrap (p/receive-it in-c)
                       (fn [new-item]
                         (if (some? new-item)
                           (do
                             (when (reduced?
                                    (try
                                      (add! @b new-item)
                                      (catch #?(:clj Throwable :cljs :any) t
                                        (if-some [r (ex-handler t)]
                                          (p/add-item @b r)
                                          @b))))
                               (close! in-c))
                             (copy-loop open?))
                           (do
                             (try
                               (add! @b)
                               (catch #?(:clj Throwable :cljs :any) t
                                 (if-some [r (ex-handler t)]
                                   (p/add-item @b r)
                                   @b)))
                             (copy-loop false))))))
               nil)]))))
      true)
     (full-duplex in-c out-c))))

(defn ^:no-doc exchange-locked? [left-control  left-result right-control  right-result]
  (if (plt/change-condition-state left-control :waiting :claimed)
    (if-not (plt/is-full left-result)
      (if (plt/change-condition-state right-control :waiting :claimed)
        (if-not (plt/is-full right-result)
          true
          (do
            (plt/change-condition-state right-control :claimed :waiting)
            (plt/signal right-control)
            (plt/change-condition-state left-control :claimed :waiting)
            (plt/signal left-control)
            false))
        (do
          (plt/change-condition-state left-control :claimed :waiting)
          (plt/signal left-control)
          false))
      (do
        (plt/change-condition-state left-control :claimed :waiting)
        (plt/signal left-control)
        false))
    false))


(defn ^:no-doc half-exchange-locked? [left-control  left-result]
  (if (plt/change-condition-state left-control :waiting :claimed)
    (if-not (plt/is-full left-result)
      true
      (do
        (plt/change-condition-state left-control :claimed :waiting)
        (plt/signal left-control)
        false))
    false))

(defn ^:no-doc do-closed [^objects left left-head lock]
  (let [left-resume  (aget left 0)
        left-control (aget left 1)
        left-ng      (aget left 2)
        left-value   (aget left 3)
        left-result  (aget left 4)
        left-event   (aget left 5)]
    ;; TODO check result
    (if (half-exchange-locked? left-control left-result)
      (do
        (plt/set-value left-result nil)
        (plt/unlock lock)
        (plt/change-condition-state left-control :claimed :synced)
        (plt/signal left-control)
        (left-resume left-event nil left-ng)
        (plt/run-listeners left-result)
        (plt/do-lock lock)
        true)
      (do
        (when (plt/check-condition-state left-control :synced)
          (q/unlink6 left-head left))
        (plt/unlock lock)
        (plt/do-lock lock)
        false))))

(defn ^:no-doc do-exchange [left-control right-control  left-result  right-result right-value left-value lock left-resume right-resume left-event right-event left-ng right-ng]
  ;; don't hold any locks by the time callbacks are called
  (plt/change-condition-state right-control :claimed :synced)
  (plt/change-condition-state left-control :claimed :synced)
  (plt/set-value left-result right-value)
  (plt/set-value right-result left-value)
  (plt/unlock lock)
  (plt/signal left-control)
  (plt/signal right-control)
  (left-resume left-event right-value left-ng)
  (right-resume right-event left-value right-ng)
  (plt/run-listeners left-result)
  (plt/run-listeners right-result)
  (plt/do-lock lock))

;; TODO make is-closed volatile field

(defn ^:no-doc channel-close! [^DuplexChannel channel]
  (let [is-closed (.-is-closed channel)
        lock      (.-lock channel)]
    (when (plt/change-condition-state is-closed :waiting :synced)
      (plt/signal is-closed)
      (plt/do-lock lock)
      (loop []
        ;; TODO do-closed might execute callbacks while locked
        (if-let [^objects left (q/peek6 (.-left-head channel))]
          (if-let [^objects right (q/peek6 (.-right-head channel))]
            nil ; TODO trigger loop?
            (when (do-closed left (.-left-head channel) lock)
              (recur)))
          (if-let [^objects right (q/peek6 (.-right-head channel))]
            (when (do-closed right (.-right-head channel) lock)
              (recur))
            nil ; TODO trigger loop?
            )))
      (plt/unlock lock)
      #_(plt/remove-condition-listener is-closed k))
    nil))

;; TODO single queue?

;; TODO make event link in queue?

(defn ^:no-doc duplex-channel-event
  [^DuplexChannel channel' write-head value]
  (let [lock      (.-lock channel')
        result (plt/result-box)
        is-closed (.-is-closed channel')]
    (reify
      p/Event
      (try-event [event resume _resume-with-error control nack-group]
        (if (and (plt/is-full result)
                 (plt/change-condition-state control :waiting :synced))
          (do
            (plt/signal control)
            (resume event (plt/get-value result) nack-group))
          (do
            (plt/do-lock lock)
            (let [entry (doto (q/add-entry6 write-head)
                          (aset 0 resume)
                          (aset 1 control)
                          (aset 2 nack-group)
                          (aset 3 value)
                          (aset 4 result)
                          (aset 5 event))]
              (loop []
                (if-let [^objects left (q/peek6 (.-left-head channel'))]
                  (let [left-resume  (aget left 0)
                        left-control (aget left 1)
                        left-ng      (aget left 2)
                        left-value   (aget left 3)
                        left-result  (aget left 4)
                        left-event   (aget left 5)]
                    (if (and (plt/is-full left-result)
                             (plt/change-condition-state left-control :waiting :synced))
                      (do
                        (plt/unlock lock)
                        (plt/signal left-control)
                        (left-resume left-event (plt/get-value left-result) left-ng)
                        (plt/do-lock lock)
                        (recur))
                      (if-let [^objects right (q/peek6 (.-right-head channel'))]
                        (let [right-resume  (aget right 0)
                              right-control (aget right 1)
                              right-ng      (aget right 2)
                              right-value   (aget right 3)
                              right-result  (aget right 4)
                              right-event   (aget right 5)]
                          (if (and (plt/is-full right-result)
                                   (plt/change-condition-state right-control :waiting :synced))
                            (do
                              (plt/unlock lock)
                              (plt/signal right-control)
                              (right-resume right-event (plt/get-value right-result) right-ng)
                              (plt/do-lock lock)
                              (recur))
                            (if-not (plt/check-condition-state left-control :synced)
                              (if-not (plt/check-condition-state right-control :synced)
                                (if (exchange-locked? left-control left-result right-control right-result)
                                  (do
                                    (do-exchange left-control
                                                 right-control
                                                 left-result
                                                 right-result
                                                 right-value
                                                 left-value
                                                 lock
                                                 left-resume
                                                 right-resume
                                                 left-event
                                                 right-event
                                                 left-ng
                                                 right-ng)
                                    (recur))
                                  (recur))
                                (do
                                  (q/unlink6 (.-right-head channel') right)
                                  (recur)))
                              (do
                                (q/unlink6 (.-left-head channel') left)
                                (recur)))))
                        (if (plt/check-condition-state is-closed :synced)
                          (do
                            (do-closed left (.-left-head channel') lock)
                            (when (identical? left (q/peek6 (.-left-head channel')))
                              (q/unlink6 (.-left-head channel') left))
                            (recur))
                          nil))))
                  (if-let [^objects right (q/peek6 (.-right-head channel'))]
                    (if (plt/check-condition-state is-closed :synced)
                      (do
                        (do-closed right (.-right-head channel') lock)
                        (when (identical? right (q/peek6 (.-right-head channel')))
                          (q/unlink6 (.-right-head channel') right))
                        (recur))
                      nil)
                    nil)))
              (when-not (plt/check-condition-state control :synced)
                (let [f (fn [v]
                         (when (plt/change-condition-state control :waiting :synced)
                           (plt/signal control)
                           (resume event v nack-group)))
                      _ (plt/add-listener result f)
                      control-listener (fn [k]
                                         (when (plt/check-condition-state control :synced)
                                           (plt/remove-condition-listener control k)
                                           (plt/do-lock lock)
                                           (plt/remove-listener result f)
                                           ;; TODO remove result listener on synced
                                           (when (identical? control (aget entry 1)) ;; TODO required for re-using queue nodes
                                             (q/unlink6 write-head entry))
                                           (plt/unlock lock))
                                         (when (and (plt/is-full result)
                                                    (plt/change-condition-state control :waiting :synced))
                                           (plt/signal control)
                                           (resume event (plt/get-value result) nack-group)))
                      k (plt/unique-key)]
                  (plt/add-condition-listener control k control-listener)
                  #_(plt/add-condition-listener result k result-listener)))
              (plt/unlock lock))))))))

(defn send
  "Creates an event where synchronization completes after the given value
  has been sent via the given channel, or after the given channel
  closes. Returns true or false"
  [channel value]
  (assert (some? value))
  (p/send-it channel value))

(defn receive
  "Creates an event where synchronization completes after receivng a value
  via the given channel or the channel is closed. Returns the received
  value or nil."
  [channel]
  (p/receive-it channel))

(defn close!
  "Close a channel."
  [channel]
  (p/close! channel))

(defn full-duplex
  "Given a SendChannel and a ReceiveChannel return a single channel that
is both send and receivable."
  [send-channel receive-channel]
  (reify
    p/SendChannel
    (send-it [_ it]
      (p/send-it send-channel it))
    p/ReceiveChannel
    (receive-it [_]
      (p/receive-it receive-channel))
    p/Channel
    (close! [_]
      (close! send-channel))))

(defn send-only
  "Given a full duplex channel return a channel that is only a
SendChannel"
  [channel]
  (reify
    p/SendChannel
    (send-it [_ it]
      (p/send-it channel it))
    p/Channel
    (close! [_]
      (close! channel))))

(defn receive-only
  "Given a full duplex channel return a channel that is only a
ReceiveChannel"
  [channel]
  (reify
    p/ReceiveChannel
    (receive-it [_]
      (p/receive-it channel))
    p/Channel
    (close! [_]
      (close! channel))))

(defn sync!
  "Run a callback after the given event occurs. Returns a
  CompletableFuture (or whatever the platform supplied promise like
  type is) that will be completed after the event occurs."
  ([evt]
   (let [cf (plt/promise-like-type)]
     (p/-sync
      (-> evt
          (wrap (fn [value] (plt/complete-plt cf value)))
          (wrap-handler (fn [exception] (plt/complete-plt-error cf exception)))
          
          ))
     cf))
  ([evt callback]
   (sync! (wrap evt callback))))

#?(:clj

   #_(defn sync!!
     "Stop and wait for the given event to occur"
     [evt]
     @(sync! evt))

   #_(defn sync!!
     "Stop and wait for the given event to occur"
     [evt]
     (let [cf (plt/spsc)]
       (p/-sync
        (-> evt
            (wrap (fn [value] (cf value nil)))
            (wrap-handler (fn [exception] (cf nil exception)))))
       @cf))

   ;; TODO :/

   (do

     (deftype E [^:volatile-mutable selector ^:volatile-mutable value ^Thread t]
       clojure.lang.IDeref
       (deref [_]
         (loop []
           (if (= 0 selector)
             (do
               (java.util.concurrent.locks.LockSupport/park)
               (recur))
             (if (= 1 selector)
               value
               (throw value)))))
       clojure.lang.IFn
       (invoke [_ x]
         (set! value x)
         (set! selector 1)
         (java.util.concurrent.locks.LockSupport/unpark t))
       (invoke [_ _ x]
         (set! value x)
         (set! selector 2)
         (java.util.concurrent.locks.LockSupport/unpark t)))

     (defn sync!!
       "Stop and wait for the given event to occur"
       [evt]
       (let [b (E. 0 nil (Thread/currentThread))]
         (p/-sync
          (-> evt
              (wrap b)
              (wrap-handler (fn [value] (b nil value)))))
         @b))))

(defn pubsub
  "A pubsub allows messages to be sent to topics. input is a channel of
messages, topic-fn is message -> topic. All listeners to a topic are
sent messages in parallel. The pubsub will not move on to the next
message until all listeners have recieved a given message."
  [input topic-fn]
  (letfn [(mass-send [value outputs]
            (barrier
             (fn [evt b!]
               (let [c (count outputs)
                     results (atom outputs)
                     e (plt/condition-variable 0)]
                 (plt/add-condition-listener e
                                             (plt/unique-key)
                                             (fn [_]
                                               (when (= c (plt/get-condition-state e))
                                                 (b!))))
                 (doseq [[channel _] outputs]
                   (sync! (wrap (send channel value)
                                (fn [send-result]
                                  (when-not send-result
                                    (swap! results dissoc channel))
                                  (loop []
                                    (let [n (long (plt/get-condition-state e))]
                                      (if (plt/change-condition-state e n (inc n))
                                        (plt/signal e)
                                        (recur))))))))
                 (wrap evt (fn [_] @results))))))
          (main-loop [commands outputs]
            (sync!
             (choice
              [(wrap (receive commands)
                     (fn [command]
                       (case (nth command 0)
                         :subscribe (main-loop commands
                                               (assoc-in outputs [(nth command 1)
                                                                  (nth command 2)]
                                                         (nth command 3)))
                         :unsubscribe (let [o (dissoc (get outputs (nth command 1)) (nth command 2) )]
                                        (if (seq o)
                                          (main-loop commands (assoc outputs (nth command 1) o))
                                          (main-loop commands (dissoc outputs (nth commands 1))))))))
               (wrap (receive input)
                     (fn [new-msg]
                       (if new-msg
                         (let [t (topic-fn new-msg)
                               topic-outputs (get outputs t)]
                           (sync! (wrap (mass-send new-msg topic-outputs)
                                        (fn [topic-outputs]
                                          (if (seq topic-outputs)
                                            (main-loop commands
                                                       (assoc outputs t topic-outputs))
                                            (main-loop commands
                                                       (dissoc outputs t)))))))
                         (doseq [[_topic topic-outputs] outputs
                                 output topic-outputs]
                           (when (:close? output)
                             (p/close! (:ch output)))))))])))]
    (let [commands (buffered-channel 1024)]
      (main-loop commands {})
      (reify
        p/PubSub
        (subscribe [_ topic channel close?]
          (wrap (send commands [:subscribe topic channel close?]) (cc/constantly channel)))
        (unsubscribe [_ topic channel]
          (wrap (send commands [:unsubscribe topic channel])
                (cc/constantly channel)))
        p/SendChannel
        (send-it [_ it]
          (p/send-it input it))))))

(defn sub
  "Given a pubsub add a channel as a listener to a given topic. Returns
an event that must be synchronized on. After that event occurs the
  listener has been added to the pubsub. If close? is true then the
  listener channel will be closed if the input to the pubsub is
  closed."
  ([ps topic channel]
   (sub ps topic channel true))
  ([ps topic channel close?]
   (p/subscribe ps topic channel close?)))

(def sub! (comp sync! sub))

(defn unsub
  "Remove a channel as a listener for the given topic from a
pubsub. Returns an event that must be synchronized on."
  [ps topic channel]
  (p/unsubscribe ps topic channel))

(def unsub! (comp sync! unsub))
