(ns kotr.pipeline
  (:require [clojure.tools.logging :as log]
            [clojure.tools.reader.edn :as edn]
    ;  [kotr.pipeline.connector.sink-conntor-util :as u]
            [kotr.pipeline.gen :as gen]
            [kotr.pipeline.connector :as conn]
            [cheshire.core :as json]
            [kotr.meta-data-repo :as md]
            [kotr.kafka.client :as client]
            [kotr.kafka.client.connector-api]
            [kotr.kafka.client.ksql-api]
            [kotr.kafka.client.schema-api]
            [kotr.pipeline.gen.protocol :as p]))

(def context (client/init-context))

(defn load-app-config [app-file]
  (-> (slurp app-file)
      (edn/read-string)))


(defn init-context [m]
  (let [m (if (string? m)
            (load-app-config m)
            m)]
    (alter-var-root #'context (constantly m))))


(defn rtry! [f & args]
  (try
    (apply f args)
    (catch Exception e
      (-> (ex-data e)
          :body))))


(defn get-context []
  context)


(defn invoke
  ([op-name] (client/invoke context {:op op-name}))
  ([op-name request]
   (client/invoke context {:op op-name :request request})))


#_(defn print-table [] (client/print-table))
#_(defn print-json [] (client/print-json))

(defn print-topic
  ([topic-name] (print-topic topic-name 1 println))
  ([topic-name v]
   (if (number? v)
     (print-topic topic-name v println)
     (print-topic topic-name 1 v)))
  ([topic-name total callback]
   (client/validate-topic-name context topic-name)
   (client/print-topic-async context topic-name total callback)))


(defn print-stream [stream-name total-requested]
  (client/validate-stream-name context stream-name)
  (let [query-str (str "select * from " stream-name " emit changes; ")
        ;total (atom -1)
        out (client/query context query-str total-requested)]
    out
    #_(client/query-async context query-str (fn [v]
                                              (swap! total inc)
                                              (when (= total-requested @total)
                                                (client/stop-query context query-str)
                                                )
                                              (println v)
                                              ))
    )
  )


(defn get-error [total-requested]
  (let [error-topic-name (get context :error-stream "KSQL_PROCESSING_LOG")
        w (str "select * from " error-topic-name " EMIT CHANGES;")
        ;total (atom -1)
        out (client/query context w total-requested)
        out (mapv (fn [m]
                    (-> m
                        (select-keys [:rowtime :rowkey :logger :level :time])
                        (assoc :type (get-in m [:message :TYPE]))
                        (assoc :type (get-in m [:message :TYPE]))
                        (assoc :error-msg (get-in m [:message :RECORDPROCESSINGERROR :ERRORMESSAGE]))
                        (assoc :record (get-in m [:message :RECORDPROCESSINGERROR :RECORD]))
                        ;  (update :record (fn [v] (json/parse-string v true) ))
                        )
                    ) out)
        ; out (clojure.walk/postwalk out )
        ]
    out

    )

  )


(defn print-topic-as-table
  ([topic-name] (print-topic-as-table topic-name 1))
  ([topic-name total]
   (client/validate-topic-name context topic-name)
   (-> (client/as-json-response context topic-name total)
       (clojure.pprint/print-table))))


(defn topic-as-json
  ([topic-name] (topic-as-json topic-name 1))
  ([topic-name total]
   (client/validate-topic-name context topic-name)
   (client/as-json-response context topic-name total)))


(defn ksql [ksql-str]
  (client/ksql context ksql-str))

(defn query-async
  ([ksql-str] (client/query-async context ksql-str println))
  ([ksql-str callback]
   (client/query-async context ksql-str callback)))

(defn stop-query [ksql-str]
  (client/stop-query context ksql-str))



;(query "select * from KSQL_PROCESSING_LOG EMIT CHANGES;")

;(stop-query "select * from KSQL_PROCESSING_LOG EMIT CHANGES;")

(defn print-table [v]
  (clojure.pprint/print-table v)
  )



(defn count-topic-row
  [topic-name]
  (client/count-topic-row context topic-name))




(defn get-schema []
  (md/get-kafka-schema context)
  ;(gen/get-schema)
  )


(defn create-dataflow-req
  ([mapping] (create-dataflow-req {} mapping))
  ([schema-m mapping]
   (gen/create-dataflow-req context mapping schema-m)))


(defn export-ksql-to-file [file-name mapping]
  (let [o (gen/create-dataflow-req context mapping (get-schema))
        o (clojure.string/join "\n" o )]
    (spit file-name o :append true)))


(defn import-ksql-from-file [file-name]
  (->>
      (clojure.string/split (slurp file-name) #"\n")
      (remove (fn [v] (clojure.string/blank? v) ))
      (vec)
      (gen/get-delta-ksql-statement context)
      (client/ksql-batch context)
      )
  )


(defn create-dataflow
  [mapping]
  (let [current-stream-schema-m (md/get-kafka-schema context)
        out (gen/create-dataflow-req context mapping current-stream-schema-m)]
    (->> out
         (gen/get-delta-ksql-statement context)
         (client/ksql-batch context))))



(defn delete-dataflow [mapping]
  (let [schema-m (md/get-kafka-schema context)
        ksql (gen/delete-dataflow-req context mapping schema-m)]
    ; (println "--" ksql )
    (client/ksql-batch context ksql))
  #_(try

      (catch Exception e
        (let [_ (log/error e)
              w (-> (ex-data e)
                    :body
                    (json/parse-string true)
                    :message)]
          w))))


(defn delete-dataflow-with-data [mapping]
  (let [schema-m (md/get-kafka-schema context)
        m (gen/delete-dataflow-req-with-data context mapping schema-m)
        ksql-coll (get m :ksql)
        topic-coll (get m :topic)
        schema-name-coll (get m :schema)]

    (doseq [ksql ksql-coll]
      (client/ksql context ksql))
    (client/delete-topics context topic-coll)
    (doseq [schema-name schema-name-coll]
      (invoke "delete-schema" schema-name)))

  #_(try

      (catch Exception e
        (let [_ (log/error e)
              w (-> (ex-data e)
                    :body
                    (json/parse-string true)
                    :message)]
          w))))


(defn update-dataflow [req]
  (do (delete-dataflow req)
      (create-dataflow req)))

(defn update-dataflow-with-data [req]
  (do (delete-dataflow-with-data req)
      (create-dataflow req)))


;;;;;;;;;;;;;;;;;;;; End dataflow api

;(def delete-dataflow gen/delete-dataflow)
;(def delete-dataflow-with-data gen/delete-dataflow-with-data)

#_(defn create-dataflow-by-source-mapping

    [coll]
    (let [ksql-coll (gen/create-dataflow-req-by-file-name context coll)
          ksql-coll (gen/get-delta-ksql-statement context ksql-coll)
          ]
      ;ksql-coll
      (when-not (empty? ksql-coll)
        (client/ksql-batch context ksql-coll))))


#_(defn delete-dataflow-by-source-mapping [base-dir connector-mapping]
    ;(clojure.pprint/pprint req-coll)
    (let [schema-m (md/get-kafka-schema context)
          ksql (gen/delete-dataflow-req-by-source-mapping context base-dir schema-m connector-mapping)]
      (client/ksql-batch context ksql)))

;(def create-dataflow-by-source-mapping gen/create-dataflow-by-source-mapping)
;(def create-dataflow-req-from-source gen/create-dataflow-req-from-source)
;(def delete-dataflow-by-source-mapping gen/delete-dataflow-by-source-mapping)





(defn create-source-connector
  ([coll] (create-source-connector "" coll))
  ([base-dir connector-mapping]
    ;(create-dataflow-by-source-mapping base-dir connector-mapping)
   (let [conn-coll (->> connector-mapping
                        (conn/create-source-connector-req base-dir)
                        (conn/find-delta-connection context))]
     (when-not (empty? conn-coll)
       (log/info "Creating new connection")
       (invoke "create-connector-batch" conn-coll)))))


(defn delete-source-connector
  ([coll] (delete-source-connector "" coll))
  ([base-dir connector-mapping]
   (let [connector-name-list (conn/get-source-connector-names base-dir connector-mapping)]
     (doseq [connector-name connector-name-list]
       (invoke "delete-connector" connector-name)))))


(defn pause-source-connector
  ([coll] (delete-source-connector "" coll))
  ([base-dir connector-mapping]
   (let [connector-name-list (conn/get-source-connector-names base-dir connector-mapping)]
     (doseq [connector-name connector-name-list]
       (invoke "pause-connector" connector-name)))))


(defn resume-source-connector
  ([coll] (delete-source-connector "" coll))
  ([base-dir connector-mapping]
   (let [connector-name-list (conn/get-source-connector-names base-dir connector-mapping)]
     (doseq [connector-name connector-name-list]
       (invoke "resume-connector" connector-name)))))


;(def create-source-connector gen/create-source-connector)
;(def delete-source-connector gen/delete-source-connector)
#_(defn create-sink-connector
    ([sink-mapping] (create-sink-connector "" sink-mapping))
    ([base-dir sink-mapping]
     (u/create-sink-connector context base-dir sink-mapping)))


(defn create-sink-connector

  [base-dir sink-connector-mapping]
  (let [[conn-coll ksql-coll] (conn/create-sink-connector context base-dir sink-connector-mapping)

        ; req-coll (into others w)
        ;   conn-coll (crate-sink-connector-req connector-schema)
        ;   _ (clojure.pprint/pprint conn-coll)
        conn-coll (conn/find-delta-connection context conn-coll)
        ]

    ;(clojure.pprint/pprint w)
    ;(clojure.pprint/pprint create-sql-for-sink)
    ;(clojure.pprint/pprint sql-coll)

    (when-not (empty? ksql-coll)
      (let [ksql-coll (gen/get-delta-ksql-statement context ksql-coll)]
        (client/ksql-batch context ksql-coll)
        )
      )

    (when-not (empty? conn-coll)
      (log/info "Creating new connection")
      (invoke "create-connector-batch" conn-coll))

    )
  )

;(def push-stardog u/push-stardog)

(defn as-entity-name [f-name]
  (let [w (clojure.string/split f-name #"/")]
    (if-let [t (last w)]
      (first (clojure.string/split t #"\."))
      f-name)))


(defn push-stardog [context model-coll]
  (log/info "Push stardog kafka context " context)
  (let [c (client/topic-config-for-metadata "_pipeline_metadata")]
    (doseq [w model-coll]
      (let [[ds_file data_file key entity_name] w
            data_file (as-entity-name data_file)
            v {:ds-name     (as-entity-name ds_file)
               :entity_name (or data_file entity_name)
               :key         key}
            k (or entity_name data_file)]
        (client/publish context c k v)))))

#_(defn undeploy-connector-batch [connector-list]
    (doseq [c connector-list]
      (invoke "delete-connector" c)))


(defn as-dataflow-req
  [v]
  (-> v
      (clojure.string/split #"\n")
      (->> (into [] (comp (map (fn [v] (clojure.string/split v #";")))
                          (map (fn [v] (into [] (map clojure.string/trim) v))))))))



#_(defmulti pipeline-invoke (fn [api-name _] api-name))

#_(defmethod pipeline-invoke
    :default
    [api-name app-context]
    (let [api-list (-> (into #{} (keys (methods pipeline-invoke)))
                       (disj :default))
          error-msg (str "\napi name not found, available api \n\n" (clojure.string/join "\n" api-list))]
      (throw (ex-info error-msg {:api-name      api-name
                                 :available-api api-list}))))






#_(defn -main [& args]
    (let [[file-name api-name] args]
      (log/info (str "invoking context-file  " file-name " with name ") api-name)
      (->> (load-app-config file-name)
           (pipeline-invoke api-name))))


(comment


  u/context


  (init-context "app/sample/lcontext.edn")

  (create-source-connector "app/sample/" "app/sample/mapping_connector/source_mapping.csv")

  (create-dataflow "app/sample/mapping_transformation/trans_party_mapping.csv")

  (create-dataflow "app/sample/mapping_transformation/trans_policy_mapping.csv")


  ;(delete-dataflow "app/sample/mapping/trans_party_mapping.csv")

  (u/create-sink-connector "app/sample/" "app/sample/mapping_connector/raw_sink_mapping.csv")

  (u/create-sink-connector "app/sample/" "app/sample/mapping_connector/refined_sink_mapping.csv")



  (->> [[["my_party4" "my_partyid" "tnf_party/party_id" "string" "doc..."]
         ["my_party4" "party_id" "(gen_id_md5 (concat tnf_party/party_id '123' ) '-123'  )"]
         ["my_party4" "policy_number2" "123"]]]
       (create-dataflow-req)
       ;(create-dataflow "app/demo/")
       ;(update-dataflow )
       )


  (->> [[["party_female" "party_id" "tnf_party/party_id" "string" "doc..."]
         ["party_female" "ramdom_key" "(gen_id_md5 (concat tnf_party/party_id '123' ) '-123'  )"]
         ["party_female" "policy_number2" "123"]
         ["party_female" nil "(where (= tnf_party/gender 'F'))"]
         ]]
       (create-dataflow-req)
       ;(create-dataflow "app/demo/")
       ;(update-dataflow )
       )


  (->> (print-topic "role" 3)
       ; (clojure.pprint/print-table)
       )



  (invoke "show-functions")


  (client/consume u/context "ADDRESS3" (fn [v]
                                         (when (< 0 (count v))
                                           (println (take 2 v))
                                           #_(clojure.pprint/print-table (take 10 v))
                                           ;  (println (count v) )
                                           )))


  @client/consume-state-atom

  (reset! client/consume-state-atom {})

  (-> (invoke "describe-stream" "role")
      ; (convert-stream-to-avro-schema)
      )


  (invoke "describe-schema" "role_staging-value")


  (package
    "app/demo/"
    {:sink-connectors         ["app/demo/mapping/sink-mapping.csv"]
     :source-connectors       ["app/demo/mapping/source-mapping.csv"]
     :mapping-transformations ["app/demo/mapping/trans-mapping.csv"]
     }
    )


  (invoke "show-topics")

  (invoke "show-schemas")

  (invoke "describe-schema" "id3_address_in-value")

  (invoke "show-stream-name")

  (invoke "show-table-name")

  (invoke "show-topics-name")

  (->> (print-topic "id3_polrelp_policy_relationship" 3)
       (clojure.pprint/print-table)
       )

  (ksql "CREATE TABLE address3 AS SELECT city, COUNT(*) as total FROM tnf_party_address GROUP BY city EMIT CHANGES;")

  (query-async "SELECT city, COUNT(*) as total FROM tnf_party_address GROUP BY city EMIT CHANGES;"
               #_(fn [v]
                   (println "--" v)

                   )
               )

  (stop-query "SELECT city, COUNT(*) as total FROM tnf_party_address GROUP BY city EMIT CHANGES;")

  (invoke "show-stream-name")

  (invoke "show-connectors")

  (invoke "show-functions")

  (invoke "show-connectors-status")

  (invoke "delete-all-connector")

  (invoke "describe-connector" "product_staging_csv_source")


  (deploy-source-connector "app/demo/" "app/demo/mapping/source_mapping.csv")

  (deploy-dataflow "app/demo/" "app/demo/mapping/trans_party_mapping.csv")


  (deploy-dataflow "app/demo/" "app/demo/mapping/trans_policy_mapping.csv")

  (deploy-sink-connector "app/demo/" "app/demo/mapping/sink_mapping.csv")

  (gen-ksql "app/demo/" "app/demo/mapping/trans-mapping.csv")

  (undeploy-dataflow "app/demo/mapping/trans_policy_mapping.csv")




  (->> [[["my_party4" "my_partyid" "tnf_party/party_id" "string" "doc..."]
         ["my_party4" "party_id" "tnf_party/party_id"]
         ["my_party4" "policy_number2" "123"]]]
       (deploy-dataflow "app/demo/"))

  (->> [[["my_party4" "my_partyid" "tnf_party/party_id" "string" "doc..."]
         ["my_party4" "party_id" "tnf_party/party_id"]
         ["my_party4" "policy_number2" "123"]]]
       (undeploy-dataflow))


  (->> [["repo/sql_sink.json" "my_party4" "my_partyid" "my_party4"]]
       (deploy-sink-connector "app/demo/")
       ;(push-to-kafka)
       ;(gen-schema)
       )




  (invoke "describe-schema" "ID3_PERSON_IN-value")

  (->> (invoke "show-data" {:topic-name "_pipeline_metadata" :total 2})
       #_(clojure.pprint/print-table))




  (push-stardog
    [["tnf_db" "ID3_PERSON_IN" "BSACSG_Party_Surrogate_Key" "ID3_PERSON_IN"]
     ["tnf_db" "ID3_ZVSPOLP_Policy" "FBFNCU_Policy_Id" "ID3_ZVSPOLP_Policy"]
     ["tnf_db" "tnf_party" "party_id" "tnf_party"]
     ["tnf_db" "tnf_party_address" "party_id" "tnf_party_address"]])




  (push-stardog [["tnf_db" "my_party" "my_partyid" "my_party"]])


  (->> [["tnf_policy21" "policy_number" "id3_zvspolp_policy/FBFPCU_State_Id" "string" "doc..."]
        ["tnf_policy21" "policy_number65" "123"]
        ["tnf_policy21" "policy_number21" "123"]
        ["tnf_policy21" "policy_number223" "123"]]
       ;(deploy-dataflow)
       (undeploy-dataflow)
       )



  (->> (invoke "print-topic" {:topic-name "TNF_PARTY" :total 4})
       (clojure.pprint/print-table))

  (-> " tnf_party;party_id;id3_person_in/BSACSG_Party_Surrogate_Key;;String
        tnf_party;last_name;id3_person_in/BSB9NA_Last_Name;;String
        tnf_party;middle_name;id3_person_in/BSCCNA_Middle_Name;;String
        tnf_party;first_name;id3_person_in/BSCBNA_First_Name;;String
        tnf_party;prefix;id3_person_in/BSCENA_Prefix_Name;;String
        tnf_party;suffix;id3_person_in/BSCDNA_Suffix_Name;;String
        tnf_party;short_name;id3_person_in/BSENNA_Alternate_Correspond_Name;;String
        tnf_party;gender;id3_person_in/BSDCST_Gender_Code;;String
        tnf_party;date_of_birth;id3_person_in/BSACDX_Birth_Date;;String
        tnf_party;date_of_death;id3_person_in/BSADDX_Deceased_Date;;String
        tnf_party;tax_id;id3_party_in/AVFFNB_Tax_Id_Number;;String
        tnf_party;tax_id_type_code;id3_party_in/AVDRST_Tax_Id_Type;;String
        tnf_party;person_indicator;id3_party_in/AVDQST_Person_Ind;;String
        tnf_party;company_name;;;String
        tnf_party;source_system_id;9;;String
        tnf_party;;(join id3_person_in/BSACSG_Party_Surrogate_Key id3_party_in/AVACSG_Party_Surrogate_Key);;"
      (as-dataflow-req)
      #_(undeploy-dataflow))


  (-> "tnf_party;party_id;tp_stage_producer/Producer_Id;;String
       tnf_party;last_name;tp_stage_producer/LASTNAME;;String
       tnf_party;middle_name;tp_stage_producer/MIDDLENAME;;String
       tnf_party;first_name;tp_stage_producer/FIRSTNAME;;String
       tnf_party;prefix;tp_stage_producer/PREFIX;;String
       tnf_party;suffix;tp_stage_producer/SUFFIX;;String
       tnf_party;short_name;tp_stage_producer/NICKNAME;;String
       tnf_party;gender;;;String
       tnf_party;date_of_birth;tp_stage_producer/BIRTHDAY;;String
       tnf_party;date_of_death;;;String
       tnf_party;tax_id;tp_stage_producer/TAXID;;String
       tnf_party;tax_id_type_code;;;String
       tnf_party;person_indicator;tp_stage_producer/ISPERSON;;String
       tnf_party;company_name;tp_stage_producer/COMPANYNAME;;String
       tnf_party;source_system_id;31;;String
       tnf_party;;(left_join tp_stage_producer/TAXID id3_party_in/AVFFNB_Tax_Id_Number );;
       tnf_party;;(where (= id3_party_in/AVFFNB_Tax_Id_Number NULL));"
      (as-dataflow-req)
      (deploy-dataflow))


  (invoke "drop-stream" "TNF_PARTY")

  (invoke "describe-stream" "TNF_PARTY")

  (invoke "show-streams")

  (ekconn/read-template "app/allianz_life/config/csv_source.json")


  (-> "tnf_party6;party_id;id3_person_in/BSACSG_Party_Surrogate_Key;;String
       tnf_party6;last_name;id3_person_in/BSB9NA_Last_Name;;String
       tnf_party6;middle_name;id3_person_in/BSCCNA_Middle_Name;;String
       tnf_party6;first_name;id3_person_in/BSCBNA_First_Name;;String
       tnf_party6;full_name;(concat (constant 9) id3_person_in/BSCBNA_First_Name id3_person_in/BSCCNA_Middle_Name id3_person_in/BSB9NA_Last_Name)  ;;String"
      (as-dataflow-req)
      (deploy-dataflow))


  (->> (invoke "print-topic" {:topic-name "MY_PARTY4" :total 1})
       (clojure.pprint/print-table))

  )