(ns com.lambdasoftware.conveyor.client
  (:require [com.lambdasoftware.conveyor.protocol :as protocol]
            [taoensso.timbre :as log]
            [gloss.io :as gio]
            [clojure.string :as str]
            [aleph.tcp :as tcp]
            [manifold.deferred :as d]
            [manifold.stream :as s]
            [manifold.time :as t]))

(defn next-tmpid [conn]
  (::tmpid
   (swap! conn update-in [::tmpid] inc)))

(defn next-key [conn]
  (::current-id
   (swap! conn update-in [::current-id] inc)))

(defn transaction? [conn-or-tx]
  (and (associative? conn-or-tx)
       (= ::tx (::type conn-or-tx))))

(defn connected? [conn]
  (and (map? @conn)
       (not (nil? (::client @conn)))))

(defn register-handler
  ([conn type cb] (register-handler conn type cb (constantly true)))
  ([conn type cb pred]
   (swap! conn update-in [::handlers type] conj {:cb cb :pred pred})))

(defn get-tmpid [tx-or-conn]
  (if (transaction? tx-or-conn)
    (::tmpid tx-or-conn)
    0))

(defn send-message [tx-or-conn msg]
  (let [conn (if (transaction? tx-or-conn)
               (do (log/debug (str "Sending message as part of transaction: " (::tmpid tx-or-conn)))
                   (::conn tx-or-conn))
               tx-or-conn)]
    (if (= ::connected (::state @conn))
      (do
        (when (not= :heartbeat (:type msg))
          (log/debug (str "Sending message: " msg)))
        (s/put! (::client @conn)
                (cond-> msg
                  (= :create-event (:type msg)) (assoc :tmpid (get-tmpid tx-or-conn)))))
      (log/warn (str "Attempted to send message over connection in state (" (::state @conn) "): " msg)))))

(defn disconnect
  ([conn] (disconnect conn "Client disconnected normally"))
  ([conn msg]
   (if (= ::connected (::state @conn))
     (let [{:keys [::client ::msg-timeout]} @conn
           ack (d/deferred)
           on-disconnect #(do (d/success! ack ::ack)
                              (log/info (str "Disconnecting with message: " msg))
                              (swap! conn assoc
                                     ::on-disconnect nil
                                     ::state ::disconnecting))]
       (swap! conn assoc ::on-disconnect on-disconnect)
       (send-message conn {:type :client-disconnect
                           :message msg})
       (d/on-realized (d/timeout! ack msg-timeout ::no-ack)
                      (fn [res]
                        (when (= ::no-ack res)
                          (log/warn "Server did not respond to disconnect - closing socket anyway.")
                          (s/close! client)))
                      (fn [err] (log/warn (str "Unexpected error in disconnecting client: " err)))))
     (log/warn (str "Cannot disconnect from connection in non-connected state: " (::state @conn))))))

(defn close-subscription-streams [conn]
  (let [subs (::feed-subscriptions @conn)
        streams (apply concat (vals subs))]
    (doseq [stream streams]
      (s/close! stream))))

(defn disconnect-on-no-ack [conn ack timeout]
  (d/on-realized (d/timeout! ack timeout ::no-ack)
    (fn [res]
      (when (= ::no-ack res)
        (log/warn (format "Expected message not received from server in %d ms - disconnecting" timeout))
        (disconnect conn)))
    (fn [err] (log/warn (str "Unexpected error: " err)))))

(defn heartbeat [conn]
  (let [{:keys [::client ::heartbeat-order ::timeout]} @conn
        ack (d/deferred)]
    (register-handler conn
                      :heartbeat
                      (fn [msg]
                        (when (connected? conn)
                          (swap! conn assoc
                                 ::heartbeat-order (inc heartbeat-order)
                                 ::last-heartbeat (System/currentTimeMillis))
                          (d/success! ack ::ack))))
    (send-message conn {:type :heartbeat
                        :order heartbeat-order})
    (disconnect-on-no-ack conn ack (.intValue (* timeout 0.75)))
    ack))

(defn matching-subscriptions
  "Given a path and a map of path prefixes to subscriptions,
  get all subscriptions who belong to a key that is a prefix
  of the path."
  [path subs-map]
  (apply concat
         (keep (fn [[p subs]]
                 (when (str/starts-with? path p)
                   (log/debug (format "Matched path %s: %s" path p))
                   subs))
               subs-map)))

(defn emit-to-subscribers [conn evt]
  (let [{:keys [feed]} evt
        subs (matching-subscriptions feed (::feed-subscriptions @conn))]
    (when (seq subs)
      (log/debug (str "Emitting message to subscribers: " evt))
      (doseq [sub subs]
        (s/put! sub evt)))))

(defn handle-message [msg conn]
  (let [{:keys [type]} msg]
    (when (not= :heartbeat type)
      (log/debug (str "Received message: " msg)))
    (condp = type
      :server-disconnect (do (log/info (format "Received server disconnect: %s" (:message msg)))
                             (when-let [on-disconnect (::on-disconnect @conn)]
                               (on-disconnect))
                             (s/close! (::client @conn)))
      :event (emit-to-subscribers conn (update-in msg [:data] protocol/unpack-payload))
      :error (log/warn (str "Received error message: " msg)) ;; TODO: allow clients to define an error handler as part of a connection
      nil)
    (doseq [{:keys [cb pred]} (get-in @conn [::handlers type])]
      (when (pred msg)
        (cb msg)
        ;; TODO: See if we need to call the removal code on timeout - factor out
        (swap! conn update-in [::handlers type]
          #(remove (fn [handler]
                     (= cb (:cb handler)))
                   %))))))

(defn connect
  ([] (connect {}))
  ([{:keys [host port timeout msg-timeout]
      :or {host "127.0.0.1"
           port 4455
           timeout 5000
           msg-timeout 10000}
     :as cfg}]
   (log/info (format "Connecting to %s:%d (timeout=%d,msg-timeout=%d)" host port timeout msg-timeout))
   (let [client @(d/chain' (tcp/client {:host host :port port})
                           protocol/wrap-encoded-duplex-stream
                           #(d/timeout! % timeout ::error))]
     (if (= ::error client)
       (do (log/warn "Timed out while attempting to connect")
           (atom {::state ::error}))
       (let [conn (atom {::state ::connected
                         ::heartbeat-order 0
                         ::last-heartbeat 0
                         ::timeout timeout
                         ::msg-timeout msg-timeout
                         ::client client
                         ::handlers {}
                         ::on-disconnect nil
                         ::current-id 0
                         ::tmpid 1
                         ::feed-subscriptions {}})
             cancel-heartbeat (t/every 3000 #(heartbeat conn))]
         (s/consume
          (fn [msg]
            (try
              (handle-message msg conn)
              (catch Throwable e
                (log/warn (str "Caught exception while handling message: " (.getMessage e))))))
          client)
         (s/on-closed client
                      (fn []
                        (cancel-heartbeat)
                        (close-subscription-streams conn)
                        (reset! conn {::state ::closed})
                        (log/info "Socket successfully closed")))
         conn)))))

(defn get-feed
  "Retrieves all of the events that are part of a specific feed.
  Returns a deferred representing the feed's events."
  ([conn path] (get-feed conn path {}))
  ([conn path {:keys [before-ts after-ts
                      before-tx after-tx]
               :or {before-ts 0, after-ts 0
                    before-tx 0, after-tx 0}}]
   {:pre [(= ::connected (::state @conn))]}
   (let [resp (d/deferred)
         k (next-key conn)]
     (register-handler conn :feed
                       (fn [{:keys [events key]}]
                         (when
                           (d/success! resp (map #(update-in % [:data] protocol/unpack-payload)
                                                 events))))
                       #(= k (:key %)))
     (send-message conn {:type :get-feed
                         :feed path
                         :before-ts before-ts
                         :after-ts after-ts
                         :before-tx before-tx
                         :after-tx after-tx
                         :key k})
     (d/timeout! resp (::msg-timeout @conn)))))

(defn create-feed
  "Creates a new feed, given a URL-like path, e.g.: \"/patriot/orders/services\""
  [conn path]
  {:pre [(= ::connected (::state @conn))]}
  (log/info (str "Creating feed: " path))
  (let [resp (d/deferred)
        k (next-key conn)]
    (register-handler conn :feed-created
                      (fn [{:keys [key feed]}]
                        (log/info (str "Feed created: " path))
                        (d/success! resp feed))
                      #(= k (:key %)))
    (send-message conn {:type :create-feed
                        :feed path
                        :key k})
    (d/timeout! resp (::msg-timeout @conn))))

(defn create-event
  "Creates an event at a specific feed path. The path must have been previously created.
  Event is expected to be some arbitrary payload as a raw byte array."
  [tx-or-conn path event]
  (let [conn (if (transaction? tx-or-conn)
                 (::conn tx-or-conn)
                 tx-or-conn)
        resp (d/deferred)
        k (next-key conn)]
    (when (= ::connected (::state @conn))
      (register-handler conn :event-created
                        (fn [msg]
                          (log/debug (str "Event created at " path ": " msg))
                          (d/success! resp [:ok (select-keys msg [:ts :tx-id])]))
                        #(= k (:key %)))
      (send-message conn {:type :create-event
                          :feed path
                          :data (gio/to-byte-buffer event)
                          :key k})
      (d/timeout! resp (::msg-timeout @conn)))))

(defn remove-stream [conn path stream]
  ;{:pre [(= ::connected (::state @conn))]}
  (log/info (str "Removing stream: " path))
  (swap! conn update-in [::feed-subscriptions path]
         (fn [streams]
           (remove #(= % stream) streams))))

(defn subscribe-to-stream
  "Subscribe to all future events in a stream"
  [conn path]
  {:pre [(= ::connected (::state @conn))]}
  (let [paths (keys (::feed-subscriptions conn))]
    (if (not-any? #(clojure.string/starts-with? % path) paths)
      (let [ack (d/deferred)
            k (next-key conn)
            {:keys [::msg-timeout]} @conn]
        (register-handler conn :subscribed-to-feed
                          (fn [msg]
                            (log/debug (str "Subscribed to feed: " path))
                            (d/success! ack :ok))
                          #(= k (:key %)))
        (send-message conn {:type :subscribe-feed
                            :feed path
                            :key k})
        (d/on-realized (d/timeout! ack msg-timeout ::no-ack)
                       (fn [res]
                         (when (= ::no-ack res)
                           (log/warn (str "Could not subscribe to feed due to timeout: " path))))
                       (fn [err] (log/warn (format "Unexpected error when trying to subscribe to feed, \"%s\": %s" path (str err))))))
      (log/debug (format "Already subscribed to %s or upstream. Not re-subscribing" path)))))

(defn event-stream
  "Obtain a stream of all events that are emitted for a certain path and any sub-paths."
  [conn path]
  {:pre [(= ::connected (::state @conn))]}
  ;; TODO: When there does not exist a subscription to this stream or any higher up, subscribe
  (let [s (s/stream)]
    (s/on-closed s #(remove-stream conn path s))
    (subscribe-to-stream conn path)
    (swap! conn update-in [::feed-subscriptions path] conj s)
    s))

(defmacro with-transaction
  "Wrap a number of statements in a single transaction that will succeed or fail together.
  When anything in the body of the transaction throws an exception that is not caught,
  the transaciton will be rolled back. Otherwise, it will be committed at the close of
  the with-transaction expression.

  Example:
    (with-transaction [tx conn]
      (create-event tx \"/path/to/stream\" {:event-num 1})
      (something-that-may-throw!)
      (create-event tx \"/path/to/stream\" {:event-num 2})
      (create-event tx \"/path/to/stream\" {:event-num 3}))"
  [binding & body]
  `(let [conn# ~(second binding)
         tmpid# (next-tmpid conn#)
         ~(first binding) {::type ::tx
                           ::tmpid tmpid#
                           ::conn conn#}]
     (send-message conn# {:type :begin-tx
                          :tmpid tmpid#
                          :key (next-key conn#)})
     (try
       (do ~@body)
       (send-message conn# {:type :commit-tx
                            :tmpid tmpid#
                            :key (next-key conn#)})
       (catch Exception exc#
         (log/warn (str "Error executing transaction: " (.getMessage exc#)))
         (.printStackTrace exc#)
         (send-message conn# {:type :rollback-tx
                              :tmpid tmpid#
                              :key (next-key conn#)})))))
