(ns com.lambdasoftware.conveyor.event-sourcing
  "Event sourcing convenience functions"
  (:require [com.lambdasoftware.conveyor.client :as client]
            [taoensso.timbre :as log]
            [clojure.string :as str]
            [taoensso.nippy :as nippy]
            [manifold.deferred :as d]
            [manifold.stream :as s]
            [manifold.bus :as bus]))

(defn queriable? [m]
  (and (map? m)
       (every? #(contains? m %) [:state :queries :type])))

(defn fresh-aggregate [aggr domain events]
  (assoc aggr
         :domain domain
         :state {}
         :events (merge {:new [] :saved []}
                        events)))

(defn fresh-read-model [model domain]
  (assoc model :domain domain))

(defn domain [conn domain-name]
  (let [parts (str/split domain-name #"[.]")
        path (str "/" (str/join "/" parts))
        d (atom {::conn conn
                 ::path path
                 ::queriable {}
                 ::read-model-bus (bus/event-bus)})]
    (client/create-feed conn path)
    d))

(defn register-aggregate [domain agg-key aggregate]
  (swap! domain assoc-in [::queriable agg-key] (assoc aggregate
                                                      :type ::aggregate
                                                      :key agg-key))
  (log/info (str "Registered aggregate: " (name agg-key)))
  domain)

(defn on-save
  "Register a function to be called whenever an aggregate is saved.
  
  Params:
    handler - Function to be called when aggregate is saved: (handler agg-key agg-state)

  Return: Domain with handler assoc'd in
  "
  [domain handler]
  (swap! domain assoc ::on-save handler)
  domain)

(defn notify-read-model-subscribers [domain rm-key state data]
  (bus/publish! (::read-model-bus @domain) rm-key [state data]))

(defn read-model-handler [domain rm-key read-model handlers state]
  (fn [event]
    (try
      (let [data (nippy/thaw (:data event))
            {:keys [set-id]} read-model]
        (set-id state (:tx-id event))
        (if-let [handler (get handlers (:type data))]
          (do (handler state data)
              (notify-read-model-subscribers domain rm-key state data))
          (when-let [default-handler (:_default handlers)]
            (default-handler state data))))
      (catch Throwable e
        (log/warn (str "Error handling event in read model: " (.getMessage e)))
        (.printStackTrace e)))))

(defn register-read-model
  "Registers a read model to receive events for any of its subscriptions.

  Details:
     1. Gets event stream as of the present
     2. Request all events prior to the latest tx-id processed by the read model
     3. Once all events have been received, connects the subscription stream to the handler,
        filtering ids that are less than or equal to the latest transaction-id received
        from the initial batch. This ensures exactly-once processing.

  Arguments:
    domain: Domain
    rm-key: Keyword used to identify the read model
    read-model: Map conforming to the read-model spec (see examples)
    state: Any object to be passed as a second parameter to every function
           registered on the read model"
  [domain rm-key read-model state]
  {:pre [(nil? (get-in @domain [::queriable rm-key]))]}
  (let [{:keys [::conn ::path]} @domain
        {:keys [last-id subscriptions]} read-model
        tx-id (last-id state)
        out-str (s/stream)]
    (log/info (str "Initialized read model, " rm-key ", from tx-id " tx-id))
    (doseq [[stream handlers] subscriptions
            :let [stream-path (str path "/" stream)]]
      (client/create-feed conn stream-path)
      (let [event-stream (client/event-stream conn stream-path)]
        ;; Put prior messages on stream
        (d/chain (client/get-feed conn stream-path {:after-tx tx-id})
                 #(s/put-all! out-str %)
                 (fn [success?]
                   (if success?
                     (do (log/info "Wrote historical events to stream.")
                         (s/connect
                          (s/filter #(> (:tx-id %) tx-id) event-stream)
                          out-str))
                     (do (log/error (format "Error writing historical data - not subscribing to current! (closed=%b,drained=%b)"
                                        (s/closed? out-str)
                                        (s/drained? out-str)))
                         (s/close! event-stream)
                         (s/close! out-str)))))
        (s/consume (read-model-handler domain rm-key read-model handlers state)
                   out-str)))
    (swap! domain update-in [::queriable] assoc rm-key (assoc read-model
                                                              :type ::read-model
                                                              :state state)))
  (log/info (str "Registered read model: " (name rm-key)))
  domain)

(defn subscribe-to-read-model
  "Subscribes to all messages successfully handled by a given read model. All
  handled messages are forwarded on to `sink` after they have been handled by
  the read model. This allows applications to, e.g. build a real-time update
  mechanism by pushing the most up to date information to clients that are
  querying a specific read model.

  Returns a sink that will receive events after they have been handled by a read
  model."
  [domain rm-key]
  (bus/subscribe (::read-model-bus @domain) rm-key))

(defn handle-event [agg evt]
  (let [{:keys [state handlers]} agg
        {event-type :type} evt]
    (if-let [handler (get handlers event-type)]
      (update-in agg [:state] #(handler % evt))
      (do (log/error (str "Cannot handle event type: " (or event-type "Unspecified")))
          agg))))

(defn create [domain ak data]
  {:pre [(= ::aggregate (get-in @domain [::queriable ak :type]))]}
  (let [agg (get-in @domain [::queriable ak])
        init-type (get agg :init-event :initialized)
        uuid (str (java.util.UUID/randomUUID))
        init-evt (assoc data :id uuid :type init-type)]
    (-> (fresh-aggregate agg domain {:new [init-evt]})
        (handle-event init-evt))))

(defn materialize
  "Retrieve the full state of an aggregate, including all computed properties.
  Computed properties are declared under a :computed key on the aggregate and
  essentially function as no-arg queries.  

  Example:
  ;; Aggregate that has been registered on some domain
  {:name \"Counter\"
   :init-event :counter/init
   :handlers {:counter/init (constantly {:count 0})
              :counter/incremented (fn [state _] (update state :count inc))}
   :computed {:is-large? #(> (:count %) 10)}}

  (materialize agg)
  ;; => {:count 13 :is-large? true}
  "
  [agg]
  (reduce (fn [state [prop getter]]
            (assoc state prop (getter state)))
          (:state agg)
          (get agg :computed [])))

(defn- query* [queries query params state]
  (when-let [query-fn (get queries query)]
    (apply query-fn (conj params state))))

(defn query [queriable query & params]
  {:pre [(queriable? queriable)]}
  (let [{:keys [state queries type]} queriable]
    (if (= ::aggregate type)
      (condp = query
        :_id (:id state)
        :_prop (get state (first params))
        :_state (materialize queriable)
        (query* queries query params state))
      (query* queries query params state))))

(defn cmd [{:keys [state commands handlers] :as agg} c & params]
  (if-let [cmd-fn (get commands c)]
    (let [events (apply cmd-fn (conj params state))]
      (reduce handle-event
              (update-in agg [:events :new] #(into [] (concat % events)))
              events))))

(defn save [{:keys [events domain key name] :as agg}]
  (let [{:keys [::conn ::path ::on-save]} @domain
        id (get-in agg [:state :id])
        agg-path (str path "/" name "/" id)]
    (client/create-feed conn agg-path)
    (client/with-transaction [tx conn]
      (doseq [new-event (:new events)]
        (client/create-event tx agg-path (nippy/freeze new-event))))
    (when on-save
      (future (on-save domain key (materialize agg))))
    (-> agg
        (assoc-in [:events :new] [])
        (update-in [:events :saved] #(into [] (concat % (:new events)))))))

(defn fetch [domain ak id]
  {:pre [(= ::aggregate (get-in @domain [::queriable ak :type]))]}
  (let [{:keys [::path ::conn ::queriable]} @domain
        aggr (get queriable ak)
        agg-path (str path "/" (:name aggr) "/" id)]
    (d/chain
     (client/get-feed conn agg-path)
     (fn [events]
       (let [events (map #(-> % :data nippy/thaw) events)]
         (reduce handle-event
                 (fresh-aggregate aggr domain {:saved events})
                 events))))))

(defn read-model
  "Retrieve a read model from the domain that may subsequently be queried"
  [domain rm-key]
  {:pre [(= ::read-model (get-in @domain [::queriable rm-key :type]))]}
  (let [model (get-in @domain [::queriable rm-key])]
    (fresh-read-model model domain)))

(defn simple-setter
  "Returns a function that copies an attribute from the event
  to the aggregate state."
  [attr]
  (fn [state evt]
    (assoc state attr (get evt attr))))
