(ns cellophane.om-tutorials-test
  (:refer-clojure :exclude [read])
  (:require [clojure.test :refer [deftest testing is are]]
            [cellophane.test-utils :refer [remove-whitespace]]
            [cellophane.next :as cellophane :refer [defui]]
            [cellophane.dom :as dom]))

;; =============================================================================
;; Quick Start

(def animals-app-state
  (atom
    {:app/title "Animals"
     :animals/list
     [[1 "Ant"] [2 "Antelope"] [3 "Bird"] [4 "Cat"] [5 "Dog"]
      [6 "Lion"] [7 "Mouse"] [8 "Monkey"] [9 "Snake"] [10 "Zebra"]]}))

(defmulti animals-read (fn [env key params] key))

(defmethod animals-read :default
  [{:keys [state] :as env} key params]
  (let [st @state]
    (if-let [[_ value] (find st key)]
      {:value value}
      {:value :not-found})))

(defmethod animals-read :animals/list
  [{:keys [state] :as env} key {:keys [start end]}]
  {:value (subvec (:animals/list @state) start end)})

(defui AnimalsList
  static cellophane/IQueryParams
  (params [this]
    {:start 0 :end 10})
  static cellophane/IQuery
  (query [this]
    '[:app/title (:animals/list {:start ?start :end ?end})])
  Object
  (render [this]
    (let [{:keys [app/title animals/list]} (cellophane/props this)]
      (dom/div nil
        (dom/h2 nil title)
        (apply dom/ul nil
          (map
            (fn [[i name]]
              (dom/li nil (str i ". " name)))
            list))))))

(def animals-reconciler
  (cellophane/reconciler
    {:state animals-app-state
     :parser (cellophane/parser {:read animals-read})}))

(deftest test-render-animals-tutorial
  (let [result-markup (remove-whitespace
                        "<div data-reactroot=\"\" data-reactid=\"1\">
                           <h2 data-reactid=\"2\">Animals</h2>
                           <ul data-reactid=\"3\">
                             <li data-reactid=\"4\">1. Ant</li>
                             <li data-reactid=\"5\">2. Antelope</li>
                             <li data-reactid=\"6\">3. Bird</li>
                             <li data-reactid=\"7\">4. Cat</li>
                             <li data-reactid=\"8\">5. Dog</li>
                             <li data-reactid=\"9\">6. Lion</li>
                             <li data-reactid=\"10\">7. Mouse</li>
                             <li data-reactid=\"11\">8. Monkey</li>
                             <li data-reactid=\"12\">9. Snake</li>
                             <li data-reactid=\"13\">10. Zebra</li>
                           </ul>
                         </div>")]
    (testing "render with factory"
      (let [ctor (cellophane/factory AnimalsList)]
        (is (= (str (#'dom/render-to-str* (ctor @animals-app-state))) result-markup))))
    (testing "render with reconciler & add-root!"
      (let [c (cellophane/add-root! animals-reconciler AnimalsList nil)
            markup-str (str (#'dom/render-to-str* c))]
        (is (= (class (cellophane/app-root animals-reconciler)) AnimalsList))
        (is (= markup-str result-markup))))))

;; =============================================================================
;; Om Links tutorial

(def links-init-data
  {:current-user {:email "bob.smith@gmail.com"}
   :items [{:id 0 :title "Foo"}
           {:id 1 :title "Bar"}
           {:id 2 :title "Baz"}]})

(defmulti links-read cellophane/dispatch)

(defmethod links-read :items
  [{:keys [query state]} k _]
  (let [st @state]
    {:value (cellophane/db->tree query (get st k) st)}))

(defui LinksItem
  static cellophane/Ident
  (ident [_ {:keys [id]}]
    [:item/by-id id])
  static cellophane/IQuery
  (query [_]
    '[:id :title [:current-user _]])
  Object
  (render [this]
    (let [{:keys [title current-user]} (cellophane/props this)]
      (dom/li nil
        (dom/div nil title)
        (dom/div nil (:email current-user))))))

(def links-item (cellophane/factory LinksItem))

(defui LinksSomeList
  static cellophane/IQuery
  (query [_]
    [{:items (cellophane/get-query LinksItem)}])
  Object
  (render [this]
    (dom/div nil
      (dom/h2 nil "A List!")
      (dom/ul nil
        (map links-item (-> this cellophane/props :items))))))

(def links-reconciler
  (cellophane/reconciler
    {:state links-init-data
     :parser (cellophane/parser {:read links-read})}))

(deftest test-render-links-tutorial
  (let [c (cellophane/add-root! links-reconciler LinksSomeList nil)]
    (is (= (str (#'dom/render-to-str* c))
           (remove-whitespace
             "<div data-reactroot=\"\" data-reactid=\"1\">
                <h2 data-reactid=\"2\">A List!</h2>
                <ul data-reactid=\"3\">
                  <li data-reactid=\"4\">
                    <div data-reactid=\"5\">Foo</div>
                    <div data-reactid=\"6\">bob.smith@gmail.com</div>
                  </li>
                  <li data-reactid=\"7\">
                    <div data-reactid=\"8\">Bar</div>
                    <div data-reactid=\"9\">bob.smith@gmail.com</div>
                  </li>
                  <li data-reactid=\"10\">
                    <div data-reactid=\"11\">Baz</div>
                    <div data-reactid=\"12\">bob.smith@gmail.com</div>
                  </li>
                </ul>
              </div>")))))

;; =============================================================================
;; Componentes, Identity & Normalization

(def cian-init-data
  {:list/one [{:name "John" :points 0}
              {:name "Mary" :points 0}
              {:name "Bob"  :points 0}]
   :list/two [{:name "Mary" :points 0 :age 27}
              {:name "Gwen" :points 0}
              {:name "Jeff" :points 0}]})

;; -----------------------------------------------------------------------------
;; Parsing

(defmulti cian-read cellophane/dispatch)

(defn get-people [state key]
  (let [st @state]
    (into [] (map #(get-in st %)) (get st key))))

(defmethod cian-read :list/one
  [{:keys [state] :as env} key params]
  {:value (get-people state key)})

(defmethod cian-read :list/two
  [{:keys [state] :as env} key params]
  {:value (get-people state key)})

(defmulti cian-mutate cellophane/dispatch)

(defmethod cian-mutate 'points/increment
  [{:keys [state]} _ {:keys [name]}]
  {:action
   (fn []
     (swap! state update-in
       [:person/by-name name :points]
       inc))})

(defmethod cian-mutate 'points/decrement
  [{:keys [state]} _ {:keys [name]}]
  {:action
   (fn []
     (swap! state update-in
       [:person/by-name name :points]
       #(let [n (dec %)] (if (neg? n) 0 n))))})

;; -----------------------------------------------------------------------------
;; Components

(defui Person
  static cellophane/Ident
  (ident [this {:keys [name]}]
    [:person/by-name name])
  static cellophane/IQuery
  (query [this]
    '[:name :points :age])
  Object
  (render [this]
    (println "Render Person" (-> this cellophane/props :name))
    (let [{:keys [points name foo] :as props} (cellophane/props this)]
      (dom/li nil
        (dom/label nil (str name ", points: " points))
        (dom/button
          #js {:onClick
               (fn [e]
                 (cellophane/transact! this
                   `[(points/increment ~props)]))}
          "+")
        (dom/button
          #js {:onClick
               (fn [e]
                 (cellophane/transact! this
                   `[(points/decrement ~props)]))}
          "-")))))

(def person (cellophane/factory Person {:keyfn :name}))

(defui ListView
  Object
  (render [this]
    ;(println "Render ListView" (-> this cellophane/path first))
    (let [list (cellophane/props this)]
      (apply dom/ul nil
        (map person list)))))

(def list-view (cellophane/factory ListView))

(defui RootView
  static cellophane/IQuery
  (query [this]
    (let [subquery (cellophane/get-query Person)]
      `[{:list/one ~subquery} {:list/two ~subquery}]))
  Object
  (render [this]
    (println "Render RootView")
    (let [{:keys [list/one list/two]} (cellophane/props this)]
      (apply dom/div nil
        [(dom/h2 nil "List A")
         (list-view one)
         (dom/h2 nil "List B")
         (list-view two)]))))

(def cian-reconciler
  (cellophane/reconciler
    {:state  cian-init-data
     :parser (cellophane/parser {:read cian-read :mutate cian-mutate})}))

(deftest test-cian-tutorial
  (let [c (cellophane/add-root! cian-reconciler RootView nil)]
    (is (= (str (#'dom/render-to-str* c))
           (remove-whitespace "<div data-reactroot=\"\" data-reactid=\"1\">
                                 <h2 data-reactid=\"2\">List A</h2>
                                 <ul data-reactid=\"3\">
                                   <li data-reactid=\"4\">
                                     <label data-reactid=\"5\">John, points: 0</label>
                                     <button data-reactid=\"6\">+</button>
                                     <button data-reactid=\"7\">-</button>
                                   </li>
                                   <li data-reactid=\"8\">
                                     <label data-reactid=\"9\">Mary, points: 0</label>
                                     <button data-reactid=\"10\">+</button>
                                     <button data-reactid=\"11\">-</button>
                                   </li>
                                   <li data-reactid=\"12\">
                                     <label data-reactid=\"13\">Bob, points: 0</label>
                                     <button data-reactid=\"14\">+</button>
                                     <button data-reactid=\"15\">-</button>
                                   </li>
                                 </ul>
                                 <h2 data-reactid=\"16\">List B</h2>
                                 <ul data-reactid=\"17\">
                                   <li data-reactid=\"18\">
                                     <label data-reactid=\"19\">Mary, points: 0</label>
                                     <button data-reactid=\"20\">+</button>
                                     <button data-reactid=\"21\">-</button>
                                   </li>
                                   <li data-reactid=\"22\">
                                     <label data-reactid=\"23\">Gwen, points: 0</label>
                                     <button data-reactid=\"24\">+</button>
                                     <button data-reactid=\"25\">-</button>
                                   </li>
                                   <li data-reactid=\"26\">
                                     <label data-reactid=\"27\">Jeff, points: 0</label>
                                     <button data-reactid=\"28\">+</button>
                                     <button data-reactid=\"29\">-</button>
                                   </li>
                                 </ul>
                               </div>")))))

;; =============================================================================
;; Queries with unions

(def union-init-data
  {:dashboard/items
   [{:id 0 :type :dashboard/post
     :author "Laura Smith"
     :title "A Post!"
     :content "Lorem ipsum dolor sit amet, quem atomorum te quo"
     :favorites 0}
    {:id 1 :type :dashboard/photo
     :title "A Photo!"
     :image "photo.jpg"
     :caption "Lorem ipsum"
     :favorites 0}
    {:id 2 :type :dashboard/post
     :author "Jim Jacobs"
     :title "Another Post!"
     :content "Lorem ipsum dolor sit amet, quem atomorum te quo"
     :favorites 0}
    {:id 3 :type :dashboard/graphic
     :title "Charts and Stufff!"
     :image "chart.jpg"
     :favorites 0}
    {:id 4 :type :dashboard/post
     :author "May Fields"
     :title "Yet Another Post!"
     :content "Lorem ipsum dolor sit amet, quem atomorum te quo"
     :favorites 0}]})

(defui Post
  static cellophane/IQuery
  (query [this]
    [:id :type :title :author :content])
  Object
  (render [this]
    (let [{:keys [title author content] :as props} (cellophane/props this)]
      (dom/div nil
        (dom/h3 nil title)
        (dom/h4 nil author)
        (dom/p nil content)))))

(def post (cellophane/factory Post))

(defui Photo
  static cellophane/IQuery
  (query [this]
    [:id :type :title :image :caption])
  Object
  (render [this]
    (let [{:keys [title image caption]} (cellophane/props this)]
      (dom/div nil
        (dom/h3 nil (str "Photo: " title))
        (dom/div nil image)
        (dom/p nil "Caption: ")))))

(def photo (cellophane/factory Photo))

(defui Graphic
  static cellophane/IQuery
  (query [this]
    [:id :type :title :image])
  Object
  (render [this]
    (let [{:keys [title image]} (cellophane/props this)]
      (dom/div nil
        (dom/h3 nil (str "Graphic: " title))
        (dom/div nil image)))))

(def graphic (cellophane/factory Graphic))

(defui DashboardItem
  static cellophane/Ident
  (ident [this {:keys [id type]}]
    [type id])
  static cellophane/IQuery
  (query [this]
    (zipmap
      [:dashboard/post :dashboard/photo :dashboard/graphic]
      (map #(conj % :favorites)
        [(cellophane/get-query Post)
         (cellophane/get-query Photo)
         (cellophane/get-query Graphic)])))
  Object
  (render [this]
    (let [{:keys [id type favorites] :as props} (cellophane/props this)]
      (dom/li
        #js {:style #js {:padding 10 :borderBottom "1px solid black"}}
        (dom/div nil
          (({:dashboard/post    post
             :dashboard/photo   photo
             :dashboard/graphic graphic} type)
            (cellophane/props this)))
        (dom/div nil
          (dom/p nil (str "Favorites: " favorites))
          (dom/button
            #js {:onClick
                 (fn [e]
                   (cellophane/transact! this
                     `[(dashboard/favorite {:ref [~type ~id]})]))}
            "Favorite!"))))))

(def dashboard-item (cellophane/factory DashboardItem))

(defui Dashboard
  static cellophane/IQuery
  (query [this]
    [{:dashboard/items (cellophane/get-query DashboardItem)}])
  Object
  (render [this]
    (let [{:keys [dashboard/items]} (cellophane/props this)]
      (apply dom/ul
        #js {:style #js {:padding 0}}
        (map dashboard-item items)))))

(defmulti union-read cellophane/dispatch)

(defmethod union-read :dashboard/items
  [{:keys [state]} k _]
  (let [st @state]
    {:value (into [] (map #(get-in st %)) (get st k))}))

(defmulti mutate cellophane/dispatch)

(defmethod mutate 'dashboard/favorite
  [{:keys [state]} k {:keys [ref]}]
  {:action
   (fn []
     (swap! state update-in (conj ref :favorites) inc))})

(def union-reconciler
  (cellophane/reconciler
    {:state  union-init-data
     :parser (cellophane/parser {:read union-read :mutate mutate})}))


(deftest test-unions-tutorial
    (let [c (cellophane/add-root! union-reconciler Dashboard nil)]
      (is (= (str (#'dom/render-to-str* c))
             (remove-whitespace
               "<ul style=\"padding:0;\" data-reactroot=\"\" data-reactid=\"1\">
                 <li style=\"padding:10px;border-bottom:1px solid black;\" data-reactid=\"2\">
                   <div data-reactid=\"3\">
                     <div data-reactid=\"4\">
                       <h3 data-reactid=\"5\">A Post!</h3>
                       <h4 data-reactid=\"6\">Laura Smith</h4>
                       <p data-reactid=\"7\">Lorem ipsum dolor sit amet, quem atomorum te quo</p>
                     </div>
                   </div>
                   <div data-reactid=\"8\">
                     <p data-reactid=\"9\">Favorites: 0</p>
                     <button data-reactid=\"10\">Favorite!</button>
                   </div>
                 </li>
                 <li style=\"padding:10px;border-bottom:1px solid black;\" data-reactid=\"11\">
                   <div data-reactid=\"12\">
                     <div data-reactid=\"13\">
                       <h3 data-reactid=\"14\">Photo: A Photo!</h3>
                       <div data-reactid=\"15\">photo.jpg</div>
                       <p data-reactid=\"16\">Caption: </p>
                     </div>
                   </div>
                   <div data-reactid=\"17\">
                     <p data-reactid=\"18\">Favorites: 0</p>
                     <button data-reactid=\"19\">Favorite!</button>
                   </div>
                 </li>
                 <li style=\"padding:10px;border-bottom:1px solid black;\" data-reactid=\"20\">
                   <div data-reactid=\"21\">
                     <div data-reactid=\"22\">
                       <h3 data-reactid=\"23\">Another Post!</h3>
                       <h4 data-reactid=\"24\">Jim Jacobs</h4>
                       <p data-reactid=\"25\">Lorem ipsum dolor sit amet, quem atomorum te quo</p>
                     </div>
                   </div>
                   <div data-reactid=\"26\">
                     <p data-reactid=\"27\">Favorites: 0</p>
                     <button data-reactid=\"28\">Favorite!</button>
                   </div>
                 </li>
                 <li style=\"padding:10px;border-bottom:1px solid black;\" data-reactid=\"29\">
                   <div data-reactid=\"30\">
                     <div data-reactid=\"31\">
                       <h3 data-reactid=\"32\">Graphic: Charts and Stufff!</h3>
                       <div data-reactid=\"33\">chart.jpg</div>
                     </div>
                   </div>
                   <div data-reactid=\"34\">
                     <p data-reactid=\"35\">Favorites: 0</p>
                     <button data-reactid=\"36\">Favorite!</button>
                   </div>
                 </li>
                 <li style=\"padding:10px;border-bottom:1px solid black;\" data-reactid=\"37\">
                   <div data-reactid=\"38\">
                     <div data-reactid=\"39\">
                       <h3 data-reactid=\"40\">Yet Another Post!</h3>
                       <h4 data-reactid=\"41\">May Fields</h4>
                       <p data-reactid=\"42\">Lorem ipsum dolor sit amet, quem atomorum te quo</p>
                     </div>
                   </div>
                   <div data-reactid=\"43\">
                     <p data-reactid=\"44\">Favorites: 0</p>
                     <button data-reactid=\"45\">Favorite!</button>
                   </div>
                 </li>
               </ul>")))))

