(ns com.kurogitsune.logicadb.core
  (:require [clojure.java.jdbc :as j]
            [honeysql.core :as sql]
            [honeysql.helpers :refer :all]
            [pandect.algo.sha3-512 :as pd]
            [com.kurogitsune.logica.core :as lg]
            [clojurewerkz.serialism.core :as s]
            [clojure.core.match :refer [match]]))

(defn connected-db [path]
  {:classname   "org.sqlite.JDBC"
   :subprotocol "sqlite"
   :subname    	path
   })
   
(defn create-logic-ref [db table-name]
  (try (j/db-do-commands db
    (j/create-table-ddl table-name
      [:hash :binary]))
    (catch Exception e (println e))))
    
(defn create-logic-base [db table-name]
  (try (j/db-do-commands db
    (j/create-table-ddl table-name
      [:hash :binary]
      [:orderLogic :integer]
      [:type :text]
      [:name :text]
      [:param :text]
			[:eval :text]
      [:selfestablished :text]
      [:x0 :binary]
      [:x1 :binary]
      [:x2 :binary]
      [:x3 :binary]
      [:x4 :binary]))
    (catch Exception e (println e))))
   
(defn insert-unique! [db table data] 
  (let [dup (j/query db (sql/format (sql/build :select :hash :from table :where [:= :hash (:hash data)])))]
    (if (empty? dup) (j/insert! db table data))))
    
(defn serialized [l] (s/serialize l :json))
(defn deserialized [n] (s/deserialize n :json))

(defn safe-nth [x n] (try (nth x n) (catch Exception e nil)))
(defn nth-pval [p n] (safe-nth (vec (:args p)) (* 2 n)))

(defn dbhash [x] (pd/sha3-512-bytes (serialized x)))

(defn insertToDB 
  [db table l] 
  (defn predicateOrFuncToDB [db table l]
    (let [xs (filter some? [(nth-pval l 0) (nth-pval l 1) (nth-pval l 2) (nth-pval l 3) (nth-pval l 4)])]
      (let [xhs (map (fn [x] (dbhash (second x))) xs)]
        (let [xh0 (safe-nth xhs 0) xh1 (safe-nth xhs 1) xh2 (safe-nth xhs 2) xh3 (safe-nth xhs 3) xh4 (safe-nth xhs 4)]
          (let [p (dissoc (assoc (into {} l) :orderLogic (:order l) :hash (dbhash l) :x0 xh0 :x1 xh1 :x2 xh2 :x3 xh3 :x4 xh4) :args :order )] 
            (insert-unique! db table p)
            (doseq [x xs] (insertToDB db table (second x))))))))
  (match [l] 
    [{:type "predicate"}] (predicateOrFuncToDB db table l)
    [{:type "variable"}]
      (let [v (dissoc (assoc (into {} l) :hash (dbhash l) :orderLogic (:order l)) :order)]
        (insert-unique! db table v))
    [{:type "object"}]
      (let [o (assoc (into {} l) :hash (dbhash l))]
        (insert-unique! db table o))   
    [{:type "func"}] (predicateOrFuncToDB db table l)
    [{:type "and" :a a :b b}]
      (let [an (dissoc (assoc (into {} l) :hash (dbhash l) :x0 (dbhash a) :x1 (dbhash b)) :a :b)]
        (insert-unique! db table an)
        (insertToDB db table a)
        (insertToDB db table b))
    [{:type "or" :a a :b b}]
      (let [o (dissoc (assoc (into {} l) :hash (dbhash l) :x0 (dbhash a) :x1 (dbhash b)) :a :b)]
        (insert-unique! db table o)
        (insertToDB db table a)
        (insertToDB db table b))
    [{:type "not" :a a}]
      (let [n (dissoc (assoc (into {} l) :hash (dbhash l) :x0 (dbhash a)) :a)]
        (insert-unique! db table n)
        (insertToDB db table a))
    [{:type "imp" :a a :b b}]
      (let [i (dissoc (assoc (into {} l) :hash (dbhash l) :x0 (dbhash a) :x1 (dbhash b)) :a :b)]
        (insert-unique! db table i)
        (insertToDB db table a)
        (insertToDB db table b))
    [{:type "all" :x x :px px}]
      (let [a (dissoc (assoc (into {} l) :hash (dbhash l) :param x :x0 (dbhash px)) :x :px)]
        (insert-unique! db table a)
        (insertToDB db table px))
    [{:type "exists" :x x :px px}]
      (let [e (dissoc (assoc (into {} l) :hash (dbhash l) :param x :x0 (dbhash px)) :x :px)]
        (insert-unique! db table e)
        (insertToDB db table px))
    [_] nil
    ))
    
(defn selectFromHash 
  [db table hash]
  (let [q (sql/build :select :* :from table :where [:= :hash hash])]
    (j/query db (sql/format q))))

(defn selectOneFromHash [db table hash] (first (selectFromHash db table hash)))

(defn recoverFromDB 
  [db table ldb]
  (defn recoverArgsFromDB [db table ldb hashes]
    (let [xs (filter some? (not-empty (map (fn [h] (selectOneFromHash db :logic h)) hashes)))]
      (let [values (map (fn [x] (recoverFromDB db table x)) xs)]
        (let [args (into {} (map-indexed (fn [i x] [(keyword (str "x" i)) x]) values))] args))))
  (match [ldb]
    [{:type "predicate" :orderlogic order :name name :x0 xh0 :x1 xh1 :x2 xh2 :x3 xh3 :x4 xh4}]
      (lg/predicateOf order name (recoverArgsFromDB db table ldb [xh0 xh1 xh2 xh3 xh4]))
    [{:type "variable" :orderlogic order :name name}] (lg/variableOf order name)
    [{:type "object" :name name}] (lg/objectOf name) 
    [{:type "predicate" :orderlogic order :name name :eval e :x0 xh0 :x1 xh1 :x2 xh2 :x3 xh3 :x4 xh4 :selfestablished f}]
      (lg/funcOf order name e (assoc (recoverArgsFromDB db table ldb [xh0 xh1 xh2 xh3 xh4]) :selfestablished f))
    [{:type "and" :x0 a :x1 b}] 
      (lg/andOf (recoverFromDB db table (selectOneFromHash db :logic a)) (recoverFromDB db table (selectOneFromHash db :logic b)))
    [{:type "or" :x0 a :x1 b}] 
      (lg/impOf (recoverFromDB db table (selectOneFromHash db :logic a)) (recoverFromDB db table (selectOneFromHash db :logic b)))
    [{:type "not" :x0 a}] 
      (lg/notOf (recoverFromDB db table (selectOneFromHash db :logic a)))
    [{:type "imp" :x0 a :x1 b}] 
      (lg/impOf (recoverFromDB db table (selectOneFromHash db :logic a)) (recoverFromDB db table (selectOneFromHash db :logic b)))
    [{:type "all" :param x :x0 px}] 
      (lg/allOf x (recoverFromDB db table (selectOneFromHash db :logic px)))
    [{:type "exists" :param x :x0 px}] 
      (lg/existsOf x (recoverFromDB db table (selectOneFromHash db :logic px)))
    [_] nil
  ))

(defn addLogicDB [db table l] 
  (insertToDB db :logic l)
  (insert-unique! db table {:hash (dbhash l)}))
(defn removeLogicDB [db table l] 
  (let [q (sql/build :delete-from table :where [:= :hash (dbhash l)])]
    (j/execute! db (sql/format q))))
(defn addLogicsDB [db table ls] (doseq [l ls] (addLogicDB db table l)))
(defn removeLogicsDB [db table ls] (doseq [l ls] (removeLogicDB db table l)))

(defn getLogicsDB [db table] 
    (let [qhash (sql/build :select :* :from table)]
      (let [hashes (j/query db (sql/format qhash))]
        (let [qxs (map (fn [hash] (sql/build :select :* :from :logic :where [:= :hash (:hash hash)])) hashes)]
          (let [xs (flatten (map (fn [qx] (j/query db (sql/format qx))) qxs))]
           (map (fn [x] (recoverFromDB db :logic x)) xs))))))

(defn executed [request if-error] 
	(binding [*ns* (find-ns 'com.kurogitsune.logicadb.core)] 
		(try (eval (read-string request)) (catch Exception e (let [] (println e) if-error)))))

(defn xkv-folded [xkvs]
  (first (flatten (flatten xkvs))))

(defn xkv-replaced-with [logic lcs xkv]
  (match [logic]
    ;; {:x0 "x0"} && {:x0 (variableOf "x0")} && {:x0 foo} -> {:x0 [foo]} 
    [{:args args}] (xkv-folded (map (fn [a] (map (fn [kv] (if (= (keyword (:name (second a))) (first kv)) {(first kv) (map (fn [lc] ((first kv) (:args lc))) lcs)}) ) xkv ))args))
    [_] xkv
  ))
(defn xkv-input [logic xkv]
  (reduce (fn [l kv] 
    (match [l (first (second kv))] ;; @todo fix hack
      [{:args args} {:type t}] (assoc l :args (assoc args (first kv) (first (second kv))))  
      [_ _] l
    )) logic xkv))
(defn perm [f as bs] (flatten (map (fn [a] (map (fn [b] (f a b)) bs)) as)))

(defn self-established [logic]
  (and (some? (:selfestablished logic)) ((executed (:selfestablished logic) (fn [x] false)) logic)))

(defn findConstVerOf [logics logic xkv]
  ;; 前後関係を考慮したい。p(x)かつ[x<10](x)ならば、など、pのxが後ろにかかる。
  (match [(xkv-input logic xkv)]
    [{:type "and" :a a :b b}] (perm (fn [a1 b1] (lg/andOf a1 b1)) (findConstVerOf logics a xkv) (findConstVerOf logics b (xkv-replaced-with a (findConstVerOf logics a xkv) xkv)))
    [l1] (filter (fn [l] (or (lg/isConstVerOf l l1) (let [se (and (self-established l1) (= (:args l) (:args l1)))] se))) logics)
  ))

(defn substituted1 [l x lc]
	(match [l lc]
		[{:type "predicate"} {:type "predicate"}]
		(let [arg ((keyword x) (:args lc))] (if (some? arg) (assoc l :args (clojure.set/union (:args l ) (assoc (:args l) (keyword x) arg) )) l))
		[_ {:type "and" :a a :b b}] (substituted1 (substituted1 l x a) x b)
		[_ {:type "or" :a a :b b}] (substituted1 (substituted1 l x a) x b)
		[_ {:type "not" :a a}] (substituted1 l x a)
		[_ _] l
	))

(defn deductions [logics]
  (let [result
    (map
      (fn [l] 
        (match [l] 
          [{:type "all" :x x :px {:type "imp" :a a :b b}}] (map (fn [x] (lg/substituted b x)) (findConstVerOf logics a {(keyword x) x})) 
          [_] nil
        ))
      logics)]
    (into #{} (filter some? (flatten result)))))

(defn inductions [logics]
  (let [result
    (map (fn [l1] (map (fn [l2] (if (not (= l1 l2)) (lg/induction (lg/andOf l1 l2)) nil)) logics)) logics)]
    (into #{} (filter some? (flatten result)))))

(defn abductions [logics]
  (let [result
    (map
      (fn [l] 
        (match [l] 
          [{:type "all" :x x :px {:type "imp" :a a :b b}}] 
          (flatten (map (fn [bx] (map (fn [ax] (if (= (:args ax) (:args bx)) ax)) (findConstVerOf logics a {(keyword x) x}))) (findConstVerOf logics b {(keyword x) x}))) 
          [_] nil
        ))
      logics)]
    (into #{} (filter some? (flatten result)))))

(defn introduce [logics]
  (let [result
    (map
      (fn [l] 
        (match [l]
          [{:type "all" :x x1 :px {:type "imp" :a {:type "exists" :x x2 :px a } :b b}}] 
						(if (= x1 x2)
							(let [acs (findConstVerOf logics a {(keyword x1) x1})] 
							 (map (fn [ac] (substituted1 b x1 ac)) acs))
							nil) 
          [_] nil
        ))
      logics)]
    (into #{} (filter some? (flatten result)))))

(defrecord brain-db [db facts candidates hiddenfacts temp])
(defn create-brain-db [db] (->brain-db db #{} #{} #{} #{}))
(defn addLogic [brain table l]
  (addLogicDB (:db @brain) table l)
  (reset! brain (update-in @brain [(keyword table)] clojure.set/union #{l})))
(defn removeLogic [brain table l]
  (removeLogicDB (:db @brain) table l)
  (reset! brain (update-in @brain [(keyword table)] clojure.set/difference #{l})))
(defn addLogics [brain table ls] (doseq [l ls] (addLogic brain table l)))
(defn removeLogics [brain table ls] (doseq [l ls] (removeLogic brain table l)))

(defn safeDropTable [db table] (try (j/db-do-commands db (j/drop-table-ddl table)) (catch Exception e nil)))

(defn substituted [p pc] 
  (let 
    [newArgs (map 
      (fn [x]
        (match [(second x)]
          [{:type "variable" :order o :name n}] (first (filter (fn [x2] (= (keyword n) (first x2))) (:args pc)))
          [{:type "func" :eval f}] ((lg/executed f) (second (first (filter (fn [x2] (= (first x) (first x2))) (:args pc)))))
          [_] [(:first x) (:second x)]
          ))
      (:args p))] 
    (lg/predicateOf (:order p) (:name p) (into {} newArgs))))

(defn eval-functions-applied [brain logics]
  (let [fs (filter (fn [l] (some? (:eval l))) logics)]
    (reduce (fn [l r] ((executed (:eval r) (fn [brain self] brain)) l r)) brain fs)))

(defn think [brain] 
  (eval-functions-applied brain (:facts @brain)))
