(ns com.kurogitsune.logica.core
  (:require [clojure.set :refer :all]))
  
(use '[clojure.core.match :refer [match]])
(require '[clojure.core.reducers :as r])

(defrecord lobject [type name])
(defrecord lvariable [type order name])
(defrecord lpredicate [type order name args])
;; funcは例えばsuccのように真偽でなく引数と同じオーダーのLogicを返すLogic
(defrecord lfunc [type order name e args])

(defn objectOf [name] (->lobject "object" name))
(defn variableOf [order name] (->lvariable "variable" order name))
(defn predicateOf [order name args] (->lpredicate "predicate" order name args))
(defn funcOf [order name e args] (->lfunc "func" order name e args))

(defrecord lor [type a b])
(defrecord land [type a b])
(defrecord lnot [type a])
(defrecord limp [type a b])

(defn orOf [a b] (->lor "or" a b))
(defn andOf [a b] (->land "and" a b))
(defn notOf [a] (->lnot "not" a))
(defn impOf [a b] (->limp "imp" a b))

(defrecord lall [type x px])
(defrecord lexists [type x px])

(defn allOf [x px] (->lall "all" x px))
(defn existsOf [x px] (->lexists "exists" x px))

(defn executed [request] (binding [*ns* (find-ns 'com.kurogitsune.logica.core)] (eval (read-string request))))

(defn isConst [t] 
  (match [t]
    [{:type "object"}] true
    [{:type "variable"}] false
    [{:type "predicate" :args xs}] (every? true? (map (fn [x] (isConst (second x))) xs ))
    [{:type "imp" :a a :b b}] (and (isConst a) (isConst b)) 
    [{:type "and" :a a :b b}] (and (isConst a) (isConst b))  
    [{:type "or" :a a :b b}] (and (isConst a) (isConst b))  
    [{:type "not" :a a}] (isConst a) 
    [_] false
  ))
  
(defn isFunc [t]
  (match [t]
    [{:type "func"}] true
    [_] false
  ))
  
(defn isConstVerOf [pc p] 
  (and 
    (isConst pc)
    (= (:name pc) (:name p))
    ;; @todo すでにp側にconstがある場合は一致しなければならない
    (every? true? (map (fn [x] (isConst (second x))) (:args pc)))
    ;; funcの場合,abductionが機能してしまうので除外
    (every? false? (map (fn [x] (isFunc (second x))) (:args p)))
    (every? true? 
      (map 
        (fn [x] (not (= nil (filter 
          (fn [x2] (= (first x2) (first x))) 
          (:args p)))))
        (:args pc)))))
        
(defn substituted [p pc] 
  (let 
    [newArgs (map 
      (fn [x] 
        ()
        (match [(second x)]
          [{:type "variable" :order o :name n}] (first (filter (fn [x2] (= (keyword n) (first x2))) (:args pc)))
          [{:type "func" :e f}] 
            (let [arg (first (map (fn [l] (:args (second l))) (filter (fn [x2] (= (first x) (first x2))) (:args pc))))] 
              ((executed f) (flatten (vec arg))))
          [_] [(first x) (second x)])
          )
      (:args p))] 
    (predicateOf (:order p) (:name p) (into {} newArgs))))
      
(defn unapplyName [p nameX] (first (filter (fn [x] (= nameX (:name (second x)))) (:args p))))
(defn unapplyKey [p keyX] (first (filter (fn [x] (= keyX (first x))) (:args p))))

(defn variableVersion [p] 
    (let 
      [newArgs (map-indexed
        (fn [i x] 
          (match [(second x)]
            [_] [(first x) (variableOf 0 (str "x" i))]
          )) 
        (:args p))] 
      (predicateOf (:order p) (:name p) (into {} newArgs))))

(defn holder [l] (->> (:args l) (filter (fn [x] (= "variable" (:type (second x))))) (map (fn [x] (first x)))))
(defn holder-match [p q] (= (holder p) (holder q)))
(defn const [l] (->> (:args l) (filter (fn [x] (not (= "variable" (:type (second x)))))) (into {})))
(defn same-const [p q] (into {} (clojure.set/intersection (into #{} (const p)) (into #{} (const q)))))
(defn diff-const [p q] (into {} (clojure.set/difference (into #{} (const p)) (into #{} (const q)))))
(defn variable-version-with-const [l cs]
	(let [v (variableVersion l)]
		(assoc v :args (reduce (fn [l r] (assoc l (first r) (second r))) (:args v) cs))))

(defn deduction [l]
  (match [l]
    [{:type "and" :a {:type "all" :x x :px {:type "imp" :a p :b q}} :b pc}] (if (isConstVerOf pc p) (substituted q pc) nil) 
    [_] nil ))
    
(defn induction [l]
  (match [l]
    [{:type "and" 
      :a {:type "predicate" :order op :args ap}  
      :b {:type "predicate" :order oq :args aq}}]
    (let [cs (same-const (:a l) (:b l)) csa (diff-const (:a l) (:b l)) csb (diff-const (:b l) (:a l))]
			(if 
				(and (some? cs) (holder-match (:a l) (:b l))) (allOf "x0" (impOf (variable-version-with-const (:a l) csa) (variable-version-with-const (:b l) csb)))))  
    [_] nil ))
   
(defn abduction [l]
  (match [l]
    [{:type "and" :a qc :b {:type "all" :x x :px {:type "imp" :a p :b q}}}] (if (isConstVerOf qc q) (substituted p qc) nil) 
    [_] nil ))
  
(defn simplified [l] 
  (match [l] 
    [{:type "not" :a {:type "not" :a a}}] (simplified a)
    [_] l
  ))

(defn notAll [l] (if (isConst l) (notOf l) 
  (match [l] 
    [{:type "all" :x x :px px}] (existsOf x (notOf px)) 
    [{:type "exists" :x x :px px}] (allOf x px) 
    [_] (existsOf "x0" (notOf l)))))
