(ns com.kurogitsune.ccgjp
	(:require [clojure.core.match :refer [match]]
						[clojure.core.cache :as c]))

(defrecord T [T])

(defn is-compatible [a b]
	(let [result (match [a b]
		[[an (spec-a :guard set?)] [bn (spec-b :guard set?)]] (and (is-compatible an bn) (clojure.set/subset? spec-a spec-b))
		[[an (spec-a :guard set?)] bn] (is-compatible an bn)
		[(va :guard sequential?) (vb :guard sequential?)] (every? (fn [x] (is-compatible (first x) (second x))) (map vector va vb))
		[_ (_ :guard (fn [x] (= a b)))] true
		[_ (_ :guard (fn [x] (:T a)))] true
		[_ (_ :guard (fn [x] (:T b)))] true
		[_ _] false
		)]
		result))

(defn extract-t [typed a]
	(match [typed a]
		[(vt :guard sequential?) (va :guard sequential?)] (apply merge (filter some? (map (fn [x] (extract-t (first x) (second x))) (map vector vt va))))
		[et ea] (if (:T et) (let [] {et ea}))))

(defn max-t [typed]
	(if (and (sequential? typed) (not (empty? typed))) (apply max (map (fn [x] (max-t x)) typed)) (get typed :T 0)))

(defn num-ts [typed]
	(if (and (sequential? typed) (not (empty? typed))) (not-empty (filter some? (reduce clojure.set/union (map (fn [x] (num-ts x)) typed)))) #{(:T typed)}))

(defn is-t-zero [p]
	(zero? (count (num-ts p))))

(defn replaced-t [t n]
	(if (not (nil? (:T t))) (let [c (:T t)] (if (<= c n) (->T (+ n c)) t))
		(if (map? t) (into {} (map (fn [x] [(first x) (replaced-t (second x) n)]) t))
			(if (sequential? t) (vec (map (fn [x] (replaced-t x n)) t))
				t
			))))

(defn complete-t [typed t]
	(let [t2 (replaced-t t (max-t typed))] ;; 代入先が被らないようにカウントアップ
		(if (sequential? typed) (vec (map (fn [x] (complete-t x t2)) typed))
			(get t2 typed typed)
		)))

(defn type-lifted-normal [a] (let [t (->T (+ 1 (max-t a)))] [t "/" [t "\\" a]]))
(defn type-lifted-reversed [a] (let [t (->T (+ 1 (max-t a)))] [t "\\" [t "/" a]]))
															
(defn is-apply-func [s]
	(match [s]
		[[[p1 [a "/" b] f1] [p2 c f2]]] (is-compatible c b) ;; 関数適用規則
		[[[p1 a f1] [p2 [b "\\" c] f2]]] (is-compatible a c)
		:else false))

(defn is-merge-func [s]
	(match [s]
		[[[p1 [a "/" b] f1] [p2 args f2]]] ;; 一般化関数合成規則
		(and 
			(let [seps (take-nth 2 (rest args))] (apply = "/" seps))
			(is-compatible b (first args)))
		[[[p1 args f1] [p2 [a "\\" b] f2]]]
		(and 
			(let [seps (take-nth 2 (rest args))] (apply = "\\" seps))
			(is-compatible b (first args)))
		:else false))

(defn is-cross-func [s]
	(match [s]
		[[[p1 [[a "/" b] "\\" c] f1] [p2 [d "\\" e] f2]]] (and (is-compatible b d) (is-compatible c e)) ;; 関数交差置換規則
		[[[p1 [a "/" b] f1] [p2 [[c "\\" d] "/" e] f2]]] (and (is-compatible a d) (is-compatible b e))
		:else false))

(defn mix-normal [a] 
	(let [t (->T (+ 1 (max-t a)))] 
		[[t "/" (get a 2)] "/" [t "/" (get a 0)]]))
(defn mix-reversed [a] 
	(let [t (->T (+ 1 (max-t a)))] 
		[[t "\\" (get a 2)] "\\" [t "\\" (get a 0)]]))

(defn is-mix-func [s]
	(match [s]
		[[[p1 [a "/" args] f1] p2]]
		(and 
			(let [seps (take-nth 2 (rest args))] (apply = "\\" seps))
			(is-compatible a (first args)))
		[[[p1 [a "\\" args] f1] p2]]
		(and 
			(let [seps (take-nth 2 (rest args))] (apply = "/" seps))
			(is-compatible a (first args)))
		[[p1 [p2 [b "/" args] f2]]] 
		(and 
			(let [seps (take-nth 2 (rest args))] (apply = "\\" seps))
			(is-compatible b (first args)))
		[[p1 [p2 [b "\\" args] f2]]] 
		(and 
			(let [seps (take-nth 2 (rest args))] (apply = "/" seps))
			(is-compatible b (first args)))
		:else false
	))

(def ^:const max-depth 2)
(def ^:const max-t-val 2)

(defn is-not-type-complex [depth a] (and (< depth max-depth) (< (count (num-ts a)) max-t-val)))

(defn n-args [f]
  (-> f class .getDeclaredMethods first .getParameterTypes alength))

(defn applied [f1 f2]
	(let []
		(if (and (some? f1) (fn? f1)) (f1 f2) "Failed")))

(def cache (atom (c/fifo-cache-factory {})))

(defn cached [seed result]
	(swap! cache (fn [x] (c/miss x seed result)))
	result)

(defn build-tree-impl [acc f]
	(let [next-acc 
		(apply clojure.set/union (map (fn [xs] xs
			(let [pairs (map f (map vector xs (rest xs)))
						result (if (some true? (map nil? pairs)) nil (into #{} (map-indexed (fn [i p] (vec (concat (take i xs) [p] (drop (+ 2 i) xs)))) pairs)))]
			result
		)) acc))]
		(if (= 2 (count (first next-acc))) next-acc (build-tree-impl next-acc f))
		))

(defn build-tree [xs f] (build-tree-impl #{xs} f))

(defn combine
	([s] (combine s "" 0))
	([s span depth] (reset! cache (c/fifo-cache-factory {})) (combine s span depth true true true))
	([s span depth use-lift? use-mix? whole-t?]
		(defn tree-combine [full flag-lift flag-mix flag-whole-t]
			(let [
			flatten-x (fn [x] (if (and (some? (first x)) (some? (second x))) (not-empty (reduce clojure.set/union (into #{} (map (fn [a] 
				(reduce clojure.set/union (into #{} (map (fn [b] (combine [a b] span depth flag-lift flag-mix flag-whole-t)) (second x))))) (first x)))))))
			treed-full (build-tree (map (fn [f] [f]) full) flatten-x)
			combined-full (into #{} (map flatten-x treed-full))]
			(reduce clojure.set/union combined-full)))
	 	(let [some-zero-in-combined (fn [lz] #{(not-empty (vec (some (fn [combined] (some (fn [p] (if (is-t-zero p) p)) combined)) lz)))})
					all-t-in-combined (fn [lz] (map vec (reduce clojure.set/union lz)))]
			(if (c/has? @cache [(map (partial take 2) s) use-lift? use-mix? whole-t?])
				(get @cache [(map (partial take 2) s) use-lift? use-mix? whole-t?]) 
				(cached [(map (partial take 2) s) use-lift? use-mix? whole-t?]
			(match [s]
				[[p nil]] nil
				[[nil p]] nil
				[[nil nil]] nil
				[([[p1 [a "/" b] f1] [p2 c f2]] :guard is-apply-func)] ;; 関数適用規則
				(let []
					#{[(str p1 span p2) (complete-t a (extract-t b c)) (applied f1 f2)]}) 
				[([[p1 a f1] [p2 [b "\\" c] f2]] :guard is-apply-func)]
				(let [] 
					#{[(str p1 span p2) (complete-t b (extract-t c a)) (applied f2 f1)]})
				[([[p1 [a "/" b] f1] [p2 args f2]] :guard is-merge-func)] ;; 一般化関数合成規則
				#{[(str p1 span p2) 
					(let [e (extract-t b (first args))] 
						(vec (concat [(complete-t a e) "/"] (vec (map (fn [x] (complete-t x e)) (rest (rest args))))))) (fn [x] (if (and (fn? f1) (fn? f2)) (f1 (f2 x))))]}
				[([[p1 args f1] [p2 [a "\\" b] f2]] :guard is-merge-func)]
				#{[(str p1 span p2) 
					(let [e (extract-t b (first args))] 
						(vec (concat [(complete-t a e) "\\"] (vec (map (fn [x] (complete-t x e)) (rest (rest args))))))) (fn [x] (if (and (fn? f1) (fn? f2)) (f2 (f1 x))))]}
				[([[p1 [[a "/" b] "\\" c] f1] [p2 [d "\\" e] f2]] :guard is-cross-func)] ;; 関数交差置換規則
				#{[(str p1 span p2) [a "/" e] (fn [x] (if (and (fn? f1) (fn? f2)) ((f1 x) (f2 x))))]}
				[([[p1 [a "/" b] f1] [p2 [[c "\\" d] "/" e] f2]] :guard is-cross-func)]
				#{[(str p1 span p2) [c "/" b] (fn [x] (if (and (fn? f1) (fn? f2)) ((f2 x) (f1 x))))]}
				[(ps :guard 
					(fn [x] (let [rels (take-nth 2 (rest x))] 
						(and 
							(> (count rels) 0)
							(every? (fn [r] (= (second r) "CONJ")) rels) 
							(apply = (map first rels))))))] 
				#{[(clojure.string/join " " (map first ps)) (second (first ps)) (applied (get (second ps) 2) (map (fn [x] (get x 2)) (take-nth 2 ps)))]}
				[(full :guard empty?)] nil
				[(full :guard (fn [x] (and (every? sequential? x) (> (count x) 2))))]
				(tree-combine full use-lift? use-mix? whole-t?)
				;; 一致しない場合、型繰り上げ規則を適用
				[[[p1 (a :guard (partial is-not-type-complex depth)) f1] [p2 (b :guard (partial is-not-type-complex depth)) f2]]] 
				(let [aln (type-lifted-normal a) 
							alr (type-lifted-reversed a) 
							bln (type-lifted-normal b) 
							blr (type-lifted-reversed b)
							ft1 (fn [f] (if (fn? f) (f f1)))
							ft2 (fn [f] (if (fn? f) (f f2)))
							fm1 (fn [g] (fn [x] (if (and (fn? g) (fn? f1)) (g (f1 x)))))
							fm2 (fn [g] (fn [x] (if (and (fn? g) (fn? f2)) (g (f2 x)))))]
					(defn some-in-complex [filter-func]
						(filter-func [
							(if use-lift? (filter-func [
								(combine [[p1 aln ft1] [p2 b f2]] span (+ depth 1) use-lift? use-mix? whole-t?)
								(combine [[p1 alr ft1] [p2 b f2]] span (+ depth 1) use-lift? use-mix? whole-t?)
								(combine [[p1 a f1] [p2 bln ft2]] span (+ depth 1) use-lift? use-mix? whole-t?)
								(combine [[p1 a f1] [p2 blr ft2]] span (+ depth 1) use-lift? use-mix? whole-t?)]))
							(if use-mix? (match [a]
								[[ap "/" args-a]] 
									(let [an (mix-normal a)] 
										(filter-func [
											(combine [[p1 an fm1] [p2 b f2]] span (+ depth 1) use-lift? use-mix? whole-t?)
											(combine [[p1 an fm1] [p2 bln ft2]] span (+ depth 1) use-lift? use-mix? whole-t?)
											(combine [[p1 an fm1] [p2 blr ft2]] span (+ depth 1) use-lift? use-mix? whole-t?)
										]))
								[[ap "\\" args-a]] 
									(let [ar (mix-reversed a)] 
										(filter-func [
											(combine [[p1 ar fm1] [p2 b f2]] span (+ depth 1) use-lift? use-mix? whole-t?)
											(combine [[p1 ar fm1] [p2 bln ft2]] span (+ depth 1) use-lift? use-mix? whole-t?)
											(combine [[p1 ar fm1] [p2 blr ft2]] span (+ depth 1) use-lift? use-mix? whole-t?)
										]))
								[_] nil
							))
							(if use-mix? (match [b]
								[[bp "/" args-b]] 
									(let [bn (mix-normal b)] 
										(filter-func [
											(combine [[p1 a f1] [p2 bn fm2]] span (+ depth 1) use-lift? use-mix? whole-t?)
											(combine [[p1 aln ft1] [p2 bn fm2]] span (+ depth 1) use-lift? use-mix? whole-t?)
											(combine [[p1 alr ft1] [p2 bn fm2]] span (+ depth 1) use-lift? use-mix? whole-t?)
										]))
								[[bp "\\" args-b]] 
									(let [br (mix-reversed b)]
										(filter-func [
											(combine [[p1 a f1] [p2 br fm2]] span (+ depth 1) use-lift? use-mix? whole-t?)
											(combine [[p1 aln ft1] [p2 br fm2]] span (+ depth 1) use-lift? use-mix? whole-t?)
											(combine [[p1 alr ft1] [p2 br fm2]] span (+ depth 1) use-lift? use-mix? whole-t?)
										]))
								[_] nil
							))
							])
					)
					(if whole-t? (some-in-complex all-t-in-combined) (some-in-complex some-zero-in-combined))
					)
				[[[p1 a f]]] #{[p1 a f]}
				[_] nil
			))))
	))

(defn combine-simple [s] (reset! cache (c/fifo-cache-factory {})) (combine s "" 0 false false false))
(defn combine-with-lift [s] (reset! cache (c/fifo-cache-factory {})) (combine s "" 0 true false false))
(defn combine-with-lift-mix [s] (reset! cache (c/fifo-cache-factory {})) (combine s "" 0 true true false))
(defn combine-full [s] (reset! cache (c/fifo-cache-factory {})) (combine s "" 9 true true true))

