(ns k16.kid
  (:require
   [clojure.string :as str]
   [k16.base58 :as base58])
  (:import
   com.fasterxml.uuid.Generators
   com.fasterxml.uuid.NoArgGenerator
   com.fasterxml.uuid.impl.UUIDUtil
   java.time.Instant
   java.util.UUID))

(set! *warn-on-reflection* true)

(defonce ^:private ^NoArgGenerator GENERATOR
  (delay (Generators/timeBasedEpochGenerator)))

(defn- v7
  ([]
   (v7 @GENERATOR))
  ([^NoArgGenerator generator]
   (.generate generator)))

(defn uuid->kid [^UUID uuid]
  (-> uuid
      UUIDUtil/asByteArray
      base58/encode))

(defn uuid->bytes [^UUID uuid]
  (UUIDUtil/asByteArray uuid))

(defn kid->bytes ^bytes [^String kid]
  (let [without-prefix (if (<= (String/.length kid) 22)
                         kid
                         (second (str/split kid #"_")))]
    (base58/decode without-prefix)))

(defn bytes->kid
  ([^bytes kid]
   (bytes->kid kid nil))
  ([^bytes kid ^String prefix]
   (let [kid (base58/encode kid)]
     (if prefix
       (str prefix "_" kid)
       kid))))

(defn kid->uuid [^String kid]
  (-> kid
      kid->bytes
      UUIDUtil/uuid))

(defn- uuid->instant [^UUID uuid]
  (-> (.getMostSignificantBits uuid)
      (bit-shift-right 16)
      (Instant/ofEpochMilli)))

(defn kid->instant [kid]
  (-> kid kid->uuid uuid->instant))

(defn generate-kid
  ([]
   (uuid->kid (v7)))
  ([prefix]
   (->> (v7)
        uuid->kid
        (str prefix "_"))))

(defn generate-kid-bytes []
  (uuid->bytes (v7)))
