(ns k16.glitch
  (:require
   [k16.glitch.impl :as impl]
   [malli.core :as m]
   [malli.error :as me]
   [malli.transform :as mt]
   [malli.util :as mu]
   [meta-merge.core :as metamerge])
  (:import
   [k16.glitch Glitch]))

(set! *warn-on-reflection* true)

(def ?Glitch
  [:map {:closed true}
   [:code :keyword]
   [:message :string]
   [:data {:optional true} :map]

   [:display-safe {:optional true :default false} :boolean]
   [:display-message {:optional true} :string]
   [:display-data {:optional true} :map]

   [:http-status-code {:optional true} :int]])

(def ?GlitchDefinition
  (mu/merge
   ?Glitch
   [:map
    [:parent {:optional true} :any]
    [:message {:optional true} :string]

    [:props-schema {:optional true} :any]

    [:constructor {:optional true}
     [:function
      [:=> [:cat [:map
                  [:props :map]
                  [:glitch ?Glitch]
                  [:definition :map]
                  [:cause {:optional true} [:maybe :any]]]]
       ?Glitch]]]]))

(declare invalid_type)
(declare schema_invalid)
(declare validate-schema!)
(declare fault)

(defn glitch?
  ([glitch] (glitch? nil glitch))
  ([definition glitch]
   (if definition
     (and (instance? Glitch glitch)
          (= (:code definition) (:code glitch)))
     (instance? Glitch glitch))))

(defn build-glitch [{:keys [definition message props cause]}]
  (when (:props-schema definition)
    (validate-schema! (:props-schema definition)
                      (or props {})
                      "Glitch :props did not conform to :props-schema"))

  (let [constructor-fn (:constructor definition)

        glitch-keys [:code :message :data
                     :display-data :display-message
                     :display-safe :http-status-code]

        overrides (cond-> (select-keys props glitch-keys)
                    message (assoc :message message))

        glitch (merge (select-keys definition glitch-keys)
                      overrides)]

    (if constructor-fn
      (as-> (constructor-fn
             {:props props
              :glitch glitch
              :definition definition
              :cause cause})
            $
        (select-keys $ glitch-keys)
        (metamerge/meta-merge glitch $ overrides))
      glitch)))

(defmacro ex
  ([definition]
   `(ex ~definition nil nil nil))

  ([definition message-or-props-or-cause]
   `(cond
      (string? ~message-or-props-or-cause)
      (ex ~definition ~message-or-props-or-cause nil nil)

      (map? ~message-or-props-or-cause)
      (ex ~definition nil ~message-or-props-or-cause nil)

      (instance? Throwable ~message-or-props-or-cause)
      (ex ~definition nil nil ~message-or-props-or-cause)

      (nil? ~message-or-props-or-cause)
      (ex ~definition nil nil nil)

      :else
      (throw (impl/->glitch* {:code :fault
                              :message "ex used with incorrect arguments"
                              :data {:definition ~definition
                                     :arg [~message-or-props-or-cause]}}))))

  ([definition message-or-props props-or-cause]
   `(let [message?# (or (string? ~message-or-props)
                        (nil? ~message-or-props))
          cause?# (or (instance? Throwable ~props-or-cause)
                      (nil? ~props-or-cause))]
      (cond
        (and message?# (map? ~props-or-cause))
        (ex ~definition ~message-or-props ~props-or-cause nil)

        (and message?# cause?#)
        (ex ~definition ~message-or-props nil ~props-or-cause)

        cause?#
        (ex ~definition nil ~message-or-props ~props-or-cause)

        :else
        (throw (impl/->glitch* {:code :fault
                                :message "ex used with incorrect arguments"
                                :data {:definition ~definition
                                       :rest-of-args [~message-or-props ~props-or-cause]}})))))

  ([definition message props cause]
   `(let [glitch# (build-glitch {:definition ~definition
                                 :message ~message
                                 :props ~props
                                 :cause ~cause})]
      (impl/->glitch* glitch# ~cause))))

#_{:clj-kondo/ignore [:clojure-lsp/unused-public-var]}
(defmacro ex!
  "Same as `ex` but this throws the constructed glitch directly"
  ([definition]
   `(throw (ex ~definition)))
  ([definition message|props|cause]
   `(throw (ex ~definition ~message|props|cause)))
  ([definition message|props props|cause]
   `(throw (ex ~definition ~message|props ~props|cause)))
  ([definition message props cause]
   `(throw (ex ~definition ~message ~props ~cause))))

#_{:clj-kondo/ignore [:clojure-lsp/unused-public-var]}
(defn ex->glitch [e glitch]
  (when-not (instance? Exception e)
    (throw (ex invalid_type
               {:data {:expected-type Exception
                       :actual-type (type e)}})))

  (if (glitch? e)
    e
    (ex fault (merge {:message (.getMessage ^Exception e)
                      :data (or (ex-data e) {})}
                     glitch)
        e)))

(defn display-safe
  "Construct a new glitch with has been made display-safe"
  [glitch]
  (when-not (glitch? glitch)
    (throw (ex invalid_type
               {:data {:expected-type Glitch
                       :actual-type (type glitch)}})))

  (ex (assoc (impl/get-state glitch)
             :display-safe true)
      glitch))

#_{:clj-kondo/ignore [:clojure-lsp/unused-public-var]}
(defn display-safe!
  "Like `gl/display-safe` except this throws the error"
  [glitch]
  (throw (display-safe glitch)))

(def ^:private make-schema-validator
  (memoize m/validator))

(defn validate-schema!
  "Validate the given `data` against a given `?schema` throwing a schema_invalid glitch if
  the data does not conform to the schema.
  
  This function compiles and memoizes the given ?schema for general performance improvements.
  Note therefor that if provided with high cardinality schema data this may unexpectedly
  increase process memory."
  ([?schema data] (validate-schema! ?schema data nil))
  ([?schema data message]
   (let [validator (make-schema-validator ?schema)]
     (when-not (validator data)
       (ex! schema_invalid (cond-> {:schema ?schema
                                    :input data}
                             message (assoc :message message)))))))

(defn ->definition [config]
  (validate-schema! ?GlitchDefinition config "Glitch definition did not conform to schema")
  (let [config (m/decode ?GlitchDefinition config (mt/default-value-transformer {::mt/add-optional-keys true}))
        parent (:parent config)
        definition (cond-> config
                     (:data-schema config) (update :data-schema with-meta {:displace true})
                     (:parent config) (-> (assoc :lineage ^:prepend [(:code parent)])))]
    (metamerge/meta-merge (dissoc parent :code) definition)))

(def schema_invalid
  (->definition
   {:code :glitch/schema_invalid
    :props-schema [:map
                   [:schema :any]
                   [:input :any]]
    :constructor (fn [{:keys [props glitch]}]
                   (let [{:keys [schema input]} props
                         message (or (:message glitch) "Input data did not conform to schema")]
                     {:message message
                      :data {:input input
                             :errors (-> (m/explain schema input)
                                         me/humanize)}}))}))

(def invalid_type
  (->definition
   {:code :glitch/invalid_type
    :props-schema [:map
                   [:expected :any]
                   [:received :any]]
    :constructor (fn [{:keys [props]}]
                   (let [{:keys [expected received]} props]
                     {:message (str "Expected type [" expected "] but got [" received "]")
                      :data {:expected expected
                             :received received}}))}))

#_{:clj-kondo/ignore [:clojure-lsp/unused-public-var]}
(def bad_request
  (->definition
   {:code :bad_request
    :message "Bad Request"
    :http-status-code 400}))

#_{:clj-kondo/ignore [:clojure-lsp/unused-public-var]}
(def authentication_error
  (->definition
   {:code :authentication_error
    :message "Authentication Error"
    :http-status-code 401}))

#_{:clj-kondo/ignore [:clojure-lsp/unused-public-var]}
(def authorization_error
  (->definition
   {:code :authorization_error
    :message "Forbidden: No access to resource"
    :http-status-code 403}))

(def not_found
  (->definition
   {:code :not_found
    :http-status-code 404
    :props-schema [:map [:resource {:optional true} [:maybe :string]]]
    :constructor (fn [{:keys [props]}]
                   (let [resource (:resource props)
                         resource-provided? (contains? props :resource)

                         message (when resource-provided?
                                   (str " [" (if resource resource "nil") "]"))]
                     {:message (str "Resource not found" message)
                      :data (select-keys props [:resource])}))}))

#_{:clj-kondo/ignore [:clojure-lsp/unused-public-var]}
(def conflict
  (->definition
   {:code :conflict
    :message "Conflict"
    :http-status-code 409}))

#_{:clj-kondo/ignore [:clojure-lsp/unused-public-var]}
(def payload_too_large
  (->definition
   {:code :payload_too_large
    :message "Payload too large"
    :http-status-code 413}))

#_{:clj-kondo/ignore [:clojure-lsp/unused-public-var]}
(def unprocessable_entity
  (->definition
   {:code :unprocessable_entity
    :message "Unprocessable entity"
    :http-status-code 422}))

#_{:clj-kondo/ignore [:clojure-lsp/unused-public-var]}
(def too_many_requests
  (->definition
   {:code :too_many_requests
    :message "Too many requests"
    :http-status-code 429}))

(def fault
  (->definition
   {:code :fault
    :message "Server Error"
    :http-status-code 500}))

#_{:clj-kondo/ignore [:clojure-lsp/unused-public-var]}
(def not_implemented
  (->definition
   {:code :not_implemented
    :message "Not Implemented"
    :http-status-code 501}))

#_{:clj-kondo/ignore [:clojure-lsp/unused-public-var]}
(def bad_gateway
  (->definition
   {:code :bad_gateway
    :message "Bad Gateway"
    :http-status-code 502}))

#_{:clj-kondo/ignore [:clojure-lsp/unused-public-var]}
(def service_unavailable
  (->definition
   {:code :service_unavailable
    :message "Service Unavailable"
    :http-status-code 503}))

#_{:clj-kondo/ignore [:clojure-lsp/unused-public-var]}
(def gateway_timeout
  (->definition
   {:code :gateway_timeout
    :message "Gateway Timeout"
    :http-status-code 504}))
