(ns com.keminglabs.util.set
  (:require [clojure.set :refer [intersection union]]))


;;Modified from
;;https://github.com/francoisdevlin/devlinsf-clojure-utils/blob/master/src/lib/sfd/table_utils.clj
;;http://stackoverflow.com/questions/13009939/outer-join-in-clojure?lq=1

(defn inner-style
  [left-keys right-keys]
  (intersection (set left-keys) (set right-keys)))

(defn left-outer-style
  [left-keys right-keys]
  left-keys)

(defn right-outer-style
  [left-keys right-keys]
  right-keys)

(defn full-outer-style
  [left-keys right-keys]
  (union (set left-keys) (set right-keys)))

(defn join-worker
  "This is an internal method to be used in each join function."
  ([join-style left-coll right-coll join-fn] (join-worker join-style left-coll right-coll join-fn join-fn))
  ([join-style left-coll right-coll left-join-fn right-join-fn]
     (let [keys-a (keys (first left-coll)) ;The column names of coll-a
           keys-b (keys (first right-coll)) ;The column names of coll-b
           indexed-left (group-by left-join-fn left-coll) ;indexes the coll-a using join-fn-a
           indexed-right (group-by right-join-fn right-coll) ;indexes the coll-b using join-fn-b
           desired-joins (join-style (keys indexed-left) (keys indexed-right))]
         
         (into #{} (for [joined-value desired-joins
                         left-side (get indexed-left joined-value [{}])
                         right-side (get indexed-right joined-value [{}])]
                       (merge left-side right-side))))))

(defmacro defjoin
  [join-name join-style doc-string]
  `(do
     (defn ~join-name
       ~(str doc-string
             "\n  This function takes a left collection, a right collection, and at least one join function.  If only one
join function is provided, it is used on both the left & right hand sides.")
       ([~'left-coll ~'right-coll ~'join-fn]
          (join-worker ~join-style ~'left-coll ~'right-coll ~'join-fn ~'join-fn))
       ([~'left-coll ~'right-coll ~'left-join-fn ~'right-join-fn]
          (join-worker ~join-style ~'left-coll ~'right-coll ~'left-join-fn ~'right-join-fn)))))

(defjoin inner-join inner-style
  "This is for performing inner joins.  The join value must exist in both lists.")
(defjoin left-outer-join left-outer-style
  "This is for performing left outer joins.  The join value must exist in the left hand list.")
(defjoin right-outer-join right-outer-style
  "This is for performing right outer joins.  The join value must exist in the right hand list.")
(defjoin full-outer-join full-outer-style
  "This is for performing full outer joins.  The join value may exist in either list.")

(defn natural-join
  "Performs the natural join.  If there are no keys that intersect, the join is not performed."
  [left-coll right-coll]
  (let [intersect (apply intersection
                         (map (comp set keys first)
                              [left-coll right-coll]))]
    (if (empty? intersect)
      []
      (inner-join left-coll right-coll (apply juxt intersect)))))

(defn cross-join
  "DAMN CLOJURE IS AWESOME"
  [left-coll right-coll]
  (inner-join left-coll right-coll (constantly 1)))