(ns cljx.core
  (:use [cljx.rules :only [cljs-rules]]
        [clojure.java.io :only [reader make-parents]])
  (:require [clojure.string :as string]
            kibit.check)
  (:import [java.io File]))

(def ^:private warning-str ";; This file autogenerated from ")

;;Taken from clojure.tools.namespace
(defn cljx-source-file?
  "Returns true if file is a normal file with a .cljx extension."
  [^File file]
  (and (.isFile file)
       (.endsWith (.getName file) ".cljx")))

(defn find-cljx-sources-in-dir
  "Searches recursively under dir for CLJX files.
Returns a sequence of File objects, in breadth-first sort order."
  [^File dir]
  ;; Use sort by absolute path to get breadth-first search.
  (sort-by #(.getAbsolutePath ^File %)
           (filter cljx-source-file? (file-seq dir))))

;; not using clojure.walk/walk because of http://groups.google.com/group/clojure-dev/browse_frm/thread/f8e2f2b0276783a9
(defn- walk
  [inner outer form]
  (cond
   (list? form) (outer (with-meta (apply list (map inner form)) (meta form)))
   (instance? clojure.lang.IMapEntry form) (outer (vec (map inner form)))
   (seq? form) (outer (doall (map inner form)))
   (coll? form) (outer (into (empty form) (map inner form)))
   :else (outer form)))

(defn- postwalk
  [f form]
  (walk (partial postwalk f) f form))

(defn- remove-exclusions
  [{:keys [nested-exclusions]} munged-exprs]
  (if-not nested-exclusions
    (remove #(= % :cljx.core/exclude) munged-exprs)
    (postwalk
      #(cond
         (instance? clojure.lang.IMapEntry %) %
         (seq? %) (with-meta
                    (apply list (remove (partial = :cljx.core/exclude) %))
                    (meta %))
         (coll? %) (into (empty %) (remove (partial = :cljx.core/exclude) %))
         :else %)
      munged-exprs)))

(defn munge-expr
  [expr {:keys [rules nested-exclusions] :as options}]
  (->> (kibit.check/check-expr expr
                               :rules rules
                               :guard identity
                               :resolution :toplevel)
    (map #(or (:alt %) (:expr %)))
    (remove-exclusions options)))

(defn munge-forms
  [reader {:keys [rules nested-exclusions] :as options}]
  (->> (kibit.check/check-reader reader
                                 :rules rules
                                 :guard identity
                                 :resolution :toplevel)
    (map #(or (:alt %) (:expr %)))
    (remove-exclusions options)))

(defn- write-on-correct-lines
  [line-number form]
  (let [offset (if (and (list? form) (-> form meta :line))
                 (-> form meta :line (- line-number) (max 0))
                 0)
        write-form (fn [offset form delim delim2]
                     (#'clojure.core/print-meta form *out*)
                     (print delim)
                     (let [ret (reduce write-on-correct-lines (+ offset line-number) form)]
                       (print delim2)
                       ret))]
    (dotimes [_ offset] (println))
    (cond
      (list? form) (write-form offset form "(" ")")
      (set? form) (write-form offset form "#{" "}")
      (vector? form) (write-form offset form "[" "]")
      (map? form) (write-form offset (apply concat form) "{" "}")
      :else (do (pr form) (print " ") (+ line-number offset)))))

(defn generate
  [cljx-path {:keys [output-path extension rules
                     nested-exclusions maintain-form-position] :as options}]
  (println "Rewriting" cljx-path "to" output-path
           (str "(" extension ")")
           "with" (count rules) "rules.")
     
  (doseq [f (find-cljx-sources-in-dir (File. cljx-path))]
    (let [munged-forms (munge-forms (reader f) options)
          generated-f  (File. (-> (.getPath f)
                                (string/replace cljx-path output-path)
                                (string/replace #"cljx$" extension)))]
      (make-parents generated-f)
      (spit generated-f
        (with-out-str
          (print warning-str)
          (println (.getPath f))
          (if maintain-form-position
            (reduce write-on-correct-lines 2 munged-forms)
            (doseq [form munged-forms] (prn form))))))))

(defn- cljx-compile [builds]
  "The actual static transform, separated out so it can be called repeatedly."
  (doseq [build builds
          :let [{:keys [source-paths output-path extension rules
                        include-meta nested-exclusions maintain-form-position] :as opts}
                (merge {:extension "clj" :include-meta false
                        :maintain-form-position false :nested-exclusions false}
                  build)]]
    (let [rules (if-not (symbol? rules)
                  ;; TODO now that we are evaluating within the context of the
                  ;; user's project, there should be no reason for this eval to
                  ;; exist any more
                  (eval rules)
                  (do
                    (require (symbol (namespace rules)))
                    @(resolve rules)))]
      (doseq [p source-paths]
        (binding [*print-meta* include-meta]
          (generate p (assoc opts :rules rules)))))))
