goog.provide('Cl.LinearExpression');

goog.require('Cl');
goog.require('Cl.AbstractVariable');
goog.require('Cl.HashTable');

goog.scope(function() {
  var AbstractVariable = Cl.AbstractVariable;
  var HashTable = Cl.HashTable;
;
;
;
/**
 * @constructor
 */
Cl.LinearExpression = function(clv, value, constant) {
  this._constant = constant || 0;
  this._terms = new HashTable();
  if (clv instanceof AbstractVariable) {
    this._terms.put(clv, value || 1);
  } else if (typeof clv === "number") {
    this._constant = clv;
  }
};

Cl.LinearExpression.prototype.initializeFromHash = function(_constant, terms) {
  this._constant = _constant;
  this._terms = terms.clone();
  return this;
};
Cl.LinearExpression.prototype.clone = function() {
  return new Cl.LinearExpression().initializeFromHash(this._constant, this._terms);
};
Cl.LinearExpression.prototype.multiplyMe = function(x) {
  this._constant *= x;
  this._terms.each(goog.bind(function(clv, coeff) {
    return this._terms.put(clv, coeff * x);
  }, this));
  return this;
};
Cl.LinearExpression.prototype.times = function(x) {
  var expr;
  if (typeof x === "number") {
    return this.clone().multiplyMe(x);
  } else {
    expr = x;
    if (this.isConstant()) {
      return expr.times(this._constant);
    } else if (expr.isConstant()) {
      return this.times(expr._constant);
    } else {
      throw new Cl.errors.NonlinearExpression();
    }
  }
};
Cl.LinearExpression.prototype.plus = function(expr) {
  if (expr instanceof Cl.LinearExpression) {
    return this.clone().addExpression(expr, 1);
  } else if (expr instanceof Cl.Variable) {
    return this.clone().addVariable(expr, 1);
  }
};
Cl.LinearExpression.prototype.minus = function(expr) {
  if (expr instanceof Cl.LinearExpression) {
    return this.clone().addExpression(expr, -1);
  } else if (expr instanceof Cl.Variable) {
    return this.clone().addVariable(expr, -1);
  }
};
Cl.LinearExpression.prototype.divide = function(x) {
  if (typeof x === "number") {
    if (Cl.approx(x, 0)) {
      throw new Cl.errors.NonlinearExpression();
    }
    return this.times(1 / x);
  } else if (x instanceof Cl.LinearExpression) {
    if (!x.isConstant()) {
      throw new Cl.errors.NonlinearExpression();
    }
    return this.times(1 / x._constant);
  }
};
Cl.LinearExpression.prototype.divFrom = function(expr) {
  if ((!this.isConstant()) || Cl.approx(this._constant, 0)) {
    throw new Cl.errors.NonlinearExpression();
  }
  return expr.divide(this._constant);
};
Cl.LinearExpression.prototype.subtractFrom = function(expr) {
  return expr.minus(this);
};
Cl.LinearExpression.prototype.addExpression = function(expr, n, subject, solver) {
  if (expr instanceof AbstractVariable) {
    expr = new Cl.LinearExpression(expr);
  }
  this.incrementConstant(n * expr.constant());
  n = n || 1;
  expr.terms().each(goog.bind(function(clv, coeff) {
    return this.addVariable(clv, coeff * n, subject, solver);
  }, this));
  return this;
};
Cl.LinearExpression.prototype.addVariable = function(v, c, subject, solver) {
  var coeff, new_coefficient;
  c = c || 1;
  coeff = this._terms.get(v);
  if (coeff) {
    new_coefficient = coeff + c;
    if (Cl.approx(new_coefficient, 0)) {
      if (solver) {
        solver.noteRemovedVariable(v, subject);
      }
      this._terms.remove(v);
    } else {
      this._terms.put(v, new_coefficient);
    }
  } else if (!Cl.approx(c, 0)) {
    this._terms.put(v, c);
    if (solver) {
      solver.noteAddedVariable(v, subject);
    }
  }
  return this;
};
Cl.LinearExpression.prototype.setVariable = function(v, c) {
  this._terms.put(v, c);
  return this;
};
Cl.LinearExpression.prototype.anyPivotableVariable = function() {
  if (this.isConstant()) {
    throw new Cl.errors.InternalError("anyPivotableVariable called on a constant");
  }
  this._terms.each(function(clv, c) {
    if (clv.isPivotable()) {
      return clv;
    }
  });
  return null;
};
Cl.LinearExpression.prototype.substituteOut = function(outvar, expr, subject, solver) {
  var multiplier;
  multiplier = this._terms.remove(outvar);
  this.incrementConstant(multiplier * expr.constant());
  return expr.terms().each(goog.bind(function(clv, coeff) {
    var new_coeff, old_coeff;
    old_coeff = this._terms.get(clv);
    if (old_coeff) {
      new_coeff = old_coeff + multiplier * coeff;
      if (Cl.approx(new_coeff, 0)) {
        solver.noteRemovedVariable(clv, subject);
        return this._terms.remove(clv);
      } else {
        return this._terms.put(clv, new_coeff);
      }
    } else {
      this._terms.put(clv, multiplier * coeff);
      return solver.noteAddedVariable(clv, subject);
    }
  }, this));
};
Cl.LinearExpression.prototype.changeSubject = function(old_subject, new_subject) {
  return this._terms.put(old_subject, this.newSubject(new_subject));
};
Cl.LinearExpression.prototype.newSubject = function(subject) {
  var reciprocal;
  reciprocal = 1 / this._terms.remove(subject);
  this.multiplyMe(-reciprocal);
  return reciprocal;
};
Cl.LinearExpression.prototype.coefficientFor = function(clv) {
  return this._terms.get(clv) || 0;
};
Cl.LinearExpression.prototype.constant = function() {
  return this._constant;
};
Cl.LinearExpression.prototype.set_constant = function(_constant) {
  this._constant = _constant;
};
Cl.LinearExpression.prototype.terms = function() {
  return this._terms;
};
Cl.LinearExpression.prototype.incrementConstant = function(c) {
  return this._constant += c;
};
Cl.LinearExpression.prototype.isConstant = function() {
  return this._terms.size() === 0;
};
Cl.LinearExpression.prototype.toString = function() {
  var bstr, needsplus;
  bstr = '';
  needsplus = false;
  if (!Cl.approx(this._constant, 0) || this.isConstant()) {
    bstr += this._constant;
    if (this.isConstant()) {
      return bstr;
    } else {
      needsplus = true;
    }
  }
  this._terms.each(function(clv, coeff) {
    if (needsplus) {
      bstr += " + ";
    }
    bstr += coeff + "*" + clv;
    return needsplus = true;
  });
  return bstr;
};;

}); // close goog.scope()
