(ns jjttjj.clj-ib
  (:require [jjttjj.clj-ib
             [mapping :refer [map-> ->map]]
             [translation :refer [translate integer-account-value?
                                  numeric-account-value?
                                  boolean-account-value?]]]
            [taoensso.timbre :as log]
            [clojure.xml :as xml]))


(defn handler-ewrapper
  "Creates a wrapper that flattens the Interactive Brokers EWrapper interface,
   calling a single function with maps that all have a :type to indicate what type
   of messages was received, and the massaged parameters from the event."
  [response-handler]
  (reify
    com.ib.client.EWrapper

    ;;; Connection and Server
    (currentTime [this time]
      (response-handler
       {:type :current-time
        :value (translate :from-ib :date-time time)}))

    (^void error [this ^int requestId ^int errorCode ^String message]
     (response-handler
      {:type :error :request-id requestId :code errorCode
       :message message}))

    (^void error [this ^String message]
     (response-handler {:type :error :message message}))

    (^void error [this ^Exception ex]
     (response-handler {:type :error :exception (.toString ex)}))

    (connectionClosed [this]
      (response-handler {:type :connection-closed}))

    ;;; Market Data
    (tickPrice [this tickerId field price canAutoExecute]
      (response-handler {:type :tick
                         :field (translate :from-ib :tick-field-code field)
                         :ticker-id tickerId
                         :value price
                         :can-auto-execute? (= 1 canAutoExecute)}))

    (tickSize [this tickerId field size]
      (response-handler {:type :tick
                         :field (translate :from-ib :tick-field-code field)
                         :ticker-id tickerId
                         :value size}))

    (tickOptionComputation [this tickerId field impliedVol delta optPrice
                            pvDividend gamma vega theta undPrice]
      (response-handler {:type :tick
                         :field (translate :from-ib :tick-field-code field)
                         :ticker-id tickerId
                         :implied-volatility impliedVol
                         :option-price optPrice
                         :pv-dividends pvDividend
                         :underlying-price undPrice
                         :delta delta :gamma gamma :theta theta :vega vega}))

    (tickGeneric [this tickerId tickType value]
      (response-handler {:type :tick
                         :field (translate :from-ib :tick-field-code tickType)
                         :ticker-id tickerId :value value}))

    (tickString [this tickerId tickType value]
      (let [field (translate :from-ib :tick-field-code tickType)]
        (cond
          (= field :last-timestamp)
          (response-handler {:type :tick :field field
                             :ticker-id tickerId
                             :value (translate :from-ib :date-time value)})

          :else
          (response-handler {:type :tick :field field
                             :ticker-id tickerId
                             :value val}))))

    (tickEFP [this tickerId tickType basisPoints formattedBasisPoints
              impliedFuture holdDays futureExpiry dividendImpact dividendsToExpiry]
      (response-handler {:type :tick
                         :field (translate :from-ib :tick-field-code tickType)
                         :ticker-id tickerId
                         :basis-points basisPoints
                         :formatted-basis-points formattedBasisPoints
                         :implied-future impliedFuture :hold-days holdDays
                         :future-expiry futureExpiry
                         :dividend-impact dividendImpact
                         :dividends-to-expiry dividendsToExpiry}))

    (tickSnapshotEnd [this reqId]
      (response-handler {:type :tick-snapshot-end :request-id reqId}))

    (marketDataType [this reqId type]
      (response-handler {:type :market-data-type
                         :request-id reqId
                         :market-data-type (translate :from-ib :market-data-type type)}))

    ;;; Orders
    (orderStatus [this orderId status filled remaining avgFillPrice permId
                  parentId lastFillPrice clientId whyHeld]
      (response-handler {:type :order-status :order-id orderId
                         :status (translate :from-ib :order-status status)
                         :filled filled :remaining remaining
                         :average-fill-price avgFillPrice
                         :permanent-id permId :parent-id parentId
                         :last-fill-price lastFillPrice :client-id clientId
                         :why-held whyHeld}))

    (openOrder [this orderId contract order orderState]
      (response-handler {:type :open-order :order-id orderId :contract (->map contract)
                         :order (->map order) :order-state (->map orderState)}))

    (openOrderEnd [this]
      (response-handler {:type :open-order-end}))

    (nextValidId [this orderId]
      (response-handler {:type :next-valid-order-id :value orderId}))

    ;;; Account and Portfolio
    (updateAccountValue [this key value currency accountName]
      (let [avk (translate :from-ib :account-value-key key)
            val (cond
                  (integer-account-value? avk) (Integer/parseInt value)
                  (numeric-account-value? avk) (Double/parseDouble value)
                  (boolean-account-value? avk) (Boolean/parseBoolean value)
                  :else value)]
        (response-handler {:type :update-account-value :key avk :value val
                           :currency currency :account accountName})))

    (accountDownloadEnd [this account-code]
      (response-handler {:type :account-download-end :account-code account-code}))

    (updatePortfolio [this contract position marketPrice marketValue averageCost
                      unrealizedPNL realizedPNL accountName]
      (response-handler {:type :update-portfolio :contract (->map contract)
                         :position position :market-price marketPrice
                         :market-value marketValue :average-cost averageCost
                         :unrealized-gain-loss unrealizedPNL
                         :realized-gain-loss realizedPNL
                         :account accountName}))

    (updateAccountTime [this timeStamp]
      (response-handler {:type :update-account-time
                         :value (translate :from-ib :time-of-day timeStamp)}))

    ;;; Contract Details
    (contractDetails [this requestId contractDetails]
      (let [{:keys [trading-hours liquid-hours time-zone-id] :as m} (->map contractDetails)]
        (response-handler {:type :contract-details
                           :request-id requestId
                           :value (-> m
                                      (assoc :trading-hours (translate :from-ib :trading-hours [time-zone-id trading-hours]))
                                      (assoc :liquid-hours  (translate :from-ib :trading-hours [time-zone-id liquid-hours])))})))

    (bondContractDetails [this requestId contractDetails]
      (response-handler {:type :contract-details :request-id requestId
                         :value (->map contractDetails)}))

    (contractDetailsEnd [this requestId]
      (response-handler {:type :contract-details-end :request-id requestId}))

    ;;; Execution Details
    (execDetails [this requestId contract execution]
      (response-handler {:type :execution-details :request-id requestId
                         :contract (->map contract)
                         :value (->map execution)}))

    (execDetailsEnd [this requestId]
      (response-handler {:type :execution-details-end :request-id requestId}))

    (commissionReport [this commissionReport]
      (response-handler {:type :commission-report :report (->map commissionReport)}))

    ;;; Market Depth
    (updateMktDepth [this tickerId position operation side price size]
      (response-handler {:type :update-market-depth
                         :ticker-id tickerId
                         :position position
                         :operation (translate :from-ib :market-depth-row-operation
                                               operation)
                         :side (translate :from-ib :market-depth-side side)
                         :price price :size size}))

    (updateMktDepthL2 [this tickerId position marketMaker operation side price size]
      (response-handler {:type :update-market-depth-level-2
                         :ticker-id tickerId :position position
                         :market-maker marketMaker
                         :operation (translate :from-ib :market-depth-row-operation
                                               operation)
                         :side (translate :from-ib :market-depth-side side)
                         :price price :size size}))

    ;;; News Bulletin
    (updateNewsBulletin [this msgId msgType message origExchange]
      (response-handler {:type (condp = msgType
                                 0 :news-bulletin
                                 1 :exchange-unavailable
                                 2 :exchange-available)
                         :id msgId :message message :exchange origExchange}))

    ;;; Financial Advisors
    (managedAccounts [this accountsList]
      (response-handler {:type :managed-accounts
                         :accounts (->> (.split accountsList ",") (map #(.trim %)) vec)}))

    (receiveFA [this faDataType xml]
      (response-handler {:type (condp = faDataType
                                 1 :financial-advisor-groups
                                 2 :financial-advisor-profile
                                 3 :financial-advisor-account-aliases)
                         :value xml}))

    ;;; Historical Data
    (historicalData [this requestId date open high low close volume count wap hasGaps]
      (if (.startsWith date "finished")
        (response-handler {:type :price-bar-complete :request-id requestId})
        (response-handler
         {:type :price-bar :request-id requestId
          :time (translate :from-ib :timestamp date)
          :open open :high high :low low :close close :volume volume
          :trade-count count :WAP wap :has-gaps? hasGaps})))

    ;;; Market Scanners
    (scannerParameters [this xml]
      (response-handler {:type :scan-parameters :value xml}))

    (scannerData [this requestId rank contractDetails distance benchmark
                  projection legsStr]
      (response-handler {:type :scan-result :request-id requestId :rank rank
                         :contract-details (->map contractDetails)
                         :distance distance :benchmark benchmark
                         :projection projection :legs legsStr}))

    (scannerDataEnd [this requestId]
      (response-handler {:type :scan-end :request-id requestId}))

    ;;; Real Time Bars
    (realtimeBar [this requestId time open high low close volume wap count]
      (response-handler {:type :price-bar :request-id requestId
                         :time (translate :from-ib :date-time time)
                         :open open :high high :low low :close close :volume volume
                         :count count :WAP wap}))

    ;;; Fundamental Data
    (fundamentalData [this requestId xml]
      (let [report-xml (-> (java.io.ByteArrayInputStream (.getBytes xml))
                           xml/parse)]
        (response-handler {:type :fundamental-data :request-id requestId
                           :report report-xml})))))

(defn- counter-fn []
  (let [counter (atom (int 0))]
    (fn [] (swap! counter #(inc (int %))))))

(defonce next-id (counter-fn))

(defprotocol ResponseSource
  (response-chan [this]
                 [this buf-or-n]
                 [this buf-or-n xform]
                 [this buf-or-n xform ex-handler]
                 "Returns a new channel that is tapped from a main response channel"))

;;connection and server
(defn connect
  ([socket] (connect socket "localhost" 7496))
  ([socket host port]
   (connect socket host port (next-id)))
  ([socket host port client-id]
   (log/debug "connecting on host:" host "port:" port "client-id:" client-id)
   (.eConnect socket host port client-id)))

(defn disconnect [socket]
  (log/debug "Disconnecting")
  (.eDisconnect socket))

(defn connected? [socket]
  (.isConnected socket))

(defn set-server-log-level [socket log-level]
  (.setServerLogLevel socket (translate :to-ib :log-level log-level)))

(defn request-current-time [socket]
  (.reqCurrentTime socket))

(defn connection-time [socket]
  (translate :from-ib :connection-time (.TwsConnectionTime socket)))

;;market data
(defn request-market-data
  ([socket contract]
   (request-market-data socket contract "" false))
  ([socket contract tick-list snapshot?]
   (let [ticker-id (next-id)]
     (.reqMktData socket ticker-id
                  (map-> com.ib.client.Contract contract)
                  (translate :to-ib :tick-list tick-list)
                  snapshot?)
     ticker-id)))

(defn cancel-market-data [socket ticker-id]
  (.cancelMktData socket ticker-id))

;;orders
(defn place-order [socket contract order]
  (let [order-id (next-id)]
    (.placeOrder socket order-id
                 (map-> com.ib.client.Contract contract)
                 (map-> com.ib.client.Order order))
    order-id))

(defn cancel-order [socket order-id]
  (.cancelOrder socket order-id))

(defn request-open-orders [socket]
  (.reqOpenOrders socket))
(defn request-all-open-orders [socket]
  (.reqAllOpenOrders socket))
;;account and portfolio
(defn request-account-updates [socket subscribe? account-code]
  (.reqAccountUpdates socket subscribe? account-code))


;;executions
;;contract details
(defn request-contract-details [socket contract]
  (let [request-id (next-id)]
    (.reqContractDetails
     socket request-id
     (map-> com.ib.client.Contract contract))
    request-id))


;;market depth
(defn request-market-depth [socket contract num-rows]
  (let [ticker-id (next-id)]
    (.reqMktDepth socket
                  ticker-id
                  (map-> com.ib.client.Contract contract)
                  num-rows)
    ticker-id))
(defn cancel-market-depth [socket ticker-id]
  (.cancelMktDepth socket ticker-id))

;;news bulletins
(defn request-news-bulletins [socket] (request-news-bulletins socket true))
(defn request-news-bulletins [socket all-messages?]
  (.reqNewsBulletins socket all-messages?))
(defn cancel-news-bulletins [socket]
  (.cancelNewsBulletins socket))

;;historical data
(defn request-historical-data [socket contract end-date
                               duration duration-unit
                               bar-size bar-size-unit
                               what-to-show use-regular-trading-hours?]
  (let [[acceptable-duration acceptable-duration-unit]
        (translate :to-ib :acceptable-duration [duration duration-unit])
        request-id (next-id)]
    (.reqHistoricalData socket
                        request-id
                        (map-> com.ib.client.Contract contract)
                        (translate :to-ib :date-time end-date)
                        (translate :to-ib :duration [acceptable-duration
                                                     acceptable-duration-unit])
                        (translate :to-ib :bar-size [bar-size bar-size-unit])
                        (translate :to-ib :what-to-show what-to-show)
                        (if use-regular-trading-hours? 1 0)
                        2) ;;formatDate. 2 for long date, 1 for string date
    request-id))
;;real-time-bars
(defn request-real-time-bars [socket contract what-to-show use-regular-trading-hours?]
  (let [ticker-id (next-id)]
    (.reqRealTimeBars socket
                      ticker-id
                      (map-> com.ib.client.Contract contract)
                      5 ;;barSize, 5seconds, the only currently supported option
                      (translate :to-ib :what-to-show what-to-show)
                      use-regular-trading-hours?)
    ticker-id))

(defn cancel-real-time-bars [socket ticker-id]
  (.cancelRealTimeBars socket ticker-id))

;;fundamental data
(defn request-fundamental-data [socket contract report-type]
  (let [request-id (next-id)]
    (.reqFundamentalData socket request-id
                         (map-> com.ib.client.Contract contract)
                         (translate :to-ib :report-type report-type))
    request-id))
(defn cancel-fundamental-data [socket request-id]
  (.cancelFundamentalData socket request-id))


(defn ib-socket [f]
  (com.ib.client.EClientSocket. (handler-ewrapper f)))

